/*
    XCruise - A directory browser
    Copyright (C) 1999  Yusuke Shinyama <euske@cl.cs.titech.ac.jp>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 *  draw.c
 *	 Draw the universe on a window
 */


#include "xcruise.h"

#include <limits.h>


/*  Local type definitions
 */
typedef struct LineBuff {
    int      lines;
    XSegment seg[GalaxyPoly];
} LineBuff;


/*  Show a message atop of the window
 */
void alert(char* s, int ypos)
{
    int x, len = strlen(s);
    x = (screenWidth - XTextWidth(alrtfont, s, len))/2;
    XDrawString(disp, mywin, alrtgc, x,
		screenHeight/2 + ypos*alrtfont->ascent , s, len);
    XFlush(disp);
}

/*  Draw a cursor
 */
void drawCursor(int cx, int cy)
{
    XSegment seg[4];
    cx += screenWidth/2;
    cy = screenHeight/2-cy;
    seg[0].y1 = seg[1].y1 = seg[0].y2 = seg[1].y2 = cy;
    seg[2].x1 = seg[3].x1 = seg[2].x2 = seg[3].x2 = cx;
    seg[0].x1 = cx - CursorSize;
    seg[0].x2 = cx - CursorSize*2;
    seg[1].x1 = cx + CursorSize;
    seg[1].x2 = cx + CursorSize*2;
    seg[2].y1 = cy - CursorSize;
    seg[2].y2 = cy - CursorSize*2;
    seg[3].y1 = cy + CursorSize;
    seg[3].y2 = cy + CursorSize*2;
    XDrawSegments(disp, myscreen, cursorgc, seg, 4);
}

/*  Draw informations
 */
void drawInfo(void)
{
    char	s[20];
    sprintf(s, "%3.1f", vspeed);
    XDrawString(disp, myscreen, infogc, 30, 60,
		"Velocity:", cstrlen("Velocity:"));
    XDrawString(disp, myscreen, infogc, 30, 100,
		"Current Galaxy:", cstrlen("Current Galaxy:"));
    XDrawString(disp, myscreen, infogc, 280, 60, s, strlen(s));
    XDrawString(disp, myscreen, infogc, 280, 100, 
		curzone->g.path, strlen(curzone->g.path));
    XDrawString(disp, myscreen, infogc, screenWidth-300, screenHeight-140,
		"Button1 : Forward", cstrlen("Button1 : Forward"));
    XDrawString(disp, myscreen, infogc, screenWidth-300, screenHeight-100,
		"Button2 : Backward", cstrlen("Button2 : Backward"));
    XDrawString(disp, myscreen, infogc, screenWidth-300, screenHeight-60,
		"  [Q]   : Quit", cstrlen("  [Q]   : Quit"));
}

/*  Draw measures
 */
void drawMeasure(int rot1, int rot2)
{
    int		i, x, x0, y0, tw;
    char    s[20];
    XSetClipRectangles(disp, meangc, 0, 0, &meanhclip, 1, YXBanded);
    i = ((rot1 - (MeasureSize/2/MeasureUnit))/30) * 30;
    x = - MeasureUnit * (rot1 - i);
    y0 = meanhclip.y;
    while(x < MeasureSize + 20) {
	x0 = x + screenWidth/2;
	XDrawLine(disp, myscreen, meangc, x0, y0, x0, y0+10);
	XDrawLine(disp, myscreen, meangc,
		  x0+15*MeasureUnit, y0, x0+15*MeasureUnit, y0+5);
	sprintf(s, "%d", (i+360)%360);
	tw = XTextWidth(meanfont, s, strlen(s));
	XDrawString(disp, myscreen, meangc, 
		    x0-tw/2, y0+12+meanfont->ascent, s, strlen(s));
	x += 30*MeasureUnit;
	i += 30;
    }
    XSetClipRectangles(disp, meangc, 0, 0, &meanvclip, 1, YXBanded);
    i = ((rot2 - (MeasureSize/2/MeasureUnit))/30) * 30;
    x = - MeasureUnit * (rot2 - i);
    x0 = meanvclip.x;
    while(x < MeasureSize + 20) {
	y0 = x + screenHeight/2;
	XDrawLine(disp, myscreen, meangc, x0, y0, x0+10, y0);
	XDrawLine(disp, myscreen, meangc,
		  x0, y0+15*MeasureUnit, x0+5, y0+15*MeasureUnit);
	sprintf(s, "%d", (i+360)%360);
	XDrawString(disp, myscreen, meangc,
		    x0+12, y0+meanfont->ascent/2, s, strlen(s));
	x += 30*MeasureUnit;
	i += 30;
    }
}

/*  Transfer a image buffer to the main window
 */
void redrawScreen(void)
{
    XCopyArea(disp, myscreen, mywin, defaultgc, 
	      0, 0, screenWidth, screenHeight, 0, 0);
}



/*  Draw stars
 */

/*  Draw various size texts (used for showing stars' names)
 */
static void drawText(int x, int y, int size, char* s)
{
    GC sizegc;
    XFontStruct* sizefont;
    int len = strlen(s);
    if (0 < size) {
	if (size < namesize[0]) {
	    sizegc = namegc[0];
	    sizefont = namefont[0];
	} else if (size < namesize[1]) {
	    sizegc = namegc[1];
	    sizefont = namefont[1];
	} else if (size < namesize[2]) {
	    sizegc = namegc[2];
	    sizefont = namefont[2];
	} else {
	    sizegc = namegc[3];
	    sizefont = namefont[3];
	}
	x -= XTextWidth(sizefont, s, len)/2;
	XDrawString(disp, myscreen, sizegc, x, y, s, len);
    }
}

/*  Insert a star into the z-buffer
 */
static void insertZStar(galaxyinfo* g, star* s1)
{
    ZStar* zs0 = g->view;
    ZStar* zs1;
    int	z;
    if (INT_MAX < s1->i.perspos.z) {
	z = INT_MAX;
    } else if (s1->i.perspos.z < INT_MIN) {
	z = INT_MIN;
    } else {
	z = s1->i.perspos.z;
    }
    if ((s1->i.type == T_Planet && 0 < z) ||
	(s1->i.type == T_Galaxy && 0 < z+s1->i.r) ||
	(s1->i.type == T_Wormhole)) {
	zs0[g->nview].z = INT_MIN;
	while(z < zs0->z)
	    zs0++;
	for(zs1 = &g->view[g->nview]; zs0 < zs1; zs1--)
	    *zs1 = *(zs1-1);
	zs0->z = z;
	zs0->s = s1;
	g->nview++;
    }
}

/*  Calculate perspective position
 */
static double calcPersPos(Point3D* res, Point3D* pos0)
{
    Point3D difc;
    difc.x = pos0->x - viewp.x;
    difc.y = pos0->y - viewp.y;
    difc.z = pos0->z - viewp.z;
    res->x = product(rightv, difc);
    res->y = product(upperv, difc);
    res->z = product(forwardv, difc);
    
    return(vlength(difc));
}

/*  Translate 3D to 2D
 */
#define ClipMin -10000.0
#define ClipMax 10000.0
static void trans2DPos(short* x, short* y, Point3D* perspos)
{
    double x0, y0;
    x0 = screenSize * perspos->x / perspos->z + screenWidth/2;
    y0 =-screenSize * perspos->y / perspos->z + screenHeight/2;
    if (x0 < ClipMin)
    { x0 = ClipMin; y0 = ClipMin * y0 / x0; }
    else if (ClipMax < x0)
    { x0 = ClipMax; y0 = ClipMax * y0 / x0; }
    if (y0 < ClipMin)
    { x0 = ClipMin * x0 / y0; y0 = ClipMin; }
    else if (ClipMax < y0)
    { x0 = ClipMax * x0 / y0; y0 = ClipMax; }
    *x = x0; *y = y0;
}

static short trans2DR(double r0, double dist)
{
    return(screenSize * r0 / dist);
}

/*  Clip a line which penetrates the screen
 */
#define ZClip	0.1
static int clipLine(Point3D* p0, Point3D* p1)
{
    double	z;
    if (ZClip < p0->z && ZClip < p1->z)
	return(true);
    if (p0->z <= ZClip && p1->z <= ZClip)
	return(false);
    z = (ZClip - p0->z)/(p1->z - p0->z);
    if (p0->z <= ZClip) {
	p0->x = (p1->x - p0->x)*z + p0->x;
	p0->y = (p1->y - p0->y)*z + p0->y;
	p0->z = ZClip;
    } else {
	p1->x = (p1->x - p0->x)*z + p0->x;
	p1->y = (p1->y - p0->y)*z + p0->y;
	p1->z = ZClip;
    }
    return(true);
}

/*  Draw a planet
 */
static void drawPlanet(planetinfo* p)
{
    short   x, y, r, c;
    c = (p->info.parent == curzone)?
	p->info.col : (p->info.col+CurrentColors);
    trans2DPos(&x, &y, &p->info.perspos);
    
    r = trans2DR(p->info.r, p->info.dist);
    if (r < screenSize && inValid(x, y, r)) {
	XFillArc(disp, myscreen, stargc[c], x-r, y-r, r*2, r*2, 0, 360*64);
	drawText(x, y-r, r/4, p->info.name);
    }
}

/*  Draw a galaxy (recursively)
 */
static void drawStars0(star* s);
static void drawGalaxy(galaxyinfo* g);

static void drawGalaxy(galaxyinfo* g)
{
    LineBuff lbefore, lafter;
    Point3D	px, py, mpx, mpy, p0, p1, pos;
    int		i, istep, sid0, sid1;
    short  	r, c, x, y;
    XSegment* bseg,* aseg;
    double	rv, pz;

    pos = g->info.perspos;
    c = (g->info.parent == curzone)? 
	g->info.col : (g->info.col+CurrentColors);
    trans2DPos(&x, &y, &pos);
    r = trans2DR(rv = g->info.r, g->info.dist);

    /* it's drawable */
    if (g->info.dist < rv || inValid(x, y, r)) {
	/* pre-calc */
	px = g->info.px;	py = g->info.py;
	mpx.x = rv * product(rightv, px);
	mpx.y = rv * product(upperv, px);
	mpx.z = rv * product(forwardv, px);
	mpy.x = rv * product(rightv, py);
	mpy.y = rv * product(upperv, py);
	mpy.z = rv * product(forwardv, py);

	if (0 < pos.z)
	    drawText(x, y-r, r/4, g->info.name);

	istep = 1;
	if (r < Poly8R)
	    istep = 8;
	else if (r < Poly16R)
	    istep = 4;
	else if (r < Poly32R)
	    istep = 2;

	/* calc the ring */
	lbefore.lines = lafter.lines = 0;
	bseg = lbefore.seg;
	aseg = lafter.seg;
	p0.x = mpx.x*cirx[GalaxyPoly-istep] + 
	    mpy.x*ciry[GalaxyPoly-istep] + pos.x;
	p0.y = mpx.y*cirx[GalaxyPoly-istep] +
	    mpy.y*ciry[GalaxyPoly-istep] + pos.y;
	p0.z = (pz = mpx.z*cirx[GalaxyPoly-istep] + 
		mpy.z*ciry[GalaxyPoly-istep]) + pos.z;
	sid0 = (0.0 <= pz);
	
	for(i = 0; i < GalaxyPoly; i += istep) {
	    p1.x = mpx.x*cirx[i] + mpy.x*ciry[i] + pos.x;
	    p1.y = mpx.y*cirx[i] + mpy.y*ciry[i] + pos.y;
	    p1.z = (pz = mpx.z*cirx[i] + mpy.z*ciry[i]) + pos.z;
	    sid1 = (0.0 <= pz);
	    if (clipLine(&p0, &p1)) {
		if (sid0 && sid1) {
		    trans2DPos(&bseg->x1, &bseg->y1, &p0);
		    trans2DPos(&bseg->x2, &bseg->y2, &p1);
		    lbefore.lines++; bseg++;
		} else {
		    trans2DPos(&aseg->x1, &aseg->y1, &p0);
		    trans2DPos(&aseg->x2, &aseg->y2, &p1);
		    lafter.lines++; aseg++;
		}
	    }
	    p0 = p1;
	    sid0 = sid1;
	}

	/* draw the first half */
	XDrawSegments(disp, myscreen, stargc[c], lbefore.seg, lbefore.lines);
	
	/* draw the inside galaxy */
	if (g->expanded) {
	    if ((star*)g != &universe && r < ViewR) {
		g->marked = 1;
	    } else {
		ZStar* z1;
		star* s1;
		if (g->info.dist < g->info.r &&
		    g->info.parent == curzone) {
		    stpswin = (star*)g;
		}
		g->nview = 0;
		s1 = g->children;
		for(i = 0; i < g->nst; i++, s1++) {
		    s1->i.dist = calcPersPos(&s1->i.perspos, &s1->i.pos);
		    insertZStar(g, s1);
		}
		z1 = g->view;
		for(i = 0; i < g->nview; i++, z1++)
		    drawStars0(z1->s);
	    }
	} else if (ViewR < r) {
	    expandGalaxy(g);
	}
	
	/* draw the second half */
	XDrawSegments(disp, myscreen, stargc[c], lafter.seg, lafter.lines);
    }
}

/*  Draw a planet or a galaxy
 */
static void drawStars0(star* s)
{
    switch(s->i.type) {
    case T_Galaxy:
	drawGalaxy(&s->g);
	break;
    case T_Planet:
	drawPlanet(&s->p);
	break;
    }
}

/*  Draw wormholes
 */
void drawWormholes(void)
{
    int		n, i, nline;
    short   x, y, r, c;
    Point3D* pts;
    Point3D	p0, p1;
    wormholeinfo*	w1;
    XSegment seg[WormPoints];
    XSegment* nowseg;
    
    for(n = 0; n < numlinks; n++) {
	w1 = curlinks[n];
	pts = w1->wpt;
	c = (w1->info.parent == curzone)? 
	    w1->info.col : (w1->info.col+CurrentColors);
	calcPersPos(&w1->info.perspos, &w1->info.pos);
	if (0 < w1->info.perspos.z) {
	    trans2DPos(&x, &y, &w1->info.perspos);
	    r = trans2DR(w1->info.r, w1->info.dist);
	    XDrawArc(disp, myscreen, stargc[c], 
		     x-r, y-r, r*2, r*2, 0, 360*64);
	    drawText(x, y-r, r/4, w1->info.name);
	}
	p0 = w1->info.perspos;
	nowseg = seg;
	nline = 0;
	for(i = 1; i < WormPoints; i++) {
	    calcPersPos(&p1, pts++);
	    if (clipLine(&p0, &p1)) {
		trans2DPos(&nowseg->x1, &nowseg->y1, &p0);
		trans2DPos(&nowseg->x2, &nowseg->y2, &p1);
		nline++; nowseg++;
	    }
	    p0 = p1;
	}
	XDrawSegments(disp, myscreen, stargc[c], seg, nline);
    }
}

/*  Draw the whole universe
 */
void drawTheUniverse(void)
{
    stpswin = NULL;
    universe.i.dist = calcPersPos(&universe.i.perspos, &universe.i.pos);
    drawStars0(&universe);
    drawWormholes();
}
