/*
 * entity.c: Entity manipulation (creation/init, removal)
 * Copyright 1999 David Lawrence (philaw@camtech.net.au)
 * Last modified July 27 '99.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include "header.h"
#include "win.h"
#include "entity.h"

extern struct timeval game_init;
extern entity *root, *player;
#if FOCUSFOLLOWSNUKE
extern entity *focus;
#endif

int id_num = 0; /* Used for unique ID for each entity */
int num_entities = 0;

entity *create_entity(int type)
{

  entity *ent;
  ent = root;

  /* Set ent to last entity in list */
  while( ent->next != NULL )
    ent = ent->next;

  if( (ent->next = (entity *)malloc(sizeof(entity))) == NULL) {
    fprintf(stderr,"Malloc error creating entity %d\n", id_num);
    die("Exiting!\n", 1);
  }

  ent->next->ent_id = id_num++; /* We want tux to be 0 */
  num_entities++;

#if VERBOSE  
  printf("Created entity, there are %d entities\n", num_entities);
#endif

  /* Initialise entity */
  ent = ent->next;

  ent->type = type;
  ent->mode = ALIVE;
  ent->x_v = 0;
  ent->y_v = 0;
  ent->dir = 4;
  ent->frame = 0;
  ent->img_no = 0;
  ent->last_frame = game_init;
  ent->next = NULL;

  if( ent->type >= ITEM_OFFSET )
    ent->class = ITEM;
  else if( ent->type >= PROJECTILE_OFFSET )
    ent->class = PROJECTILE;
  else if( ent->type >= NEUTRAL_OFFSET )
    ent->class = NEUTRAL;
  else if( ent->type >= BADDIE_OFFSET )
    ent->class = BADDIE;
  else
    ent->class = GOODIE;
  
  /* Class specific data */
  switch( ent->class ) {
  case GOODIE:
    ent->width = 32;
    ent->height = 32;

    switch( ent->type ) {
    case VI:
      ent->speed = 16;
      ent->health = 60;
      break;
    case GNU:
      ent->speed = 4;
      ent->health = 150;
      break;
    default:
      ent->speed = 8;
      ent->health = 100;
    }
    break;
  case BADDIE:
    ent->width = 32;
    ent->height = 32;
    ent->speed = 4;
    ent->health = 0;
    switch( ent->type ) {
    case TROLL: /* 15 */
      ent->health += 5;
    case FLAMER: /* 10 */
    case CLIPPY:
      ent->health += 5;
    default: /* 5 */
      ent->health += 5;
    }
    break;
  case NEUTRAL:
    ent->width = 24;
    ent->height = 24;
    ent->speed = 4;
    ent->dir = rand()%8; /* Just to make big packs of them look interesting */
    ent->health = 1;
    break;
  case PROJECTILE:
  case ITEM:
    ent->width = 32;
    ent->height = 32;
    ent->health = 1;
    break;
  }

  return ent; /* Pointer to newly made entity */

}

/* Remove entity that with appropriate ent_id.
   returns the entity which was before the one
   removed, so if you use ent = remove_entity(ent->id)
   ent->next should be the same before and after removal */
entity *remove_entity(int ent_id)
{

  entity *ent;
  entity *tmp_ptr;
  int type, id;

  ent = root;

  /* We want ent->next to be the entity we are going to remove */
  while( ent->next != NULL && ent->next->ent_id != ent_id )
    ent = ent->next;

  if( ent->next == NULL ) {
    fprintf(stderr, "Fell off the end of the list in remove entity!\n"
	    "We were looking for ent_id of %d\n", ent_id);
    return (entity *)NULL;
  }

#if FOCUSFOLLOWSNUKE
  if( focus == ent->next )
    focus = player;
#endif

  id = ent->next->ent_id;
  type = ent->next->type;

  num_entities--;

#if VERBOSE
  printf("Removed entity(type %d, id %d) Total:%d entities\n",
	 type, id, num_entities);
#endif

  tmp_ptr = ent->next->next;
  free(ent->next);
  ent->next = tmp_ptr;

  return ent;

}


/* All means all but tux and root */
void remove_all_entities(void)
{

  int i = 0;
  entity *ent, *next;

  ent = player->next; /* player == root->next */

  while( ent != NULL ) {

    next = ent->next;
    free(ent);
    ent = next;
    i++;
    num_entities--;

  }

  /* Reset player list member */
  player->next = NULL;

#if VERBOSE
  fprintf(stderr, "Free'd %d entities\n", i);
#endif
  /* We only have 2 (1 actual) entities left */

}
