/* Terraform - (C) 1997-2000 Robert Gasch (r.gasch@chello.nl)
 *  - http://212.187.12.197/RNG/terraform/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <math.h>
#include "HeightFieldGenSubdiv.h"
#include "GlobalTrace.h"
#include "GlobalSanityCheck.h"
#include "MathGauss.h"
#include "string.h"


#define EMPTY	0.0

/*
 *  constructor: initialize
 */
HeightFieldGenSubdiv::HeightFieldGenSubdiv (HeightField *HF) 
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "+++ HeightFieldGenSubdiv\n");

	SanityCheck::bailout ((!HF), "HF==NULL", "HeightFieldGenSubdiv::HeightFieldGenSubdiv");

	p_HF = HF;
	p_mGauss = NULL;
	d_prefSize = d_reqSize = -1;
	d_RscaleFactor = 0.5;
}


/*
 *  destructor: noting to clean up
 */
HeightFieldGenSubdiv::~HeightFieldGenSubdiv ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_FLOW, "--- HeightFieldGenSubdiv\n");
}


/*
 *  setPreferredSize: expand the HF size until it is a power of 2
 */
int HeightFieldGenSubdiv::setPreferredSize (int n)
{
	char		buf[80];
	int		pow2=1;
	double		v;

	d_reqSize = -1;

	while ((v=pow(2,pow2++)) < n)
		;
	if (v != n)			// size is not power of 2
		{
		d_reqSize = n;
		d_prefSize = (int)v;
		sprintf (buf, "Expanding %d to %d ... \n",  d_reqSize, d_prefSize);
		GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, buf);
		}
	return (d_prefSize);
}


/*
 *  setRequestedSize: extract the HF of requested size from the power of 2 HF
 */
void HeightFieldGenSubdiv::setRequestedSize ()
{
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "downsizing ... \n");

	int             x, xx, y, yy, 
			d, lim; 
	PTYPE           *hf2; 

	d = (d_prefSize-d_reqSize)/2;
	lim = d_prefSize-d;
	if ((d_prefSize-d_reqSize)%2 == 1)	// make sure we don't overcount
		lim--;
	hf2 = new PTYPE[d_reqSize*d_reqSize];

	for (y=d, yy=0; y<lim; y++, yy++)
		for (x=d, xx=0; x<lim; x++, xx++)
			hf2[yy*d_reqSize+xx] = p_HF->getEl(x,y);

	p_HF->init (hf2, d_reqSize, d_reqSize, p_HF->getName());
}


/*
 *  avgvvals: used by Diamond generation routine
 */
float HeightFieldGenSubdiv::avgyvals (int i, int j, int strut, int n)
{
	int	width = d_prefSize;

	if (i == 0)
	    return ((p_HF->getEl(i,(j-strut)&(width-1)) +
		     p_HF->getEl(i,(j+strut)&(width-1)) +
		     p_HF->getEl((i+strut)&(width-1),j)) / 3);
	else 
	if (i == n-1)
	    return ((p_HF->getEl(i,(j-strut)&(width-1)) +
		     p_HF->getEl(i,(j+strut)&(width-1)) +
		     p_HF->getEl((i-strut)&(width-1),j)) / 3);
	else 
	if (j == 0)
	    return ((p_HF->getEl((i-strut)&(width-1),j) +
		     p_HF->getEl((i+strut)&(width-1),j) +
		     p_HF->getEl(i,(j+strut)&(width-1))) / 3);
	else 
	if (j == n-1)
	    return ((p_HF->getEl((i-strut)&(width-1),j) +
		     p_HF->getEl((i+strut)&(width-1),j) +
		     p_HF->getEl(i,(j-strut)&(width-1))) / 3);
	else
	    return ((p_HF->getEl((i-strut)&(width-1),j) +
		     p_HF->getEl((i+strut)&(width-1),j) +
		     p_HF->getEl(i,(j-strut)&(width-1)) +
		     p_HF->getEl(i,(j+strut)&(width-1))) / 4);
}


/*
 *  avgvvals2: used by Diamond generation routine
 */
float HeightFieldGenSubdiv::avgyvals2 (int i, int j, int strut, int n)
{
	int     tstrut = strut/2;
	int	width = d_prefSize;
    
	return ((p_HF->getEl((i-tstrut)&(width-1),(j-tstrut)&(width-1)) +
		 p_HF->getEl((i-tstrut)&(width-1),(j+tstrut)&(width-1)) +
		 p_HF->getEl((i+tstrut)&(width-1),(j-tstrut)&(width-1)) +
		 p_HF->getEl((i+tstrut)&(width-1),(j+tstrut)&(width-1))) / 4);
}


/*
 *  generate: generation routine which provides a wrapper for all the 
 * 	different generation routing this object has. 
 */
int HeightFieldGenSubdiv::generate (int mode, int n, float scaleFactor,
				    int seed)
{
	int	rc;

	if (mode == 1)
		rc = generateRecursive (n, scaleFactor, seed);
	else
	if (mode == 2)
		rc = generateOffset (n, scaleFactor, seed);
	else
	if (mode == 3)
		rc = generateMidpoint (n, scaleFactor, seed);
	else
	if (mode == 4)
		rc = generateDiamond (n, scaleFactor, seed);
	else
		{
		SanityCheck::warning ((n>4), "n>4, defaulting to recursive generation", "HeightFieldGenSubdiv::generate");
		rc = generateRecursive (n, scaleFactor, seed);
		}

	return rc;
}
    

/*
 * Plasma example - diamond-square midpoint subdivision on a grid.
 * The surface is iteratively computed by calculating the center point of
 * the initial square (where the four corners of the surface are the
 * corners of the square), then of the resulting diamonds, then of
 * squares again, etc.
 * 
 * Code adapted from 'fillSurf' example posted to Usenet by
 * Paul Martz (pmartz@dsd.es.com).  
 * 
 * Original copyright notice:
 * Use this code anyway you want as long as you don't sell it for money.
 */
int HeightFieldGenSubdiv::generateDiamond (int	n, 
					float	scaleFactor,
					int 	seed)
{
	int 	i, j,
		strut, tstrut; 
	bool	oddline; 
	float	maxDelta = n;

	SanityCheck::bailout ((n<5), "n<5", "HeightFieldGenSubdiv::generateDiamond");
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Generating using Diamond Squares ...\n");

	p_mGauss = new MathGauss (seed);

	n = setPreferredSize (n);
	p_HF->newHF (n, n);

	// initialize things 
	strut = n / 2;
    
	// create fractal surface from seeded values 
	tstrut = strut;
	while (tstrut>0)
	    {
	    oddline = false;
	    for (i=0; i<n; i+=tstrut)
		{
		oddline = (!oddline);
		for (j=0; j<n; j+=tstrut)
		    {
		    if ((oddline) && (j==0)) 
			j+=tstrut;
		    p_HF->setEl (i,j, (avgyvals (i, j, tstrut, n) + 
					p_mGauss->rnd()*maxDelta));
		    j+=tstrut;
		    }
		}

	    if (tstrut/2 != 0)
		for (i=tstrut/2; i<n; i+=tstrut)
		    for (j=tstrut/2; j<n; j+=tstrut)
			p_HF->setEl(i,j, (avgyvals2(i, j, tstrut, n) + 
					     p_mGauss->rnd()*maxDelta));
	    tstrut /= 2;
	    maxDelta *= scaleFactor;
	    }

	if (d_reqSize != -1)
		setRequestedSize ();
	else
		p_HF->gatherStatistics ();
	p_HF->setSaved (FALSE);
	delete p_mGauss;
	p_mGauss = NULL;

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Height Field Generation done\n");

	return 0;
}



/*
 * Plasma example - offset square midpoint subdivision on a grid.
 * This method is similar to standard subdivision, but it is 'offset'
 * in that the new points at each level are offset from the previous
 * level by half the square size.  The original corner points at level
 * x are only used once to form a weighted average, and then become the
 * center points of the new squares:
 *
 * It should be clear that this method includes some non-local context
 * when calculating new heights.  With standard subdivision, the only
 * heights that can ever influence the area inside the original square
 * are the original corner points.  With offset squares, most of the
 * new corner points lie outside the original square and therefore
 * are influenced by more distant points.  This feature can be a problem,
 * because if you don't want the map to be toroidal (wrapped in both
 * x and y), you need to generate a large number of points outside the
 * final terrain area.  For this example, I just wrap x and y.
 *
 * Original copyright notice:
 * This code was inspired by looking at Tim Clark's (?) Mars demo. You are 
 * free to use my code, my ideas, whatever. I have benefited enormously from 
 * the free information on the Internet, and I would like to keep that 
 * process going. James McNeill (mcneja@wwc.edu)
 */
int HeightFieldGenSubdiv::generateOffset (int	n, 
					float	scaleFactor,
					int 	seed)
{
	float	maxDelta = n;
	//PTYPE	*hf = NULL;

	SanityCheck::bailout ((n<5), "n<5", "HeightFieldGenSubdiv::generateOffset");
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Generating using Offset Squares ...\n");

	p_mGauss = new MathGauss (seed);

	n = setPreferredSize (n);
	p_HF->newHF (n, n);

	int row_offset = 0;  // start at zero for first row

	for (int square_size = n; square_size > 1; square_size /= 2)
	    {
	    for (int x1 = row_offset; x1 < n; x1 += square_size)
		{
		for (int y1 = row_offset; y1 < n; y1 += square_size)
			{
			// Get the four corner points.
			int x2 = (x1 + square_size) % n;
			int y2 = (y1 + square_size) % n;

			float i1 = p_HF->getEl(x1,y1);
			float i2 = p_HF->getEl(x2,y1);
			float i3 = p_HF->getEl(x1,y2);
			float i4 = p_HF->getEl(x2,y2);

			// Obtain new points by averaging the corner points.
			float p1 = ((i1 * 9) + (i2 * 3) + (i3 * 3) + (i4)) / 16;
			float p2 = ((i1 * 3) + (i2 * 9) + (i3) + (i4 * 3)) / 16;
			float p3 = ((i1 * 3) + (i2) + (i3 * 9) + (i4 * 3)) / 16;
			float p4 = ((i1) + (i2 * 3) + (i3 * 3) + (i4 * 9)) / 16;
    
			// Add a random offset to each new point.
			p1 = (int)(p1 + p_mGauss->rnd()*maxDelta);
			p2 = (int)(p2 + p_mGauss->rnd()*maxDelta);
			p3 = (int)(p3 + p_mGauss->rnd()*maxDelta);
			p4 = (int)(p4 + p_mGauss->rnd()*maxDelta);
    
			// Write out the generated points.
			int x3 = (x1 + square_size/4) % n;
			int y3 = (y1 + square_size/4) % n;
			x2 = (x3 + square_size/2) % n;
			y2 = (y3 + square_size/2) % n;
    
			//printf ("%d, %d\n, %d, %d\n, %d, %d\n, %d %d\n\n", 
			//	x3, y3, x2, y3, x3, y2, x2, y2);

			p_HF->setEl(x3,y3, p1);
			p_HF->setEl(x2,y3, p2);
			p_HF->setEl(x3,y2, p3);
			p_HF->setEl(x2,y2, p4);
	     		}
		}
	    row_offset = square_size/4;  // set offset for next row
	    maxDelta *= scaleFactor;
	   }

	if (d_reqSize != -1)
		setRequestedSize ();
	else
		p_HF->gatherStatistics ();
	p_HF->setSaved (FALSE);
	delete p_mGauss;
	p_mGauss = NULL;

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Height Field Generation done\n");

	return 0;
}


/*
 * Plasma example: basic midpoint subdivision on a grid where 
 * grid size is a power of 2. 
 *
 * K is scaling factor (ratio of altitude variation to length),
 * H is constant noise factor, independent of area size;
 * 
 * typical values passed in are (0.5, 0.0) so that, with a width cell
 * grid, starting values range from -128..128.  This still runs the
 * risk of overflow, but can still be acceptable.
 * 
 * altitude at each level is uniform random, + or - (h*K+H)/2
 * where h is the stepsize (length of area side)
 * 
 * Code base on a Usenet posting by Andrea Griffini
 */
int HeightFieldGenSubdiv::generateMidpoint 	(int	n, 
					float	scaleFactor,
					int 	seed)
{
	int 	i,j,i2,j2,i3,j3,h,h2;
	float	a,b,c,d, 
		maxDelta = n; 

	SanityCheck::bailout ((n<5), "n<5", "HeightFieldGenSubdiv::generateMidpoint");
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Generating using Midpoint Subdivision ...\n");

	p_mGauss = new MathGauss (seed);

	n = setPreferredSize (n);
	p_HF->newHF (n, n);

	for (h=n; h>1; h=h2)
	    {
	    h2 = h/2;
	    for (i=0; i<n; i+=h)
		{
		i2 = ((i+h)&(n-1));
		for (j=0; j<n; j+=h)
			{
			j2 = ((j+h)&(n-1));
			i3 = i+h2; 
			j3 = j+h2;
			a = p_HF->getEl(i,j);
			b = p_HF->getEl(i2,j);
			c = p_HF->getEl(i,j2);
			d = p_HF->getEl(i2,j2);

			//printf ("%d, %d\n, %d, %d\n, %d, %d\n, %d %d\n\n", 
			//	i3, j, i, j3, i2, j3, j3, j2);

			// center
			p_HF->setEl (i3,j3, (a+b+c+d)/4 + (p_mGauss->rnd()*maxDelta));

			// top, left, right, bottom
			if (i==0)
			    p_HF->setEl (i3, j, ((a+b)/2 + (p_mGauss->rnd()*maxDelta)));
			if (j==0)
			    p_HF->setEl (i, j3, ((a+c)/2 + (p_mGauss->rnd()*maxDelta)));
			p_HF->setEl (i2, j3, ((b+d)/2 + (p_mGauss->rnd()*maxDelta)));
			p_HF->setEl (i3, j2, ((c+d)/2 + (p_mGauss->rnd()*maxDelta)));
			}
		}
	   maxDelta *= scaleFactor;
	   }

	if (d_reqSize != -1)
		setRequestedSize ();
	else
		p_HF->gatherStatistics ();
	p_HF->setSaved (FALSE);
	delete p_mGauss;
	p_mGauss = NULL;

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Height Field Generation done\n");

	return 0;
}


/*
 *  generateRecursive: create a landscape through recursive square subdivision.
 *	This recursive generation code is actully mine (RNG)
 */
int HeightFieldGenSubdiv::generateRecursive 	(int	n, 
					float	scaleFactor,
					int 	seed)
{
	char 		buf[80];
	int 		cx, cy; 		// center x, y
	float		initDelta=n;

	SanityCheck::bailout ((n<5), "n<5", "HeightFieldGenSubdiv::generate");

	d_RscaleFactor = scaleFactor;
	p_mGauss = new MathGauss (seed);

	n = setPreferredSize (n);
	p_HF->newHF (n, n);

	// setup initial square
	cx = cy = (int) ((n-1)/2);
	p_HF->setEl (0, p_mGauss->gauss()*initDelta);
	p_HF->setEl (cx, cy, p_mGauss->gauss()*initDelta);
	p_HF->setEl (0, cy, p_mGauss->gauss()*initDelta);
	p_HF->setEl (cx, 0, p_mGauss->gauss()*initDelta);
	p_HF->setEl (cx, n-1, p_mGauss->gauss()*initDelta);
	p_HF->setEl (n-1, cy, p_mGauss->gauss()*initDelta);
	p_HF->setEl (p_HF->getSize()-1, p_mGauss->gauss()*initDelta);
	if (GlobalTrace::isSet (GlobalTrace::TRACE_DEBUG))
		{
		sprintf (buf, "Seeding at (%d, %d); %f, %f, %f, %f  ...\n",  cx, cy, 
			p_HF->getEl(0,cy), p_HF->getEl(cx,0), 
        		p_HF->getEl(cx,n-1), p_HF->getEl(n-1,cy));
		GlobalTrace::trace (TRUE, buf);
		}

	// process subsquares recursively
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Q1 ... ");
	processRSquare (0, 0, cx, cy, initDelta);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Q2 ... ");
	processRSquare (cx, 0, n-1, cy, initDelta);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Q3 ... ");
	processRSquare (0, cy, cx, n-1, initDelta);
	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "Q4 ... ");
	processRSquare (cx, cy, n-1, n-1, initDelta);

	if (d_reqSize != -1)
		setRequestedSize ();
	else
		p_HF->gatherStatistics ();
	p_HF->setSaved (FALSE);
	delete p_mGauss;
	p_mGauss = NULL;

	GlobalTrace::trace (GlobalTrace::TRACE_VERBOSE, "done\n");

	return 0;
}



/*
 *  processRSquare: recursive function breaks a square into 4 sub-squares, 
 *	initializes the new points 
 *	the recursive generation code is actully mine
 */
void HeightFieldGenSubdiv::processRSquare (int x1, int y1, int x2, int y2, 
					  float maxDelta)
{
	int 	cx = (x1+x2)/2, 	// center 
		cy = (y1+y2)/2, 
		dx = ABS (x2-x1),	// distance
		dy = ABS (y2-y1),
		pos;

	// square has reached minimum size; we're done
	if (dx<=1 && dy<=1) 
		return ;

	// we're in a normal square: process
	//printf ("Processing: %d, %d;  %d, %d\n", x1, y1, x2, y2);

	// center point
	p_HF->setEl(cx,cy, ( p_HF->getEl(x1,y1) + p_HF->getEl(x2,y1) + 
			     p_HF->getEl(x2,y2) + p_HF->getEl(x2,y1) ) / 4 ); 
	p_HF->setEl (cx,cy, p_HF->getEl(cx, cy)+(p_mGauss->gauss()*maxDelta));

	// process vertex points which haven't been touched yet
	pos = p_HF->Pos2Off(x1, cy);
	if (p_HF->getEl(pos) == EMPTY)
		{
		p_HF->setEl (pos, ( p_HF->getEl(x1,y1) + p_HF->getEl(x1,y2) ) / 2.0);
		p_HF->setEl (pos, p_HF->getEl(pos) + (p_mGauss->gauss()*maxDelta));
		}

	pos = p_HF->Pos2Off(cx, y1);
	if (p_HF->getEl(pos) == EMPTY)
		{
		p_HF->setEl (pos, ( p_HF->getEl(x1,y1) + p_HF->getEl(x2,y1) ) / 2.0);
		p_HF->setEl (pos, p_HF->getEl(pos) + (p_mGauss->gauss()*maxDelta));
		}

	pos = p_HF->Pos2Off(cx, y2);
	if (p_HF->getEl(pos) == EMPTY)
		{
		p_HF->setEl (pos, ( p_HF->getEl(x1,y2) + p_HF->getEl(x2,y2) ) / 2.0);
		p_HF->setEl (pos, p_HF->getEl(pos) + (p_mGauss->gauss()*maxDelta));
		}

	pos = p_HF->Pos2Off(x2, cy);
	if (p_HF->getEl(pos) == EMPTY)
		{
		p_HF->setEl (pos, ( p_HF->getEl(x2,y1) + p_HF->getEl(x2,y2) ) / 2.0);
		p_HF->setEl (pos, p_HF->getEl(pos) + (p_mGauss->gauss()*maxDelta));
		}

	//printf ("Values: %f: %f, %f, %f, %f\n", hf[p_HF->Pos2Off(cx, cy)], 
	//	hf[p_HF->Pos2Off(0, cy)], hf[p_HF->Pos2Off(cx,0)],
	//	hf[p_HF->Pos2Off(cx,y2)], hf[p_HF->Pos2Off(x2,cy)]);

	// process subsquares recursively
	processRSquare (x1, y1, cx, cy, maxDelta*d_RscaleFactor);
	processRSquare (cx, y1, x2, cy, maxDelta*d_RscaleFactor);
	processRSquare (x1, cy, cx, y2, maxDelta*d_RscaleFactor);
	processRSquare (cx, cy, x2, y2, maxDelta*d_RscaleFactor);
}

