/*
 * bltWinUtil.c --
 *
 *	This module contains WIN32 routines not included in the Tcl/Tk
 *	libraries.
 *
 * Copyright 1998 by Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no event
 * shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or performance
 * of this software.
 *
 */

#include <bltInt.h>
#include <X11/Xutil.h>
#include <X11/Xlib.h>

/*
 * Data structure for setting graphics context.
 */
typedef struct {
    int function;		/* logical operation */
    unsigned long plane_mask;	/* plane mask */
    unsigned long foreground;	/* foreground pixel */
    unsigned long background;	/* background pixel */
    int line_width;		/* line width */
    int line_style;		/* LineSolid, LineOnOffDash, LineDoubleDash */
    int cap_style;		/* CapNotLast, CapButt,
				   CapRound, CapProjecting */
    int join_style;		/* JoinMiter, JoinRound, JoinBevel */
    int fill_style;		/* FillSolid, FillTiled,
				   FillStippled, FillOpaeueStippled */
    int fill_rule;		/* EvenOddRule, WindingRule */
    int arc_mode;		/* ArcChord, ArcPieSlice */
    Pixmap tile;		/* tile pixmap for tiling operations */
    Pixmap stipple;		/* stipple 1 plane pixmap for stipping */
    int ts_x_origin;		/* offset for tile or stipple operations */
    int ts_y_origin;
    Font font;			/* default text font for text operations */
    int subwindow_mode;		/* ClipByChildren, IncludeInferiors */
    Bool graphics_exposures;	/* boolean, should exposures be generated */
    int clip_x_origin;		/* origin for clipping */
    int clip_y_origin;
    Pixmap clip_mask;		/* bitmap clipping; other calls for rects */
    int dash_offset;		/* patterned/dashed line information */
    char dashes;		/* Should be -1 to flag extended information */
    int nDashValues;
    char dashValues[12];
} XGCValuesEx;

static int tkpWinRopModes[] =
{
    R2_BLACK,			/* GXclear */
    R2_MASKPEN,			/* GXand */
    R2_MASKPENNOT,		/* GXandReverse */
    R2_COPYPEN,			/* GXcopy */
    R2_MASKNOTPEN,		/* GXandInverted */
    R2_NOT,			/* GXnoop */
    R2_XORPEN,			/* GXxor */
    R2_MERGEPEN,		/* GXor */
    R2_NOTMERGEPEN,		/* GXnor */
    R2_NOTXORPEN,		/* GXequiv */
    R2_NOT,			/* GXinvert */
    R2_MERGEPENNOT,		/* GXorReverse */
    R2_NOTCOPYPEN,		/* GXcopyInverted */
    R2_MERGENOTPEN,		/* GXorInverted */
    R2_NOTMASKPEN,		/* GXnand */
    R2_WHITE			/* GXset */
};
#define MASKPAT		0x00E20746 /* dest = (src & pat) | (!src & dst) */
#define COPYFG		0x00CA0749 /* dest = (pat & src) | (!pat & dst) */
#define COPYBG		0x00AC0744 /* dest = (!pat & src) | (pat & dst) */
/*
 * Translation table between X gc functions and Win32 BitBlt op modes.  Some
 * of the operations defined in X don't have names, so we have to construct
 * new opcodes for those functions.  This is arcane and probably not all that
 * useful, but at least it's accurate.
 */

#define NOTSRCAND	(DWORD)0x00220326	/* dest = (NOT source) AND dest */
#define NOTSRCINVERT	(DWORD)0x00990066	/* dest = (NOT source) XOR dest */
#define SRCORREVERSE	(DWORD)0x00DD0228	/* dest = source OR (NOT dest) */
#define SRCNAND		(DWORD)0x007700E6	/* dest = NOT (source AND dest) */


static int bltModes[] =
{
    BLACKNESS,			/* GXclear */
    SRCAND,			/* GXand */
    SRCERASE,			/* GXandReverse */
    SRCCOPY,			/* GXcopy */
    NOTSRCAND,			/* GXandInverted */
    PATCOPY,			/* GXnoop */
    SRCINVERT,			/* GXxor */
    SRCPAINT,			/* GXor */
    NOTSRCERASE,		/* GXnor */
    NOTSRCINVERT,		/* GXequiv */
    DSTINVERT,			/* GXinvert */
    SRCORREVERSE,		/* GXorReverse */
    NOTSRCCOPY,			/* GXcopyInverted */
    MERGEPAINT,			/* GXorInverted */
    SRCNAND,			/* GXnand */
    WHITENESS			/* GXset */
};

void *
Blt_EmulateCalloc(
    unsigned int nElems, 
    unsigned int sizeOfElem)
{
    void *array;
    unsigned int nBytes;

    nBytes = nElems * sizeOfElem;
    array = Tcl_Alloc(nBytes);
    memset(array, 0, nBytes);
    return array;
}

char *
Blt_EmulateStrdup(const char *string)
{
    unsigned int nBytes;
    char *array;

    nBytes = strlen(string) + 1;
    array = (char *)Tcl_Alloc(nBytes * sizeof(char));
    strcpy(array, string);
    return array;
}

double
drand48()
{
    return (double) rand() / (double)RAND_MAX;
}

void
srand48(seed)
    unsigned int seed;
{
    srand(seed);
}

int
Blt_GetPlatformId(void)
{
    static int platformId = 0;

    if (platformId == 0) {
	OSVERSIONINFO opsysInfo;

	opsysInfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	if (GetVersionEx(&opsysInfo)) {
	    platformId = opsysInfo.dwPlatformId;
	}
    }
    return platformId;
}

char *
Blt_LastError(void)
{
    static char buffer[1024];
    int length;

    FormatMessage(
	FORMAT_MESSAGE_FROM_SYSTEM,
	NULL,
	GetLastError(),
	MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),	/* Default language */
	buffer,
	1024,
	NULL);
    length = strlen(buffer);
    if (buffer[length - 2] == '\r') {
	buffer[length - 2] = '\0';
    }
    return buffer;
}

/*
 *----------------------------------------------------------------------
 *
 * XFree --
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXFree(void *ptr)
{
    free(ptr);
}

/*
 *----------------------------------------------------------------------
 *
 * XMaxRequestSize --
 *
 *----------------------------------------------------------------------
 */
long
Blt_EmulateXMaxRequestSize(Display *display)
{
    return USHRT_MAX;
}

/*
 *----------------------------------------------------------------------
 *
 * XLowerWindow --
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXLowerWindow(
    Display *display, 
    Window window)
{
    HWND hWnd;

    hWnd = Tk_GetHWND(window);
    display->request++;
    SetWindowPos(hWnd, HWND_BOTTOM, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
}

/*
 *----------------------------------------------------------------------
 *
 * XRaiseWindow --
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXRaiseWindow(
    Display *display, 
    Window window)
{
    HWND hWnd;

    hWnd = Tk_GetHWND(window);
    display->request++;
    SetWindowPos(hWnd, HWND_TOP, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE);
}

/*
 *----------------------------------------------------------------------
 *
 * XUnmapWindow --
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXUnmapWindow(
    Display *display, 
    Window window)
{
    HWND hWnd;

    hWnd = Tk_GetHWND(window);
    display->request++;
    ShowWindow(hWnd, SW_HIDE);
    /* SetWindowPos(hWnd, HWND_TOP, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE); */
}

/*
 *----------------------------------------------------------------------
 *
 * XWarpPointer --
 *
 *	If destWindow is None, moves the pointer by the offsets (destX,
 *	destY) relative to the current position of the pointer.
 *	If destWindow is a window, moves the pointer to the offsets
 *	(destX, destY) relative to the origin of destWindow.  However,
 *	if srcWindow is a window, the move only takes place if the window
 *	srcWindow contains the pointer and if the specified rectangle of
 *	srcWindow contains the pointer.
 *
 *	The srcX and srcY coordinates are relative to the origin of
 *	srcWindow.  If srcHeight is zero, it is replaced with the current
 *	height of srcWindow minus srcY.  If srcWidth is zero, it is
 *	replaced with the current width of srcWindow minus srcX.
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXWarpPointer(
    Display *display,
    Window srcWindow,
    Window destWindow,
    int srcX,
    int srcY,
    unsigned int srcWidth,
    unsigned int srcHeight,
    int destX,
    int destY)
{
    HWND hWnd;
    POINT point;

    hWnd = Tk_GetHWND(destWindow);
    point.x = destX, point.y = destY;
    if (ClientToScreen(hWnd, &point)) {
	SetCursorPos(point.x, point.y);
    }
}

static Tcl_HashTable gcTable;
static int gcInitialized = FALSE;

typedef struct DashInfo {
    HDC dc;
    int count;
    COLORREF color;
    int offset, nBits;
} DashInfo;

void
Blt_SetDashes(
    Display *display, 
    GC gc,
    Dashes *dashesPtr)
{
    register int i;
    XGCValuesEx *gcPtr = (XGCValuesEx *)gc;

    /* This must be used only with a privately created GC */
    assert((int)gcPtr->dashes == -1);
    gcPtr->nDashValues = dashesPtr->nValues;
    gcPtr->dash_offset = dashesPtr->offset;
    for (i = 0; i < dashesPtr->nValues; i++) {
	gcPtr->dashValues[i] = dashesPtr->valueArr[i];
    }
}

static int
GetDashInfo(
    HDC dc, 
    GC gc, 
    DashInfo *infoPtr)
{
    int dashOffset, dashValue;

    dashValue = 0;
    dashOffset = gc->dash_offset;
    if ((int)gc->dashes == -1) {
	XGCValuesEx *gcPtr = (XGCValuesEx *)gc;

	if (gcPtr->nDashValues == 1) {
	    dashValue = gcPtr->dashValues[0];
	}
    } else if (gc->dashes > 0) {
	dashValue = (int)gc->dashes;
    }
    if (dashValue == 0) {
	return FALSE;
    }
    infoPtr->dc = dc;
    infoPtr->nBits = dashValue;
    infoPtr->offset = dashOffset;
    infoPtr->count = 0;
    infoPtr->color = gc->foreground;
    return TRUE;
}

void
Blt_SetROP2(HDC dc, int function)
{
    SetROP2(dc, tkpWinRopModes[function]);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_EmulateXCreateGC --
 *
 *	Allocate a new extended GC, and initialize the specified fields.
 *
 * Results:
 *	Returns a newly allocated GC.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

GC
Blt_EmulateXCreateGC(
    Display *display,
    Drawable drawable,
    unsigned long mask,
    XGCValues *valuesPtr)
{
    XGCValuesEx *gcPtr;

    gcPtr = (XGCValuesEx *)malloc(sizeof(XGCValuesEx));
    if (gcPtr == NULL) {
	return None;
    }
    gcPtr->function = (mask & GCFunction)
	? valuesPtr->function : GXcopy;
    gcPtr->plane_mask = (mask & GCPlaneMask)
	? valuesPtr->plane_mask : ~0;
    gcPtr->foreground = (mask & GCForeground)
	? valuesPtr->foreground : 0;
    gcPtr->background = (mask & GCBackground)
	? valuesPtr->background : 0xffffff;
    gcPtr->line_width = (mask & GCLineWidth)
	? valuesPtr->line_width : 0;
    gcPtr->line_style = (mask & GCLineStyle)
	? valuesPtr->line_style : LineSolid;
    gcPtr->cap_style = (mask & GCCapStyle)
	? valuesPtr->cap_style : CapNotLast;
    gcPtr->join_style = (mask & GCJoinStyle)
	? valuesPtr->join_style : JoinMiter;
    gcPtr->fill_style = (mask & GCFillStyle)
	? valuesPtr->fill_style : FillSolid;
    gcPtr->fill_rule = (mask & GCFillRule)
	? valuesPtr->fill_rule : WindingRule;
    gcPtr->arc_mode = (mask & GCArcMode)
	? valuesPtr->arc_mode : ArcPieSlice;
    gcPtr->tile = (mask & GCTile)
	? valuesPtr->tile : None;
    gcPtr->stipple = (mask & GCStipple)
	? valuesPtr->stipple : None;
    gcPtr->ts_x_origin = (mask & GCTileStipXOrigin)
	? valuesPtr->ts_x_origin : 0;
    gcPtr->ts_y_origin = (mask & GCTileStipYOrigin)
	? valuesPtr->ts_y_origin : 0;
    gcPtr->font = (mask & GCFont)
	? valuesPtr->font : None;
    gcPtr->subwindow_mode = (mask & GCSubwindowMode)
	? valuesPtr->subwindow_mode : ClipByChildren;
    gcPtr->graphics_exposures = (mask & GCGraphicsExposures)
	? valuesPtr->graphics_exposures : True;
    gcPtr->clip_x_origin = (mask & GCClipXOrigin)
	? valuesPtr->clip_x_origin : 0;
    gcPtr->clip_y_origin = (mask & GCClipYOrigin)
	? valuesPtr->clip_y_origin : 0;
    gcPtr->dash_offset = (mask & GCDashOffset)
	? valuesPtr->dash_offset : 0;
    gcPtr->dashes = (mask & GCDashList)
	? valuesPtr->dashes : -1;	/* Mark that this an extended GC */
    gcPtr->nDashValues = 0;
    if (mask & GCClipMask) {
	struct ClipMask {
	    int type;		/* One of TKP_CLIP_PIXMAP or TKP_CLIP_REGION */
	    Pixmap pixmap;
	}       *clipPtr;
	clipPtr = (struct ClipMask *)malloc(sizeof(struct ClipMask));
#define TKP_CLIP_PIXMAP 0
	clipPtr->type = TKP_CLIP_PIXMAP;
	clipPtr->pixmap = valuesPtr->clip_mask;
	gcPtr->clip_mask = (Pixmap) clipPtr;
    } else {
	gcPtr->clip_mask = None;
    }
    return (GC)gcPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_GCToPen --
 *
 *	Set up the graphics port from the given GC.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The current port is adjusted.
 *
 *----------------------------------------------------------------------
 */
HPEN
Blt_GCToPen(dc, gc)
    HDC dc;
    GC gc;
{
    DWORD lineAttrs, lineStyle;
    DWORD dashArr[12];
    DWORD *dashPtr;
    int nValues, lineWidth;
    LOGBRUSH lBrush;
    HPEN pen;

    nValues = 0;
    lineWidth = (gc->line_width < 1) ? 1 : gc->line_width;
    if ((gc->line_style == LineOnOffDash) ||
	(gc->line_style == LineDoubleDash)) {
	XGCValuesEx *gcPtr = (XGCValuesEx *)gc;

	if ((int)gc->dashes == -1) {
	    register int i;

	    nValues = gcPtr->nDashValues;
	    for (i = 0; i < nValues; i++) {
		dashArr[i] = (DWORD) gcPtr->dashValues[i];
	    }
	    if (nValues == 1) {
		dashArr[1] = dashArr[0];
		nValues = 2;
	    }
	} else {
	    dashArr[1] = dashArr[0] = (DWORD) gc->dashes;
	    nValues = 2;
	    gc->dashes = -1;
	}
    }
    switch (nValues) {
    case 0:
	lineStyle = PS_SOLID;
	break;
    case 3:
	lineStyle = PS_DASHDOT;
	break;
    case 4:
	lineStyle = PS_DASHDOTDOT;
	break;
    case 2:
    default:
	/* PS_DASH style dash length is too long. */
	lineStyle = PS_DOT;
	break;
    }

    lBrush.lbStyle = BS_SOLID;
    lBrush.lbColor = gc->foreground;
    lBrush.lbHatch = 0;		/* Value is ignored with style is BS_SOLID */

    lineAttrs = 0;
    switch (gc->cap_style) {
    case CapNotLast:
    case CapButt:
	lineAttrs |= PS_ENDCAP_FLAT;
	break;
    case CapRound:
	lineAttrs |= PS_ENDCAP_ROUND;
	break;
    default:
	lineAttrs |= PS_ENDCAP_SQUARE;
	break;
    }
    switch (gc->join_style) {
    case JoinMiter:
	lineAttrs |= PS_JOIN_MITER;
	break;
    case JoinRound:
	lineAttrs |= PS_JOIN_ROUND;
	break;
    default:
	lineAttrs |= PS_JOIN_BEVEL;
	break;
    }
    pen = NULL;
    SetBkMode(dc, TRANSPARENT);
    if ((lineStyle == PS_SOLID) ||
	(Blt_GetPlatformId() == VER_PLATFORM_WIN32_NT)) {
	/*
	 * If the line style is solid or we're running on NT, first
	 * try to use a geometric pen.
	 */
	if (nValues > 0) {
	    lineStyle = PS_USERSTYLE;
	    dashPtr = dashArr;
	} else {
	    dashPtr = NULL;
	}
	pen = ExtCreatePen(PS_GEOMETRIC | lineAttrs | lineStyle, lineWidth,
	    &lBrush, nValues, dashPtr);
    } else {
	/* Windows 95: we'll sacrifice thick lines for dashes. */
	pen = ExtCreatePen(PS_COSMETIC | lineAttrs | lineStyle, 1, &lBrush,
	    0, NULL);
    }
    assert(pen != NULL);
    if (pen == NULL) {
	pen = CreatePen(PS_COSMETIC | PS_SOLID, 1, gc->foreground);
    }
    return pen;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_GetBitmapData --
 *
 *	Returns the DIB bits from a bitmap.
 *
 * Results:
 *	Returns a byte array of bitmap data or NULL if an error
 *	occurred.  The parameter pitchPtr returns the number
 *	of bytes per row.
 *
 *----------------------------------------------------------------------
 */
unsigned char *
Blt_GetBitmapData(
    Display *display,		/* Display of bitmap */
    Pixmap bitmap,		/* Bitmap to query */
    int width,			/* Width of bitmap */
    int height,			/* Height of bitmap */
    int *pitchPtr)
{				/* (out) Number of bytes per row */
    TkWinDCState state;
    HDC dc;
    int result;
    unsigned char *bits;
    unsigned int size;
    HBITMAP hBitmap;
    BITMAPINFOHEADER bi, *biPtr;
    HANDLE hMem, hMem2;
    int bytesPerRow;

    hBitmap = ((TkWinDrawable *)bitmap)->bitmap.handle;
    bi.biSize = sizeof(BITMAPINFOHEADER);
    bi.biPlanes = 1;
    bi.biBitCount = 1;
    bi.biCompression = BI_RGB;
    bi.biWidth = width;
    bi.biHeight = height;
    size = sizeof(BITMAPINFOHEADER) + 2 * sizeof(RGBQUAD);
    hMem = GlobalAlloc(GHND, size);
    biPtr = (BITMAPINFOHEADER *)GlobalLock(hMem);
    *biPtr = bi;
    dc = TkWinGetDrawableDC(display, bitmap, &state);
    result = GetDIBits(dc, hBitmap, 0, height, (LPVOID)NULL, 
	(BITMAPINFO *)biPtr, DIB_RGB_COLORS);
    TkWinReleaseDrawableDC(bitmap, dc, &state);
    if (!result) {
	GlobalUnlock(hMem);
	GlobalFree(hMem);
	return NULL;
    }
    bi = *biPtr;
    GlobalUnlock(hMem);
    bytesPerRow = ((width + 31) & ~31) / 8;
    if (bi.biSizeImage == 0) {
         bi.biSizeImage = bytesPerRow * height;
    }	
    bits = (unsigned char *)malloc(bi.biSizeImage);
    if (bits == NULL) {
	GlobalFree(hMem);
        return NULL;
    }
    hMem2 = GlobalReAlloc(hMem, size + bi.biSizeImage, 0);
    if (hMem2 == NULL) {
	GlobalFree(hMem);
        return NULL;
    }
    hMem = hMem2;
    biPtr = (LPBITMAPINFOHEADER)GlobalLock(hMem);
    dc = TkWinGetDrawableDC(display, bitmap, &state);
    result = GetDIBits(dc, hBitmap, 0, height, (unsigned char *)biPtr + size, 
        (BITMAPINFO *)biPtr, DIB_RGB_COLORS);
    TkWinReleaseDrawableDC(bitmap, dc, &state);
    if (!result) {
	OutputDebugString("GetDIBits failed\n");
	GlobalUnlock(hMem);
	GlobalFree(hMem);
	free((char *)bits);
	return NULL;
    }
    memcpy (bits, (unsigned char *)biPtr + size, bi.biSizeImage);
    *pitchPtr = bytesPerRow;
    GlobalUnlock(hMem);
    GlobalFree(hMem);
    return bits;
}

/*
 *----------------------------------------------------------------------
 *
 * CreateRotatedFont --
 *
 *	Creates a rotated copy of the given font.  This only works 
 *	for TrueType fonts.
 *
 * Results:
 *	Returns the newly create font or NULL if the font could not
 *	be created.
 *
 *----------------------------------------------------------------------
 */
HFONT
CreateRotatedFont(
    unsigned long fontId,	/* Font identifier (actually a Tk_Font) */
    double theta)
{				/* Number of degrees to rotate font */
    TkFontAttributes *faPtr;	/* Set of attributes to match. */
    TkFont *fontPtr;
    HFONT hFont;
    LOGFONT lf;

    fontPtr = (TkFont *) fontId;
    faPtr = &fontPtr->fa;
    ZeroMemory(&lf, sizeof(LOGFONT));
    lf.lfHeight = -faPtr->pointsize;
    if (lf.lfHeight < 0) {
	HDC dc;

	dc = GetDC(NULL);
	lf.lfHeight = -MulDiv(faPtr->pointsize,
	    GetDeviceCaps(dc, LOGPIXELSY), 72);
	ReleaseDC(NULL, dc);
    }
    lf.lfWidth = 0;
    lf.lfEscapement = lf.lfOrientation = ROUND(theta * 10.0);
#define TK_FW_NORMAL	0
    lf.lfWeight = (faPtr->weight == TK_FW_NORMAL) ? FW_NORMAL : FW_BOLD;
    lf.lfItalic = faPtr->slant;
    lf.lfUnderline = faPtr->underline;
    lf.lfStrikeOut = faPtr->overstrike;
    lf.lfCharSet = DEFAULT_CHARSET;
    lf.lfOutPrecision = OUT_TT_ONLY_PRECIS;
    lf.lfClipPrecision = CLIP_DEFAULT_PRECIS;
    lf.lfQuality = DEFAULT_QUALITY;
    lf.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;

    if (faPtr->family == NULL) {
	lf.lfFaceName[0] = '\0';
    } else {
#if HAVE_UTF
	if (Blt_GetPlatformId() == VER_PLATFORM_WIN32_NT) {
	    Tcl_Encoding encoding;
	    Tcl_DString dString;

	    Tcl_DStringInit(&dString);
	    encoding = Tcl_GetEncoding(NULL, "unicode");
	    Tcl_UtfToExternalDString(encoding, faPtr->family, -1, &dString);
	    strcpy(lf.lfFaceName, Tcl_DStringValue(&dString));
	    Tcl_DStringFree(&dString);
	} else {
	    strcpy(lf.lfFaceName, faPtr->family);
	}
#else
	strcpy(lf.lfFaceName, faPtr->family);
#endif
    }
    /*
     * Replace the standard X and Mac family names with the names that
     * MS Windows likes.
     */
    if ((strcasecmp(lf.lfFaceName, "Times") == 0)
	|| (strcasecmp(lf.lfFaceName, "New York") == 0)) {
	strcpy(lf.lfFaceName, "Times New Roman");
    } else if ((strcasecmp(lf.lfFaceName, "Courier") == 0)
	|| (strcasecmp(lf.lfFaceName, "Monaco") == 0)) {
	strcpy(lf.lfFaceName, "Courier New");
    } else if ((strcasecmp(lf.lfFaceName, "Helvetica") == 0)
	|| (strcasecmp(lf.lfFaceName, "Geneva") == 0)) {
	strcpy(lf.lfFaceName, "Arial");
    }
    hFont = CreateFontIndirect(&lf);
    if (hFont == NULL) {
#ifdef WINDEBUG
	PurifyPrintf("can't create font: %s\n", Blt_LastError());
#endif
	return NULL;
    }
    return hFont;
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawRectangles --
 *
 *       Draws the outlines of the specified rectangles as if a
 *       five-point PolyLine protocol request were specified for each
 *       rectangle:
 *
 *             [x,y] [x+width,y] [x+width,y+height] [x,y+height]
 *             [x,y]
 *
 *      For the specified rectangles, these functions do not draw a
 *      pixel more than once.  XDrawRectangles draws the rectangles in
 *      the order listed in the array.  If rectangles intersect, the
 *      intersecting pixels are drawn multiple times.  Draws a
 *      rectangle.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws rectangles on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXDrawRectangles(
    Display *display,
    Drawable drawable,
    GC gc,
    XRectangle *rectArr,
    int nRects)
{
    HPEN pen, oldPen;
    TkWinDCState state;
    HBRUSH brush, oldBrush;
    HDC dc;
    register XRectangle *rectPtr;
    register int i;

    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    pen = Blt_GCToPen(dc, gc);
    brush = GetStockObject(NULL_BRUSH);
    oldPen = SelectPen(dc, pen);
    oldBrush = SelectBrush(dc, brush);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    rectPtr = rectArr;
    for (i = 0; i < nRects; i++, rectPtr++) {
	Rectangle(dc, (int)rectPtr->x, (int)rectPtr->y,
	    (int)(rectPtr->x + rectPtr->width + 1),
	    (int)(rectPtr->y + rectPtr->height + 1));
    }
    DeletePen(SelectPen(dc, oldPen));
    DeleteBrush(SelectBrush(dc, oldBrush));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

#ifdef notdef
/*
 * Implements the "pixeling" of small arcs, because GDI-performance
 * for this is awful
 * was made especially for BLT, graph4 demo now runs 4x faster
 *
 */
/* O-outer , I-inner, B-both */
#define NEITHER_ 0
#define OUTLINE 1
#define FILL 2
#define BOTH (OUTLINE|FILL)
#define MINIARCS 5
static int arcus0[1] =
{
    BOTH
};
static int arcus1[4] =
{
    BOTH, BOTH,
    BOTH, BOTH
};

static int arcus2[9] =
{
    NEITHER, OUTLINE, NEITHER,
    OUTLINE, FILL, OUTLINE,
    NEITHER, OUTLINE, NEITHER
};

static int arcus3[16] =
{
    NEITHER, OUTLINE, OUTLINE, NEITHER,
    OUTLINE, FILL, FILL, OUTLINE,
    OUTLINE, FILL, FILL, OUTLINE,
    NEITHER, OUTLINE, OUTLINE, NEITHER
};

static int arcus4[25] =
{
    NEITHER, OUTLINE, OUTLINE, OUTLINE, NEITHER,
    OUTLINE, FILL, FILL, FILL, OUTLINE,
    OUTLINE, FILL, FILL, FILL, OUTLINE,
    OUTLINE, FILL, FILL, FILL, OUTLINE,
    NEITHER, OUTLINE, OUTLINE, OUTLINE, NEITHER
};

static int *arcis[MINIARCS] =
{
    arcus0, arcus1, arcus2, arcus3, arcus4
};

static void
DrawMiniArc(
    HDC dc,
    int width,
    int x,
    int y,
    int mask,
    COLORREF inner,
    COLORREF outer)
{
    int *arc;
    int i, j;

    if (width > MINIARCS) {
	return;
    }
    arc = arcis[width];
    for (i = 0; i <= width; i++) {
	for (j = 0; j <= width; j++) {
	    bit = (mask & *arc);
	    if (bit & OUTLINE) {
		SetPixelV(dc, x + i, y + j, outer);
	    } else if (bit & FILL) {
		SetPixelV(dc, x + i, y + j, inner);
	    }
	    arc++;
	}
    }
}

#endif

/*
 *----------------------------------------------------------------------
 *
 * DrawArc --
 *
 *	This procedure handles the rendering of drawn or filled
 *	arcs and chords.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Renders the requested arcs.
 *
 *----------------------------------------------------------------------
 */
static void
DrawArc(dc, arcMode, arcPtr, pen, brush)
    HDC dc;
    int arcMode;		/* Mode: either ArcChord or ArcPieSlice */
    XArc *arcPtr;
    HPEN pen;
    HBRUSH brush;
{
    int start, extent, clockwise;
    int xstart, ystart, xend, yend;
    double radian_start, radian_end, xr, yr;
    double dx, dy;

    if ((arcPtr->angle1 == 0) && (arcPtr->angle2 == 23040)) {
	/* Handle special case of circle or ellipse */
	Ellipse(dc, arcPtr->x, arcPtr->y, arcPtr->x + arcPtr->width + 1,
	    arcPtr->y + arcPtr->height + 1);
	return;
    }
    start = arcPtr->angle1, extent = arcPtr->angle2;
    clockwise = (extent < 0);	/* Non-zero if clockwise */

    /*
     * Compute the absolute starting and ending angles in normalized radians.
     * Swap the start and end if drawing clockwise.
     */
    start = start % (64 * 360);
    if (start < 0) {
	start += (64 * 360);
    }
    extent = (start + extent) % (64 * 360);
    if (extent < 0) {
	extent += (64 * 360);
    }
    if (clockwise) {
	int tmp = start;
	start = extent;
	extent = tmp;
    }
#define XAngleToRadians(a) ((double)(a) / 64 * M_PI / 180);
    radian_start = XAngleToRadians(start);
    radian_end = XAngleToRadians(extent);

    /*
     * Now compute points on the radial lines that define the starting and
     * ending angles.  Be sure to take into account that the y-coordinate
     * system is inverted.
     */
    dx = arcPtr->width * 0.5;
    dy = arcPtr->height * 0.5;

    xr = arcPtr->x + dx;
    yr = arcPtr->y + dy;
    xstart = (int)((xr + cos(radian_start) * dx) + 0.5);
    ystart = (int)((yr + sin(-radian_start) * dy) + 0.5);
    xend = (int)((xr + cos(radian_end) * dx) + 0.5);
    yend = (int)((yr + sin(-radian_end) * dy) + 0.5);

    /*
     * Now draw a filled or open figure.  Note that we have to
     * increase the size of the bounding box by one to account for the
     * difference in pixel definitions between X and Windows.
     */

    if (brush == 0) {
	/*
	 * Note that this call will leave a gap of one pixel at the
	 * end of the arc for thin arcs.  We can't use ArcTo because
	 * it's only supported under Windows NT.
	 */
	Arc(dc, arcPtr->x, arcPtr->y, arcPtr->x + arcPtr->width + 1,
	    arcPtr->y + arcPtr->height + 1, xstart, ystart, xend, yend);
	/* FIXME: */
    } else {
	if (arcMode == ArcChord) {
	    Chord(dc, arcPtr->x, arcPtr->y, arcPtr->x + arcPtr->width + 1,
		arcPtr->y + arcPtr->height + 1, xstart, ystart, xend, yend);
	} else if (arcMode == ArcPieSlice) {
	    Pie(dc, arcPtr->x, arcPtr->y, arcPtr->x + arcPtr->width + 1,
		arcPtr->y + arcPtr->height + 1, xstart, ystart, xend, yend);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawArcs --
 *
 *	Draws multiple circular or elliptical arcs.  Each arc is
 *	specified by a rectangle and two angles.  The center of the
 *	circle or ellipse is the center of the rect- angle, and the
 *	major and minor axes are specified by the width and height.
 *	Positive angles indicate counterclock- wise motion, and
 *	negative angles indicate clockwise motion.  If the magnitude
 *	of angle2 is greater than 360 degrees, XDrawArcs truncates it
 *	to 360 degrees.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws an arc for each array element on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXDrawArcs(
    Display *display,
    Drawable drawable,
    GC gc,
    XArc *arcArr,
    int nArcs)
{
    HPEN pen, oldPen;
    HBRUSH brush, oldBrush;
    register int i;
    HDC dc;
    TkWinDCState state;
    register XArc *arcPtr;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    pen = Blt_GCToPen(dc, gc);
    oldPen = SelectPen(dc, pen);
    brush = GetStockBrush(NULL_BRUSH);
    oldBrush = SelectBrush(dc, brush);
    for (arcPtr = arcArr, i = 0; i < nArcs; i++, arcPtr++) {
	DrawArc(dc, gc->arc_mode, arcPtr, pen, 0);
    }
    DeleteBrush(SelectBrush(dc, oldBrush));
    DeletePen(SelectPen(dc, oldPen));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * XFillArcs --
 *
 *	Draw a filled arc.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws a filled arc for each array element on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXFillArcs(
    Display *display,
    Drawable drawable,
    GC gc,
    XArc *arcArr,
    int nArcs)
{
    HBRUSH brush, oldBrush;
    HPEN pen, oldPen;
    register int i;
    HDC dc;
    register XArc *arcPtr;
    TkWinDCState state;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    pen = Blt_GCToPen(dc, gc);
    oldPen = SelectPen(dc, pen);
    brush = CreateSolidBrush(gc->foreground);
    oldBrush = SelectBrush(dc, brush);
    for (arcPtr = arcArr, i = 0; i < nArcs; i++, arcPtr++) {
	DrawArc(dc, gc->arc_mode, arcPtr, pen, brush);
    }
    DeleteBrush(SelectBrush(dc, oldBrush));
    DeletePen(SelectPen(dc, oldPen));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawLines --
 *
 *	Draw connected lines.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Renders a series of connected lines.
 *
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 *
 * ConvertPoints --
 *
 *	Convert an array of X points to an array of Win32 points.
 *
 * Results:
 *	Returns the converted array of POINTs.
 *
 * Side effects:
 *	Allocates a block of memory that should not be freed.
 *
 *----------------------------------------------------------------------
 */

static POINT *
ConvertPoints(points, npoints, mode, bbox)
    XPoint *points;
    int npoints;
    int mode;			/* CoordModeOrigin or CoordModePrevious. */
    RECT *bbox;			/* Bounding box of points. */
{
    static POINT *winPoints = NULL;	/* Array of points that is reused. */
    static int nWinPoints = -1;	/* Current size of point array. */
    int i;

    /*
     * To avoid paying the cost of a malloc on every drawing routine,
     * we reuse the last array if it is large enough.
     */

    if (npoints > nWinPoints) {
	if (winPoints != NULL) {
	    ckfree((char *)winPoints);
	}
	winPoints = (POINT *) malloc(sizeof(POINT) * npoints);
	if (winPoints == NULL) {
	    nWinPoints = -1;
	    return NULL;
	}
	nWinPoints = npoints;
    }
    bbox->left = bbox->right = points[0].x;
    bbox->top = bbox->bottom = points[0].y;

    if (mode == CoordModeOrigin) {
	for (i = 0; i < npoints; i++) {
	    winPoints[i].x = points[i].x;
	    winPoints[i].y = points[i].y;
	    bbox->left = MIN(bbox->left, winPoints[i].x);
	    bbox->right = MAX(bbox->right, winPoints[i].x);
	    bbox->top = MIN(bbox->top, winPoints[i].y);
	    bbox->bottom = MAX(bbox->bottom, winPoints[i].y);
	}
    } else {
	winPoints[0].x = points[0].x;
	winPoints[0].y = points[0].y;
	for (i = 1; i < npoints; i++) {
	    winPoints[i].x = winPoints[i - 1].x + points[i].x;
	    winPoints[i].y = winPoints[i - 1].y + points[i].y;
	    bbox->left = MIN(bbox->left, winPoints[i].x);
	    bbox->right = MAX(bbox->right, winPoints[i].x);
	    bbox->top = MIN(bbox->top, winPoints[i].y);
	    bbox->bottom = MAX(bbox->bottom, winPoints[i].y);
	}
    }
    return winPoints;
}


static void CALLBACK
DrawDot(
    int x, int y,		/* Coordinates of point */
    LPARAM clientData)
{				/* Line information */
    DashInfo *infoPtr = (DashInfo *) clientData;
    int count;

    infoPtr->count++;
    count = (infoPtr->count + infoPtr->offset) / infoPtr->nBits;
    if (count & 0x1) {
	SetPixelV(infoPtr->dc, x, y, infoPtr->color);
    }
}


void
Blt_EmulateXDrawLines(
    Display *display,
    Drawable drawable,
    GC gc,
    XPoint *pointArr,
    int nPoints,
    int mode)
{
    TkWinDCState state;
    HDC dc;

    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    if (gc->line_style != LineSolid) {
	/* Handle dotted lines specially */
	register XPoint *p1, *p2;
	register int i;
	DashInfo info;

	if (!GetDashInfo(dc, gc, &info)) {
	    goto solidLine;
	}
	p1 = pointArr;
	p2 = p1 + 1;
	for (i = 1; i < nPoints; i++, p1++, p2++) {
	    LineDDA(p1->x, p1->y, p2->x, p2->y, DrawDot, (LPARAM) & info);
	}
    } else {
	RECT rect;
	HPEN pen, oldPen;
	HBRUSH brush, oldBrush;
	POINT *pts;

      solidLine:
	pen = Blt_GCToPen(dc, gc);
	oldPen = SelectPen(dc, pen);
	brush = CreateSolidBrush(gc->foreground);
	oldBrush = SelectBrush(dc, brush);
	SetROP2(dc, tkpWinRopModes[gc->function]);
	pts = ConvertPoints(pointArr, nPoints, mode, &rect);
	Polyline(dc, pts, nPoints);
	DeletePen(SelectPen(dc, oldPen));
	DeleteBrush(SelectBrush(dc, oldBrush));
    }
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

void
Blt_EmulateXDrawLine(
    Display *display,
    Drawable drawable,
    GC gc,
    int x1, int y1,
    int x2, int y2)
{
    TkWinDCState state;
    HDC dc;

    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    if (gc->line_style != LineSolid) {
	/* Handle dotted lines specially */
	DashInfo info;

	if (!GetDashInfo(dc, gc, &info)) {
	    goto solidLine;
	}
	LineDDA(x1, y1, x2, y2, DrawDot, (LPARAM) & info);
    } else {
	HPEN pen, oldPen;
	HBRUSH brush, oldBrush;

      solidLine:
	pen = Blt_GCToPen(dc, gc);
	oldPen = SelectPen(dc, pen);
	brush = CreateSolidBrush(gc->foreground);
	oldBrush = SelectBrush(dc, brush);
	SetROP2(dc, tkpWinRopModes[gc->function]);
	MoveToEx(dc, x1, y1, (LPPOINT) NULL);
	LineTo(dc, x2, y2);
	DeletePen(SelectPen(dc, oldPen));
	DeleteBrush(SelectBrush(dc, oldBrush));
    }
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawSegments --
 *
 *	Draws multiple, unconnected lines. For each segment, draws a
 *	line between (x1, y1) and (x2, y2).  It draws the lines in the
 *	order listed in the array of XSegment structures and does not
 *	perform joining at coincident endpoints.  For any given line,
 *	does not draw a pixel more than once. If lines intersect, the
 *	intersecting pixels are drawn multiple times.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws unconnected line segments on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXDrawSegments(
    Display *display,
    Drawable drawable,
    GC gc,
    XSegment *segArr,
    int nSegments)
{
    register int i;
    HDC dc;
    register XSegment *segPtr;
    TkWinDCState state;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    if (gc->line_style != LineSolid) {
	/* Handle dotted lines specially */
	DashInfo info;

	if (!GetDashInfo(dc, gc, &info)) {
	    goto solidLine;
	}
	for (segPtr = segArr, i = 0; i < nSegments; i++, segPtr++) {
	    info.count = 0;
	    LineDDA(segPtr->x1, segPtr->y1, segPtr->x2, segPtr->y2,
		DrawDot, (LPARAM)&info);
	}
    } else {
	HPEN pen, oldPen;

      solidLine:
	pen = Blt_GCToPen(dc, gc);
	oldPen = SelectPen(dc, pen);
	for (segPtr = segArr, i = 0; i < nSegments; i++, segPtr++) {
	    MoveToEx(dc, segPtr->x1, segPtr->y1, (LPPOINT) NULL);
	    LineTo(dc, segPtr->x2, segPtr->y2);
	}
	DeletePen(SelectPen(dc, oldPen));
    }
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawRectangle --
 *
 *       Draws the outlines of the specified rectangle as if a
 *       five-point PolyLine protocol request were specified for each
 *       rectangle:
 *
 *             [x,y] [x+width,y] [x+width,y+height] [x,y+height]
 *             [x,y]
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws a rectangle on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXDrawRectangle(
    Display *display,
    Drawable drawable,
    GC gc,
    int x, int y,
    unsigned int width,
    unsigned int height)
{
    TkWinDCState state;
    HPEN pen, oldPen;
    HBRUSH brush, oldBrush;
    HDC dc;

    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    pen = Blt_GCToPen(dc, gc);
    brush = GetStockObject(NULL_BRUSH);
    oldPen = SelectPen(dc, pen);
    oldBrush = SelectBrush(dc, brush);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    if (gc->line_style != LineSolid) {
	/* Handle dotted lines specially */
	register int x2, y2;
	DashInfo info;

	if (!GetDashInfo(dc, gc, &info)) {
	    goto solidLine;
	}
	x2 = x + width;
	y2 = y + height;
	LineDDA(x, y, x2, y, DrawDot, (LPARAM)&info);
	LineDDA(x2, y, x2, y2, DrawDot, (LPARAM)&info);
	LineDDA(x2, y2, x, y2, DrawDot, (LPARAM)&info);
	LineDDA(x, y2, x, y, DrawDot, (LPARAM)&info);
    } else {
      solidLine:
	Rectangle(dc, x, y, x + width + 1, y + height + 1);
    }
    DeletePen(SelectPen(dc, oldPen));
    DeleteBrush(SelectBrush(dc, oldBrush));
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_EmulateXDrawPoints --
 *
 *	Uses the foreground pixel and function components of the GC to
 *	draw a multiple points into the specified drawable.
 *      CoordModeOrigin treats all coordinates as relative to the
 *	origin, and CoordModePrevious treats all coordinates after
 *	the first as relative to the previous point.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws points on the specified drawable.
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXDrawPoints(
    Display *display,
    Drawable drawable,
    GC gc,
    XPoint *pointArr,
    int nPoints,
    int mode)
{				/* Ignored. CoordModeOrigin is assumed. */
    register int i;
    HDC dc;
    register XPoint *pointPtr;
    TkWinDCState state;

    display->request++;
    if (drawable == None) {
	return;
    }
    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, tkpWinRopModes[gc->function]);
    for (pointPtr = pointArr, i = 0; i < nPoints; i++, pointPtr++) {
	SetPixelV(dc, pointPtr->x, pointPtr->y, gc->foreground);
    }
    TkWinReleaseDrawableDC(drawable, dc, &state);
}

/*
 *----------------------------------------------------------------------
 *
 * XReparentWindow --
 *
 *	If the specified window is mapped, automatically performs an
 *	UnmapWindow request on it, removes it from its current
 *	position in the hierarchy, and inserts it as the child of the
 *	specified parent.  The window is placed in the stacking order
 *	on top with respect to sibling windows.
 *
 *	Note: In WIN32 you can't reparent to/from another application.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Reparents the specified window.
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXReparentWindow(
    Display *display,
    Window window,
    Window parent,
    int x,
    int y)
{
    HWND child, newParent;

    child = Tk_GetHWND(window);
    newParent = Tk_GetHWND(parent);
    SetParent(child, newParent);
    SetWindowLong(child, GWL_STYLE, WS_CHILD | WS_CLIPCHILDREN |
	WS_CLIPSIBLINGS);

    XMoveWindow(display, window, x, y);
    XRaiseWindow(display, window);
    XMapWindow(display, window);
}

void
Blt_EmulateXSetDashes(
    Display *display,
    GC gc,
    int dashOffset,
    _Xconst char *dashList,
    int n)
{
    gc->dashes = (unsigned char)strlen(dashList);
    gc->dash_offset = (int)dashList;
}

/*
 *----------------------------------------------------------------------
 *
 * XDrawString --
 *
 *	Draw a single string in the current font.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Renders the specified string in the drawable.
 *
 *----------------------------------------------------------------------
 */

void
Blt_EmulateXDrawString(
    Display *display,
    Drawable drawable,
    GC gc,
    int x,
    int y,
    _Xconst char *string,
    int length)
{
    if (drawable == None) {
	return;
    }
    Tk_DrawChars(display, drawable, gc, (Tk_Font)gc->font, string,
	length, x, y);
}

static void
TileArea(
    HDC destDC, 
    HDC srcDC, 
    int xTileOrigin, 
    int yTileOrigin,
    int tileWidth,
    int tileHeight,
    int x, 
    int y,
    int width,
    int height)
{
    int destX, destY;
    int destWidth, destHeight;
    int srcX, srcY;
    int xOrigin, yOrigin;
    int delta;
    int left, top, right, bottom;

    xOrigin = x, yOrigin = y;
    if (x < xTileOrigin) {
	delta = (xTileOrigin - x) % tileWidth;
	if (delta > 0) {
	    xOrigin -= (tileWidth - delta);
	}
    } else if (x > xTileOrigin) {
	delta = (x - xTileOrigin) % tileWidth;
	if (delta > 0) {
	    xOrigin -= delta;
	}
    }
    if (y < yTileOrigin) {
	delta = (yTileOrigin - y) % tileHeight;
	if (delta > 0) {
	    yOrigin -= (tileHeight - delta);
	}
    } else if (y >= yTileOrigin) {
	delta = (y - yTileOrigin) % tileHeight;
	if (delta > 0) {
	    yOrigin -= delta;
	}
    }
#ifdef notdef
    PurifyPrintf("tile is (%d,%d,%d,%d)\n", xTileOrigin, yTileOrigin, 
		 tileWidth, tileHeight);
    PurifyPrintf("region is (%d,%d,%d,%d)\n", x, y, width, height);
    PurifyPrintf("starting at %d,%d\n", xOrigin, yOrigin);
#endif
    left = x;
    right = x + width;
    top = y;
    bottom = y + height;
    for (y = yOrigin; y < bottom; y += tileHeight) {
	srcY = 0;
	destY = y;
	destHeight = tileHeight;
	if (y < top) {
	    srcY = (top - y);
	    destHeight = tileHeight - srcY;
	    destY = top;
	} 
	if ((destY + destHeight) > bottom) {
	    destHeight = (bottom - destY);
	}
	for (x = xOrigin; x < right; x += tileWidth) {
	    srcX = 0;
	    destX = x;
	    destWidth = tileWidth;
	    if (x < left) {
		srcX = (left - x);
		destWidth = tileWidth - srcX;
		destX = left;
	    } 
	    if ((destX + destWidth) > right) {
		destWidth = (right - destX);
	    }
#ifdef notdef
	    PurifyPrintf("drawing pattern (%d,%d,%d,%d) at %d,%d\n",
			 srcX , srcY, destWidth, destHeight, destX, destY);
#endif
	    BitBlt(destDC, destX, destY, destWidth, destHeight, 
		srcDC, srcX, srcY, SRCCOPY);
	}
    }
}


/*
 *----------------------------------------------------------------------
 *
 * XFillRectangles --
 *
 *	Fill multiple rectangular areas in the given drawable.
 *	Handles tiling.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws onto the specified drawable.
 *
 *----------------------------------------------------------------------
 */

void
Blt_EmulateXFillRectangles(
    Display *display,
    Drawable drawable,
    GC gc,
    XRectangle *rectArr,
    int nRectangles)
{
    HDC hDC;
    int i;
    RECT rect;
    TkWinDCState state;
    TkWinDrawable *twdPtr;
    HBRUSH oldBrush, hFgBrush, hBgBrush, hBrush;
    HBITMAP oldBitmap, hBitmap;
    HDC memDC;
    BITMAP bm;
    
    if (drawable == None) {
	return;
    }
    hDC = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(hDC, tkpWinRopModes[gc->function]);

    switch(gc->fill_style) {
    case FillTiled:
	if (gc->tile == None) {
	    goto fillSolid;
        }
	if ((GetDeviceCaps(hDC, RASTERCAPS) & RC_BITBLT) == 0) {
	    goto fillSolid;
	}
        twdPtr = (TkWinDrawable *)gc->tile;
	GetObject(twdPtr->bitmap.handle, sizeof(BITMAP), &bm);
	memDC = CreateCompatibleDC(hDC);
	oldBitmap = SelectBitmap(memDC, twdPtr->bitmap.handle);
        for (i = 0; i < nRectangles; i++) {
	    TileArea(hDC, memDC, gc->ts_x_origin, gc->ts_y_origin, bm.bmWidth, 
		bm.bmHeight, (int)rectArr[i].x, (int)rectArr[i].y, 
		(int)rectArr[i].width, (int)rectArr[i].height);
        }
	SelectBitmap(memDC, oldBitmap);
	DeleteDC(memDC);
        break; 

    case FillOpaqueStippled:
    case FillStippled:
	if (gc->stipple == None) {
	    goto fillSolid;
	}
        twdPtr = (TkWinDrawable *)gc->stipple;
	if (twdPtr->type != TWD_BITMAP) {
	    panic("unexpected drawable type in stipple");
	}
	hBrush = CreatePatternBrush(twdPtr->bitmap.handle);
	SetBrushOrgEx(hDC, gc->ts_x_origin, gc->ts_y_origin, NULL);
	oldBrush = SelectBrush(hDC, hBrush);
	memDC = CreateCompatibleDC(hDC);

	/*
	 * For each rectangle, create a drawing surface which is the size of
	 * the rectangle and fill it with the background color.  Then merge the
	 * result with the stipple pattern.
	 */
        hFgBrush = CreateSolidBrush(gc->foreground);
	hBgBrush = CreateSolidBrush(gc->background);
	for (i = 0; i < nRectangles; i++) {
	    hBitmap = CreateCompatibleBitmap(hDC, rectArr[i].width,
		    rectArr[i].height);
	    oldBitmap = SelectObject(memDC, hBitmap);
	    rect.left = rect.top = 0;
	    rect.right = rectArr[i].width;
	    rect.bottom = rectArr[i].height;
	    FillRect(memDC, &rect, hFgBrush);
	    BitBlt(hDC, rectArr[i].x, rectArr[i].y, rectArr[i].width,
		 rectArr[i].height, memDC, 0, 0, COPYBG);
	    if (gc->fill_style == FillOpaqueStippled) {
		FillRect(memDC, &rect, hBgBrush);
		BitBlt(hDC, rectArr[i].x, rectArr[i].y, rectArr[i].width, 
			rectArr[i].height, memDC, 0, 0, COPYFG);
	    }
	    SelectObject(memDC, oldBitmap);
	    DeleteObject(hBitmap);
	}
	DeleteBrush(hFgBrush);
	DeleteBrush(hBgBrush);
	DeleteDC(memDC);
	SelectBrush(hDC, oldBrush);
	DeleteBrush(hBrush);
 	break;

    case FillSolid:
	fillSolid:
	memDC = CreateCompatibleDC(hDC);
        hFgBrush = CreateSolidBrush(gc->foreground);
	for (i = 0; i < nRectangles; i++) {
	    hBitmap = CreateCompatibleBitmap(hDC, rectArr[i].width,
		    rectArr[i].height);
	    oldBitmap = SelectObject(memDC, hBitmap);
	    rect.left = rect.top = 0;
	    rect.right = rectArr[i].width;
	    rect.bottom = rectArr[i].height;
	    FillRect(memDC, &rect, hFgBrush);
	    BitBlt(hDC, rectArr[i].x, rectArr[i].y, rectArr[i].width,
		 rectArr[i].height, memDC, 0, 0, SRCCOPY);
	    SelectObject(memDC, oldBitmap);
	    DeleteObject(hBitmap);
	}
	DeleteBrush(hFgBrush);
	DeleteDC(memDC);
 	break;
    }
    TkWinReleaseDrawableDC(drawable, hDC, &state);
}

void
Blt_EmulateXFillRectangle(
    Display *display,
    Drawable drawable,
    GC gc,
    int x,
    int y,
    unsigned int width,
    unsigned int height)
{
    HDC hDC;
    RECT rect;
    TkWinDCState state;
    TkWinDrawable *twdPtr;
    HBRUSH oldBrush, hFgBrush, hBgBrush, hBrush;
    HBITMAP oldBitmap, hBitmap;
    HDC memDC;
    BITMAP bm;

    if (drawable == None) {
	return;
    }
    hDC = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(hDC, tkpWinRopModes[gc->function]);
    rect.left = rect.top = 0;
    rect.right = width;
    rect.bottom = height;

    switch(gc->fill_style) {
    case FillTiled:
	if (gc->tile == None) { 
	    goto fillSolid;
	}
        if ((GetDeviceCaps(hDC, RASTERCAPS) & RC_BITBLT) == 0) {
	    goto fillSolid;
	}
        twdPtr = (TkWinDrawable *)gc->tile;
	GetObject(twdPtr->bitmap.handle, sizeof(BITMAP), &bm);
	memDC = CreateCompatibleDC(hDC);
	oldBitmap = SelectBitmap(memDC, twdPtr->bitmap.handle);
	TileArea(hDC, memDC, gc->ts_x_origin, gc->ts_y_origin, bm.bmWidth, 
		bm.bmHeight, x, y, width, height);
	SelectBitmap(memDC, oldBitmap);
	DeleteDC(memDC);
        break; 

    case FillOpaqueStippled:
    case FillStippled:
	if (gc->stipple == None) {
	    goto fillSolid;
	}
        twdPtr = (TkWinDrawable *)gc->stipple;
	if (twdPtr->type != TWD_BITMAP) {
	    panic("unexpected drawable type in stipple");
	}
	hBrush = CreatePatternBrush(twdPtr->bitmap.handle);
	SetBrushOrgEx(hDC, gc->ts_x_origin, gc->ts_y_origin, NULL);
	oldBrush = SelectBrush(hDC, hBrush);
	memDC = CreateCompatibleDC(hDC);

	hFgBrush = CreateSolidBrush(gc->foreground);
	hBgBrush = CreateSolidBrush(gc->background);
	hBitmap = CreateCompatibleBitmap(hDC, width, height);
	oldBitmap = SelectObject(memDC, hBitmap);
	FillRect(memDC, &rect, hFgBrush);
        SetBkMode(hDC, TRANSPARENT);
	BitBlt(hDC, x, y, width, height, memDC, 0, 0, COPYFG);
	if (gc->fill_style == FillOpaqueStippled) {
	    FillRect(memDC, &rect, hBgBrush);
	    BitBlt(hDC, x, y, width, height, memDC, 0, 0, COPYBG);
	}
	DeleteBrush(hFgBrush);
	DeleteBrush(hBgBrush);

	SelectBitmap(memDC, oldBitmap);
	DeleteBitmap(hBitmap);
	DeleteDC(memDC);
	SelectBrush(hDC, oldBrush);
 	break;

    case FillSolid:
      fillSolid:
	/* TkWinFillRect(hDC, x, y, width, height, gc->foreground);  */
	memDC = CreateCompatibleDC(hDC);
        hFgBrush = CreateSolidBrush(gc->foreground);
	hBitmap = CreateCompatibleBitmap(hDC, width, height);
	oldBitmap = SelectBitmap(memDC, hBitmap);
	rect.left = rect.top = 0;
	rect.right = width;
	rect.bottom = height;
	FillRect(memDC, &rect, hFgBrush);
	BitBlt(hDC, x, y, width, height, memDC, 0, 0, SRCCOPY);
	SelectObject(memDC, oldBitmap);
	DeleteBitmap(hBitmap);
	DeleteBrush(hFgBrush);
	DeleteDC(memDC);
 	break;
    }
    TkWinReleaseDrawableDC(drawable, hDC, &state);
}

int
Blt_DrawRotatedText(
    Display *display,
    Drawable drawable,
    int x, int y,
    double theta,
    TextStyle *stylePtr,
    TextLayout *layoutPtr)
{
    HFONT hFont, oldFont;
    TkWinDCState state;
    TEXTMETRIC tm;
    HDC dc;
    int isTT;

    dc = TkWinGetDrawableDC(display, drawable, &state);
    SetROP2(dc, stylePtr->gc->function);
    hFont = CreateRotatedFont(stylePtr->gc->font, theta);
    if (hFont == NULL) {
	return FALSE;
    }
    oldFont = SelectFont(dc, hFont);
    isTT = ((GetTextMetrics(dc, &tm)) && (tm.tmPitchAndFamily & TMPF_TRUETYPE));
    if (isTT) {
	int bbWidth, bbHeight;
	double sinTheta, cosTheta;
	Point2D p, s, center;
	register TextSegment *segPtr;
	register int i;

	Blt_GetBoundingBox(layoutPtr->width, layoutPtr->height, theta,
	    &bbWidth, &bbHeight, (XPoint *)NULL);
	Blt_TranslateAnchor(x, y, bbWidth, bbHeight, stylePtr->anchor, &x, &y);
	center.x = (double)layoutPtr->width * -0.5;
	center.y = (double)layoutPtr->height * -0.5;
	theta = (-theta / 180.0) * M_PI;
	sinTheta = sin(theta), cosTheta = cos(theta);
	for (segPtr = layoutPtr->segArr, i = 0; i < layoutPtr->nSegments;
	    i++, segPtr++) {
	    p.x = center.x + (double)segPtr->x;
	    p.y = center.y + (double)segPtr->y;
	    s.x = x + (p.x * cosTheta) - (p.y * sinTheta) + (bbWidth * 0.5);
	    s.y = y + (p.x * sinTheta) + (p.y * cosTheta) + (bbHeight * 0.5);
	    segPtr->sx = ROUND(s.x);
	    segPtr->sy = ROUND(s.y);
	}
	SetBkMode(dc, TRANSPARENT);
	SetTextAlign(dc, TA_LEFT | TA_BASELINE);
	if (stylePtr->state & (STATE_DISABLED | STATE_EMPHASIS)) {
	    TkBorder *borderPtr = (TkBorder *) stylePtr->border;
	    XColor *color1, *color2;

	    color1 = borderPtr->lightColor, color2 = borderPtr->darkColor;
	    if (stylePtr->state & STATE_EMPHASIS) {
		XColor *hold;

		hold = color1, color1 = color2, color2 = hold;
	    }
	    if (color1 != NULL) {
		SetTextColor(dc, color1->pixel);
		for (segPtr = layoutPtr->segArr, i = 0;
		    i < layoutPtr->nSegments; i++, segPtr++) {
		    TextOut(dc, segPtr->sx, segPtr->sy, segPtr->text,
			segPtr->count);
		}
	    }
	    if (color2 != NULL) {
		SetTextColor(dc, color2->pixel);
		for (segPtr = layoutPtr->segArr, i = 0;
		    i < layoutPtr->nSegments; i++, segPtr++) {
		    TextOut(dc, segPtr->sx + 1, segPtr->sy + 1, segPtr->text,
			segPtr->count);
		}
	    }
	    goto done;		/* Done */
	}
	if ((stylePtr->shadow.offset > 0) && (stylePtr->shadow.color != NULL)) {
	    SetTextColor(dc, stylePtr->shadow.color->pixel);
	    for (segPtr = layoutPtr->segArr, i = 0;
		i < layoutPtr->nSegments; i++, segPtr++) {
		TextOut(dc, segPtr->sx + stylePtr->shadow.offset,
		    segPtr->sy + stylePtr->shadow.offset,
		    segPtr->text, segPtr->count);
	    }
	}
	SetTextColor(dc, stylePtr->color->pixel);
	for (segPtr = layoutPtr->segArr, i = 0; i < layoutPtr->nSegments;
	    i++, segPtr++) {
	    TextOut(dc, segPtr->sx, segPtr->sy, segPtr->text, segPtr->count);
	}
    }
  done:
    SelectFont(dc, oldFont);
    DeleteFont(hFont);
    TkWinReleaseDrawableDC(drawable, dc, &state);
    return isTT;
}


static void
DrawPixel(HDC hDC, int x, int y, COLORREF color)
{
    HDC memDC;
    HBRUSH hFgBrush;
    HBITMAP oldBitmap, hBitmap;
    RECT rect;
    int size;
 
	size=10;
    memDC = CreateCompatibleDC(hDC);
    hFgBrush = CreateSolidBrush(color);
    hBitmap = CreateCompatibleBitmap(hDC, size, size);
    oldBitmap = SelectObject(memDC, hBitmap);
    rect.left = rect.top = 0;
    rect.right = rect.bottom = size;
    FillRect(memDC, &rect, hFgBrush);
    BitBlt(hDC, x, y, size, size, memDC, 0, 0, SRCCOPY);
    SelectObject(memDC, oldBitmap);
    DeleteObject(hBitmap);
    DeleteBrush(hFgBrush);
    DeleteDC(memDC);
}

/*
 *----------------------------------------------------------------------
 *
 * PixelateBitmap --
 *
 *	Draws a masked bitmap in given device (should be printer)
 *	context.  Bit operations on print devices usually fail because
 *	there's no way to read back from the device surface to get the
 *	previous pixel values, rendering BitBlt useless. The bandaid
 *	solution here is to draw 1x1 pixel rectangles at each
 *	coordinate as directed by the the mask and source bitmaps.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Draws onto the specified drawable.
 *
 *----------------------------------------------------------------------
 */
static void
PixelateBitmap(
    Display *display,
    Drawable drawable,
    Pixmap srcBitmap,
    Pixmap maskBitmap,
    int width,
    int height,
    GC gc,
    int destX,
    int destY)
{
    register int x, y;
    register int dx, dy;
    int pixel;
    unsigned char *srcBits;
    register unsigned char *srcPtr;
    int bitPos, bytesPerRow;
    HDC hDC;
    TkWinDCState state;

    srcBits = Blt_GetBitmapData(display, srcBitmap, width, height, 
	&bytesPerRow);
    if (srcBits == NULL) {
	return;
    }
    hDC = TkWinGetDrawableDC(display, drawable, &state);
    if (maskBitmap != None) {
        register unsigned char *maskPtr;
        unsigned char *maskBits;
        maskBits = Blt_GetBitmapData(display, maskBitmap, width, height,
	    &bytesPerRow);
        bytesPerRow = ((width + 31) & ~31) / 8;
        for (dy = destY, y = height - 1; y >= 0; y--, dy++) {
	    maskPtr = maskBits + (bytesPerRow * y);
	    srcPtr = srcBits + (bytesPerRow * y);
	    for (dx = destX, x = 0; x < width; x++, dx++) {
	        bitPos = x % 8;
	        pixel = (*maskPtr & (0x80 >> bitPos));
	        if (pixel) {
		    pixel = (*srcPtr & (0x80 >> bitPos));
		    DrawPixel(hDC, dx, dy, 
		        (pixel) ? gc->foreground : gc->background);
	        }
	        if (bitPos == 7) {
		    srcPtr++, maskPtr++;
	        }
	    }			/* x */
        }
        free((char *)maskBits);
    } else {
        bytesPerRow = ((width + 31) & ~31) / 8;
        for (dy = destY, y = height - 1; y >= 0; y--, dy++) {
	    srcPtr = srcBits + (bytesPerRow * y);
	    for (dx = destX, x = 0; x < width; x++, dx++) {
	        bitPos = x % 8;
		pixel = (*srcPtr & (0x80 >> bitPos));
		DrawPixel(hDC, dx, dy, 
		        (pixel) ? gc->foreground : gc->background);
	        if (bitPos == 7) {
		    srcPtr++;
	        }
	    }
	}
    }
    TkWinReleaseDrawableDC(drawable, hDC, &state);
    free((char *)srcBits);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_EmulateXCopyPlane --
 *
 *	Copies a bitmap from a source drawable to a destination
 *	drawable.  The plane argument specifies which bit plane of
 *	the source contains the bitmap.  Note that this implementation
 *	ignores the gc->function.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Changes the destination drawable.
 *
 *----------------------------------------------------------------------
 */
void
Blt_EmulateXCopyPlane(
    Display *display,
    Drawable src,
    Drawable dest,
    GC gc,
    int srcX,
    int srcY,
    unsigned int width,
    unsigned int height,
    int destX,
    int destY,
    unsigned long plane)
{
    HDC srcDC, destDC;
    TkWinDCState srcState, destState;
    HBRUSH bgBrush, fgBrush, oldBrush;
    TkpClipMask *clipPtr = (TkpClipMask *) gc->clip_mask;

    display->request++;

    if (plane != 1) {
	panic("Unexpected plane specified for XCopyPlane");
    }
    srcDC = TkWinGetDrawableDC(display, src, &srcState);

    if (src != dest) {
	destDC = TkWinGetDrawableDC(display, dest, &destState);
    } else {
	destDC = srcDC;
    }

    if ((clipPtr == NULL) || (clipPtr->type == TKP_CLIP_REGION)) {

	/*
	 * Case 1: opaque bitmaps.  Windows handles the conversion
	 * from one bit to multiple bits by setting 0 to the
	 * foreground color, and 1 to the background color (seems
	 * backwards, but there you are).
	 */

	if (clipPtr && clipPtr->type == TKP_CLIP_REGION) {
	    SelectClipRgn(destDC, (HRGN) clipPtr->value.region);
	    OffsetClipRgn(destDC, gc->clip_x_origin, gc->clip_y_origin);
	}
	SetBkMode(destDC, OPAQUE);
	SetBkColor(destDC, gc->foreground);
	SetTextColor(destDC, gc->background);
	BitBlt(destDC, destX, destY, width, height, srcDC, srcX, srcY,
	    SRCCOPY);

	SelectClipRgn(destDC, NULL);
    } else if (clipPtr->type == TKP_CLIP_PIXMAP) {
	if (clipPtr->value.pixmap == src) {
	    /*
	     * Case 2: transparent bitmaps are handled by setting the
	     * destination to the foreground color whenever the source
	     * pixel is set.
	     */
	    if ((GetDeviceCaps(destDC, RASTERCAPS) & RC_BITBLT) == 0) {
		PixelateBitmap(display, dest, src, None, width, height, gc,
		    destX, destY);
	    } else {
		fgBrush = CreateSolidBrush(gc->foreground);
		oldBrush = SelectObject(destDC, fgBrush);
		BitBlt(destDC, destX, destY, width, height, srcDC,
		    srcX, srcY, MASKPAT);
		SelectObject(destDC, oldBrush);
		DeleteObject(fgBrush);
	    }
	} else {

	    /*
	     * Case 3: two arbitrary bitmaps.  Copy the source rectangle
	     * into a color pixmap.  Use the result as a brush when
	     * copying the clip mask into the destination.
	     */

	    HDC memDC, maskDC;
	    HBITMAP bitmap;
	    TkWinDCState maskState;

#ifdef WINDEBUG
	    PurifyPrintf("devcaps %x\n", GetDeviceCaps(destDC, TECHNOLOGY));
#endif
	    if ((GetDeviceCaps(destDC, RASTERCAPS) & RC_BITBLT) == 0) {
		PixelateBitmap(display, dest, src, clipPtr->value.pixmap,
		    width, height, gc, destX, destY);
	    } else {
		fgBrush = CreateSolidBrush(gc->foreground);
		bgBrush = CreateSolidBrush(gc->background);
		maskDC = TkWinGetDrawableDC(display, clipPtr->value.pixmap,
		    &maskState);
		memDC = CreateCompatibleDC(destDC);
		bitmap = CreateBitmap(width, height, 1, 1, NULL);
		SelectObject(memDC, bitmap);

		/*
		 * Set foreground bits.  We create a new bitmap containing
		 * (source AND mask), then use it to set the foreground color
		 * into the destination.
		 */

		BitBlt(memDC, 0, 0, width, height, srcDC, srcX, srcY, SRCCOPY);
		BitBlt(memDC, 0, 0, width, height, maskDC,
		    destX - gc->clip_x_origin, destY - gc->clip_y_origin,
		    SRCAND);
		oldBrush = SelectObject(destDC, fgBrush);
		BitBlt(destDC, destX, destY, width, height, memDC, 0, 0,
		    MASKPAT);

		/*
		 * Set background bits.  Same as foreground, except we use
		 * ((NOT source) AND mask) and the background brush.
		 */

		BitBlt(memDC, 0, 0, width, height, srcDC, srcX, srcY,
		    NOTSRCCOPY);
		BitBlt(memDC, 0, 0, width, height, maskDC,
		    destX - gc->clip_x_origin, destY - gc->clip_y_origin,
		    SRCAND);
		SelectObject(destDC, bgBrush);
		BitBlt(destDC, destX, destY, width, height, memDC, 0, 0,
		    MASKPAT);
		TkWinReleaseDrawableDC(clipPtr->value.pixmap, maskDC,
		    &maskState);
		SelectObject(destDC, oldBrush);
		DeleteDC(memDC);
		DeleteObject(bitmap);
		DeleteObject(fgBrush);
		DeleteObject(bgBrush);
	    }
	}
    }
    if (src != dest) {
	TkWinReleaseDrawableDC(dest, destDC, &destState);
    }
    TkWinReleaseDrawableDC(src, srcDC, &srcState);
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_EmulateXCopyArea --
 *
 *	Copies data from one drawable to another using block transfer
 *	routines.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Data is moved from a window or bitmap to a second window or
 *	bitmap.
 *
 *----------------------------------------------------------------------
 */

void
Blt_EmulateXCopyArea(
    Display *display,
    Drawable src,
    Drawable dest,
    GC gc,
    int srcX,
    int srcY,
    unsigned int width,
    unsigned int height,
    int destX,
    int destY)
{
    HDC srcDC, destDC;
    TkWinDCState srcState, destState;
    TkpClipMask *clipPtr = (TkpClipMask *) gc->clip_mask;
    POINT pts[2];

    srcDC = TkWinGetDrawableDC(display, src, &srcState);
    if (src != dest) {
	destDC = TkWinGetDrawableDC(display, dest, &destState);
    } else {
	destDC = srcDC;
    }

    if (clipPtr && clipPtr->type == TKP_CLIP_REGION) {
	SelectClipRgn(destDC, (HRGN) clipPtr->value.region);
	OffsetClipRgn(destDC, gc->clip_x_origin, gc->clip_y_origin);
    }
    pts[0].x = destX, pts[0].y = destY;
    pts[1].x = destX + width, pts[1].y = destY + height;
    LPtoDP(destDC, pts, 2);
    BitBlt(destDC, pts[0].x, pts[0].y, pts[1].x - pts[0].x,
	pts[1].y - pts[0].y, srcDC, srcX, srcY,
	bltModes[gc->function]);

    SelectClipRgn(destDC, NULL);

    if (src != dest) {
	TkWinReleaseDrawableDC(dest, destDC, &destState);
    }
    TkWinReleaseDrawableDC(src, srcDC, &srcState);
}
