/********************************************************************************
*                                                                               *
*                           S t r i n g   O b j e c t                           *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXString.h,v 1.2 1999/11/11 23:44:00 jeroen Exp $                        *
********************************************************************************/
#ifndef FXSTRING_H
#define FXSTRING_H



/**
  FXString provides essential string manipulation capabilities to FOX.
*/

/// A string class
class FXAPI FXString {
private:
  FXchar* str;      // String data
public:
  
  /// Create empty string
  FXString();
  
  /// Copy construct
  FXString(const FXString& s);

  /// Construct and init
  FXString(const FXchar* s);

  /// Construct and init with substring
  FXString(const FXchar* s,FXint n);

  /// Construct and fill with constant
  FXString(FXchar c,FXint n);
  
  /// Construct string from two parts
  FXString(const FXchar *s1,const FXchar* s2);
  
  /// Size to some desired capacity
  void size(FXint sz);

  /// Size of text data
  FXint size() const;

  /// Get text contents
  const FXchar* text() const { return (const FXchar*)str; }

  /// Get text contents (non-null)
  const FXchar* value() const { return (const FXchar*)(str?str:""); }

  /// See if string is empty
  FXbool empty() const { return !str || !str[0]; }

  /// Easy indexing
  FXchar& operator[](FXint i){ return str[i]; }
  const FXchar& operator[](FXint i) const { return str[i]; }

  /// Assignment
  FXString& operator=(const FXString& s);
  FXString& operator=(const FXchar* s);

  /// Length of text only
  FXint length() const;

  /// Fill with a constant
  FXString& fill(FXchar c,FXint len);
  
  /// Fill up to current length
  FXString& fill(FXchar c);
  
  /// Convert to lower case
  FXString& lower();
  
  /// Convert to upper case
  FXString& upper();
  
  /// Extract partition of delimiter separated string
  FXString extract(FXint part,FXchar delim) const;
  FXString extract(FXint part,FXchar delim,FXchar esc) const;

  /// Insert string
  FXString& insert(FXint pos,const FXString& s);
  FXString& insert(FXint pos,const FXchar* s);
  FXString& insert(FXint pos,const FXchar* s,FXint n);
  
  /// Prepend string
  FXString& prepend(const FXString& s);
  FXString& prepend(const FXchar *s);
  FXString& prepend(const FXchar *s,FXint n);
  
  /// Append string
  FXString& append(const FXString& s);
  FXString& append(const FXchar *s);
  FXString& append(const FXchar *s,FXint n);
  
  /// Replace part of string
  FXString& replace(FXint pos,FXint len,const FXString& s);
  FXString& replace(FXint pos,FXint len,const FXchar *s);
  FXString& replace(FXint pos,FXint len,const FXchar *s,FXint n);
  
  /// Remove substring
  FXString& remove(FXint pos,FXint n=1);
  
  /// Remove leading and trailing whitespace
  FXString& trim();
  
  /// Remove leading whitespace
  FXString& trimBegin();
  
  /// Remove trailing whitespace
  FXString& trimEnd();
  
  /// Truncate string at pos
  FXString& trunc(FXint pos);

  /// Clear
  FXString& clear();

  /// Get leftmost part
  FXString left(FXint n) const;
  
  /// Get rightmost part
  FXString right(FXint n) const;
  
  /// Get some part in the middle
  FXString mid(FXint pos,FXint n) const;
  
  /// Return all characters before the nth occurrence of ch,
  /// or entire string if not found
  FXString before(FXchar ch,FXint n=1) const;
  
  /// Return all characters after the nth occurrence of ch,
  /// or empty string if not found
  FXString after(FXchar ch,FXint n=1) const;
  
  /// Find a character, searching forward; return position or -1
  FXint findf(FXchar c,FXint pos=0) const;
  
  /// Find a character, searching backward; return position or -1
  FXint findb(FXchar c,FXint pos=2147483647) const;
  
  /// Find number of occurances of character in string
  FXint count(FXchar c) const;
  
  /// Format a string a-la printf
  FXString& format(const char *fmt,...);
  
  /// Get hash value
  FXint hash() const;
  
  /// Equality up to a length len
  friend FXAPI FXbool equal(const FXString &s1,const FXString &s2,FXint len=2147483647);
  friend FXAPI FXbool equal(const FXString &s1,const char *s2,FXint len=2147483647);
  friend FXAPI FXbool equal(const FXchar *s1,const FXString &s2,FXint len=2147483647);
  
  /// Equality
  friend FXAPI FXbool operator==(const FXString &s1,const FXString &s2);
  friend FXAPI FXbool operator==(const FXString &s1,const char *s2);
  friend FXAPI FXbool operator==(const FXchar *s1,const FXString &s2);
  
  /// Inequality
  friend FXAPI FXbool operator!=(const FXString &s1,const FXString &s2);
  friend FXAPI FXbool operator!=(const FXString &s1,const FXchar *s2);
  friend FXAPI FXbool operator!=(const FXchar *s1,const FXString &s2);
  
  /// Less than
  friend FXAPI FXbool operator<(const FXString &s1,const FXString &s2);
  friend FXAPI FXbool operator<(const FXString &s1,const FXchar *s2);
  friend FXAPI FXbool operator<(const FXchar *s1,const FXString &s2);
  
  /// Less than or equal
  friend FXAPI FXbool operator<=(const FXString &s1,const FXString &s2);
  friend FXAPI FXbool operator<=(const FXString &s1,const FXchar *s2);
  friend FXAPI FXbool operator<=(const FXchar *s1,const FXString &s2); 
  
  /// Greater than
  friend FXAPI FXbool operator>(const FXString &s1,const FXString &s2);
  friend FXAPI FXbool operator>(const FXString &s1,const FXchar *s2);
  friend FXAPI FXbool operator>(const FXchar *s1,const FXString &s2);
  
  /// Greater than or equal
  friend FXAPI FXbool operator>=(const FXString &s1,const FXString &s2);
  friend FXAPI FXbool operator>=(const FXString &s1,const FXchar *s2);
  friend FXAPI FXbool operator>=(const FXchar *s1,const FXString &s2);

  /// Append
  FXString& operator+=(const FXString& s);
  FXString& operator+=(const FXchar* s);

  /// Concatenation operators
  friend FXAPI FXString operator+(const FXString& s1,const FXString& s2);
  friend FXAPI FXString operator+(const FXString& s1,const FXchar* s2);
  friend FXAPI FXString operator+(const FXchar* s1,const FXString& s2);

  /// Saving and loading
  friend FXAPI FXStream& operator<<(FXStream& store,const FXString& s);
  friend FXAPI FXStream& operator>>(FXStream& store,FXString& s);

  /// Format a string a-la printf
  friend FXAPI FXString FXStringFormat(const FXchar *fmt,...);
  
  /// Conversion of numbers to strings
  friend FXAPI FXString FXStringVal(FXint num);
  friend FXAPI FXString FXStringVal(FXuint num);
  friend FXAPI FXString FXStringVal(FXfloat num,FXint prec=6,FXExponent exp=EXP_AUTO);
  friend FXAPI FXString FXStringVal(FXdouble num,FXint prec=6,FXExponent exp=EXP_AUTO);
  
  /// Conversion of string to numbers
  friend FXAPI FXint FXIntVal(const FXString& s);
  friend FXAPI FXuint FXUIntVal(const FXString& s);
  friend FXAPI FXfloat FXFloatVal(const FXString& s);
  friend FXAPI FXdouble FXDoubleVal(const FXString& s);

  /// Delete
 ~FXString();
  };


#endif
