/********************************************************************************
*                                                                               *
*                       M e n u   T i t l e   W i d g e t                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXMenuTitle.cpp,v 1.4 1999/10/20 03:39:09 jeroen Exp $                   *
********************************************************************************/
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXGIFIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXComposite.h"
#include "FXShell.h"
#include "FXPopup.h"
#include "FXMenuPane.h"
#include "FXMenuCaption.h"
#include "FXMenuTitle.h"

/*
  Notes:
  - Accelerators.
  - Help text from constructor is third part; second part should be
    accelerator key combination.
  - When menu label changes, hotkey might have to be adjusted.
  - Fix it so menu stays up when after Alt-F, you press Alt-E.
  - MenuItems should be derived from FXLabel.
  - Look into SEL_FOCUS_SELF some more...
*/


/*******************************************************************************/

// Map
FXDEFMAP(FXMenuTitle) FXMenuTitleMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXMenuTitle::onPaint),
  FXMAPFUNC(SEL_ENTER,0,FXMenuTitle::onEnter),
  FXMAPFUNC(SEL_LEAVE,0,FXMenuTitle::onLeave),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXMenuTitle::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXMenuTitle::onLeftBtnRelease),
  FXMAPFUNC(SEL_MIDDLEBUTTONPRESS,0,FXMenuTitle::onDefault),
  FXMAPFUNC(SEL_MIDDLEBUTTONRELEASE,0,FXMenuTitle::onDefault),
  FXMAPFUNC(SEL_RIGHTBUTTONPRESS,0,FXMenuTitle::onDefault),
  FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,0,FXMenuTitle::onDefault),
  FXMAPFUNC(SEL_KEYPRESS,0,FXMenuTitle::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXMenuTitle::onKeyRelease),
  FXMAPFUNC(SEL_KEYPRESS,FXWindow::ID_HOTKEY,FXMenuTitle::onHotKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,FXWindow::ID_HOTKEY,FXMenuTitle::onHotKeyRelease),
  FXMAPFUNC(SEL_FOCUS_UP,0,FXMenuTitle::onFocusUp),
  FXMAPFUNC(SEL_FOCUS_DOWN,0,FXMenuTitle::onFocusDown),
  FXMAPFUNC(SEL_FOCUSIN,0,FXMenuTitle::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXMenuTitle::onFocusOut),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_POST,FXMenuTitle::onCmdPost),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_UNPOST,FXMenuTitle::onCmdUnpost),
  };

  
// Object implementation
FXIMPLEMENT(FXMenuTitle,FXMenuCaption,FXMenuTitleMap,ARRAYNUMBER(FXMenuTitleMap))



// Make a menu title button
FXMenuTitle::FXMenuTitle(FXComposite* p,const FXString& text,FXIcon* ic,FXMenuPane* pup,FXuint opts):
  FXMenuCaption(p,text,ic,opts){
  flags|=FLAG_ENABLED;
  options&=~LAYOUT_FILL_X;
  pane=pup;
  }


// Create window
void FXMenuTitle::create(){
  FXMenuCaption::create();
  if(pane) pane->create();
  }


// Create window
void FXMenuTitle::detach(){
  FXMenuCaption::detach();
  if(pane) pane->detach();
  }


// Get default width
FXint FXMenuTitle::getDefaultWidth(){
  FXint tw,iw,w;
  tw=iw=w=0;
  if(!label.empty()) tw=font->getTextWidth(label.text(),label.length());
  if(icon) iw=icon->getWidth()+5;
  w=tw+iw;
  if(iw && tw) w+=5;
  return 12+w;
  }


// Get default height
FXint FXMenuTitle::getDefaultHeight(){
  FXint th,ih;
  th=ih=0;
  if(!label.empty()) th=font->getFontHeight()+5;
  if(icon) ih=icon->getHeight()+5;
  return FXMAX(th,ih);
  }


// If window can have focus
FXbool FXMenuTitle::canFocus() const { 
  return 1; 
  }


// When pressed, perform ungrab, then process normally
long FXMenuTitle::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    if(flags&FLAG_ACTIVE){
      handle(this,MKUINT(ID_UNPOST,SEL_COMMAND),NULL);
      }
    else{
      handle(this,MKUINT(ID_POST,SEL_COMMAND),NULL);
      }
    return 1;
    }
  return 0;
  }


// Released left button
long FXMenuTitle::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(isEnabled()){
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    if(event->moved){
      handle(this,MKUINT(ID_UNPOST,SEL_COMMAND),ptr);
      }
    return 1;
    }
  return 0;
  }


// Keyboard press; forward to menu pane
long FXMenuTitle::onKeyPress(FXObject*,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(isEnabled()){
    FXTRACE((200,"%s::onKeyPress %08x keysym=0x%04x state=%04x\n",getClassName(),this,event->code,event->state));
    if(target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr)) return 1;
    if(pane && pane->shown() && pane->handle(pane,sel,ptr)) return 1;
    }
  return 0;
  }


// Keyboard release; forward to menu pane
long FXMenuTitle::onKeyRelease(FXObject*,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(isEnabled()){
    FXTRACE((200,"%s::onKeyRelease %08x keysym=0x%04x state=%04x\n",getClassName(),this,event->code,event->state));
    if(target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr)) return 1;
    if(pane && pane->shown() && pane->handle(pane,sel,ptr)) return 1;
    }
  return 0;
  }


// Hot key combination pressed
long FXMenuTitle::onHotKeyPress(FXObject*,FXSelector,void* ptr){
  FXTRACE((200,"%s::onHotKeyPress %08x\n",getClassName(),this));
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    }
  return 1;
  }


// Hot key combination released
long FXMenuTitle::onHotKeyRelease(FXObject*,FXSelector,void*){
  FXTRACE((200,"%s::onHotKeyRelease %08x\n",getClassName(),this));
  if(isEnabled()){ 
    if(flags&FLAG_ACTIVE){
      handle(this,MKUINT(ID_UNPOST,SEL_COMMAND),NULL);
      }
    else{
      handle(this,MKUINT(ID_POST,SEL_COMMAND),NULL);
      }
    }
  return 1;
  }


// Post the menu
long FXMenuTitle::onCmdPost(FXObject*,FXSelector,void*){
  FXint x,y;
  if(pane && !pane->shown()){
    translateCoordinatesTo(x,y,getRoot(),0,0);
    pane->popup(getParent(),x-1,y+height);
    if(!getParent()->grabbed()) getParent()->grab();
    }
  flags|=FLAG_ACTIVE;
  update();
  return 1;
  }


// Unpost the menu
long FXMenuTitle::onCmdUnpost(FXObject*,FXSelector,void*){
  if(pane && pane->shown()){
    pane->popdown();
    if(getParent()->grabbed()) getParent()->ungrab();
    }
  flags&=~FLAG_ACTIVE;
  update();
  return 1;
  }


// Focus moved down
long FXMenuTitle::onFocusDown(FXObject*,FXSelector,void*){
  if(pane && !pane->shown()){
    handle(this,MKUINT(ID_POST,SEL_COMMAND),NULL);
    return 1;
    }
  return 0;
  }


// Focus moved up
long FXMenuTitle::onFocusUp(FXObject*,FXSelector,void*){
  if(pane && pane->shown()){
    handle(this,MKUINT(ID_UNPOST,SEL_COMMAND),NULL);
    return 1;
    }
  return 0;
  }


// Into focus chain
void FXMenuTitle::setFocus(){
  FXWindow *menuitem=getParent()->getFocus();
  FXbool active=menuitem && menuitem->isActive();
  FXMenuCaption::setFocus();
  if(active) handle(this,MKUINT(ID_POST,SEL_COMMAND),NULL);
  }
    


// Out of focus chain
void FXMenuTitle::killFocus(){
  FXMenuCaption::killFocus();
  handle(this,MKUINT(ID_UNPOST,SEL_COMMAND),NULL);
  }


// Enter
long FXMenuTitle::onEnter(FXObject* sender,FXSelector sel,void* ptr){
  FXMenuCaption::onEnter(sender,sel,ptr);
  if(isEnabled() && canFocus() && getParent()->getFocus()) setFocus();
  return 1;
  }


// Leave
long FXMenuTitle::onLeave(FXObject* sender,FXSelector sel,void* ptr){
  FXMenuCaption::onLeave(sender,sel,ptr);
  return 1;
  }


// Handle repaint 
long FXMenuTitle::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent *ev=(FXEvent*)ptr;
  FXDCWindow dc(this,ev);
  FXint xx,yy;
  if(isActive()){
    dc.setForeground(selbackColor);
    dc.fillRectangle(0,0,width,height);
    xx=6;
    if(icon){
      dc.drawIcon(icon,5,(height-icon->getHeight())/2);
      xx=5+5+icon->getWidth();
      }
    if(!label.empty()){
      yy=font->getFontAscent()+(height-font->getFontHeight())/2;
      dc.setTextFont(font);
      dc.setForeground(seltextColor);
      dc.drawText(xx,yy,label.text(),label.length());
      if(0<=hotoff) dc.drawLine(xx+font->getTextWidth(label.text(),hotoff),yy+1,xx+font->getTextWidth(label.text(),hotoff+1)-1,yy+1);
      }
    }
  else if(isEnabled()){
    dc.setForeground(backColor);
    dc.fillRectangle(0,0,width,height);
    xx=6;
    if(icon){
      dc.drawIcon(icon,5,(height-icon->getHeight())/2);
      xx=5+5+icon->getWidth();
      }
    if(!label.empty()){
      yy=font->getFontAscent()+(height-font->getFontHeight())/2;
      dc.setTextFont(font);
      dc.setForeground(textColor);
      dc.drawText(xx,yy,label.text(),label.length());
      if(0<=hotoff) dc.drawLine(xx+font->getTextWidth(label.text(),hotoff),yy+1,xx+font->getTextWidth(label.text(),hotoff+1)-1,yy+1);
      }
    }
  else{
    dc.setForeground(backColor);
    dc.fillRectangle(0,0,width,height);
    xx=6;
    if(icon){
      dc.drawIconShaded(icon,5,(height-icon->getHeight())/2);
      xx=5+5+icon->getWidth();
      }
    if(!label.empty()){
      yy=font->getFontAscent()+(height-font->getFontHeight())/2;
      dc.setTextFont(font);
      dc.setForeground(hiliteColor);
      dc.drawText(xx+1,yy-1,label.text(),label.length());
      dc.setForeground(shadowColor);
      dc.drawText(xx,yy,label.text(),label.length());
      if(0<=hotoff) dc.drawLine(xx+font->getTextWidth(label.text(),hotoff),yy+1,xx+font->getTextWidth(label.text(),hotoff+1)-1,yy+1);
      }
    }
  return 1;
  }


// Gained focus
long FXMenuTitle::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXMenuCaption::onFocusIn(sender,sel,ptr);
  update();
  return 1;
  }

  
// Lost focus
long FXMenuTitle::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXMenuCaption::onFocusOut(sender,sel,ptr);
  update();
  return 1;
  }


// Test if logically inside
FXbool FXMenuTitle::contains(FXint parentx,FXint parenty) const {
  FXint x,y;
  if(FXMenuCaption::contains(parentx,parenty)) return 1;
  if(getPopup() && getPopup()->shown()){
    getParent()->translateCoordinatesTo(x,y,getRoot(),parentx,parenty);
    if(getPopup()->contains(x,y)) return 1;
    }
  return 0;
  }


// Save object to stream
void FXMenuTitle::save(FXStream& store) const {
  FXMenuCaption::save(store);
  store << pane;
  }


// Load object from stream
void FXMenuTitle::load(FXStream& store){
  FXMenuCaption::load(store);
  store >> pane;
  }  


// Delete it
FXMenuTitle::~FXMenuTitle(){
  pane=(FXMenuPane*)-1;
  }


