/********************************************************************************
*                                                                               *
*                         S t a t u s b a r   W i d g e t                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXStatusbar.cpp,v 1.5 1999/11/04 23:37:19 jeroen Exp $                   *
********************************************************************************/
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXComposite.h"
#include "FXPacker.h"
#include "FXHorizontalFrame.h"
#include "FXDragCorner.h"
#include "FXStatusbar.h"


#define PANESPACING   4
#define UPPERSPACING  0


/* 
  Notes:
  - Fallback text is displayed when neither cursor window or
    target object supply temporary help string.
*/


/*******************************************************************************/

// Map
FXDEFMAP(FXStatusline) FXStatuslineMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXStatusline::onPaint),
  FXMAPFUNC(SEL_UPDATE,0,FXStatusline::onUpdate),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE,FXStatusline::onCmdSetStringValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETSTRINGVALUE,FXStatusline::onCmdGetStringValue),
  };


// Default message
const FXchar FXStatusline::defaultMessage[]="Ready.";


// Object implementation
FXIMPLEMENT(FXStatusline,FXFrame,FXStatuslineMap,ARRAYNUMBER(FXStatuslineMap))

  
// Deserialization
FXStatusline::FXStatusline(){
  flags|=FLAG_SHOWN;
  }

  
// Construct and init
FXStatusline::FXStatusline(FXComposite* p,FXObject* tgt,FXSelector sel):
  FXFrame(p,FRAME_SUNKEN|LAYOUT_LEFT|LAYOUT_FILL_Y|LAYOUT_FILL_X,0,0,0,0, 4,4,2,2){
  flags|=FLAG_SHOWN;
  status=defaultMessage;
  normal=defaultMessage;
  font=getApp()->normalFont;
  textColor=getApp()->foreColor;
  textHighlightColor=getApp()->foreColor;
  target=tgt;
  message=sel;
  }


// Create Window
void FXStatusline::create(){
  FXFrame::create();
  font->create();
  }


// Detach Window
void FXStatusline::detach(){
  FXFrame::detach();
  font->detach();
  }


// Get default width; as text changes often, exact content does not matter
FXint FXStatusline::getDefaultWidth(){
  return padleft+padright+(border<<1)+8;
  }


// Get default height; just care about font height
FXint FXStatusline::getDefaultHeight(){
  return font->getFontHeight()+padtop+padbottom+(border<<1);
  }


// Slightly different from Frame border
long FXStatusline::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent *ev=(FXEvent*)ptr;
  FXDCWindow dc(this,ev);
  FXint ty=padtop+(height-padtop-padbottom-font->getFontHeight())/2;
  FXint pos,len;
  dc.setForeground(backColor);
  dc.fillRectangle(ev->rect.x,ev->rect.y,ev->rect.w,ev->rect.h);
  if(!status.empty()){
    dc.setTextFont(font);
    pos=status.findf('\n');
    len=status.length();
    if(pos>=0){
      dc.setForeground(textHighlightColor);
      dc.drawText(padleft,ty+font->getFontAscent(),status.text(),pos);
      dc.setForeground(textColor);
      dc.drawText(padleft+font->getTextWidth(status.text(),pos),ty+font->getFontAscent(),status.text()+pos+1,len-pos-1);
      }
    else{
      dc.setForeground(textColor);
      dc.drawText(padleft,ty+font->getFontAscent(),status.text(),len);
      }
    }
  drawFrame(dc,0,0,width,height);
  return 1;
  }


// If the cursor is inside a widget, flash its help text;
// Otherwise, unflash back to the regular status message.
long FXStatusline::onUpdate(FXObject* sender,FXSelector sel,void* ptr){
  FXWindow *helpsource;
  
  // Regular GUI update
  FXFrame::onUpdate(sender,sel,ptr);
  
  // Ask the help source for a new status text first
  helpsource=getApp()->getCursorWindow();
  if(helpsource){
    if(helpsource->handle(this,MKUINT(FXWindow::ID_QUERY_HELP,SEL_UPDATE),NULL)){
      return 1;
      }
    }
  
  // Ask target; this should be the normal help text
  // indicating the state the program is in currently.
  if(target){
    if(target->handle(this,MKUINT(message,SEL_UPDATE),NULL)){
      return 1;
      }
    }
  
  // Otherwise, supply normal message
  setText(normal);
  return 1;
  }


// Update value from a message
long FXStatusline::onCmdSetStringValue(FXObject*,FXSelector,void* ptr){
  if(ptr==NULL){ fxerror("%s::onCmdSetStringValue: NULL pointer.\n",getClassName()); }
  setText(*((FXString*)ptr));
  return 1;
  }


// Obtain value from text field
long FXStatusline::onCmdGetStringValue(FXObject*,FXSelector,void* ptr){
  if(ptr==NULL){ fxerror("%s::onCmdGetStringValue: NULL pointer.\n",getClassName()); }
  *((FXString*)ptr)=getText();
  return 1;
  }


// Set currently displayed message
void FXStatusline::setText(const FXString& text){
  if(status!=text){
    status=text;
    update(border,border,width-(border<<1),height-(border<<1));
    }
  }


// Set permanently displayed message
void FXStatusline::setNormalText(const FXString& text){
  if(normal!=text){
    normal=text;
    update(border,border,width-(border<<1),height-(border<<1));
    }
  }


// Change the font
void FXStatusline::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  if(font!=fnt){
    font=fnt;
    recalc();
    update();
    }
  }


// Set text color
void FXStatusline::setTextColor(FXColor clr){
  textColor=clr;
  update(border,border,width-(border<<1),height-(border<<1));
  }


// Set text highlight color
void FXStatusline::setTextHighlightColor(FXColor clr){
  textHighlightColor=clr;
  update(border,border,width-(border<<1),height-(border<<1));
  }


// Save object to stream
void FXStatusline::save(FXStream& store) const {
  FXFrame::save(store);
  store << status;
  store << normal;
  store << font;
  store << textColor;
  store << textHighlightColor;
  }


// Load object from stream
void FXStatusline::load(FXStream& store){
  FXFrame::load(store);
  store >> status;
  store >> normal;
  store >> font;
  store >> textColor;
  store >> textHighlightColor;
  }


// Destruct
FXStatusline::~FXStatusline(){
  font=(FXFont*)-1;
  }


/*******************************************************************************/


// Object implementation
FXIMPLEMENT(FXStatusbar,FXHorizontalFrame,NULL,0)


// Make a status bar
FXStatusbar::FXStatusbar(FXComposite* p,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb,FXint hs,FXint vs):
  FXHorizontalFrame(p,opts,x,y,w,h,pl,pr,pt,pb,hs,vs){
  corner=new FXDragCorner(this);
  status=new FXStatusline(this);
  }


// Compute minimum width based on child layout hints
int FXStatusbar::getDefaultWidth(){
  register FXint w,wcum,numc;
  register FXWindow* child;
  register FXuint hints;
  wcum=numc=0;
  for(child=corner->getNext(); child; child=child->getNext()){
    if(child->shown()){
      hints=child->getLayoutHints();
      if(hints&LAYOUT_FIX_WIDTH) w=child->getWidth(); 
      else w=child->getDefaultWidth();
      wcum+=w;
      numc++;
      }
    }
  if(numc>1) wcum+=(numc-1)*hspacing; 
  if(corner->shown() && (numc>1)) wcum+=corner->getDefaultWidth();
  return padleft+padright+wcum+(border<<1);
  }


// Compute minimum height based on child layout hints
int FXStatusbar::getDefaultHeight(){
  register FXint h,hmax;
  register FXWindow* child;
  register FXuint hints;
  hmax=0;
  for(child=corner->getNext(); child; child=child->getNext()){
    if(child->shown()){
      hints=child->getLayoutHints();
      if(hints&LAYOUT_FIX_HEIGHT) h=child->getHeight(); 
      else h=child->getDefaultHeight();
      if(hmax<h) hmax=h;
      }
    }
  h=padtop+padbottom+hmax;
  if(corner->shown() && (h<corner->getDefaultHeight())) h=corner->getDefaultHeight();
  return h+(border<<1);
  }


// Recalculate layout
void FXStatusbar::layout(){
  FXint left,right,top,bottom;
  FXint remain,extra_space,total_space,t;
  FXint x,y,w,h;
  FXint numc=0;
  FXint sumexpand=0;
  FXint numexpand=0;
  FXint e=0;
  FXuint hints;
  FXWindow* child;

  // Placement rectangle; right/bottom non-inclusive
  left=border+padleft;
  right=width-border-padright;
  top=border+padtop;
  bottom=height-border-padbottom;
  remain=right-left;


  // Find number of paddable children and total width
  for(child=corner->getNext(); child; child=child->getNext()){
    if(child->shown()){
      hints=child->getLayoutHints();
      if(hints&LAYOUT_FIX_WIDTH) w=child->getWidth();
      else w=child->getDefaultWidth();
      FXASSERT(w>=0);
      if((hints&LAYOUT_CENTER_X) || ((hints&LAYOUT_FILL_X) && !(hints&LAYOUT_FIX_WIDTH))){
        sumexpand+=w; 
        numexpand+=1;
        }
      else{
        remain-=w;
        }
      numc++;
      }
    }

  // Child spacing
  if(numc>1) remain-=hspacing*(numc-1);
    
  // Substract corner width
  if(corner->shown() && (numc>1)){
    right-=corner->getDefaultWidth();
    remain-=corner->getDefaultWidth();
    }

  // Do the layout
  for(child=corner->getNext(); child; child=child->getNext()){
    if(child->shown()){
      hints=child->getLayoutHints();

      // Layout child in Y
      y=child->getY();
      if(hints&LAYOUT_FIX_HEIGHT) h=child->getHeight(); 
      else h=child->getDefaultHeight();
      extra_space=0;
      if((hints&LAYOUT_FILL_Y) && !(hints&LAYOUT_FIX_HEIGHT)){
        h=bottom-top;
        if(h<0) h=0;
        }
      else if(hints&LAYOUT_CENTER_Y){
        if(h<(bottom-top)) extra_space=(bottom-top-h)/2;
        }
      if(hints&LAYOUT_BOTTOM)
        y=bottom-extra_space-h;
      else /*hints&LAYOUT_TOP*/
        y=top+extra_space;

      // Layout child in X
      x=child->getX();
      if(hints&LAYOUT_FIX_WIDTH) w=child->getWidth(); 
      else w=child->getDefaultWidth();
      extra_space=0;
      total_space=0;
      if((hints&LAYOUT_FILL_X) && !(hints&LAYOUT_FIX_WIDTH)){
        if(sumexpand>0){
          t=w*remain;
          FXASSERT(sumexpand>0);
          w=t/sumexpand;
          e+=t%sumexpand;
          if(e>=sumexpand){w++;e-=sumexpand;}
          }
        else{
          FXASSERT(numexpand>0);
          w=remain/numexpand;
          e+=remain%numexpand;
          if(e>=numexpand){w++;e-=numexpand;}
          }
        }
      else if(hints&LAYOUT_CENTER_X){
        if(sumexpand>0){
          t=w*remain;
          FXASSERT(sumexpand>0);
          total_space=t/sumexpand-w;
          e+=t%sumexpand;
          if(e>=sumexpand){total_space++;e-=sumexpand;}
          }
        else{
          FXASSERT(numexpand>0);
          total_space=remain/numexpand-w;
          e+=remain%numexpand;
          if(e>=numexpand){total_space++;e-=numexpand;}
          }
        extra_space=total_space/2;
        }
      if(hints&LAYOUT_RIGHT){
        x=right-w-extra_space;
        right=right-w-hspacing-total_space;
        }
      else{/*hints&LAYOUT_LEFT*/
        x=left+extra_space;
        left=left+w+hspacing+total_space;
        }
      child->position(x,y,w,h);
      }
    }

  // Just make sure corner grip's on top
  if(corner->shown()){
    if(numc>1)
      corner->position(width-border-corner->getDefaultWidth(),height-border-corner->getDefaultHeight(),corner->getDefaultWidth(),corner->getDefaultHeight());
    else
      corner->position(width-padright-border-corner->getDefaultWidth(),height-border-padbottom-corner->getDefaultHeight(),corner->getDefaultWidth(),corner->getDefaultHeight());
    corner->raise();
    }
  flags&=~FLAG_DIRTY;
  }



// Save object to stream
void FXStatusbar::save(FXStream& store) const {
  FXComposite::save(store);
  store << corner;
  store << status;
  }


// Load object from stream
void FXStatusbar::load(FXStream& store){
  FXComposite::load(store);
  store >> corner;
  store >> status;
  }


// Destruct
FXStatusbar::~FXStatusbar(){
  corner=(FXDragCorner*)-1;
  status=(FXStatusline*)-1;
  }



