/********************************************************************************
*                                                                               *
*                               T a b   O b j e c t                             *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXTab.cpp,v 1.5 1999/10/29 07:58:39 jeroen Exp $                         *
********************************************************************************/
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXComposite.h"
#include "FXPacker.h"
#include "FXTab.h"


/*
  Notes:
  - Should focus go to tab items?
  - Should callbacks come from tab items?
  - FXTabItems should catch SEL_ACTIVATE etc.
  - Should redesign this stuff a little.
  - Tab items should observe various border styles.
  - TAB/TABTAB should go into content, arrow keys navigate between tabs.
  - FXTabBook: pane's hints make no sense to observe
  - We hide the panes in FXTabBook.  This way, we don't have to change
    the position of each pane when the FXTabBook itself changes. 
    Only the active pane needs to be moved.
*/


#define TAB_ORIENT_MASK    (TAB_TOP|TAB_LEFT|TAB_RIGHT|TAB_BOTTOM)
#define TABBOOK_MASK       (TABBOOK_SIDEWAYS|TABBOOK_BOTTOMTABS)

/*******************************************************************************/

// Map
FXDEFMAP(FXTabItem) FXTabItemMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXTabItem::onPaint),
  FXMAPFUNC(SEL_FOCUSIN,0,FXTabItem::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXTabItem::onFocusOut),
  FXMAPFUNC(SEL_ACTIVATE,0,FXTabItem::onActivate),
  FXMAPFUNC(SEL_DEACTIVATE,0,FXTabItem::onDeactivate),
  FXMAPFUNC(SEL_KEYPRESS,FXWindow::ID_HOTKEY,FXTabItem::onHotKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,FXWindow::ID_HOTKEY,FXTabItem::onHotKeyRelease),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_TIP,FXTabItem::onQueryTip),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_HELP,FXTabItem::onQueryHelp),
  };


// Object implementation
FXIMPLEMENT(FXTabItem,FXLabel,FXTabItemMap,ARRAYNUMBER(FXTabItemMap))


// Tab item
FXTabItem::FXTabItem(FXTabBar* p,const FXString& text,FXIcon* ic,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb):
  FXLabel(p,text,ic,opts,x,y,w,h,pl,pr,pt,pb){
  tip=text.extract(1,'\t');
  help=text.extract(2,'\t');
  flags|=FLAG_ENABLED;
  border=2;
  }


// Enable the window
void FXTabItem::enable(){
  if(!(flags&FLAG_ENABLED)){
    FXWindow::enable();
    update();
    }
  }


// Disable the window
void FXTabItem::disable(){
  if(flags&FLAG_ENABLED){
    FXWindow::disable();
    update();
    }
  }


// If window can have focus
FXbool FXTabItem::canFocus() const { return 1; }


// Gained focus
long FXTabItem::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXLabel::onFocusIn(sender,sel,ptr);
  update(border,border,width-(border<<1),height-(border<<1));
  return 1;
  }

  
// Lost focus
long FXTabItem::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXLabel::onFocusOut(sender,sel,ptr);
  update(border,border,width-(border<<1),height-(border<<1));
  return 1;
  }


// Hot key combination pressed
long FXTabItem::onHotKeyPress(FXObject*,FXSelector,void* ptr){
  FXTRACE((200,"%s::onHotKeyPress %08x\n",getClassName(),this));
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
    }
  return 1;
  }


// Hot key combination released
long FXTabItem::onHotKeyRelease(FXObject*,FXSelector,void* ptr){
  FXTRACE((200,"%s::onHotKeyRelease %08x\n",getClassName(),this));
  flags&=~FLAG_TIP;
  if(isEnabled()){ 
    handle(this,MKUINT(0,SEL_DEACTIVATE),ptr); 
    }
  return 1;
  }


// Handle repaint 
long FXTabItem::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent *ev=(FXEvent*)ptr;
  FXDCWindow dc(this,ev);
  FXint tw=0,th=0,iw=0,ih=0,tx,ty,ix,iy;
  dc.setForeground(backColor);
  dc.fillRectangle(ev->rect.x,ev->rect.y,ev->rect.w,ev->rect.h);
  switch(options&TAB_ORIENT_MASK){
    case TAB_LEFT:
      dc.setForeground(hiliteColor);
      dc.drawLine(2,0,width-1,0);
      dc.drawLine(0,2,1,1);
      dc.drawLine(0,height-2,0,2);
      dc.setForeground(shadowColor);
      dc.drawLine(2,height-2,width-1,height-2);
      dc.setForeground(borderColor);
      dc.drawLine(3,height-1,width-1,height-1);
      break;
    case TAB_RIGHT:
      dc.setForeground(hiliteColor);
      dc.drawLine(0,0,width-3,0);
      dc.drawLine(width-3,0,width-1,3);
      dc.setForeground(shadowColor);
      dc.drawLine(width-2,2,width-2,height-2);
      dc.drawLine(0,height-2,width-2,height-2);
      dc.setForeground(borderColor);
      dc.drawLine(0,height-1,width-3,height-1);
      dc.drawLine(width-1,3,width-1,height-4);
      dc.drawLine(width-3,height-1,width-1,height-4);
      break;
    case TAB_BOTTOM:
      dc.setForeground(hiliteColor);
      dc.drawLine(0,0,0,height-4);
      dc.drawLine(0,height-4,1,height-2);
      dc.setForeground(shadowColor);
      dc.drawLine(2,height-2,width-3,height-2);
      dc.drawLine(width-2,0,width-2,height-3);
      dc.drawLine(width-2,0,width-1,0);
      dc.setForeground(borderColor);
      dc.drawLine(3,height-1,width-4,height-1);
      dc.drawLine(width-4,height-1,width-1,height-4);
      dc.drawLine(width-1,1,width-1,height-4);
      break;
    case TAB_TOP:
      dc.setForeground(hiliteColor);
      dc.drawLine(0,height-1,0,2);
      dc.drawLine(0,2,2,0);
      dc.drawLine(2,0,width-3,0);
      dc.setForeground(shadowColor);
      dc.drawLine(width-2,1,width-2,height-1);
      dc.setForeground(borderColor);
      dc.drawLine(width-2,1,width-1,2);
      dc.drawLine(width-1,2,width-1,height-2);
      dc.setForeground(hiliteColor);
      dc.drawLine(width-1,height-1,width-1,height-1);
      break;
    }
  if(!label.empty()){
    tw=font->getTextWidth(label.text(),label.length());
    th=font->getFontHeight();
    }
  if(icon){
    iw=icon->getWidth();
    ih=icon->getHeight();
    }
  just_x(tx,ix,tw,iw);
  just_y(ty,iy,th,ih);
  if(icon){
    if(isEnabled())
      dc.drawIcon(icon,ix,iy);
    else
      dc.drawIconShaded(icon,ix,iy);
    }
  if(!label.empty()){
    dc.setTextFont(font);
    if(isEnabled()){
      dc.setForeground(textColor);
      drawLabel(dc,label,hotoff,tx,ty,tw,th);
      if(hasFocus()){
        drawFocusRectangle(dc,border+2,border+2,width-2*border-4,height-2*border-4);
        }
      }
    else{
      dc.setForeground(hiliteColor);
      drawLabel(dc,label,hotoff,tx+1,ty+1,tw,th);
      dc.setForeground(shadowColor);
      drawLabel(dc,label,hotoff,tx,ty,tw,th);
      }
    }
  return 1;
  }


// We were asked about status text
long FXTabItem::onQueryHelp(FXObject* sender,FXSelector,void*){
  if(!help.empty() && (flags&FLAG_HELP)){
    sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&help);
    return 1;
    }
  return 0;
  }


// We were asked about tip text
long FXTabItem::onQueryTip(FXObject* sender,FXSelector,void*){
  if(!tip.empty() && (flags&FLAG_TIP)){
    sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&tip);
    return 1;
    }
  return 0;
  }


// Pressed left button
long FXTabItem::onActivate(FXObject*,FXSelector,void* ptr){
  getParent()->handle(this,MKUINT(FXTabBar::ID_OPEN_ITEM,SEL_COMMAND),ptr);
  return 1;
  }


// Released left button
long FXTabItem::onDeactivate(FXObject*,FXSelector,void*){
  return 1;
  }


// Change help text
void FXTabItem::setHelpText(const FXString& text){
  help=text;
  }


// Change tip text
void FXTabItem::setTipText(const FXString& text){
  tip=text;
  }


// Get tab style
FXuint FXTabItem::getTabOrientation() const {
  return (options&TAB_ORIENT_MASK);
  }


// Set tab style
void FXTabItem::setTabOrientation(FXuint style){
  FXuint opts=(options&~TAB_ORIENT_MASK) | (style&TAB_ORIENT_MASK);
  if(options!=opts){
    options=opts;
    recalc();
    update();
    }
  }


// Save object to stream
void FXTabItem::save(FXStream& store) const {
  FXLabel::save(store);
  store << tip;
  store << help;
  }


// Load object from stream
void FXTabItem::load(FXStream& store){
  FXLabel::load(store);
  store >> tip;
  store >> help;
  }  


/*******************************************************************************/

// Map
FXDEFMAP(FXTabBar) FXTabBarMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXTabBar::onPaint),
  FXMAPFUNC(SEL_FOCUS_NEXT,0,FXTabBar::onFocusNext),
  FXMAPFUNC(SEL_FOCUS_PREV,0,FXTabBar::onFocusPrev),
  FXMAPFUNC(SEL_FOCUS_UP,0,FXTabBar::onFocusUp),
  FXMAPFUNC(SEL_FOCUS_DOWN,0,FXTabBar::onFocusDown),
  FXMAPFUNC(SEL_FOCUS_LEFT,0,FXTabBar::onFocusLeft),
  FXMAPFUNC(SEL_FOCUS_RIGHT,0,FXTabBar::onFocusRight),
  FXMAPFUNC(SEL_COMMAND,FXTabBar::ID_OPEN_ITEM,FXTabBar::onOpenItem),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,FXTabBar::onCmdSetValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTVALUE,FXTabBar::onCmdSetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTVALUE,FXTabBar::onCmdGetIntValue),
  FXMAPFUNCS(SEL_UPDATE,FXTabBar::ID_OPEN_FIRST,FXTabBar::ID_OPEN_LAST,FXTabBar::onUpdOpen),
  FXMAPFUNCS(SEL_COMMAND,FXTabBar::ID_OPEN_FIRST,FXTabBar::ID_OPEN_LAST,FXTabBar::onCmdOpen),
  };


// Object implementation
FXIMPLEMENT(FXTabBar,FXPacker,FXTabBarMap,ARRAYNUMBER(FXTabBarMap))


// Make a tab bar 
FXTabBar::FXTabBar(FXComposite* p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb):
  FXPacker(p,opts,x,y,w,h,pl,pr,pt,pb,0,0){
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  current=0;
  }


// Get width
FXint FXTabBar::getDefaultWidth(){
  register FXint w,wtabs,wmaxtab,t,ntabs;
  register FXuint hints;
  register FXWindow *child;
  if(options&TABBOOK_SIDEWAYS){
    wtabs=0;
    for(child=getFirst(); child; child=child->getNext()){
      if(child->shown()){
        hints=child->getLayoutHints();
        if(hints&LAYOUT_FIX_WIDTH) t=child->getWidth(); else t=child->getDefaultWidth();
        if(t>wtabs) wtabs=t;
        }
      }
    w=wtabs;
    }
  else{
    wtabs=wmaxtab=ntabs=0;
    for(child=getFirst(); child; child=child->getNext()){
      if(child->shown()){
        hints=child->getLayoutHints();
        if(hints&LAYOUT_FIX_WIDTH) t=child->getWidth(); else t=child->getDefaultWidth();
        if(t>wmaxtab) wmaxtab=t;
        wtabs+=t;
        ntabs++;
        }
      }
    if(options&PACK_UNIFORM_WIDTH) wtabs=ntabs*wmaxtab;
    w=wtabs+5;
    }
  return w+padleft+padright+(border<<1);
  }


// Get height
FXint FXTabBar::getDefaultHeight(){
  register FXint h,htabs,hmaxtab,t,ntabs;
  register FXuint hints;
  register FXWindow *child;
  if(options&TABBOOK_SIDEWAYS){
    htabs=hmaxtab=ntabs=0;
    for(child=getFirst(); child; child=child->getNext()){
      if(child->shown()){
        hints=child->getLayoutHints();
        if(hints&LAYOUT_FIX_HEIGHT) t=child->getHeight(); else t=child->getDefaultHeight();
        if(t>hmaxtab) hmaxtab=t;
        htabs+=t;
        ntabs++;
        }
      }
    if(options&PACK_UNIFORM_HEIGHT) htabs=ntabs*hmaxtab;
    h=htabs+5;
    }
  else{
    htabs=0;
    for(child=getFirst(); child; child=child->getNext()){
      if(child->shown()){
        hints=child->getLayoutHints();
        if(hints&LAYOUT_FIX_HEIGHT) t=child->getHeight(); else t=child->getDefaultHeight();
        if(t>htabs) htabs=t;
        }
      }
    h=htabs;
    }
  return h+padtop+padbottom+(border<<1);
  }


// Recalculate layout
void FXTabBar::layout(){
  register int i,x,y,w,h,wmaxtab,hmaxtab,newcurrent;
  register FXWindow *raisetab=NULL;
  register FXWindow *tab;
  register FXuint hints;
  
  newcurrent=-1;

  // Measure tabs again
  wmaxtab=hmaxtab=0;
  for(tab=getFirst(),i=0; tab; tab=tab->getNext(),i++){
    if(tab->shown()){
      hints=tab->getLayoutHints();
      if(hints&LAYOUT_FIX_WIDTH) w=tab->getWidth(); else w=tab->getDefaultWidth();
      if(hints&LAYOUT_FIX_HEIGHT) h=tab->getHeight(); else h=tab->getDefaultHeight();
      if(w>wmaxtab) wmaxtab=w;
      if(h>hmaxtab) hmaxtab=h;
      if(newcurrent<0 || i<=current) newcurrent=i;
      }
    }
  
  // Changes current only if old current no longer visible
  current=newcurrent;
  
  // Tabs on left or right
  if(options&TABBOOK_SIDEWAYS){
    
    // Placements for tab items and tab panels
    y=border+padtop;
    if(options&TABBOOK_BOTTOMTABS){         // Right tabs
      x=width-padright-border-wmaxtab;
      }
    else{
      x=border+padleft;
      }
    
    // Place all of the children
    for(i=0,tab=getFirst(); tab; tab=tab->getNext(),i++){
      if(tab->shown()){
        hints=tab->getLayoutHints();
        if(hints&LAYOUT_FIX_HEIGHT) h=tab->getHeight();
        else if(options&PACK_UNIFORM_HEIGHT) h=hmaxtab;
        else h=tab->getDefaultHeight();
        if(current==i){
          if(options&TABBOOK_BOTTOMTABS)      // Right tabs
            tab->position(x-2,y,wmaxtab+2,h+3);
          else
            tab->position(x,y,wmaxtab+2,h+3);
          tab->update(0,0,wmaxtab+2,h+3);
          raisetab=tab;
          }
        else{
          if(options&TABBOOK_BOTTOMTABS)      // Right tabs
            tab->position(x-2,y+2,wmaxtab,h);
          else
            tab->position(x+2,y+2,wmaxtab,h);
          tab->update(0,0,wmaxtab,h);
          }
        y+=h;
        }
      }
    }
  
  // Tabs on top or bottom
  else{
    
    // Placements for tab items and tab panels
    x=border+padleft;
    if(options&TABBOOK_BOTTOMTABS){         // Bottom tabs
      y=height-padbottom-border-hmaxtab;
      }
    else{
      y=border+padtop;
      }
    
    // Place all of the children
    for(i=0,tab=getFirst(); tab; tab=tab->getNext(),i++){
      if(tab->shown()){
        hints=tab->getLayoutHints();
        if(hints&LAYOUT_FIX_WIDTH) w=tab->getWidth();
        else if(options&PACK_UNIFORM_WIDTH) w=wmaxtab;
        else w=tab->getDefaultWidth();
        if(current==i){
          if(options&TABBOOK_BOTTOMTABS)      // Bottom tabs
            tab->position(x,y-2,w+3,hmaxtab+2);
          else
            tab->position(x,y,w+3,hmaxtab+2);
          tab->update(0,0,w+3,hmaxtab+2);
          raisetab=tab;
          }
        else{
          if(options&TABBOOK_BOTTOMTABS)      // Bottom tabs
            tab->position(x+2,y-2,w,hmaxtab);
          else
            tab->position(x+2,y+2,w,hmaxtab);
          tab->update(0,0,w,hmaxtab);
          }
        x+=w;
        }
      }
    }
  if(raisetab) raisetab->raise();
  flags&=~FLAG_DIRTY;
  }


// Set current subwindow
void FXTabBar::setCurrent(FXint panel){
  if(0<=panel && panel!=current){
    current=panel;
    recalc();
    }
  }


// Handle repaint 
long FXTabBar::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent *ev=(FXEvent*)ptr;
  FXDCWindow dc(this,ev);
  dc.setForeground(backColor);
  dc.fillRectangle(ev->rect.x,ev->rect.y,ev->rect.w,ev->rect.h);
  drawFrame(dc,0,0,width,height);
  return 1;
  }


// Focus moved to next visible tab
long FXTabBar::onFocusNext(FXObject*,FXSelector,void* ptr){
  FXWindow *child=getFocus();
  if(child) child=child->getNext(); else child=getFirst();
  while(child && !child->shown()) child=child->getNext();
  if(child){
    setCurrent(indexOfChild(child));
    child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    return 1;
    }
  return 0;
  }


// Focus moved to previous visible tab
long FXTabBar::onFocusPrev(FXObject*,FXSelector,void* ptr){
  FXWindow *child=getFocus();
  if(child) child=child->getPrev(); else child=getLast();
  while(child && !child->shown()) child=child->getPrev();
  if(child){
    setCurrent(indexOfChild(child));
    child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    return 1;
    }
  return 0;
  }


// Focus moved up
long FXTabBar::onFocusUp(FXObject*,FXSelector,void* ptr){
  if(options&TABBOOK_SIDEWAYS){
    return handle(this,MKUINT(0,SEL_FOCUS_PREV),ptr);
    }
  return 0;
  }


// Focus moved down
long FXTabBar::onFocusDown(FXObject*,FXSelector,void* ptr){
  if(options&TABBOOK_SIDEWAYS){
    return handle(this,MKUINT(0,SEL_FOCUS_NEXT),ptr);
    }
  return 0;
  }


// Focus moved left
long FXTabBar::onFocusLeft(FXObject*,FXSelector,void* ptr){
  if(!(options&TABBOOK_SIDEWAYS)){
    return handle(this,MKUINT(0,SEL_FOCUS_PREV),ptr);
    }
  return 0;
  }


// Focus moved right
long FXTabBar::onFocusRight(FXObject*,FXSelector,void* ptr){
  if(!(options&TABBOOK_SIDEWAYS)){
    return handle(this,MKUINT(0,SEL_FOCUS_NEXT),ptr);
    }
  return 0;
  }


// Update value from a message
long FXTabBar::onCmdSetValue(FXObject*,FXSelector,void* ptr){
  setCurrent((FXint)(long)ptr);
  return 1;
  }


// Update value from a message
long FXTabBar::onCmdSetIntValue(FXObject*,FXSelector,void* ptr){
  if(ptr==NULL){ fxerror("%s::onCmdSetIntValue: NULL pointer.\n",getClassName()); }
  setCurrent(*((FXint*)ptr));
  return 1;
  }

// Obtain value from text field
long FXTabBar::onCmdGetIntValue(FXObject*,FXSelector,void* ptr){
  if(ptr==NULL){ fxerror("%s::onCmdGetIntValue: NULL pointer.\n",getClassName()); }
  *((FXint*)ptr)=current;
  return 1;
  }


// Open item
long FXTabBar::onCmdOpen(FXObject*,FXSelector sel,void*){
  setCurrent(SELID(sel)-ID_OPEN_FIRST);
  return 1;
  }


// Update the nth button
long FXTabBar::onUpdOpen(FXObject* sender,FXSelector sel,void* ptr){
  FXuint msg=((SELID(sel)-ID_OPEN_FIRST)==(FXuint)current) ? ID_CHECK : ID_UNCHECK;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// The sender of the message is the item to open up
long FXTabBar::onOpenItem(FXObject* sender,FXSelector,void*){
  FXint which=indexOfChild((FXWindow*)sender);
  if(0<=which && which!=current){
    setCurrent(which);
    if(target) target->handle(this,MKUINT(message,SEL_COMMAND),(void*)which);
    }
  return 1;
  }


// Get tab style
FXuint FXTabBar::getTabStyle() const {
  return (options&TABBOOK_MASK);
  }


// Set tab style
void FXTabBar::setTabStyle(FXuint style){
  FXuint opts=(options&~TABBOOK_MASK) | (style&TABBOOK_MASK);
  if(options!=opts){
    options=opts;
    recalc();
    update();
    }
  }


// Save object to stream
void FXTabBar::save(FXStream& store) const {
  FXPacker::save(store);
  store << current;
  }


// Load object from stream
void FXTabBar::load(FXStream& store){
  FXPacker::load(store);
  store >> current;
  }  


/*******************************************************************************/

FXDEFMAP(FXTabBook) FXTabBookMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXTabBook::onPaint),
  FXMAPFUNC(SEL_FOCUS_NEXT,0,FXTabBook::onFocusNext),
  FXMAPFUNC(SEL_FOCUS_PREV,0,FXTabBook::onFocusPrev),
  FXMAPFUNC(SEL_FOCUS_UP,0,FXTabBook::onFocusUp),
  FXMAPFUNC(SEL_FOCUS_DOWN,0,FXTabBook::onFocusDown),
  FXMAPFUNC(SEL_FOCUS_LEFT,0,FXTabBook::onFocusLeft),
  FXMAPFUNC(SEL_FOCUS_RIGHT,0,FXTabBook::onFocusRight),
  FXMAPFUNC(SEL_COMMAND,FXTabBar::ID_OPEN_ITEM,FXTabBook::onOpenItem),
  };


// Object implementation
FXIMPLEMENT(FXTabBook,FXTabBar,FXTabBookMap,ARRAYNUMBER(FXTabBookMap))


// Make a tab book
FXTabBook::FXTabBook(FXComposite* p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb):
  FXTabBar(p,tgt,sel,opts,x,y,w,h,pl,pr,pt,pb){
  }


// Get width
FXint FXTabBook::getDefaultWidth(){
  register FXint w,wtabs,wmaxtab,wpnls,t,ntabs;
  register FXuint hints;
  register FXWindow *tab,*pane;
  
  // Left or right tabs
  if(options&TABBOOK_SIDEWAYS){
    wtabs=wpnls=0;
    for(tab=getFirst(); tab && tab->getNext(); tab=tab->getNext()->getNext()){
      pane=tab->getNext();
      if(tab->shown()){
        hints=tab->getLayoutHints();
        if(hints&LAYOUT_FIX_WIDTH) t=tab->getWidth(); else t=tab->getDefaultWidth();
        if(t>wtabs) wtabs=t;
        t=pane->getDefaultWidth();
        if(t>wpnls) wpnls=t;
        }
      }
    w=wtabs+wpnls;
    }
  
  // Top or bottom tabs
  else{
    wtabs=wpnls=wmaxtab=ntabs=0;
    for(tab=getFirst(); tab && tab->getNext(); tab=tab->getNext()->getNext()){
      pane=tab->getNext();
      if(tab->shown()){
        hints=tab->getLayoutHints();
        if(hints&LAYOUT_FIX_WIDTH) t=tab->getWidth(); else t=tab->getDefaultWidth();
        if(t>wmaxtab) wmaxtab=t;
        wtabs+=t;
        t=pane->getDefaultWidth();
        if(t>wpnls) wpnls=t;
        ntabs++;
        }
      }
    if(options&PACK_UNIFORM_WIDTH) wtabs=ntabs*wmaxtab;
    w=FXMAX(wtabs,wpnls)+5;
    }
  return w+padleft+padright+(border<<1);
  }


// Get height
FXint FXTabBook::getDefaultHeight(){
  register FXint h,htabs,hmaxtab,hpnls,t,ntabs;
  register FXuint hints;
  register FXWindow *tab,*pane;
  
  // Left or right tabs
  if(options&TABBOOK_SIDEWAYS){
    htabs=hpnls=hmaxtab=ntabs=0;
    for(tab=getFirst(); tab && tab->getNext(); tab=tab->getNext()->getNext()){
      pane=tab->getNext();
      if(tab->shown()){
        hints=tab->getLayoutHints();
        if(hints&LAYOUT_FIX_HEIGHT) t=tab->getHeight(); else t=tab->getDefaultHeight();
        if(t>hmaxtab) hmaxtab=t;
        htabs+=t;
        t=pane->getDefaultHeight();
        if(t>hpnls) hpnls=t;
        ntabs++;
        }
      }
    if(options&PACK_UNIFORM_HEIGHT) htabs=ntabs*hmaxtab;
    h=FXMAX(htabs,hpnls)+5;
    }
  
  // Top or bottom tabs
  else{
    htabs=hpnls=0;
    for(tab=getFirst(); tab && tab->getNext(); tab=tab->getNext()->getNext()){
      pane=tab->getNext();
      if(tab->shown()){
        hints=tab->getLayoutHints();
        if(hints&LAYOUT_FIX_HEIGHT) t=tab->getHeight(); else t=tab->getDefaultHeight();
        if(t>htabs) htabs=t;
        t=pane->getDefaultHeight();
        if(t>hpnls) hpnls=t;
        }
      }
    h=htabs+hpnls;
    }
  return h+padtop+padbottom+(border<<1);
  }


// Recalculate layout
void FXTabBook::layout(){
  register int i,x,y,w,h,px,py,pw,ph,wmaxtab,hmaxtab,newcurrent;
  register FXWindow *raisepane=NULL;
  register FXWindow *raisetab=NULL;
  register FXWindow *pane,*tab;
  register FXuint hints;
  
  newcurrent=-1;

  // Measure tabs again
  wmaxtab=hmaxtab=0;
  for(tab=getFirst(),i=0; tab && tab->getNext(); tab=tab->getNext()->getNext(),i++){
    pane=tab->getNext();
    if(tab->shown()){
      hints=tab->getLayoutHints();
      if(hints&LAYOUT_FIX_WIDTH) w=tab->getWidth(); else w=tab->getDefaultWidth();
      if(hints&LAYOUT_FIX_HEIGHT) h=tab->getHeight(); else h=tab->getDefaultHeight();
      if(w>wmaxtab) wmaxtab=w;
      if(h>hmaxtab) hmaxtab=h;
      if(newcurrent<0 || i<=current) newcurrent=i;
      }
    }
  
  // This will change only if current now invisible
  current=newcurrent;
  
  // Left or right tabs
  if(options&TABBOOK_SIDEWAYS){
    
    // Placements for tab items and tab panels
    y=border+padtop;
    py=y;
    pw=width-padleft-padright-(border<<1)-wmaxtab;
    ph=height-padtop-padbottom-(border<<1);
    if(options&TABBOOK_BOTTOMTABS){         // Right tabs
      x=width-padright-border-wmaxtab;
      px=border+padleft;
      }
    else{
      x=border+padleft;
      px=x+wmaxtab;
      }
    
    // Place all of the children
    for(tab=getFirst(),i=0; tab && tab->getNext(); tab=tab->getNext()->getNext(),i++){
      pane=tab->getNext();
      if(tab->shown()){
        hints=tab->getLayoutHints();
        if(hints&LAYOUT_FIX_HEIGHT) h=tab->getHeight();
        else if(options&PACK_UNIFORM_HEIGHT) h=hmaxtab;
        else h=tab->getDefaultHeight();
        if(current==i){
          if(options&TABBOOK_BOTTOMTABS)      // Right tabs
            tab->position(x-2,y,wmaxtab+2,h+3);
          else
            tab->position(x,y,wmaxtab+2,h+3);
          tab->update(0,0,wmaxtab+2,h+3);
          pane->position(px,py,pw,ph);
          pane->show();
          raisetab=tab;
          raisepane=pane;
          }
        else{
          if(options&TABBOOK_BOTTOMTABS)      // Right tabs
            tab->position(x-2,y+2,wmaxtab,h);
          else
            tab->position(x+2,y+2,wmaxtab,h);
          tab->update(0,0,wmaxtab,h);
          pane->hide();
          }
        y+=h;
        }
      else{
        pane->hide();
        }
      }
    
    // Hide spurious last tab
    if(tab) tab->resize(0,0);
    }
  
  // Top or bottom tabs
  else{
    
    // Placements for tab items and tab panels
    x=border+padleft;
    px=x;
    pw=width-padleft-padright-(border<<1);
    ph=height-padtop-padbottom-(border<<1)-hmaxtab;
    if(options&TABBOOK_BOTTOMTABS){         // Bottom tabs
      y=height-padbottom-border-hmaxtab;
      py=border+padtop;
      }
    else{
      y=border+padtop;
      py=y+hmaxtab;
      }
    
    // Place all of the children
    for(tab=getFirst(),i=0; tab && tab->getNext(); tab=tab->getNext()->getNext(),i++){
      pane=tab->getNext();
      if(tab->shown()){
        hints=tab->getLayoutHints();
        if(hints&LAYOUT_FIX_WIDTH) w=tab->getWidth();
        else if(options&PACK_UNIFORM_WIDTH) w=wmaxtab;
        else w=tab->getDefaultWidth();
        if(current==i){
          if(options&TABBOOK_BOTTOMTABS)      // Bottom tabs
            tab->position(x,y-2,w+3,hmaxtab+2);
          else
            tab->position(x,y,w+3,hmaxtab+2);
          tab->update(0,0,w+3,hmaxtab+2);
          pane->position(px,py,pw,ph);
          pane->show();
          raisepane=pane;
          raisetab=tab;
          }
        else{
          if(options&TABBOOK_BOTTOMTABS)      // Bottom tabs
            tab->position(x+2,y-2,w,hmaxtab);
          else
            tab->position(x+2,y+2,w,hmaxtab);
          tab->update(0,0,w,hmaxtab);
          pane->hide();
          }
        x+=w;
        }
      else{
        pane->hide();
        }
      }

    // Hide spurious last tab
    if(tab) tab->resize(0,0);
    }
    
  // Raise tab over panel and panel over all other tabs
  if(raisepane) raisepane->raise();
  if(raisetab) raisetab->raise();
  
  flags&=~FLAG_DIRTY;
  }


// The sender of the message is the item to open up
long FXTabBook::onOpenItem(FXObject* sender,FXSelector,void*){
  FXint which=indexOfChild((FXWindow*)sender)/2;
  if(0<=which && which!=current){
    setCurrent(which);
    if(target) target->handle(this,MKUINT(message,SEL_COMMAND),(void*)which);
    }
  return 1;
  }


// Handle repaint 
long FXTabBook::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent *ev=(FXEvent*)ptr;
  FXDCWindow dc(this,ev);
  dc.setForeground(backColor);
  dc.fillRectangle(ev->rect.x,ev->rect.y,ev->rect.w,ev->rect.h);
  drawFrame(dc,0,0,width,height);
  return 1;
  }


// Focus moved to next tab
long FXTabBook::onFocusNext(FXObject*,FXSelector,void* ptr){
  FXWindow *child=getFocus();
  FXint which;
  if(child){
    child=child->getNext();
    if(!child) return 0;
    which=indexOfChild(child);
    if(which&1){
      child=child->getNext();
      which++;
      }
    }
  else{
    child=getFirst();
    which=0;
    }
  while(child && !child->shown()){ child=child->getNext(); which++; }
  if(child){
    setCurrent(which>>1);
    child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    return 1;
    }
  return 0;
  }


// Focus moved to previous
long FXTabBook::onFocusPrev(FXObject*,FXSelector,void* ptr){
  FXWindow *child=getFocus();
  FXint which;
  if(child){
    child=child->getPrev();
    if(!child) return 0;
    which=indexOfChild(child);
    if(which&1){
      child=child->getPrev();
      }
    }
  else{
    child=getLast();
    if(!child) return 0;
    which=indexOfChild(child);
    if(which&1){
      child=child->getPrev();
      }
    }
  if(which&1){
    while(child && child->getPrev() && !child->shown()){ child=child->getPrev()->getPrev(); which-=2; }
    }
  else{
    while(child && !child->shown()){ child=child->getPrev(); which--; }
    }
  if(child){
    setCurrent(which>>1);
    child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    return 1;
    }
  return 0;
  }


// Focus moved up
long FXTabBook::onFocusUp(FXObject*,FXSelector sel,void* ptr){
  if(options&TABBOOK_SIDEWAYS){
    return handle(this,MKUINT(0,SEL_FOCUS_PREV),ptr);
    }
  if(getFocus()){
    FXWindow *child=NULL;
    if(indexOfChild(getFocus())&1){     // We're on a panel
      if(!(options&TABBOOK_BOTTOMTABS)) child=getFocus()->getPrev();
      }
    else{                               // We're on a tab
      if(options&TABBOOK_BOTTOMTABS) child=getFocus()->getNext();
      }
    if(child){
      if(child->isEnabled() && child->canFocus()){
        child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
        return 1;
        }
      if(child->isComposite() && child->handle(this,sel,ptr)){
        return 1;
        }
      }
    }
  return 0;
  }


// Focus moved down
long FXTabBook::onFocusDown(FXObject*,FXSelector sel,void* ptr){
  if(options&TABBOOK_SIDEWAYS){
    return handle(this,MKUINT(0,SEL_FOCUS_NEXT),ptr);
    }
  if(getFocus()){
    FXWindow *child=NULL;
    if(indexOfChild(getFocus())&1){     // We're on a panel
      if(options&TABBOOK_BOTTOMTABS) child=getFocus()->getPrev();
      }
    else{                               // We're on a tab
      if(!(options&TABBOOK_BOTTOMTABS)) child=getFocus()->getNext();
      }
    if(child){
      if(child->isEnabled() && child->canFocus()){
        child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
        return 1;
        }
      if(child->isComposite() && child->handle(this,sel,ptr)){
        return 1;
        }
      }
    }
  return 0;
  }


// Focus moved left
long FXTabBook::onFocusLeft(FXObject*,FXSelector sel,void* ptr){
  if(!(options&TABBOOK_SIDEWAYS)){
    return handle(this,MKUINT(0,SEL_FOCUS_PREV),ptr);
    }
  if(getFocus()){
    FXWindow *child=NULL;
    if(indexOfChild(getFocus())&1){     // We're on a panel
      if(!(options&TABBOOK_BOTTOMTABS)) child=getFocus()->getPrev();
      }
    else{                               // We're on a tab
      if(options&TABBOOK_BOTTOMTABS) child=getFocus()->getNext();
      }
    if(child){
      if(child->isEnabled() && child->canFocus()){
        child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
        return 1;
        }
      if(child->isComposite() && child->handle(this,sel,ptr)){
        return 1;
        }
      }
    }
  return 0;
  }


// Focus moved right
long FXTabBook::onFocusRight(FXObject*,FXSelector sel,void* ptr){
  if(!(options&TABBOOK_SIDEWAYS)){
    return handle(this,MKUINT(0,SEL_FOCUS_NEXT),ptr);
    }
  if(getFocus()){
    FXWindow *child=NULL;
    if(indexOfChild(getFocus())&1){     // We're on a panel
      if(options&TABBOOK_BOTTOMTABS) child=getFocus()->getPrev();
      }
    else{                               // We're on a tab
      if(!(options&TABBOOK_BOTTOMTABS)) child=getFocus()->getNext();
      }
    if(child){
      if(child->isEnabled() && child->canFocus()){
        child->handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
        return 1;
        }
      if(child->isComposite() && child->handle(this,sel,ptr)){
        return 1;
        }
      }
    }
  return 0;
  }

