/********************************************************************************
*                                                                               *
*                          U t i l i t y   F u n c t i o n s                    *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: fxutils.cpp,v 1.5 1999/11/17 07:31:08 jeroen Exp $                      *
********************************************************************************/
#include "xincs.h"
#include "fxdefs.h"
#include "fxkeys.h"


/*
  To do:
  - Those functions manipulating strings should perhaps become FXString type
    functions?
  - Need to pass buffer-size argument to all those fxdirpart() etc. functions
    to protect against memory overruns (make it a default argument last one
    so as to not impact anyone).
  - Revive my old malloc() replacement library to detect memory block overwrites.
*/

/*******************************************************************************/

// Bit-reversal
const FXuchar fxbitreverse[256]={
  0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0,
  0x10, 0x90, 0x50, 0xd0, 0x30, 0xb0, 0x70, 0xf0,
  0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8,
  0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8,
  0x04, 0x84, 0x44, 0xc4, 0x24, 0xa4, 0x64, 0xe4,
  0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4,
  0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec,
  0x1c, 0x9c, 0x5c, 0xdc, 0x3c, 0xbc, 0x7c, 0xfc,
  0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2,
  0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2,
  0x0a, 0x8a, 0x4a, 0xca, 0x2a, 0xaa, 0x6a, 0xea,
  0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa,
  0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6,
  0x16, 0x96, 0x56, 0xd6, 0x36, 0xb6, 0x76, 0xf6,
  0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee,
  0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe,
  0x01, 0x81, 0x41, 0xc1, 0x21, 0xa1, 0x61, 0xe1,
  0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1,
  0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9,
  0x19, 0x99, 0x59, 0xd9, 0x39, 0xb9, 0x79, 0xf9,
  0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5,
  0x15, 0x95, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5,
  0x0d, 0x8d, 0x4d, 0xcd, 0x2d, 0xad, 0x6d, 0xed,
  0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd,
  0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3,
  0x13, 0x93, 0x53, 0xd3, 0x33, 0xb3, 0x73, 0xf3,
  0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb,
  0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb,
  0x07, 0x87, 0x47, 0xc7, 0x27, 0xa7, 0x67, 0xe7,
  0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7,
  0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef,
  0x1f, 0x9f, 0x5f, 0xdf, 0x3f, 0xbf, 0x7f, 0xff
  };


/*******************************************************************************/


// Global flag which controls tracing level
unsigned int fxTraceLevel=0;


/*******************************************************************************/

// Allocate memory
FXint fxmalloc(void** ptr,unsigned long size){
  FXASSERT(ptr);
  *ptr=malloc(size);
  return *ptr!=NULL;
  }


// Allocate cleaned memory
FXint fxcalloc(void** ptr,unsigned long size){
  FXASSERT(ptr);
  *ptr=calloc(size,1);
  return *ptr!=NULL;
  }


// Resize memory
FXint fxresize(void** ptr,unsigned long size){
  FXASSERT(ptr);
  void *p=realloc(*ptr,size);
  if(p){ *ptr=p; return TRUE; }
  return FALSE;
  }


// Free memory, resets ptr to NULL afterward
void fxfree(void** ptr){
  FXASSERT(ptr);
  if(*ptr) free(*ptr);
  *ptr=NULL;
  }


// String duplicate
FXchar *fxstrdup(const FXchar* str){
  FXchar *copy;
  FXMALLOC(&copy,FXchar,strlen(str)+1);
  strcpy(copy,str);
  return copy;
  }


// Case independent compare
int fxstrcasecmp(const FXchar* s1,const FXchar* s2){
  register FXint c1,c2;
  if(s1 && s2){
    do{
      c1=toupper(*s1++);
      c2=toupper(*s2++);
      }
    while(c1 && (c1==c2));
    return c1-c2;
    }
  return 0;
  }


// Assert failed routine
void fxassert(const char* expression,const char* filename,unsigned int lineno){
#ifndef WIN32
  fprintf(stderr,"%s:%d: FXASSERT(%s) failed.\n",filename,lineno,expression);
#else
  char msg[256];
  sprintf(msg,"%s(%d): FXASSERT(%s) failed.\n",filename,lineno,expression);
#ifdef _WINDOWS
  OutputDebugString(msg);
  fprintf(stderr,"%s",msg); // if a console is available
#else
  fprintf(stderr,"%s",msg);
#endif
#endif
  }


// Log message to [typically] stderr
void fxmessage(const char* format,...){
#ifndef WIN32
  va_list arguments;
  va_start(arguments,format);
  vfprintf(stderr,format,arguments);
  va_end(arguments);
#else
  char msg[256];
  va_list arguments;
  va_start(arguments,format);
  vsprintf(msg,format,arguments);
  va_end(arguments);
#ifdef _WINDOWS
  OutputDebugString(msg);
  fprintf(stderr,"%s",msg); // if a console is available
#else
  fprintf(stderr,"%s",msg);
#endif
#endif
  }


// Error routine
void fxerror(const char* format,...){
#ifndef WIN32
  va_list arguments;
  va_start(arguments,format);
  vfprintf(stderr,format,arguments);
  va_end(arguments);
  exit(1);
#else
  char msg[256];
  va_list arguments;
  va_start(arguments,format);
  vsprintf(msg,format,arguments);
  va_end(arguments);
#ifdef _WINDOWS
  OutputDebugString(msg);
  fprintf(stderr,"%s",msg); // if a console is available
  MessageBox(NULL,msg,NULL,MB_OK|MB_ICONEXCLAMATION|MB_APPLMODAL);
  DebugBreak();
#else
  fprintf(stderr,"%s",msg);
  exit(1);
#endif
#endif
  }


// Warning routine
void fxwarning(const char* format,...){
#ifndef WIN32
  va_list arguments;
  va_start(arguments,format);
  vfprintf(stderr,format,arguments);
  va_end(arguments);
#else
  char msg[256];
  va_list arguments;
  va_start(arguments,format);
  vsprintf(msg,format,arguments);
  va_end(arguments);
#ifdef _WINDOWS
  OutputDebugString(msg);
  fprintf(stderr,"%s",msg); // if a console is available
  MessageBox(NULL,msg,NULL,MB_OK|MB_ICONINFORMATION|MB_APPLMODAL);
#else
  fprintf(stderr,"%s",msg);
#endif
#endif
  }


// Trace printout routine
void fxtrace(unsigned int level,const char* format,...){
  if(fxTraceLevel>level){
#ifndef WIN32
    va_list arguments;
    va_start(arguments,format);
    vfprintf(stderr,format,arguments);
    va_end(arguments);
#else
    char msg[256];
    va_list arguments;
    va_start(arguments,format);
    vsprintf(msg,format,arguments);
    va_end(arguments);
#ifdef _WINDOWS
    OutputDebugString(msg);
    fprintf(stderr,"%s",msg); // if a console is available
#else
    fprintf(stderr,"%s",msg);
#endif
#endif
    }
  }


// Sleep n microseconds
void fxsleep(unsigned int n){
#ifdef WIN32
  unsigned int zzz=n/1000;
  if(zzz==0) zzz=1;
  Sleep(zzz);
#else
#ifdef __USE_POSIX199309
  struct timespec value;
  value.tv_nsec = 1000 * (n%1000000);
  value.tv_sec = n/1000000;
  nanosleep(&value,NULL);
#else
#ifndef BROKEN_SELECT  
  struct timeval value;
  value.tv_usec = n % 1000000;
  value.tv_sec = n / 1000000;
  select(1,0,0,0,&value);
#else
  unsigned int zzz=n/1000000;
  if(zzz==0) zzz=1;
  if(zzz){
    while((zzz=sleep(zzz))>0) ;
    }
#endif
#endif
#endif
  }


// Character value of keysym+control state
FXchar fxkeyval(FXuint keysym,FXuint state){
  if((state&CONTROLMASK) && 0x60<=keysym && keysym<=0x7f) return keysym-0x60;
  if(state&0x2000){   // Patch from mohme@hgb-leipzig.de
    switch(keysym){ 
      case 'q': return '@';
      case '7': return '{';
      case '8': return '[';
      case '9': return ']';
      case '0': return '}';
      case 0xdf: return '\\';
      }
    }
  if(KEY_space<=keysym && keysym<=255) return keysym;
  if(KEY_BackSpace<=keysym && keysym<=KEY_Return) return keysym&255;
  if(KEY_KP_0<=keysym && keysym<=KEY_KP_9) return '0'-KEY_KP_0+keysym;
  if(KEY_Escape==keysym) return '\033';
  if(KEY_KP_Space==keysym) return ' ';
  if(KEY_KP_Tab==keysym) return '\t';
  if(KEY_KP_Enter==keysym) return '\015';
  if(KEY_KP_Add==keysym) return '+';
  if(KEY_KP_Subtract==keysym) return '-';
  if(KEY_KP_Multiply==keysym) return '*';
  if(KEY_KP_Divide==keysym) return '/';
  if(KEY_KP_Equal==keysym) return '=';
  if(KEY_KP_Decimal==keysym) return '.';
  return 0;
  }

#define ISSEP(ch) ((ch)=='+' || (ch)=='-' || (ch)==' ')


// Parse for accelerator key codes in a string
FXHotKey fxparseaccel(const FXchar* s){
  FXuint code=0,mods=0;
  if(s){
    while(s[0] && s[0]!='\t' && s[0]!='\n'){
      if(ISSEP(s[0])){
        s++;
        continue;
        }
      if((toupper(s[0])=='C')&&(toupper(s[1])=='T')&&(toupper(s[2])=='L')&&ISSEP(s[3])){
        mods|=CONTROLMASK;
        s+=4;
        continue;
        }
      if((toupper(s[0])=='A')&&(toupper(s[1])=='L')&&(toupper(s[2])=='T')&&ISSEP(s[3])){
        mods|=ALTMASK;
        s+=4;
        continue;
        }
      if((toupper(s[0])=='S')&&(toupper(s[1])=='H')&&(toupper(s[2])=='I')&&(toupper(s[3])=='F')&&(toupper(s[4])=='T')&&ISSEP(s[5])){
        mods|=SHIFTMASK;
        s+=6;
        continue;
        }
      if(isalnum(s[0]) && (s[1]=='\0' || s[1]=='\t' || s[1]=='\n')){
        if(mods&SHIFTMASK) 
          code=toupper(s[0])+KEY_space-' ';
        else
          code=tolower(s[0])+KEY_space-' ';
        }
      FXTRACE((150,"fxparseaccel: code = %04x mods=%04x\n",code,mods));
      return MKUINT(code,mods);
      }
    }
  return 0;
  }


// Parse for hot key in a string
FXHotKey fxparsehotkey(const FXchar* s){
  FXuint code,mods;
  if(s){
    while(s[0] && s[0]!='\t'){
      if(s[0]=='&'){
        if(s[1]!='&'){
          if(isalnum(s[1])){
            mods=ALTMASK;
            code=tolower(s[1])+KEY_space-' ';
            FXTRACE((150,"fxparsehotkey: code = %04x mods=%04x\n",code,mods));
            return MKUINT(code,mods);
            }
          break;
          }
        s++;
        }
      s++;
      }
    }
  return 0;
  }


// Locate hot key underline offset from begin of string
FXint fxfindhotkeyoffset(const FXchar* s){
  register FXint pos=0;
  if(s){
    while(s[pos] && s[pos]!='\t'){
      if(s[pos]=='&'){
        if(s[pos+1]!='&') return pos;
        pos++;
        }
      pos++;
      }
    }
  return -1;
  }


#define FOLD(c)	  ((flags&FILEMATCH_CASEFOLD)&&islower(c)?toupper(c):(c))


// Match a file name with a pattern; returning 1 if it matches, 0 if not.
FXint fxfilematch(const char *pattern,const char *string,FXuint flags){
  if(pattern && string){
    register const char *p=pattern,*n=string;
    register char c,c1,cstart,cend;
    register int neg;
    
    // Loop over characters of pattern
    while((c=*p++)!='\0'){
      c=FOLD(c);
      switch(c){

        // Single character wild card
        case '?':
	  if(*n=='\0') return 0;
	  else if((flags&FILEMATCH_FILE_NAME) && *n==PATHSEP) return 0;
	  else if((flags&FILEMATCH_PERIOD) && *n=='.' && (n==string || ((flags&FILEMATCH_FILE_NAME) && n[-1]==PATHSEP))) return 0;
	  break;

        // Many character wild card
        case '*':
	  if((flags&FILEMATCH_PERIOD) && *n=='.' && (n==string || ((flags&FILEMATCH_FILE_NAME) && n[-1]==PATHSEP))) return 0;

	  for(c=*p++; c=='?' || c=='*'; c=*p++,n++){
	    if(((flags&FILEMATCH_FILE_NAME) && *n==PATHSEP) || (c=='?' && *n=='\0')) return 0;
            }

	  if(c=='\0') return 1;

	  c1=(!(flags&FILEMATCH_NOESCAPE) && c=='\\') ? *p : c;
	  c1=FOLD(c1);
          for(--p; *n!='\0'; n++){
	    if((c=='[' || FOLD(*n)==c1) && fxfilematch(p,n,flags&~FILEMATCH_PERIOD)) return 1;
            }
          return 0;

        // Character range wild card
        case '[':
	  if(*n=='\0') return 0;

	  if((flags&FILEMATCH_PERIOD) && *n=='.' && (n==string || ((flags&FILEMATCH_FILE_NAME) && n[-1]==PATHSEP))) return 0;

	  // Nonzero if the sense of the character class is inverted.
	  neg=(*p=='!' || *p=='^');

	  if(neg) ++p;

	  c=*p++;
	  for(;;){
	    cstart=c;
            cend=c;

	    if(!(flags&FILEMATCH_NOESCAPE) && c=='\\') cstart=cend=*p++;

	    cstart=cend=FOLD(cstart);

	    // [ (unterminated) loses.
	    if(c=='\0') return 0;

	    c=*p++;
	    c=FOLD(c);

	    // [/] can never match.
	    if((flags&FILEMATCH_FILE_NAME) && c==PATHSEP) return 0;

	    if(c=='-' && *p!=']'){
	      cend = *p++;
	      if(!(flags&FILEMATCH_NOESCAPE) && cend=='\\') cend=*p++;
	      if(cend=='\0') return 0;
	      cend=FOLD(cend);
	      c=*p++;
	      }

	    if(FOLD(*n)>=cstart && FOLD(*n)<=cend) goto matched;

	    if(c==']') break;
	    }
	  if(!neg) return 0;
	  break;

matched:  // Skip the rest of the [...] that already matched.
	  while(c!=']'){

	    // [... (unterminated) loses.
	    if(c=='\0') return 0;

	    c=*p++;

	    // XXX 1003.2d11 is unclear if this is right.
	    if(!(flags&FILEMATCH_NOESCAPE) && c=='\\') ++p;
	    }
	  if(neg) return 0;
	  break;

        // Escaped character
        case '\\':
	  if(!(flags&FILEMATCH_NOESCAPE)){
	    c=*p++;
	    c=FOLD(c);
	    }
          // Fall through...

        // Normal character
        default:
	  if(FOLD(*n)!=c) return 0;
	  break;
        }

      ++n;
      }

    if(*n=='\0') return 1;

    // The FILEMATCH_LEADING_DIR flag says that "foo*" matches "foobar/frobozz".
    if((flags&FILEMATCH_LEADING_DIR) && *n==PATHSEP) return 1;
    }

  return 0;
  }



// Get highlight color
FXColor makeHiliteColor(FXColor clr){
  FXuint r,g,b;
  r=FXREDVAL(clr);
  g=FXGREENVAL(clr);
  b=FXBLUEVAL(clr);
  r=FXMAX(31,r);
  g=FXMAX(31,g);
  b=FXMAX(31,b);
  r=(133*r)/100;
  g=(133*g)/100;
  b=(133*b)/100;
  r=FXMIN(255,r);
  g=FXMIN(255,g);
  b=FXMIN(255,b);
  return FXRGB(r,g,b);
  }


// Get shadow color
FXColor makeShadowColor(FXColor clr){
  FXuint r,g,b;
  r=FXREDVAL(clr);
  g=FXGREENVAL(clr);
  b=FXBLUEVAL(clr);
  r=(66*r)/100;
  g=(66*g)/100;
  b=(66*b)/100;
  return FXRGB(r,g,b);
  }


// Get user name from uid
FXchar* fxgetusername(FXchar* result,FXuint uid){
  if(!result){fxerror("fxgetusername: NULL result argument.\n");}
#ifndef WIN32
  struct passwd *pwd;
  if((pwd=getpwuid(uid))!=NULL) 
    strcpy(result,pwd->pw_name); 
  else 
    sprintf(result,"%d",uid);
#else
  sprintf(result,"%d",uid);
#endif
  return result;
  }


// Get group name from gid
FXchar* fxgetgroupname(FXchar* result,FXuint gid){
  if(!result){fxerror("fxgetgroupname: NULL result argument.\n");}
#ifndef WIN32
  struct group *grp;
  if((grp=getgrgid(gid))!=NULL) 
    strcpy(result,grp->gr_name); 
  else 
    sprintf(result,"%d",gid);
#else
  sprintf(result,"%d",gid);
#endif
  return result;
  }


// Get permissions string from mode
FXchar* fxgetpermissions(FXchar* result,FXuint mode){
  if(!result){fxerror("fxgetpermissions: NULL result argument.\n");}
#ifndef WIN32
  result[0]=S_ISLNK(mode) ? 'l' : S_ISREG(mode) ? '-' : S_ISDIR(mode) ? 'd' : S_ISCHR(mode) ? 'c' : S_ISBLK(mode) ? 'b' : S_ISFIFO(mode) ? 'p' : S_ISSOCK(mode) ? 's' : '?';
  result[1]=(mode&S_IRUSR) ? 'r' : '-';
  result[2]=(mode&S_IWUSR) ? 'w' : '-';
  result[3]=(mode&S_ISUID) ? 's' : (mode&S_IXUSR) ? 'x' : '-';
  result[4]=(mode&S_IRGRP) ? 'r' : '-';
  result[5]=(mode&S_IWGRP) ? 'w' : '-';
  result[6]=(mode&S_ISGID) ? 's' : (mode&S_IXGRP) ? 'x' : '-';
  result[7]=(mode&S_IROTH) ? 'r' : '-';
  result[8]=(mode&S_IWOTH) ? 'w' : '-';
  result[9]=(mode&S_ISVTX) ? 't' : (mode&S_IXOTH) ? 'x' : '-';
  result[10]=0;
#else
  result[0]='-';
  result[1]='r';
  result[2]='w';
  result[3]='x';
  result[4]='r';
  result[5]='w';
  result[6]='x';
  result[7]='r';
  result[8]='w';
  result[9]='x';
  result[10]=0;
#endif
  return result;
  }


// Return TRUE iff s is a prefix of t
FXbool fxprefix(const FXchar* s,const FXchar* t){
  const FXchar *p,*q;
  for(p=s,q=t; *p && *q && *p==*q; p++,q++);
  return !*p;
  }


// Return TRUE iff s is a suffix of t
FXbool fxsuffix(const FXchar* s, const FXchar* t){
  const FXchar *p,*q;
  for(p=s+strlen(s),q=t+strlen(t); p>s && q>t && *p==*q; p--,q--);
  return p==s && *p==*q;
  }


// Expand ~ in filenames; using HOME environment variable
// Note "~name" expands to "~name", "~/name" expands to "$HOME/name", 
// and "~" expands to "$HOME"
FXchar* fxexpand(FXchar* result,const FXchar* name){
  if(!result){fxerror("fxexpand: NULL result argument.\n");}
  *result='\0';
  if(name){
    if(name[0]=='~'){
      if(name[1]=='\0' || name[1]==PATHSEP){
        fxgethomedir(result);
        name++;
        }
      }
    strcat(result,name);
    }
  return result;
  }
        

// Construct a full pathname from the given directory and file name
FXchar* fxpathname(FXchar* result,const FXchar* dirname,const FXchar* filename){
  int l;
  if(!result){fxerror("fxpathname: NULL result argument.\n");}
  fxexpand(result,dirname);
  if((l=strlen(result)) && result[l-1]!=PATHSEP) result[l++]=PATHSEP;
  strcpy(result+l,filename);
  return result;
  }


// Return the directory part of pathname
FXchar* fxdirpart(FXchar* result,const FXchar* pathname){
  FXchar *p;
  if(!result){fxerror("fxdirpart: NULL result argument.\n");}
  fxexpand(result,pathname);
  if((p=strrchr(result,PATHSEP))){      // Fix by sancelot@usa.net
    if(fxexists(result)){
      if(fxisfile(result)){
        if(p==result) p++;
        *p='\0';
        }
      }
    else{
      if(p==result) p++;
      *p='\0';
      }
    }
  else{
    fxgetcurrentdir(result);
    }
  return result;
  }


// Return the file title, i.e. filename less extension part of pathname
FXchar* fxfiletitle(FXchar* result,const FXchar* pathname){
  FXchar *p;
  if(!result){fxerror("fxfiletitle: NULL result argument.\n");}
  result[0]='\0';
  if(pathname){
    p=(FXchar*)strrchr(pathname,PATHSEP); // Borland C++ compiler says strrchr() returns a const char*
    if(p) pathname=p+1;
    strcpy(result,pathname);
    p=strrchr(result,'.');
    if(p) *p=0;
    }
  return result;
  }


// Return the filename part of pathname
FXchar* fxfilepart(FXchar* result,const FXchar* pathname){
  const FXchar *p; // Borland C++ compiler says strrchr() returns const char*
  if(!result){fxerror("fxfilepart: NULL result argument.\n");}
  p=strrchr(pathname,PATHSEP);
  if(p) pathname=p+1;
  strcpy(result,pathname);
  return result;
  }


// Return the extension part of pathname
FXchar* fxfileext(FXchar* result,const FXchar* pathname){
  const FXchar* p; // Borland C++ compiler says strrchr() returns const char*
  if(!result){fxerror("fxfileext: NULL result argument.\n");}
  *result='\0';
  p=strrchr(pathname,PATHSEP);
  if(p) pathname=p+1;
  if((p=strrchr(pathname,'.'))) return strcpy(result,p+1);
  return result;
  }


// Check whether a file exists
FXbool fxexists(const FXchar *name){
  struct stat info;
  if(!name){fxerror("fxexists: NULL name argument.\n");}
  return stat(name,&info)==0;
  }


// Check whether its a directory
FXbool fxisdir(const FXchar *name){
  struct stat info;
  if(!name){fxerror("fxisdir: NULL name argument.\n");}
#ifndef WIN32
  return (stat(name,&info)==0) && S_ISDIR(info.st_mode);
#else
  return (stat(name,&info)==0) && (GetFileAttributes(name)&FILE_ATTRIBUTE_DIRECTORY);
#endif
  }


// Check whether its a file
FXbool fxisfile(const FXchar *name){
  struct stat info;
  if(!name){fxerror("fxisdir: NULL name argument.\n");}
#ifndef WIN32
  return (stat(name,&info)==0) && S_ISREG(info.st_mode);
#else
  return (stat(name,&info)==0) && !(GetFileAttributes(name)&FILE_ATTRIBUTE_DIRECTORY);
#endif
  }


// Create a unique numbered backup file name for the given pathname 
FXchar* fxbakname(FXchar* result,const FXchar* pathname){
  FXchar* p;
  int n=0;
  if(!result){fxerror("fxbakname: NULL result argument.\n");}
  if(!pathname){fxerror("fxbakname: NULL pathname argument.\n");}
  p=result+strlen(pathname);
  strcpy(result,pathname);
  do{
    sprintf(p,".~%d~",++n);
    } 
  while(fxexists(result));
  return result;
  }



// Check whether two files are identical (refer to the same inode)
FXbool fxidentical(const FXchar *name1,const FXchar *name2){
  if(!name1){fxerror("fxidentical: NULL name1 argument.\n");}
  if(!name2){fxerror("fxidentical: NULL name2 argument.\n");}
#ifndef WIN32
  struct stat info1,info2;
  return !lstat(name1,&info1) && !lstat(name2,&info2) && info1.st_ino==info2.st_ino;
#else
  HANDLE hFile1=CreateFile(name1,GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
  if(hFile1==INVALID_HANDLE_VALUE) return FALSE;
  HANDLE hFile2=CreateFile(name2,GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
  if(hFile2==INVALID_HANDLE_VALUE){
    CloseHandle(hFile1);
    return FALSE;
    }
  BY_HANDLE_FILE_INFORMATION info1;
  BY_HANDLE_FILE_INFORMATION info2;
  FXbool same=FALSE;
  if(GetFileInformationByHandle(hFile1,&info1) && GetFileInformationByHandle(hFile2,&info2)){
    if(info1.nFileIndexLow==info2.nFileIndexLow &&
       info1.nFileIndexHigh==info2.nFileIndexHigh &&
       info1.dwVolumeSerialNumber==info2.dwVolumeSerialNumber) same=TRUE;
    }
  CloseHandle(hFile1);
  CloseHandle(hFile2);
  return same;
#endif
  }


// Split a string into substrings delimited by a given character
FXchar *fxsplit(FXchar*& s,FXchar c){
  FXchar *t=s;
  if(s && (s=strchr(s,c))) *s++='\0';
  return t;
  }


// Return the shortest path equivalent to pathname (remove . and ..)
FXchar *fxshortestpath(FXchar *result,const FXchar *pathname){
  FXchar path[MAXPATHLEN+1],*cur,*last,*part,*tmp;
  if(!result){fxerror("fxshortestpath: NULL result argument.\n");}
  if(!pathname){fxerror("fxshortestpath: NULL pathname argument.\n");}
  strcpy(path,pathname);
  cur=result;
  *cur='\0';
  tmp=path;
  last=NULL;
  if(*tmp==PATHSEP){ 
    *cur++=PATHSEP;
    *cur='\0';
    tmp++; 
    }
  while((part=fxsplit(tmp,PATHSEP))){
    if(strcmp(part,".")==0)
      ;
    else if(strcmp(part,"..")==0 && (last=strrchr(result,PATHSEP))){
      if(last>result)
        cur=last;
      else
        cur=last+1;
      *cur='\0';
      } 
    else{
      if(cur>result && *(cur-1)!=PATHSEP) *cur++=PATHSEP;
      strcpy(cur,part);
      cur+=strlen(part);
      }
    }
  
  // Remove trailing backslash(es)
  while(result<cur-1 && *(cur-1)==PATHSEP && *(cur-2)!=':'){
    *--cur='\0';
    }

#ifdef WIN32
  // Use GetFullPathName() to ensure that the correct drive letter is used
  DWORD dwSize=GetFullPathName(result,0,NULL,NULL);
  if(dwSize>0){
    FXchar *buffer;
    if(FXMALLOC(&buffer,FXchar,dwSize+1)){
      if(0!=GetFullPathName(result,dwSize,buffer,NULL))
	strcpy(result,buffer);
      FXFREE(&buffer);
      }
    }
#endif

  return result;
  }


// Return directory one level above given one
FXchar* fxupdir(FXchar* result,const FXchar *dirname){
  FXchar *ptr;
  
  // Tests
  if(!result){fxerror("fxupdir: NULL result argument.\n");}
  if(!dirname){fxerror("fxupdir: NULL dirname argument.\n");}

  // Copy
  strcpy(result,dirname);

  // Find the last PATHSEP
  ptr=strrchr(result,PATHSEP);
  
  // If dirname had a trailing PATHSEP, lop it off and try again
  if(ptr && *(ptr+1)=='\0'){*ptr='\0';ptr=strrchr(result,PATHSEP);}

  // Try lop off the last directory
  if(ptr){
    
    // Is this the toplevel directory already?
    // The test is a bit complicated as we need to take a possible drive letter into account
    if((result[0]==PATHSEP && ptr==result) || (isalpha(result[0]) && result[1]==':' && result[2]==PATHSEP && ptr==result+2)){
      *(ptr+1)='\0';
      }
    else{
      *ptr='\0';
      }
    }
  return result;
  }


// Test if pathname is the toplevel directory
FXbool fxistopdir(const FXchar* path){
  return path==NULL || (path[0]==PATHSEP && path[1]=='\0') || (isalpha(path[0]) && path[1]==':' && path[2]==PATHSEP && path[3]=='\0');
  }


// Get current working directory
FXchar* fxgetcurrentdir(FXchar *result){
  if(!result){fxerror("fxgetcurrentdir: NULL result argument.\n");}
#ifndef WIN32
  if(getcwd(result,MAXPATHLEN)==NULL){fxerror("fxgetcurrentdir: failed.\n");}
#else
  GetCurrentDirectory(MAXPATHLEN,result);
#endif
  return result;
  }


// Make a directory
FXint fxmkdir(const FXchar* pathname,FXuint mode){
#if defined(WIN32) && !defined(__CYGWIN__)
  return _mkdir(pathname);
#else
  return mkdir(pathname,mode);
#endif
  }


// Get home directory
FXchar* fxgethomedir(FXchar *result){
  static FXchar* home=NULL;
  if(!result){fxerror("fxgethomedir: NULL result argument.\n");}
  if(!home){home=getenv("HOME");if(!home) home="";}
  strcpy(result,home);
  return result;
  }


// Translate filename to an absolute pathname; ~ in filename is expanded,
// and if the resulting pathname is still relative, basename is prepended
FXchar *fxabspath(FXchar *result,const FXchar *basename,const FXchar *filename){
  char s[MAXPATHLEN+1],t[MAXPATHLEN+1],cwd[MAXPATHLEN+1];
  
  // Test
  if(!result){fxerror("fxabspath: NULL result argument.\n");}

  // Expand tilde's
  fxexpand(s,filename);
  
  // Absolute path may be preceded by a drive letter as in C:\blabla\blabla
  // or just \blabla\blabla or on unix just /blabla/blabla
  if(!(s[0]==PATHSEP || (isalpha(s[0]) && s[1]==':' && s[2]==PATHSEP))){
    if(basename==NULL || *basename=='\0'){
      fxgetcurrentdir(cwd);
      basename=cwd;
      }
    fxpathname(t,basename,s);
    return fxshortestpath(result,t);
    }
  return fxshortestpath(result,s);
  }


// Search path for file name. A ~ in path is expanded. If name is absolute
// it is returned unchanged. Otherwise the absolute name is returned in
// result. If name is not found on path, NULL is returned. 
FXchar *fxsearchpath(FXchar *result,const FXchar *path,const FXchar *name){
  FXchar fullname[MAXPATHLEN+1];
  FXchar basedir[MAXPATHLEN+1];
  const FXchar *s,*t;
  if(!result){fxerror("fxsearchpath: NULL result argument.\n");}
  if(!path){fxerror("fxsearchpath: NULL path argument.\n");}
  if(!name){fxerror("fxsearchpath: NULL name argument.\n");}
  if(name[0]==PATHSEP || (isalpha(name[0]) && name[1]==':' && name[2]==PATHSEP)){
    strcpy(result,name);
    if(fxexists(result)) return result;
    return NULL;
    }
  fxgetcurrentdir(basedir);
  for(s=path; *s; s=t){
    int l;
    t=strchr(s,PATHLISTSEP);
    if(!t) t=s+strlen(s);
    if(s!=t){
      if(s[0]=='.'){ 
        if(t==s+1) s=t; else if(s[1]==PATHSEP) s+=2; 
        }
      l=t-s;
      strncpy(fullname,s,l);
      if(l>0 && fullname[l-1]!=PATHSEP){ fullname[l]=PATHSEP; l++; }
      strcpy(fullname+l,name);
      fxabspath(result,basedir,fullname);
      if(fxexists(result)) return result;
      }
    if(*t) t++;
    }
  return NULL;
  }


// Get process id
FXint fxgetpid(){
#ifndef WIN32
  return getpid();
#else
  return (int)GetCurrentProcessId();
#endif
  }


// Convert RGB to HSV
void fxrgb_to_hsv(FXfloat& h,FXfloat& s,FXfloat& v,FXfloat r,FXfloat g,FXfloat b){
  FXfloat t,delta;
  v=r; if(g>v) v=g; if(b>v) v=b;
  t=r; if(g<t) t=g; if(b<t) t=b;
  delta=v-t;
  if(v!=0.0) 
    s=delta/v; 
  else 
    s=0.0;
  if(s==0.0){
    h=0.0;
    }
  else{
    if(r==v) 
      h=(g-b)/delta;
    else if(g==v) 
      h=2.0f+(b-r)/delta;
    else if(b==v) 
      h=4.0f+(r-g)/delta;
    h=h*60.0f;
    if(h<0.0) h=h+360;
    }
  }


// Convert to RGB
void fxhsv_to_rgb(FXfloat& r,FXfloat& g,FXfloat& b,FXfloat h,FXfloat s,FXfloat v){
  FXfloat f,w,q,t;
  FXint i;
  if(s==0.0){
    r=v;
    g=v;
    b=v;
    }
  else{
    if(h==360.0) h=0.0;
    h=h/60.0f;
    i=(FXint)h;
    f=h-i;
    w=v*(1.0f-s);
    q=v*(1.0f-(s*f));
    t=v*(1.0f-(s*(1.0f-f)));
    switch(i){
      case 0: r=v; g=t; b=w; break;
      case 1: r=q; g=v; b=w; break;
      case 2: r=w; g=v; b=t; break;
      case 3: r=w; g=q; b=v; break;
      case 4: r=t; g=w; b=v; break;
      case 5: r=v; g=w; b=q; break;
      }
    }
  }


// Calculate a hash value from a string; algorithm same as in perl
FXint fxstrhash(const FXchar* str){
  register FXint h=0;
  register FXint g;
  while(*str) {
    h=(h<<4)+*str++;
    g=h&0xF0000000;
    if(g) h^=g>>24;
    h&=0x0fffffff;
    }
  FXASSERT(h<=0x0fffffff);
  return h;
  }


// Parse font description
FXbool fxparsefontdesc(FXFontDesc& fontdesc,const FXchar* string){
  return string && 7==sscanf(string,"%s %u %u %u %u %u %u",fontdesc.face,&fontdesc.size,&fontdesc.weight,&fontdesc.slant,&fontdesc.encoding,&fontdesc.setwidth,&fontdesc.flags);
  }


// Unparse font description
FXbool fxunparsefontdesc(FXchar *string,const FXFontDesc& fontdesc){
  sprintf(string,"%s %u %u %u %u %u %u",fontdesc.face,fontdesc.size,fontdesc.weight,fontdesc.slant,fontdesc.encoding,fontdesc.setwidth,fontdesc.flags);
  return TRUE;
  }


#ifdef FX_NATIVE_WIN32

// Return the current state of the modifier keys and mouse buttons
unsigned int fxmodifierkeys(){
  FXuint state=0;
  if(GetKeyState(VK_SHIFT)<0) state|=SHIFTMASK;
  if(GetKeyState(VK_CAPITAL)>0) state|=CAPSLOCKMASK;
  if(GetKeyState(VK_CONTROL)<0) state|=CONTROLMASK;
  if(GetKeyState(VK_MENU)<0) state|=ALTMASK;
  if(GetKeyState(VK_NUMLOCK)>0) state|=NUMLOCKMASK;
  if(GetKeyState(VK_SCROLL)>0) state|=SCROLLLOCKMASK;
  if(GetKeyState(VK_LBUTTON)<0) state|=LEFTBUTTONMASK;
  if(GetKeyState(VK_MBUTTON)<0) state|=MIDDLEBUTTONMASK;
  if(GetKeyState(VK_RBUTTON)<0) state|=RIGHTBUTTONMASK;
  return state;
  }


// Convert font size (in decipoints) to device logical units
int fxpointsize_to_height(HDC hdc,unsigned size){
  SetGraphicsMode(hdc,GM_ADVANCED);
  XFORM xform;
  ModifyWorldTransform(hdc,&xform,MWT_IDENTITY);
  SetViewportOrgEx(hdc,0,0,NULL);
  SetWindowOrgEx(hdc,0,0,NULL);
  FLOAT cyDpi=(FLOAT)(25.4*GetDeviceCaps(hdc,VERTRES)/GetDeviceCaps(hdc,VERTSIZE));
  POINT pt;
  pt.x=0;
  pt.y=(int)(size*cyDpi/72);
  DPtoLP(hdc,&pt,1);
  return -(int)(fabs(pt.y)/10.0+0.5);
  }


// Convert logical units to decipoints
unsigned fxheight_to_pointsize(HDC hdc,int height){
  SetGraphicsMode(hdc,GM_ADVANCED);
  XFORM xform;
  ModifyWorldTransform(hdc,&xform,MWT_IDENTITY);
  SetViewportOrgEx(hdc,0,0,NULL);
  SetWindowOrgEx(hdc,0,0,NULL);
  FLOAT cyDpi=(FLOAT)(25.4*GetDeviceCaps(hdc,VERTRES)/GetDeviceCaps(hdc,VERTSIZE));
  POINT pt;
  pt.x=0;
  pt.y=10*height;
  LPtoDP(hdc,&pt,1);
  return (FXuint)(72*pt.y/cyDpi);
  }

#endif
