/* $Id: filter.c,v 1.15 1999/03/20 15:28:22 becka Exp $
******************************************************************************

   Filter-mouse - generic mouse event generator.

   Copyright (C) 1998 Andreas Beck      [becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdlib.h>
#include <string.h>

#include <ggi/internal/gii-dl.h>

static void send_devinfo(gii_input *inp);

enum maptype { 
	MAP_NONE, 
	MAP_KEY_TO_BUTTON,
	MAP_KEY_TO_REL,
	MAP_KEY_TO_ABS,
	MAP_ACCELERATE
};

typedef struct mapping_entry {

	struct mapping_entry *next;

	enum maptype type;
	uint32	number;		/* key/axis */
	uint32	modifier_mask;	/* all modifiers in mask */
	uint32	modifier_value;	/* must match value */
	uint32	to;		/* key/axis */
	sint32	speed;		/* ticks to go */
	union {
		struct accel {
			int treshold,linear,quadratic;
		} accel[4];
	} priv;
} mapping_entry;

static mapping_entry test[]= {
	{
		NULL,
		MAP_KEY_TO_BUTTON,
		
		GIIK_Enter,
		(1<<GII_KM_ALT),
		(1<<GII_KM_ALT),

		GII_PBUTTON_LEFT,
		0,
	},
	{
		NULL,
		MAP_KEY_TO_REL,

		GIIK_Left,
		(1<<GII_KM_ALT),
		(1<<GII_KM_ALT),

		0,
		-1,
	},
	{
		NULL,
		MAP_KEY_TO_REL,

		GIIK_Right,
		(1<<GII_KM_ALT),
		(1<<GII_KM_ALT),

		0,
		1,
	},
	{
		NULL,
		MAP_KEY_TO_REL,

		GIIK_Up,
		(1<<GII_KM_ALT),
		(1<<GII_KM_ALT),

		1,
		-1,
	},
	{
		NULL,
		MAP_KEY_TO_REL,

		GIIK_Down,
		(1<<GII_KM_ALT),
		(1<<GII_KM_ALT),

		1,
		1,
	},
	{
		NULL,
		MAP_ACCELERATE,
		
		0,
		0,0,
		0,0,
		{
			{	{ 0, 400,   0 },
				{ 0, 400,   0 },
				{ 0, 400,   0 },
				{ 0, 400,   0 }
			}
		}
	}
};

typedef struct {
	mapping_entry *entry;
} fmouse_priv;

static void
fmouse_send_pbutton(gii_input *inp, uint8 type, uint32 nr)
{
	gii_event ev;

	_giiEventBlank(&ev, sizeof(gii_pbutton_event));
	ev.pbutton.type = type;
	ev.pbutton.size = sizeof(gii_pbutton_event);
	ev.pbutton.origin = inp->origin;
	ev.pbutton.target = GII_EV_TARGET_ALL;
	ev.pbutton.button = nr;
	_giiEvQueueAdd(inp, &ev);
}

static void
fmouse_send_pmoveall(gii_input *inp, uint8 type, sint32 x, sint32 y, sint32 z, sint32 w)
{
	gii_event ev;

	_giiEventBlank(&ev, sizeof(gii_pmove_event));
	ev.pmove.type = type;
	ev.pmove.size = sizeof(gii_pmove_event);
	ev.pmove.origin = inp->origin;
	ev.pmove.target = GII_EV_TARGET_ALL;
	ev.pmove.x = x;
	ev.pmove.y = y;
	ev.pmove.z = z;
	ev.pmove.wheel = w;
	_giiEvQueueAdd(inp, &ev);
}

static void
fmouse_send_pmove(gii_input *inp, uint8 type, uint32 nr, sint32 speed)
{
	int x,y,z,w;
	x=y=z=w=0;
	switch(nr)
	{ case 0:
	  default:
		x = speed;break;
	  case 1:
		y = speed;break;
	  case 2:
		z = speed;break;
	  case 3:
		w = speed;break;
	}
	fmouse_send_pmoveall(inp, type, x, y, z, w);
}

#define abs(x) (x<0 ? -x : x)

void doaccel(gii_input *inp,gii_event *event,struct accel *acc)
{
	int x,w,erg[4];
	
	for (x=0;x<4;x++) {
		w=0;
		switch(x) {
			case 0:w=event->pmove.x;break;
			case 1:w=event->pmove.y;break;
			case 2:w=event->pmove.z;break;
			case 3:w=event->pmove.wheel;break;
		}
		erg[x]=abs(w)<acc[x].treshold ? 0 : 
		       (double)w*acc[x].linear/100.0 +
		       (double)w*abs(w)*acc[x].quadratic/10000.0;
	}
	fmouse_send_pmoveall(inp, evPtrRelative, 
				erg[0], erg[1], erg[2], erg[3]);
	return;
}

static int
GII_fmouse_handler(gii_input *inp, gii_event *event)
{
	fmouse_priv   *priv = inp->priv;
	mapping_entry *entry;
	int ret = 0;
	static int di_sent=0;
	
	if (di_sent==0) { di_sent=1;send_devinfo(inp); }
	
	GIIDPRINT_MISC("filter-mouse: Filt check.\n");
	if (event->any.origin==inp->origin) 
		return 0;	/* avoid recursion ! */
	GIIDPRINT_MISC("filter-mouse: Real check.\n");

	entry = priv->entry;
	while (entry) {
		switch (entry->type) {
		case MAP_KEY_TO_BUTTON:
			if ((event->any.type==evKeyPress ||
			     event->any.type==evKeyRelease) &&
			    event->key.sym==entry->number &&
			    (event->key.modifiers & entry->modifier_mask)
			    == entry->modifier_value) {
				fmouse_send_pbutton(inp, event->any.type
						    == evKeyRelease
						    ? evPtrButtonRelease
						    : evPtrButtonPress,
						    entry->to);
				GIIDPRINT_MISC("filter-mouse:Have Key-to-button.\n");
			}
			break;
		case MAP_KEY_TO_REL:
			if ((event->any.type==evKeyPress ||
			       event->any.type==evKeyRepeat) &&
			    event->key.sym==entry->number &&
			    (event->key.modifiers & entry->modifier_mask)
			    == entry->modifier_value) {
				fmouse_send_pmove(inp, evPtrRelative,
						  entry->to, entry->speed);
				GIIDPRINT_MISC("filter-mouse:Have Key-to-Rel Press.\n");
			}
			break;
		case MAP_KEY_TO_ABS:
			if ((event->any.type==evKeyPress ||
			     event->any.type==evKeyRepeat) &&
			    event->key.sym==entry->number &&
			    (event->key.modifiers & entry->modifier_mask) 
			    == entry->modifier_value ) {
				fmouse_send_pmove(inp, evPtrAbsolute,
						  entry->to, entry->speed);
				GIIDPRINT_MISC("filter-mouse:Have Key-to-Abs Press.\n");
			}
			break;
		case MAP_ACCELERATE:
			{
				if ((event->any.type==evPtrRelative)) {
	
					GIIDPRINT_MISC("filter-mouse:Have accelerate.\n");
					doaccel(inp, event, entry->priv.accel);
					ret=1; /* Eat old event */
				}
			}
			break;
		default:
			GIIDPRINT_MISC("filter-mouse: illegal mapping.\n");
			break;
		}
		entry = entry->next;
	}
	
	return ret;
}

static int GII_fmouse_close(gii_input *inp)
{
	GIIDPRINT_MISC("GII_fmouse_close(%p) called\n", inp);

	free(inp->priv);

	return 0;
}

static gii_cmddata_getdevinfo devinfo =
{
	"Mouse filter",	/* long device name */
	"mouf",		/* shorthand */
	emPointer,	/* can_generate */
	4,		/* num_buttons	(no supported device have more) */
	0		/* num_axes 	(only for valuators) */
};

static gii_deviceinfo xdevinfo =
{
	NULL,
	0,
	&devinfo,
	NULL
};

static void send_devinfo(gii_input *inp)
{
	gii_event ev;
	gii_cmddata_getdevinfo *dinfo;
	int size = sizeof(gii_cmd_nodata_event)+sizeof(gii_cmddata_getdevinfo);

	_giiEventBlank(&ev, size);
	
	ev.any.size   = size;
	ev.any.type   = evCommand;
	ev.any.origin = inp->origin;
	ev.cmd.code   = GII_CMDCODE_GETDEVINFO;

	dinfo = (gii_cmddata_getdevinfo *) &ev.cmd.data;
	*dinfo = devinfo;

	_giiEvQueueAdd(inp, &ev);
}


int GIIdlinit(gii_input *inp, const char *args, void *argptr)
{
	fmouse_priv   *priv;

	GIIDPRINT_MISC("filter-mouse init(%p, \"%s\") called\n", inp,
		       args ? args : "");

	priv = malloc(sizeof(fmouse_priv));
        if (priv == NULL) return GGI_ENOMEM;

	test[0].next = &test[1];
	test[1].next = &test[2];
	test[2].next = &test[3];
	test[3].next = &test[4];
	test[4].next = &test[5];
	priv->entry = &test[0];	/* TEST */
	
	inp->priv       = priv;
	inp->GIIhandler = GII_fmouse_handler;
	inp->GIIclose   = GII_fmouse_close;
	inp->devinfo    = &xdevinfo;
	xdevinfo.origin = inp->origin;

	GIIDPRINT_MISC("filter-mouse fully up\n");

	return 0;
}
