///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2000, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#include <string.h>
#include <stdio.h>

#include "config.h"

#if USE_PRIVATE_HEAPS
#include "privateheap.h"
#define HEAPTYPE privateHeap
#else
#define HEAPTYPE threadHeap
#include "threadheap.h"
#endif

#include "processheap.h"

hoardLockType processHeap::_bufferLock;
superblock * processHeap::_recycleBin;


processHeap::processHeap (void)
#if HEAP_FRAG_STATS
  : _currentAllocated (0),
  _currentRequested (0),
  _maxAllocated (0),
  _inUseAtMaxAllocated (0),
  _maxRequested (0)
#endif
{
  _recycleBin = NULL;
  int i;
  // The process heap is heap 0.
  setIndex (0);
  for (i = 0; i < MAX_HEAPS; i++) {
    // Set every thread's process heap to this one.
    theap[i].setpHeap (this);
    // Set every thread heap's index.
    theap[i].setIndex (i + 1);
  }
#if HEAP_LOG
  for (i = 0; i < MAX_HEAPS + 1; i++) {
    char fname[255];
    sprintf (fname, "log%d", i);
    unlink (fname);
    _log[i].open (fname);
  }
#endif
#if HEAP_FRAG_STATS
  hoardLockInit (_statsLock);
#endif
  hoardLockInit (_bufferLock);
}


// Obtain a superblock buffer from our store.  We always ask the
// system for at least the REFILL number of superblocks (above), and
// then parcel them out to the threads.
// Returns 1 if this is a recycled superblock.
int processHeap::getSuperblockBuffer (char *& ptr)
{
  static char * superblocksBuffer = NULL;
  static int superblocksBufferCount = 0;

  hoardLock (_bufferLock);

  // If we're out of buffer space,
  // refill with a chunk that can hold the refill number of superblocks
  // if there are none available in the recycle bin.

  int isRecycled = 0;
  if (_recycleBin) {
    // Reuse a superblock.
    ptr = (char *) _recycleBin;
    ((superblock *) ptr)->remove();
    _recycleBin = _recycleBin->getNext();
    isRecycled = 1;
  } else {
    if (superblocksBufferCount == 0) {
      // Refill the superblock buffer.
      superblocksBuffer
	= (char *) hoardSbrk (SUPERBLOCK_SIZE * REFILL_NUMBER_OF_SUPERBLOCKS);
      superblocksBufferCount = REFILL_NUMBER_OF_SUPERBLOCKS;
    }
    // Carve out a superblock and return it.
    ptr = superblocksBuffer;
    superblocksBuffer += SUPERBLOCK_SIZE;
    superblocksBufferCount--;
  }

  hoardUnlock (_bufferLock);
  return isRecycled;
}


// Print out statistics information.
void processHeap::stats (void) {
#if HEAP_STATS
  int umax = 0;
  int amax = 0;
  for (int j = 0; j < MAX_HEAPS; j++) {
    for (int i = 0; i < SIZE_CLASSES; i++) {
      // printf ("[%d] U=%d, A=%d\n", i, theap[j].maxInUse(i), theap[j].maxAllocated(i));
      amax += theap[j].maxAllocated(i) * sizeFromClass (i);
      umax += theap[j].maxInUse(i) * sizeFromClass (i);
    }
  }
  printf ("Amax <= %d, Umax <= %d\n", amax, umax);
#if HEAP_FRAG_STATS
  amax = getMaxAllocated();
  umax = getMaxRequested();
  printf ("Maximum allocated = %d\nMaximum in use = %d\nIn use at max allocated = %d\n", amax, umax, getInUseAtMaxAllocated());
  printf ("Still in use = %d\n", _currentRequested);
  printf ("Fragmentation (3) = %f\n", (float) amax / (float) getInUseAtMaxAllocated());
  printf ("Fragmentation (4) = %f\n", (float) amax / (float) umax);
#endif

#endif // HEAP_STATS
}



#if HEAP_FRAG_STATS
void processHeap::setAllocated (int requestedSize,
				int actualSize)
{
  hoardLock (_statsLock);
  // printf ("setAllocated (%d, %d)\n", requestedSize, actualSize);
  _currentRequested += requestedSize;
  _currentAllocated += actualSize;
  if (_currentRequested > _maxRequested) {
    _maxRequested = _currentRequested;
  }
  if (_currentAllocated > _maxAllocated) {
    _maxAllocated = _currentAllocated;
    _inUseAtMaxAllocated = _currentRequested;
  }
  hoardUnlock (_statsLock);
}


void processHeap::setDeallocated (int requestedSize,
				  int actualSize)
{
  hoardLock (_statsLock);
  _currentRequested -= requestedSize;
  _currentAllocated -= actualSize;
  hoardUnlock (_statsLock);
}
#endif


// free (ptr, pheap):
//   inputs: a pointer to an object allocated by malloc().
//   side effects: returns the block to the object's superblock;
//                 updates the thread heap's statistics;
//                 may release the superblock to the process heap.

void processHeap::free (void * ptr)
{

  // Return if ptr is 0.
  // This is the behavior prescribed by the standard.
  if (ptr == 0) {
    return;
  }

  // Find the block and superblock corresponding to this ptr.

  block * b = (block *) ptr - 1;
  assert (b->isValid());

  // Check to see if this block came from a memalign() call.
  if (((unsigned long) b->getNext() & 1) == 1) {
    b = (block *) ((unsigned long) b->getNext() & ~1);
    assert (b->isValid());
  }    

  b->markFree();

  superblock * sb = b->getSuperblock();
  assert (sb);
  assert (sb->isValid());

#if HEAP_FRAG_STATS
  setDeallocated (b->getRequestedSize(), 0);
#endif

  const int sizeclass = sb->getBlockSizeClass();

  //
  // Return the block to the superblock,
  // find the heap that owns this superblock
  // and update its statistics.
  //

  hoardHeap * owner = getAndLockOwner (sb);

#if HEAP_LOG
  MemoryRequest m;
  m.free (ptr);
  getLog (getIndex()).append(m);
#endif

  owner->freeBlock (b, sb, sizeclass, this);
  owner->unlock();
}
