/* File "hangul.c":
 * Handles conversion of KSC5601 hangul characters
 * and romanised hangul to internal format (and back). */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "basic.h"
#include "pools.h"
#include "tries.h"

#undef GLOBAL
#define GLOBAL
#include "hangul.h"

/* variables ================================================================*/

GLOBAL bool_t convert_to_ksc = TRUE; /* default: convert hancode to KSC. */

/* In this conversion module, we use four representations of
 * Hangul characters and syllables:
 * 1. KSC5601 code, used by a couple of programs (MULE for example),
 *    which codes single letters as well as whole syllables by codes
 *    that use two characters in the range of 0xa1 to 0xfe.
 * 2. Trigem code, which decodes syllables and letters by a 16-bit word
 *    divided as follows:
 *    (MSB) | S  |  I  I  I  I  I | V V V V V | F F F F F | (LSB)
 *          | 15 | 14 13 12 11 10 | 9 8 7 6 5 | 4 3 2 1 0 |
 *    where "S" is set if the code represents a syllable (no single letter)
 *          "IIIII" is the code of the initial consonant(s)
 *          (see "initial_consonants")
 *          "VVVVV" is the code of the vowel (see "vowels")
 *          "FFFFF" is the code of the final consonant(s)
 *          (see "final_consonants")
 * 3. Hancode, which is the internal, letter oriented code. The letters are
 *    represented by codes in the range of 0xa1-0xba, and 0xa0 marks the
 *    beginning of a syllable.
 * 4. Roman code, which is similar to Hancode, but it uses the latin letters
 *    and adopts the Yale standard for Hangul romanization.
 *    Here, every syllable is leaded by a dot ".". */

/* the KSC table ============================================================*/

/* For each character in the KSC5601 character set,
 * <ksc_table> defines a hancode string. */
#define KSC_TABLE_SIZE 2401 /* number of entries in the kcs_table */

/* For each KSC5601 two-byte code, a hancode string is assigned.
 * This table is sorted in order of the KSC codes. */
LOCAL struct 
{ 
  u_short_t trigem; 
  u_short_t ksc; 
  string_t hancode; 
} ksc_table[KSC_TABLE_SIZE] =
{
  {0x0841, 0xa4a1, NULL}, {0x0c41, 0xa4a2, NULL}, {0x0444, 0xa4a3, NULL}, 
  {0x1041, 0xa4a4, NULL}, {0x0446, 0xa4a5, NULL}, {0x0447, 0xa4a6, NULL}, 
  {0x1441, 0xa4a7, NULL}, {0x1841, 0xa4a8, NULL}, {0x1c41, 0xa4a9, NULL},
  {0x044a, 0xa4aa, NULL}, {0x044b, 0xa4ab, NULL}, {0x044c, 0xa4ac, NULL}, 
  {0x044d, 0xa4ad, NULL}, {0x044e, 0xa4ae, NULL}, {0x044f, 0xa4af, NULL}, 
  {0x0450, 0xa4b0, NULL}, {0x2041, 0xa4b1, NULL}, {0x2441, 0xa4b2, NULL}, 
  {0x2841, 0xa4b3, NULL}, {0x0454, 0xa4b4, NULL}, {0x2c41, 0xa4b5, NULL}, 
  {0x3041, 0xa4b6, NULL}, {0x3441, 0xa4b7, NULL}, {0x3841, 0xa4b8, NULL}, 
  {0x3c41, 0xa4b9, NULL}, {0x4041, 0xa4ba, NULL}, {0x4441, 0xa4bb, NULL}, 
  {0x4841, 0xa4bc, NULL}, {0x4c41, 0xa4bd, NULL}, {0x5041, 0xa4be, NULL}, 
  {0x0461, 0xa4bf, NULL}, {0x0481, 0xa4c0, NULL}, {0x04a1, 0xa4c1, NULL}, 
  {0x04c1, 0xa4c2, NULL}, {0x04e1, 0xa4c3, NULL}, {0x0541, 0xa4c4, NULL}, 
  {0x0561, 0xa4c5, NULL}, {0x0581, 0xa4c6, NULL}, {0x05a1, 0xa4c7, NULL}, 
  {0x05c1, 0xa4c8, NULL}, {0x05e1, 0xa4c9, NULL}, {0x0641, 0xa4ca, NULL}, 
  {0x0661, 0xa4cb, NULL}, {0x0681, 0xa4cc, NULL}, {0x06a1, 0xa4cd, NULL}, 
  {0x06c1, 0xa4ce, NULL}, {0x06e1, 0xa4cf, NULL}, {0x0741, 0xa4d0, NULL}, 
  {0x0761, 0xa4d1, NULL}, {0x0781, 0xa4d2, NULL}, {0x07a1, 0xa4d3, NULL},
  {0x8861, 0xb0a1, NULL}, {0x8862, 0xb0a2, NULL}, {0x8865, 0xb0a3, NULL},
  {0x8868, 0xb0a4, NULL}, {0x8869, 0xb0a5, NULL}, {0x886a, 0xb0a6, NULL},
  {0x886b, 0xb0a7, NULL}, {0x8871, 0xb0a8, NULL}, {0x8873, 0xb0a9, NULL},
  {0x8874, 0xb0aa, NULL}, {0x8875, 0xb0ab, NULL}, {0x8876, 0xb0ac, NULL},
  {0x8877, 0xb0ad, NULL}, {0x8878, 0xb0ae, NULL}, {0x8879, 0xb0af, NULL},
  {0x887b, 0xb0b0, NULL}, {0x887c, 0xb0b1, NULL}, {0x887d, 0xb0b2, NULL},
  {0x8881, 0xb0b3, NULL}, {0x8882, 0xb0b4, NULL}, {0x8885, 0xb0b5, NULL},
  {0x8889, 0xb0b6, NULL}, {0x8891, 0xb0b7, NULL}, {0x8893, 0xb0b8, NULL},
  {0x8895, 0xb0b9, NULL}, {0x8896, 0xb0ba, NULL}, {0x8897, 0xb0bb, NULL},
  {0x88a1, 0xb0bc, NULL}, {0x88a2, 0xb0bd, NULL}, {0x88a5, 0xb0be, NULL},
  {0x88a9, 0xb0bf, NULL}, {0x88b5, 0xb0c0, NULL}, {0x88b7, 0xb0c1, NULL},
  {0x88c1, 0xb0c2, NULL}, {0x88c5, 0xb0c3, NULL}, {0x88c9, 0xb0c4, NULL},
  {0x88e1, 0xb0c5, NULL}, {0x88e2, 0xb0c6, NULL}, {0x88e5, 0xb0c7, NULL},
  {0x88e8, 0xb0c8, NULL}, {0x88e9, 0xb0c9, NULL}, {0x88eb, 0xb0ca, NULL},
  {0x88f1, 0xb0cb, NULL}, {0x88f3, 0xb0cc, NULL}, {0x88f5, 0xb0cd, NULL},
  {0x88f6, 0xb0ce, NULL}, {0x88f7, 0xb0cf, NULL}, {0x88f8, 0xb0d0, NULL},
  {0x88fb, 0xb0d1, NULL}, {0x88fc, 0xb0d2, NULL}, {0x88fd, 0xb0d3, NULL},
  {0x8941, 0xb0d4, NULL}, {0x8945, 0xb0d5, NULL}, {0x8949, 0xb0d6, NULL},
  {0x8951, 0xb0d7, NULL}, {0x8953, 0xb0d8, NULL}, {0x8955, 0xb0d9, NULL},
  {0x8956, 0xb0da, NULL}, {0x8957, 0xb0db, NULL}, {0x8961, 0xb0dc, NULL},
  {0x8962, 0xb0dd, NULL}, {0x8963, 0xb0de, NULL}, {0x8965, 0xb0df, NULL},
  {0x8968, 0xb0e0, NULL}, {0x8969, 0xb0e1, NULL}, {0x8971, 0xb0e2, NULL},
  {0x8973, 0xb0e3, NULL}, {0x8975, 0xb0e4, NULL}, {0x8976, 0xb0e5, NULL},
  {0x8977, 0xb0e6, NULL}, {0x897b, 0xb0e7, NULL}, {0x8981, 0xb0e8, NULL},
  {0x8985, 0xb0e9, NULL}, {0x8989, 0xb0ea, NULL}, {0x8993, 0xb0eb, NULL},
  {0x8995, 0xb0ec, NULL}, {0x89a1, 0xb0ed, NULL}, {0x89a2, 0xb0ee, NULL},
  {0x89a5, 0xb0ef, NULL}, {0x89a8, 0xb0f0, NULL}, {0x89a9, 0xb0f1, NULL},
  {0x89ab, 0xb0f2, NULL}, {0x89ad, 0xb0f3, NULL}, {0x89b0, 0xb0f4, NULL},
  {0x89b1, 0xb0f5, NULL}, {0x89b3, 0xb0f6, NULL}, {0x89b5, 0xb0f7, NULL},
  {0x89b7, 0xb0f8, NULL}, {0x89b8, 0xb0f9, NULL}, {0x89c1, 0xb0fa, NULL},
  {0x89c2, 0xb0fb, NULL}, {0x89c5, 0xb0fc, NULL}, {0x89c9, 0xb0fd, NULL},
  {0x89cb, 0xb0fe, NULL}, {0x89d1, 0xb1a1, NULL}, {0x89d3, 0xb1a2, NULL},
  {0x89d5, 0xb1a3, NULL}, {0x89d7, 0xb1a4, NULL}, {0x89e1, 0xb1a5, NULL},
  {0x89e5, 0xb1a6, NULL}, {0x89e9, 0xb1a7, NULL}, {0x89f3, 0xb1a8, NULL},
  {0x89f6, 0xb1a9, NULL}, {0x89f7, 0xb1aa, NULL}, {0x8a41, 0xb1ab, NULL},
  {0x8a42, 0xb1ac, NULL}, {0x8a45, 0xb1ad, NULL}, {0x8a49, 0xb1ae, NULL},
  {0x8a51, 0xb1af, NULL}, {0x8a53, 0xb1b0, NULL}, {0x8a55, 0xb1b1, NULL},
  {0x8a57, 0xb1b2, NULL}, {0x8a61, 0xb1b3, NULL}, {0x8a65, 0xb1b4, NULL},
  {0x8a69, 0xb1b5, NULL}, {0x8a73, 0xb1b6, NULL}, {0x8a75, 0xb1b7, NULL},
  {0x8a81, 0xb1b8, NULL}, {0x8a82, 0xb1b9, NULL}, {0x8a85, 0xb1ba, NULL},
  {0x8a88, 0xb1bb, NULL}, {0x8a89, 0xb1bc, NULL}, {0x8a8a, 0xb1bd, NULL},
  {0x8a8b, 0xb1be, NULL}, {0x8a90, 0xb1bf, NULL}, {0x8a91, 0xb1c0, NULL},
  {0x8a93, 0xb1c1, NULL}, {0x8a95, 0xb1c2, NULL}, {0x8a97, 0xb1c3, NULL},
  {0x8a98, 0xb1c4, NULL}, {0x8aa1, 0xb1c5, NULL}, {0x8aa2, 0xb1c6, NULL},
  {0x8aa5, 0xb1c7, NULL}, {0x8aa9, 0xb1c8, NULL}, {0x8ab6, 0xb1c9, NULL},
  {0x8ab7, 0xb1ca, NULL}, {0x8ac1, 0xb1cb, NULL}, {0x8ad5, 0xb1cc, NULL},
  {0x8ae1, 0xb1cd, NULL}, {0x8ae2, 0xb1ce, NULL}, {0x8ae5, 0xb1cf, NULL},
  {0x8ae9, 0xb1d0, NULL}, {0x8af1, 0xb1d1, NULL}, {0x8af3, 0xb1d2, NULL},
  {0x8af5, 0xb1d3, NULL}, {0x8b41, 0xb1d4, NULL}, {0x8b45, 0xb1d5, NULL},
  {0x8b49, 0xb1d6, NULL}, {0x8b61, 0xb1d7, NULL}, {0x8b62, 0xb1d8, NULL},
  {0x8b65, 0xb1d9, NULL}, {0x8b68, 0xb1da, NULL}, {0x8b69, 0xb1db, NULL},
  {0x8b6a, 0xb1dc, NULL}, {0x8b71, 0xb1dd, NULL}, {0x8b73, 0xb1de, NULL},
  {0x8b75, 0xb1df, NULL}, {0x8b77, 0xb1e0, NULL}, {0x8b81, 0xb1e1, NULL},
  {0x8ba1, 0xb1e2, NULL}, {0x8ba2, 0xb1e3, NULL}, {0x8ba5, 0xb1e4, NULL},
  {0x8ba8, 0xb1e5, NULL}, {0x8ba9, 0xb1e6, NULL}, {0x8bab, 0xb1e7, NULL},
  {0x8bb1, 0xb1e8, NULL}, {0x8bb3, 0xb1e9, NULL}, {0x8bb5, 0xb1ea, NULL},
  {0x8bb7, 0xb1eb, NULL}, {0x8bb8, 0xb1ec, NULL}, {0x8bbc, 0xb1ed, NULL},
  {0x8c61, 0xb1ee, NULL}, {0x8c62, 0xb1ef, NULL}, {0x8c63, 0xb1f0, NULL},
  {0x8c65, 0xb1f1, NULL}, {0x8c69, 0xb1f2, NULL}, {0x8c6b, 0xb1f3, NULL},
  {0x8c71, 0xb1f4, NULL}, {0x8c73, 0xb1f5, NULL}, {0x8c75, 0xb1f6, NULL},
  {0x8c76, 0xb1f7, NULL}, {0x8c77, 0xb1f8, NULL}, {0x8c7b, 0xb1f9, NULL},
  {0x8c81, 0xb1fa, NULL}, {0x8c82, 0xb1fb, NULL}, {0x8c85, 0xb1fc, NULL},
  {0x8c89, 0xb1fd, NULL}, {0x8c91, 0xb1fe, NULL}, {0x8c93, 0xb2a1, NULL},
  {0x8c95, 0xb2a2, NULL}, {0x8c96, 0xb2a3, NULL}, {0x8c97, 0xb2a4, NULL},
  {0x8ca1, 0xb2a5, NULL}, {0x8ca2, 0xb2a6, NULL}, {0x8ca9, 0xb2a7, NULL},
  {0x8ce1, 0xb2a8, NULL}, {0x8ce2, 0xb2a9, NULL}, {0x8ce3, 0xb2aa, NULL},
  {0x8ce5, 0xb2ab, NULL}, {0x8ce9, 0xb2ac, NULL}, {0x8cf1, 0xb2ad, NULL},
  {0x8cf3, 0xb2ae, NULL}, {0x8cf5, 0xb2af, NULL}, {0x8cf6, 0xb2b0, NULL},
  {0x8cf7, 0xb2b1, NULL}, {0x8d41, 0xb2b2, NULL}, {0x8d42, 0xb2b3, NULL},
  {0x8d45, 0xb2b4, NULL}, {0x8d51, 0xb2b5, NULL}, {0x8d55, 0xb2b6, NULL},
  {0x8d57, 0xb2b7, NULL}, {0x8d61, 0xb2b8, NULL}, {0x8d65, 0xb2b9, NULL},
  {0x8d69, 0xb2ba, NULL}, {0x8d75, 0xb2bb, NULL}, {0x8d76, 0xb2bc, NULL},
  {0x8d7b, 0xb2bd, NULL}, {0x8d81, 0xb2be, NULL}, {0x8da1, 0xb2bf, NULL},
  {0x8da2, 0xb2c0, NULL}, {0x8da5, 0xb2c1, NULL}, {0x8da7, 0xb2c2, NULL},
  {0x8da9, 0xb2c3, NULL}, {0x8db1, 0xb2c4, NULL}, {0x8db3, 0xb2c5, NULL},
  {0x8db5, 0xb2c6, NULL}, {0x8db7, 0xb2c7, NULL}, {0x8db8, 0xb2c8, NULL},
  {0x8db9, 0xb2c9, NULL}, {0x8dc1, 0xb2ca, NULL}, {0x8dc2, 0xb2cb, NULL},
  {0x8dc9, 0xb2cc, NULL}, {0x8dd6, 0xb2cd, NULL}, {0x8dd7, 0xb2ce, NULL},
  {0x8de1, 0xb2cf, NULL}, {0x8de2, 0xb2d0, NULL}, {0x8df7, 0xb2d1, NULL},
  {0x8e41, 0xb2d2, NULL}, {0x8e45, 0xb2d3, NULL}, {0x8e49, 0xb2d4, NULL},
  {0x8e51, 0xb2d5, NULL}, {0x8e53, 0xb2d6, NULL}, {0x8e57, 0xb2d7, NULL},
  {0x8e61, 0xb2d8, NULL}, {0x8e81, 0xb2d9, NULL}, {0x8e82, 0xb2da, NULL},
  {0x8e85, 0xb2db, NULL}, {0x8e89, 0xb2dc, NULL}, {0x8e90, 0xb2dd, NULL},
  {0x8e91, 0xb2de, NULL}, {0x8e93, 0xb2df, NULL}, {0x8e95, 0xb2e0, NULL},
  {0x8e97, 0xb2e1, NULL}, {0x8e98, 0xb2e2, NULL}, {0x8ea1, 0xb2e3, NULL},
  {0x8ea9, 0xb2e4, NULL}, {0x8eb6, 0xb2e5, NULL}, {0x8eb7, 0xb2e6, NULL},
  {0x8ec1, 0xb2e7, NULL}, {0x8ec2, 0xb2e8, NULL}, {0x8ec5, 0xb2e9, NULL},
  {0x8ec9, 0xb2ea, NULL}, {0x8ed1, 0xb2eb, NULL}, {0x8ed3, 0xb2ec, NULL},
  {0x8ed6, 0xb2ed, NULL}, {0x8ee1, 0xb2ee, NULL}, {0x8ee5, 0xb2ef, NULL},
  {0x8ee9, 0xb2f0, NULL}, {0x8ef1, 0xb2f1, NULL}, {0x8ef3, 0xb2f2, NULL},
  {0x8f41, 0xb2f3, NULL}, {0x8f61, 0xb2f4, NULL}, {0x8f62, 0xb2f5, NULL},
  {0x8f65, 0xb2f6, NULL}, {0x8f67, 0xb2f7, NULL}, {0x8f69, 0xb2f8, NULL},
  {0x8f6b, 0xb2f9, NULL}, {0x8f70, 0xb2fa, NULL}, {0x8f71, 0xb2fb, NULL},
  {0x8f73, 0xb2fc, NULL}, {0x8f75, 0xb2fd, NULL}, {0x8f77, 0xb2fe, NULL},
  {0x8f7b, 0xb3a1, NULL}, {0x8fa1, 0xb3a2, NULL}, {0x8fa2, 0xb3a3, NULL},
  {0x8fa5, 0xb3a4, NULL}, {0x8fa9, 0xb3a5, NULL}, {0x8fb1, 0xb3a6, NULL},
  {0x8fb3, 0xb3a7, NULL}, {0x8fb5, 0xb3a8, NULL}, {0x8fb7, 0xb3a9, NULL},
  {0x9061, 0xb3aa, NULL}, {0x9062, 0xb3ab, NULL}, {0x9063, 0xb3ac, NULL},
  {0x9065, 0xb3ad, NULL}, {0x9068, 0xb3ae, NULL}, {0x9069, 0xb3af, NULL},
  {0x906a, 0xb3b0, NULL}, {0x906b, 0xb3b1, NULL}, {0x9071, 0xb3b2, NULL},
  {0x9073, 0xb3b3, NULL}, {0x9075, 0xb3b4, NULL}, {0x9076, 0xb3b5, NULL},
  {0x9077, 0xb3b6, NULL}, {0x9078, 0xb3b7, NULL}, {0x9079, 0xb3b8, NULL},
  {0x907b, 0xb3b9, NULL}, {0x907d, 0xb3ba, NULL}, {0x9081, 0xb3bb, NULL},
  {0x9082, 0xb3bc, NULL}, {0x9085, 0xb3bd, NULL}, {0x9089, 0xb3be, NULL},
  {0x9091, 0xb3bf, NULL}, {0x9093, 0xb3c0, NULL}, {0x9095, 0xb3c1, NULL},
  {0x9096, 0xb3c2, NULL}, {0x9097, 0xb3c3, NULL}, {0x90a1, 0xb3c4, NULL},
  {0x90a2, 0xb3c5, NULL}, {0x90a5, 0xb3c6, NULL}, {0x90a9, 0xb3c7, NULL},
  {0x90b1, 0xb3c8, NULL}, {0x90b7, 0xb3c9, NULL}, {0x90e1, 0xb3ca, NULL},
  {0x90e2, 0xb3cb, NULL}, {0x90e4, 0xb3cc, NULL}, {0x90e5, 0xb3cd, NULL},
  {0x90e9, 0xb3ce, NULL}, {0x90eb, 0xb3cf, NULL}, {0x90ec, 0xb3d0, NULL},
  {0x90f1, 0xb3d1, NULL}, {0x90f3, 0xb3d2, NULL}, {0x90f5, 0xb3d3, NULL},
  {0x90f6, 0xb3d4, NULL}, {0x90f7, 0xb3d5, NULL}, {0x90fd, 0xb3d6, NULL},
  {0x9141, 0xb3d7, NULL}, {0x9142, 0xb3d8, NULL}, {0x9145, 0xb3d9, NULL},
  {0x9149, 0xb3da, NULL}, {0x9151, 0xb3db, NULL}, {0x9153, 0xb3dc, NULL},
  {0x9155, 0xb3dd, NULL}, {0x9156, 0xb3de, NULL}, {0x9157, 0xb3df, NULL},
  {0x9161, 0xb3e0, NULL}, {0x9162, 0xb3e1, NULL}, {0x9165, 0xb3e2, NULL},
  {0x9169, 0xb3e3, NULL}, {0x9171, 0xb3e4, NULL}, {0x9173, 0xb3e5, NULL},
  {0x9176, 0xb3e6, NULL}, {0x9177, 0xb3e7, NULL}, {0x917a, 0xb3e8, NULL},
  {0x9181, 0xb3e9, NULL}, {0x9185, 0xb3ea, NULL}, {0x91a1, 0xb3eb, NULL},
  {0x91a2, 0xb3ec, NULL}, {0x91a5, 0xb3ed, NULL}, {0x91a9, 0xb3ee, NULL},
  {0x91ab, 0xb3ef, NULL}, {0x91b1, 0xb3f0, NULL}, {0x91b3, 0xb3f1, NULL},
  {0x91b5, 0xb3f2, NULL}, {0x91b7, 0xb3f3, NULL}, {0x91bc, 0xb3f4, NULL},
  {0x91bd, 0xb3f5, NULL}, {0x91c1, 0xb3f6, NULL}, {0x91c5, 0xb3f7, NULL},
  {0x91c9, 0xb3f8, NULL}, {0x91d6, 0xb3f9, NULL}, {0x9241, 0xb3fa, NULL},
  {0x9245, 0xb3fb, NULL}, {0x9249, 0xb3fc, NULL}, {0x9251, 0xb3fd, NULL},
  {0x9253, 0xb3fe, NULL}, {0x9255, 0xb4a1, NULL}, {0x9261, 0xb4a2, NULL},
  {0x9262, 0xb4a3, NULL}, {0x9265, 0xb4a4, NULL}, {0x9269, 0xb4a5, NULL},
  {0x9273, 0xb4a6, NULL}, {0x9275, 0xb4a7, NULL}, {0x9277, 0xb4a8, NULL},
  {0x9281, 0xb4a9, NULL}, {0x9282, 0xb4aa, NULL}, {0x9285, 0xb4ab, NULL},
  {0x9288, 0xb4ac, NULL}, {0x9289, 0xb4ad, NULL}, {0x9291, 0xb4ae, NULL},
  {0x9293, 0xb4af, NULL}, {0x9295, 0xb4b0, NULL}, {0x9297, 0xb4b1, NULL},
  {0x92a1, 0xb4b2, NULL}, {0x92b6, 0xb4b3, NULL}, {0x92c1, 0xb4b4, NULL},
  {0x92e1, 0xb4b5, NULL}, {0x92e5, 0xb4b6, NULL}, {0x92e9, 0xb4b7, NULL},
  {0x92f1, 0xb4b8, NULL}, {0x92f3, 0xb4b9, NULL}, {0x9341, 0xb4ba, NULL},
  {0x9342, 0xb4bb, NULL}, {0x9349, 0xb4bc, NULL}, {0x9351, 0xb4bd, NULL},
  {0x9353, 0xb4be, NULL}, {0x9357, 0xb4bf, NULL}, {0x9361, 0xb4c0, NULL},
  {0x9362, 0xb4c1, NULL}, {0x9365, 0xb4c2, NULL}, {0x9369, 0xb4c3, NULL},
  {0x936a, 0xb4c4, NULL}, {0x936b, 0xb4c5, NULL}, {0x9371, 0xb4c6, NULL},
  {0x9373, 0xb4c7, NULL}, {0x9375, 0xb4c8, NULL}, {0x9377, 0xb4c9, NULL},
  {0x9378, 0xb4ca, NULL}, {0x937c, 0xb4cb, NULL}, {0x9381, 0xb4cc, NULL},
  {0x9385, 0xb4cd, NULL}, {0x9389, 0xb4ce, NULL}, {0x93a1, 0xb4cf, NULL},
  {0x93a2, 0xb4d0, NULL}, {0x93a5, 0xb4d1, NULL}, {0x93a9, 0xb4d2, NULL},
  {0x93ab, 0xb4d3, NULL}, {0x93b1, 0xb4d4, NULL}, {0x93b3, 0xb4d5, NULL},
  {0x93b5, 0xb4d6, NULL}, {0x93b7, 0xb4d7, NULL}, {0x93bc, 0xb4d8, NULL},
  {0x9461, 0xb4d9, NULL}, {0x9462, 0xb4da, NULL}, {0x9463, 0xb4db, NULL},
  {0x9465, 0xb4dc, NULL}, {0x9468, 0xb4dd, NULL}, {0x9469, 0xb4de, NULL},
  {0x946a, 0xb4df, NULL}, {0x946b, 0xb4e0, NULL}, {0x946c, 0xb4e1, NULL},
  {0x9470, 0xb4e2, NULL}, {0x9471, 0xb4e3, NULL}, {0x9473, 0xb4e4, NULL},
  {0x9475, 0xb4e5, NULL}, {0x9476, 0xb4e6, NULL}, {0x9477, 0xb4e7, NULL},
  {0x9478, 0xb4e8, NULL}, {0x9479, 0xb4e9, NULL}, {0x947d, 0xb4ea, NULL},
  {0x9481, 0xb4eb, NULL}, {0x9482, 0xb4ec, NULL}, {0x9485, 0xb4ed, NULL},
  {0x9489, 0xb4ee, NULL}, {0x9491, 0xb4ef, NULL}, {0x9493, 0xb4f0, NULL},
  {0x9495, 0xb4f1, NULL}, {0x9496, 0xb4f2, NULL}, {0x9497, 0xb4f3, NULL},
  {0x94a1, 0xb4f4, NULL}, {0x94e1, 0xb4f5, NULL}, {0x94e2, 0xb4f6, NULL},
  {0x94e3, 0xb4f7, NULL}, {0x94e5, 0xb4f8, NULL}, {0x94e8, 0xb4f9, NULL},
  {0x94e9, 0xb4fa, NULL}, {0x94eb, 0xb4fb, NULL}, {0x94ec, 0xb4fc, NULL},
  {0x94f1, 0xb4fd, NULL}, {0x94f3, 0xb4fe, NULL}, {0x94f5, 0xb5a1, NULL},
  {0x94f7, 0xb5a2, NULL}, {0x94f9, 0xb5a3, NULL}, {0x94fc, 0xb5a4, NULL},
  {0x9541, 0xb5a5, NULL}, {0x9542, 0xb5a6, NULL}, {0x9545, 0xb5a7, NULL},
  {0x9549, 0xb5a8, NULL}, {0x9551, 0xb5a9, NULL}, {0x9553, 0xb5aa, NULL},
  {0x9555, 0xb5ab, NULL}, {0x9556, 0xb5ac, NULL}, {0x9557, 0xb5ad, NULL},
  {0x9561, 0xb5ae, NULL}, {0x9565, 0xb5af, NULL}, {0x9569, 0xb5b0, NULL},
  {0x9576, 0xb5b1, NULL}, {0x9577, 0xb5b2, NULL}, {0x9581, 0xb5b3, NULL},
  {0x9585, 0xb5b4, NULL}, {0x95a1, 0xb5b5, NULL}, {0x95a2, 0xb5b6, NULL},
  {0x95a5, 0xb5b7, NULL}, {0x95a8, 0xb5b8, NULL}, {0x95a9, 0xb5b9, NULL},
  {0x95ab, 0xb5ba, NULL}, {0x95ad, 0xb5bb, NULL}, {0x95b1, 0xb5bc, NULL},
  {0x95b3, 0xb5bd, NULL}, {0x95b5, 0xb5be, NULL}, {0x95b7, 0xb5bf, NULL},
  {0x95b9, 0xb5c0, NULL}, {0x95bb, 0xb5c1, NULL}, {0x95c1, 0xb5c2, NULL},
  {0x95c5, 0xb5c3, NULL}, {0x95c9, 0xb5c4, NULL}, {0x95e1, 0xb5c5, NULL},
  {0x95f6, 0xb5c6, NULL}, {0x9641, 0xb5c7, NULL}, {0x9645, 0xb5c8, NULL},
  {0x9649, 0xb5c9, NULL}, {0x9651, 0xb5ca, NULL}, {0x9653, 0xb5cb, NULL},
  {0x9655, 0xb5cc, NULL}, {0x9661, 0xb5cd, NULL}, {0x9681, 0xb5ce, NULL},
  {0x9682, 0xb5cf, NULL}, {0x9685, 0xb5d0, NULL}, {0x9689, 0xb5d1, NULL},
  {0x9691, 0xb5d2, NULL}, {0x9693, 0xb5d3, NULL}, {0x9695, 0xb5d4, NULL},
  {0x9697, 0xb5d5, NULL}, {0x96a1, 0xb5d6, NULL}, {0x96b6, 0xb5d7, NULL},
  {0x96c1, 0xb5d8, NULL}, {0x96d7, 0xb5d9, NULL}, {0x96e1, 0xb5da, NULL},
  {0x96e5, 0xb5db, NULL}, {0x96e9, 0xb5dc, NULL}, {0x96f3, 0xb5dd, NULL},
  {0x96f5, 0xb5de, NULL}, {0x96f7, 0xb5df, NULL}, {0x9741, 0xb5e0, NULL},
  {0x9745, 0xb5e1, NULL}, {0x9749, 0xb5e2, NULL}, {0x9751, 0xb5e3, NULL},
  {0x9757, 0xb5e4, NULL}, {0x9761, 0xb5e5, NULL}, {0x9762, 0xb5e6, NULL},
  {0x9765, 0xb5e7, NULL}, {0x9768, 0xb5e8, NULL}, {0x9769, 0xb5e9, NULL},
  {0x976b, 0xb5ea, NULL}, {0x9771, 0xb5eb, NULL}, {0x9773, 0xb5ec, NULL},
  {0x9775, 0xb5ed, NULL}, {0x9777, 0xb5ee, NULL}, {0x9781, 0xb5ef, NULL},
  {0x97a1, 0xb5f0, NULL}, {0x97a2, 0xb5f1, NULL}, {0x97a5, 0xb5f2, NULL},
  {0x97a8, 0xb5f3, NULL}, {0x97a9, 0xb5f4, NULL}, {0x97b1, 0xb5f5, NULL},
  {0x97b3, 0xb5f6, NULL}, {0x97b5, 0xb5f7, NULL}, {0x97b6, 0xb5f8, NULL},
  {0x97b7, 0xb5f9, NULL}, {0x97b8, 0xb5fa, NULL}, {0x9861, 0xb5fb, NULL},
  {0x9862, 0xb5fc, NULL}, {0x9865, 0xb5fd, NULL}, {0x9869, 0xb5fe, NULL},
  {0x9871, 0xb6a1, NULL}, {0x9873, 0xb6a2, NULL}, {0x9875, 0xb6a3, NULL},
  {0x9876, 0xb6a4, NULL}, {0x9877, 0xb6a5, NULL}, {0x987d, 0xb6a6, NULL},
  {0x9881, 0xb6a7, NULL}, {0x9882, 0xb6a8, NULL}, {0x9885, 0xb6a9, NULL},
  {0x9889, 0xb6aa, NULL}, {0x9891, 0xb6ab, NULL}, {0x9893, 0xb6ac, NULL},
  {0x9895, 0xb6ad, NULL}, {0x9896, 0xb6ae, NULL}, {0x9897, 0xb6af, NULL},
  {0x98e1, 0xb6b0, NULL}, {0x98e2, 0xb6b1, NULL}, {0x98e5, 0xb6b2, NULL},
  {0x98e9, 0xb6b3, NULL}, {0x98eb, 0xb6b4, NULL}, {0x98ec, 0xb6b5, NULL},
  {0x98f1, 0xb6b6, NULL}, {0x98f3, 0xb6b7, NULL}, {0x98f5, 0xb6b8, NULL},
  {0x98f6, 0xb6b9, NULL}, {0x98f7, 0xb6ba, NULL}, {0x98fd, 0xb6bb, NULL},
  {0x9941, 0xb6bc, NULL}, {0x9942, 0xb6bd, NULL}, {0x9945, 0xb6be, NULL},
  {0x9949, 0xb6bf, NULL}, {0x9951, 0xb6c0, NULL}, {0x9953, 0xb6c1, NULL},
  {0x9955, 0xb6c2, NULL}, {0x9956, 0xb6c3, NULL}, {0x9957, 0xb6c4, NULL},
  {0x9961, 0xb6c5, NULL}, {0x9976, 0xb6c6, NULL}, {0x99a1, 0xb6c7, NULL},
  {0x99a2, 0xb6c8, NULL}, {0x99a5, 0xb6c9, NULL}, {0x99a9, 0xb6ca, NULL},
  {0x99b7, 0xb6cb, NULL}, {0x99c1, 0xb6cc, NULL}, {0x99c9, 0xb6cd, NULL},
  {0x99e1, 0xb6ce, NULL}, {0x9a41, 0xb6cf, NULL}, {0x9a45, 0xb6d0, NULL},
  {0x9a81, 0xb6d1, NULL}, {0x9a82, 0xb6d2, NULL}, {0x9a85, 0xb6d3, NULL},
  {0x9a89, 0xb6d4, NULL}, {0x9a90, 0xb6d5, NULL}, {0x9a91, 0xb6d6, NULL},
  {0x9a97, 0xb6d7, NULL}, {0x9ac1, 0xb6d8, NULL}, {0x9ae1, 0xb6d9, NULL},
  {0x9ae5, 0xb6da, NULL}, {0x9ae9, 0xb6db, NULL}, {0x9af1, 0xb6dc, NULL},
  {0x9af3, 0xb6dd, NULL}, {0x9af7, 0xb6de, NULL}, {0x9b61, 0xb6df, NULL},
  {0x9b62, 0xb6e0, NULL}, {0x9b65, 0xb6e1, NULL}, {0x9b68, 0xb6e2, NULL},
  {0x9b69, 0xb6e3, NULL}, {0x9b71, 0xb6e4, NULL}, {0x9b73, 0xb6e5, NULL},
  {0x9b75, 0xb6e6, NULL}, {0x9b81, 0xb6e7, NULL}, {0x9b85, 0xb6e8, NULL},
  {0x9b89, 0xb6e9, NULL}, {0x9b91, 0xb6ea, NULL}, {0x9b93, 0xb6eb, NULL},
  {0x9ba1, 0xb6ec, NULL}, {0x9ba5, 0xb6ed, NULL}, {0x9ba9, 0xb6ee, NULL},
  {0x9bb1, 0xb6ef, NULL}, {0x9bb3, 0xb6f0, NULL}, {0x9bb5, 0xb6f1, NULL},
  {0x9bb7, 0xb6f2, NULL}, {0x9c61, 0xb6f3, NULL}, {0x9c62, 0xb6f4, NULL},
  {0x9c65, 0xb6f5, NULL}, {0x9c69, 0xb6f6, NULL}, {0x9c71, 0xb6f7, NULL},
  {0x9c73, 0xb6f8, NULL}, {0x9c75, 0xb6f9, NULL}, {0x9c76, 0xb6fa, NULL},
  {0x9c77, 0xb6fb, NULL}, {0x9c78, 0xb6fc, NULL}, {0x9c7c, 0xb6fd, NULL},
  {0x9c7d, 0xb6fe, NULL}, {0x9c81, 0xb7a1, NULL}, {0x9c82, 0xb7a2, NULL},
  {0x9c85, 0xb7a3, NULL}, {0x9c89, 0xb7a4, NULL}, {0x9c91, 0xb7a5, NULL},
  {0x9c93, 0xb7a6, NULL}, {0x9c95, 0xb7a7, NULL}, {0x9c96, 0xb7a8, NULL},
  {0x9c97, 0xb7a9, NULL}, {0x9ca1, 0xb7aa, NULL}, {0x9ca2, 0xb7ab, NULL},
  {0x9ca5, 0xb7ac, NULL}, {0x9cb5, 0xb7ad, NULL}, {0x9cb7, 0xb7ae, NULL},
  {0x9ce1, 0xb7af, NULL}, {0x9ce2, 0xb7b0, NULL}, {0x9ce5, 0xb7b1, NULL},
  {0x9ce9, 0xb7b2, NULL}, {0x9cf1, 0xb7b3, NULL}, {0x9cf3, 0xb7b4, NULL},
  {0x9cf5, 0xb7b5, NULL}, {0x9cf6, 0xb7b6, NULL}, {0x9cf7, 0xb7b7, NULL},
  {0x9cfd, 0xb7b8, NULL}, {0x9d41, 0xb7b9, NULL}, {0x9d42, 0xb7ba, NULL},
  {0x9d45, 0xb7bb, NULL}, {0x9d49, 0xb7bc, NULL}, {0x9d51, 0xb7bd, NULL},
  {0x9d53, 0xb7be, NULL}, {0x9d55, 0xb7bf, NULL}, {0x9d57, 0xb7c0, NULL},
  {0x9d61, 0xb7c1, NULL}, {0x9d62, 0xb7c2, NULL}, {0x9d65, 0xb7c3, NULL},
  {0x9d69, 0xb7c4, NULL}, {0x9d71, 0xb7c5, NULL}, {0x9d73, 0xb7c6, NULL},
  {0x9d75, 0xb7c7, NULL}, {0x9d76, 0xb7c8, NULL}, {0x9d77, 0xb7c9, NULL},
  {0x9d81, 0xb7ca, NULL}, {0x9d85, 0xb7cb, NULL}, {0x9d93, 0xb7cc, NULL},
  {0x9d95, 0xb7cd, NULL}, {0x9da1, 0xb7ce, NULL}, {0x9da2, 0xb7cf, NULL},
  {0x9da5, 0xb7d0, NULL}, {0x9da9, 0xb7d1, NULL}, {0x9db1, 0xb7d2, NULL},
  {0x9db3, 0xb7d3, NULL}, {0x9db5, 0xb7d4, NULL}, {0x9db7, 0xb7d5, NULL},
  {0x9dc1, 0xb7d6, NULL}, {0x9dc5, 0xb7d7, NULL}, {0x9dd7, 0xb7d8, NULL},
  {0x9df6, 0xb7d9, NULL}, {0x9e41, 0xb7da, NULL}, {0x9e45, 0xb7db, NULL},
  {0x9e49, 0xb7dc, NULL}, {0x9e51, 0xb7dd, NULL}, {0x9e53, 0xb7de, NULL},
  {0x9e55, 0xb7df, NULL}, {0x9e57, 0xb7e0, NULL}, {0x9e61, 0xb7e1, NULL},
  {0x9e65, 0xb7e2, NULL}, {0x9e69, 0xb7e3, NULL}, {0x9e73, 0xb7e4, NULL},
  {0x9e75, 0xb7e5, NULL}, {0x9e77, 0xb7e6, NULL}, {0x9e81, 0xb7e7, NULL},
  {0x9e82, 0xb7e8, NULL}, {0x9e85, 0xb7e9, NULL}, {0x9e89, 0xb7ea, NULL},
  {0x9e91, 0xb7eb, NULL}, {0x9e93, 0xb7ec, NULL}, {0x9e95, 0xb7ed, NULL},
  {0x9e97, 0xb7ee, NULL}, {0x9ea1, 0xb7ef, NULL}, {0x9eb6, 0xb7f0, NULL},
  {0x9ec1, 0xb7f1, NULL}, {0x9ee1, 0xb7f2, NULL}, {0x9ee2, 0xb7f3, NULL},
  {0x9ee5, 0xb7f4, NULL}, {0x9ee9, 0xb7f5, NULL}, {0x9ef1, 0xb7f6, NULL},
  {0x9ef5, 0xb7f7, NULL}, {0x9ef7, 0xb7f8, NULL}, {0x9f41, 0xb7f9, NULL},
  {0x9f42, 0xb7fa, NULL}, {0x9f45, 0xb7fb, NULL}, {0x9f49, 0xb7fc, NULL},
  {0x9f51, 0xb7fd, NULL}, {0x9f53, 0xb7fe, NULL}, {0x9f55, 0xb8a1, NULL},
  {0x9f57, 0xb8a2, NULL}, {0x9f61, 0xb8a3, NULL}, {0x9f62, 0xb8a4, NULL},
  {0x9f65, 0xb8a5, NULL}, {0x9f69, 0xb8a6, NULL}, {0x9f71, 0xb8a7, NULL},
  {0x9f73, 0xb8a8, NULL}, {0x9f75, 0xb8a9, NULL}, {0x9f77, 0xb8aa, NULL},
  {0x9f78, 0xb8ab, NULL}, {0x9f7b, 0xb8ac, NULL}, {0x9f7c, 0xb8ad, NULL},
  {0x9fa1, 0xb8ae, NULL}, {0x9fa2, 0xb8af, NULL}, {0x9fa5, 0xb8b0, NULL},
  {0x9fa9, 0xb8b1, NULL}, {0x9fb1, 0xb8b2, NULL}, {0x9fb3, 0xb8b3, NULL},
  {0x9fb5, 0xb8b4, NULL}, {0x9fb7, 0xb8b5, NULL}, {0xa061, 0xb8b6, NULL},
  {0xa062, 0xb8b7, NULL}, {0xa065, 0xb8b8, NULL}, {0xa067, 0xb8b9, NULL},
  {0xa068, 0xb8ba, NULL}, {0xa069, 0xb8bb, NULL}, {0xa06a, 0xb8bc, NULL},
  {0xa06b, 0xb8bd, NULL}, {0xa071, 0xb8be, NULL}, {0xa073, 0xb8bf, NULL},
  {0xa075, 0xb8c0, NULL}, {0xa077, 0xb8c1, NULL}, {0xa078, 0xb8c2, NULL},
  {0xa07b, 0xb8c3, NULL}, {0xa07d, 0xb8c4, NULL}, {0xa081, 0xb8c5, NULL},
  {0xa082, 0xb8c6, NULL}, {0xa085, 0xb8c7, NULL}, {0xa089, 0xb8c8, NULL},
  {0xa091, 0xb8c9, NULL}, {0xa093, 0xb8ca, NULL}, {0xa095, 0xb8cb, NULL},
  {0xa096, 0xb8cc, NULL}, {0xa097, 0xb8cd, NULL}, {0xa098, 0xb8ce, NULL},
  {0xa0a1, 0xb8cf, NULL}, {0xa0a2, 0xb8d0, NULL}, {0xa0a9, 0xb8d1, NULL},
  {0xa0b7, 0xb8d2, NULL}, {0xa0e1, 0xb8d3, NULL}, {0xa0e2, 0xb8d4, NULL},
  {0xa0e5, 0xb8d5, NULL}, {0xa0e9, 0xb8d6, NULL}, {0xa0eb, 0xb8d7, NULL},
  {0xa0f1, 0xb8d8, NULL}, {0xa0f3, 0xb8d9, NULL}, {0xa0f5, 0xb8da, NULL},
  {0xa0f7, 0xb8db, NULL}, {0xa0f8, 0xb8dc, NULL}, {0xa0fd, 0xb8dd, NULL},
  {0xa141, 0xb8de, NULL}, {0xa142, 0xb8df, NULL}, {0xa145, 0xb8e0, NULL},
  {0xa149, 0xb8e1, NULL}, {0xa151, 0xb8e2, NULL}, {0xa153, 0xb8e3, NULL},
  {0xa155, 0xb8e4, NULL}, {0xa156, 0xb8e5, NULL}, {0xa157, 0xb8e6, NULL},
  {0xa161, 0xb8e7, NULL}, {0xa162, 0xb8e8, NULL}, {0xa165, 0xb8e9, NULL},
  {0xa169, 0xb8ea, NULL}, {0xa175, 0xb8eb, NULL}, {0xa176, 0xb8ec, NULL},
  {0xa177, 0xb8ed, NULL}, {0xa179, 0xb8ee, NULL}, {0xa181, 0xb8ef, NULL},
  {0xa1a1, 0xb8f0, NULL}, {0xa1a2, 0xb8f1, NULL}, {0xa1a4, 0xb8f2, NULL},
  {0xa1a5, 0xb8f3, NULL}, {0xa1a9, 0xb8f4, NULL}, {0xa1ab, 0xb8f5, NULL},
  {0xa1b1, 0xb8f6, NULL}, {0xa1b3, 0xb8f7, NULL}, {0xa1b5, 0xb8f8, NULL},
  {0xa1b7, 0xb8f9, NULL}, {0xa1c1, 0xb8fa, NULL}, {0xa1c5, 0xb8fb, NULL},
  {0xa1d6, 0xb8fc, NULL}, {0xa1d7, 0xb8fd, NULL}, {0xa241, 0xb8fe, NULL},
  {0xa245, 0xb9a1, NULL}, {0xa249, 0xb9a2, NULL}, {0xa253, 0xb9a3, NULL},
  {0xa255, 0xb9a4, NULL}, {0xa257, 0xb9a5, NULL}, {0xa261, 0xb9a6, NULL},
  {0xa265, 0xb9a7, NULL}, {0xa269, 0xb9a8, NULL}, {0xa273, 0xb9a9, NULL},
  {0xa275, 0xb9aa, NULL}, {0xa281, 0xb9ab, NULL}, {0xa282, 0xb9ac, NULL},
  {0xa283, 0xb9ad, NULL}, {0xa285, 0xb9ae, NULL}, {0xa288, 0xb9af, NULL},
  {0xa289, 0xb9b0, NULL}, {0xa28a, 0xb9b1, NULL}, {0xa28b, 0xb9b2, NULL},
  {0xa291, 0xb9b3, NULL}, {0xa293, 0xb9b4, NULL}, {0xa295, 0xb9b5, NULL},
  {0xa297, 0xb9b6, NULL}, {0xa29b, 0xb9b7, NULL}, {0xa29d, 0xb9b8, NULL},
  {0xa2a1, 0xb9b9, NULL}, {0xa2a5, 0xb9ba, NULL}, {0xa2a9, 0xb9bb, NULL},
  {0xa2b3, 0xb9bc, NULL}, {0xa2b5, 0xb9bd, NULL}, {0xa2c1, 0xb9be, NULL},
  {0xa2e1, 0xb9bf, NULL}, {0xa2e5, 0xb9c0, NULL}, {0xa2e9, 0xb9c1, NULL},
  {0xa341, 0xb9c2, NULL}, {0xa345, 0xb9c3, NULL}, {0xa349, 0xb9c4, NULL},
  {0xa351, 0xb9c5, NULL}, {0xa355, 0xb9c6, NULL}, {0xa361, 0xb9c7, NULL},
  {0xa365, 0xb9c8, NULL}, {0xa369, 0xb9c9, NULL}, {0xa371, 0xb9ca, NULL},
  {0xa375, 0xb9cb, NULL}, {0xa3a1, 0xb9cc, NULL}, {0xa3a2, 0xb9cd, NULL},
  {0xa3a5, 0xb9ce, NULL}, {0xa3a8, 0xb9cf, NULL}, {0xa3a9, 0xb9d0, NULL},
  {0xa3ab, 0xb9d1, NULL}, {0xa3b1, 0xb9d2, NULL}, {0xa3b3, 0xb9d3, NULL},
  {0xa3b5, 0xb9d4, NULL}, {0xa3b6, 0xb9d5, NULL}, {0xa3b7, 0xb9d6, NULL},
  {0xa3b9, 0xb9d7, NULL}, {0xa3bb, 0xb9d8, NULL}, {0xa461, 0xb9d9, NULL},
  {0xa462, 0xb9da, NULL}, {0xa463, 0xb9db, NULL}, {0xa464, 0xb9dc, NULL},
  {0xa465, 0xb9dd, NULL}, {0xa468, 0xb9de, NULL}, {0xa469, 0xb9df, NULL},
  {0xa46a, 0xb9e0, NULL}, {0xa46b, 0xb9e1, NULL}, {0xa46c, 0xb9e2, NULL},
  {0xa471, 0xb9e3, NULL}, {0xa473, 0xb9e4, NULL}, {0xa475, 0xb9e5, NULL},
  {0xa477, 0xb9e6, NULL}, {0xa47b, 0xb9e7, NULL}, {0xa481, 0xb9e8, NULL},
  {0xa482, 0xb9e9, NULL}, {0xa485, 0xb9ea, NULL}, {0xa489, 0xb9eb, NULL},
  {0xa491, 0xb9ec, NULL}, {0xa493, 0xb9ed, NULL}, {0xa495, 0xb9ee, NULL},
  {0xa496, 0xb9ef, NULL}, {0xa497, 0xb9f0, NULL}, {0xa49b, 0xb9f1, NULL},
  {0xa4a1, 0xb9f2, NULL}, {0xa4a2, 0xb9f3, NULL}, {0xa4a5, 0xb9f4, NULL},
  {0xa4b3, 0xb9f5, NULL}, {0xa4e1, 0xb9f6, NULL}, {0xa4e2, 0xb9f7, NULL},
  {0xa4e5, 0xb9f8, NULL}, {0xa4e8, 0xb9f9, NULL}, {0xa4e9, 0xb9fa, NULL},
  {0xa4eb, 0xb9fb, NULL}, {0xa4f1, 0xb9fc, NULL}, {0xa4f3, 0xb9fd, NULL},
  {0xa4f5, 0xb9fe, NULL}, {0xa4f7, 0xbaa1, NULL}, {0xa4f8, 0xbaa2, NULL},
  {0xa541, 0xbaa3, NULL}, {0xa542, 0xbaa4, NULL}, {0xa545, 0xbaa5, NULL},
  {0xa548, 0xbaa6, NULL}, {0xa549, 0xbaa7, NULL}, {0xa551, 0xbaa8, NULL},
  {0xa553, 0xbaa9, NULL}, {0xa555, 0xbaaa, NULL}, {0xa556, 0xbaab, NULL},
  {0xa557, 0xbaac, NULL}, {0xa561, 0xbaad, NULL}, {0xa562, 0xbaae, NULL},
  {0xa565, 0xbaaf, NULL}, {0xa569, 0xbab0, NULL}, {0xa573, 0xbab1, NULL},
  {0xa575, 0xbab2, NULL}, {0xa576, 0xbab3, NULL}, {0xa577, 0xbab4, NULL},
  {0xa57b, 0xbab5, NULL}, {0xa581, 0xbab6, NULL}, {0xa585, 0xbab7, NULL},
  {0xa5a1, 0xbab8, NULL}, {0xa5a2, 0xbab9, NULL}, {0xa5a3, 0xbaba, NULL},
  {0xa5a5, 0xbabb, NULL}, {0xa5a9, 0xbabc, NULL}, {0xa5b1, 0xbabd, NULL},
  {0xa5b3, 0xbabe, NULL}, {0xa5b5, 0xbabf, NULL}, {0xa5b7, 0xbac0, NULL},
  {0xa5c1, 0xbac1, NULL}, {0xa5c5, 0xbac2, NULL}, {0xa5d6, 0xbac3, NULL},
  {0xa5e1, 0xbac4, NULL}, {0xa5f6, 0xbac5, NULL}, {0xa641, 0xbac6, NULL},
  {0xa642, 0xbac7, NULL}, {0xa645, 0xbac8, NULL}, {0xa649, 0xbac9, NULL},
  {0xa651, 0xbaca, NULL}, {0xa653, 0xbacb, NULL}, {0xa661, 0xbacc, NULL},
  {0xa665, 0xbacd, NULL}, {0xa681, 0xbace, NULL}, {0xa682, 0xbacf, NULL},
  {0xa685, 0xbad0, NULL}, {0xa688, 0xbad1, NULL}, {0xa689, 0xbad2, NULL},
  {0xa68a, 0xbad3, NULL}, {0xa68b, 0xbad4, NULL}, {0xa691, 0xbad5, NULL},
  {0xa693, 0xbad6, NULL}, {0xa695, 0xbad7, NULL}, {0xa697, 0xbad8, NULL},
  {0xa69b, 0xbad9, NULL}, {0xa69c, 0xbada, NULL}, {0xa6a1, 0xbadb, NULL},
  {0xa6a9, 0xbadc, NULL}, {0xa6b6, 0xbadd, NULL}, {0xa6c1, 0xbade, NULL},
  {0xa6e1, 0xbadf, NULL}, {0xa6e2, 0xbae0, NULL}, {0xa6e5, 0xbae1, NULL},
  {0xa6e9, 0xbae2, NULL}, {0xa6f7, 0xbae3, NULL}, {0xa741, 0xbae4, NULL},
  {0xa745, 0xbae5, NULL}, {0xa749, 0xbae6, NULL}, {0xa751, 0xbae7, NULL},
  {0xa755, 0xbae8, NULL}, {0xa757, 0xbae9, NULL}, {0xa761, 0xbaea, NULL},
  {0xa762, 0xbaeb, NULL}, {0xa765, 0xbaec, NULL}, {0xa769, 0xbaed, NULL},
  {0xa771, 0xbaee, NULL}, {0xa773, 0xbaef, NULL}, {0xa775, 0xbaf0, NULL},
  {0xa7a1, 0xbaf1, NULL}, {0xa7a2, 0xbaf2, NULL}, {0xa7a5, 0xbaf3, NULL},
  {0xa7a9, 0xbaf4, NULL}, {0xa7ab, 0xbaf5, NULL}, {0xa7b1, 0xbaf6, NULL},
  {0xa7b3, 0xbaf7, NULL}, {0xa7b5, 0xbaf8, NULL}, {0xa7b7, 0xbaf9, NULL},
  {0xa7b8, 0xbafa, NULL}, {0xa7b9, 0xbafb, NULL}, {0xa861, 0xbafc, NULL},
  {0xa862, 0xbafd, NULL}, {0xa865, 0xbafe, NULL}, {0xa869, 0xbba1, NULL},
  {0xa86b, 0xbba2, NULL}, {0xa871, 0xbba3, NULL}, {0xa873, 0xbba4, NULL},
  {0xa875, 0xbba5, NULL}, {0xa876, 0xbba6, NULL}, {0xa877, 0xbba7, NULL},
  {0xa87d, 0xbba8, NULL}, {0xa881, 0xbba9, NULL}, {0xa882, 0xbbaa, NULL},
  {0xa885, 0xbbab, NULL}, {0xa889, 0xbbac, NULL}, {0xa891, 0xbbad, NULL},
  {0xa893, 0xbbae, NULL}, {0xa895, 0xbbaf, NULL}, {0xa896, 0xbbb0, NULL},
  {0xa897, 0xbbb1, NULL}, {0xa8a1, 0xbbb2, NULL}, {0xa8a2, 0xbbb3, NULL},
  {0xa8b1, 0xbbb4, NULL}, {0xa8e1, 0xbbb5, NULL}, {0xa8e2, 0xbbb6, NULL},
  {0xa8e5, 0xbbb7, NULL}, {0xa8e8, 0xbbb8, NULL}, {0xa8e9, 0xbbb9, NULL},
  {0xa8f1, 0xbbba, NULL}, {0xa8f5, 0xbbbb, NULL}, {0xa8f6, 0xbbbc, NULL},
  {0xa8f7, 0xbbbd, NULL}, {0xa941, 0xbbbe, NULL}, {0xa957, 0xbbbf, NULL},
  {0xa961, 0xbbc0, NULL}, {0xa962, 0xbbc1, NULL}, {0xa971, 0xbbc2, NULL},
  {0xa973, 0xbbc3, NULL}, {0xa975, 0xbbc4, NULL}, {0xa976, 0xbbc5, NULL},
  {0xa977, 0xbbc6, NULL}, {0xa9a1, 0xbbc7, NULL}, {0xa9a2, 0xbbc8, NULL},
  {0xa9a5, 0xbbc9, NULL}, {0xa9a9, 0xbbca, NULL}, {0xa9b1, 0xbbcb, NULL},
  {0xa9b3, 0xbbcc, NULL}, {0xa9b7, 0xbbcd, NULL}, {0xaa41, 0xbbce, NULL},
  {0xaa61, 0xbbcf, NULL}, {0xaa77, 0xbbd0, NULL}, {0xaa81, 0xbbd1, NULL},
  {0xaa82, 0xbbd2, NULL}, {0xaa85, 0xbbd3, NULL}, {0xaa89, 0xbbd4, NULL},
  {0xaa91, 0xbbd5, NULL}, {0xaa95, 0xbbd6, NULL}, {0xaa97, 0xbbd7, NULL},
  {0xab41, 0xbbd8, NULL}, {0xab57, 0xbbd9, NULL}, {0xab61, 0xbbda, NULL},
  {0xab65, 0xbbdb, NULL}, {0xab69, 0xbbdc, NULL}, {0xab71, 0xbbdd, NULL},
  {0xab73, 0xbbde, NULL}, {0xaba1, 0xbbdf, NULL}, {0xaba2, 0xbbe0, NULL},
  {0xaba5, 0xbbe1, NULL}, {0xaba9, 0xbbe2, NULL}, {0xabb1, 0xbbe3, NULL},
  {0xabb3, 0xbbe4, NULL}, {0xabb5, 0xbbe5, NULL}, {0xabb7, 0xbbe6, NULL},
  {0xac61, 0xbbe7, NULL}, {0xac62, 0xbbe8, NULL}, {0xac64, 0xbbe9, NULL},
  {0xac65, 0xbbea, NULL}, {0xac68, 0xbbeb, NULL}, {0xac69, 0xbbec, NULL},
  {0xac6a, 0xbbed, NULL}, {0xac6b, 0xbbee, NULL}, {0xac71, 0xbbef, NULL},
  {0xac73, 0xbbf0, NULL}, {0xac75, 0xbbf1, NULL}, {0xac76, 0xbbf2, NULL},
  {0xac77, 0xbbf3, NULL}, {0xac7b, 0xbbf4, NULL}, {0xac81, 0xbbf5, NULL},
  {0xac82, 0xbbf6, NULL}, {0xac85, 0xbbf7, NULL}, {0xac89, 0xbbf8, NULL},
  {0xac91, 0xbbf9, NULL}, {0xac93, 0xbbfa, NULL}, {0xac95, 0xbbfb, NULL},
  {0xac96, 0xbbfc, NULL}, {0xac97, 0xbbfd, NULL}, {0xaca1, 0xbbfe, NULL},
  {0xaca2, 0xbca1, NULL}, {0xaca5, 0xbca2, NULL}, {0xaca9, 0xbca3, NULL},
  {0xacb1, 0xbca4, NULL}, {0xacb3, 0xbca5, NULL}, {0xacb5, 0xbca6, NULL},
  {0xacb7, 0xbca7, NULL}, {0xacc1, 0xbca8, NULL}, {0xacc5, 0xbca9, NULL},
  {0xacc9, 0xbcaa, NULL}, {0xacd1, 0xbcab, NULL}, {0xacd7, 0xbcac, NULL},
  {0xace1, 0xbcad, NULL}, {0xace2, 0xbcae, NULL}, {0xace3, 0xbcaf, NULL},
  {0xace4, 0xbcb0, NULL}, {0xace5, 0xbcb1, NULL}, {0xace8, 0xbcb2, NULL},
  {0xace9, 0xbcb3, NULL}, {0xaceb, 0xbcb4, NULL}, {0xacec, 0xbcb5, NULL},
  {0xacf1, 0xbcb6, NULL}, {0xacf3, 0xbcb7, NULL}, {0xacf5, 0xbcb8, NULL},
  {0xacf6, 0xbcb9, NULL}, {0xacf7, 0xbcba, NULL}, {0xacfc, 0xbcbb, NULL},
  {0xad41, 0xbcbc, NULL}, {0xad42, 0xbcbd, NULL}, {0xad45, 0xbcbe, NULL},
  {0xad49, 0xbcbf, NULL}, {0xad51, 0xbcc0, NULL}, {0xad53, 0xbcc1, NULL},
  {0xad55, 0xbcc2, NULL}, {0xad56, 0xbcc3, NULL}, {0xad57, 0xbcc4, NULL},
  {0xad61, 0xbcc5, NULL}, {0xad62, 0xbcc6, NULL}, {0xad65, 0xbcc7, NULL},
  {0xad69, 0xbcc8, NULL}, {0xad71, 0xbcc9, NULL}, {0xad73, 0xbcca, NULL},
  {0xad75, 0xbccb, NULL}, {0xad76, 0xbccc, NULL}, {0xad77, 0xbccd, NULL},
  {0xad81, 0xbcce, NULL}, {0xad85, 0xbccf, NULL}, {0xad89, 0xbcd0, NULL},
  {0xad97, 0xbcd1, NULL}, {0xada1, 0xbcd2, NULL}, {0xada2, 0xbcd3, NULL},
  {0xada3, 0xbcd4, NULL}, {0xada5, 0xbcd5, NULL}, {0xada9, 0xbcd6, NULL},
  {0xadab, 0xbcd7, NULL}, {0xadb1, 0xbcd8, NULL}, {0xadb3, 0xbcd9, NULL},
  {0xadb5, 0xbcda, NULL}, {0xadb7, 0xbcdb, NULL}, {0xadbb, 0xbcdc, NULL},
  {0xadc1, 0xbcdd, NULL}, {0xadc2, 0xbcde, NULL}, {0xadc5, 0xbcdf, NULL},
  {0xadc9, 0xbce0, NULL}, {0xadd7, 0xbce1, NULL}, {0xade1, 0xbce2, NULL},
  {0xade5, 0xbce3, NULL}, {0xade9, 0xbce4, NULL}, {0xadf1, 0xbce5, NULL},
  {0xadf5, 0xbce6, NULL}, {0xadf6, 0xbce7, NULL}, {0xae41, 0xbce8, NULL},
  {0xae45, 0xbce9, NULL}, {0xae49, 0xbcea, NULL}, {0xae51, 0xbceb, NULL},
  {0xae53, 0xbcec, NULL}, {0xae55, 0xbced, NULL}, {0xae61, 0xbcee, NULL},
  {0xae62, 0xbcef, NULL}, {0xae65, 0xbcf0, NULL}, {0xae69, 0xbcf1, NULL},
  {0xae71, 0xbcf2, NULL}, {0xae73, 0xbcf3, NULL}, {0xae75, 0xbcf4, NULL},
  {0xae77, 0xbcf5, NULL}, {0xae81, 0xbcf6, NULL}, {0xae82, 0xbcf7, NULL},
  {0xae85, 0xbcf8, NULL}, {0xae88, 0xbcf9, NULL}, {0xae89, 0xbcfa, NULL},
  {0xae91, 0xbcfb, NULL}, {0xae93, 0xbcfc, NULL}, {0xae95, 0xbcfd, NULL},
  {0xae97, 0xbcfe, NULL}, {0xae99, 0xbda1, NULL}, {0xae9b, 0xbda2, NULL},
  {0xae9c, 0xbda3, NULL}, {0xaea1, 0xbda4, NULL}, {0xaeb6, 0xbda5, NULL},
  {0xaec1, 0xbda6, NULL}, {0xaec2, 0xbda7, NULL}, {0xaec5, 0xbda8, NULL},
  {0xaec9, 0xbda9, NULL}, {0xaed1, 0xbdaa, NULL}, {0xaed7, 0xbdab, NULL},
  {0xaee1, 0xbdac, NULL}, {0xaee2, 0xbdad, NULL}, {0xaee5, 0xbdae, NULL},
  {0xaee9, 0xbdaf, NULL}, {0xaef1, 0xbdb0, NULL}, {0xaef3, 0xbdb1, NULL},
  {0xaef5, 0xbdb2, NULL}, {0xaef7, 0xbdb3, NULL}, {0xaf41, 0xbdb4, NULL},
  {0xaf42, 0xbdb5, NULL}, {0xaf49, 0xbdb6, NULL}, {0xaf51, 0xbdb7, NULL},
  {0xaf55, 0xbdb8, NULL}, {0xaf57, 0xbdb9, NULL}, {0xaf61, 0xbdba, NULL},
  {0xaf62, 0xbdbb, NULL}, {0xaf65, 0xbdbc, NULL}, {0xaf69, 0xbdbd, NULL},
  {0xaf6a, 0xbdbe, NULL}, {0xaf71, 0xbdbf, NULL}, {0xaf73, 0xbdc0, NULL},
  {0xaf75, 0xbdc1, NULL}, {0xaf77, 0xbdc2, NULL}, {0xafa1, 0xbdc3, NULL},
  {0xafa2, 0xbdc4, NULL}, {0xafa5, 0xbdc5, NULL}, {0xafa8, 0xbdc6, NULL},
  {0xafa9, 0xbdc7, NULL}, {0xafb0, 0xbdc8, NULL}, {0xafb1, 0xbdc9, NULL},
  {0xafb3, 0xbdca, NULL}, {0xafb5, 0xbdcb, NULL}, {0xafb7, 0xbdcc, NULL},
  {0xafbc, 0xbdcd, NULL}, {0xb061, 0xbdce, NULL}, {0xb062, 0xbdcf, NULL},
  {0xb064, 0xbdd0, NULL}, {0xb065, 0xbdd1, NULL}, {0xb069, 0xbdd2, NULL},
  {0xb071, 0xbdd3, NULL}, {0xb073, 0xbdd4, NULL}, {0xb076, 0xbdd5, NULL},
  {0xb077, 0xbdd6, NULL}, {0xb07d, 0xbdd7, NULL}, {0xb081, 0xbdd8, NULL},
  {0xb082, 0xbdd9, NULL}, {0xb085, 0xbdda, NULL}, {0xb089, 0xbddb, NULL},
  {0xb091, 0xbddc, NULL}, {0xb093, 0xbddd, NULL}, {0xb096, 0xbdde, NULL},
  {0xb097, 0xbddf, NULL}, {0xb0b7, 0xbde0, NULL}, {0xb0e1, 0xbde1, NULL},
  {0xb0e2, 0xbde2, NULL}, {0xb0e5, 0xbde3, NULL}, {0xb0e9, 0xbde4, NULL},
  {0xb0eb, 0xbde5, NULL}, {0xb0f1, 0xbde6, NULL}, {0xb0f3, 0xbde7, NULL},
  {0xb0f6, 0xbde8, NULL}, {0xb0f7, 0xbde9, NULL}, {0xb141, 0xbdea, NULL},
  {0xb145, 0xbdeb, NULL}, {0xb149, 0xbdec, NULL}, {0xb185, 0xbded, NULL},
  {0xb1a1, 0xbdee, NULL}, {0xb1a2, 0xbdef, NULL}, {0xb1a5, 0xbdf0, NULL},
  {0xb1a8, 0xbdf1, NULL}, {0xb1a9, 0xbdf2, NULL}, {0xb1ab, 0xbdf3, NULL},
  {0xb1b1, 0xbdf4, NULL}, {0xb1b3, 0xbdf5, NULL}, {0xb1b7, 0xbdf6, NULL},
  {0xb1c1, 0xbdf7, NULL}, {0xb1c2, 0xbdf8, NULL}, {0xb1c5, 0xbdf9, NULL},
  {0xb1d6, 0xbdfa, NULL}, {0xb1e1, 0xbdfb, NULL}, {0xb1f6, 0xbdfc, NULL},
  {0xb241, 0xbdfd, NULL}, {0xb245, 0xbdfe, NULL}, {0xb249, 0xbea1, NULL},
  {0xb251, 0xbea2, NULL}, {0xb253, 0xbea3, NULL}, {0xb261, 0xbea4, NULL},
  {0xb281, 0xbea5, NULL}, {0xb282, 0xbea6, NULL}, {0xb285, 0xbea7, NULL},
  {0xb289, 0xbea8, NULL}, {0xb291, 0xbea9, NULL}, {0xb293, 0xbeaa, NULL},
  {0xb297, 0xbeab, NULL}, {0xb2a1, 0xbeac, NULL}, {0xb2b6, 0xbead, NULL},
  {0xb2c1, 0xbeae, NULL}, {0xb2e1, 0xbeaf, NULL}, {0xb2e5, 0xbeb0, NULL},
  {0xb357, 0xbeb1, NULL}, {0xb361, 0xbeb2, NULL}, {0xb362, 0xbeb3, NULL},
  {0xb365, 0xbeb4, NULL}, {0xb369, 0xbeb5, NULL}, {0xb36b, 0xbeb6, NULL},
  {0xb370, 0xbeb7, NULL}, {0xb371, 0xbeb8, NULL}, {0xb373, 0xbeb9, NULL},
  {0xb381, 0xbeba, NULL}, {0xb385, 0xbebb, NULL}, {0xb389, 0xbebc, NULL},
  {0xb391, 0xbebd, NULL}, {0xb3a1, 0xbebe, NULL}, {0xb3a2, 0xbebf, NULL},
  {0xb3a5, 0xbec0, NULL}, {0xb3a9, 0xbec1, NULL}, {0xb3b1, 0xbec2, NULL},
  {0xb3b3, 0xbec3, NULL}, {0xb3b5, 0xbec4, NULL}, {0xb3b7, 0xbec5, NULL},
  {0xb461, 0xbec6, NULL}, {0xb462, 0xbec7, NULL}, {0xb465, 0xbec8, NULL},
  {0xb466, 0xbec9, NULL}, {0xb467, 0xbeca, NULL}, {0xb469, 0xbecb, NULL},
  {0xb46a, 0xbecc, NULL}, {0xb46b, 0xbecd, NULL}, {0xb470, 0xbece, NULL},
  {0xb471, 0xbecf, NULL}, {0xb473, 0xbed0, NULL}, {0xb475, 0xbed1, NULL},
  {0xb476, 0xbed2, NULL}, {0xb477, 0xbed3, NULL}, {0xb47b, 0xbed4, NULL},
  {0xb47c, 0xbed5, NULL}, {0xb481, 0xbed6, NULL}, {0xb482, 0xbed7, NULL},
  {0xb485, 0xbed8, NULL}, {0xb489, 0xbed9, NULL}, {0xb491, 0xbeda, NULL},
  {0xb493, 0xbedb, NULL}, {0xb495, 0xbedc, NULL}, {0xb496, 0xbedd, NULL},
  {0xb497, 0xbede, NULL}, {0xb4a1, 0xbedf, NULL}, {0xb4a2, 0xbee0, NULL},
  {0xb4a5, 0xbee1, NULL}, {0xb4a9, 0xbee2, NULL}, {0xb4ac, 0xbee3, NULL},
  {0xb4b1, 0xbee4, NULL}, {0xb4b3, 0xbee5, NULL}, {0xb4b5, 0xbee6, NULL},
  {0xb4b7, 0xbee7, NULL}, {0xb4bb, 0xbee8, NULL}, {0xb4bd, 0xbee9, NULL},
  {0xb4c1, 0xbeea, NULL}, {0xb4c5, 0xbeeb, NULL}, {0xb4c9, 0xbeec, NULL},
  {0xb4d3, 0xbeed, NULL}, {0xb4e1, 0xbeee, NULL}, {0xb4e2, 0xbeef, NULL},
  {0xb4e5, 0xbef0, NULL}, {0xb4e6, 0xbef1, NULL}, {0xb4e8, 0xbef2, NULL},
  {0xb4e9, 0xbef3, NULL}, {0xb4ea, 0xbef4, NULL}, {0xb4eb, 0xbef5, NULL},
  {0xb4f1, 0xbef6, NULL}, {0xb4f3, 0xbef7, NULL}, {0xb4f4, 0xbef8, NULL},
  {0xb4f5, 0xbef9, NULL}, {0xb4f6, 0xbefa, NULL}, {0xb4f7, 0xbefb, NULL},
  {0xb4f8, 0xbefc, NULL}, {0xb4fa, 0xbefd, NULL}, {0xb4fc, 0xbefe, NULL},
  {0xb541, 0xbfa1, NULL}, {0xb542, 0xbfa2, NULL}, {0xb545, 0xbfa3, NULL},
  {0xb549, 0xbfa4, NULL}, {0xb551, 0xbfa5, NULL}, {0xb553, 0xbfa6, NULL},
  {0xb555, 0xbfa7, NULL}, {0xb557, 0xbfa8, NULL}, {0xb561, 0xbfa9, NULL},
  {0xb562, 0xbfaa, NULL}, {0xb563, 0xbfab, NULL}, {0xb565, 0xbfac, NULL},
  {0xb569, 0xbfad, NULL}, {0xb56b, 0xbfae, NULL}, {0xb56c, 0xbfaf, NULL},
  {0xb571, 0xbfb0, NULL}, {0xb573, 0xbfb1, NULL}, {0xb574, 0xbfb2, NULL},
  {0xb575, 0xbfb3, NULL}, {0xb576, 0xbfb4, NULL}, {0xb577, 0xbfb5, NULL},
  {0xb57b, 0xbfb6, NULL}, {0xb57c, 0xbfb7, NULL}, {0xb57d, 0xbfb8, NULL},
  {0xb581, 0xbfb9, NULL}, {0xb585, 0xbfba, NULL}, {0xb589, 0xbfbb, NULL},
  {0xb591, 0xbfbc, NULL}, {0xb593, 0xbfbd, NULL}, {0xb595, 0xbfbe, NULL},
  {0xb596, 0xbfbf, NULL}, {0xb5a1, 0xbfc0, NULL}, {0xb5a2, 0xbfc1, NULL},
  {0xb5a5, 0xbfc2, NULL}, {0xb5a9, 0xbfc3, NULL}, {0xb5aa, 0xbfc4, NULL},
  {0xb5ab, 0xbfc5, NULL}, {0xb5ad, 0xbfc6, NULL}, {0xb5b0, 0xbfc7, NULL},
  {0xb5b1, 0xbfc8, NULL}, {0xb5b3, 0xbfc9, NULL}, {0xb5b5, 0xbfca, NULL},
  {0xb5b7, 0xbfcb, NULL}, {0xb5b9, 0xbfcc, NULL}, {0xb5c1, 0xbfcd, NULL},
  {0xb5c2, 0xbfce, NULL}, {0xb5c5, 0xbfcf, NULL}, {0xb5c9, 0xbfd0, NULL},
  {0xb5d1, 0xbfd1, NULL}, {0xb5d3, 0xbfd2, NULL}, {0xb5d5, 0xbfd3, NULL},
  {0xb5d6, 0xbfd4, NULL}, {0xb5d7, 0xbfd5, NULL}, {0xb5e1, 0xbfd6, NULL},
  {0xb5e2, 0xbfd7, NULL}, {0xb5e5, 0xbfd8, NULL}, {0xb5f1, 0xbfd9, NULL},
  {0xb5f5, 0xbfda, NULL}, {0xb5f7, 0xbfdb, NULL}, {0xb641, 0xbfdc, NULL},
  {0xb642, 0xbfdd, NULL}, {0xb645, 0xbfde, NULL}, {0xb649, 0xbfdf, NULL},
  {0xb651, 0xbfe0, NULL}, {0xb653, 0xbfe1, NULL}, {0xb655, 0xbfe2, NULL},
  {0xb657, 0xbfe3, NULL}, {0xb661, 0xbfe4, NULL}, {0xb662, 0xbfe5, NULL},
  {0xb665, 0xbfe6, NULL}, {0xb669, 0xbfe7, NULL}, {0xb671, 0xbfe8, NULL},
  {0xb673, 0xbfe9, NULL}, {0xb675, 0xbfea, NULL}, {0xb677, 0xbfeb, NULL},
  {0xb681, 0xbfec, NULL}, {0xb682, 0xbfed, NULL}, {0xb685, 0xbfee, NULL},
  {0xb689, 0xbfef, NULL}, {0xb68a, 0xbff0, NULL}, {0xb68b, 0xbff1, NULL},
  {0xb691, 0xbff2, NULL}, {0xb693, 0xbff3, NULL}, {0xb695, 0xbff4, NULL},
  {0xb697, 0xbff5, NULL}, {0xb6a1, 0xbff6, NULL}, {0xb6a2, 0xbff7, NULL},
  {0xb6a5, 0xbff8, NULL}, {0xb6a9, 0xbff9, NULL}, {0xb6b1, 0xbffa, NULL},
  {0xb6b3, 0xbffb, NULL}, {0xb6b6, 0xbffc, NULL}, {0xb6b7, 0xbffd, NULL},
  {0xb6c1, 0xbffe, NULL}, {0xb6c2, 0xc0a1, NULL}, {0xb6c5, 0xc0a2, NULL},
  {0xb6c9, 0xc0a3, NULL}, {0xb6d1, 0xc0a4, NULL}, {0xb6d3, 0xc0a5, NULL},
  {0xb6d7, 0xc0a6, NULL}, {0xb6e1, 0xc0a7, NULL}, {0xb6e2, 0xc0a8, NULL},
  {0xb6e5, 0xc0a9, NULL}, {0xb6e9, 0xc0aa, NULL}, {0xb6f1, 0xc0ab, NULL},
  {0xb6f3, 0xc0ac, NULL}, {0xb6f5, 0xc0ad, NULL}, {0xb6f7, 0xc0ae, NULL},
  {0xb741, 0xc0af, NULL}, {0xb742, 0xc0b0, NULL}, {0xb745, 0xc0b1, NULL},
  {0xb749, 0xc0b2, NULL}, {0xb751, 0xc0b3, NULL}, {0xb753, 0xc0b4, NULL},
  {0xb755, 0xc0b5, NULL}, {0xb757, 0xc0b6, NULL}, {0xb759, 0xc0b7, NULL},
  {0xb761, 0xc0b8, NULL}, {0xb762, 0xc0b9, NULL}, {0xb765, 0xc0ba, NULL},
  {0xb769, 0xc0bb, NULL}, {0xb76f, 0xc0bc, NULL}, {0xb771, 0xc0bd, NULL},
  {0xb773, 0xc0be, NULL}, {0xb775, 0xc0bf, NULL}, {0xb777, 0xc0c0, NULL},
  {0xb778, 0xc0c1, NULL}, {0xb779, 0xc0c2, NULL}, {0xb77a, 0xc0c3, NULL},
  {0xb77b, 0xc0c4, NULL}, {0xb77c, 0xc0c5, NULL}, {0xb77d, 0xc0c6, NULL},
  {0xb781, 0xc0c7, NULL}, {0xb785, 0xc0c8, NULL}, {0xb789, 0xc0c9, NULL},
  {0xb791, 0xc0ca, NULL}, {0xb795, 0xc0cb, NULL}, {0xb7a1, 0xc0cc, NULL},
  {0xb7a2, 0xc0cd, NULL}, {0xb7a5, 0xc0ce, NULL}, {0xb7a9, 0xc0cf, NULL},
  {0xb7aa, 0xc0d0, NULL}, {0xb7ab, 0xc0d1, NULL}, {0xb7b0, 0xc0d2, NULL},
  {0xb7b1, 0xc0d3, NULL}, {0xb7b3, 0xc0d4, NULL}, {0xb7b5, 0xc0d5, NULL},
  {0xb7b6, 0xc0d6, NULL}, {0xb7b7, 0xc0d7, NULL}, {0xb7b8, 0xc0d8, NULL},
  {0xb7bc, 0xc0d9, NULL}, {0xb861, 0xc0da, NULL}, {0xb862, 0xc0db, NULL},
  {0xb865, 0xc0dc, NULL}, {0xb867, 0xc0dd, NULL}, {0xb868, 0xc0de, NULL},
  {0xb869, 0xc0df, NULL}, {0xb86b, 0xc0e0, NULL}, {0xb871, 0xc0e1, NULL},
  {0xb873, 0xc0e2, NULL}, {0xb875, 0xc0e3, NULL}, {0xb876, 0xc0e4, NULL},
  {0xb877, 0xc0e5, NULL}, {0xb878, 0xc0e6, NULL}, {0xb881, 0xc0e7, NULL},
  {0xb882, 0xc0e8, NULL}, {0xb885, 0xc0e9, NULL}, {0xb889, 0xc0ea, NULL},
  {0xb891, 0xc0eb, NULL}, {0xb893, 0xc0ec, NULL}, {0xb895, 0xc0ed, NULL},
  {0xb896, 0xc0ee, NULL}, {0xb897, 0xc0ef, NULL}, {0xb8a1, 0xc0f0, NULL},
  {0xb8a2, 0xc0f1, NULL}, {0xb8a5, 0xc0f2, NULL}, {0xb8a7, 0xc0f3, NULL},
  {0xb8a9, 0xc0f4, NULL}, {0xb8b1, 0xc0f5, NULL}, {0xb8b7, 0xc0f6, NULL},
  {0xb8c1, 0xc0f7, NULL}, {0xb8c5, 0xc0f8, NULL}, {0xb8c9, 0xc0f9, NULL},
  {0xb8e1, 0xc0fa, NULL}, {0xb8e2, 0xc0fb, NULL}, {0xb8e5, 0xc0fc, NULL},
  {0xb8e9, 0xc0fd, NULL}, {0xb8eb, 0xc0fe, NULL}, {0xb8f1, 0xc1a1, NULL},
  {0xb8f3, 0xc1a2, NULL}, {0xb8f5, 0xc1a3, NULL}, {0xb8f7, 0xc1a4, NULL},
  {0xb8f8, 0xc1a5, NULL}, {0xb941, 0xc1a6, NULL}, {0xb942, 0xc1a7, NULL},
  {0xb945, 0xc1a8, NULL}, {0xb949, 0xc1a9, NULL}, {0xb951, 0xc1aa, NULL},
  {0xb953, 0xc1ab, NULL}, {0xb955, 0xc1ac, NULL}, {0xb957, 0xc1ad, NULL},
  {0xb961, 0xc1ae, NULL}, {0xb965, 0xc1af, NULL}, {0xb969, 0xc1b0, NULL},
  {0xb971, 0xc1b1, NULL}, {0xb973, 0xc1b2, NULL}, {0xb976, 0xc1b3, NULL},
  {0xb977, 0xc1b4, NULL}, {0xb981, 0xc1b5, NULL}, {0xb9a1, 0xc1b6, NULL},
  {0xb9a2, 0xc1b7, NULL}, {0xb9a5, 0xc1b8, NULL}, {0xb9a9, 0xc1b9, NULL},
  {0xb9ab, 0xc1ba, NULL}, {0xb9b1, 0xc1bb, NULL}, {0xb9b3, 0xc1bc, NULL},
  {0xb9b5, 0xc1bd, NULL}, {0xb9b7, 0xc1be, NULL}, {0xb9b8, 0xc1bf, NULL},
  {0xb9b9, 0xc1c0, NULL}, {0xb9bd, 0xc1c1, NULL}, {0xb9c1, 0xc1c2, NULL},
  {0xb9c2, 0xc1c3, NULL}, {0xb9c9, 0xc1c4, NULL}, {0xb9d3, 0xc1c5, NULL},
  {0xb9d5, 0xc1c6, NULL}, {0xb9d7, 0xc1c7, NULL}, {0xb9e1, 0xc1c8, NULL},
  {0xb9f6, 0xc1c9, NULL}, {0xb9f7, 0xc1ca, NULL}, {0xba41, 0xc1cb, NULL},
  {0xba45, 0xc1cc, NULL}, {0xba49, 0xc1cd, NULL}, {0xba51, 0xc1ce, NULL},
  {0xba53, 0xc1cf, NULL}, {0xba55, 0xc1d0, NULL}, {0xba57, 0xc1d1, NULL},
  {0xba61, 0xc1d2, NULL}, {0xba62, 0xc1d3, NULL}, {0xba65, 0xc1d4, NULL},
  {0xba77, 0xc1d5, NULL}, {0xba81, 0xc1d6, NULL}, {0xba82, 0xc1d7, NULL},
  {0xba85, 0xc1d8, NULL}, {0xba89, 0xc1d9, NULL}, {0xba8a, 0xc1da, NULL},
  {0xba8b, 0xc1db, NULL}, {0xba91, 0xc1dc, NULL}, {0xba93, 0xc1dd, NULL},
  {0xba95, 0xc1de, NULL}, {0xba97, 0xc1df, NULL}, {0xbaa1, 0xc1e0, NULL},
  {0xbab6, 0xc1e1, NULL}, {0xbac1, 0xc1e2, NULL}, {0xbae1, 0xc1e3, NULL},
  {0xbae2, 0xc1e4, NULL}, {0xbae5, 0xc1e5, NULL}, {0xbae9, 0xc1e6, NULL},
  {0xbaf1, 0xc1e7, NULL}, {0xbaf3, 0xc1e8, NULL}, {0xbaf5, 0xc1e9, NULL},
  {0xbb41, 0xc1ea, NULL}, {0xbb45, 0xc1eb, NULL}, {0xbb49, 0xc1ec, NULL},
  {0xbb51, 0xc1ed, NULL}, {0xbb61, 0xc1ee, NULL}, {0xbb62, 0xc1ef, NULL},
  {0xbb65, 0xc1f0, NULL}, {0xbb69, 0xc1f1, NULL}, {0xbb71, 0xc1f2, NULL},
  {0xbb73, 0xc1f3, NULL}, {0xbb75, 0xc1f4, NULL}, {0xbb77, 0xc1f5, NULL},
  {0xbba1, 0xc1f6, NULL}, {0xbba2, 0xc1f7, NULL}, {0xbba5, 0xc1f8, NULL},
  {0xbba8, 0xc1f9, NULL}, {0xbba9, 0xc1fa, NULL}, {0xbbab, 0xc1fb, NULL},
  {0xbbb1, 0xc1fc, NULL}, {0xbbb3, 0xc1fd, NULL}, {0xbbb5, 0xc1fe, NULL},
  {0xbbb7, 0xc2a1, NULL}, {0xbbb8, 0xc2a2, NULL}, {0xbbbb, 0xc2a3, NULL},
  {0xbbbc, 0xc2a4, NULL}, {0xbc61, 0xc2a5, NULL}, {0xbc62, 0xc2a6, NULL},
  {0xbc65, 0xc2a7, NULL}, {0xbc67, 0xc2a8, NULL}, {0xbc69, 0xc2a9, NULL},
  {0xbc6c, 0xc2aa, NULL}, {0xbc71, 0xc2ab, NULL}, {0xbc73, 0xc2ac, NULL},
  {0xbc75, 0xc2ad, NULL}, {0xbc76, 0xc2ae, NULL}, {0xbc77, 0xc2af, NULL},
  {0xbc81, 0xc2b0, NULL}, {0xbc82, 0xc2b1, NULL}, {0xbc85, 0xc2b2, NULL},
  {0xbc89, 0xc2b3, NULL}, {0xbc91, 0xc2b4, NULL}, {0xbc93, 0xc2b5, NULL},
  {0xbc95, 0xc2b6, NULL}, {0xbc96, 0xc2b7, NULL}, {0xbc97, 0xc2b8, NULL},
  {0xbca1, 0xc2b9, NULL}, {0xbca5, 0xc2ba, NULL}, {0xbcb7, 0xc2bb, NULL},
  {0xbce1, 0xc2bc, NULL}, {0xbce2, 0xc2bd, NULL}, {0xbce5, 0xc2be, NULL},
  {0xbce9, 0xc2bf, NULL}, {0xbcf1, 0xc2c0, NULL}, {0xbcf3, 0xc2c1, NULL},
  {0xbcf5, 0xc2c2, NULL}, {0xbcf6, 0xc2c3, NULL}, {0xbcf7, 0xc2c4, NULL},
  {0xbd41, 0xc2c5, NULL}, {0xbd57, 0xc2c6, NULL}, {0xbd61, 0xc2c7, NULL},
  {0xbd76, 0xc2c8, NULL}, {0xbda1, 0xc2c9, NULL}, {0xbda2, 0xc2ca, NULL},
  {0xbda5, 0xc2cb, NULL}, {0xbda9, 0xc2cc, NULL}, {0xbdb1, 0xc2cd, NULL},
  {0xbdb3, 0xc2ce, NULL}, {0xbdb5, 0xc2cf, NULL}, {0xbdb7, 0xc2d0, NULL},
  {0xbdb9, 0xc2d1, NULL}, {0xbdc1, 0xc2d2, NULL}, {0xbdc2, 0xc2d3, NULL},
  {0xbdc9, 0xc2d4, NULL}, {0xbdd6, 0xc2d5, NULL}, {0xbde1, 0xc2d6, NULL},
  {0xbdf6, 0xc2d7, NULL}, {0xbe41, 0xc2d8, NULL}, {0xbe45, 0xc2d9, NULL},
  {0xbe49, 0xc2da, NULL}, {0xbe51, 0xc2db, NULL}, {0xbe53, 0xc2dc, NULL},
  {0xbe77, 0xc2dd, NULL}, {0xbe81, 0xc2de, NULL}, {0xbe82, 0xc2df, NULL},
  {0xbe85, 0xc2e0, NULL}, {0xbe89, 0xc2e1, NULL}, {0xbe91, 0xc2e2, NULL},
  {0xbe93, 0xc2e3, NULL}, {0xbe97, 0xc2e4, NULL}, {0xbea1, 0xc2e5, NULL},
  {0xbeb6, 0xc2e6, NULL}, {0xbeb7, 0xc2e7, NULL}, {0xbee1, 0xc2e8, NULL},
  {0xbf41, 0xc2e9, NULL}, {0xbf61, 0xc2ea, NULL}, {0xbf71, 0xc2eb, NULL},
  {0xbf75, 0xc2ec, NULL}, {0xbf77, 0xc2ed, NULL}, {0xbfa1, 0xc2ee, NULL},
  {0xbfa2, 0xc2ef, NULL}, {0xbfa5, 0xc2f0, NULL}, {0xbfa9, 0xc2f1, NULL},
  {0xbfb1, 0xc2f2, NULL}, {0xbfb3, 0xc2f3, NULL}, {0xbfb7, 0xc2f4, NULL},
  {0xbfb8, 0xc2f5, NULL}, {0xbfbd, 0xc2f6, NULL}, {0xc061, 0xc2f7, NULL},
  {0xc062, 0xc2f8, NULL}, {0xc065, 0xc2f9, NULL}, {0xc067, 0xc2fa, NULL},
  {0xc069, 0xc2fb, NULL}, {0xc071, 0xc2fc, NULL}, {0xc073, 0xc2fd, NULL},
  {0xc075, 0xc2fe, NULL}, {0xc076, 0xc3a1, NULL}, {0xc077, 0xc3a2, NULL},
  {0xc078, 0xc3a3, NULL}, {0xc081, 0xc3a4, NULL}, {0xc082, 0xc3a5, NULL},
  {0xc085, 0xc3a6, NULL}, {0xc089, 0xc3a7, NULL}, {0xc091, 0xc3a8, NULL},
  {0xc093, 0xc3a9, NULL}, {0xc095, 0xc3aa, NULL}, {0xc096, 0xc3ab, NULL},
  {0xc097, 0xc3ac, NULL}, {0xc0a1, 0xc3ad, NULL}, {0xc0a5, 0xc3ae, NULL},
  {0xc0a7, 0xc3af, NULL}, {0xc0a9, 0xc3b0, NULL}, {0xc0b1, 0xc3b1, NULL},
  {0xc0b7, 0xc3b2, NULL}, {0xc0e1, 0xc3b3, NULL}, {0xc0e2, 0xc3b4, NULL},
  {0xc0e5, 0xc3b5, NULL}, {0xc0e9, 0xc3b6, NULL}, {0xc0f1, 0xc3b7, NULL},
  {0xc0f3, 0xc3b8, NULL}, {0xc0f5, 0xc3b9, NULL}, {0xc0f6, 0xc3ba, NULL},
  {0xc0f7, 0xc3bb, NULL}, {0xc141, 0xc3bc, NULL}, {0xc142, 0xc3bd, NULL},
  {0xc145, 0xc3be, NULL}, {0xc149, 0xc3bf, NULL}, {0xc151, 0xc3c0, NULL},
  {0xc153, 0xc3c1, NULL}, {0xc155, 0xc3c2, NULL}, {0xc157, 0xc3c3, NULL},
  {0xc161, 0xc3c4, NULL}, {0xc165, 0xc3c5, NULL}, {0xc176, 0xc3c6, NULL},
  {0xc181, 0xc3c7, NULL}, {0xc185, 0xc3c8, NULL}, {0xc197, 0xc3c9, NULL},
  {0xc1a1, 0xc3ca, NULL}, {0xc1a2, 0xc3cb, NULL}, {0xc1a5, 0xc3cc, NULL},
  {0xc1a9, 0xc3cd, NULL}, {0xc1b1, 0xc3ce, NULL}, {0xc1b3, 0xc3cf, NULL},
  {0xc1b5, 0xc3d0, NULL}, {0xc1b7, 0xc3d1, NULL}, {0xc1c1, 0xc3d2, NULL},
  {0xc1c5, 0xc3d3, NULL}, {0xc1c9, 0xc3d4, NULL}, {0xc1d7, 0xc3d5, NULL},
  {0xc241, 0xc3d6, NULL}, {0xc245, 0xc3d7, NULL}, {0xc249, 0xc3d8, NULL},
  {0xc251, 0xc3d9, NULL}, {0xc253, 0xc3da, NULL}, {0xc255, 0xc3db, NULL},
  {0xc257, 0xc3dc, NULL}, {0xc261, 0xc3dd, NULL}, {0xc271, 0xc3de, NULL},
  {0xc281, 0xc3df, NULL}, {0xc282, 0xc3e0, NULL}, {0xc285, 0xc3e1, NULL},
  {0xc289, 0xc3e2, NULL}, {0xc291, 0xc3e3, NULL}, {0xc293, 0xc3e4, NULL},
  {0xc295, 0xc3e5, NULL}, {0xc297, 0xc3e6, NULL}, {0xc2a1, 0xc3e7, NULL},
  {0xc2b6, 0xc3e8, NULL}, {0xc2c1, 0xc3e9, NULL}, {0xc2c5, 0xc3ea, NULL},
  {0xc2e1, 0xc3eb, NULL}, {0xc2e5, 0xc3ec, NULL}, {0xc2e9, 0xc3ed, NULL},
  {0xc2f1, 0xc3ee, NULL}, {0xc2f3, 0xc3ef, NULL}, {0xc2f5, 0xc3f0, NULL},
  {0xc2f7, 0xc3f1, NULL}, {0xc341, 0xc3f2, NULL}, {0xc345, 0xc3f3, NULL},
  {0xc349, 0xc3f4, NULL}, {0xc351, 0xc3f5, NULL}, {0xc357, 0xc3f6, NULL},
  {0xc361, 0xc3f7, NULL}, {0xc362, 0xc3f8, NULL}, {0xc365, 0xc3f9, NULL},
  {0xc369, 0xc3fa, NULL}, {0xc371, 0xc3fb, NULL}, {0xc373, 0xc3fc, NULL},
  {0xc375, 0xc3fd, NULL}, {0xc377, 0xc3fe, NULL}, {0xc3a1, 0xc4a1, NULL},
  {0xc3a2, 0xc4a2, NULL}, {0xc3a5, 0xc4a3, NULL}, {0xc3a8, 0xc4a4, NULL},
  {0xc3a9, 0xc4a5, NULL}, {0xc3aa, 0xc4a6, NULL}, {0xc3b1, 0xc4a7, NULL},
  {0xc3b3, 0xc4a8, NULL}, {0xc3b5, 0xc4a9, NULL}, {0xc3b7, 0xc4aa, NULL},
  {0xc461, 0xc4ab, NULL}, {0xc462, 0xc4ac, NULL}, {0xc465, 0xc4ad, NULL},
  {0xc469, 0xc4ae, NULL}, {0xc471, 0xc4af, NULL}, {0xc473, 0xc4b0, NULL},
  {0xc475, 0xc4b1, NULL}, {0xc477, 0xc4b2, NULL}, {0xc481, 0xc4b3, NULL},
  {0xc482, 0xc4b4, NULL}, {0xc485, 0xc4b5, NULL}, {0xc489, 0xc4b6, NULL},
  {0xc491, 0xc4b7, NULL}, {0xc493, 0xc4b8, NULL}, {0xc495, 0xc4b9, NULL},
  {0xc496, 0xc4ba, NULL}, {0xc497, 0xc4bb, NULL}, {0xc4a1, 0xc4bc, NULL},
  {0xc4a2, 0xc4bd, NULL}, {0xc4b7, 0xc4be, NULL}, {0xc4e1, 0xc4bf, NULL},
  {0xc4e2, 0xc4c0, NULL}, {0xc4e5, 0xc4c1, NULL}, {0xc4e8, 0xc4c2, NULL},
  {0xc4e9, 0xc4c3, NULL}, {0xc4f1, 0xc4c4, NULL}, {0xc4f3, 0xc4c5, NULL},
  {0xc4f5, 0xc4c6, NULL}, {0xc4f6, 0xc4c7, NULL}, {0xc4f7, 0xc4c8, NULL},
  {0xc541, 0xc4c9, NULL}, {0xc542, 0xc4ca, NULL}, {0xc545, 0xc4cb, NULL},
  {0xc549, 0xc4cc, NULL}, {0xc551, 0xc4cd, NULL}, {0xc553, 0xc4ce, NULL},
  {0xc555, 0xc4cf, NULL}, {0xc557, 0xc4d0, NULL}, {0xc561, 0xc4d1, NULL},
  {0xc565, 0xc4d2, NULL}, {0xc569, 0xc4d3, NULL}, {0xc571, 0xc4d4, NULL},
  {0xc573, 0xc4d5, NULL}, {0xc575, 0xc4d6, NULL}, {0xc576, 0xc4d7, NULL},
  {0xc577, 0xc4d8, NULL}, {0xc581, 0xc4d9, NULL}, {0xc5a1, 0xc4da, NULL},
  {0xc5a2, 0xc4db, NULL}, {0xc5a5, 0xc4dc, NULL}, {0xc5a9, 0xc4dd, NULL},
  {0xc5b1, 0xc4de, NULL}, {0xc5b3, 0xc4df, NULL}, {0xc5b5, 0xc4e0, NULL},
  {0xc5b7, 0xc4e1, NULL}, {0xc5c1, 0xc4e2, NULL}, {0xc5c2, 0xc4e3, NULL},
  {0xc5c5, 0xc4e4, NULL}, {0xc5c9, 0xc4e5, NULL}, {0xc5d1, 0xc4e6, NULL},
  {0xc5d7, 0xc4e7, NULL}, {0xc5e1, 0xc4e8, NULL}, {0xc5f7, 0xc4e9, NULL},
  {0xc641, 0xc4ea, NULL}, {0xc649, 0xc4eb, NULL}, {0xc661, 0xc4ec, NULL},
  {0xc681, 0xc4ed, NULL}, {0xc682, 0xc4ee, NULL}, {0xc685, 0xc4ef, NULL},
  {0xc689, 0xc4f0, NULL}, {0xc691, 0xc4f1, NULL}, {0xc693, 0xc4f2, NULL},
  {0xc695, 0xc4f3, NULL}, {0xc697, 0xc4f4, NULL}, {0xc6a1, 0xc4f5, NULL},
  {0xc6a5, 0xc4f6, NULL}, {0xc6a9, 0xc4f7, NULL}, {0xc6b7, 0xc4f8, NULL},
  {0xc6c1, 0xc4f9, NULL}, {0xc6d7, 0xc4fa, NULL}, {0xc6e1, 0xc4fb, NULL},
  {0xc6e2, 0xc4fc, NULL}, {0xc6e5, 0xc4fd, NULL}, {0xc6e9, 0xc4fe, NULL},
  {0xc6f1, 0xc5a1, NULL}, {0xc6f3, 0xc5a2, NULL}, {0xc6f5, 0xc5a3, NULL},
  {0xc6f7, 0xc5a4, NULL}, {0xc741, 0xc5a5, NULL}, {0xc745, 0xc5a6, NULL},
  {0xc749, 0xc5a7, NULL}, {0xc751, 0xc5a8, NULL}, {0xc761, 0xc5a9, NULL},
  {0xc762, 0xc5aa, NULL}, {0xc765, 0xc5ab, NULL}, {0xc769, 0xc5ac, NULL},
  {0xc771, 0xc5ad, NULL}, {0xc773, 0xc5ae, NULL}, {0xc777, 0xc5af, NULL},
  {0xc7a1, 0xc5b0, NULL}, {0xc7a2, 0xc5b1, NULL}, {0xc7a5, 0xc5b2, NULL},
  {0xc7a9, 0xc5b3, NULL}, {0xc7b1, 0xc5b4, NULL}, {0xc7b3, 0xc5b5, NULL},
  {0xc7b5, 0xc5b6, NULL}, {0xc7b7, 0xc5b7, NULL}, {0xc861, 0xc5b8, NULL},
  {0xc862, 0xc5b9, NULL}, {0xc865, 0xc5ba, NULL}, {0xc869, 0xc5bb, NULL},
  {0xc86a, 0xc5bc, NULL}, {0xc871, 0xc5bd, NULL}, {0xc873, 0xc5be, NULL},
  {0xc875, 0xc5bf, NULL}, {0xc876, 0xc5c0, NULL}, {0xc877, 0xc5c1, NULL},
  {0xc881, 0xc5c2, NULL}, {0xc882, 0xc5c3, NULL}, {0xc885, 0xc5c4, NULL},
  {0xc889, 0xc5c5, NULL}, {0xc891, 0xc5c6, NULL}, {0xc893, 0xc5c7, NULL},
  {0xc895, 0xc5c8, NULL}, {0xc896, 0xc5c9, NULL}, {0xc897, 0xc5ca, NULL},
  {0xc8a1, 0xc5cb, NULL}, {0xc8b7, 0xc5cc, NULL}, {0xc8e1, 0xc5cd, NULL},
  {0xc8e2, 0xc5ce, NULL}, {0xc8e5, 0xc5cf, NULL}, {0xc8e9, 0xc5d0, NULL},
  {0xc8eb, 0xc5d1, NULL}, {0xc8f1, 0xc5d2, NULL}, {0xc8f3, 0xc5d3, NULL},
  {0xc8f5, 0xc5d4, NULL}, {0xc8f6, 0xc5d5, NULL}, {0xc8f7, 0xc5d6, NULL},
  {0xc941, 0xc5d7, NULL}, {0xc942, 0xc5d8, NULL}, {0xc945, 0xc5d9, NULL},
  {0xc949, 0xc5da, NULL}, {0xc951, 0xc5db, NULL}, {0xc953, 0xc5dc, NULL},
  {0xc955, 0xc5dd, NULL}, {0xc957, 0xc5de, NULL}, {0xc961, 0xc5df, NULL},
  {0xc965, 0xc5e0, NULL}, {0xc976, 0xc5e1, NULL}, {0xc981, 0xc5e2, NULL},
  {0xc985, 0xc5e3, NULL}, {0xc9a1, 0xc5e4, NULL}, {0xc9a2, 0xc5e5, NULL},
  {0xc9a5, 0xc5e6, NULL}, {0xc9a9, 0xc5e7, NULL}, {0xc9b1, 0xc5e8, NULL},
  {0xc9b3, 0xc5e9, NULL}, {0xc9b5, 0xc5ea, NULL}, {0xc9b7, 0xc5eb, NULL},
  {0xc9bc, 0xc5ec, NULL}, {0xc9c1, 0xc5ed, NULL}, {0xc9c5, 0xc5ee, NULL},
  {0xc9e1, 0xc5ef, NULL}, {0xca41, 0xc5f0, NULL}, {0xca45, 0xc5f1, NULL},
  {0xca55, 0xc5f2, NULL}, {0xca57, 0xc5f3, NULL}, {0xca61, 0xc5f4, NULL},
  {0xca81, 0xc5f5, NULL}, {0xca82, 0xc5f6, NULL}, {0xca85, 0xc5f7, NULL},
  {0xca89, 0xc5f8, NULL}, {0xca91, 0xc5f9, NULL}, {0xca93, 0xc5fa, NULL},
  {0xca95, 0xc5fb, NULL}, {0xca97, 0xc5fc, NULL}, {0xcaa1, 0xc5fd, NULL},
  {0xcab6, 0xc5fe, NULL}, {0xcac1, 0xc6a1, NULL}, {0xcae1, 0xc6a2, NULL},
  {0xcae2, 0xc6a3, NULL}, {0xcae5, 0xc6a4, NULL}, {0xcae9, 0xc6a5, NULL},
  {0xcaf1, 0xc6a6, NULL}, {0xcaf3, 0xc6a7, NULL}, {0xcaf7, 0xc6a8, NULL},
  {0xcb41, 0xc6a9, NULL}, {0xcb45, 0xc6aa, NULL}, {0xcb49, 0xc6ab, NULL},
  {0xcb51, 0xc6ac, NULL}, {0xcb57, 0xc6ad, NULL}, {0xcb61, 0xc6ae, NULL},
  {0xcb62, 0xc6af, NULL}, {0xcb65, 0xc6b0, NULL}, {0xcb68, 0xc6b1, NULL},
  {0xcb69, 0xc6b2, NULL}, {0xcb6b, 0xc6b3, NULL}, {0xcb71, 0xc6b4, NULL},
  {0xcb73, 0xc6b5, NULL}, {0xcb75, 0xc6b6, NULL}, {0xcb81, 0xc6b7, NULL},
  {0xcb85, 0xc6b8, NULL}, {0xcb89, 0xc6b9, NULL}, {0xcb91, 0xc6ba, NULL},
  {0xcb93, 0xc6bb, NULL}, {0xcba1, 0xc6bc, NULL}, {0xcba2, 0xc6bd, NULL},
  {0xcba5, 0xc6be, NULL}, {0xcba9, 0xc6bf, NULL}, {0xcbb1, 0xc6c0, NULL},
  {0xcbb3, 0xc6c1, NULL}, {0xcbb5, 0xc6c2, NULL}, {0xcbb7, 0xc6c3, NULL},
  {0xcc61, 0xc6c4, NULL}, {0xcc62, 0xc6c5, NULL}, {0xcc63, 0xc6c6, NULL},
  {0xcc65, 0xc6c7, NULL}, {0xcc69, 0xc6c8, NULL}, {0xcc6b, 0xc6c9, NULL},
  {0xcc71, 0xc6ca, NULL}, {0xcc73, 0xc6cb, NULL}, {0xcc75, 0xc6cc, NULL},
  {0xcc76, 0xc6cd, NULL}, {0xcc77, 0xc6ce, NULL}, {0xcc7b, 0xc6cf, NULL},
  {0xcc81, 0xc6d0, NULL}, {0xcc82, 0xc6d1, NULL}, {0xcc85, 0xc6d2, NULL},
  {0xcc89, 0xc6d3, NULL}, {0xcc91, 0xc6d4, NULL}, {0xcc93, 0xc6d5, NULL},
  {0xcc95, 0xc6d6, NULL}, {0xcc96, 0xc6d7, NULL}, {0xcc97, 0xc6d8, NULL},
  {0xcca1, 0xc6d9, NULL}, {0xcca2, 0xc6da, NULL}, {0xcce1, 0xc6db, NULL},
  {0xcce2, 0xc6dc, NULL}, {0xcce5, 0xc6dd, NULL}, {0xcce9, 0xc6de, NULL},
  {0xccf1, 0xc6df, NULL}, {0xccf3, 0xc6e0, NULL}, {0xccf5, 0xc6e1, NULL},
  {0xccf6, 0xc6e2, NULL}, {0xccf7, 0xc6e3, NULL}, {0xcd41, 0xc6e4, NULL},
  {0xcd42, 0xc6e5, NULL}, {0xcd45, 0xc6e6, NULL}, {0xcd49, 0xc6e7, NULL},
  {0xcd51, 0xc6e8, NULL}, {0xcd53, 0xc6e9, NULL}, {0xcd55, 0xc6ea, NULL},
  {0xcd57, 0xc6eb, NULL}, {0xcd61, 0xc6ec, NULL}, {0xcd65, 0xc6ed, NULL},
  {0xcd69, 0xc6ee, NULL}, {0xcd71, 0xc6ef, NULL}, {0xcd73, 0xc6f0, NULL},
  {0xcd76, 0xc6f1, NULL}, {0xcd77, 0xc6f2, NULL}, {0xcd81, 0xc6f3, NULL},
  {0xcd89, 0xc6f4, NULL}, {0xcd93, 0xc6f5, NULL}, {0xcd95, 0xc6f6, NULL},
  {0xcda1, 0xc6f7, NULL}, {0xcda2, 0xc6f8, NULL}, {0xcda5, 0xc6f9, NULL},
  {0xcda9, 0xc6fa, NULL}, {0xcdb1, 0xc6fb, NULL}, {0xcdb3, 0xc6fc, NULL},
  {0xcdb5, 0xc6fd, NULL}, {0xcdb7, 0xc6fe, NULL}, {0xcdc1, 0xc7a1, NULL},
  {0xcdd7, 0xc7a2, NULL}, {0xce41, 0xc7a3, NULL}, {0xce45, 0xc7a4, NULL},
  {0xce61, 0xc7a5, NULL}, {0xce65, 0xc7a6, NULL}, {0xce69, 0xc7a7, NULL},
  {0xce73, 0xc7a8, NULL}, {0xce75, 0xc7a9, NULL}, {0xce81, 0xc7aa, NULL},
  {0xce82, 0xc7ab, NULL}, {0xce85, 0xc7ac, NULL}, {0xce88, 0xc7ad, NULL},
  {0xce89, 0xc7ae, NULL}, {0xce8b, 0xc7af, NULL}, {0xce91, 0xc7b0, NULL},
  {0xce93, 0xc7b1, NULL}, {0xce95, 0xc7b2, NULL}, {0xce97, 0xc7b3, NULL},
  {0xcea1, 0xc7b4, NULL}, {0xceb7, 0xc7b5, NULL}, {0xcee1, 0xc7b6, NULL},
  {0xcee5, 0xc7b7, NULL}, {0xcee9, 0xc7b8, NULL}, {0xcef1, 0xc7b9, NULL},
  {0xcef5, 0xc7ba, NULL}, {0xcf41, 0xc7bb, NULL}, {0xcf45, 0xc7bc, NULL},
  {0xcf49, 0xc7bd, NULL}, {0xcf51, 0xc7be, NULL}, {0xcf55, 0xc7bf, NULL},
  {0xcf57, 0xc7c0, NULL}, {0xcf61, 0xc7c1, NULL}, {0xcf65, 0xc7c2, NULL},
  {0xcf69, 0xc7c3, NULL}, {0xcf71, 0xc7c4, NULL}, {0xcf73, 0xc7c5, NULL},
  {0xcf75, 0xc7c6, NULL}, {0xcfa1, 0xc7c7, NULL}, {0xcfa2, 0xc7c8, NULL},
  {0xcfa5, 0xc7c9, NULL}, {0xcfa9, 0xc7ca, NULL}, {0xcfb1, 0xc7cb, NULL},
  {0xcfb3, 0xc7cc, NULL}, {0xcfb5, 0xc7cd, NULL}, {0xcfb7, 0xc7ce, NULL},
  {0xd061, 0xc7cf, NULL}, {0xd062, 0xc7d0, NULL}, {0xd065, 0xc7d1, NULL},
  {0xd069, 0xc7d2, NULL}, {0xd06e, 0xc7d3, NULL}, {0xd071, 0xc7d4, NULL},
  {0xd073, 0xc7d5, NULL}, {0xd075, 0xc7d6, NULL}, {0xd077, 0xc7d7, NULL},
  {0xd081, 0xc7d8, NULL}, {0xd082, 0xc7d9, NULL}, {0xd085, 0xc7da, NULL},
  {0xd089, 0xc7db, NULL}, {0xd091, 0xc7dc, NULL}, {0xd093, 0xc7dd, NULL},
  {0xd095, 0xc7de, NULL}, {0xd096, 0xc7df, NULL}, {0xd097, 0xc7e0, NULL},
  {0xd0a1, 0xc7e1, NULL}, {0xd0b7, 0xc7e2, NULL}, {0xd0e1, 0xc7e3, NULL},
  {0xd0e2, 0xc7e4, NULL}, {0xd0e5, 0xc7e5, NULL}, {0xd0e9, 0xc7e6, NULL},
  {0xd0eb, 0xc7e7, NULL}, {0xd0f1, 0xc7e8, NULL}, {0xd0f3, 0xc7e9, NULL},
  {0xd0f5, 0xc7ea, NULL}, {0xd0f7, 0xc7eb, NULL}, {0xd141, 0xc7ec, NULL},
  {0xd142, 0xc7ed, NULL}, {0xd145, 0xc7ee, NULL}, {0xd149, 0xc7ef, NULL},
  {0xd151, 0xc7f0, NULL}, {0xd153, 0xc7f1, NULL}, {0xd155, 0xc7f2, NULL},
  {0xd157, 0xc7f3, NULL}, {0xd161, 0xc7f4, NULL}, {0xd162, 0xc7f5, NULL},
  {0xd165, 0xc7f6, NULL}, {0xd169, 0xc7f7, NULL}, {0xd171, 0xc7f8, NULL},
  {0xd173, 0xc7f9, NULL}, {0xd175, 0xc7fa, NULL}, {0xd176, 0xc7fb, NULL},
  {0xd177, 0xc7fc, NULL}, {0xd181, 0xc7fd, NULL}, {0xd185, 0xc7fe, NULL},
  {0xd189, 0xc8a1, NULL}, {0xd193, 0xc8a2, NULL}, {0xd1a1, 0xc8a3, NULL},
  {0xd1a2, 0xc8a4, NULL}, {0xd1a5, 0xc8a5, NULL}, {0xd1a9, 0xc8a6, NULL},
  {0xd1ae, 0xc8a7, NULL}, {0xd1b1, 0xc8a8, NULL}, {0xd1b3, 0xc8a9, NULL},
  {0xd1b5, 0xc8aa, NULL}, {0xd1b7, 0xc8ab, NULL}, {0xd1bb, 0xc8ac, NULL},
  {0xd1c1, 0xc8ad, NULL}, {0xd1c2, 0xc8ae, NULL}, {0xd1c5, 0xc8af, NULL},
  {0xd1c9, 0xc8b0, NULL}, {0xd1d5, 0xc8b1, NULL}, {0xd1d7, 0xc8b2, NULL},
  {0xd1e1, 0xc8b3, NULL}, {0xd1e2, 0xc8b4, NULL}, {0xd1e5, 0xc8b5, NULL},
  {0xd1f5, 0xc8b6, NULL}, {0xd1f7, 0xc8b7, NULL}, {0xd241, 0xc8b8, NULL},
  {0xd242, 0xc8b9, NULL}, {0xd245, 0xc8ba, NULL}, {0xd249, 0xc8bb, NULL},
  {0xd253, 0xc8bc, NULL}, {0xd255, 0xc8bd, NULL}, {0xd257, 0xc8be, NULL},
  {0xd261, 0xc8bf, NULL}, {0xd265, 0xc8c0, NULL}, {0xd269, 0xc8c1, NULL},
  {0xd273, 0xc8c2, NULL}, {0xd275, 0xc8c3, NULL}, {0xd281, 0xc8c4, NULL},
  {0xd282, 0xc8c5, NULL}, {0xd285, 0xc8c6, NULL}, {0xd289, 0xc8c7, NULL},
  {0xd28e, 0xc8c8, NULL}, {0xd291, 0xc8c9, NULL}, {0xd295, 0xc8ca, NULL},
  {0xd297, 0xc8cb, NULL}, {0xd2a1, 0xc8cc, NULL}, {0xd2a5, 0xc8cd, NULL},
  {0xd2a9, 0xc8ce, NULL}, {0xd2b1, 0xc8cf, NULL}, {0xd2b7, 0xc8d0, NULL},
  {0xd2c1, 0xc8d1, NULL}, {0xd2c2, 0xc8d2, NULL}, {0xd2c5, 0xc8d3, NULL},
  {0xd2c9, 0xc8d4, NULL}, {0xd2d7, 0xc8d5, NULL}, {0xd2e1, 0xc8d6, NULL},
  {0xd2e2, 0xc8d7, NULL}, {0xd2e5, 0xc8d8, NULL}, {0xd2e9, 0xc8d9, NULL},
  {0xd2f1, 0xc8da, NULL}, {0xd2f3, 0xc8db, NULL}, {0xd2f5, 0xc8dc, NULL},
  {0xd2f7, 0xc8dd, NULL}, {0xd341, 0xc8de, NULL}, {0xd342, 0xc8df, NULL},
  {0xd345, 0xc8e0, NULL}, {0xd349, 0xc8e1, NULL}, {0xd351, 0xc8e2, NULL},
  {0xd355, 0xc8e3, NULL}, {0xd357, 0xc8e4, NULL}, {0xd361, 0xc8e5, NULL},
  {0xd362, 0xc8e6, NULL}, {0xd365, 0xc8e7, NULL}, {0xd367, 0xc8e8, NULL},
  {0xd368, 0xc8e9, NULL}, {0xd369, 0xc8ea, NULL}, {0xd36a, 0xc8eb, NULL},
  {0xd371, 0xc8ec, NULL}, {0xd373, 0xc8ed, NULL}, {0xd375, 0xc8ee, NULL},
  {0xd377, 0xc8ef, NULL}, {0xd37b, 0xc8f0, NULL}, {0xd381, 0xc8f1, NULL},
  {0xd385, 0xc8f2, NULL}, {0xd389, 0xc8f3, NULL}, {0xd391, 0xc8f4, NULL},
  {0xd393, 0xc8f5, NULL}, {0xd397, 0xc8f6, NULL}, {0xd3a1, 0xc8f7, NULL},
  {0xd3a2, 0xc8f8, NULL}, {0xd3a5, 0xc8f9, NULL}, {0xd3a9, 0xc8fa, NULL},
  {0xd3b1, 0xc8fb, NULL}, {0xd3b3, 0xc8fc, NULL}, {0xd3b5, 0xc8fd, NULL},
  {0xd3b7, 0xc8fe, NULL}
};

/* Hancode representation of Hangul characters ==============================*/

#define DOT "\xa0"
#define A "\xa1"
#define B "\xa2"
#define C "\xa3"
#define D "\xa4"
#define E "\xa5"
#define G "\xa7"
#define H "\xa8"
#define I "\xa9"
#define K "\xab"
#define L "\xac"
#define M "\xad"
#define N "\xae"
#define O "\xaf"
#define P "\xb0"
#define S "\xb3"
#define T "\xb4"
#define U "\xb5"
#define W "\xb7"
#define X "\xb8"
#define Z "\xba"
#define NUL ""

#define DOT_C '\xa0'
#define A_C '\xa1'
#define B_C '\xa2'
#define C_C '\xa3'
#define D_C '\xa4'
#define E_C '\xa5'
#define G_C '\xa7'
#define H_C '\xa8'
#define I_C '\xa9'
#define K_C '\xab'
#define L_C '\xac'
#define M_C '\xad'
#define N_C '\xae'
#define O_C '\xaf'
#define P_C '\xb0'
#define S_C '\xb3'
#define T_C '\xb4'
#define U_C '\xb5'
#define W_C '\xb7'
#define X_C '\xb8'
#define Z_C '\xba'

#define VOWELS A E I O U W

/* tables for decoding trigem codes =========================================*/

LOCAL string_t initial_consonants[32] = 
{ /* the initial consonants in a syllable */
  NUL, NUL,   K, K K,   N,   T, T T,   L, 
    M,   P, P P,   S, S S,   X,   C, C C,
    Z,   G,   D,   B,   H, NUL, NUL, NUL,
  NUL, NUL, NUL, NUL, NUL, NUL, NUL, NUL
};

LOCAL string_t vowels[32] =
{ /* the vowels in a syllable */
  NUL, NUL, NUL,   A,   A I, I A, I A I,     E,
  NUL, NUL, E I, I E, I E I,   O,   O A, O A I,
  NUL, NUL, O I, I O,     W, W E, W E I,   W I,
  NUL, NUL, I W,   U,   U I,   I,   NUL,   NUL
};

LOCAL string_t final_consonants[32] =
{ /* the final consonants in a syllable */
 NUL, NUL,   K, K K, K S,   N, N C, N H,
   T,   L, L K, L M, L P, L S, L D, L B,
 L H,   M, NUL,   P, P S,   S, S S,   X,
   C,   Z,   G,   D,   B,   H, NUL, NUL
};

#define SYLLABLE(trigem)   (((trigem) & 0x8000) != 0)

#define INITIAL_CONSONANT(trigem)   (initial_consonants[((trigem) >> 10) & 31])

#define VOWEL(trigem)   (vowels[((trigem) >> 5) & 31])

#define FINAL_CONSONANT(trigem) (final_consonants[(trigem) & 31])

/*---------------------------------------------------------------------------*/

#define STRING_BUFFER_SIZE 500

LOCAL pool_t string_pool;       /* string pool with all hancode syllables */
LOCAL long_t *hancode_trie;     /* trie used to segmentise hancode syllables */
LOCAL long_t hancode_trie_root; /* root node index of <hancode_trie> */

/* functions ================================================================*/

LOCAL int compare_hancode_entries (const void *entry1, const void *entry2)
/* Compare two trie entries. */
{
  trie_entry_t *hancode1 = (trie_entry_t *) entry1;
  trie_entry_t *hancode2 = (trie_entry_t *) entry2;

  return strcmp_no_case (hancode1->key, hancode2->key);
}

/*---------------------------------------------------------------------------*/

GLOBAL void init_hangul (void)
/* Initialise the hangul module. */
{
  /* KSC-code/Hancode pairs for trie generation */
  trie_entry_t hancodes[KSC_TABLE_SIZE]; 
  short_t i;
  pool_t hancode_trie_pool;

  string_pool = new_pool (sizeof (char));

  /* Build hancode strings from <ksc_table>; copy entries to <hancodes>. */
  for (i = 0; i < KSC_TABLE_SIZE; i++) 
  {
    char string_buffer[STRING_BUFFER_SIZE];
    string_t string;
    u_short_t trigem;
    
    /* Build hancode string out of trigem code. */
    trigem = ksc_table[i].trigem;
    string = string_buffer;
    if (SYLLABLE (trigem))
      string = copy_string (string, DOT, string_buffer + STRING_BUFFER_SIZE);
    
    string = copy_string (string, INITIAL_CONSONANT (trigem), 
			  string_buffer + STRING_BUFFER_SIZE);
    string = copy_string (string, VOWEL (trigem), 
			  string_buffer + STRING_BUFFER_SIZE);
    string = copy_string (string, FINAL_CONSONANT (trigem), 
			  string_buffer + STRING_BUFFER_SIZE);
    ksc_table[i].hancode = copy_string_to_pool (string_pool, string_buffer, 
						NULL);
    
    /* Copy to trie entry table. */
    hancodes[i].key = ksc_table[i].hancode;
    hancodes[i].content = ksc_table[i].ksc;
  }
  
  /* Sort the hancode strings and build the trie. */ 
  qsort (hancodes, KSC_TABLE_SIZE, sizeof (trie_entry_t), 
	 compare_hancode_entries);
  new_trie (KSC_TABLE_SIZE, hancodes, &hancode_trie_pool, &hancode_trie_root);
  hancode_trie = pool_to_vector (hancode_trie_pool);
  free_pool (hancode_trie_pool);
}  

/*---------------------------------------------------------------------------*/

GLOBAL void terminate_hangul (void)
/* Terminate the hangul module. */
{
  free (hancode_trie);
  free_pool (string_pool);
}  

/* conversion of Hancode to romanised Hangul ================================*/

/* Convert Hancode string <hancode_string> to romanised Hangul. */

GLOBAL string_t hancode_to_roman (string_t hancode_string)
{
  /* (nearly) YALE roman representation for each of the Hancode letters */
  static string_t romans[27] = 
  {
    ".", "a", "ph", "c", "th", "e", "", "kh", "h", "i", "",
    "k", "l", "m", "n", "o", "p", "", "", "s", "t",
    "u", "", "wu", "ng", "", "ch"
  };
  static char roman_string[STRING_BUFFER_SIZE];
  string_t roman_string_ptr, hancode_string_ptr, roman_segment;
  
  roman_string_ptr = roman_string;
  hancode_string_ptr = hancode_string;
  while (*hancode_string_ptr != EOS) 
  {
    if (ORD (*hancode_string_ptr) >= ORD (DOT_C) 
	&& ORD (*hancode_string_ptr) <= ORD (Z_C))
      /* Convert this character. */
    {
      roman_string_ptr = copy_string (roman_string_ptr, "{", 
				      roman_string + STRING_BUFFER_SIZE);
      
      while (ORD (*hancode_string_ptr) >= ORD (DOT_C)
	     && ORD (*hancode_string_ptr) <= ORD (Z_C)) 
      {
	roman_segment = romans[ORD(*hancode_string_ptr) - 0xa0];
	
	/* Handle some special cases. */
	switch (*hancode_string_ptr)
	{
	case X_C:
	  if (hancode_string_ptr > hancode_string 
	      && hancode_string_ptr[-1] == DOT_C)
	    roman_segment = "";
	  break;
	case I_C:
	  if ((hancode_string_ptr > hancode_string
	       && strchr (VOWELS, hancode_string_ptr[-1]) != NULL) 
	      || (hancode_string_ptr[1] != EOS 
		  && strchr (VOWELS, hancode_string_ptr[1]) != NULL))
	    roman_segment = "y";
	  break;
	case O_C:
	  if (hancode_string_ptr[1] == A_C)
	    roman_segment = "w";
	  break;
	case W_C:
	  if (hancode_string_ptr > hancode_string 
	      && hancode_string_ptr[-1] == I_C)
	    roman_segment = "u";
	  else if (hancode_string_ptr[1] != EOS 
		   && strchr (VOWELS, hancode_string_ptr[1]) != NULL)
	    roman_segment = "w";
	  break;
	default:
	  break;
	}
	
	roman_string_ptr = copy_string (roman_string_ptr, roman_segment, 
					roman_string + STRING_BUFFER_SIZE);
	hancode_string_ptr++;
      }
      
      roman_string_ptr = copy_string (roman_string_ptr, "}", 
				      roman_string + STRING_BUFFER_SIZE);
    }
    else 
      /* Copy this character literally. */
    {
      if (roman_string_ptr + 2 >= roman_string + STRING_BUFFER_SIZE)
	error ("string too long");
      
      *roman_string_ptr++ = *hancode_string_ptr++;
    }
  }
  
  *roman_string_ptr = EOS;
  
  return roman_string;
}

/* conversion of romanised Hangul to Hancode ================================*/

/* Convert romanised Hangul string <roman_string> to Hancode. */

GLOBAL string_t roman_to_hancode (string_t roman_string)
{
  /* all letter sequences that can be converted to hancode */
  static struct {string_t roman; string_t hancode;} romans[] = {
    /* Two-letter strings must come first. */
    {"ch", Z}, {"kh", G}, {"th", D}, {"ph", B}, {"wu", W}, {"ng", X}, 
    {"wa", O A},
    
    {"a", A}, {"c", C}, {"e", E}, {"h", H}, {"i", I}, {"k", K}, {"l", L}, 
    {"m", M}, {"n", N}, {"o", O}, {"p", P}, {"r", L}, {"s", S}, {"t", T}, 
    {"u", U}, {"w", W}, {"x", X}, {"y", I}, {".", DOT},
    {NULL, NULL}
  };
  static char hancode_string[STRING_BUFFER_SIZE];
  string_t roman_string_ptr, hancode_string_ptr;
  short_t i;
  
  roman_string_ptr = roman_string;
  hancode_string_ptr = hancode_string;
  while (*roman_string_ptr != EOS) 
  {
    if (*roman_string_ptr == '{') 
    {
      roman_string_ptr++;
      while (*roman_string_ptr != '}') 
      {
	if (*roman_string_ptr == EOS)
	  error ("missing \"}\" in romanised Hangul");
	
	/* Insert an "x" at the beginning of a syllable
	 * if a vowel is following. */
	if (roman_string_ptr[-1] == '.' 
	    && strrchr ("aeiouwy", 
			TO_LOWER (roman_string_ptr[0])) != NULL)
	  hancode_string_ptr 
	    = copy_string (hancode_string_ptr, X, 
			   hancode_string + STRING_BUFFER_SIZE);
	
	for (i = 0; romans[i].roman != NULL; i++) 
	{
	  if (strncmp_no_case (roman_string_ptr, romans[i].roman, 
			       strlen (romans[i].roman)) == 0)
	  {
	    hancode_string_ptr = copy_string (hancode_string_ptr, 
					      romans[i].hancode, 
					      hancode_string 
					      + STRING_BUFFER_SIZE);
	    roman_string_ptr += strlen (romans[i].roman);
	    break;
	  }
	}
	
	if (romans[i].roman == NULL)
	  error ("\"%c\" is not a romanised Hangul letter", 
		 *roman_string_ptr); 
      }
      /* Jump over closing "}" */
      roman_string_ptr++;
    } 
    else /* Copy literally. */
    {
      if (hancode_string_ptr + 2 >= hancode_string + STRING_BUFFER_SIZE)
	error ("string too long");
      
      *hancode_string_ptr++ = *roman_string_ptr++;
    }
  }
  
  *hancode_string_ptr = EOS;
  
  return hancode_string;
}

/* conversion of Hancode to KSC5601 =========================================*/

GLOBAL string_t hancode_to_ksc (string_t hancode_string)
/* Convert Hancode string <hancode_string> to KSC5601 code. */
{
  static char ksc_string[STRING_BUFFER_SIZE];
  string_t ksc_string_end = ksc_string + STRING_BUFFER_SIZE;
  string_t ksc_string_ptr, hancode_string_ptr;

  ksc_string_ptr = ksc_string;
  hancode_string_ptr = hancode_string;
  while (*hancode_string_ptr != EOS) 
  { /* KSC code is actually u_short_t, but trie entries are of type long_t. */
    long_t ksc_code, code; 
    long_t trie_node;
    string_t string_ptr;
    
    /* Search the trie until we have found the longest segment. */
    trie_node = hancode_trie_root;
    string_ptr = hancode_string_ptr;
    ksc_code = 0;
    while (lookup_trie (hancode_trie, &trie_node, &string_ptr, &code)) 
    {
      hancode_string_ptr = string_ptr; 
      ksc_code = code;
    }
    
    if (ksc_code != 0) /* Copy a KSC character. */
    { 
      if (ksc_string_ptr + 3 >= ksc_string_end)
	error ("string too long");
      *ksc_string_ptr++ = (ksc_code >> 8);
      *ksc_string_ptr++ = (ksc_code & 0xff);
    } 
    else if (*hancode_string_ptr == DOT_C)
    {
      ksc_string_ptr = copy_string (ksc_string_ptr, "{.}", ksc_string_end);
      hancode_string_ptr++;
    } 
    else /* Copy an ASCII character. */
    {
      if (ksc_string_ptr + 2 >= ksc_string_end)
	error ("string too long");
      
      *ksc_string_ptr++ = *hancode_string_ptr++;
    }
  }	
  
  *ksc_string_ptr = EOS;

  return ksc_string;
}

/* conversion of KSC5601 to Hancode =========================================*/

LOCAL string_t ksc_code_to_hancode (u_short_t ksc_code)
/* Return the Hancode string for <ksc_code>.
 * The result is a constant string. */
{
  short_t lower, upper; /* boundaries of searched range in <ksc_table> */
  
  lower = 0;
  upper = KSC_TABLE_SIZE - 1;
  
  /* Search binarily for <ksc_code> in <ksc_table>. */
  while (lower < upper) 
  {
    short_t middle = (lower + upper) / 2;
    
    if (ksc_table[middle].ksc < ksc_code)
      lower = middle + 1;
    else if (ksc_table[middle].ksc > ksc_code)
      upper = middle - 1;
    else
      lower = upper = middle;
  }
  
  if (lower > upper || ksc_table[lower].ksc != ksc_code)
    error ("unknown KSC code in string");
  
  return ksc_table[lower].hancode;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t ksc_to_hancode (string_t ksc_string)
/* Convert KSC5601 string <ksc_string> to hancode format.
 * The returned string remains valid until this function is called again. */
{
  static char hancode_string[STRING_BUFFER_SIZE];
  string_t ksc_string_ptr, hancode_string_ptr;
  
  hancode_string_ptr = hancode_string;
  ksc_string_ptr = ksc_string;
  while (*ksc_string_ptr != EOS) 
  {
    if (ORD (*ksc_string_ptr) < 0xa0) /* Copy an ASCII character. */
    {
      if (hancode_string_ptr + 2 >= hancode_string + STRING_BUFFER_SIZE)
	error ("string too long");
      
      *hancode_string_ptr++ = *ksc_string_ptr++;
    } 
    else /* Copy a KSC two-byte character. */
    {
      u_short_t ksc_code;
      
      if (ORD (ksc_string_ptr[1]) < 0xa0)
	error ("illegal KSC character");
      
      ksc_code = (ORD (ksc_string_ptr[0]) << 8) | ORD (ksc_string_ptr[1]);
      ksc_string_ptr += 2;
      hancode_string_ptr = 
	copy_string (hancode_string_ptr, ksc_code_to_hancode (ksc_code), 
		     hancode_string + STRING_BUFFER_SIZE);
    }
  }
  
  *hancode_string_ptr = EOS;
  
  return hancode_string;
}
