/* File "malmake.c":
 * Reads a project file and compiles all symbol files, rule files, and
 * lexicon files that must be updated. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#define _XOPEN_SOURCE
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "basic.h"
#include "files.h"
#include "malaga_files.h"
#include "input.h"

#undef GLOBAL
#define GLOBAL

/*---------------------------------------------------------------------------*/

#define MALSYM ("malsym" HANGUL_SUFFIX)
#define MALRUL ("malrul" HANGUL_SUFFIX)
#define MALLEX ("mallex" HANGUL_SUFFIX)

/*---------------------------------------------------------------------------*/

LOCAL string_t program_name; /* the name the program has been called by */

/* the source and object name of a malaga file */
typedef struct SOURCE_FILE_T
{
  struct SOURCE_FILE_T *next; /* next source file in this list */
  string_t name;              /* name of the next source file */
} source_file_t;

typedef struct
{
  source_file_t *source_files;
  string_t object_file;
} files_t;

/* the names of the files needed by Malaga */
LOCAL files_t symbol_files; 
LOCAL files_t extended_symbol_files;
LOCAL files_t lexicon_files;
LOCAL files_t allomorph_files;
LOCAL files_t morphology_files;
LOCAL files_t syntax_files;
LOCAL files_t navigation_files;

/*---------------------------------------------------------------------------*/

GLOBAL void error (string_t format, ...)
/* Print an error message and exit. */
{
  va_list arg;

  fprintf (stderr, "%s: ", program_name);

  va_start (arg, format);
  vfprintf (stderr, format, arg);
  va_end (arg);

  fprintf (stderr, "\n");

  exit (1);
}

/*---------------------------------------------------------------------------*/

LOCAL void execute (string_t program_name, string_t arg1, string_t arg2, 
		    string_t file_name)
/* Execute a process and wait for successful termination. */
{
  pid_t pid;
  int status;

  if (file_name == NULL) 
    file_name = arg1;

  fprintf (stderr, "compiling \"%s\"\n", name_in_path (file_name));
  switch (pid = fork ()) 
  {
  case -1:
    error ("can't execute \"%s\"", program_name);
    break;
    
  case 0:
    execlp (program_name, program_name, arg1, arg2, NULL);
    error ("can't execute \"%s\"", program_name);
    break;
    
  default:
    waitpid (pid, &status, 0);
    if (! WIFEXITED (status) || WEXITSTATUS (status) != 0)
      error ("compilation failed");
    
    break;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void free_files (files_t *files)
/* Free the space occupied by <files>. */
{
  source_file_t *source_file;
  
  free (files->object_file);
  
  source_file = files->source_files;
  while (source_file != NULL) 
  {
    source_file_t *next_file = source_file->next;
    
    free (source_file->name);
    free (source_file);
    source_file = next_file;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void free_all_files (void)
/* Free the space occupied by the file names. */
{
  free_files (&navigation_files);
  free_files (&syntax_files);
  free_files (&morphology_files);
  free_files (&lexicon_files);
  free_files (&extended_symbol_files);
  free_files (&symbol_files);
  free_files (&allomorph_files);
}

/*---------------------------------------------------------------------------*/

LOCAL time_t file_time (string_t file_name, bool_t must_exist)
/* Get the last modification time of <file_name>. */
{
  struct stat status;
  
  /* Assume earliest possible time for non-existing files. */
  if (stat (file_name, &status) == 0)
    return status.st_mtime;
  else if (must_exist)
    error ("can't access \"%s\"", file_name);

  return 0;
}

/*---------------------------------------------------------------------------*/

LOCAL void read_project_file (string_t project_file)
/* Read the project file. */
{
  FILE *project_stream;

  project_stream = fopen_save (project_file, "r");
  while (! feof (project_stream)) 
  {
    char project_line[INPUT_LINE_LENGTH];
    string_t project_line_ptr, argument, extension;
    files_t *files;
    
    read_line (project_stream, project_line, INPUT_LINE_LENGTH);
    project_line_ptr = project_line;
    
    if (*project_line_ptr == EOS)
      continue;
    
    argument = parse_word (&project_line_ptr);
    
    extension = NULL;
    files = NULL;
    if (strcmp_no_case (argument, "sym:") == 0) 
    {
      files = &symbol_files;
      extension = "sym_c";
    } 
    if (strcmp_no_case (argument, "esym:") == 0) 
    {
      files = &extended_symbol_files;
      extension = "esym_c";
    } 
    else if (strcmp_no_case (argument, "all:") == 0) 
    {
      files = &allomorph_files;
      extension = "all_c";
    } 
    else if (strcmp_no_case (argument, "lex:") == 0) 
    {
      files = &lexicon_files;
      extension = "lex_c";
    } 
    else if (strcmp_no_case (argument, "mor:") == 0) 
    {
      files = &morphology_files;
      extension = "mor_c";
    } 
    else if (strcmp_no_case (argument, "syn:") == 0) 
    {
      files = &syntax_files;
      extension = "syn_c";
    }
    else if (strcmp_no_case (argument, "nav:") == 0) 
    {
      files = &navigation_files;
      extension = "nav_c";
    }
    else if (strcmp_no_case (argument, "include:") == 0)
    {
      string_t include_file = parse_word (&project_line_ptr);
      string_t include_path = new_string (absolute_path (include_file,
							 project_file));
      
      parse_end (project_line_ptr);
      read_project_file (include_path);
      free (include_path);
      free (include_file);
    }
    
    free (argument);
    
    if (files != NULL) 
      /* Read the files in the current line
       * and include them into the appropriate file_list. */
    {
      while (*project_line_ptr != EOS) 
      {
	source_file_t *new_file;
	source_file_t **source_file_ptr;
	
	/* Read the next file name in the project file. */
	argument = parse_word (&project_line_ptr);
	
	/* Allocate a new source_file node. */
	new_file = (source_file_t *) new_mem (sizeof (source_file_t));
	new_file->name = new_string (absolute_path (argument, project_file));
	free (argument);

	/* Set the object file if it's the first source file. */
	if (files->object_file == NULL)
	  files->object_file = replace_extension (new_file->name, extension);
	
	/* Walk to the end of the list and append the new file. */
	source_file_ptr = &files->source_files;
	while (*source_file_ptr != NULL)
	  source_file_ptr = &(*source_file_ptr)->next;
	*source_file_ptr = new_file;
      }
    }
  }
  fclose_save (project_stream, project_file);
}

/*---------------------------------------------------------------------------*/

LOCAL bool_t current_code_version (string_t file)
/* Return if <file> is compiled with the current code version. */
{
  FILE *stream;
  common_header_t header;
  bool_t up_to_date;
  long_t code_version;
  struct stat status;
  
  if (has_extension (file, "sym_c") || has_extension (file, "esym_c"))
    code_version = SYMBOL_CODE_VERSION;
  else if (has_extension (file, "all_c") || has_extension (file, "mor_c")
	   || has_extension (file, "syn_c") || has_extension (file, "nav_c"))
    code_version = RULE_CODE_VERSION;
  else if (has_extension (file, "lex_c"))
    code_version = LEXICON_CODE_VERSION;
  else
    error ("internal (unknown file suffix)");

  /* Assume earliest possible time for non-existing files. */
  if (stat (file, &status) == -1)
    return FALSE;

  stream = fopen_save (file, "rb");
  fread_save (&header, sizeof (header), 1, stream, file);
  up_to_date = (header.code_version == code_version);
  fclose_save (stream, file);

  return up_to_date;
}

/*---------------------------------------------------------------------------*/

LOCAL bool_t is_obsolete (files_t *files, ...)
/* Check if <files>->object_file is older than one of <files>
 * or one of the optional files. */
{
  va_list arg;
  time_t object_time = file_time (files->object_file, FALSE);
  source_file_t *source_file;
  string_t file_name;
  bool_t obsolete;

  obsolete = FALSE;
  for (source_file = files->source_files; 
       source_file != NULL; 
       source_file = source_file->next) 
  {
    if (object_time < file_time (source_file->name, TRUE))
      obsolete = TRUE;
  }

  va_start (arg, files);
  for (file_name = va_arg (arg, string_t);
       file_name != NULL; 
       file_name = va_arg (arg, string_t)) 
  {
    if (object_time < file_time (file_name, TRUE))
      obsolete = TRUE;
  }
  va_end (arg);

  if (! current_code_version (files->object_file))
    obsolete = TRUE;

  return obsolete;
}

/*---------------------------------------------------------------------------*/

LOCAL void program_message (void)
/* Print some information about the program. */
{
  printf ("%s (%s) - Copyright (C) 1995-1998 Bjoern Beutel\n",
	  program_name, MALAGA_VERSION);
  printf ("This program comes with ABSOLUTELY NO WARRANTY.\n");
  printf ("This is free software which you may redistribute "
	  "under certain conditions.\n");
  printf ("For details, refer to the GNU General Public License.\n");
}

/*---------------------------------------------------------------------------*/

GLOBAL int main (int argc, string_t argv[])
/* The main function of "malmake". */
{
  string_t project_file;
  short_t i;

  program_name = name_in_path (argv[0]);

  /* Parse arguments. */

  if (argc == 2 && (strcmp_no_case (argv[1], "-version") == 0
		    || strcmp_no_case (argv[1], "-v") == 0))
  {
    program_message ();
    exit (0);
  }

  project_file = NULL;
  for (i = 1; i < argc; i++) 
  {
    string_t argument = ((*argv[i] == '-') 
			 ? argv[i] : absolute_path (argv[i], NULL));
    
    if (has_extension (argument, "pro"))
      set_file_name (&project_file, argument, NULL);
    else
      error ("illegal argument \"%s\"", argument);
  }

  if (project_file == NULL)
    error ("missing project file name");
  
  read_project_file (project_file);
  
  /* Test dependencies and compile files. */
  
  if (symbol_files.source_files == NULL)
    error ("missing symbol file names in project file");

  if (is_obsolete (&symbol_files, NULL))
    execute (MALSYM, symbol_files.source_files->name, NULL, NULL);

  if (extended_symbol_files.source_files != NULL)
  {
    if (is_obsolete (&extended_symbol_files, symbol_files.object_file, NULL))
      execute (MALSYM, extended_symbol_files.source_files->name,
	       symbol_files.object_file, NULL);
  }
  else
    extended_symbol_files.object_file = new_string (symbol_files.object_file);

  if (allomorph_files.source_files == NULL)
    error ("missing allomorph file names in project file");

  if (is_obsolete (&allomorph_files, symbol_files.object_file, NULL))
    execute (MALRUL, allomorph_files.source_files->name, 
	     symbol_files.object_file, NULL);
  
  if (lexicon_files.source_files == NULL)
    error ("missing lexicon file names in project file");
  
  if (is_obsolete (&lexicon_files, symbol_files.object_file, 
		   allomorph_files.object_file, NULL))
    execute (MALLEX, project_file, NULL, lexicon_files.source_files->name);
  
  if (morphology_files.source_files != NULL
      && is_obsolete (&morphology_files, symbol_files.object_file, NULL))
    execute (MALRUL, morphology_files.source_files->name, 
	     symbol_files.object_file, NULL);
  
  if (syntax_files.source_files != NULL 
      && is_obsolete (&syntax_files, extended_symbol_files.object_file, NULL))
    execute (MALRUL, syntax_files.source_files->name, 
	     extended_symbol_files.object_file, NULL);
  
  if (navigation_files.source_files != NULL 
      && is_obsolete (&navigation_files, extended_symbol_files.object_file, 
		      NULL))
    execute (MALRUL, navigation_files.source_files->name, 
	     extended_symbol_files.object_file, NULL);
  
  fprintf (stderr, "\"%s\" is up to date\n", name_in_path (project_file));
  
  free_all_files ();
  free (project_file);
  
  return 0;
}
