/* File "options.c":
 * Options that can be set by malaga and mallex. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "input.h"
#include "commands.h"
#include "instr_type.h"
#include "rule_type.h"
#include "rules.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL
#include "options.h"

/* constants ================================================================*/

#define HIDDEN_ATTRIBS_MAX 8 /* maximum of hidden attributes */

/* commands =================================================================*/

LOCAL void do_sort_records (string_t arguments)
/* Select the order in which attributes in a record are printed. */
{
  if (*arguments == EOS)
    printf ("sort-records: %s\n", 
	    attribute_order == INTERNAL_ORDER ? "internal" :
	    attribute_order == ALPHABETIC_ORDER ? "alphabetic" :
	    attribute_order == DEFINITION_ORDER ? "definition" : "");
  else
  {
    string_t argument;

    argument = parse_word (&arguments);
    if (strcmp_no_case (argument, "internal") == 0)
      attribute_order = INTERNAL_ORDER;
    else if (strcmp_no_case (argument, "alphabetic") == 0)
      attribute_order = ALPHABETIC_ORDER;
    else if (strcmp_no_case (argument, "definition") == 0)
      attribute_order = DEFINITION_ORDER;
    else 
      error ("\"internal\", \"alphabetic\" or \"definition\" expected, "
	     "not \"%s\"\n", argument);
    
    free (argument);
  }

  parse_end (arguments);
}

GLOBAL command_t sort_records_option =
{
  "sort-records", do_sort_records,
  "Arguments:\n"
  "  internal -- print record attributes in internal order\n"
  "  alphabetic -- print record attributes in alphabetic order\n"
  "  definition -- print record attributes in symbol-table order\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_hidden (string_t arguments)
/* Hide specified attributes when printing values. */
{
  static symbol_t hidden_attribs[HIDDEN_ATTRIBS_MAX];
  long_t i;
  
  hidden_attributes = hidden_attribs;
  
  if (*arguments == EOS)
  {
    if (num_hidden_attributes == 0)
      printf ("hidden: none\n");
    else
    {
      printf ("hidden:");
      for (i = 0; i < num_hidden_attributes; i++)
	printf (" %s", symbol_name (hidden_attributes[i]));
      printf ("\n");
    }
  }
  else
  {
    while (*arguments != EOS)
    {
      string_t argument = parse_word (&arguments);
      
      if (strcmp_no_case (argument, "none") == 0)
	num_hidden_attributes = 0;
      else if (*argument == '+')
      {
	symbol_t attribute = find_symbol (argument + 1);
	
	/* Check if attribute is already in list. */
	for (i = 0; i < num_hidden_attributes; i++)
	{
	  if (hidden_attributes[i] == attribute)
	    break;
	}
	
	/* Add attribute to list. */
	if (i == num_hidden_attributes)
	{
	  if (num_hidden_attributes >= HIDDEN_ATTRIBS_MAX)
	    error ("too many attributes to hide");
	  
	  hidden_attributes[num_hidden_attributes] = attribute;
	  num_hidden_attributes++;
	}
      }
      else if (*argument == '-')
      {
	symbol_t attribute = find_symbol (argument + 1);
	
	/* Find attribute in list. */
	for (i = 0; i < num_hidden_attributes; i++)
	{
	  if (hidden_attributes[i] == attribute)
	  {
	    /* Shift the following attributes */ 
	    for (i++; i < num_hidden_attributes; i++)
	      hidden_attributes[i-1] = hidden_attributes[i];
	    
	    num_hidden_attributes--;
	  }
	}
      }
      else
	error ("\"+...\", \"-...\" or \"none\" expected, not \"%s\"", 
	       argument);
      
      free (argument);
    }
  }
}

GLOBAL command_t hidden_option =
{
  "hidden", do_hidden,
  "Set hidden attributes in categories.\n"
  "Arguments: one or more of the following:\n"
  "  none -- don't hide any attributes\n"
  "  +<attribute> -- also hide <attribute>\n"
  "  -<attribute> -- don't hide <attribute> any longer\n"
};

/*---------------------------------------------------------------------------*/

#ifdef HANGUL

LOCAL void do_hangul (string_t arguments)
/* Switch to hangul output. */
{
  if (*arguments == EOS)
    printf ("hangul: %s\n", convert_to_ksc ? "yes" : "no");
  else
    convert_to_ksc = parse_yes_no (&arguments);

  parse_end (arguments);
}

GLOBAL command_t hangul_option =
{
  "hangul", do_hangul,
  "Arguments:\n"
  "  yes -- display output in KSC5601 Hangul characters\n"
  "  no -- display output in romanised Hangul\n"
};

#endif /* HANGUL */

/*---------------------------------------------------------------------------*/

LOCAL void do_heap_size (string_t arguments)
/* Change the value heap size. */
{
  if (*arguments == EOS)
    printf ("heap-size: %ld\n", get_value_heap_size ());
  else
  {
    long_t size;

    if (pc != -1)
      error ("can't set heap-size while executing rules");
    free_value_heap ();

    size = parse_integer (&arguments);
    if (size < 0)
      error ("don't use negative heap size");

    set_value_heap_size (size);
  }
  parse_end (arguments);
}

GLOBAL command_t heap_size_option =
{
  "heap-size", do_heap_size,
  "Arguments: <size> -- the new heap size\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_output_option (string_t arguments)
/* Determine if output is printed after analysis. */
{
  if (*arguments == EOS)
    printf ("output: %s\n", show_output ? "yes" : "no");
  else
    show_output = parse_yes_no (&arguments);
  parse_end (arguments);
}

GLOBAL command_t output_option =
{
  "output", do_output_option,
  "Arguments:\n"
  "  yes -- show output after analysis\n"
  "  no -- don't show output after analysis\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_result_option (string_t arguments)
/* Determine if result is shown after analysis. */
{
  if (*arguments == EOS)
    printf ("result: %s\n", show_result ? "yes" : "no");
  else
    show_result = parse_yes_no (&arguments);
  parse_end (arguments);
}

GLOBAL command_t result_option =
{
  "result", do_result_option,
  "Arguments:\n"
  "  yes -- show result after analysis\n"
  "  no -- don't show result after analysis\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_variables_option (string_t arguments)
/* Determine if variables are displayed in debug mode. */
{
  if (*arguments == EOS)
    printf ("variables: %s\n", show_variables ? "yes" : "no");
  else
    show_variables = parse_yes_no (&arguments);
  parse_end (arguments);
}

GLOBAL command_t variables_option =
{
  "variables", do_variables_option,
  "Arguments:\n"
  "  yes -- show variables automatically in debug mode\n"
  "  no -- don't show variables automatically\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_switch_option (string_t arguments)
/* Set switches that can be read from Malaga rules. */
{
  if (*arguments == EOS)
  {
    if (! start_switches ())
      printf ("switch: no switches defined\n");
    else
    {
      symbol_t key;
      value_t value;
      
      while (TRUE)
      {
	value = get_next_switch (&key);
	if (value == NULL)
	  break;

	printf ("switch \"%s\": ", symbol_name (key));
	print_value (value);
	printf ("\n");
      }
    }
  }
  else
  {
    string_t name;
    value_t value;

    name = parse_word (&arguments);

    if (IS_DIGIT (*arguments) || (*arguments == '-'))
      value = double_to_value (parse_double (&arguments));
    else if (*arguments == '"')
    {
      string_t string;

      string = parse_word (&arguments);
      value = string_to_value (string, NULL);
      free (string);
    }
    else
    {
      string_t symbol;

      symbol = parse_word (&arguments);
      value = symbol_to_value (find_symbol (symbol));
      free (symbol);
    }

    parse_end (arguments);
    set_switch (find_symbol (name), value);
    free (name);
  }
}

GLOBAL command_t switch_option =
{
  "switch", do_switch_option, 
  "Set a switch that can be read from Malaga rule files.\n"
  "Arguments: <switch> <value>\n"
  "  <value> may be a string, a symbol, or a number value\n"
};

/*---------------------------------------------------------------------------*/
