/* File "rule_code.c":
 * This module manages the emission of instructions
 * and keeps track of the stack index.
 * It also holds buffers for the compiled code. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "instr_type.h"
#include "rule_type.h"
#include "files.h"
#include "malaga_files.h"

#undef GLOBAL
#define GLOBAL

#include "rule_code.h"

/* constants ================================================================*/

#define BUFFER_SIZE 400 /* number of instructions that can be held in buffer */

/* variables ================================================================*/

LOCAL instr_t instr_buffer[BUFFER_SIZE];
LOCAL value_t value_buffer[BUFFER_SIZE];
/* The instruction buffer and its associated constant values.
 * They are implemented as two independent buffers instead of
 * a buffer of records because <value_buffer> is needed
 * for <clear_value_heap_except>. */

LOCAL short_t buffer_index = 0;
/* the index in both <instr_buffer> and <value_buffer> */

/* functions ================================================================*/

GLOBAL void init_code (short_t file_type)
/* Initialise <code> to contain compilation data for a file of <file_type>.
 * <file_type> may be ALLO_RULE_FILE, MORPHO_RULE_FILE or SYNTAX_RULE_FILE */
{
  code.file_type = file_type;
  code.stack_index = 0;
  code.next_instr_index = 0;
  code.rule_pool = new_pool (sizeof (rule_t));
  code.rule_set_pool = new_pool (sizeof (long_t));
  code.instr_pool = new_pool (sizeof (instr_t));
  code.value_pool = new_pool (sizeof (cell_t));
  code.string_pool = new_pool (sizeof (char));
  code.src_line_pool = new_pool (sizeof (src_line_t));
  code.var_pool = new_pool (sizeof (var_t));
  code.var_scope_pool = new_pool (sizeof (var_scope_t));
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_code (void)
/* Free all memory used by <code>. */
{
  free_pool (code.rule_pool);
  free_pool (code.rule_set_pool);
  free_pool (code.instr_pool);
  free_pool (code.value_pool);
  free_pool (code.string_pool);
  free_pool (code.src_line_pool);
  free_pool (code.var_pool);
  free_pool (code.var_scope_pool);
}

/*---------------------------------------------------------------------------*/

GLOBAL void write_code (string_t file_name)
/* Write <code> to <file_name>. */
{ 
  FILE *stream;
  rule_header_t rule_header;

  stream = fopen_save (file_name, "wb");

  /* Set rule file header data. */
  set_header (&rule_header.common_header, RULE_FILE, RULE_CODE_VERSION);
  rule_header.initial_rule_set = code.initial_rule_set;
  rule_header.initial_cat = code.initial_cat;
  rule_header.robust_rule = code.robust_rule;
  rule_header.pruning_rule = code.pruning_rule;
  rule_header.allo_rule = code.allo_rule;
  rule_header.filter_rule = code.filter_rule;
  rule_header.input_rule = code.input_rule;
  rule_header.rules_size = pool_items (code.rule_pool);
  rule_header.rule_sets_size = pool_items (code.rule_set_pool);
  rule_header.instrs_size = pool_items (code.instr_pool);
  rule_header.values_size = pool_items (code.value_pool);
  rule_header.src_lines_size = pool_items (code.src_line_pool);
  rule_header.vars_size = pool_items (code.var_pool);
  rule_header.var_scopes_size = pool_items (code.var_scope_pool);
  rule_header.strings_size = pool_items (code.string_pool);

  /* Write the header. */
  fwrite_save (&rule_header, sizeof (rule_header), 1, stream, file_name); 

  /* Write the tables. */
  write_pool (code.rule_pool, stream, file_name);
  write_pool (code.rule_set_pool, stream, file_name);
  write_pool (code.instr_pool, stream, file_name);
  write_pool (code.value_pool, stream, file_name);
  write_pool (code.src_line_pool, stream, file_name);
  write_pool (code.var_pool, stream, file_name);
  write_pool (code.var_scope_pool, stream, file_name);
  write_pool (code.string_pool, stream, file_name);

  fclose_save (stream, file_name);
}

/*---------------------------------------------------------------------------*/

LOCAL void set_stack_index (u_byte_t opcode, long_t info)
/* Set the stack index according to the given instruction. */
{ 
  /* We can not check for stack overflow here (sorry), because in
   * "choose" and "foreach", a PUSH_NULL instruction is patched, so
   * stack index is (transiently) not quite the real thing. */

  switch (opcode) 
  {
  case INS_ERROR:
  case INS_TERMINATE:
  case INS_NOP:
  case INS_SUCCEED:
  case INS_UNARY_MINUS_OP:
  case INS_GET_ATTRIBUTE:
  case INS_REMOVE_ATTRIBUTE:
  case INS_STD_FUNCTION:
  case INS_MATCH:
  case INS_ITERATE:
  case INS_JUMP:
  case INS_JUMP_NOW:
  case INS_JUMP_LATER:
    /* These instructions leave the stack size unchanged. */
    break;
  case INS_PUSH_VAR:
  case INS_PUSH_CONST:
  case INS_PUSH_SYMBOL:
  case INS_PUSH_PATTERN_VAR:
  case INS_PUSH_1ST_ELEMENT:
  case INS_JUMP_SUBRULE: /* subtract number of parameters from stack_index */
    code.stack_index++;
    break;
  case INS_PUSH_NULL:
    code.stack_index += info;
    break;
  case INS_ADD_END_STATE:
  case INS_ADD_STATE:
  case INS_TERMINATE_IF_NULL:
  case INS_DOT_OPERATION:
  case INS_PLUS_OPERATION:
  case INS_MINUS_OPERATION:
  case INS_ASTERISK_OPERATION:
  case INS_SLASH_OPERATION:
  case INS_SET_VAR:
  case INS_PLUS_VAR:
  case INS_MINUS_VAR:
  case INS_ASTERISK_VAR:
  case INS_SLASH_VAR:
  case INS_JUMP_IF_NULL:
  case INS_JUMP_IF_NOT_NULL:
  case INS_JUMP_IF_YES:
  case INS_JUMP_IF_NO:
  case INS_RETURN: /* no instruction after this instruction is executed. */
    code.stack_index -= 1;
    break;
  case INS_ADD_ALLO:
  case INS_SET_VAR_PATH:
  case INS_PLUS_VAR_PATH:
  case INS_MINUS_VAR_PATH:
  case INS_ASTERISK_VAR_PATH:
  case INS_SLASH_VAR_PATH:
  case INS_JUMP_IF_EQUAL:
  case INS_JUMP_IF_NOT_EQUAL:
  case INS_JUMP_IF_CONGR:
  case INS_JUMP_IF_NOT_CONGR:
  case INS_JUMP_IF_IN:
  case INS_JUMP_IF_NOT_IN:
  case INS_JUMP_IF_LESS:
  case INS_JUMP_IF_NOT_LESS:
  case INS_JUMP_IF_GREATER:
  case INS_JUMP_IF_NOT_GREATER:
    code.stack_index -= 2;
    break;
  case INS_POP:
    code.stack_index -= info;
    break;
  case INS_BUILD_LIST:
  case INS_BUILD_PATH:
    code.stack_index -= (info - 1);
    break;
  case INS_BUILD_RECORD:
    code.stack_index -= (2*info - 1);
    break;
  default:
    error ("internal (instruction %d unknown in rule_code)", opcode);
  }
}

/*---------------------------------------------------------------------------*/

LOCAL instr_t *emit_instr_local (u_byte_t opcode, long_t info)
/* Emit an instruction to the instruction pool. DO NOT FLUSH BUFFER!
 * Return the address of the instruction in the pool. */
{
  instr_t instr;
  instr_t *instr_ptr;

  if (info < INSTR_INFO_MIN || info > INSTR_INFO_MAX)
    error ("internal (instruction info out of range)");

  /* Fill the next instruction */
  instr = INSTR (opcode, info);

  /* Generate instruction. */
  instr_ptr = (instr_t *) copy_to_pool (code.instr_pool, &instr, 1, NULL);
  set_stack_index (opcode, info);
  code.next_instr_index = pool_items (code.instr_pool);

  return instr_ptr;
}

/* functions that support constant folding of values ========================*/

GLOBAL void buffer_instr (u_byte_t opcode, long_t info)
/* Buffer the instructions BUILD_LIST, BUILD_RECORD, PUSH_SYMBOL,
 * and PUSH_CONST for constant folding. */

{
  DB_ASSERT (info >= INSTR_INFO_MIN && info <= INSTR_INFO_MAX);

  clear_value_heap_except (buffer_index, value_buffer);

  switch (opcode) 
  {
  case INS_PUSH_SYMBOL:
    if (buffer_index >= BUFFER_SIZE)
      error ("constant value too complex");
    
    instr_buffer[buffer_index] = INSTR (INS_PUSH_SYMBOL, info);
    value_buffer[buffer_index] = symbol_to_value (info);
    buffer_index++;
    break;
    
  case INS_PUSH_CONST:
    if (buffer_index >= BUFFER_SIZE)
      error ("constant value too complex");
    
    instr_buffer[buffer_index] = INSTR (INS_PUSH_CONST, info);
    value_buffer[buffer_index] = (value_t) pool_item (code.value_pool, info);
    buffer_index++;
    break;
    
  case INS_BUILD_LIST:
    if (buffer_index >= info) /* Execute the operation in the buffer. */
    {
      instr_buffer[buffer_index - info] = INSTR (INS_PUSH_CONST, -1);
      value_buffer[buffer_index - info] 
	= build_list (info, value_buffer + buffer_index - info);
      buffer_index -= (info - 1);
    } 
    else
      emit_instr (opcode, info);
    break;
    
  case INS_BUILD_RECORD:
    if (buffer_index >= 2*info) /* Execute the operation in the buffer. */
    {
      instr_buffer[buffer_index - 2*info] = INSTR (INS_PUSH_CONST, -1);
      value_buffer[buffer_index - 2*info]
	= build_record (info, value_buffer + buffer_index - 2*info);
      buffer_index -= (2*info - 1);
    }
    else
      emit_instr (opcode, info);
    break;
    
  case INS_BUILD_PATH:
    if (buffer_index >= info) /* Execute the operation in the buffer. */
    {
      instr_buffer[buffer_index - info] = INSTR (INS_PUSH_CONST, -1);
      value_buffer[buffer_index - info] 
	= build_path (info, value_buffer + buffer_index - info);
      buffer_index -= (info - 1);
    } 
    else
      emit_instr (opcode, info);
    break;

  case INS_DOT_OPERATION:
  case INS_PLUS_OPERATION:
  case INS_MINUS_OPERATION:
  case INS_ASTERISK_OPERATION:
  case INS_SLASH_OPERATION:
    if (buffer_index >= 2) /* Execute the operation in the buffer. */
    {
      value_t arg1, arg2, result;

      arg1 = value_buffer[buffer_index - 2];
      arg2 = value_buffer[buffer_index - 1];
      switch (opcode)
      {
      case INS_DOT_OPERATION: 
	result = dot_operation (arg1, arg2); 
	break;
      case INS_PLUS_OPERATION: 
	result = plus_operation (arg1, arg2); 
	break;
      case INS_MINUS_OPERATION: 
	result = minus_operation (arg1, arg2); 
	break;
      case INS_ASTERISK_OPERATION: 
	result = asterisk_operation (arg1, arg2); 
	break;
      case INS_SLASH_OPERATION: 
	result = slash_operation (arg1, arg2); 
	break;
      default:
	error ("internal (unknown instruction in constant folding)");
      }
      instr_buffer[buffer_index - 2] = INSTR (INS_PUSH_CONST, -1);
      value_buffer[buffer_index - 2] = result;
      buffer_index -= 1;
    }
    else
      emit_instr (opcode, info);
    break;

  case INS_UNARY_MINUS_OP:
  case INS_GET_ATTRIBUTE:
  case INS_REMOVE_ATTRIBUTE:
    if (buffer_index >= 1) /* Execute the operation in the buffer. */
    {
      value_t arg, result;
      
      arg = value_buffer[buffer_index - 1];
      switch (opcode)
      {
      case INS_UNARY_MINUS_OP: 
	result = unary_minus_operation (arg); 
	break;
      case INS_GET_ATTRIBUTE: 
	result = get_attribute (arg, info); 
	break;
      case INS_REMOVE_ATTRIBUTE: 
	result = remove_attribute (arg, info); 
	break;
      default:
	error ("internal (unknown instruction in constant folding)");
      }
      instr_buffer[buffer_index - 1] = INSTR (INS_PUSH_CONST, -1);
      value_buffer[buffer_index - 1] = result;
    } 
    else
      emit_instr (opcode, info);
    break;

  default:
    emit_instr (opcode, info);
    break;
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void buffer_push_const_instr (value_t value)
/* Buffer the instruction PUSH_CONST with the given value. */
{
  if (buffer_index >= BUFFER_SIZE)
    error ("constant value too complex");

  instr_buffer[buffer_index] = INSTR (INS_PUSH_CONST, -1);
  value_buffer[buffer_index] = value;
  buffer_index++;
}

/*---------------------------------------------------------------------------*/

GLOBAL void flush_buffer (void)
/* Emit the instructions that are still in the buffer. */
{
  short_t i;

  for (i = 0; i < buffer_index; i++) 
  {
    switch (OPCODE (instr_buffer[i])) 
    {
    case INS_PUSH_CONST:
    {
      long_t value_index;
      
      if (INSTR_INFO (instr_buffer[i]) == -1)
	copy_value_to_pool (code.value_pool, value_buffer[i], &value_index);
      else
	value_index = INSTR_INFO (instr_buffer[i]);

      emit_instr_local (INS_PUSH_CONST, value_index);
      break;
    }

    case INS_PUSH_SYMBOL: 
      emit_instr_local (INS_PUSH_SYMBOL, INSTR_INFO (instr_buffer[i]));
      break;
    }
  }

  buffer_index = 0;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_buffer_top_value (void)
/* Test if the buffer contains a value and return the top value. 
 * The value is valid until "clear_value_heap" is called again. */
{
  if (buffer_index == 0)
    error ("internal (buffer is empty)");

  buffer_index--;
  return value_buffer[buffer_index];
}

/*---------------------------------------------------------------------------*/

GLOBAL instr_t *emit_instr (u_byte_t opcode, long_t info)
/* Emit an instruction to the instruction pool (flushes buffer before)
 * and return the address of the instruction in the pool. */
{
  flush_buffer ();
  return emit_instr_local (opcode, info);
}

/*---------------------------------------------------------------------------*/
