/* File "rule_symbols.c":
 * Implements the symbol table for variables and rules. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "instr_type.h"
#include "rule_type.h"
#include "rule_code.h"

#undef GLOBAL
#define GLOBAL

#include "rule_symbols.h"

/* types ====================================================================*/

/* The table for variables and rules is implemented as a binary tree.
 * It is sorted alphabetically after variable and rule names. */

typedef struct COMP_SCOPE_T
/* information about a certain definition of a variable and its scope */
{
  struct COMP_SCOPE_T *next_scope;
  long_t first_instr; /* index of first instruction
		       * where variable is accessable */
  long_t last_instr;  /* index of last instruction
		       * where variable is accessable.
		       * -1 means "scope not yet closed" */
  long_t stack_index;
} comp_scope_t;

typedef struct SYMBOL_NODE_T
/* a symbol table node, contains information for variable and rule usage */
{
  struct SYMBOL_NODE_T *left_son;  /* sons with name that come before <name> */
  struct SYMBOL_NODE_T *right_son; /* sons with name that come after <name> */
  string_t name;                   /* name of the node in <code.string_pool> */
  long_t name_index;               /* name index in <code.string_pool> */

  long_t rule_index;               /* rule index in <code.rule_pool> or -1 */
  rule_t *rule_ptr;                /* rule pointer or NULL */

  long_t const_index;              /* index in <code.value_pool> for 
				    * named constants  or -1 */
  comp_scope_t *first_scope;       /* first use of <name> as var or NULL */
  comp_scope_t *current_scope;     /* current use of <name> as a var or NULL */
} symbol_node_t;

/* variables ================================================================*/

LOCAL symbol_node_t *symbol_table = NULL; /* root node of the symbol table */

/* basic functions ==========================================================*/

LOCAL symbol_node_t *find_symbol_node (string_t name, 
				       bool_t insert_if_necessary)
/* Find and return a symbol node with <name>.
 * if <insert_if_necessary> == TRUE, insert a new node if necessary,
 * else return NULL if the node doesn't exist. */
{
  /* points to a pointer to the current node (this pointer may be NULL) */
  symbol_node_t **node_ptr; 
  
  node_ptr = &symbol_table;
  while (TRUE) 
  {
    short_t comp_result;
    symbol_node_t *node;
    
    if (*node_ptr == NULL)
    {
      if (insert_if_necessary) /* We have to insert the node. */
      {
	/* Allocate and initialise node. */
	node = (symbol_node_t *) new_mem (sizeof (symbol_node_t));
	node->left_son = NULL;
	node->right_son = NULL;
	node->name = copy_string_to_pool (code.string_pool, name, 
					  &node->name_index);
	node->rule_index = -1;
	node->rule_ptr = NULL;
	node->const_index = -1;
	node->first_scope = NULL;
	node->current_scope = NULL;
	
	/* Link <node> into tree. */
	*node_ptr = node;
	return node;
      }
      else
	return NULL;
    }

    /* Link <node> into tree. */
    node = *node_ptr;
    comp_result = strcmp_no_case (name, node->name);
    if (comp_result < 0)
      node_ptr = &node->left_son;
    else if (comp_result > 0)
      node_ptr = &node->right_son;
    else
      return node;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void traverse_tree (symbol_node_t *tree, 
			  void (*node_function) (symbol_node_t *))
/* Traverse <tree> alphabetically; for each node, execute <node_function>. */
{
  if (tree != NULL)
  {
    traverse_tree (tree->left_son, node_function);
    node_function (tree);
    traverse_tree (tree->right_son, node_function);
  }
}

/* functions that use the symbol table for rules ============================*/

LOCAL void create_rule_node (symbol_node_t *node)
/* Create an empty rule entry for <node>. */
{
  rule_t rule;

  rule.name = node->name_index;
  rule.first_instr = -1;
  rule.type = -1;
  rule.num_params = -1;
  node->rule_ptr = (rule_t *) copy_to_pool (code.rule_pool, &rule, 1, 
					    &node->rule_index);
}

/*---------------------------------------------------------------------------*/

GLOBAL void enter_function (string_t name, long_t index)
/* Associate standard function <name> with <index>. */
{
  symbol_node_t *node;

  node = find_symbol_node (name, TRUE);
  if (node->rule_ptr != NULL || node->rule_index != -1)
    error ("function \"%s\" already defined", name);

  node->rule_index = index;
}

/*---------------------------------------------------------------------------*/

GLOBAL long_t enter_rule (string_t name, 
			  long_t first_instr, 
			  rule_type_t type, 
			  long_t num_params)
/* Enter rule <name> of <type> that starts at <first_instr> into the
 * symbol table. The rule number will be returned. */
{
  symbol_node_t *node;

  node = find_symbol_node (name, TRUE);

  if (node->rule_ptr != NULL)
  {
    if (node->rule_ptr->first_instr != -1)
      error ("rule \"%s\" defined twice", name);
    else if (type == SUBRULE)
    {
      if (node->rule_ptr->type != SUBRULE)
	error ("rule \"%s\" already used in a rule set", name);

      if (node->rule_ptr->num_params != num_params)
	error ("\"%s\" has been called with %ld parameters", 
	       name, node->rule_ptr->num_params);
    }
    else if (type == COMBI_RULE || type == END_RULE)
    {
      if (node->rule_ptr->type != COMBI_RULE)
	error ("rule \"%s\" already used as a subrule", name);
    }
    else
      error ("internal (undefined rule)");
  }
  else
  {
    if (node->rule_index != -1)
      error ("\"%s\" is a standard function");

    create_rule_node (node);
  }

  node->rule_ptr->first_instr = first_instr;
  node->rule_ptr->type = type;
  node->rule_ptr->num_params = num_params;
  return node->rule_index;
}

/*---------------------------------------------------------------------------*/

GLOBAL rule_t *find_rule_or_function (string_t name, long_t *rule_number)
/* Find rule <name> and return its rule descriptor.
 * If <rule_number> != NULL, save its rule index in <*rule_number>.
 * If the rule descriptor is NULL, the name describes a standard function. */
{
  symbol_node_t *node;

  node = find_symbol_node (name, TRUE);

  if (node->rule_ptr == NULL && node->rule_index == -1)
    create_rule_node (node);

  if (rule_number != NULL)
    *rule_number = node->rule_index;

  return node->rule_ptr;
}

/*---------------------------------------------------------------------------*/

LOCAL void check_rule_defined (symbol_node_t *node)
/* If <node> is (also) a rule, check if it is defined. */
{
  if (node->rule_ptr != NULL && node->rule_ptr->first_instr == -1)
    error ("rule \"%s\" not defined", node->name);
}

/*---------------------------------------------------------------------------*/

GLOBAL void check_rules_defined (void)
/* Check if all rules in the symbol tree are defined. */
{
  traverse_tree (symbol_table, check_rule_defined);
}

/* functions that use the symbol table for variables ========================*/

GLOBAL long_t find_variable (string_t name)
/* Find variable <name> in the symbol table and return its stack index. */
{
  symbol_node_t *node = find_symbol_node (name, FALSE);
  comp_scope_t *scope;

  if (node != NULL) 
  {
    scope = node->current_scope;
    if (scope != NULL && scope->last_instr == -1)
      return scope->stack_index;
  }
  
  error ("variable \"$%s\" is not defined", name);
  return -1;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t define_variable (string_t name, long_t stack_index)
/* Define the value on index <stack_index> to be a new variable
 * and return a copy of its name <name> in the string pool. */
{
  symbol_node_t *node;
  comp_scope_t *scope;

  node = find_symbol_node (name, TRUE);
  DB_ASSERT (node != NULL);
  
  if (node->current_scope != NULL && node->current_scope->last_instr == -1)
    error ("variable \"$%s\" defined twice", name);

  /* Allocate a new scope and initialise it. */
  scope = (comp_scope_t *) new_mem (sizeof (comp_scope_t));
  scope->next_scope = NULL;
  scope->first_instr = code.next_instr_index;
  scope->last_instr = -1; /* scope is not yet closed */
  scope->stack_index = stack_index;

  /* Link scope to other scopes of same variable. */
  if (node->current_scope == NULL) /* no scope for this variable yet */
    node->first_scope = scope;
  else
    node->current_scope->next_scope = scope;

  node->current_scope = scope;

  return node->name;
}

/*---------------------------------------------------------------------------*/

GLOBAL void undefine_variable (string_t name)
/* Close the scope of variable <name>. */
{
  symbol_node_t *node;

  node = find_symbol_node (name, FALSE);

  DB_ASSERT (node != NULL);
  DB_ASSERT (node->current_scope != NULL);
  DB_ASSERT (node->current_scope->last_instr == -1);

  node->current_scope->last_instr = code.next_instr_index;
}

/*---------------------------------------------------------------------------*/

LOCAL void dump_variable (symbol_node_t *node)
/* Dump all scopes of <node> as a variable. */
{
  comp_scope_t *scope; /* a scope in the symbol table */

  scope = node->first_scope;

  if (scope != NULL) 
  {
    /* Copy all scopes to the var-scope pool */
    var_t var;             /* a new entry in <code.var_pool> */
    var_scope_t var_scope; /* a new entry in <code.var_scope_pool> */
    
    var.name = node->name_index;
    var.first_scope = pool_items (code.var_scope_pool);
    var.number_of_scopes = 0;
    
    while (scope != NULL) 
    {
      var_scope.first_instr = scope->first_instr;
      var_scope.last_instr = scope->last_instr;
      var_scope.stack_index = scope->stack_index;
      
      copy_to_pool (code.var_scope_pool, (void *) &var_scope, 1, NULL);
      var.number_of_scopes++;
      scope = scope->next_scope;
    }
    copy_to_pool (code.var_pool, (void *) &var, 1, NULL);
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void dump_variables (void)
/* Dump all variables and variables scopes in the table. */
{
  /* Emit all variables in alphabetical order. */
  traverse_tree (symbol_table, dump_variable);
}

/* functions that use the symbol table for variables ========================*/

GLOBAL long_t find_constant (string_t name)
/* Find constant <name> in the symbol table and return its value pool index. */
{
  symbol_node_t *node = find_symbol_node (name, FALSE);

  if (node == NULL || node->const_index == -1)
    error ("constant \"@%s\" is not defined", name);
	
  return node->const_index;
}

/*---------------------------------------------------------------------------*/

GLOBAL void define_constant (string_t name, long_t const_index)
/* Define constant <name> with value at <const_index> in the value pool. */
{
  symbol_node_t *node = find_symbol_node (name, TRUE);

  if (node->const_index != -1)
    error ("constant \"@%s\" is already defined", name);

  node->const_index = const_index;
}

/*---------------------------------------------------------------------------*/

LOCAL void free_symbol_tree (symbol_node_t *node)
/* Free all memory used by <node> (including the node itself). */
{
  comp_scope_t *scope, *next_scope;

  /* We cannot use "traverse_tree" here because we must traverse
   * <node>->left_son and <node>->right_son before we delete <node>. */
  if (node != NULL) 
  {
    free_symbol_tree (node->left_son);
    free_symbol_tree (node->right_son);
    
    /* Free the linked list of scopes. */
    scope = node->first_scope;
    while (scope != NULL) 
    {
      next_scope = scope->next_scope;
      free (scope);
      scope = next_scope;
    }
    
    free (node);
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void free_symbols (void)
/* Free all memory used by the symbol table. */
{
  free_symbol_tree (symbol_table);
  symbol_table = NULL;
}
