# File "allomorph.tcl":
# Main module to display allomorphs produced by mallex.

# This file is part of Malaga, a system for Left Associative Grammars.
# Copyright (C) 1995-1998 Bjoern Beutel
#
# Bjoern Beutel
# Universitaet Erlangen-Nuernberg
# Abteilung fuer Computerlinguistik
# Bismarckstrasse 12
# D-91054 Erlangen
# e-mail: malaga@linguistik.uni-erlangen.de
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Global variables:
# $allomorph_numbers -- the list of allomorph numbers
# $allomorph_cat($number) -- the categorie of allomorph $number

#------------------------------------------------------------------------------

# Display allomorphs.

proc allomorph_display {} {

  global font font_size char_set allomorph_numbers allomorph_cat allomorph_surf
  
  .allomorph.frame.canvas delete matrix_tag
  
  set canvas_border 10
  set char_height $font_size(allomorph)
  set current_y $canvas_border
  set start_x $canvas_border
  set max_x $start_x
  
  # Run through the allomorph list:
  set first_allomorph 1
  foreach number $allomorph_numbers {
    
    # Space between allomorphs
    if {! $first_allomorph} {
      incr current_y $char_height
    }

    # Display allomorph number
    set number_id [.allomorph.frame.canvas create matrix \
		   $start_x $current_y \
		   "! \"$number: \"" \
		   -font $font(allomorph) \
		   -char_set $char_set \
		   -tags matrix_tag]
    
    # Display allomorph surface
    set matrix_id [.allomorph.frame.canvas create matrix \
		   [widget_right .allomorph.frame.canvas $number_id ] \
		   $current_y \
		   "\"$allomorph_surf($number)\"" \
		   -font $font(allomorph) \
		   -char_set $char_set \
		   -tags matrix_tag]
    set max_x [maximum $max_x \
	       [widget_right .allomorph.frame.canvas $matrix_id]]
    set current_y [widget_bottom .allomorph.frame.canvas $matrix_id]
    
    # Display allomorph category
    set matrix_id [.allomorph.frame.canvas create matrix \
		   [widget_right .allomorph.frame.canvas $number_id ] \
		   $current_y \
		   $allomorph_cat($number) \
		   -font $font(allomorph) \
		   -char_set $char_set \
		   -tags matrix_tag]
    set max_x [maximum $max_x \
	       [widget_right .allomorph.frame.canvas $matrix_id]]
    set current_y [widget_bottom .allomorph.frame.canvas $matrix_id]
    
    # Center allomorph number vertically
    .allomorph.frame.canvas move $number_id \
    0 [expr ($current_y \
	     - [widget_bottom .allomorph.frame.canvas $number_id])/2]
    
    set first_allomorph 0
  }
  
  # Reconfigure scroll bars:
  .allomorph.frame.canvas configure \
  -scrollregion \
  "0 0 [expr $max_x + $canvas_border] [expr $current_y + $canvas_border]"
}

#------------------------------------------------------------------------------

# Read the malaga allomorphs.

proc allomorph_read {} {

  global allomorph_numbers allomorph_cat allomorph_surf
  
  set allomorph_numbers {}
  
  # Read all allomorphs
  while {[set line [read_line stdin]] != "end" } {
    set allomorph_number [lindex $line 0]
    lappend allomorph_numbers $allomorph_number
    set allomorph_surf($allomorph_number) [lindex $line 1]
    set allomorph_cat($allomorph_number) [lindex $line 2]
  }
}

#------------------------------------------------------------------------------

# Create window where to display the allomorphs

proc allomorph_open_window {} {

  global allomorph_geometry

  if [winfo exists .allomorph] {return}

  if {! [info exists allomorph_geometry]} {set allomorph_geometry 600x400}
  
  # Define start window.
  toplevel .allomorph
  wm title .allomorph "Malaga Allomorphs"
  wm iconname .allomorph "Allomorphs"
  wm minsize .allomorph 100 100
  wm geometry .allomorph $allomorph_geometry
  wm focusmodel .allomorph active

  # Define the different menus.
  frame .allomorph.menu -relief raised -borderwidth 1
  pack .allomorph.menu -side top -fill x
  create_window_menu allomorph
  create_font_menu allomorph
  tk_menuBar .allomorph.menu .allomorph.menu.window .allomorph.menu.fonts
  
  create_scroll_frame allomorph
  focus .allomorph.frame.canvas
}

#------------------------------------------------------------------------------

# Close the allomorph window

proc allomorph_close_window {} {

  global allomorph_geometry
  
  if [winfo exists .allomorph] {
    set allomorph_geometry [winfo geometry .allomorph]
    destroy .allomorph
  }
}

#------------------------------------------------------------------------------

proc allomorph_redraw {mode} {

  allomorph_open_window

  if {$mode == "reload"} {allomorph_read}
  allomorph_display
  if {[wm state .allomorph] == "normal"} {raise .allomorph}
}

#------------------------------------------------------------------------------

# Initialize the variables of "allomorph".

proc allomorph_init {} {

  global font font_size

  set font_size(allomorph) 14
  set font(allomorph) $font($font_size(allomorph))
}

#------------------------------------------------------------------------------
