/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysaudio.c,v 1.19 1999/07/26 20:07:52 slouken Exp $";
#endif

/* Allow access to a raw mixing buffer */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#ifdef linux
#include <linux/soundcard.h>
#endif
#ifdef __FreeBSD__
#include <machine/soundcard.h>
#endif
#ifdef ESD_SUPPORT
#include <esd.h>
#endif

#include "SDL_audio.h"
#include "SDL_error.h"
#include "SDL_audiomem.h"
#include "SDL_audio_c.h"
#include "SDL_timer.h"

#ifndef _PATH_DEV_DSP
#define _PATH_DEV_DSP	"/dev/dsp"
#endif

/* Open the audio device for playback, and don't block if busy */
#define OPEN_FLAGS	(O_WRONLY|O_NONBLOCK)

/* The file descriptor for the audio device */
static int audio_fd = -1;

/* The parent process id, to detect when application quits */
static pid_t parent = 0;

/* Raw mixing buffer */
static Uint8 *mixbuf;
static int    mixlen;

/* Support for audio timing using a timer, instead of select() */
static double frame_ticks;
static double next_frame;
#define FUDGE_TICKS	10	/* The scheduler overhead ticks per frame */

void SDL_SYS_WaitAudio(void)
{
	fd_set fdset;

	/* Check to see if the thread-parent process is still alive */
	{ static int cnt = 0;
		/* Note that this only works with thread implementations 
		   that use a different process id for each thread.
		*/
		if (parent && (((++cnt)%10) == 0)) { /* Check every 10 loops */
			if ( kill(parent, 0) < 0 ) {
				SDL_AudioEnabled = 0;
			}
		}
	}

	/* See if we need to use timed audio synchronization */
	if ( frame_ticks ) {
		/* Use timer for general audio synchronization */
		Sint32 ticks;

		ticks = ((Sint32)(next_frame - SDL_GetTicks()))-FUDGE_TICKS;
		if ( ticks > 0 ) {
			SDL_Delay(ticks);
		}
	}

	/* Use select() for audio synchronization */
	FD_ZERO(&fdset);
	FD_SET(audio_fd, &fdset);
	select(audio_fd+1, NULL, &fdset, NULL, NULL);
}

void SDL_SYS_PlayAudio(void)
{
	int written;

	/* Write the audio data, checking for EAGAIN on broken audio drivers */
	do {
		written = write(audio_fd, mixbuf, mixlen);
		if ( (written < 0) && ((errno == 0) || (errno == EAGAIN)) ) {
			SDL_Delay(1);	/* Let a little CPU time go by */
		}
	} while ( (written < 0) && 
	          ((errno == 0) || (errno == EAGAIN) || (errno == EINTR)) );

	/* If timer synchronization is enabled, set the next write frame */
	if ( frame_ticks ) {
		next_frame += frame_ticks;
	}

	/* If we couldn't write, assume fatal error for now */
	if ( written < 0 ) {
		SDL_AudioEnabled = 0;
	}
}

Uint8 *SDL_SYS_GetAudioBuf(void)
{
	return(mixbuf);
}

void SDL_SYS_CloseAudio(void)
{
	if ( mixbuf != NULL ) {
		SDL_FreeAudioMem(mixbuf);
		mixbuf = NULL;
	}
	if ( audio_fd >= 0 ) {
		close(audio_fd);
		audio_fd = -1;
	}
}

static int SDL_ReopenAudio(char *audiodev, int format, int stereo,
							SDL_AudioSpec *spec)
{
	int frag_spec;
	int value;

	/* Close and then reopen the audio device */
	close(audio_fd);
	audio_fd = open(audiodev, OPEN_FLAGS, 0);
	if ( audio_fd < 0 ) {
		SDL_SetError("Couldn't open %s: %s", audiodev, strerror(errno));
		return(-1);
	}

	/* Calculate the final parameters for this audio specification */
	SDL_CalculateAudioSpec(spec);

	/* Determine the power of two of the fragment size */
	for ( frag_spec = 0; (0x01<<frag_spec) < spec->size; ++frag_spec );
	if ( (0x01<<frag_spec) != spec->size ) {
		SDL_SetError("Fragment size must be a power of two");
		return(-1);
	}
	frag_spec |= 0x00020000;	/* two fragments, for low latency */

	/* Set the audio buffering parameters */
	if ( ioctl(audio_fd, SNDCTL_DSP_SETFRAGMENT, &frag_spec) < 0 ) {
		SDL_SetError("Couldn't set audio fragment spec");
		return(-1);
	}
#ifdef DEBUG_AUDIO
	{ audio_buf_info info;
	  ioctl(audio_fd, SNDCTL_DSP_GETOSPACE, &info);
	  fprintf(stderr, "fragments = %d\n", info.fragments);
	  fprintf(stderr, "fragstotal = %d\n", info.fragstotal);
	  fprintf(stderr, "fragsize = %d\n", info.fragsize);
	  fprintf(stderr, "bytes = %d\n", info.bytes);
	}
#endif

	/* Set the audio format */
	value = format;
	if ( (ioctl(audio_fd, SNDCTL_DSP_SETFMT, &value) < 0) ||
						(value != format) ) {
		SDL_SetError("Couldn't set audio format");
		return(-1);
	}

	/* Set mono or stereo audio */
	value = (spec->channels > 1);
	if ( (ioctl(audio_fd, SNDCTL_DSP_STEREO, &stereo) < 0) ||
						(value != stereo) ) {
		SDL_SetError("Couldn't set audio channels");
		return(-1);
	}

	/* Set the DSP frequency */
	value = spec->freq;
	if ( ioctl(audio_fd, SOUND_PCM_WRITE_RATE, &value) < 0 ) {
		SDL_SetError("Couldn't set audio frequency");
		return(-1);
	}
	spec->freq = value;

	/* We successfully re-opened the audio */
	return(0);
}

/* If we can't open /dev/dsp, try to fall back to ESD audio support */
#ifdef ESD_SUPPORT
int SDL_ESD_OpenAudio(SDL_AudioSpec *spec)
{
	esd_format_t format;

	/* Convert audio spec to the ESD audio format */
	format = (ESD_STREAM | ESD_PLAY);
	switch ( spec->format & 0xFF ) {
		case 8:
			format |= ESD_BITS8;
			break;
		case 16:
			format |= ESD_BITS16;
			break;
		default:
			SDL_SetError("Unsupported ESD audio format");
			return(-1);
	}
	if ( spec->channels == 1 ) {
		format |= ESD_MONO;
	} else {
		format |= ESD_STEREO;
	}

	/* Open a connection to the ESD audio server */
	audio_fd = esd_play_stream(format, spec->freq, NULL, NULL);
	if ( audio_fd < 0 ) {
		SDL_SetError("Couldn't open audio device or ESD connection");
		return(-1);
	}

	/* Calculate the final parameters for this audio specification */
	SDL_CalculateAudioSpec(spec);
	frame_ticks = (1000.0/((double)spec->freq/spec->samples));
	next_frame = SDL_GetTicks()+frame_ticks;

	/* Allocate mixing buffer */
	mixlen = spec->size;
	mixbuf = (Uint8 *)SDL_AllocAudioMem(mixlen);
	if ( mixbuf == NULL ) {
		return(-1);
	}
	memset(mixbuf, spec->silence, spec->size);

	/* We're ready to rock and roll. :-) */
	return(0);
}
#else
int SDL_ESD_OpenAudio(SDL_AudioSpec *spec)
{
	return(-1);
}
#endif

int SDL_SYS_OpenAudio(SDL_AudioSpec *spec)
{
	char *audiodev;
	char audiopath[1024];
	int format;
	int stereo;
	int value;
	Uint16 test_format;

	/* Reset the timer synchronization flag (used by ESD) */
	frame_ticks = 0.0;

	/* Open the audio device */
	if ( (audiodev=getenv("AUDIODEV")) == NULL ) {
		audiodev = _PATH_DEV_DSP;
	}
	audio_fd = open(audiodev, OPEN_FLAGS, 0);

	/* If the first open fails, look for other devices */
	if ( (audio_fd < 0) && (strlen(audiodev) < (sizeof(audiopath)-3)) ) {
		int exists, instance;
		struct stat sb;

		instance = 1;
		do { /* Don't use errno ENOENT - it may not be thread-safe */
			sprintf(audiopath, "%s%d", audiodev, instance++);
			exists = 0;
			if ( stat(audiopath, &sb) == 0 ) {
				exists = 1;
				audio_fd = open(audiopath, OPEN_FLAGS, 0); 
			}
		} while ( exists && (audio_fd < 0) );
		audiodev = audiopath;
	}
	if ( audio_fd < 0 ) {
		SDL_SetError("Couldn't open %s: %s", audiodev, strerror(errno));
		return(SDL_ESD_OpenAudio(spec));
	}
	mixbuf = NULL;

	/* Get a list of supported hardware formats */
	if ( ioctl(audio_fd, SNDCTL_DSP_GETFMTS, &value) < 0 ) {
		SDL_SetError("Couldn't get audio format list");
		return(-1);
	}

	/* Try for a closest match on audio format */
	format = 0;
	for ( test_format = SDL_FirstAudioFormat(spec->format);
						! format && test_format; ) {
#ifdef DEBUG_AUDIO
		fprintf(stderr, "Trying format 0x%4.4x\n", test_format);
#endif
		switch ( test_format ) {
			case AUDIO_U8:
				if ( value & AFMT_U8 ) {
					format = AFMT_U8;
				}
				break;
			case AUDIO_S8:
				if ( value & AFMT_S8 ) {
					format = AFMT_S8;
				}
				break;
			case AUDIO_S16LSB:
				if ( value & AFMT_S16_LE ) {
					format = AFMT_S16_LE;
				}
				break;
			case AUDIO_S16MSB:
				if ( value & AFMT_S16_BE ) {
					format = AFMT_S16_BE;
				}
				break;
			case AUDIO_U16LSB:
				if ( value & AFMT_U16_LE ) {
					format = AFMT_U16_LE;
				}
				break;
			case AUDIO_U16MSB:
				if ( value & AFMT_U16_BE ) {
					format = AFMT_U16_BE;
				}
				break;
			default:
				break;
		}
		if ( ! format ) {
			test_format = SDL_NextAudioFormat();
		}
	}
	if ( format == 0 ) {
		SDL_SetError("Couldn't find any hardware audio formats");
		return(-1);
	}
	spec->format = test_format;

	/* Set the audio format */
	value = format;
	if ( (ioctl(audio_fd, SNDCTL_DSP_SETFMT, &value) < 0) ||
						(value != format) ) {
		SDL_SetError("Couldn't set audio format");
		return(-1);
	}

	/* Set mono or stereo audio (currently only two channels supported) */
	stereo = (spec->channels > 1);
	ioctl(audio_fd, SNDCTL_DSP_STEREO, &stereo);
	if ( stereo ) {
		spec->channels = 2;
	} else {
		spec->channels = 1;
	}

	/* Because some drivers don't allow setting the buffer size
	   after setting the format, we must re-open the audio device
	   once we know what format and channels are supported
	 */
	if ( SDL_ReopenAudio(audiodev, format, stereo, spec) < 0 ) {
		/* Error is set by SDL_ReopenAudio() */
		return(-1);
	}

	/* Allocate mixing buffer */
	mixlen = spec->size;
	mixbuf = (Uint8 *)SDL_AllocAudioMem(mixlen);
	if ( mixbuf == NULL ) {
		return(-1);
	}
	memset(mixbuf, spec->silence, spec->size);

	/* Get the parent process id (we're the parent of the audio thread) */
	parent = getpid();

	/* We're ready to rock and roll. :-) */
	return(0);
}
