/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysvideo.c,v 1.43 1999/08/06 21:10:01 slouken Exp $";
#endif

/* X11 based SDL video driver implementation.
   Note:  This implementation does not currently need X11 thread locking,
          since the event thread uses a separate X connection and any
          additional locking necessary is handled internally.  However,
          if full locking is neccessary, take a look at XInitThreads().
*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#ifdef XFREE86_DGA
#include <X11/extensions/xf86dga.h>
#include <X11/extensions/xf86vmode.h>
#endif
#ifdef MTRR_SUPPORT
#include <asm/mtrr.h>
#include <sys/fcntl.h>
#endif

#include "SDL.h"
#include "SDL_error.h"
#include "SDL_timer.h"
#include "SDL_video.h"
#include "SDL_mouse.h"
#include "SDL_endian.h"
#include "SDL_sysvideo.h"
#include "SDL_video_c.h"
#include "SDL_pixels_c.h"
#include "SDL_sysmouse_c.h"
#include "SDL_syswm_c.h"
#include "SDL_events_c.h"
#include "SDL_lowvideo.h"
#include "blank_cursor.h"


/* Hardware surface functions */
static int SDL_AllocHWSurface(SDL_Surface *surface);
static int SDL_LockHWSurface(SDL_Surface *surface);
static void SDL_UnlockHWSurface(SDL_Surface *surface);
static int SDL_FlipHWSurface(SDL_Surface *surface);
static void SDL_FreeHWSurface(SDL_Surface *surface);

/* Private display data */
Display *SDL_Display = NULL;	/* Used for events and window management */
Display *GFX_Display = NULL;	/* Used for graphics and colormap stuff */
#define SDL_Screen	DefaultScreen(SDL_Display)
#define SDL_Visual	DefaultVisual(SDL_Display, SDL_Screen)
Window   SDL_Window = (Window)0; /* Shared by both displays (no X security?) */
Colormap SDL_DisplayColormap;
Colormap SDL_PrivateColormap;
Atom     SDL_DELWIN;
WMcursor *SDL_BlankCursor = NULL;
static int XF86_DGA = 0;
static int SDL_MTRR_fd = -1;
static char *SDL_windowid;
static XImage  *SDL_Ximage = NULL;
static GC	SDL_GC;
static Colormap SDL_XColorMap;
static int     *SDL_XPixels;
static SDL_Rect **SDL_modelist = NULL;
static int x11_w, x11_h;
static int switchable_modes;
static int swap_pixels;

#define X_HIDDEN_SIZE	32		/* starting hidden window size */

/* Shared memory information */
extern int XShmQueryExtension(Display *dpy);	/* Not in X11 headers */
static int use_mitshm;
static XShmSegmentInfo shminfo;

/* Shared memory error handler routine */
static int shm_error;
static int (*X_handler)(Display *, XErrorEvent *);
static int shm_errhandler(Display *d, XErrorEvent *e)
{
        if ( e->error_code == BadAccess ) {
        	++shm_error;
        	return(0);
        } else
		return(X_handler(d,e));
}

/* X11 I/O error handler routine */
static int (*XIO_handler)(Display *);
static int xio_errhandler(Display *d)
{
	/* Ack!  Lost X11 connection! */

	/* We will crash if we try to clean up our display */
	if ( SDL_Ximage ) {
		SDL_VideoSurface->pixels = NULL;
	}
	SDL_Display = NULL;

	/* Continue with the standard X11 error handler */
	return(XIO_handler(d));
}

/* Used to sort video modes largest to smallest */
static int CompareModes(const void *A, const void *B)
{
	const SDL_Rect *a = *(SDL_Rect **)A;
	const SDL_Rect *b = *(SDL_Rect **)B;

	return( (int)(b->w*b->h) - (int)(a->w*a->h) );
}

static void SDL_UpdateVideoInfo(SDL_VideoInfo *info)
{
#ifdef XFREE86_DGA
	if ( XF86_DGA ) {
		extern int _SDL_lockSO;
		int unused_int;
		char *unused_ptr;
		int total_mem;

		info->hw_available = 1;
		seteuid(0);		/* Try to set root privileges */
		XF86DGAGetVideo(SDL_Display, SDL_Screen,
			&unused_ptr, &unused_int, &unused_int, &total_mem);
		seteuid(getuid());	/* Get rid of root privileges */
		_SDL_lockSO = 1;
		info->video_mem = total_mem;
	}
#endif
}

int SDL_SYS_VideoInit(SDL_PixelFormat *vformat)
{
	char *display;
	int EventBase, ErrorBase;
	int MajorVersion, MinorVersion;
	int flags;
	int local_x11;
	XGCValues gcv;

	/* Initialize all variables that we clean on shutdown */
	SDL_Window = (Window)0;
	SDL_Ximage = NULL;
	SDL_XPixels = NULL;
	SDL_PrivateColormap = 0;
	SDL_DisplayColormap = 0;
	SDL_BlankCursor = NULL;
	SDL_modelist = NULL;

	/* Open the X11 display */
	display = NULL;		/* Get it from DISPLAY environment variable */
	if ( strncmp(XDisplayName(display), ":", 1) == 0 ) {
		local_x11 = 1;
	} else {
		local_x11 = 0;
	}
	SDL_Display = XOpenDisplay(display);
	if ( SDL_Display == NULL ) {
		SDL_SetError("Couldn't open X11 display");
		return(-1);
	}

	/* Create an alternate X display for graphics updates -- allows us
	   to do graphics updates in a separate thread from event handling.
	   Thread-safe X11 doesn't seem to exist.
	 */
	GFX_Display = XOpenDisplay(display);
	if ( GFX_Display == NULL ) {
		SDL_SetError("Couldn't open X11 display");
		return(-1);
	}

	/* Set the error handler if we lose the X display */
	XIO_handler = XSetIOErrorHandler(xio_errhandler);

	/* Register the "delete window" protocol */
	SDL_DELWIN = XInternAtom(SDL_Display, "WM_DELETE_WINDOW", False);

	/* Determine the screen depth */
	{
		XPixmapFormatValues *pix_format;
		int i, num_formats;

		vformat->BitsPerPixel = DefaultDepth(SDL_Display, SDL_Screen);
		pix_format = XListPixmapFormats(SDL_Display, &num_formats);
		if ( pix_format == NULL ) {
			SDL_SetError("Couldn't determine screen formats");
			return(-1);
		}
		for ( i=0; i<num_formats; ++i ) {
			if ( vformat->BitsPerPixel == pix_format[i].depth )
				break;
		}
		if ( i != num_formats )
			vformat->BitsPerPixel = pix_format[i].bits_per_pixel;
		XFree((char *)pix_format);
	}
	if ( vformat->BitsPerPixel > 8 ) {
		vformat->Rmask = SDL_Visual->red_mask;
	  	vformat->Gmask = SDL_Visual->green_mask;
	  	vformat->Bmask = SDL_Visual->blue_mask;
	}

	/* Check for MIT shared memory extension */
	if ( local_x11 ) {
		use_mitshm = XShmQueryExtension(SDL_Display);
	} else {
		use_mitshm = 0;
	}

#ifdef XFREE86_DGA
	/* Check for XFree86 Direct Graphics Access extension
	   Note:  Only root can perform Direct Graphics Access
	 */
	XF86_DGA = 0;
	switchable_modes = 0;
	seteuid(0);		/* Try to set root privileges */
	if ( local_x11 &&
	     ((geteuid() == 0) || (access("/dev/mem", R_OK|W_OK) == 0)) &&
	     XF86DGAQueryExtension(SDL_Display, &EventBase, &ErrorBase) &&
	     XF86DGAQueryVersion(SDL_Display, &MajorVersion, &MinorVersion) &&
	     XF86DGAQueryDirectVideo(SDL_Display, SDL_Screen, &flags) && 
					(flags & XF86DGADirectPresent) ) {
		int buggy_X11;
		int i, nmodes;
		XF86VidModeModeInfo **modes;

		XF86_DGA = 1;
		/* Metro-X 4.3.0 and earlier has a broken implementation of
		   XF86VidModeGetAllModeLines() - it hangs the client.
		 */
		buggy_X11 = 0;
		if ( strcmp(ServerVendor(SDL_Display),
					"Metro Link Incorporated") == 0 ) {
			FILE *metro_fp;

			metro_fp =
				fopen("/usr/X11R6/lib/X11/Metro/.version", "r");
			if ( metro_fp != NULL ) {
				int major, minor, patch, version;
				major = 0; minor = 0; patch = 0;
				fscanf(metro_fp, "%d.%d.%d",
						&major, &minor, &patch);
				version = major*100+minor*10+patch;
				if ( version < 431 ) {
					buggy_X11 = 1;
				}
				fclose(metro_fp);
			}
		}
#if defined(__alpha__) || defined(__powerpc__)
		/* The alpha and PPC XFree86 servers are also buggy */
		buggy_X11 = 1;
#endif
		/* Enumerate the available fullscreen modes */
		if ( ! buggy_X11 &&
		     XF86VidModeQueryExtension(SDL_Display,
					&EventBase, &ErrorBase) &&
		     XF86VidModeQueryVersion(SDL_Display,
					&MajorVersion, &MinorVersion) &&
		     XF86VidModeGetAllModeLines(SDL_Display, SDL_Screen,
							&nmodes, &modes) ) {
			SDL_modelist = (SDL_Rect **)
					malloc((nmodes+1)*sizeof(SDL_Rect *));
			if ( SDL_modelist ) {
				for ( i=0; i<nmodes; ++i ) {
					SDL_modelist[i] = (SDL_Rect *)
						malloc(sizeof(SDL_Rect));
					if ( SDL_modelist[i] == NULL ) {
						SDL_OutOfMemory();
						seteuid(getuid());
						return(-1);
					}
					SDL_modelist[i]->x = 0;
					SDL_modelist[i]->y = 0;
					SDL_modelist[i]->w = modes[i]->hdisplay;
					SDL_modelist[i]->h = modes[i]->vdisplay;
				}
				SDL_modelist[i] = NULL;

				/* Sort modes largest to smallest */
				qsort(SDL_modelist, nmodes, sizeof(SDL_Rect *),
								CompareModes);
			}
			switchable_modes = 1;
		} else {
			SDL_modelist = (SDL_Rect **)
					malloc((1+1)*sizeof(SDL_Rect *));
			if ( SDL_modelist ) {
				SDL_modelist[0] = (SDL_Rect *)
						malloc(sizeof(SDL_Rect));
				if ( SDL_modelist[0] == NULL ) {
					SDL_OutOfMemory();
					seteuid(getuid());
					return(-1);
				}
				SDL_modelist[0]->x = 0;
				SDL_modelist[0]->y = 0;
				SDL_modelist[0]->w = WidthOfScreen(DefaultScreenOfDisplay(SDL_Display));
				SDL_modelist[0]->h = HeightOfScreen(DefaultScreenOfDisplay(SDL_Display));
				SDL_modelist[1] = NULL;
			}
		}
#ifdef DEBUG_XF86
		if ( SDL_modelist ) {
			int i;

			fprintf(stderr, "XF86 mode list:\n");
			for ( i=0; SDL_modelist[i]; ++i ) {
				fprintf(stderr, "\t%dx%d\n",
					SDL_modelist[i]->w, SDL_modelist[i]->h);
			}
		}
#endif
#ifdef MTRR_SUPPORT
		/* Try to set up "Memory Type Range Register" access */
		SDL_MTRR_fd = open("/proc/mtrr", O_WRONLY, 0);
		if ( SDL_MTRR_fd >= 0 ) {
			struct mtrr_sentry entry;
			int entry_base;
			int unused;
			int entry_size;

			/* Get the base and length of the video framebuffer */
			XF86DGAGetVideoLL(SDL_Display, SDL_Screen,
				&entry_base, &unused, &unused, &entry_size);

			/* Set the entry values */
			entry.base = (unsigned long)entry_base;
			entry.size = (unsigned long)entry_size;
			entry.size *= 1024;
			entry.type = MTRR_TYPE_WRCOMB;

			/* Magic code to satisfy kernel requirements */
			if ( entry.size < 0x100000 ) {
				entry.size = 0x100000;
			} else
			if ( entry.size % 0x400000 ) {
				entry.size += (0x400000-1);
				entry.size &= ~(0x400000-1);
			}

			/* Add the framebuffer region to the control list */
			ioctl(SDL_MTRR_fd, MTRRIOC_ADD_ENTRY, &entry);
		}
#endif
	}
#ifdef VERBOSE_WARNINGS
	/* Warn of buffer overflows if this program is set-uid */
	if ( geteuid() != getuid() ) {
		fprintf(stderr,
		"Warning: This set-uid program may be a security risk\n");
	}
#endif
	seteuid(getuid());	/* Temporarily get rid of root privileges */

#ifdef DEBUG_XF86
	fprintf(stderr, "XFree86 DGA is %s\n", XF86_DGA?"enabled":"disabled");
#endif
#endif /* XFREE86_DGA */

	/* Create (or use) the X11 window */
	SDL_windowid = getenv("SDL_WINDOWID");
	if ( SDL_windowid ) {
		SDL_Window = atol(SDL_windowid);
	} else {
		SDL_Window = XCreateSimpleWindow(SDL_Display,
				DefaultRootWindow(SDL_Display), 0, 0,
				X_HIDDEN_SIZE, X_HIDDEN_SIZE, 1,
				BlackPixel(SDL_Display, SDL_Screen),
				BlackPixel(SDL_Display, SDL_Screen));
	}

	/* Set the class hints so we can get an icon (AfterStep) */
	{
		XClassHint *classhints;

		classhints = XAllocClassHint();
		if ( classhints != NULL ) {
			classhints->res_name = "SDL_App";
			classhints->res_class = "SDL_App";
			XSetClassHint(SDL_Display, SDL_Window, classhints);
			XFree(classhints);
		}
	}

	/* Allow the window to be deleted by the window manager */
	XSetWMProtocols(SDL_Display, SDL_Window, &SDL_DELWIN, 1);

	/* Cache the window in the server, when possible */
	{
		unsigned long mask;
		Screen *xscreen;
		XSetWindowAttributes a;

		xscreen = DefaultScreenOfDisplay(SDL_Display);
		if ( DoesBackingStore(xscreen) != NotUseful ) {
			mask = CWBackingStore;
			a.backing_store = DoesBackingStore(xscreen);
		} else {
			mask = CWSaveUnder;
			a.save_under = DoesSaveUnders(xscreen);
		}
		XChangeWindowAttributes(SDL_Display, SDL_Window, mask, &a);
	}

	/* Create the graphics context */
	gcv.graphics_exposures = False;
	SDL_GC = XCreateGC(SDL_Display, SDL_Window, GCGraphicsExposures, &gcv);
	if ( ! SDL_GC ) {
		SDL_SetError("Couldn't create graphics context");
		return(-1);
	}

	/* If the surface is palettized, create the colormap */
	if ( vformat->BitsPerPixel == 8 ) {
		int     ncolors;

		/* Get the default display colormap (used by icons) */
		SDL_DisplayColormap = DefaultColormap(SDL_Display, SDL_Screen);
		SDL_XColorMap = SDL_DisplayColormap;

		/* Allocate the pixel flags */
		ncolors = (1<<vformat->BitsPerPixel);
		SDL_XPixels = (int *)malloc(ncolors*sizeof(int));
		if ( SDL_XPixels == NULL ) {
			SDL_OutOfMemory();
			return(-1);
		}
		memset(SDL_XPixels, 0, ncolors*sizeof(*SDL_XPixels));
	}

	/* Create the blank cursor */
	SDL_BlankCursor = SDL_CreateWMCursor(NULL, blank_cdata, blank_cmask,
					BLANK_CWIDTH, BLANK_CHEIGHT,
						BLANK_CHOTX, BLANK_CHOTY);

	/* Set the events that we look for */
	if ( SDL_windowid == NULL ) {
		XSelectInput(SDL_Display, SDL_Window,
				( FocusChangeMask | PropertyChangeMask
				| EnterWindowMask | LeaveWindowMask
				| KeyPressMask | KeyReleaseMask
				| ButtonPressMask | ButtonReleaseMask
				| PointerMotionMask
				| ExposureMask | StructureNotifyMask ) );
	}

	/* See whether or not we need to swap pixels */
	swap_pixels = 0;
	if ( SDL_BYTEORDER == SDL_LIL_ENDIAN ) {
		if ( XImageByteOrder(SDL_Display) == MSBFirst ) {
			swap_pixels = 1;
		}
	} else {
		if ( XImageByteOrder(SDL_Display) == LSBFirst ) {
			swap_pixels = 1;
		}
	}

	/* Fill in some window manager capabilities */
	SDL_HWCaps.info.wm_available = 1;
	SDL_WMCaps.SetCaption = SDL_SYS_SetWMCaption;
	SDL_WMCaps.SetIcon = SDL_SYS_SetWMIcon;
	SDL_WMCaps.GetWMInfo = SDL_SYS_GetWMInfo;

	/* Fill in our hardware acceleration capabilities */
	SDL_UpdateVideoInfo(&SDL_HWCaps.info);
	SDL_HWCaps.AllocHWSurface = SDL_AllocHWSurface;
	SDL_HWCaps.LockHWSurface = SDL_LockHWSurface;
	SDL_HWCaps.UnlockHWSurface = SDL_UnlockHWSurface;
	SDL_HWCaps.FlipHWSurface = SDL_FlipHWSurface;
	SDL_HWCaps.FreeHWSurface = SDL_FreeHWSurface;

	/* We're done! */
	XSync(SDL_Display, False);
	return(0);
}

SDL_Rect **SDL_SYS_ListModes(SDL_Surface *screen,
				SDL_PixelFormat *format, Uint32 flags)
{
	if ( format->BitsPerPixel == screen->format->BitsPerPixel ) {
		if ( (flags & SDL_FULLSCREEN) && XF86_DGA ) {
			return(SDL_modelist);
		} else {
			return((SDL_Rect **)-1);
		}
	} else {
		return((SDL_Rect **)0);
	}
}

/* Various screen update functions available */
static void SDL_NormalUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);
static void SDL_MITSHMUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);
#ifdef XFREE86_DGA
static void SDL_DirectUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);
static void SDL_BankedUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);
#endif

/* Private structure to keep track of the video card parameters */
struct {
	int active;
	int width;
	int lines;
	int page;
	int w, h;
	int flip;
	char *base[2];
} vinfo = { 0, 0, 0, 0, 0, 0, 0, { NULL, NULL } };

static int SetupWindow(SDL_Surface *current, Uint32 flags)
{
	/* Allocate shared memory if possible */
	if ( use_mitshm ) {
		shminfo.shmid = shmget(IPC_PRIVATE, current->h*current->pitch,
								IPC_CREAT|0777);
		if ( shminfo.shmid >= 0 ) {
			shminfo.shmaddr = (char *)shmat(shminfo.shmid, 0, 0);
			shminfo.readOnly = False;
			if ( shminfo.shmaddr != (char *)-1 ) {
				shm_error = False;
				X_handler = XSetErrorHandler(shm_errhandler);
				XShmAttach(SDL_Display, &shminfo);
				XSync(SDL_Display, True);
				XSetErrorHandler(X_handler);
				if ( shm_error == True )
					shmdt(shminfo.shmaddr);
			} else {
				shm_error = True;
			}
			shmctl(shminfo.shmid, IPC_RMID, NULL);
		} else {
			shm_error = True;
		}
		if ( shm_error == True )
			use_mitshm = 0;
	}
	if ( use_mitshm ) {
		current->pixels = shminfo.shmaddr;
	} else {
		current->pixels = malloc(current->h*current->pitch);
	}
	if ( current->pixels == NULL ) {
		SDL_OutOfMemory();
		return(-1);
	}

	if ( use_mitshm ) {
		SDL_Ximage = XShmCreateImage(SDL_Display, SDL_Visual,
			DefaultDepth(SDL_Display, SDL_Screen),
			ZPixmap, shminfo.shmaddr, &shminfo, 
						current->w, current->h);
	} else {
		SDL_Ximage = XCreateImage(SDL_Display, SDL_Visual,
			DefaultDepth(SDL_Display, SDL_Screen),
			ZPixmap, 0, (char *)current->pixels, 
			current->w, current->h,
			((current->format)->BytesPerPixel == 3) ? 32 :
				(current->format)->BytesPerPixel*8, 0);
	}
	if ( SDL_Ximage == NULL ) {
		SDL_SetError("Couldn't create XImage");
		if ( use_mitshm ) {
			XShmDetach(SDL_Display, &shminfo);
			XSync(SDL_Display, False);
			shmdt(shminfo.shmaddr);
			current->pixels = NULL;
		}
		return(-1);
	}

	/* Determine what blit function to use */
	if ( use_mitshm ) {
		SDL_SYS_UpdateRects = SDL_MITSHMUpdate;
	} else {
		SDL_SYS_UpdateRects = SDL_NormalUpdate;
	}
	return(0);
}

static int SetupDirect(SDL_Surface *current, Uint32 flags)
{
#ifdef XFREE86_DGA
	extern void _SDL_Quit(void);
	extern int _SDL_lockSO;

	int banked;
	int vram;

	/* Set the requested video mode */
	if ( switchable_modes ) {
		int i, nmodes;
		XF86VidModeModeInfo **modes;

		/* Save the original video mode settings */
		if ( ! x11_w || ! x11_h ) {
			XF86DGAGetViewPortSize(SDL_Display, SDL_Screen,
							&x11_w, &x11_h);
		}

		/* Get the mode we need to set, and set it */
		XF86VidModeGetAllModeLines(SDL_Display, SDL_Screen,
						&nmodes, &modes);
		for ( i=0; i<nmodes; ++i ) {
			if ( (current->w == modes[i]->hdisplay) && 
			     (current->h == modes[i]->vdisplay) ) {
				break;
			}
		}
		if ( i == nmodes ) {
			/* This shouldn't happen */
			SDL_SetError("Requested video mode not available");
			return(-1);
		}
		XF86VidModeSwitchToMode(SDL_Display, SDL_Screen, modes[i]);
	}

	/* Get the parameters for the video card */
	seteuid(0);		/* Try to set root privileges */
	XF86DGAGetVideo(SDL_Display, SDL_Screen,
			&vinfo.base[0], &vinfo.width, &vinfo.page, &vram);
	vinfo.width *= current->format->BytesPerPixel;
        vinfo.width = (vinfo.width + 3) & ~3;       /* 4-byte aligning */
	vinfo.lines = (vinfo.page/vinfo.width);
	vinfo.w = current->w;
	vinfo.h = current->h;
	vinfo.flip = 0;
	vinfo.base[1] = vinfo.base[0]+vinfo.width*vinfo.h;
	banked = (vinfo.lines < current->h);
#ifdef DEBUG_XF86
	fprintf(stderr,
"Video: ram = %dK, pagesize = %d, w = %d, h = %d, lines = %d, pitch = %d\n",
		vram, vinfo.page, vinfo.w, vinfo.h, vinfo.lines, vinfo.width);
#endif

	/* Arggghh.  The XFree86 3.3.2 implementation of XF86DGAGetVideo()
	   sets atexit() and signal handler functions to the DGA cleanup 
	   function which calls _exit() internally.  This blows away any 
	   atexit() call we might have made previously, and causes a 
	   segmentation fault on exit if we, as a shared library, are 
	   dynamically unloaded.  Hack, hack, hack.
	*/
	atexit(SDL_Quit);
	_SDL_lockSO = 1;

	/* Set direct graphics access */
	XF86DGASetViewPort(SDL_Display, SDL_Screen, 0, 0);
	XF86DGADirectVideo(SDL_Display, SDL_Screen,
		XF86DGADirectGraphics|XF86DGADirectMouse|XF86DGADirectKeyb);
	XF86DGASetVidPage(SDL_Display, SDL_Screen, 0);
	seteuid(getuid());	/* Temporarily get rid of root privileges */

	/* Make the input as raw as possible */
	XGrabKeyboard(SDL_Display, SDL_Window, True, GrabModeAsync,
						GrabModeAsync, CurrentTime);
	XGrabPointer(SDL_Display, SDL_Window, True, 
			(ButtonPressMask|ButtonReleaseMask|PointerMotionMask),
			GrabModeAsync, GrabModeAsync, None, None, CurrentTime);

	/* Fill in the current video surface */
	if ( banked ) {
		current->flags |= SDL_SWSURFACE;
		current->pixels = malloc(current->h*current->pitch);
	} else {
		current->flags |= SDL_HWSURFACE;
		current->pitch = vinfo.width;
		if ( ((flags & SDL_DOUBLEBUF) == SDL_DOUBLEBUF) &&
					(vinfo.lines >= vinfo.h*2) ) {
			current->flags |= SDL_DOUBLEBUF;
			current->pixels = vinfo.base[!vinfo.flip];
		} else {
			current->pixels = vinfo.base[0];
		}
	}
	if ( current->pixels == NULL ) {
		SDL_OutOfMemory();
		return(-1);
	}
	current->flags |= SDL_FULLSCREEN;

	/* Determine what blit function to use */
	if ( banked ) {
		SDL_SYS_UpdateRects = SDL_BankedUpdate;
	} else {
		SDL_SYS_UpdateRects = SDL_DirectUpdate;
	}
	vinfo.active = 1;
	return(0);
#else
	SDL_SetError("Logic error: Fullscreen requested without DGA");
	return(-1);
#endif /* XFREE86_DGA */
}

static void ReleaseMode(SDL_Surface *screen)
{
	if ( SDL_Ximage ) {
		XDestroyImage(SDL_Ximage);
		if ( use_mitshm ) {
			XShmDetach(SDL_Display, &shminfo);
			XSync(SDL_Display, False);
			shmdt(shminfo.shmaddr);
		}
		screen->pixels = NULL;
		SDL_Ximage = NULL;
	}
#ifdef XFREE86_DGA
	if ( vinfo.active ) {
		/* Release the direct input and graphics */
		XUngrabPointer(SDL_Display, CurrentTime);
		XUngrabKeyboard(SDL_Display, CurrentTime);
		XF86DGADirectVideo(SDL_Display, SDL_Screen, 0);

		/* Set the original video mode */
		if ( switchable_modes ) {
			XF86VidModeModeInfo **modes;
			int i, nmodes;

			XF86VidModeGetAllModeLines(SDL_Display, SDL_Screen,
							&nmodes, &modes);
			for ( i=0; i<nmodes; ++i ) {
				if ( (x11_w == modes[i]->hdisplay) && 
				     (x11_h == modes[i]->vdisplay) ) {
					break;
				}
			}
			if ( i != nmodes ) {
				XF86VidModeSwitchToMode(SDL_Display,
						SDL_Screen, modes[i]);
			}
		}
		if ( (screen->flags & SDL_HWSURFACE) == SDL_SWSURFACE ) {
			free(screen->pixels);
		}
		vinfo.active = 0;
	}
#endif
}

static int SDL_ShowWindow(int width, int height, Uint32 flags)
{
	XSizeHints *sizehints;
	XWindowAttributes xwinattr;

	/* Resize the window */
	if ( flags & SDL_FULLSCREEN ) {
		XSetWindowAttributes xsetattr;
		Window u1; int u2; unsigned int u3;

		/* Prevent window manager from grabbing us on placement */
		XSetTransientForHint(SDL_Display, SDL_Window, SDL_Window);
		xsetattr.override_redirect = True;
		XChangeWindowAttributes(SDL_Display, SDL_Window,
					CWOverrideRedirect, &xsetattr);

		/* Make our window cover the entire screen */
		XGetGeometry(SDL_Display, DefaultRootWindow(SDL_Display),
				&u1, &u2, &u2,
				(unsigned int *)&width,
				(unsigned int *)&height,
				&u3, &u3);
	} else {
		XSetWindowAttributes xsetattr;

		/* If we were fullscreen, unmap the window */
		XGetWindowAttributes(SDL_Display, SDL_Window, &xwinattr);
		if ( xwinattr.override_redirect == True ) {
			XUnmapWindow(SDL_Display, SDL_Window);
		}

		/* Allow window manager to grab us and place the window */
		XDeleteProperty(SDL_Display, SDL_Window, XA_WM_TRANSIENT_FOR);
		xsetattr.override_redirect = False;
		XChangeWindowAttributes(SDL_Display, SDL_Window,
					CWOverrideRedirect, &xsetattr);
	}
	sizehints = XAllocSizeHints();
	if ( sizehints == NULL ) {
		SDL_SetError("Couldn't allocate size hints");
		return(-1);
	}
	sizehints->min_width   = width;
	sizehints->min_height  = height;
	sizehints->max_width   = width;
	sizehints->max_height  = height;
	sizehints->flags       = (PMinSize | PMaxSize);
	XSetWMNormalHints(SDL_Display, SDL_Window, sizehints);
	XFree(sizehints);
	XResizeWindow(SDL_Display, SDL_Window, width, height);

	/* Map the window onto the display, if needed */
	XGetWindowAttributes(SDL_Display, SDL_Window, &xwinattr);
	if ( xwinattr.map_state == IsUnmapped ) {
		XEvent event;

		XMapRaised(SDL_Display, SDL_Window);
		do {
			XNextEvent(SDL_Display, &event);
		} while ( event.type != MapNotify );
	}
	return(0);
}

SDL_Surface *SDL_SYS_SetVideoMode(SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags)
{
	/* Lock the event thread, in multi-threading environments */
	SDL_Lock_EventThread();

	/* Check the combination of flags we were passed */
	if ( flags & SDL_FULLSCREEN ) {
		/* Clear fullscreen flag if not supported */
		if ( ! XF86_DGA || (SDL_windowid != NULL) ) {
			flags &= ~SDL_FULLSCREEN;
		}
#if defined(XFREE86_DGA) && defined(VERBOSE_WARNINGS)
		if ( geteuid() != 0 ) {
			fprintf(stderr,
				"You must be root to set a fullscreen mode\n");
		}
#endif
	}
	if ( (flags&SDL_FULLSCREEN) != SDL_FULLSCREEN ) {
		/* There's no windowed double-buffering */
		flags &= ~SDL_DOUBLEBUF;
	}
	if ( (flags&SDL_DOUBLEBUF) == SDL_DOUBLEBUF ) {
		/* Use hardware surfaces when double-buffering */
		flags |= SDL_HWSURFACE;
	}

	/* Release any previously set mode */
	ReleaseMode(current);

	/* Show the window, if we own it */
	if ( SDL_windowid == NULL ) {
		if ( SDL_ShowWindow(width, height, flags) < 0 ) {
			current = NULL;
			goto finished;
		}
	}

	/* Set up the new mode framebuffer */
	current->flags = 0;		/* Clear flags */
	current->w = width;
	current->h = height;
	current->pitch = SDL_CalculatePitch(current);
	if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		if ( SetupDirect(current, flags) == 0 ) {
			flags |= SDL_HWPALETTE;
		} else {
			/* Failed, fall back to window */
			flags &= ~SDL_FULLSCREEN;
		}
	}
	if ( (flags & SDL_FULLSCREEN) != SDL_FULLSCREEN ) {
		if ( SetupWindow(current, flags) < 0 ) {
			current = NULL;
			goto finished;
		}
	}

	/* Set the appropriate colormap */
	if ( current->format->palette != NULL ) {
		if ( flags & SDL_HWPALETTE ) {
			current->flags |= SDL_HWPALETTE;
			if ( SDL_PrivateColormap == 0 ) {
				SDL_PrivateColormap = XCreateColormap(SDL_Display, SDL_Window, SDL_Visual, AllocAll);
			}
			SDL_XColorMap = SDL_PrivateColormap;
		} else {
			/* Free any currently allocated private colormap */
			if ( SDL_PrivateColormap != 0 ) {
				XFreeColormap(SDL_Display, SDL_PrivateColormap);
				SDL_PrivateColormap = 0;
			}
			SDL_XColorMap = SDL_DisplayColormap;
		}
		XSetWindowColormap(SDL_Display, SDL_Window, SDL_XColorMap);
	}

	/* Make sure that our window has input focus if fullscreen */
	if ( (flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
		XSetInputFocus(SDL_Display, SDL_Window,
					RevertToPointerRoot, CurrentTime);
	}
	XSync(SDL_Display, False);

	/* We're done */
finished:
	SDL_Unlock_EventThread();
	return(current);
}

/* We don't actually allow hardware surfaces other than the main one */
static int SDL_AllocHWSurface(SDL_Surface *surface)
{
	return(-1);
}
static void SDL_FreeHWSurface(SDL_Surface *surface)
{
	return;
}

/* We need to wait for vertical retrace on page flipped displays */
static int SDL_LockHWSurface(SDL_Surface *surface)
{
#ifdef XFREE86_DGA
	if ( (surface->flags & SDL_DOUBLEBUF) == SDL_DOUBLEBUF ) {
		while ( ! XF86DGAViewPortChanged(GFX_Display,
				DefaultScreen(GFX_Display), 2) ) {
			SDL_Delay(1);
		}
	}
#endif
	return(0);
}
static void SDL_UnlockHWSurface(SDL_Surface *surface)
{
	return;
}

/* We can flip DGA memory! */
static int SDL_FlipHWSurface(SDL_Surface *surface)
{
#ifdef XFREE86_DGA
	/* Choose your page (double-buffered for now) */
	vinfo.flip = !vinfo.flip;

	/* Set it up */
	XF86DGASetViewPort(GFX_Display, DefaultScreen(GFX_Display),
					0, vinfo.flip*vinfo.h);
	surface->pixels = vinfo.base[!vinfo.flip];
#endif
	return(0);
}

#ifdef DGA_BROKEN_VIEWPORT
#define FIXED_VIEWPORT	0x80
void SDL_FixViewPort(void)
{
	if ( vinfo.active && !(vinfo.active&FIXED_VIEWPORT) ) {
		XF86DGASetViewPort(GFX_Display, DefaultScreen(GFX_Display),
						0, vinfo.flip*vinfo.h);
		vinfo.active |= FIXED_VIEWPORT;
	}
}
#endif /* DGA_BROKEN_VIEWPORT */

/* Byte-swap the pixels in the display image */
static void SDL_SwapPixels(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	int i;
	int x, minx, maxx;
	int y, miny, maxy;

	switch (screen->format->BytesPerPixel) {
	    case 2: {
		Uint16 *spot;
		for ( i=0; i<numrects; ++i ) {
			minx = rects[i].x;
			maxx = rects[i].x+rects[i].w;
			miny = rects[i].y;
			maxy = rects[i].y+rects[i].h;
			for ( y=miny; y<maxy; ++y ) {
				spot = (Uint16 *) ((Uint8 *)screen->pixels +
						y * screen->pitch + minx * 2);
				for ( x=minx; x<maxx; ++x, ++spot ) {
					*spot = SDL_Swap16(*spot);
				}
			}
		}
	    }
	    break;

	    case 4: {
		Uint32 *spot;
		for ( i=0; i<numrects; ++i ) {
			minx = rects[i].x;
			maxx = rects[i].x+rects[i].w;
			miny = rects[i].y;
			maxy = rects[i].y+rects[i].h;
			for ( y=miny; y<maxy; ++y ) {
				spot = (Uint32 *) ((Uint8 *)screen->pixels +
						y * screen->pitch + minx * 4);
				for ( x=minx; x<maxx; ++x, ++spot ) {
					*spot = SDL_Swap32(*spot);
				}
			}
		}
	    }
	    break;

	    default:
		/* should never get here */
		break;
	}
}

static void SDL_NormalUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	int i;

	/* Check for endian-swapped X server, swap if necessary (VERY slow!) */
	if ( swap_pixels && ((screen->format->BytesPerPixel%2) == 0) ) {
		SDL_SwapPixels(screen, numrects, rects);
		for ( i=0; i<numrects; ++i ) {
			XPutImage(GFX_Display, SDL_Window, SDL_GC, SDL_Ximage,
				rects[i].x, rects[i].y,
				rects[i].x, rects[i].y, rects[i].w, rects[i].h);
		}
		SDL_SwapPixels(screen, numrects, rects);
	} else {
		for ( i=0; i<numrects; ++i ) {
			XPutImage(GFX_Display, SDL_Window, SDL_GC, SDL_Ximage,
				rects[i].x, rects[i].y,
				rects[i].x, rects[i].y, rects[i].w, rects[i].h);
		}
	}
	XSync(GFX_Display, False);
}

static void SDL_MITSHMUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
	int i;

	for ( i=0; i<numrects; ++i ) {
		XShmPutImage(GFX_Display, SDL_Window, SDL_GC, SDL_Ximage,
				rects[i].x, rects[i].y,
				rects[i].x, rects[i].y, rects[i].w, rects[i].h,
									False);
	}
	XSync(GFX_Display, False);
}

static void SDL_DirectUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
}

/* This is very slow.  Ideally, you would organize all the updates into
   lists of bank-sorted updates, and perform all updates for each bank
   in turn.  In the general case, I don't think it's worth the effort.
*/
static void SDL_BankedUpdate(SDL_Surface *screen, int numrects, SDL_Rect *rects)
{
#ifdef XFREE86_DGA
	int i, w, h;
	Uint8 *src, *dst;
	int xoffset;
	int bankleft;
	int bank;
	int wl, wr;

	/* Fixed by Alex Jones */
	for ( i=0; i<numrects; ++i ) {
		xoffset = rects[i].x * screen->format->BytesPerPixel;
		src = (Uint8 *)screen->pixels +
			rects[i].y * screen->pitch + xoffset;
		h = rects[i].h;
		w = rects[i].w * screen->format->BytesPerPixel;

		/* Get the starting bank and offset */
		bank = 0;
		dst = vinfo.base[0] + xoffset;
		bankleft = vinfo.page - xoffset;

		bankleft -= vinfo.width * rects[i].y;
		dst += vinfo.width * rects[i].y;
		if ( bankleft <= 0 ) {
			bank = -bankleft / vinfo.page + 1;
			bankleft += bank * vinfo.page;
			dst -= bank * vinfo.page;
		}

 		XF86DGASetVidPage(GFX_Display,DefaultScreen(GFX_Display),bank);
		while ( h-- ) {

			if ( w <= bankleft ) {
				wl = w;
				wr = 0;
			} else {
				wl = bankleft;
				wr = w - bankleft;
			}

			memcpy(dst, src, wl);

			bankleft -= vinfo.width;
			dst += vinfo.width;
			if ( bankleft <= 0 ) {
				XF86DGASetVidPage(GFX_Display,
					DefaultScreen(GFX_Display), ++bank);
				bankleft += vinfo.page;
				dst -= vinfo.page;
				memcpy(vinfo.base[0], src+wl, wr);
			}
			src += screen->pitch;
		}
	}
#endif /* XFREE86_DGA */
}

int SDL_SYS_SetColors(SDL_Surface *screen, int firstcolor, int ncolors)
{
	SDL_Color *colors;
	int      alloct_all;
	int      i;
	XColor  *xcmap;

	alloct_all = 1;
	colors = screen->format->palette->colors;
	ncolors = screen->format->palette->ncolors;
	xcmap = (XColor *)malloc(ncolors*sizeof(*xcmap));
	if ( xcmap == NULL ) {
		return(0);
	}

	/* It's easy if we have a private colormap */
	if ( (screen->flags & SDL_HWPALETTE) == SDL_HWPALETTE ) {
		for ( i=0; i<ncolors; ++i ) {
			xcmap[i].pixel = i;
			xcmap[i].red   = (colors[i].r<<8)|colors[i].r;
			xcmap[i].green = (colors[i].g<<8)|colors[i].g;
			xcmap[i].blue  = (colors[i].b<<8)|colors[i].b;
			xcmap[i].flags = (DoRed|DoGreen|DoBlue);
		}
		XStoreColors(GFX_Display, SDL_XColorMap, xcmap, ncolors);
#ifdef XFREE86_DGA
		if ( (screen->flags & SDL_FULLSCREEN) == SDL_FULLSCREEN ) {
			XF86DGAInstallColormap(SDL_Display, SDL_Screen,
							SDL_XColorMap);
		}
#endif
	} else {
		unsigned long pixel;
		XColor        wanted;
		XColor        color;

		/* Free existing allocated colors */
		for ( pixel=0; pixel<ncolors; ++pixel ) {
			while ( SDL_XPixels[pixel] > 0 ) {
				XFreeColors(GFX_Display, SDL_XColorMap, &pixel,
									1, 0);
				--SDL_XPixels[pixel];
			}
		}

		/* Try to allocate all the colors */
		for ( i=0; i<ncolors; ++i ) {
			color.pixel = i;
			color.red   = (colors[i].r<<8);
			color.green = (colors[i].g<<8);
			color.blue  = (colors[i].b<<8);
			color.flags = (DoRed|DoGreen|DoBlue);
			if ( alloct_all ) {
				memcpy(&wanted, &color, sizeof(wanted));
				if (XAllocColor(GFX_Display,
						SDL_XColorMap, &color)) {
					++SDL_XPixels[color.pixel];
				}
				if ( memcmp(&color, &wanted, sizeof(wanted)) )
					alloct_all = 0;
			} else {
				if (XAllocColor(GFX_Display,
						SDL_XColorMap, &color)) {
					++SDL_XPixels[color.pixel];
				}
			}
		}
		if ( ! alloct_all ) {
			/* Copy the colors from the colormap to our palette */
			for ( i=0; i<ncolors; ++i ) {
				xcmap[i].pixel = i;
				xcmap[i].flags = (DoRed|DoGreen|DoBlue);
			}
			XQueryColors(GFX_Display,SDL_XColorMap,xcmap,ncolors);
			for ( i=0; i<ncolors; ++i ) {
				colors[i].r = (xcmap[i].red>>8);
				colors[i].g = (xcmap[i].green>>8);
				colors[i].b = (xcmap[i].blue>>8);
			}
		}
	}
	free(xcmap);
	return(alloct_all);
}

/* Note:  If we are terminated, this could be called in the middle of
   another SDL video routine -- notably SDL_SYS_UpdateRects.
*/
void SDL_SYS_VideoQuit(SDL_Surface *screen)
{
	/* Shutdown everything that's still up */
	/* The event thread should be done, so we can touch SDL_Display */
	if ( SDL_Display != NULL ) {

		/* Start shutting down the windows */
		ReleaseMode(screen);
		if ( SDL_Window && (SDL_windowid == NULL) ) {
			XDestroyWindow(SDL_Display, SDL_Window);
		}
		if ( SDL_modelist ) {
			int i;

			for ( i=0; SDL_modelist[i]; ++i ) {
				free(SDL_modelist[i]);
			}
			free(SDL_modelist);
			SDL_modelist = NULL;
		}
		if ( SDL_PrivateColormap != 0 ) {
			XFreeColormap(SDL_Display, SDL_PrivateColormap);
			SDL_PrivateColormap = 0;
		}
		if ( SDL_XPixels ) {
			int numcolors;
			unsigned long pixel;

			numcolors = screen->format->palette->ncolors;
			for ( pixel=0; pixel<numcolors; ++pixel ) {
				while ( SDL_XPixels[pixel] > 0 ) {
					XFreeColors(GFX_Display,
						SDL_DisplayColormap,&pixel,1,0);
					--SDL_XPixels[pixel];
				}
			}
			free(SDL_XPixels);
			SDL_XPixels = NULL;
		} 
		if ( SDL_iconcolors ) {
			unsigned long pixel;
			int numcolors =
				((screen->format)->palette)->ncolors;
			for ( pixel=0; pixel<numcolors; ++pixel ) {
				while ( SDL_iconcolors[pixel] > 0 ) {
					XFreeColors(SDL_Display,
						SDL_DisplayColormap,&pixel,1,0);
					--SDL_iconcolors[pixel];
				}
			}
			free(SDL_iconcolors);
			SDL_iconcolors = NULL;
		} 
		/* Free that blank cursor */
		if ( SDL_BlankCursor != NULL ) {
			SDL_FreeWMCursor(NULL, SDL_BlankCursor);
			SDL_BlankCursor = NULL;
		}

		/* Close the X11 graphics connection */
		if ( GFX_Display != NULL ) {
			XCloseDisplay(GFX_Display);
			GFX_Display = NULL;
		}

		/* Free any MTRR control connections */
		if ( SDL_MTRR_fd >= 0 ) {
			close(SDL_MTRR_fd);
			SDL_MTRR_fd = -1;
		}

		/* Close the X11 display connection */
		XCloseDisplay(SDL_Display);
		SDL_Display = NULL;
	}
	if ( screen && (screen->flags & SDL_HWSURFACE) ) {
		/* Direct screen access, no memory buffer */
		screen->pixels = NULL;
	}
}
void SDL_SYS_FinalQuit(void) { }
