/*
 * tcl.c
 * A Tcl interface to libstroke.
 * By Chris Laas, chrisl@cybercom.net.
 * This file falls under the copyright of libstroke.
 */

#include <string.h>
#include <tcl.h>
#include "stroke.h"

#define STROKE_VERSION "0.2"

/* Prototypes. */
int StrokeCmd(ClientData, Tcl_Interp*, int, char*[]);

/*
 * Stroke_Init:
 *   Adds the libstroke library commands to an interpreter.
 * Results:  Standard.
 * Side effects:
 *   Adds the "stroke" command to the interpreter.
 */
int
Stroke_tcl_Init(interp)
	 Tcl_Interp *interp;
{
  stroke_init();
  if (Tcl_CreateCommand(interp, "stroke", StrokeCmd,
						(ClientData)NULL, (Tcl_CmdDeleteProc*)NULL) != TCL_OK)
	return TCL_ERROR;
  if (Tcl_PkgProvide(interp, "Stroke", STROKE_VERSION) != TCL_OK)
	return TCL_ERROR;
  return TCL_OK;
}

/*
 * Stroke_SafeInit:
 *   Adds the libstroke library commands to a safe interpreter.
 *   Libstroke is safe by nature, so this just calls Stroke_Init.
 */
int
Stroke_SafeInit(interp)
	 Tcl_Interp *interp;
{
  return Stroke_Init(interp);
}

/*
 * StrokeCmd:
 *   The command procedure for the "stroke" command.
 * Results:  Standard.
 * Side effects:
 *   Depends on first parameter.
 */
int
StrokeCmd(data, interp, argc, argv)
	 ClientData  data;   /* Unused */
	 Tcl_Interp *interp; /* Interp */
	 int         argc;   /* Arg count */
	 char       *argv[]; /* Args */
{
  if (argc < 2) {
	Tcl_AppendResult(interp, "must have at least 1 arg to stroke", NULL);
	return TCL_ERROR;
  }
  if (argv[1][0] == 'r' && strcmp(argv[1], "record") == 0) {
	int x, y;
	if (argc != 4) {
	  Tcl_AppendResult(interp, "must have 2 args to stroke record", NULL);
	  return TCL_ERROR;
	}
	if (Tcl_GetInt(interp, argv[2], &x) != TCL_OK)
	  return TCL_ERROR;
	if (Tcl_GetInt(interp, argv[3], &y) != TCL_OK)
	  return TCL_ERROR;
	stroke_record(x, y);
  } else if (argv[1][0] == 't' && strcmp(argv[1], "translate") == 0) {
	char buf[256];
	if (argc != 2) {
	  Tcl_AppendResult(interp, "stroke translate takes no args", NULL);
	  return TCL_ERROR;
	}
	if (!stroke_trans(buf)) {
	  Tcl_AppendResult(interp, "error in stroke translation", NULL);
	  return TCL_ERROR;
	}
	Tcl_AppendResult(interp, buf, NULL);
  } else {
	Tcl_AppendResult(interp, "unrecognized subcommand \"", argv[1], "\"",
					 NULL);
	return TCL_ERROR;
  }
  return TCL_OK;
}
