

/*
 *  Dr Geo an interactive geometry software
 * (C) Copyright Hilaire Fernandes  1997-1999
 * hilaire.fernandes@iname.com 
 * 
 *
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public Licences as by published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Publis License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "point.h"

extern liste_elem liste_figure;
extern GdkFont *fontobject;
extern GdkGC *font_gc;

// Common methods for point objects
void point_c::
init_nom (void)
{
  if (masque == OBJET_MACRO)
    return;
  strcpy (nom_type, _ ("this point %1"));
  strinsmsg (nom_type, nom, "%1");
}
point_s point_c::
coordonnees ()
{
  return p;
}
void point_c::
dessine (GdkPixmap * ecran, char force)
{
  int xe, ye;
  char nom1[LONGUEUR_NOM], taille;
  if ((masque == OBJET_MACRO) || (masque == TRUE && force == FALSE) || existe == FALSE)
    return;
  taille = epaisseur - 2;
  if (taille < 0)
    taille = 0;
  taille += RAYON;
  xe = ECRx (p.x);
  ye = ECRy (p.y);
  switch (style & 0xF)
    {
    case CARRE:
      rectanglefilled (ecran, xe - taille, ye - taille, taille << 1, taille << 1, couleur);
      break;
    case ROND:
      circlefilled (ecran, xe, ye, taille, couleur);
      break;
    case CROIX:
      line_good_clip (ecran, xe - taille, ye - taille, xe + taille, ye + taille, couleur);
      line_good_clip (ecran, xe + taille, ye - taille, xe - taille, ye + taille, couleur);
      break;
    }
  taille = gdk_string_width (fontobject, nom);
  gdk_draw_string (ecran, fontobject, font_gc, xe - RAYON - taille, ye + RAYON, nom);
}
char point_c::
appartient (int xm, int ym)
{
  double x, y;
  x = MONx (xm) - p.x;
  y = MONy (ym) - p.y;
  if (ABS (x) >= RI || ABS (y) >= RI)
    return FALSE;
  return TRUE;
}
void point_c::
move (int xm, int ym)
{
  double x, y;
  x = xm;
  y = ym;
  p.x += x / ECHELLE;
  p.y -= y / ECHELLE;
}
void point_c::
sauve_disk (FILE * f)
{
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
}
void point_c::
lire_disk (FILE * f)
{
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  init_nom ();
}

// Methods of point class derived from point_c
char point1::
dependance (figure_c * fig)
{
  return (s1 == fig || s2 == fig);
}
void point1::
actualise ()
{
  p.x = s1->valeur ();
  p.y = s2->valeur ();
}
liste_elem *point1::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) s1);
  liste_parent->ajoute ((void *) s2);
  return liste_parent;
}
void point1::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) s1);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) s2);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point1::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  s1 = (valeur_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  s2 = (valeur_c *) liste_figure.lire (pos);
  init_nom ();
}
// Class free 'point2'
void point2::
actualise ()
{
}
// Class point_sur_droite
char point_sur_droite::
dependance (figure_c * fig)
{
  return (parent == fig);
}
void point_sur_droite::
move (int mx, int my)
{
  droite_s co;
  double x, y, pente;
  vecteur_s u, am;
  x = mx;
  y = my;
  x = p.x + x / ECHELLE;
  y = p.y - y / ECHELLE;
  pente = parent->pente ();
  co = parent->coordonnees ();
  u = parent->directeur ();
  if (u.x == 0 && u.y == 0)
    return;
  if (fabs (pente) > 1)
    {
      p.y = y;
      if (pente != 1E100)
	p.x = (y - co.a.y) / pente + co.a.x;
      else
	p.x = co.a.x;
    }
  else
    {
      p.x = x;
      p.y = pente * (x - co.a.x) + co.a.y;
    }
  am = p - co.a;
  xm = am * u;
}
void point_sur_droite::
actualise ()
{
  vecteur_s u;
  droite_s co;
  if (!parent->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  co = parent->coordonnees ();
  u = parent->directeur ();
  p.x = xm * u.x + co.a.x;
  p.y = xm * u.y + co.a.y;
}
void point_sur_droite::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&xm, 1, sizeof (xm), f);
  pos = liste_figure.position ((void *) parent);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_droite::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&xm, 1, sizeof (xm), f);
  fread (&pos, 1, sizeof (pos), f);
  parent = (droite_c *) liste_figure.lire (pos);
  init_nom ();
}
// Class point_sur_demi_droite
char point_sur_demi_droite::
dependance (figure_c * fig)
{
  return (parent == fig);
}
void point_sur_demi_droite::
move (int mx, int my)
{
  droite_s co;
  double x, y, pente;
  vecteur_s u, am;
  x = mx;
  y = my;
  x = p.x + x / ECHELLE;
  y = p.y - y / ECHELLE;
  pente = parent->pente ();
  co = parent->coordonnees ();
  u = parent->directeur ();
  if (u.x == 0 && u.y == 0)
    {
      p = co.a;
      return;
    }
  if (fabs (pente) > 1)
    {
      p.y = y;
      if (pente != 1E100)
	p.x = (y - co.a.y) / pente + co.a.x;
      else
	p.x = co.a.x;
    }
  else
    {
      p.x = x;
      p.y = pente * (x - co.a.x) + co.a.y;
    }
  am = p - co.a;
  xm = am * u;
  if (xm < 0)
    xm = 0;
  p.x = xm * u.x + co.a.x;
  p.y = xm * u.y + co.a.y;
}
void point_sur_demi_droite::
actualise ()
{
// Ne reactualise pas l'abscisse du point sur la demi-droite mais ses coordonnees dans P
  vecteur_s u;
  droite_s co;
  if (parent->existe == FALSE)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  co = parent->coordonnees ();
  u = parent->directeur ();
  p.x = xm * u.x + co.a.x;
  p.y = xm * u.y + co.a.y;
}
void point_sur_demi_droite::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&xm, 1, sizeof (xm), f);
  pos = liste_figure.position ((void *) parent);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_demi_droite::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&xm, 1, sizeof (xm), f);
  fread (&pos, 1, sizeof (pos), f);
  parent = (droite_c *) liste_figure.lire (pos);
  init_nom ();
}
// Class point_sur_lieu
char point_sur_lieu::
dependance (figure_c * fig)
{
  return (parent == fig);
}
void point_sur_lieu::
move (int mx, int my)
{
  int nb;
  double d, d1;
  point_s l, pl, *adr;
  pl.x = p.x + ((double) mx) / ECHELLE;
  pl.y = p.y - ((double) my) / ECHELLE;		// position du curseur souris

  nb = parent->nb_objet_sur_lieu;
  adr = parent->liste_point;
  l = adr[--nb] - pl;
  xm = nb--;
  d = l * l;
//on cherche la plus proche dans le lieu
  for (; nb >= 0; nb--)
    {
      l = adr[nb] - pl;
      d1 = l * l;
      if (d1 < d)
	{
	  d = d1;
	  xm = nb;
	}
    }
}
void point_sur_lieu::
actualise ()
{
  int nb;
  point_s *adr;
  if (!parent->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  nb = parent->nb_objet_sur_lieu;
  adr = parent->liste_point;
  if (xm >= nb)
    xm = nb - 1;
  if (xm < 0)
    existe = FALSE;
  else
    p = adr[xm];
}
void point_sur_lieu::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&xm, 1, sizeof (xm), f);
  pos = liste_figure.position ((void *) parent);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_lieu::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&xm, 1, sizeof (xm), f);
  fread (&pos, 1, sizeof (pos), f);
  parent = (lieu_point_c *) liste_figure.lire (pos);
  init_nom ();
}
//Class point_milieu_segment
void point_milieu_segment::
init_nom (void)
{
  char lettre;
  char indice, tmp[8];
  if (masque == OBJET_MACRO)
    return;
  strcpy (nom_type, _ ("%1 the middle of %2"));

  strinsmsg (nom_type, parent->nom, "%2");	// l'arg2 est le nom du segment

  strinsmsg (nom_type, nom, "%1");	// l'arg1 est le nom du point

}
char point_milieu_segment::
dependance (figure_c * fig)
{
  return (parent == fig);
}
void point_milieu_segment::
actualise ()
{
  droite_s co;
  if (!parent->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  co = parent->coordonnees ();
  p = (co.a + co.b) / 2;
}
liste_elem *point_milieu_segment::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) parent);
  return liste_parent;
}
void point_milieu_segment::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) parent);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_milieu_segment::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  parent = (droite_c *) liste_figure.lire (pos);
  init_nom ();
}
//Class point_milieu_2points
void point_milieu_2points::
init_nom (void)
{
  char tmp[LONGUEUR_NOM_TYPE];
  if (masque == OBJET_MACRO)
    return;
  strcpy (nom_type, _ ("%1 the middle of %2"));
  if ((strlen (pt1->nom) + strlen (pt2->nom) + 3 + strlen (nom_type)) <= LONGUEUR_NOM_TYPE)
    sprintf (tmp, "[%s%s]", pt1->nom, pt2->nom);
  else
    strcpy (tmp, "S");
  strinsmsg (nom_type, tmp, "%2");	// arg2=nom du segment

  strinsmsg (nom_type, nom, "%1");	// arg1=nom du point

}
char point_milieu_2points::
dependance (figure_c * fig)
{
  return (pt1 == fig || pt2 == fig);
}
void point_milieu_2points::
actualise ()
{
  droite_s co;
  if (!pt1->existe || !pt2->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  co.a = pt1->coordonnees ();
  co.b = pt2->coordonnees ();
  p = (co.a + co.b) / 2;
}
liste_elem *point_milieu_2points::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) pt1);
  liste_parent->ajoute ((void *) pt2);
  return liste_parent;
}
void point_milieu_2points::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) pt1);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) pt2);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_milieu_2points::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  pt1 = (point_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  pt2 = (point_c *) liste_figure.lire (pos);
  init_nom ();
}

// Class point_sur_segment
char point_sur_segment::
dependance (figure_c * fig)
{
  return (parent == fig);
}
void point_sur_segment::
move (int mx, int my)
{
  droite_s co;
  double x, y, pente;
  vecteur_s u, am;
  x = mx;
  y = my;
  x = p.x + x / ECHELLE;
  y = p.y - y / ECHELLE;
  pente = parent->pente ();
  co = parent->coordonnees ();
  u.x = co.b.x - co.a.x;
  u.y = co.b.y - co.a.y;
  if (u.x == 0 && u.y == 0)
    return;
  if (fabs (pente) > 1)
    {
      p.y = y;
      if (pente != 1E100)
	p.x = (y - co.a.y) / pente + co.a.x;
      else
	p.x = co.a.x;
    }
  else
    {
      p.x = x;
      p.y = pente * (x - co.a.x) + co.a.y;
    }
  am = co.a - p;
  xm = sqrt (am * am) / sqrt (u * u);
  if (-am * u < 0)
    xm = 0;
  if (xm > 1)
    xm = 1;
  p.x = xm * u.x + co.a.x;
  p.y = xm * u.y + co.a.y;
}
void point_sur_segment::
actualise ()
{
// Ne reactualise pas l'abscisse du point mais sa position sur le segment
  vecteur_s u;
  droite_s co;
  if (!parent->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  co = parent->coordonnees ();
  u.x = co.b.x - co.a.x;
  u.y = co.b.y - co.a.y;
  p.x = xm * u.x + co.a.x;
  p.y = xm * u.y + co.a.y;
}
void point_sur_segment::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&xm, 1, sizeof (xm), f);
  pos = liste_figure.position ((void *) parent);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_segment::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&xm, 1, sizeof (xm), f);
  fread (&pos, 1, sizeof (pos), f);
  parent = (droite_c *) liste_figure.lire (pos);
  init_nom ();
}
// Class point_sur_cerlce
char point_sur_cercle::
dependance (figure_c * fig)
{
  return (parent == fig);
}
void point_sur_cercle::
move (int mx, int my)
{
  double k, den1, cos, sin;
  point_s centre, move;
  move.x = p.x + ((double) mx) / ECHELLE;
  move.y = p.y - ((double) my) / ECHELLE;
  centre = parent->centre ();
  move = move - centre;
  den1 = sqrt (move * move);
  if (den1 != 0)
    {
      k = parent->rayon () / den1;
      p = k * move + centre;
      cos = (p.x - centre.x);
      sin = (p.y - centre.y);
      xm = atan2 (sin, cos);
    }
}
void point_sur_cercle::
actualise ()
{
//Ne reactualise pas l'abscisse du point mais sa position sur la demi-droite
  point_s centre;
  double r;
  if (!parent->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  centre = parent->centre ();
  r = parent->rayon ();
  p.x = centre.x + r * cos (xm);
  p.y = centre.y + r * sin (xm);
}
void point_sur_cercle::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&xm, 1, sizeof (xm), f);
  pos = liste_figure.position ((void *) parent);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_cercle::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&xm, 1, sizeof (xm), f);
  fread (&pos, 1, sizeof (pos), f);
  parent = (cercle_c *) liste_figure.lire (pos);
  init_nom ();
}
// Class point_sur_arc_cerlce
char point_sur_arc_cercle::
dependance (figure_c * fig)
{
  return (parent == fig);
}
void point_sur_arc_cercle::
move (int mx, int my)
{
  point_s move;
  move.x = p.x + UNIT_ECRtoMON (mx);
  move.y = p.y - UNIT_ECRtoMON (my);
  move = move - parent->center;

  xm = atan2 (move.y, move.x) - parent->origin;

  if (parent->length > 0 && xm < 0)
    xm += 2 * PI;
  else if (parent->length < 0 && xm > 0)
    xm -= 2 * PI;

  xm /= parent->length;
  if (xm > 1)
    xm = 1;
  else if (xm < 0)
    xm = 0;
  move.x = cos (xm * parent->length + parent->origin);
  move.y = sin (xm * parent->length + parent->origin);
  p = parent->center + parent->radius * move;
}
void point_sur_arc_cercle::
actualise ()
{
//Ne reactualise pas l'abscisse du point mais ses coodonnees cartesiennes
  point_s trigo;
  if (!parent->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  trigo.x = cos (xm * parent->length + parent->origin);
  trigo.y = sin (xm * parent->length + parent->origin);

  p = parent->center + parent->radius * trigo;
}
void point_sur_arc_cercle::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&xm, 1, sizeof (xm), f);
  pos = liste_figure.position ((void *) parent);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_arc_cercle::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&xm, 1, sizeof (xm), f);
  fread (&pos, 1, sizeof (pos), f);
  parent = (arc_cercle_c *) liste_figure.lire (pos);
  init_nom ();
}
//
// point_sur_XXX_et_XXX
// point_sur_droite_et_droite
char point_sur_droite_et_droite::
dependance (figure_c * fig)
{
  return (d1 == fig || d2 == fig);
}
void point_sur_droite_et_droite::
actualise ()
{
  vecteur_s v1, v2;
  param_droite para1, para2;
  double det;
  if (!d1->existe || !d2->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  v1 = d1->directeur ();
  v2 = d2->directeur ();
  // les droites  sont-elles paralleles
  det = v1.x * v2.y - v1.y * v2.x;
  if (det == 0)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  para1 = d1->parametre ();
  para2 = d2->parametre ();
  p.x = (para2.c * para1.b - para1.c * para2.b) / det;
  p.y = (para2.a * para1.c - para1.a * para2.c) / det;
}
liste_elem *point_sur_droite_et_droite::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) d1);
  liste_parent->ajoute ((void *) d2);
  return liste_parent;
}
void point_sur_droite_et_droite::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) d1);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) d2);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_droite_et_droite::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  d1 = (droite_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  d2 = (droite_c *) liste_figure.lire (pos);
  init_nom ();
}
//      point_sur_droite_et_demi_droite
char point_sur_droite_et_demi_droite::
dependance (figure_c * fig)
{
  return (droite == fig || demi_droite == fig);
}
void point_sur_droite_et_demi_droite::
actualise ()
{
  vecteur_s v1, v2;
  param_droite para1, para2;
  double det;
  droite_s co;
  point_s ab, ap;
  if (!droite->existe || !demi_droite->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  v1 = droite->directeur ();
  v2 = demi_droite->directeur ();
  // les droites  sont-elles paralleles
  det = v1.x * v2.y - v1.y * v2.x;
  if (det == 0)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  para1 = droite->parametre ();
  para2 = demi_droite->parametre ();
  p.x = (para2.c * para1.b - para1.c * para2.b) / det;
  p.y = (para2.a * para1.c - para1.a * para2.c) / det;
  // verifier     si      le      point   est sur la demi-droite
  co = demi_droite->coordonnees ();
  ab.x = co.b.x - co.a.x;
  ab.y = co.b.y - co.a.y;
  ap.x = p.x - co.a.x;
  ap.y = p.y - co.a.y;
  if (ab.x * ap.x + ab.y * ap.y < 0)
    existe = FALSE;
}
liste_elem *point_sur_droite_et_demi_droite::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) droite);
  liste_parent->ajoute ((void *) demi_droite);
  return liste_parent;
}
void point_sur_droite_et_demi_droite::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) droite);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) demi_droite);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_droite_et_demi_droite::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  droite = (droite_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  demi_droite = (demi_droite_c *) liste_figure.lire (pos);
  init_nom ();
}
//      point_sur_droite_et_segment
char point_sur_droite_et_segment::
dependance (figure_c * fig)
{
  return (droite == fig || segment == fig);
}
void point_sur_droite_et_segment::
actualise ()
{
  vecteur_s v1, v2;
  param_droite para1, para2;
  double det;
  droite_s co;
  point_s ab, ap;
  if (!droite->existe || !segment->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  v1 = droite->directeur ();
  v2 = segment->directeur ();
  // les droites  sont-elles paralleles
  det = v1.x * v2.y - v1.y * v2.x;
  if (det == 0)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  para1 = droite->parametre ();
  para2 = segment->parametre ();
  p.x = (para2.c * para1.b - para1.c * para2.b) / det;
  p.y = (para2.a * para1.c - para1.a * para2.c) / det;
  // verifier     si      le      point   est sur le segment
  co = segment->coordonnees ();
  ab.x = co.b.x - co.a.x;
  ab.y = co.b.y - co.a.y;
  ap.x = p.x - co.a.x;
  ap.y = p.y - co.a.y;
  if (ab.x * ap.x + ab.y * ap.y < 0)
    {
      existe = FALSE;
      return;
    }
  ap.x = p.x - co.b.x;
  ap.y = p.y - co.b.y;
  if (ab.x * ap.x + ab.y * ap.y > 0)
    existe = FALSE;
}
liste_elem *point_sur_droite_et_segment::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) droite);
  liste_parent->ajoute ((void *) segment);
  return liste_parent;
}
void point_sur_droite_et_segment::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) droite);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) segment);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_droite_et_segment::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  droite = (droite_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  segment = (segment_c *) liste_figure.lire (pos);
  init_nom ();
}
//      point_sur_demi_droite_et_demi_droite
char point_sur_demi_droite_et_demi_droite::
dependance (figure_c * fig)
{
  return (dd1 == fig || dd1 == fig);
}
void point_sur_demi_droite_et_demi_droite::
actualise ()
{
  vecteur_s v1, v2;
  param_droite para1, para2;
  double det;
  droite_s co;
  point_s ab, ap;
  if (!dd1->existe || !dd2->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  v1 = dd1->directeur ();
  v2 = dd2->directeur ();
  // les droites  sont-elles paralleles
  det = v1.x * v2.y - v1.y * v2.x;
  if (det == 0)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  para1 = dd1->parametre ();
  para2 = dd2->parametre ();
  p.x = (para2.c * para1.b - para1.c * para2.b) / det;
  p.y = (para2.a * para1.c - para1.a * para2.c) / det;
  // verifier si le point est sur la 1ere demi_droite
  co = dd1->coordonnees ();
  ab.x = co.b.x - co.a.x;
  ab.y = co.b.y - co.a.y;
  ap.x = p.x - co.a.x;
  ap.y = p.y - co.a.y;
  if (ab.x * ap.x + ab.y * ap.y < 0)
    {
      existe = FALSE;
      return;
    }
  // verifier si le point est sur la 2ere demi_droite
  co = dd2->coordonnees ();
  ab.x = co.b.x - co.a.x;
  ab.y = co.b.y - co.a.y;
  ap.x = p.x - co.a.x;
  ap.y = p.y - co.a.y;
  if (ab.x * ap.x + ab.y * ap.y < 0)
    {
      existe = FALSE;
      return;
    }
}
liste_elem *point_sur_demi_droite_et_demi_droite::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) dd1);
  liste_parent->ajoute ((void *) dd2);
  return liste_parent;
}
void point_sur_demi_droite_et_demi_droite::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) dd1);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) dd2);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_demi_droite_et_demi_droite::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  dd1 = (demi_droite_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  dd2 = (demi_droite_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_demi_droite_et_segment
char point_sur_demi_droite_et_segment::
dependance (figure_c * fig)
{
  return (demi_droite == fig || segment == fig);
}
void point_sur_demi_droite_et_segment::
actualise ()
{
  vecteur_s v1, v2;
  param_droite para1, para2;
  double det;
  droite_s co;
  point_s ab, ap;
  if (!demi_droite->existe || !segment->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  v1 = demi_droite->directeur ();
  v2 = segment->directeur ();
  // les droites sont-elles paralleles
  det = v1.x * v2.y - v1.y * v2.x;
  if (det == 0)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  para1 = demi_droite->parametre ();
  para2 = segment->parametre ();
  p.x = (para2.c * para1.b - para1.c * para2.b) / det;
  p.y = (para2.a * para1.c - para1.a * para2.c) / det;
  // verifier si le point est sur la demi_droite
  co = demi_droite->coordonnees ();
  ab.x = co.b.x - co.a.x;
  ab.y = co.b.y - co.a.y;
  ap.x = p.x - co.a.x;
  ap.y = p.y - co.a.y;
  if (ab.x * ap.x + ab.y * ap.y < 0)
    {
      existe = FALSE;
      return;
    }
  // verifier si le point est sur le segment
  co = segment->coordonnees ();
  ab.x = co.b.x - co.a.x;
  ab.y = co.b.y - co.a.y;
  ap.x = p.x - co.a.x;
  ap.y = p.y - co.a.y;
  if (ab.x * ap.x + ab.y * ap.y < 0)
    {
      existe = FALSE;
      return;
    }
  ap.x = p.x - co.b.x;
  ap.y = p.y - co.b.y;
  if (ab.x * ap.x + ab.y * ap.y > 0)
    existe = FALSE;
}
liste_elem *point_sur_demi_droite_et_segment::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) demi_droite);
  liste_parent->ajoute ((void *) segment);
  return liste_parent;
}
void point_sur_demi_droite_et_segment::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) demi_droite);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) segment);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_demi_droite_et_segment::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  demi_droite = (demi_droite_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  segment = (segment_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_segment_et_segment
char point_sur_segment_et_segment::
dependance (figure_c * fig)
{
  return (s1 == fig || s2 == fig);
}
void point_sur_segment_et_segment::
actualise ()
{
  vecteur_s v1, v2;
  param_droite para1, para2;
  double det;
  droite_s co;
  point_s ab, ap;
  if (!s1->existe || !s1->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  v1 = s1->directeur ();
  v2 = s2->directeur ();
  // les droites sont-elles paralleles
  det = v1.x * v2.y - v1.y * v2.x;
  if (det == 0)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  para1 = s1->parametre ();
  para2 = s2->parametre ();
  p.x = (para2.c * para1.b - para1.c * para2.b) / det;
  p.y = (para2.a * para1.c - para1.a * para2.c) / det;
  // verifier si le point est sur le 1er segment
  co = s1->coordonnees ();
  ab.x = co.b.x - co.a.x;
  ab.y = co.b.y - co.a.y;
  ap.x = p.x - co.a.x;
  ap.y = p.y - co.a.y;
  if (ab.x * ap.x + ab.y * ap.y < 0)
    {
      existe = FALSE;
      return;
    }
  ap.x = p.x - co.b.x;
  ap.y = p.y - co.b.y;
  if (ab.x * ap.x + ab.y * ap.y > 0)
    {
      existe = FALSE;
      return;
    }
  // verifier     si      le      point   est sur le 2eme segment
  co = s2->coordonnees ();
  ab.x = co.b.x - co.a.x;
  ab.y = co.b.y - co.a.y;
  ap.x = p.x - co.a.x;
  ap.y = p.y - co.a.y;
  if (ab.x * ap.x + ab.y * ap.y < 0)
    {
      existe = FALSE;
      return;
    }
  ap.x = p.x - co.b.x;
  ap.y = p.y - co.b.y;
  if (ab.x * ap.x + ab.y * ap.y > 0)
    existe = FALSE;
}
liste_elem *point_sur_segment_et_segment::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) s1);
  liste_parent->ajoute ((void *) s2);
  return liste_parent;
}
void point_sur_segment_et_segment::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) s1);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) s2);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_segment_et_segment::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  s1 = (segment_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  s2 = (segment_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_cercle_et_droite
char point_sur_cercle_et_droite::
dependance (figure_c * fig)
{
  return (droite == fig || cercle == fig);
}
void point_sur_cercle_et_droite::
actualise ()
{
  point_s *inter, u, pt, c;
  if (!droite->existe || !cercle->existe)
    {
      existe = FALSE;
      return;
    }
  u = droite->directeur ();
  pt = (droite->coordonnees ()).a;
  c = cercle->centre ();
  inter = inter_cercle_droite (u, pt, c, cercle->rayon (), signe);
  if (inter == NULL)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  p = *inter;
  delete inter;
}
liste_elem *point_sur_cercle_et_droite::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) droite);
  liste_parent->ajoute ((void *) cercle);
  return liste_parent;
}
void point_sur_cercle_et_droite::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&signe, 1, sizeof (signe), f);
  pos = liste_figure.position ((void *) droite);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) cercle);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_cercle_et_droite::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&signe, 1, sizeof (signe), f);
  fread (&pos, 1, sizeof (pos), f);
  droite = (droite_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  cercle = (cercle_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_cercle_et_demi_droite
char point_sur_cercle_et_demi_droite::
dependance (figure_c * fig)
{
  return (demi_droite == fig || cercle == fig);
}
void point_sur_cercle_et_demi_droite::
actualise ()
{
  point_s *inter;
  droite_s co;
  point_s u, pt, c;
  if (!demi_droite->existe || !cercle->existe)
    {
      existe = FALSE;
      return;
    }
  u = demi_droite->directeur ();
  pt = (demi_droite->coordonnees ()).a;
  c = cercle->centre ();
  inter = inter_cercle_droite (u, pt, c, cercle->rayon (), signe);
  if (inter == NULL)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  p = *inter;
  delete inter;
  co = demi_droite->coordonnees ();
  u = co.b - co.a;
  pt = p - co.a;
  if (u * pt < 0)
    existe = FALSE;
}
liste_elem *point_sur_cercle_et_demi_droite::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) demi_droite);
  liste_parent->ajoute ((void *) cercle);
  return liste_parent;
}
void point_sur_cercle_et_demi_droite::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&signe, 1, sizeof (signe), f);
  pos = liste_figure.position ((void *) demi_droite);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) cercle);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_cercle_et_demi_droite::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&signe, 1, sizeof (signe), f);
  fread (&pos, 1, sizeof (pos), f);
  demi_droite = (demi_droite_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  cercle = (cercle_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_cercle_et_segment
char point_sur_cercle_et_segment::
dependance (figure_c * fig)
{
  return (segment == fig || cercle == fig);
}
void point_sur_cercle_et_segment::
actualise ()
{
  point_s *inter;
  droite_s co;
  point_s ab, ap, u, pt, c;
  if (!segment->existe || !cercle->existe)
    {
      existe = FALSE;
      return;
    }
  u = segment->directeur ();
  pt = (segment->coordonnees ()).a;
  c = cercle->centre ();
  inter = inter_cercle_droite (u, pt, c, cercle->rayon (), signe);
  if (inter == NULL)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  p = *inter;
  delete inter;
  co = segment->coordonnees ();
  u = co.b - co.a;
  pt = p - co.a;
  if (u * pt < 0)
    {
      existe = FALSE;
      return;
    }
  pt = p - co.b;
  if (u * pt > 0)
    existe = FALSE;
}
liste_elem *point_sur_cercle_et_segment::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) segment);
  liste_parent->ajoute ((void *) cercle);
  return liste_parent;
}
void point_sur_cercle_et_segment::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&signe, 1, sizeof (signe), f);
  pos = liste_figure.position ((void *) segment);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) cercle);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_cercle_et_segment::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&signe, 1, sizeof (signe), f);
  fread (&pos, 1, sizeof (pos), f);
  segment = (segment_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  cercle = (cercle_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_cercle_et_cercle
char point_sur_cercle_et_cercle::
dependance (figure_c * fig)
{
  return (cercle1 == fig || cercle2 == fig);
}
void point_sur_cercle_et_cercle::
actualise ()
{
  point_s *inter;
  point_s c1, c2, u, pta;
  double r1, r2, c, c0;
  if (!cercle1->existe || !cercle2->existe)
    {
      existe = FALSE;
      return;
    }
  c1 = cercle1->centre ();
  c2 = cercle2->centre ();
  r1 = cercle1->rayon ();
  r2 = cercle2->rayon ();
  u.x = c1.y - c2.y;
  u.y = c2.x - c1.x;
  c = (c1 * c1 - c2 * c2 + r2 * r2 - r1 * r1) / 2;
  c0 = c1 * u;
  pta.x = (c0 * (-u.x) + c * u.y) / (-u * u);
  pta.y = (c0 * (u.y) + c * u.x) / (u * u);
  inter = inter_cercle_droite (u, pta, c1, r1, signe);
  if (inter == NULL)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  p = *inter;
  delete inter;
}
liste_elem *point_sur_cercle_et_cercle::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) cercle1);
  liste_parent->ajoute ((void *) cercle2);
  return liste_parent;
}
void point_sur_cercle_et_cercle::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&signe, 1, sizeof (signe), f);
  pos = liste_figure.position ((void *) cercle1);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) cercle2);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_cercle_et_cercle::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&signe, 1, sizeof (signe), f);
  fread (&pos, 1, sizeof (pos), f);
  cercle1 = (cercle_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  cercle2 = (cercle_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_arc_cercle_et_droite
char point_sur_arc_cercle_et_droite::
dependance (figure_c * fig)
{
  return (droite == fig || arc_cercle == fig);
}
void point_sur_arc_cercle_et_droite::
actualise ()
{
  point_s *inter, u, pt;
  double am;
  if (!droite->existe || !arc_cercle->existe)
    {
      existe = FALSE;
      return;
    }
  u = droite->directeur ();
  pt = (droite->coordonnees ()).a;
  inter = inter_cercle_droite (u, pt, arc_cercle->center, arc_cercle->radius, signe);
  if (inter == NULL)
    {
      existe = FALSE;
      return;
    }
  p = *inter;
  delete inter;

  // compute angle between pt and the arc origin and check if p is on the arc
  pt = p - arc_cercle->center;
  am = atan2 (pt.y, pt.x) - arc_cercle->origin;
  if (arc_cercle->length > 0 && am < 0)
    am += 2 * PI;
  else if (arc_cercle->length < 0 && am > 0)
    am -= 2 * PI;

  if (fabs (am) > fabs (arc_cercle->length))
    existe = FALSE;
  else
    existe = TRUE;
}
liste_elem *point_sur_arc_cercle_et_droite::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) droite);
  liste_parent->ajoute ((void *) arc_cercle);
  return liste_parent;
}
void point_sur_arc_cercle_et_droite::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&signe, 1, sizeof (signe), f);
  pos = liste_figure.position ((void *) droite);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) arc_cercle);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_arc_cercle_et_droite::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&signe, 1, sizeof (signe), f);
  fread (&pos, 1, sizeof (pos), f);
  droite = (droite_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  arc_cercle = (arc_cercle_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_arc_cercle_et_demi_droite
char point_sur_arc_cercle_et_demi_droite::
dependance (figure_c * fig)
{
  return (demi_droite == fig || arc_cercle == fig);
}
void point_sur_arc_cercle_et_demi_droite::
actualise ()
{
  droite_s co;
  point_s *inter, u, pt;
  double am;
  if (!demi_droite->existe || !arc_cercle->existe)
    {
      existe = FALSE;
      return;
    }
  u = demi_droite->directeur ();
  pt = (demi_droite->coordonnees ()).a;
  inter = inter_cercle_droite (u, pt, arc_cercle->center, arc_cercle->radius, signe);
  if (inter == NULL)
    {
      existe = FALSE;
      return;
    }
  p = *inter;
  delete inter;
  // check if p on the half-line
  co = demi_droite->coordonnees ();
  u = co.b - co.a;
  pt = p - co.a;
  if (u * pt < 0)
    {
      existe = FALSE;
      return;
    }

  // compute angle between pt and the arc origin and check if p is on the arc
  pt = p - arc_cercle->center;
  am = atan2 (pt.y, pt.x) - arc_cercle->origin;
  if (arc_cercle->length > 0 && am < 0)
    am += 2 * PI;
  else if (arc_cercle->length < 0 && am > 0)
    am -= 2 * PI;

  if (fabs (am) > fabs (arc_cercle->length))
    existe = FALSE;
  else
    existe = TRUE;
}
liste_elem *point_sur_arc_cercle_et_demi_droite::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) demi_droite);
  liste_parent->ajoute ((void *) arc_cercle);
  return liste_parent;
}
void point_sur_arc_cercle_et_demi_droite::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&signe, 1, sizeof (signe), f);
  pos = liste_figure.position ((void *) demi_droite);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) arc_cercle);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_arc_cercle_et_demi_droite::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&signe, 1, sizeof (signe), f);
  fread (&pos, 1, sizeof (pos), f);
  demi_droite = (demi_droite_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  arc_cercle = (arc_cercle_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_arc_cercle_et_segment
char point_sur_arc_cercle_et_segment::
dependance (figure_c * fig)
{
  return (segment == fig || arc_cercle == fig);
}
void point_sur_arc_cercle_et_segment::
actualise ()
{
  droite_s co;
  point_s *inter, u, pt;
  double am;
  if (!segment->existe || !arc_cercle->existe)
    {
      existe = FALSE;
      return;
    }
  u = segment->directeur ();
  pt = (segment->coordonnees ()).a;
  inter = inter_cercle_droite (u, pt, arc_cercle->center, arc_cercle->radius, signe);
  if (inter == NULL)
    {
      existe = FALSE;
      return;
    }
  p = *inter;
  delete inter;
  // check if p on the segment
  co = segment->coordonnees ();
  u = co.b - co.a;
  pt = p - co.a;
  if (u * pt < 0)
    {
      existe = FALSE;
      return;
    }
  pt = p - co.b;
  if (u * pt > 0)
    {
      existe = FALSE;
      return;
    }

  // compute angle between pt and the arc origin and check if p is on the arc
  pt = p - arc_cercle->center;
  am = atan2 (pt.y, pt.x) - arc_cercle->origin;
  if (arc_cercle->length > 0 && am < 0)
    am += 2 * PI;
  else if (arc_cercle->length < 0 && am > 0)
    am -= 2 * PI;

  if (fabs (am) > fabs (arc_cercle->length))
    existe = FALSE;
  else
    existe = TRUE;
}
liste_elem *point_sur_arc_cercle_et_segment::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) segment);
  liste_parent->ajoute ((void *) arc_cercle);
  return liste_parent;
}
void point_sur_arc_cercle_et_segment::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&signe, 1, sizeof (signe), f);
  pos = liste_figure.position ((void *) segment);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) arc_cercle);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_arc_cercle_et_segment::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&signe, 1, sizeof (signe), f);
  fread (&pos, 1, sizeof (pos), f);
  segment = (segment_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  arc_cercle = (arc_cercle_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_arc_cercle_et_cercle
char point_sur_arc_cercle_et_cercle::
dependance (figure_c * fig)
{
  return (cercle == fig || arc_cercle == fig);
}
void point_sur_arc_cercle_et_cercle::
actualise ()
{
  point_s *inter, u, pta, c1, c2;
  double r1, r2, c, c0;
  double am;
  if (!cercle->existe || !arc_cercle->existe)
    {
      existe = FALSE;
      return;
    }
  c1 = arc_cercle->center;
  c2 = cercle->centre ();
  r1 = arc_cercle->radius;
  r2 = cercle->rayon ();
  u.x = c1.y - c2.y;
  u.y = c2.x - c1.x;
  c = (c1 * c1 - c2 * c2 + r2 * r2 - r1 * r1) / 2;
  c0 = c1 * u;
  pta.x = (c0 * (-u.x) + c * u.y) / (-u * u);
  pta.y = (c0 * (u.y) + c * u.x) / (u * u);
  inter = inter_cercle_droite (u, pta, c1, r1, signe);

  if (inter == NULL)
    {
      existe = FALSE;
      return;
    }
  p = *inter;
  delete inter;

  // compute angle between pt and the arc origin and check if p is on the arc
  pta = p - arc_cercle->center;
  am = atan2 (pta.y, pta.x) - arc_cercle->origin;
  if (arc_cercle->length > 0 && am < 0)
    am += 2 * PI;
  else if (arc_cercle->length < 0 && am > 0)
    am -= 2 * PI;

  if (fabs (am) > fabs (arc_cercle->length))
    existe = FALSE;
  else
    existe = TRUE;
}
liste_elem *point_sur_arc_cercle_et_cercle::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) cercle);
  liste_parent->ajoute ((void *) arc_cercle);
  return liste_parent;
}
void point_sur_arc_cercle_et_cercle::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&signe, 1, sizeof (signe), f);
  pos = liste_figure.position ((void *) cercle);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) arc_cercle);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_arc_cercle_et_cercle::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&signe, 1, sizeof (signe), f);
  fread (&pos, 1, sizeof (pos), f);
  cercle = (cercle_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  arc_cercle = (arc_cercle_c *) liste_figure.lire (pos);
  init_nom ();
}
// point_sur_arc_cercle_et_arc_cercle
char point_sur_arc_cercle_et_arc_cercle::
dependance (figure_c * fig)
{
  return (arc_cercle1 == fig || arc_cercle2 == fig);
}
void point_sur_arc_cercle_et_arc_cercle::
actualise ()
{
  point_s *inter, u, pta, c1, c2;
  double r1, r2, c, c0;
  double am;
  if (!arc_cercle1->existe || !arc_cercle2->existe)
    {
      existe = FALSE;
      return;
    }
  c1 = arc_cercle1->center;
  c2 = arc_cercle2->center;
  r1 = arc_cercle1->radius;
  r2 = arc_cercle2->radius;
  u.x = c1.y - c2.y;
  u.y = c2.x - c1.x;
  c = (c1 * c1 - c2 * c2 + r2 * r2 - r1 * r1) / 2;
  c0 = c1 * u;
  pta.x = (c0 * (-u.x) + c * u.y) / (-u * u);
  pta.y = (c0 * (u.y) + c * u.x) / (u * u);
  inter = inter_cercle_droite (u, pta, c1, r1, signe);

  if (inter == NULL)
    {
      existe = FALSE;
      return;
    }
  p = *inter;
  delete inter;

  existe = TRUE;
  // check if p is on the first arc
  pta = p - arc_cercle1->center;
  am = atan2 (pta.y, pta.x) - arc_cercle1->origin;
  if (arc_cercle1->length > 0 && am < 0)
    am += 2 * PI;
  else if (arc_cercle1->length < 0 && am > 0)
    am -= 2 * PI;

  if (fabs (am) > fabs (arc_cercle1->length))
    {
      existe = FALSE;
      return;
    }
  // check if p is on the second arc
  pta = p - arc_cercle2->center;
  am = atan2 (pta.y, pta.x) - arc_cercle2->origin;
  if (arc_cercle2->length > 0 && am < 0)
    am += 2 * PI;
  else if (arc_cercle2->length < 0 && am > 0)
    am -= 2 * PI;

  if (fabs (am) > fabs (arc_cercle2->length))
    existe = FALSE;
}
liste_elem *point_sur_arc_cercle_et_arc_cercle::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) arc_cercle1);
  liste_parent->ajoute ((void *) arc_cercle2);
  return liste_parent;
}
void point_sur_arc_cercle_et_arc_cercle::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  fwrite (&signe, 1, sizeof (signe), f);
  pos = liste_figure.position ((void *) arc_cercle1);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) arc_cercle2);
  fwrite (&pos, 1, sizeof (pos), f);
}
void point_sur_arc_cercle_et_arc_cercle::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&signe, 1, sizeof (signe), f);
  fread (&pos, 1, sizeof (pos), f);
  arc_cercle1 = (arc_cercle_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  arc_cercle2 = (arc_cercle_c *) liste_figure.lire (pos);
  init_nom ();
}

// Transformed Point
// By reflexion
char reflexion_point::
dependance (figure_c * fig)
{
  return (axe == fig || pt == fig);
}
point_s reflexion_point::
coordonnees (void)
{
  return reflexion_p (axe, pt->coordonnees ());
}
void reflexion_point::
actualise (void)
{
  if (!axe->existe || !pt->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  p = coordonnees ();
}
void reflexion_point::
move (int mx, int my)
{
}
liste_elem *reflexion_point::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) axe);
  liste_parent->ajoute ((void *) pt);
  return liste_parent;
}

void reflexion_point::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) axe);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) pt);
  fwrite (&pos, 1, sizeof (pos), f);
}
void reflexion_point::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  axe = (droite_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  pt = (point_c *) liste_figure.lire (pos);
  init_nom ();
}
// By symetry
char symetrie_point::
dependance (figure_c * fig)
{
  return (symetrie == fig || pt == fig);
}
point_s symetrie_point::
coordonnees (void)
{
  return (2 * symetrie->coordonnees () - pt->coordonnees ());
}
void symetrie_point::
actualise (void)
{
  if (!symetrie->existe || !pt->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  p = coordonnees ();
}
void symetrie_point::
move (int mx, int my)
{
}
liste_elem *symetrie_point::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) pt);
  liste_parent->ajoute ((void *) symetrie);
  return liste_parent;
}
void symetrie_point::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) pt);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) symetrie);
  fwrite (&pos, 1, sizeof (pos), f);
}
void symetrie_point::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  pt = (point_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  symetrie = (point_c *) liste_figure.lire (pos);
  init_nom ();
}
// By translation
char translation_point::
dependance (figure_c * fig)
{
  return (v == fig || pt == fig);
}
point_s translation_point::
coordonnees (void)
{
  return (v->vecteur () + pt->coordonnees ());
}
void translation_point::
actualise (void)
{
  if (!v->existe || !pt->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  p = coordonnees ();
}
void translation_point::
move (int mx, int my)
{
}
liste_elem *translation_point::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) v);
  liste_parent->ajoute ((void *) pt);
  return liste_parent;
}
void translation_point::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) pt);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) v);
  fwrite (&pos, 1, sizeof (pos), f);
}
void translation_point::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  pt = (point_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  v = (vecteur_c *) liste_figure.lire (pos);
  init_nom ();
}
// By rotation
char rotation_point::
dependance (figure_c * fig)
{
  return (v == fig || c == fig || pt == fig);
}
point_s rotation_point::
coordonnees (void)
{
  return rotation_p (c, v, pt->coordonnees ());
}
void rotation_point::
actualise (void)
{
  if (!v->existe || !c->existe || !pt->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  p = coordonnees ();
}
void rotation_point::
move (int mx, int my)
{
}
liste_elem *rotation_point::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) v);
  liste_parent->ajoute ((void *) c);
  liste_parent->ajoute ((void *) pt);
  return liste_parent;
}
void rotation_point::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) pt);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) c);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) v);
  fwrite (&pos, 1, sizeof (pos), f);
}
void rotation_point::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  pt = (point_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  c = (point_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  v = (valeur_c *) liste_figure.lire (pos);
  init_nom ();
}
// By homothetie
char homothetie_point::
dependance (figure_c * fig)
{
  return (v == fig || c == fig || pt == fig);
}
point_s homothetie_point::
coordonnees (void)
{
  return (c->p + (pt->p - c->p) * v->val);
}
void homothetie_point::
actualise (void)
{
  if (!v->existe || !c->existe || !pt->existe)
    {
      existe = FALSE;
      return;
    }
  existe = TRUE;
  p = coordonnees ();
}
void homothetie_point::
move (int mx, int my)
{
}
liste_elem *homothetie_point::
parents (liste_elem * liste_parent)
{
  liste_parent->vide ();
  liste_parent->ajoute ((void *) v);
  liste_parent->ajoute ((void *) c);
  liste_parent->ajoute ((void *) pt);
  return liste_parent;
}
void homothetie_point::
sauve_disk (FILE * f)
{
  int pos;
  sauver_parametre_base (f, this);
  fwrite (&p, 1, sizeof (p), f);
  pos = liste_figure.position ((void *) pt);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) c);
  fwrite (&pos, 1, sizeof (pos), f);
  pos = liste_figure.position ((void *) v);
  fwrite (&pos, 1, sizeof (pos), f);
}
void homothetie_point::
lire_disk (FILE * f)
{
  int pos;
  lire_parametre_base (f, this);
  fread (&p, 1, sizeof (p), f);
  fread (&pos, 1, sizeof (pos), f);
  pt = (point_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  c = (point_c *) liste_figure.lire (pos);
  fread (&pos, 1, sizeof (pos), f);
  v = (valeur_c *) liste_figure.lire (pos);
  init_nom ();
}

// Associated function for intersection with circle object
// Intersection cercle-droite   (point et vecteur directeur de la droite
// fourni)
point_s *
inter_cercle_droite (point_s & u1, point_s & pt1, point_s & centre, double rayon, char k)
{
//ne pas oublier de rendre la memoire allouer pour le pointeur sur point_s
  double delta, a, b, c;
  point_s *inter;
  a = u1.x * u1.x + u1.y * u1.y;
  b = u1.x * (pt1.x - centre.x) + u1.y * (pt1.y - centre.y);
  c = pow (pt1.x - centre.x, 2) + pow (pt1.y - centre.y, 2) - rayon * rayon;
  delta = b * b - a * c;	// discriminant reduit

  if (delta < 0 || a == 0)
    return NULL;
  inter = new point_s;
  c = (-b + k * sqrt (delta)) / a;
  *inter = pt1 + c * u1;
  return inter;
}

// Intersection cercle-droite (coefficient d'une equation cartesienne)
point_s *
inter_cercle_droite_cart (param_droite para, point_s * centre, double rayon, char k)
{
//ne pas oublier de rendre la memoire allouer pour le pointeur sur point_s
  vecteur_s n, u;
  double delta, v;
  point_s *inter;
  point_s m;
  n.x = para.a;
  n.y = para.b;
  if (n.x == 0 && n.y == 0)
    return NULL;
  v = sqrt (n.x * n.x + n.y * n.y);
  n.x /= v;
  n.y /= v;
  if (n.x != 0)
    {
      m.x = -para.c / para.a;
      m.y = 0;
    }
  else
    {
      m.x = 0;
      m.y = -para.c / para.b;
    }
  u.x = centre->x - m.x;
  u.y = centre->y - m.y;
  delta = rayon * rayon - pow (u.x * n.x + u.y * n.y, 2);
  if (delta < 0)
    return NULL;
  if (para.b != 0)
    {
      inter = new point_s;
      delta *= para.a * para.a + para.b * para.b;
      delta /= para.b * para.b;
      v = centre->x - (para.c / para.b + centre->y) * para.a / para.b + k * sqrt (delta);
      inter->x = v / (1 + para.a * para.a / (para.b * para.b));
      inter->y = -(para.a * inter->x + para.c) / para.b;
      return inter;
    }
  // ici b=0
  inter = new point_s;
  inter->y = centre->y + k * sqrt (delta);
  inter->x = -para.c / para.a;
  return inter;
}
// Intersection of two lines
// return NULL if no or infinit solution
// else return a pointer on a point_s structure
// Do not forget the memory if pointer not null
// d1 ad d2 are the cartesian coeff of the two lines
point_s *
cramer (param_droite d1, param_droite d2)
{
  point_s *s;
  double d;
  d = d1.a * d2.b - d2.a * d1.b;
  if (d == 0)
    return NULL;
  s = new point_s;
  s->x = (-d1.c * d2.b + d2.c * d1.b) / d;
  s->y = (-d1.a * d2.c + d2.a * d1.c) / d;
  return s;
}
