#include "ooglutil.h"
#include "geom.h"
#include "meshP.h"
#include "crayolaP.h"

static char msg[] = "crayMesh.c";

void *cray_mesh_HasVColor(int sel, Geom *geom, va_list args);
void *cray_mesh_UseVColor(int sel, Geom *geom, va_list args);
void *cray_mesh_EliminateColor(int sel, Geom *geom, va_list args);
void *cray_mesh_SetColorAll(int sel, Geom *geom, va_list args);
void *cray_mesh_SetColorAt(int sel, Geom *geom, va_list args);
void *cray_mesh_SetColorAtF(int sel, Geom *geom, va_list args);
void *cray_mesh_SetColorAtV(int sel, Geom *geom, va_list args);
void *cray_mesh_GetColorAt(int sel, Geom *geom, va_list args);
void *cray_mesh_GetColorAtV(int sel, Geom *geom, va_list args);

#define MAX_METHODS 11

static craySpecFunc methods[] = {
  "crayHasColor", cray_mesh_HasVColor,
  "crayHasVColor", cray_mesh_HasVColor,
  "crayCanUseVColor", crayTrue,
  "crayUseVColor", cray_mesh_UseVColor,
  "crayEliminateColor", cray_mesh_EliminateColor,
  "craySetColorAll", cray_mesh_SetColorAll,
  "craySetColorAt", cray_mesh_SetColorAt,
  "craySetColorAtF", cray_mesh_SetColorAtF,
  "craySetColorAtV", cray_mesh_SetColorAtV,
  "crayGetColorAt", cray_mesh_GetColorAt,
  "crayGetColorAtV", cray_mesh_GetColorAtV
  };

cray_mesh_init() {
  crayInitSpec(methods, MAX_METHODS, GeomClassLookup("mesh"));
  return 0;
}

void *cray_mesh_HasVColor(int sel, Geom *geom, va_list args) {
  Mesh *m = (Mesh *)geom;
  return (void *)(m->flag & MESH_C);
}

void *cray_mesh_UseVColor(int sel, Geom *geom, va_list args) {
  int i;
  Mesh *m = (Mesh *)geom;
  ColorA *def;

  if (crayHasVColor(geom, NULL)) return 0;

  def = va_arg(args, ColorA *);

  m->c = OOGLNewNE(ColorA, m->nu * m->nv, msg);
  for (i = 0; i < m->nu * m->nv; i++) {
    m->c[i].r = def->r;
    m->c[i].g = def->g;
    m->c[i].b = def->b;
    m->c[i].a = def->a;
  }
  m->flag |= MESH_C;
  return (void *)geom;
}

void *cray_mesh_EliminateColor(int sel, Geom *geom, va_list args) {
  Mesh *m = (Mesh *)geom;
  
  if (!crayHasColor(geom, NULL)) return 0;
  
  OOGLFree(m->c);
  m->c = NULL;
  m->flag ^= MESH_C;
  return (void *)geom;
}

void *cray_mesh_SetColorAll(int sel, Geom *geom, va_list args) {
  int i;
  Mesh *m = (Mesh *)geom;
  ColorA *color;
 
  if (!crayHasVColor(geom, NULL)) return NULL;
  color = va_arg(args, ColorA *);
  for (i = 0; i < m->nu * m->nv; i++) m->c[i] = *color;
  return (void *)geom;
}

void *cray_mesh_SetColorAt(int sel, Geom *geom, va_list args) {
  ColorA *color;
  int vindex, findex;
  color = va_arg(args, ColorA *);
  vindex = va_arg(args, int);
  findex = va_arg(args, int);
  if (vindex != -1) 
    return (void *)craySetColorAtV(geom, color, vindex, NULL, NULL);
  return (void *)craySetColorAtF(geom, color, findex, NULL);
}

void *cray_mesh_SetColorAtV(int sel, Geom *geom, va_list args) {
  Mesh *m = (Mesh *)geom;
  ColorA *color;
  int index;

  color = va_arg(args, ColorA *);
  index = va_arg(args, int);
  if (!crayHasVColor(geom, NULL) || index == -1) return 0;
  m->c[index].r = color->r;
  m->c[index].g = color->g;
  m->c[index].b = color->b;
  m->c[index].a = color->a;
  return (void *)color;
}

void *cray_mesh_SetColorAtF(int sel, Geom *geom, va_list args) {
  Mesh *m = (Mesh *)geom;
  ColorA *color;
  int index, u, v;

  color = va_arg(args, ColorA *);
  index = va_arg(args, int);
  if (!crayHasColor(geom, NULL) || index == -1) return 0;

  u = index % m->nu;
  v = index / m->nu;

  m->c[MESHINDEX(u, v, m)] = *color;
  m->c[MESHINDEX(u+1, v, m)] = *color;
  m->c[MESHINDEX(u+1, v+1, m)] = *color;
  m->c[MESHINDEX(u, v+1, m)] = *color;
  return (void *)m;
}

void *cray_mesh_GetColorAt(int sel, Geom *geom, va_list args) {
  ColorA *color;
  int i;
  color = va_arg(args, ColorA *);
  i = va_arg(args, int);	/* Vertex index */
  if(i < 0)
    i = va_arg(args, int);	/* If no vertex, use corresponding face index */
  if(i < 0) {
    int *edge = va_arg(args, int *);
    i = edge[0];		/* If neither, choose one of the edge verts */
  }
  return (void *)crayGetColorAtV(geom, color, i, NULL, NULL);
}
     
void *cray_mesh_GetColorAtV(int sel, Geom *geom, va_list args) {
 ColorA *color;
 int i;
 Mesh *m = (Mesh *)geom;
 if (!crayHasVColor(geom, NULL)) return 0;
 color = va_arg(args, ColorA *);
 i = va_arg(args, int);
 *color = m->c[i];
 return (void *)color;
}
