%{

#include "fexpr.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

extern void yyerror(char *);
extern int yyparse();
extern int yyrestart(void *);

#define YYDEBUG 1

struct expr_tree {
  struct expr_elem elem;
  struct expr_tree *sub1;
  struct expr_tree *sub2;
};

struct expr_tree *expr_parsed;
struct expression *expr_current;

static struct expr_parse_free {
  void *dat;
  struct expr_parse_free *next;
} *freers = NULL;

static void expr_free_dat(void *dat)
{
  struct expr_parse_free *f;
  f=malloc(sizeof(struct expr_parse_free));
  f->next=freers;
  freers=f;
  f->dat = dat;
}

static void expr_free_freers()
{
  struct expr_parse_free *f, *next=NULL;
  for (f=freers; f; f=next) {
    next = f->next;
    free(f->dat);
    free(f);
  }
  freers = NULL;
}

static void expr_free_freers_no_dat()
{
  struct expr_parse_free *f, *next=NULL;
  for (f=freers; f; f=next) {
    next = f->next;
    free(f);
  }
  freers = NULL;
}

static int count_nodes(struct expr_tree *e)
{
  if (!e) return 0;
  return count_nodes(e->sub1) + count_nodes(e->sub2) + 1;
}

static void store_nodes(struct expr_tree *e, int *pos)
{
  if (!e) return;
  store_nodes(e->sub2, pos);
  store_nodes(e->sub1, pos);
  expr_current->elems[*pos] = e->elem;
  (*pos)++;
}

static char *error_return;

char *expr_parse(struct expression *expr, char *str)
{
  int i;

  error_return = NULL;
  
  expr_current = expr;

  if (!str || !*str) return "Empty expression";
  expr_lex_reset_input(str);

  i = yyparse();		/* do all the work. */
  yyrestart(NULL);		/* clean up so we can do this again. */
  
  if (i) {
    expr_free_freers();
    return "Parse error";
  }

  if (error_return) {
    expr_free_freers();
    return error_return;
  }

  expr_free_freers_no_dat();

  /* ok. now we convert the parsed expression to a execution vector. whee. */
  expr->nelem = count_nodes(expr_parsed);
  expr->elems = malloc (expr->nelem * sizeof (struct expr_elem));
  
  /* we let i be a counter variable for where we're writing in 
   * the vectorized expression. */
  i = 0;
  store_nodes(expr_parsed, &i);
  return NULL;
}

static struct expr_monfunc lookup_monfunc(char *s)
{
  struct expr_monfunc *i;
  static struct expr_monfunc fail = { NULL, NULL, NULL };

  for (i=expr_monfuncs; i->name; i++)
    if (!strcmp(i->name,s))
      return *i;

  {
    static char buf[1024];
    sprintf(buf,"Undefined unary function: %s",s);
    error_return = buf;
  }
  return fail;
}

static struct expr_binfunc lookup_binfunc(char *s)
{
  struct expr_binfunc *i;
  static struct expr_binfunc fail = { NULL, NULL, NULL };

  for (i=expr_binfuncs; i->name; i++)
    if (!strcmp(i->name,s))
      return *i;

  {
    static char buf[1024];
    sprintf(buf,"Undefined binary function: %s",s);
    error_return = buf;
  }
  return fail;
}

static struct expr_tree *expr_parse_monop(int opr, struct expr_tree *sub)
{
  struct expr_tree *op;

  op = malloc (sizeof (struct expr_tree));
  op->sub1 = op->sub2 = NULL;
  op->elem.op = MONOP;
  op->elem.u.monop.op = opr;
  op->sub1 = sub;
  expr_free_dat(op);
  return op;
}

static struct expr_tree *expr_parse_binop(int opr, struct expr_tree *sub1, struct expr_tree *sub2)
{
  struct expr_tree *op;

  op = malloc (sizeof (struct expr_tree));
  op->sub1 = op->sub2 = NULL;
  op->elem.op = BINOP;
  op->elem.u.binop.op = opr;
  op->sub1 = sub1;
  op->sub2 = sub2;
  expr_free_dat(op);
  return op;
}

static struct expr_tree *expr_parse_monfunc(struct expr_monfunc f, struct expr_tree *sub1)
{
  struct expr_tree *op;

  op = malloc (sizeof (struct expr_tree));
  op->sub1 = op->sub2 = NULL;
  op->elem.op = MONFUNC;
  op->elem.u.monfunc.func = f.func;
  op->elem.u.monfunc.cfunc = f.cfunc;
  op->sub1 = sub1;
  expr_free_dat(op);
  return op;
}

static struct expr_tree *expr_parse_binfunc(struct expr_binfunc f, struct expr_tree *sub1, struct expr_tree *sub2)
{
  struct expr_tree *op;

  op = malloc (sizeof (struct expr_tree));
  op->sub1 = op->sub2 = NULL;
  op->elem.op = BINFUNC;
  op->elem.u.binfunc.func = f.func;
  op->elem.u.binfunc.cfunc = f.cfunc;
  op->sub1 = sub1;
  op->sub2 = sub2;
  expr_free_dat(op);
  return op;
}

static struct expr_tree *expr_parse_var(char *s)
{
  struct expr_tree *op;
  int i=0;

  op = malloc (sizeof (struct expr_tree));
  op->sub1 = op->sub2 = NULL;

  op->elem.u.pushvar.varnum = -1;

  op->elem.op = PUSHVAR;
  for (i=0; i<expr_current->nvars; i++)
    if (!strcmp(expr_current->varnames[i], s))
      op->elem.u.pushvar.varnum = i;

  if (op->elem.u.pushvar.varnum == -1) {
    static char buf[1024];
    sprintf(buf,"Undefined variable: %s", s);
    error_return = buf;
  }
  expr_free_dat(op);
  return op;
}

static struct expr_tree *expr_parse_num(double d)
{
  struct expr_tree *op;

  op = malloc (sizeof (struct expr_tree));
  op->sub1 = op->sub2 = NULL;
  op->elem.op = PUSHNUM;
  op->elem.u.pushnum.number = d;
  expr_free_dat(op);
  return op;
}

void yyerror(char *errorstr)
{
  error_return = errorstr;
}
 
%}

%union {
  char *string;
  struct expr_tree *expression;
  double number;
};

%token <string> IDENT
%token <number> NUMBER

%left '+' '-'
%left MINUS
%left '/' '*'

%type <expression> expr

%%

 totexpression: expr { expr_parsed = $1; } ;

 expr:
    '(' expr ')' { $$ = $2; }
  | expr '+' expr { $$ = expr_parse_binop('+', $1, $3); }
  | expr '-' expr { $$ = expr_parse_binop('-', $1, $3); }
  | expr '*' expr { $$ = expr_parse_binop('*', $1, $3); }
  | expr '/' expr { $$ = expr_parse_binop('/', $1, $3); }
  | '-' expr %prec MINUS { $$ = expr_parse_monop('-', $2); }
  | IDENT '(' expr ')' { $$ = expr_parse_monfunc(lookup_monfunc($1), $3); }
  | IDENT '(' expr ',' expr ')' { $$ = expr_parse_binfunc(lookup_binfunc($1), $3, $5); }
  | IDENT { $$ = expr_parse_var($1); }
  | NUMBER { $$ = expr_parse_num($1); }
  ;

  
%%
/* more c */
