/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Pat Hanrahan, Stuart Levy, Tamara Munzner, Mark Phillips */

#
/*
**	hline3.c - procedural interface to 3D line geometry
**
**	pat hanrahan
*/

#include <stdio.h>
#include <math.h>
#include "hg4.h"
#include "hpoint3.h"
#include "hplane3.h"
#include "hline3.h"
#include "transform3.h"
#include "ooglutil.h"

HLine3 *
HLn3Create()
{
    return OOGLNewE(HLine3, "HLine3");
}

void
HLn3Delete( ln )
    HLine3 *ln;
{
    OOGLFree(ln);
}

void
HLn3Print( ln )
    HLine3 *ln;
{
    printf( "ln(%s) \n", ln->type == HLN3_POINT_FORM ? "pt" : "pl" );
    Hg4Print2( ln->L );
}

void
HLn3Copy( ln1, ln2 )
    HLine3 *ln1, *ln2;
{
    memcpy( (char *)ln2, (char *)ln1, sizeof(HLine3) );
}

/*
**	Form the anti-symmetric line tensor from two points.
**
**	Note: pt * L = pl.
**	(1) pl is the plane formed from the line and the pt.
**	(2) each col of L is a plane containing the line.
**	(3) if pl is identically 0 then the pt lies on the line.
*/
int
HLn3From2HPt3s( ln, pt1, pt2 )
    HLine3  *ln;
    HPoint3 *pt1, *pt2;
{
    ln->type = HLN3_POINT_FORM;
    Hg4AntiProductPiQj( ln->L, pt1, pt2 );

    return Hg4Undefined2( ln->L );
}

/*
**	Form the anti-symmetric line tensor from two planes.
**
**	Note: K * pl = pt.
**	(1) pt is the point formed from the line and the pl.
**	(2) each row of K is a point on the line.
**	(3) if pt is identically 0 then the pl lies on the line.
*/
int
HLn3From2HPl3s( ln, pl1, pl2 )
    HLine3  *ln;
    HPlane3 *pl1, *pl2;
{
    ln->type = HLN3_PLANE_FORM;
    Hg4AntiProductPiQj( ln->L, pl1, pl2 );

    return Hg4Undefined2( ln->L );
}

int
HLn3IntersectHPl3( HLine3 *ln, HPlane3 *pl, HPoint3 *pt )
{
    HLine3 aln;

    if( ln->type == HLN3_PLANE_FORM ) {
	HLn3Dual( ln, &aln );
	ln = &aln;
    }
    return Hg4Intersect2( ln->L, pl, pt );
}

int
HLn3IntersectHPt3( HLine3 *ln, HPoint3 *pt, HPlane3 *pl )
{
    HLine3 aln;

    if( ln->type == HLN3_POINT_FORM ) {
	HLn3Dual( ln, &aln );
	ln = &aln;
    }
    return Hg4Intersect2( ln->L, pt, pl );
}


/*
**	predicate which tests for 3d line intersection and
**	if an intersection is found returns the point at which th
**	two lines cross and the plane in which the two lines lie.
**	
**	Note: One of the lines should be in the "plane-form" and the
**	other in the "point-form."
*/
int
HLn3IntersectHLn3( HLine3 *ln1, HLine3 *ln2, HPlane3 *pl, HPoint3 *pt )
{
    HLine3 ln;

    if( ln1->type == ln2->type ) {
	Hg4Dual( ln2, &ln );
	ln2 = &ln;
    }

    if( ln1->type == HLN3_POINT_FORM ) 
	return Hg4Intersect4( ln1->L, ln2->L, pt, pl );
    else
	return Hg4Intersect4( ln1->L, ln2->L, pl, pt );
}

int
HLn3Undefined( HLine3 *ln )
{
    return Hg4Undefined2( ln->L );
}

int
HLn3Infinity( HLine3 *ln )
{
    return Hg4Infinity2( ln->L, ln->type == HLN3_PLANE_FORM );
}

int
HLn3Compare( HLine3 *ln1, HLine3 *ln2 )
{
    /* Do the types have to agree? */
    return Hg4Compare2( ln1->L, ln2->L );
}

int 
HLn3CoincidentHPt3( HLine3 *ln, HPoint3 *pt )
{
    HPlane3 pl;

    return HLn3IntersectHPt3( ln, pt, &pl );
}

int
HLn3CoincidentHLn3( HLine3 *ln1, HLine3 *ln2 )
{
    HPlane3 pl;
    HPoint3 pt;

    if( HLn3IntersectHLn3( ln1, ln2, &pl, &pt ) )
	return HPl3Undefined( &pl );
    return 0;
}

int 
HLn3CoincidentHPl3( HLine3 *ln, HPlane3 *pl )
{
    HPoint3 pt;

    return HLn3IntersectHPl3( ln, pl, &pt );
}


void
HLn3Transform( Transform3 T, HLine3 *ln1, HLine3 *ln2 )
{
    /* Assume T is a point transform */
    if( ln1->type == HLN3_PLANE_FORM ) {
	HLn3Dual( ln1, ln2 );
	Hg4Transform2( T, ln2->L, ln2->L );
	HLn3Dual( ln2, ln2 );
    }
    else {
	Hg4Transform2( T, ln1->L, ln2->L );
	ln2->type = ln1->type;
    }
}

/*
**	Convert the matrix formed from planes/points to that formed
**	from points/planes.
**
**	Note:
**	(1) pu-qt+sr=0.
**	(2) (s,q,p,0) is parallel to the line.
**	(3) (-r,t,-u,0) is a plane containing the line and the origin.
**	(4) K*L = 0.
*/
void
HLn3Dual( HLine3 *ln, HLine3 *lndual )
{
    Hg4Dual( ln->L, lndual->L );
    lndual->type = HLN3_DUAL_FORM(ln->type);
}

void
HLn3Perp( HLine3 *ln, HLine3 *lnperp )
{
    HPoint3 pt;
    HPlane3 pl;

    /* LnIntersectPl( ln, &HPl3Ideal, &pt ); */
    HPt3Copy( (HPoint3 *)ln->L[3], &pt );
    HPt3Dual( &pt, &pl );
    HLn3From2HPl3s( lnperp, &pl, &HPl3Ideal );
}
