/*
   Egon Animator
   Copyright (C) 1997-1999  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/* ---
      cmds.c

      This rather bulky module contains all the functions that implement
      commands.  It also handles initialization of the interface to those
      functions.
--- */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#include "../common/common.h"
#include "../common/cmalloc.h"
#include "../common/plugin.h"
#include "../siod/siod.h"

#include "egon.h"
#include "user_interface.h"

typedef struct {
	char *name;
	 LISP(*function) ();
} s_fn_table;

extern s_fn_table fn_table[];

/* moving around */

static char *testlist[] = {	"testitem0",
	"testitem1", "testitem2", "testitem3", "testitem4", "testitem5",
	"testitem6", "testitem7", "testitem8", "testitem9"
};

/* ---
*/
static LISP listsel_test(void)
{
	int i = select_from_list("Choose an item", testlist, 10);
	if (i < 0)
		printf("User clicked Cancel\n");
	else
		printf("User chose %d\n", i);
	return NIL;
}

/* ---
*/
static LISP what_cursor_position(void)
{
	return NIL;
}

/* ---
*/
static LISP spawn(LISP command)
{
	char *argv[20], *p, cmd[1024];
	int argc = 0;

	strcpy(cmd, get_c_string(command));
	for (p = strtok(cmd, " \t\r\n");
		p && argc < 20;
		p = strtok(NULL, " \t\r\n")) {
		argv[argc++] = p;
	}
	argv[argc] = NULL;
	if (!fork()) {
		/* this is the child */
		execvp(argv[0], argv);
		exit(0);
	}
	return NIL;
}

/* Windows and buffers */

/* ---
*/
static LISP ask_for_file(void)
{
	static char path[1024], name[1024];
	char fn[1024];
	char fmt[80];
	if (path[0] == '\0')
		getcwd(path, 1024);
	name[0] = fn[0] = '\0';
	if (!select_file(path, name, NULL, fmt))
		return NIL;
	sprintf(fn, "%s/%s", path, name);
	return strcons(strlen(fn), fn);
}

/* ---
   Load a buffer from file.
*/

static LISP load_buffer(void)
{
	static char path[1024], name[1024];
	char fn[1024];
	char fmt[80];
	buffer *b;
	static int need_init = 1;

	if (need_init) {
		getcwd(path, 1024);
		need_init = 0;
	}
	strcpy(name, "");
	fn[0] = '\0';
	if (select_file(path, name, loader_patterns, fmt)) {
		sprintf(fn, "%s/%s", path, name);
		b = new_buffer(buffer_name(fn), fn);
		llpr("Loading");

		w_list->buf = b;
		activate_window(w_list);
	/* loadmatrix doesn't use the buffer argument in Egon,
	   the Scheme code is simply loaded and executed */
		if (loadmatrix(fn, b, fmt))
			llpr("New file");
	/* this will fail if load returns 1 because the string pool is full */

		b->change = FALSE;
		w_list->buf = b;
		pr_scr_flag = TRUE;
	}
	activate_window(w_list);
	return NIL;
}

/* ---
Load using an external program

   1. Ask for the program to use
   2. Run the program and save output to file in /tmp
   3. Load the file using NULL format (i.e. ask for type)
*/

static LISP load_external(void)
{
	static int loaders = 0;
	static char *loadname[20], *loadprog[20];
	static int need_init = 1;
	char program[256], param[256], fn[80], cmd[256];
	buffer *b;
	int i;

	if (need_init) {
		FILE *fp;
		char fnl[1024];
		char *p, *q, b[256];

		sprintf(fnl, "%s/egon/external.load", datadir);
		if ((fp = fopen(fnl, "r")) == NULL) {
			llpr("Can't open header file");
			return NIL;
		}
		while (!feof(fp) && loaders < 20) {
			fgets(b, 250, fp);
			if ((p = strtok(b, ":")) && (q = strtok(NULL, "\n"))) {
				loadname[loaders] = cstrdup(p);
				loadprog[loaders] = cstrdup(q);
				loaders++;
			}
		}
		fclose(fp);
		need_init = 0;
	}
	program[0] = param[0] = '\0';
	i = select_from_list("External Program:", loadname, loaders);

	if (i >= 0 && ask_for_str("Parameters:", param)) {
		sprintf(fn, "/tmp/egon%ld", (long)getpid());
		sprintf(cmd, "%s %s > %s", loadprog[i], param, fn);
		if (system(cmd)) {
			llpr("External program failed");
			return NIL;
		}

		b = new_buffer(buffer_name(fn), fn);
		llpr("Loading");

		if (loadmatrix(fn, b, NULL))
			llpr("New file");

		b->change = FALSE;
		w_list->buf = b;
		pr_scr_flag = TRUE;
	}
	activate_window(w_list);
	return NIL;
}

/* ---
*/
static LISP delete_window(void)
{
	if (!remove_window(w_list))
		llpr("Attempt to delete sole ordinary window");
	else
		pr_scr_flag = TRUE;
	activate_window(w_list);
	return NIL;
}

/* ---
*/
static LISP delete_other_windows(void)
{
	while (remove_window(w_list->next));
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP split_window_vertically(void)
{
	if (!split_window(w_list))
		llpr("This window is too small to split");
	else
		pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP other_window(void)
{
	activate_window(w_list->next);
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
   Save the buffer in the currently active window to file.
*/

static LISP save_buffer(void)
{
	llpr("Saving");

	if (savematrix(w_list->buf->path, w_list->buf, NULL))
		error_box("Couldn't save");
	else {
		w_list->buf->change = FALSE;
		pr_scr_flag = TRUE;
	}
	return NIL;
}

/* ---
   Save the buffer in the currently active window to a named file.
*/

static LISP save_buffer_as(void)
{
	static char path[1024], name[1024];
	char fn[1024];
	char fmt[80];
	char *p;
	static int need_init = 1;

	if (need_init) {
		getcwd(path, 1024);
		need_init = 0;
	}
	p = strrchr(w_list->buf->path, '/');
	if (p) strcpy(name, p+1);
	else strcpy(name, w_list->buf->path);
	fn[0] = '\0';
	if (select_file(path, name, saver_patterns, fmt)) {
		sprintf(fn, "%s/%s", path, name);
		llpr("Saving");
		if (savematrix(fn, w_list->buf, fmt))
			error_box("Couldn't save");
		else {
			w_list->buf->change = FALSE;
			strcpy(w_list->buf->path, fn);
		}
		pr_scr_flag = TRUE;
	}
	return NIL;
}

/* ---
Save using an external program

   1. Ask for the program to use
   2. Save to a file in /tmp using NULL format (i.e. ask for type)
   3. Run the program and read the file as input
*/

static LISP save_external(void)
{
	static int savers = 0;
	static char *savename[20], *saveprog[20];
	static int need_init = 1;
	char program[256], param[256], fn[80], cmd[256];
	int i;

	if (need_init) {
		FILE *fp;
		char fnl[1024];
		char *p, *q, b[256];

		sprintf(fnl, "%s/egon/external.save", datadir);
		if ((fp = fopen(fnl, "r")) == NULL) {
			llpr("Can't open saver file");
			return NIL;
		}
		while (!feof(fp) && savers < 20) {
			fgets(b, 250, fp);
			if ((p = strtok(b, ":")) && (q = strtok(NULL, "\n"))) {
				savename[savers] = cstrdup(p);
				saveprog[savers] = cstrdup(q);
				savers++;
			}
		}
		fclose(fp);
		need_init = 0;
	}

	program[0] = param[0] = '\0';
	i = select_from_list("External Program:", savename, savers);

	if (i >= 0 && ask_for_str("Parameters:", param)) {
		llpr("Saving");

		sprintf(fn, "/tmp/egon%ld", (long)getpid());
		if (savematrix(fn, w_list->buf, NULL)) {
			error_box("Couldn't save");
			return NIL;
		}
		sprintf(cmd, "%s %s < %s", program, param, fn);
		if (system(cmd)) {
			llpr("External program failed");
			return NIL;
		}
	}
	return NIL;
}

/* ---
*/
static LISP switch_to_buffer(void)
{
	buffer *b;
	char *blist[100];
	int nblist = 0, n;

	b = w_list->buf;
	do {
		b = b->next;
		blist[nblist++] = b->name;
	} while (b != w_list->buf);
	if ((n = select_from_list("Change Buffer:", blist, nblist)) >= 0)
		w_list->buf = find_buffer_by_name(blist[n]);
	activate_window(w_list);
	return NIL;
}

/* ---
*/
static LISP kill_buffer(void)
{
	buffer *b, *next_b;
	window *w;
	char *blist[100];
	int nblist = 0, n;

	b = w_list->buf;
	do {
		b = b->next;
		blist[nblist++] = b->name;
	} while (b != w_list->buf);
	if ((n = select_from_list("Kill Buffer:", blist, nblist)) >= 0) {
		if ((b = find_buffer_by_name(blist[n])) != NULL) {
			if (b != b->next) {
				next_b = free_buffer(b);
				w = w_list;
				do {
					if (w->buf == b)
						w->buf = next_b;
					w = w->next;
				} while (w != NULL && w != w_list);
				pr_scr_flag = TRUE;
			}
			else llpr("Couldn't kill last buffer");
		}
	}
	activate_window(w_list);
	return NIL;
}

/* ---
*/
static LISP preview_block(void)
{
	preview();
	return NIL;
}

/* ---
*/
static LISP print_block(void)
{
	int nr, nk;

	nr = 0;
	nk = 0;
	printer();
	return NIL;
}

/* ---
*/
static LISP execute_interpreter_command(LISP intpr)
{
	char prompt[80];
	char b[256];
	char *intname = get_c_string(intpr);
	int intp = name2interpreter(intname);
	if (intp < 0) return NIL;
	sprintf(prompt, "%s command: ", intname);
	b[0] = '\0';
	if (ask_for_str(prompt, b))
		exec_expr(intp, b);
	return NIL;
}

static char *quit_buttons[] = {"Yes", "No", "Cancel"};

/* ---
*/
static LISP lalertbox(LISP prompt, LISP buttons)
{
	char *btext[10], *p = get_c_string(prompt);
	int bno = 0, n;

	while (bno < 10 && NNULLP(buttons)) {
		char *c = get_c_string(car(buttons));
		if (c) btext[bno] = cstrdup(c);
		else btext[bno] = cstrdup("button");
		buttons = cdr(buttons);
		bno++;
	}
	if (p == NULL) p = "prompt";
	n = alert_box(p, btext, bno);
	while (bno) cfree(btext[--bno]);
	return flocons(n);
}

/* ---
*/
static LISP quit_egon(void)
{
	char prompt[256], cmd[1024];
	buffer *b = b_list;
	do {
		if (b->change) {
			sprintf(prompt, "Save %s?", b->name);
			switch (alert_box(prompt, quit_buttons, 3)) {
			case 0:
				savematrix(b->path, b, NULL);
				break;
			case 2:
				return NIL;
			default:
				break;
			}
		}
		b = b->next;
	} while (b != b_list);
        /* remove all windows, buffers and plugins */
        exit_windows();
        /* remove the temp directory for this process */
        sprintf(cmd, "rm -rf %s/%ld", siag_basedir, (long)getpid());
        system(cmd);

	exit(0);
	return NIL;
}

/* ---
*/
static LISP print_version(void)
{
	llpr(version);
	return NIL;
}

/* ---
*/
static LISP buffer_changed(LISP bname)
{
	buffer *buf;

	if (NULLP(bname)) buf = w_list->buf;
	else buf = find_buffer_by_name(get_c_string(bname));

	if (buf) buf->change = TRUE;
	return NIL;
}

/* ---
*/
static LISP set_format(LISP format)
{
	ins_format(w_list->buf, w_list->object, w_list->script,
			get_c_long(format));
	return NIL;
}

/* ---
*/
static LISP get_format(void)
{
	long fmt = ret_format(w_list->buf, w_list->object, w_list->script);
	return flocons(fmt);
}

/* These functions allow implementation of commands in Scheme
	rather than in C with Scheme wrappers */

/* ---
*/
static LISP bell(void)
{
#if 0
	XBell(XtDisplay(pLevel), 100);
#endif
	return NIL;
}

/* ---
*/
static LISP lask_for_str(LISP prompt, LISP buf)
{
	char *p, b[256];

	strcpy(b, get_c_string(buf));
	p = get_c_string(prompt);
	if (ask_for_str(p, b))
		return strcons(strlen(b), b);
	else
		return NIL;
}

/* ---
*/
static LISP lexecute(LISP cmd)
{
	execute(get_c_string(cmd));
	return NIL;
}

/* ---
*/
static LISP linput_warp_pointer(LISP value)
{
	input_warp_pointer = get_c_long(value);
	return NIL;
}

#define MAGIC_MARKER ";;; Do not change or add anything below this line.\n"

/* ---
*/
static LISP save_preferences(void)
{
        char fn1[1024], fn2[1024];
        FILE *fp1, *fp2;
        char b[1024];

        sprintf(fn1, "%s/.siag/egon.scm", getenv("HOME"));
        sprintf(fn2, "%s.BAK", fn1);
        rename(fn1, fn2);
        fp2 = fopen(fn2, "r");
        fp1 = fopen(fn1, "w");
        if (!fp1) {
                rename(fn2, fn1);
                return NIL;
        }
        while (fp2 && fgets(b, sizeof b, fp2)) {
                if (!strcmp(b, MAGIC_MARKER)) break;
                fputs(b, fp1);
        }
        fputs(MAGIC_MARKER, fp1);
        fprintf(fp1, "(tooltip-mode %ld)\n",
                get_c_long(symbol_value(cintern("*tooltip-mode*"), NIL)));

        if (fp2) fclose(fp2);
        fclose(fp1);
        return NIL;
}

/* Beginning of animation functions */

/* ---
*/
static LISP ani_c_test(void)
{
	return NIL;
}

/* ---
*/
static LISP ani_begin(void)
{
	buffer *b = w_list->buf;
	/* should free old objects first */
	b->cast = NULL;
	b->delta = 100;
	b->duration = 1000;
	b->width = 600;
	b->height = 400;
	b->bg = NULL;
	b->change = TRUE;
	w_list->object = NULL;
	w_list->script = NULL;
	pr_scr_flag = TRUE;
	return NIL;
}


/* ---
*/
static LISP select_object(LISP name)
{
	char *string = get_c_string(name);

	ani_object *o = w_list->buf->cast;

	while (o) {
		if (!strcmp(o->name, string)) break;
		o = o->next;
	}
	if (!o) {
		llpr("No such object");
	} else {
		w_list->object = o;
		w_list->script = o->script;
	}
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP select_tick(LISP tick)
{
	ani_script *s;
	int t = get_c_long(tick);
	if (!w_list->object) {
		llpr("No objects!");
		return NIL;
	}
	s = w_list->object->script;
	while (s) {
		if (s->time == t) break;
		s = s->next;
	}
	if (!s) {
		llpr("No such tick");
	} else {
		w_list->script = s;
	}
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP lani_object(LISP obj)
{
	buffer *b = w_list->buf;
	ani_object *lasto = w_list->object, *newo;
	ani_script *lasts;
	int t = get_c_long(obj);

	newo = (ani_object *)cmalloc(sizeof(ani_object));
	if (lasto) {	/* put newo after lasto */
		newo->next = lasto->next;
		lasto->next = newo;
	} else {	/* put newo first in object list */
		b->cast = newo;
		newo->next = NULL;
	}
	w_list->object = newo;
	newo->type = t;
	newo->name = type2name(t);
	lasts = newo->script = (ani_script *)cmalloc(sizeof(ani_script));
	lasts->x = 0;
	lasts->y = 0;
	lasts->width = 0;
	lasts->height = 0;
	lasts->visible = 1;
	lasts->time = 0;
	newo->fmt = 0;
	newo->string = NULL;
	lasts->next = NULL;
	w_list->script = lasts;
	b->change = TRUE;
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP set_type(LISP type)
{
	int t = get_c_long(type);

	if (!w_list->object) {
		llpr("No object selected");
	} else {
		w_list->object->type = t;
		cfree(w_list->object->name);
		w_list->object->name = type2name(t);
	}
	w_list->buf->change = TRUE;
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
FIXME: these two should free the memory used
*/

static LISP del_object(void)
{
	buffer *b = w_list->buf;
	ani_object *o = w_list->object;
	ani_object *o1 = w_list->buf->cast;

	if (!o) {
		llpr("None selected");
		return NIL;
	}
	if (o == o1) {	/* first object */
		w_list->buf->cast = o->next;
		w_list->object = o->next;
	} else {	/* some other object; find the one before */
		while (o1->next != o) o1 = o1->next;
		o1->next = o->next;
		w_list->object = o1;	/* select previous object */
	}
	if (w_list->object)
		w_list->script = w_list->object->script;
	else
		w_list->script = NULL;
	b->change = TRUE;
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP del_time(void)
{
	buffer *b = w_list->buf;
	ani_object *o = w_list->object;
	ani_script *s, *s1;

	if (!o) {
		llpr("None selected");
		return NIL;
	}
	s = w_list->script;
	s1 = o->script;
	if (s->time == 0) {
		llpr("Can't delete tick 0");
		return NIL;
	}
	while (s1->next != s) s1 = s1->next;
	s1->next = s->next;
	w_list->script = s1;
	b->change = TRUE;
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP ani_background(LISP bgp)
{
	buffer *b = w_list->buf;
	if (b->bg) cfree(b->bg);
	b->bg = cstrdup(get_c_string(bgp));
	b->change = TRUE;
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP ani_time(LISP tim)
{
	ani_object *o = w_list->object;
	ani_script *s, *news;
	unsigned int t = get_c_long(tim);
	
	if (!o) return NIL;

	/* find the place to put new tick after */
	s = o->script;
	while (s->next && s->next->time <= t)
		s = s->next;

	if (s->time == t) return NIL;	/* dup tick */

	/* simply copy the last tick, but with a new time */
	news = (ani_script *)cmalloc(sizeof(ani_script));
	*news = *s;
	news->next = s->next;
	news->time = t;
	s->next = news;
	w_list->script = news;
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP lani_duration(LISP tim)
{
	buffer *b = w_list->buf;
	b->duration = get_c_long(tim);
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP lani_delta(LISP tim)
{
	buffer *b = w_list->buf;
	b->delta = get_c_long(tim);
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP lani_geometry(LISP w, LISP h)
{
	buffer *b = w_list->buf;
	b->width = get_c_long(w);
	b->height = get_c_long(h);
	pr_scr_flag = TRUE;
	return NIL;
}

/* ---
*/
static LISP ani_end(void)
{
/*
	buffer *b = w_list->buf;
	ani_show(b->cast, 0L, b->delta, b->duration, b->width, b->height,
		b->bg, b->path);
*/
	return NIL;
}


/* ---
*/
static LISP lregister_x_font(LISP name, LISP x, LISP xi, LISP xb, LISP xbi)
{
	return flocons(register_x_font(get_c_string(name),
					get_c_string(x),
					get_c_string(xi),
					get_c_string(xb),
					get_c_string(xbi)));
}

/* ---
*/
static LISP lregister_ps_font(LISP name, LISP x, LISP xi, LISP xb, LISP xbi)
{
	return flocons(register_ps_font(get_c_string(name),
					get_c_string(x),
					get_c_string(xi),
					get_c_string(xb),
					get_c_string(xbi)));
}

/* ---
*/
static LISP lregister_t1_font(LISP name, LISP x, LISP xi, LISP xb, LISP xbi)
{
	return flocons(register_t1_font(get_c_string(name),
					get_c_string(x),
					get_c_string(xi),
					get_c_string(xb),
					get_c_string(xbi)));
}

/* ---
*/
static LISP lfont_alias(LISP alias, LISP name)
{
	return flocons(font_alias(get_c_string(alias),
				get_c_string(name)));
}

/* ---
*/
static LISP lregister_color(LISP name, LISP r, LISP g, LISP b)
{
	return flocons(register_color(get_c_string(name),
					get_c_long(r),
					get_c_long(g),
					get_c_long(b)));
}


/* ---
*/
static LISP labort(void)
{
	abort();
	return NIL;
}

/* ---
Takes an associative list of the format
((name value) (name value) ... )
and turns it into an index into the format table.
*/
static LISP lencode_format(LISP f)
{
	sfmt fmt;
	int result;

	decode_format(0, ~0, &fmt);	/* put valid fields in fmt */
	while (NNULLP(f)) {
		char *name = get_c_string(CAR(CAR(f)));
		int mask = fmt_attr2mask(name);
		LISP value = CDR(CAR(f));
		f = CDR(f);
		switch (mask) {
		case SFMT_FAMILY:
			fmt.family = get_c_string(value);
			break;
		case SFMT_SIZE:
			fmt.size = get_c_long(value);
			break;
		case SFMT_FG:
			fmt.fg = get_c_string(value);
			break;
		case SFMT_BG:
			fmt.bg = get_c_string(value);
			break;
		case SFMT_BOLD:
			fmt.bold = get_c_long(value);
			break;
		case SFMT_ITALIC:
			fmt.italic = get_c_long(value);
			break;
		case SFMT_ULINE:
			fmt.uline = get_c_long(value);
			break;
		case SFMT_HADJ:
			fmt.hadj = get_c_long(value);
			break;
		case SFMT_VADJ:
			fmt.vadj = get_c_long(value);
			break;
		case SFMT_STYLE:
			fmt.style = name2type(get_c_string(value));
			break;
		default:
			break;
		}
		
	}
	result = encode_format(~0, &fmt);
	return flocons(result);
}

static LISP SS(char *name, char *value)
{
	return cons(strcons(strlen(name), name),
		    strcons(strlen(value), value));
}

static LISP SL(char *name, int value)
{
	return cons(strcons(strlen(name), name),
		    flocons(value));
}

/* ---
Takes an index into the format table and turns it into an
associative list of the format
((name value) (name value) ... )
*/
static LISP ldecode_format(LISP f)
{
	sfmt fmt;

	decode_format(get_c_long(f), ~0, &fmt);
	return cons(SS("family", fmt.family),
		 cons(SL("size", fmt.size),
		   cons(SS("fg", fmt.fg),
		     cons(SS("bg", fmt.bg),
		       cons(SL("bold", fmt.bold),
			 cons(SL("italic", fmt.italic),
			   cons(SL("uline", fmt.uline),
			     cons(SL("hadj", fmt.hadj),
			       cons(SL("vadj", fmt.vadj),
				 cons(SS("style", type2name(fmt.style)),
				      NIL))))))))));
}

/* ---
*/
int egon_plugin_save(int ph, char *fn)
{
	return plugin_save(ph, fn);
}

static LISP lplugin_register(LISP desc, LISP ext, LISP cmd)
{
        plugin_register(get_c_string(desc),
                        get_c_string(ext),
                        get_c_string(cmd));
        return NIL;
}

static void insert_plugin(int mode, char *plugin_name)
{
        static char path[1024], name[1024];
        char fn[1024];
        char pn[1024];
        char cmd[1024];
        char fmt[80];
        char *p;
        buffer *buf = w_list->buf;
        int n;
        plugin_t plugin;

        /* ask for file name */
        if (path[0] == '\0') getcwd(path, 1024);
        name[0] = fn[0] = '\0';

        if (plugin_name) {
                char *q = strrchr(plugin_name, '/');
                if (q) strcpy(name, q+1);
                else strcpy(name, plugin_name);
                strcpy(fn, plugin_name);
        } else {
	        if (!select_file(path, name, plugin_patterns, fmt)) return;
        	sprintf(fn, "%s/%s", path, name);
	}
        plugin_unique_name(name, pn);

        plugin.row = 1;
        plugin.col = 1;
        plugin.displayed = 0;

        if (mode == PLUGIN_COPY) {
                plugin.name = cstrdup(pn);
                /* copy the file */
                p = plugin_basedir(buf, NULL);
                sprintf(pn, "%s/%s", p, plugin.name);
                sprintf(cmd, "(mkdir %s;cp %s %s)2>/dev/null", p, fn, pn);
                system(cmd);
        } else {
                strcpy(pn, fn);
                plugin.name = cstrdup(pn);
        }

        /* start the plugin */
        plugin.ph = plugin_start(pn);
        if (plugin.ph == -1) return;

        /* prepare the buffer */
        n = buf->nplugin++;
        buf->plugin = (plugin_t *)crealloc(buf->plugin,
                buf->nplugin*sizeof(plugin_t));
        buf->plugin[n] = plugin;
        buf->change = TRUE;
        pr_scr_flag = TRUE;
}

static LISP lplugin_import(LISP pn)
{
        if (NULLP(pn)) insert_plugin(PLUGIN_COPY, NULL);
        else insert_plugin(PLUGIN_COPY, get_c_string(pn));
        return NIL;
}

/* ---
broken because it still takes the name relative to basedir
*/

static LISP lplugin_link(LISP pn)
{
        if (NULLP(pn)) insert_plugin(PLUGIN_LINK, NULL);
        else insert_plugin(PLUGIN_LINK, get_c_string(pn));
        return NIL;
}

/* ---
return buffer's plug handler index or -1 for error
*/

static int select_plugin(buffer *b)
{
        char *plugin[20];
        int i;

        if (b->nplugin == 0) return -1;
        if (b->nplugin == 1) return 0;
        for (i = 0; i < 20 && i < b->nplugin; i++)
                plugin[i] = b->plugin[i].name;
        i = select_from_list("Select Plugin:", plugin, i);
        return i;
}

static LISP lplugin_select(void)
{
	return flocons(select_plugin(w_list->buf));
}

static LISP lplugin_export(void)
{
        int n;
        char fn[1024], pn[1024], cmd[1024], path[1024], fmt[80];
        buffer *buf = w_list->buf;

        /* pick plugin from list */
        n = select_plugin(buf);
        if (n == -1) return NIL;

        /* get description of the handler */
        /* ask for file name */
        getcwd(path, 1024);
        strcpy(fn, buf->plugin[n].name);
        if (!select_file(path, fn, NULL, fmt)) return NIL;

        /* save the file */
        sprintf(pn, "%s/%s", plugin_basedir(buf, NULL), buf->plugin[n].name);
        sprintf(cmd, "cp %s %s/%s", pn, path, fn);
        system(cmd);
        return NIL;
}

static LISP lplugin_delete(void)
{
        int n;
        buffer *buf = w_list->buf;

        /* pick plugin from list */
        n = select_plugin(buf);
        if (n == -1) return NIL;

        /* remove plugin */
        plugin_stop(buf->plugin[n].ph);
        cfree(buf->plugin[n].name);
        buf->nplugin--;
        for (; n < buf->nplugin; n++)
                buf->plugin[n] = buf->plugin[n+1];
        buf->change = pr_scr_flag = TRUE;
        return NIL;
}

static LISP lplugin_move(void)
{
        int n;
        buffer *buf = w_list->buf;

        /* pick plugin from list */
        n = select_plugin(buf);
        if (n == -1) return NIL;

        buf->plugin[n].row = 1;
        buf->plugin[n].col = 1;
        buf->change = pr_scr_flag = TRUE;
        return NIL;
}

static LISP lmake_backups(LISP p)
{
	make_backups = get_c_long(p);
	return NIL;
}

/* Commands that take no arguments */
s_fn_table fn_table[] =
{
	{"listsel-test", listsel_test},

	/* moving around */
	{"what-cursor-position", what_cursor_position},

	/* editing */

	/* block commands */
	{"preview", preview_block},
	{"print-block", print_block},

	/* new window */
	{"delete-window", delete_window},
	{"delete-other-windows", delete_other_windows},
	{"split-window-vertically", split_window_vertically},
	{"other-window", other_window},

	/* buffers and windows */
	{"switch-to-buffer", switch_to_buffer},
	{"kill-buffer", kill_buffer},
	{"ask-for-file", ask_for_file},
	{"load-buffer", load_buffer},
	{"save-buffer", save_buffer},
	{"save-buffer-as", save_buffer_as},
	{"load-external", load_external},
	{"save-external", save_external},

	/* help commands */
	{"print-version", print_version},

	{"quit-program", quit_egon},
	/* low level functions */
	{"bell", bell},

	{"ani-begin", ani_begin},
	{"ani-end", ani_end},
	{"del-object", del_object},
	{"del-time", del_time},
	{"ani-c-test", ani_c_test},
	{"get-format", get_format},
	{"save-preferences", save_preferences},
	{"abort", labort},
	{(char *) 0, (LISP(*)())0}
};

/* Commands that take 1 argument */

s_fn_table fn_table1[] = {
	/* low level functions */
	{"spawn", spawn},
	{"execute-interpreter-command", execute_interpreter_command},
	{"buffer-changed", buffer_changed},
	{"execute", lexecute},
	{"input-warp-pointer", linput_warp_pointer},

	{"ani-object", lani_object},
	{"ani-background", ani_background},
	{"ani-time", ani_time},
	{"ani-duration", lani_duration},
	{"ani-delta", lani_delta},
	{"select-object", select_object},
	{"select-tick", select_tick},
	{"set-type", set_type},
	{"set-format", set_format},
	{"decode-format", ldecode_format},
	{"encode-format", lencode_format},
	{"make-backups", lmake_backups},
	{NULL, NULL}
};

/* Commands that take 2 arguments */
s_fn_table fn_table2[] = {
	{"ask-for-str", lask_for_str},
	{"alertbox", lalertbox},

	{"ani-geometry", lani_geometry},
	{"font-alias", lfont_alias},
	{NULL, NULL}
};

/* ---
Set up the table of functions and names
*/

void init_cmds(void)
{
	int i;

	for (i = 0; fn_table[i].name; i++)
		init_subr_0(fn_table[i].name, fn_table[i].function);
	for (i = 0; fn_table1[i].name; i++)
		init_subr_1(fn_table1[i].name, fn_table1[i].function);
	for (i = 0; fn_table2[i].name; i++)
		init_subr_2(fn_table2[i].name, fn_table2[i].function);
	init_subr_5("register-x-font", lregister_x_font);
	init_subr_5("register-ps-font", lregister_ps_font);
	init_subr_5("register-t1-font", lregister_t1_font);
	init_subr_4("register-color", lregister_color);
	init_subr_3("plugin-register", lplugin_register);
	init_subr_0("plugin-select", lplugin_select);
        init_subr_1("plugin-import", lplugin_import);
        init_subr_0("plugin-export", lplugin_export);
        init_subr_1("plugin-link", lplugin_link);
        init_subr_0("plugin-delete", lplugin_delete);
        init_subr_0("plugin-move", lplugin_move);
}

