/*
   Pathetic Writer
   Copyright (C) 1997-1999  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/* ---
   fileio_ps.c
 
   Produces data suitable for printing on a Postscript device. I have made
   several assumptions to simplify things:
   - 1 pixel = 1 point means no scaling is necessary
   - PS font has same geometry as X font
   - no background pattern
   - don't draw the grid
   - The lpr command is used for printing

   This works for me, but will probably break (more or less) for anyone
   not using my printer (a NEC S62P) with A4 paper.
--- */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "pw.h"
#include "../common/common.h"
#include "../common/cmalloc.h"
#include "../common/plugin.h"

extern int font_size[8];

char *psformat;

static int pageno;
static int ps_paper_width, ps_paper_height;

/* ---
Sizing the output won't be a problem here, except we don't print
all the allocated but unused lines
*/

static int page_height(buffer *buf, int s, int r)
{
	int y_base, i;

	y_base = ps_paper_height - buf->top_margin;
	for (i = r; y_base > buf->bottom_margin; i++)
		y_base -= cell_height(buf, s, i);
	return (i-r);
}

static int line_start(buffer *buf, int s, int row)
{
	int x = 0;
	int w = ps_paper_width-buf->left_margin-buf->right_margin;
	int adj = buf->sht[s].text[row].adj;

	if (adj == HADJ_CENTER)
		x = (w-line_width(buf, s, row, line_length(buf, s, row)))/2;
	else if (adj == HADJ_RIGHT)
		x = w-line_width(buf, s, row, line_length(buf, s, row));

	return x+buf->left_margin;
}

static void expand(FILE *fp, int x, int y, char *from, char *name, int number)
{
        char b[1024], c[1024];
        int i;
	time_t t;
	struct tm *lt;

        unsigned int text_height, text_width;

        /* set the font */
        ps_set_font(fp, 0);

        /* first make the real header/footer string */

	/* expand the time */
	time(&t);
	lt = localtime(&t);
	strftime(c, sizeof c, from, lt);

	from = c;	/* new format string */

        i = 0;
        while (*from && (i < sizeof b)) {
                if (*from == '&') {
                        from++;
                        if (*from == '\0') break;
			else switch (*from) {
			case 'n':
                                strcpy(b+i, name);
                                i += strlen(b+i);
				break;
			case 'p':
                                sprintf(b+i, "%d", number);
                                i += strlen(b+i);
				break;
			default:
                        	b[i++] = *from;
				break;
			}
                } else b[i++] = *from;
                from++;
        }
	b[i] = '\0';

        /* next center it around (x,y) */

        text_height = ps_font_height(0);
        text_width = ps_text_width(0, b);
        fprintf(fp, "newpath\n");
        fprintf(fp, "%d %d moveto\n", x - text_width/2, y - text_height/2);

        /* finally print the string */
        fprintf(fp, "(");
        for (i = 0; b[i]; i++) {
                int c = b[i] & 0xFF;
                if (isalnum(c))
                        putc(c, fp);
                else
                        fprintf(fp, "\\%03o", c);
        }
        fprintf(fp, ")A\n");
}

static void page_header(FILE *fp, buffer *buf)
{
	char *ps_page_header = buf->header;
	if (pageno == buf->first_page_number && buf->header_on_first == 0)
		return;

	if (!ps_page_header) ps_page_header = "&n";	/* buffer name */
	expand(fp, ps_paper_width/2,
		ps_paper_height-buf->header_margin,
                ps_page_header,
		buf->name, pageno);
}

static void page_footer(FILE *fp, buffer *buf)
{
        char *ps_page_footer = buf->footer;
	if (pageno == buf->first_page_number && buf->header_on_first == 0)
		return;

        if (!ps_page_footer) ps_page_footer = "&p";     /* page number */
        expand(fp, ps_paper_width/2,
		buf->footer_margin,
                ps_page_footer,
		buf->name, pageno);
}


static int print_page(FILE *fp, buffer *buf, int s, int fromr, int tor)
{
	int i, x_base, y_base, y_pos, col;
	rich_char *line;

	fprintf(fp, "gsave\n");
	if (buf->orientation == LANDSCAPE) {
		fprintf(fp, "90 rotate\n");
		fprintf(fp, "0 %d translate\n", -ps_paper_height);
	}

	page_header(fp, buf);
	page_footer(fp, buf);

	y_base = ps_paper_height - buf->top_margin;
	for (i = fromr;
	     i <= buf->sht[s].used_lines && y_base > buf->bottom_margin;
	     i++) {
		line = buf->sht[s].text[i].p;
		y_base -= cell_height(buf, s, i);
		x_base = line_start(buf, s, i);
		y_pos = 5;
		if (rc_strlen(line) == 0) continue;

		if (buf->sht[s].text[i].sty == STY_EMBED) {
			char *p = (char *)rc_makeplain(line);
			ps_embed_print(fp, p, x_base, y_base+y_pos);
			cfree(p);
			continue;	/* don't print the text */
		}

		fprintf(fp, "newpath\n");
		fprintf(fp, "%d %d moveto\n", x_base, y_base+y_pos);

		for (col = 0; line[col].c; col++) {
			int c = (unsigned char)(line[col].c);
			int fmt = line[col].fmt;
			ps_set_font(fp, fmt);

		/* print letters and digits as is, the rest in octal */
			if (isalnum(c)) {
				fprintf(fp, "(%c)A\n", c);
			} else if (c == '\t') {
				fprintf(fp, "%d %d moveto\n",
					x_base+line_width(buf, s, i, col+1),
					y_base+y_pos);
			} else {
				fprintf(fp, "(\\%03o)A\n", c);
			}
		}
	}

	/* print plugins here! */
	for (i = 0; i < buf->sht[s].nplugin; i++) {
		int x, y, topx, topy, boty, plx, ply;
		int width, height;
		buffer_global_coords(buf, s, fromr, 0, &topx, &topy);
		boty = topy+ps_paper_height-buf->top_margin-buf->bottom_margin;
		buffer_global_coords(buf, s, buf->sht[s].plugin[i].row,
                        buf->sht[s].plugin[i].col, &plx, &ply);
                plugin_size_get(buf->sht[s].plugin[i].ph, &width, &height);
                if (ply+height < topy || ply > boty) continue;
                x = buf->left_margin
			+(ps_paper_width
				-buf->left_margin
				-buf->right_margin-width)/2;
                y = ps_paper_height-buf->top_margin-(ply-topy)-height;
                /* and let the plugin do its thing */
                plugin_print(buf->sht[s].plugin[i].ph, fp, x, y, 1);
        }

	fprintf(fp, "grestore\n");
	fprintf(fp, "showpage\n");
	return 0;
}


/* slight mod: no need to calculate the width */
static int save(char *fn, buffer *buf)
{
	FILE *fp;
	int r;
	int pages;
	time_t t;
	int s;
	int fromr, tor;

	fp = fopen(fn, "w");

#if 0
	fprintf(fp, "%%!PS-Adobe-3.0 EPSF-3.0\n");
#else
	fprintf(fp, "%%!PS-Adobe-2.0\n");
#endif
	fprintf(fp, "%%%%Creator: %s\n", version);
	fprintf(fp, "%%%%Title: %s\n", buf->name);
	t = time(NULL);
	fprintf(fp, "%%%%CreationDate: %s", ctime(&t));
	fprintf(fp, "%%%%Pages: (atend)\n");
	fprintf(fp, "%%%%PageOrder: Ascend\n");
	fprintf(fp, "%%%%BoundingBox: %d %d %d %d\n",
		0, 0, buf->paper_width, buf->paper_height);
	if (buf->orientation == LANDSCAPE) {
		fprintf(fp, "%%%%Orientation: Landscape\n");
		ps_paper_width = buf->paper_height;
		ps_paper_height = buf->paper_width;
	} else {
		fprintf(fp, "%%%%Orientation: Portrait\n");
		ps_paper_width = buf->paper_width;
		ps_paper_height = buf->paper_height;
	}
	fprintf(fp, "%%%%DocumentPaperSizes: %s\n", buf->paper_name);
	fprintf(fp, "%%%%EndComments\n");

	/* Use ISO-Latin1 encoding */
	fprintf(fp, "%%%%BeginProlog\n");
	ps_makefonts(fp);

	fprintf(fp, "/A {show} def\n");

	fprintf(fp, "%%%%EndProlog\n");

	pages = 0;
	pageno = buf->first_page_number;
	for (s = 0; s < buf->nsht; s++) {
		fromr = 1;
		tor = line_last_used(buf, s);
		for (r = fromr; r <= tor; r += page_height(buf, s, r)) {
			pages++;
			ps_set_font(fp, -1);	/* force setfont on top of page */
			fprintf(fp, "%%%%Page: %d %d\n", pages, pages);
			print_page(fp, buf, s, r, tor);
			pageno++;
		}
	}
	fprintf(fp, "%%%%Trailer:\n");
	fprintf(fp, "%%%%Pages: %d\n", pages);
	fprintf(fp, "%%%%EOF\n");
	fclose(fp);
	return 0;
}

/* ---
Conservative file format guessing: always negative (can't load)
*/

static int myformat(char *fn)
{
	return 0;
}

/* ---
*/
void fileio_ps_init()
{
	register_format(NULL, save, myformat, psformat = "Postscript (*.ps)");
}

