/*
   Copyright (C) 1996, 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the Licence, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include "../common/cmalloc.h"
#include "../common/common.h"

#include "../common/fonts.h"
#include "xfonts.h"
#include <X11/xpm.h>

#include "AnimatorP.h"

static void plugin_coords(Widget, XtPointer, int *, int *);

#define offset(field) XtOffsetOf(AnimatorRec, animator.field)
static XtResource resources[] = {
	{
		XtNanimatorCast,	/* name */
		XtCAnimatorCast,	/* class */
		XtRPointer,		/* type */
		sizeof(XtPointer),	/* size */
		offset(cast),		/* offset */
		XtRImmediate,		/* default_type */
		(XtPointer)0		/* default_addr */
	}, {
		XtNanimatorNow,
		XtCAnimatorNow,
		XtRInt,
		sizeof(int),
		offset(now),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNanimatorDelta,
		XtCAnimatorDelta,
		XtRInt,
		sizeof(int),
		offset(delta),
		XtRImmediate,
		(XtPointer)100
	}, {
		XtNanimatorDuration,
		XtCAnimatorDuration,
		XtRInt,
		sizeof(int),
		offset(duration),
		XtRImmediate,
		(XtPointer)1000
	}, {
		XtNanimatorMode,
		XtCAnimatorMode,
		XtRInt,
		sizeof(int),
		offset(newmode),
		XtRImmediate,
		(XtPointer)ANI_STOP
	}, {
		XtNanimatorBgPixmap,
		XtCAnimatorBgPixmap,
		XtRString,
		sizeof(String),
		offset(bg_pixmap),
		XtRString,
		(XtPointer)0
	}, {
		XtNanimatorPluginCoords,
		XtCAnimatorPluginCoords,
		XtRPointer,
		sizeof(XtPointer),
		offset(plugin_coords),
		XtRImmediate,
		(XtPointer)plugin_coords
	}
};
#undef offset

/* methods */
static void DoLayout();
static void Resize();
static XtGeometryResult GeometryManager();
static void ChangeManaged();
static void Redisplay(Widget, XEvent *, Region);
static void Realize(Widget, XtValueMask *, XSetWindowAttributes *);
static void Destroy(Widget);
static Boolean SetValues(Widget, Widget, Widget, ArgList, Cardinal *);

static void plugin_coords(Widget w, XtPointer p, int *x, int *y)
{
        *x = *y = 0;
}

/* actions */
static void AnimatorAction(Widget, XEvent *, String *, Cardinal *);

static XtActionsRec actions[] =
{
	{"animator", AnimatorAction},
};

/* translations */
static char translations[] =
"<Key>Return:		animator(next)		\n\
<Btn1Down>:		animator(next)		\n\
:<Key>Tab:		animator(next)		\n\
<Key>N:			animator(next)		\n\
<Key>P:			animator(previous)	\n\
<Key>S:			animator(stop)		\n\
<Key>C:			animator(continue)	\n\
<Key>Escape:		animator(quit)		\n\
<Key>Q:			animator(quit)		\n\
";

AnimatorClassRec animatorClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &compositeClassRec,
    /* class_name		*/	"Animator",
    /* widget_size		*/	sizeof(AnimatorRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	NULL,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  }, {
    /* geometry_manager   */    GeometryManager,
    /* change_managed     */    ChangeManaged,
    /* insert_child       */    XtInheritInsertChild,
    /* delete_child       */    XtInheritDeleteChild,
    /* extension          */    NULL
  }, { /* animator fields */
    /* empty			*/	0
  }
};

WidgetClass animatorWidgetClass = (WidgetClass)&animatorClassRec;

/* supporting code copied directly from animator.c */

static void ani_stepper(XtPointer client_data, XtIntervalId *id)
{
	int ani_then, ani_now;
	AnimatorWidget aw = (AnimatorWidget) client_data;

	if (!XtIsRealized((Widget)aw)) return;

	/* cancel any pending timeout */
	/* this is tricky. if ani_stepper is called by us and not by X,
	   we must remove the timeout ourselves */
	if (!id && aw->animator.waiting) {
		XtRemoveTimeOut(aw->animator.timeout);
	}
	aw->animator.waiting = False;

	ani_then = ani_now = aw->animator.now;
	if (aw->animator.mode == ANI_NEXT || aw->animator.mode == ANI_CONTINUE)
		ani_now += aw->animator.delta;
	else if (aw->animator.mode == ANI_PREVIOUS)
		ani_now -= aw->animator.delta;
	if (ani_now < 0)
		ani_now = 0;
	if (ani_now > aw->animator.duration)
		ani_now = 0;
	if (ani_now != ani_then) {
		aw->animator.now = ani_now;
		Redisplay((Widget)aw, NULL, None);
	}
	if (aw->animator.mode == ANI_CONTINUE) {
		aw->animator.timeout = XtAppAddTimeOut(
				XtWidgetToApplicationContext((Widget)aw),
				aw->animator.delta, ani_stepper,
				(XtPointer)aw);
		aw->animator.waiting = True;
	}
}

static void ani_ctl(Widget w, int mode)
{
	AnimatorWidget aw = (AnimatorWidget)w;

	switch (mode) {
	case ANI_CONTINUE:
		if (aw->animator.mode == mode) break;
	case ANI_STOP:
	case ANI_NEXT:
	case ANI_PREVIOUS:
		aw->animator.mode = mode;
		ani_stepper((XtPointer)w, NULL);
		break;
	case ANI_QUIT:
		/* can't really do that from here, can we? */
	default:
		break;
	}
}

static void AnimatorAction(Widget w, XEvent *event, String *params, Cardinal *n)
{
	if (*n < 1 || !cstrcasecmp(params[0], "next"))
		ani_ctl(w, ANI_NEXT);
	else if (!cstrcasecmp(params[0], "previous"))
		ani_ctl(w, ANI_PREVIOUS);
	else if (!cstrcasecmp(params[0], "stop"))
		ani_ctl(w, ANI_STOP);
	else if (!cstrcasecmp(params[0], "continue"))
		ani_ctl(w, ANI_CONTINUE);
	else if (!cstrcasecmp(params[0], "quit"))
		ani_ctl(w, ANI_QUIT);
}

static ani_image *i_list = NULL;

/* ---
This is self-contained: it uses only the static variable i_list to
   point to the head of the image list.
*/

ani_image *name2image(Display *display, Window ani_win, char *name)
{
	ani_image *i;
	int err;
	char b[1024];
	char *tmpfile = "/tmp/siagimage.xpm";

	if (!name) return NULL;

	for (i = i_list; i; i = i->next) {
		if (!strcmp(name, i->filename)) return i;
	}
	/* didn't find it; try loading */
	sprintf(b, "%s/common/any2xpm %s > %s", datadir, name, tmpfile);
	system(b);

	i = (ani_image *)cmalloc(sizeof(ani_image));
	i->xa.closeness = 40000;
	i->xa.exactColors = FALSE;
	i->xa.valuemask = XpmCloseness | XpmExactColors;
	err = XpmReadFileToPixmap(display, ani_win,
		tmpfile, &(i->pixmap), &(i->mask), &(i->xa));
	if (err != XpmSuccess) {
		cfree(i);
		fprintf(stderr, "XpmReadFileToPixmap returns %s\n",
			XpmGetErrorString(err));
		return NULL;
	}
	remove(tmpfile);
	i->filename = cstrdup(name);
	i->next = i_list;
	i_list = i;
	return i;
}

/* ---
this looks unnecessarily complex and it is, but I had problems
   with signedness and MAXINT wraparound
*/

static int INTERPOLATE(int x1, int x2,
		unsigned int t, unsigned int t1, unsigned int t2)
{
	long x0 = x1;		/* always pos */
	long xd = x2-x1;	/* pos or neg */
	long td1 = t-t1;	/* always pos */
	long td2 = t2-t1;	/* always pos */
	long tmp = xd*td1;	/* pos or neg */
	long rl = x0+tmp/td2;	/* pos or neg */
	int ri = rl;		/* pos or neg */

	return ri;
}

/* ---
create, draw and return a pixmap of the stage at time now.
   Caller must free
this is self-contained; no global or static vars are used,
   so it can be called at any time from anywhere with any parameters
*/

Pixmap ani_pixmap(Display *display, Window ani_win, GC ani_gc,
		ani_object *cast, char *bg_pixmap, int ani_now)
{
	int color_index;
	unsigned long color;

	int x, y, x1, y1, x2, y2;
	unsigned int b1, d1;
	Window root;
	int width, height;
	unsigned int w1, h1;
	int visible;
	int angle1, angle2;
	ani_object *actor;
	ani_image *img, *bg_image;
	Pixmap scribble;
	Window ani_root;
	int ani_x, ani_y;
	unsigned int ani_width, ani_height, ani_border, ani_depth;
	rich_char *p;

	XGetGeometry(display, ani_win, &ani_root,
		&ani_x, &ani_y, &ani_width, &ani_height,
		&ani_border, &ani_depth);
	if (ani_width > 2000 || ani_height > 2000) return None;

	scribble = XCreatePixmap(display, ani_win,
		ani_width, ani_height, ani_depth);

	/* tile the background */
	bg_image = name2image(display, ani_win, bg_pixmap);
	if (bg_image) {
		int bg_x, bg_y;
		unsigned int bg_width, bg_height, bg_border, bg_depth;

		XSetClipMask(display, ani_gc, None);

		XGetGeometry(display, bg_image->pixmap, &root, &bg_x, &bg_y,
				&bg_width, &bg_height, &bg_border, &bg_depth);
		for (y = 0; y < ani_height; y += bg_height) {
			for (x = 0; x < ani_width; x += bg_width) {
				XCopyArea(display, bg_image->pixmap, scribble,
					ani_gc, 0, 0, bg_width, bg_height, x, y);
			}
		}
	}


	/* now loop over the objects, drawing each in the right place */
	for (actor = cast; actor; actor = actor->next) {
		/* find the tick before and after ani_now and
		   interpolate to find position and size.
		   the script always contains a tick for time=0 */
		ani_script *before, *after;

		before = actor->script;		/* this is never NULL */
		while (before->next) {
			if (before->next->time > ani_now) break;
			before = before->next;
		}
		after = before->next;
		visible = before->visible;
		if (!visible) continue;

		/* set color */
		color_index = lookup_color(
				color_table[format_table[actor->fmt].fg].name);
		color = get_color(color_index);
		XSetForeground(display, ani_gc, color);

		XSetClipMask(display, ani_gc, None);

		if (!after) {	/* object remains at before */
			x = before->x;
			y = before->y;
			width = before->width;
			height = before->height;
		} else {	/* interpolate */
			x = INTERPOLATE(before->x, after->x,
				ani_now, before->time, after->time);
			y = INTERPOLATE(before->y, after->y,
				ani_now, before->time, after->time);
			width = INTERPOLATE(before->width, after->width,
				ani_now, before->time, after->time);
			height = INTERPOLATE(before->height, after->height,
				ani_now, before->time, after->time);
		}

		switch (actor->type) {
		case ANI_NONE:
			break;
		case ANI_LINE:
			x2 = x+width;
			y2 = y+height;
			XDrawLine(display, scribble,
				ani_gc, x, y, x2, y2);
			break;
		case ANI_RECTANGLE:
			XDrawRectangle(display, scribble,
				ani_gc, x, y, width, height);
			break;
		case ANI_ARC:
			angle1 = 64*90;		/* bogus!!! */
			angle2 = 64*180;
			XDrawArc(display, scribble,
				ani_gc, x, y, width, height,
				angle1, angle2);
			break;
		case ANI_ELLIPSE:
			XDrawArc(display, scribble,
				ani_gc, x, y, width, height,
				0, 64*360);
			break;
		case ANI_PIXMAP:
			img = name2image(display, ani_win,
					actor->string);
			if (!img) break;

			XGetGeometry(display, img->pixmap, &root, &x1, &y1,
				&w1, &h1, &b1, &d1);
			XSetClipOrigin(display, ani_gc, x, y);
			XSetClipMask(display, ani_gc, img->mask);

			XCopyArea(display, img->pixmap,
				scribble, ani_gc,
				0, 0, w1, h1, x, y);
			break;
		case ANI_STRING:
			/* set font */
			if (!actor->string) break;	/* sanity */
			p = rc_makerich(actor->string, actor->fmt);
			rc_strdraw(scribble, ani_gc, x, y, p, -1);
			cfree(p);
			break;
		case ANI_POINT:
			XDrawPoint(display, scribble,
				ani_gc, x, y);
			break;
		case ANI_FILLRECT:
			XFillRectangle(display, scribble,
				ani_gc, x, y, width, height);
			break;
		case ANI_FILLARC:
			angle1 = 64*90;		/* bogus!!! */
			angle2 = 64*180;
			XFillArc(display, scribble,
				ani_gc, x, y, width, height,
				angle1, angle2);
			break;
		case ANI_FILLELLIPSE:
			XFillArc(display, scribble,
				ani_gc, x, y, width, height,
				0, 64*360);
			break;
		default:
			fprintf(stderr, "bzz\n");
		}
	}

	return scribble;
}

/* end of snip from animator.c */

/* This seems to be the only way to avoid BadDrawable errors */
#define superclass (&coreClassRec)
static void Realize(Widget w, XtValueMask *valueMask,
		XSetWindowAttributes *attributes)
{
	AnimatorWidget aw = (AnimatorWidget) w;
	(*superclass->core_class.realize) (w, valueMask, attributes);
	aw->animator.waiting = False;
	aw->animator.mode = aw->animator.newmode;
}

static void Destroy(Widget w)
{
	AnimatorWidget aw = (AnimatorWidget) w;
	aw->animator.mode = ANI_QUIT;
	if (aw->animator.waiting)
		XtRemoveTimeOut(aw->animator.timeout);
}

static void Redisplay(Widget w, XEvent *xevent, Region r)
{
	AnimatorWidget aw = (AnimatorWidget) w;
	Pixmap scribble;
	Window ani_root;
	int ani_x, ani_y;
	unsigned int ani_width, ani_height, ani_border, ani_depth;
	GC ani_gc;
	unsigned long valuemask = 0;
	XGCValues values;

	if (!XtIsRealized(w)) return;	/* but that doesn't work */
	if (!XtIsManaged(w)) return;	/* what about this */

	if (!aw->animator.cast) {
		XClearWindow(XtDisplay(w), XtWindow(w));
		return;
	}

	/* must use Xlib function because we want to change the GC */
	ani_gc = XCreateGC(XtDisplay(aw), XtWindow(aw),
				valuemask, &values);
	XSetGraphicsExposures(XtDisplay(aw), ani_gc, 0);

	scribble = ani_pixmap(XtDisplay(aw), XtWindow(aw),
		ani_gc, aw->animator.cast,
		aw->animator.bg_pixmap, aw->animator.now);
	XGetGeometry(XtDisplay(aw), scribble, &ani_root,
		&ani_x, &ani_y, &ani_width, &ani_height,
		&ani_border, &ani_depth);

	XCopyArea(XtDisplay(aw), scribble, XtWindow(aw),
		ani_gc, 0, 0,
		ani_width, ani_height, 0, 0);

	/* and now we don't need this any more */
	XFreePixmap(XtDisplay(aw), scribble);
	XFreeGC(XtDisplay(aw), ani_gc);
        /* update plugin positions */
        DoLayout(aw);
}

static Boolean SetValues(Widget current, Widget request, Widget new,
			ArgList args, Cardinal *argc)
{
	AnimatorWidget curaw = (AnimatorWidget) current;
	AnimatorWidget newaw = (AnimatorWidget) new;
	Boolean do_redisplay = False;

	if (curaw->animator.newmode != newaw->animator.newmode) {
		ani_ctl((Widget)newaw, newaw->animator.newmode);
		newaw->animator.newmode = ANI_NONE;
	}
	if ((curaw->animator.now != newaw->animator.now)
		/*|| (curaw->animator.mode != newaw->animator.mode)*/) {
#ifdef WANT_FLICKER
		do_redisplay = True;
#else
		Redisplay(new, NULL, None);
#endif
	}

	if (curaw->animator.cast != newaw->animator.cast) {
#ifdef WANT_FLICKER
		do_redisplay = True;
#else
		Redisplay(new, NULL, None);
#endif
	}
	if (newaw->animator.cast == NULL) do_redisplay = True;

	return do_redisplay;
}

/*
 * Do a layout, actually assigning positions.
 */

static void DoLayout(AnimatorWidget sw)
{
        int i;
        int x, y;

        for (i = 0; i < sw->composite.num_children; i++) {
		x = y = 0;
                XtMoveWidget(sw->composite.children[i], x, y);
        }
}

/*
 * Actually layout the table
 */

static void Resize(Widget w)
{
        DoLayout((AnimatorWidget)w);
} /* Resize */

/*
 * Geometry Manager
 *
 * 'reply' is unused; we say only yeay or nay, never almost.
 */

static XtGeometryResult GeometryManager(Widget w,
                XtWidgetGeometry request, XtWidgetGeometry reply)
{
        return XtGeometryYes;
}

static void ChangeManaged(Widget w)
{
    DoLayout((AnimatorWidget)w);
}

