// 	$Id: xttpd.cc,v 1.4 1999/12/05 20:18:27 dave Exp $	
/*  xttpd  XTide web server.

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.hh"

// These browsers nowadays can get pretty verbose.
#define bufsize 10000

// Nasty global variables.
TideContext *context = NULL;
Dstr webmaster;
ZoneIndex zi;
int zoneinfo_doesnt_suck;

#include "purec++.hh"

static void
setup_socket (unsigned short port_number, int &s)
{
  struct sockaddr_in sin;
  struct hostent *hp;
  struct utsname foo;

  uname (&foo);
  if (!(hp = (struct hostent *) gethostbyname (foo.nodename))) {
    Dstr details ("Gethostbyname fails on own nodename!");
    barf (CANT_GET_SOCKET, details);
  }
  memset ((char *) &sin, '\0', sizeof sin);
  memcpy ((char *) &sin.sin_addr, hp->h_addr, hp->h_length);
  sin.sin_port = htons (port_number);
  sin.sin_family = hp->h_addrtype;
  sin.sin_addr.s_addr = INADDR_ANY;
  if ((s = socket (hp->h_addrtype, SOCK_STREAM, 0)) == -1) {
    Dstr details ("socket: ");
    details += strerror (errno);
    details += ".";
    barf (CANT_GET_SOCKET, details);
  }
  {
    int tmp = 1;
    if (setsockopt (s, SOL_SOCKET, SO_REUSEADDR,
	  	    (char *)&tmp, sizeof(tmp)) < 0) {
      Dstr details ("setsockopt: ");
      details += strerror (errno);
      details += ".";
      barf (CANT_GET_SOCKET, details);
    }
  }{
    struct linger li;
    li.l_onoff = 1;
    li.l_linger = 6000; // Hundredths of seconds (1 min)
    if (setsockopt (s, SOL_SOCKET, SO_LINGER,
                   (char *)&li, sizeof(struct linger)) < 0) {
      xperror ("setsockopt");
    }
  }
  if (bind (s, (struct sockaddr *)(&sin), sizeof sin) == -1) {
    if (errno == EADDRINUSE) {
      Dstr details ("Socket already in use.");
      barf (CANT_GET_SOCKET, details);
    } else {
      Dstr details ("bind: ");
      details += strerror (errno);
      details += ".";
      barf (CANT_GET_SOCKET, details);
    }
  }
  // backlog was 0, but this hung Digital Unix.  5 is the limit for BSD.
  if (listen (s, 5) == -1) {
    Dstr details ("listen: ");
    details += strerror (errno);
    details += ".";
    barf (CANT_GET_SOCKET, details);
  }

  webmaster = getenv ("XTTPD_FEEDBACK");
#ifdef webmasteraddr
  if (webmaster.isNull())
    webmaster = webmasteraddr;
#endif
  if (webmaster == "software@flaterco.com")
    webmaster = (char *)NULL;  // I can't handle the AOLers all by myself.
  if (webmaster.isNull()) {
    // Guess the local webmaster.  (Guessing the FQDN is harder!)
    webmaster = hp->h_name;
    if (webmaster.strchr('.') == -1) {
      // They deleted getdomainname()...  Now what am I supposed to do?
      Dstr domain;
      FILE *fp;
      if ((fp = fopen ("/etc/defaultdomain", "r"))) {
	domain.scan (fp);
	fclose (fp);
	webmaster += ".";
	webmaster += domain;
      } else if ((fp = fopen ("/etc/domain", "r"))) {
	domain.scan (fp);
	fclose (fp);
	webmaster += ".";
	webmaster += domain;
      } else if ((fp = fopen ("/etc/resolv.conf", "r"))) {
	Dstr buf;
	while (!(buf.getline(fp)).isNull()) {
	  buf /= domain;
	  if (domain == "domain") {
	    buf /= domain;
	    webmaster += ".";
	    webmaster += domain;
	    break;
	  }
	}
	fclose (fp);
      }
    }
    if (webmaster %= "www.")
      webmaster /= 4;
    webmaster *= "webmaster@";
  }
  webmaster *= "<A HREF=\"mailto:";
  webmaster += "\">";
}

static void
add_disclaimer (Dstr &foo) {
  foo += "<CENTER><H2> NOT FOR NAVIGATION </H2></CENTER>\n\
<BLOCKQUOTE>\n\
<P>This program is distributed in the hope that it will be useful, but\n\
WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  The author\n\
assumes no liability for damages arising from use of these predictions.\n\
They are not certified to be correct, and they\n\
do not incorporate the effects of tropical storms, El Ni&ntilde;o,\n\
seismic events, continental drift, or changes in global sea level.\n\
</P></BLOCKQUOTE>\n";
}

static void
end_page (int s, int pure = 0) {
  Dstr end_mess ("<HR>");
  if (pure)
    end_mess += "<P><IMG SRC=\"/purec++.png\" ALIGN=LEFT HSPACE=25></P>";
  end_mess += "<P> <A HREF=\"/\">Start over</A> <BR>\n";
  end_mess += "<A HREF=\"/index.html\">Master index</A> <BR>\n";
  end_mess += "<A HREF=\"/zones/\">Zone index</A> <BR>\n";
  end_mess += "<A HREF=\"http://www.flaterco.com/xtide/\">XTide software</A> <BR>\n";
  end_mess += webmaster;
  end_mess += "Feedback</A> <BR>\n";
  end_mess += "<A HREF=\"/tricks.html\">Hints and tricks</A></P>\n\
</BODY>\n\
</HTML>\n";
  write (s, end_mess.aschar(), end_mess.length());
  close (s);
  exit (0);
}

static void
unimp_barf (int s) {
  char *barf_message = "\
HTTP/1.0 501 Not Implemented\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> HTTP Error </TITLE></HEAD>\n\
<BODY BGCOLOR=\"FFFFFF\"><H1> HTTP Error </H1>\n\
<P> So sorry!  This server cannot satisfy your request. </P>\n";

  write (s, barf_message, strlen (barf_message));
  end_page (s);
}

static void
notfound_barf (int s) {
  char *barf_message = "\
HTTP/1.0 404 Not Found\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> Not Found </TITLE></HEAD>\n\
<BODY BGCOLOR=\"FFFFFF\"><H1> Not Found </H1>\n\
<P> So sorry!  This server cannot satisfy your request. </P>\n";

  write (s, barf_message, strlen (barf_message));
  end_page (s);
}

static void
toolong_barf (int s) {
  char *barf_message = "\
HTTP/1.0 413 Request Entity Too Large\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> HTTP Error </TITLE></HEAD>\n\
<BODY BGCOLOR=\"FFFFFF\"><H1> HTTP Error </H1>\n\
<P> So sorry!  This server cannot satisfy your request. </P>\n";

  write (s, barf_message, strlen (barf_message));
  end_page (s);
}

static void
root_page (int s) {
  Dstr root_p ("\
HTTP/1.0 200 OK\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> XTide Tide Prediction Server </TITLE></HEAD>\n\
<BODY BGCOLOR=\"FFFFFF\"><CENTER><H1> XTide Tide Prediction Server </H1></CENTER>\n\
\n\
<CENTER>\n\
<P>Copyright (C) 1998\n\
David Flater.</P>\n\
</CENTER>\n");
  add_disclaimer (root_p);
  root_p += "<BLOCKQUOTE>\n\
<P>\n\
While this program is maintained by David Flater, the web site\n\
itself is not.  Please direct feedback about problems with the web site to\n\
the local ";
  root_p += webmaster;
  root_p += "webmaster</A>.</P></BLOCKQUOTE>\n\
<P> XTide can provide tide predictions for thousands of places around the\n\
world, but first you have to select a location.\n\
There are three ways to do this.  One\n\
way is just to enter its name here, and click on Search: </P>\n\
<P><FORM METHOD=GET ACTION=\"/query\">\n\
<INPUT NAME=\"location\" TYPE=text SIZE=\"48\">\n\
<INPUT TYPE=submit VALUE=\"Search\"> <INPUT TYPE=reset VALUE=\"Clear\">\n\
</FORM></P>\n\
\n\
<P> Your other choices are the <A HREF=\"/zones/\">zone index</A>, which\n\
indexes locations by time zone, and \n\
the <A HREF=\"index.html\">master index</A>, which is just one big list\n\
of every location supported by this server.  The master index could be\n\
very large, so if you are on a slow connection, don't use it.</P>\n\
\n\
<P> All images shipped by this web server are in PNG format.  Versions of\n\
Netscape or Internet Explorer older than version 4 will not be able to\n\
display them. </P>\n";

  write (s, root_p.aschar(), root_p.length());
  end_page (s);
}

static void
start_loclist (Dstr &d) {
  context->startLocListHTML (d);
}

// This differs from TideContext::listLocationHTML because we need to
// add hyperlinks.
static void
list_location (Dstr &d, StationIndex *si, unsigned long i) {
  assert (si);
  StationRef *sr = (*si)[i];
  assert (sr);
  d += "<TR><TD><A HREF=\"/locations/";
  d += i;
  d += ".html\">";
  d += sr->name;
  d += "</A></TD><TD>";
  {
    Dstr fname;
    sr->shortHarmonicsFileName (fname);
    d += fname;
  }
  d += "</TD><TD>";
  Dstr tempc;
  sr->coordinates.print (tempc);
  d += tempc;
  d += "</TD></TR>\n";
}

static void
end_loclist (Dstr &d) {
  context->endLocListHTML (d);
}

static void
index_page (int s) {
  Dstr top_p ("\
HTTP/1.0 200 OK\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> XTide Master Index </TITLE></HEAD>\n\
<BODY BGCOLOR=\"FFFFFF\"><CENTER><H1> XTide Master Index </H1></CENTER>\n\
\n\
<P> Click on any location to get tide predictions. </P>\n");
  start_loclist (top_p);
  unsigned long i;
  StationIndex *si = context->stationIndex();
  for (i=0; i<si->length(); i++) {
    list_location (top_p, si, i);
    if ((i % maxhtmltablen == 0) && (i != 0)) {
      end_loclist (top_p);
      start_loclist (top_p);
    }
  }
  end_loclist (top_p);
  write (s, top_p.aschar(), top_p.length());
  end_page (s);
}

static void
tricks (int s) {
  Dstr top_p ("\
HTTP/1.0 200 OK\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> XTide hints and tricks </TITLE></HEAD>\n\
<BODY BGCOLOR=\"FFFFFF\"><CENTER><H1> XTide hints and tricks </H1></CENTER>\n\
\n\
<P> This web server uses numerically indexed URLs to load prediction\n\
pages.  Those URLs look like this:</P>\n\
\n\
<PRE>http://whatever/locations/nnn.html</PRE>\n\
\n\
<P>For convenience, you can also load a prediction page with a\n\
URL of the following form: </P>\n\
\n\
<PRE>http://whatever/locations/name</PRE>\n\
\n\
<P>You will receive an error if the name-based lookup gives ambiguous results,\n\
but these URLs have the advantage that they do not need to change every time\n\
the tide data are updated.</P>");

  write (s, top_p.aschar(), top_p.length());
  end_page (s);
}

static void
add_time_control (Dstr &text, Dstr &url, unsigned y, unsigned m, unsigned d) {
  text += "<H2> Time Control </H2>\n";
  text += "<P> <FORM METHOD=GET ACTION=\"";
  text += url;
  text += "\">\nYear: <SELECT NAME=\"y\">\n";
  unsigned i;
  for (i=globalfirstyear; i<=globallastyear; i++) {
    if (i == y)
      text += "<OPTION selected>";
    else
      text += "<OPTION>";
    text += i;
    text += "\n";
  }
  text += "</SELECT>\n";

  text += "Month: <SELECT NAME=\"m\">\n";
  for (i=1; i<=12; i++) {
    if (i == m)
      text += "<OPTION selected>";
    else
      text += "<OPTION>";
    text += i;
    text += "\n";
  }
  text += "</SELECT>\n";

  text += "Day: <SELECT NAME=\"d\">\n";
  for (i=1; i<=31; i++) {
    if (i == d)
      text += "<OPTION selected>";
    else
      text += "<OPTION>";
    text += i;
    text += "\n";
  }
  text += "</SELECT>\n";

  text += "<INPUT TYPE=submit VALUE=\"Go\"></FORM></P>";
  text += "<P> Note:  predictions for some locations do not cover the\n\
full span of years.  If your browser returns a blank page or a \"no data\"\n\
error, then the predictions that you requested are not available. </P>\n";
}

Timestamp
parse_time_control (Dstr &filename, Dstr timezone, Settings *settings,
Dstr &tc_out) {
  tc_out = (char *)NULL;
  Timestamp ret (time(NULL));
  int i;
  if ((i = filename.strrchr ('?')) != -1) {
    unsigned y, m, d;
    if ((sscanf (filename.ascharfrom(i), "?y=%u&m=%u&d=%u", &y, &m, &d)) == 3) {
      char temp[80];
      sprintf (temp, "%4u-%02u-%02u 00:00", y, m, d);
      Dstr in_iso (temp);
      ret = Timestamp (in_iso, timezone, settings);
      tc_out = filename.ascharfrom(i);
    }
  }
  return ret;
}

static void
load_location_page(int s, Dstr &filename, unsigned long i)
{
  StationIndex *si = context->stationIndex();
  Station *st = (*si)[i]->load (context);

  Dstr top_p ("\
HTTP/1.0 200 OK\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> ");

  top_p += st->name;
  top_p += "</TITLE></HEAD>\n<BODY BGCOLOR=\"FFFFFF\">\n<CENTER><H1> ";
  top_p += st->name;
  top_p += " <BR>\n";

  Dstr text_out, tc_out;
  Timestamp start_tm = parse_time_control (filename, st->timeZone,
    context->settings, tc_out);
  Dstr myurl ("/locations/");
  myurl += i;
  myurl += ".html";
  struct tm *starttm = start_tm.get_tm (st->timeZone, context->settings);
  unsigned y = starttm->tm_year + 1900;
  unsigned m = starttm->tm_mon + 1;
  unsigned d = starttm->tm_mday;

  if (tc_out.isNull())
    top_p += "Local time:  ";
  else
    top_p += "Requested time:  ";
  Dstr nowbuf;
  start_tm.print (nowbuf, st->timeZone, context->settings);
  top_p += nowbuf;
  top_p += " </H1></CENTER>\n";

  if (!zoneinfo_doesnt_suck) {
    top_p += "<P> Warning:  this host machine is using an obsolete time zone\n\
database.  Summer Time (Daylight Savings Time) adjustments will not be done\n\
for some locations. </P>\n";
  }

  top_p += "<P><IMG SRC=\"/graphs/";
  top_p += i;
  top_p += ".png";
  top_p += tc_out;
  top_p += "\" WIDTH=";
  top_p += context->settings->gw;
  top_p += " HEIGHT=";
  top_p += context->settings->gh;
  top_p += "></P><P><PRE>\n";

  context->textMode (st, start_tm, text_out);
  top_p += text_out;
  top_p += "</PRE></P>\n";

  add_time_control (top_p, myurl, y, m, d);

  top_p += "<H2> Tide Calendars </H2>\n";
  top_p += "<P> <A HREF=\"/calendar/month/";
  top_p += i;
  top_p += ".html";
  top_p += tc_out;
  top_p += "\"> One month</A> <BR>\n";
  top_p += "<A HREF=\"/calendar/year/";
  top_p += i;
  top_p += ".html";
  top_p += tc_out;
  top_p += "\"> One year</A> </P>\n";
  top_p += "<P> Calendars will be generated for the month/year noted\n\
at the top of this page. </P>\n";

  add_disclaimer (top_p);
  write (s, top_p.aschar(), top_p.length());
  end_page (s, 1);
}

static void
exact_location (int s, Dstr &filename, Dstr &loc)
{
  unsigned long i, count = 0;
  long match = -1;
  StationIndex *si = context->stationIndex();
  for (i=0; i<si->length(); i++)
  {
    	if ((*si)[i]->name %= loc)
	{
		if(match == -1)
			match = i;
		else
			match = -2;
      	}
      	count++;
  }

  Dstr top_p ("\
HTTP/1.0 200 OK\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> Exact Query Error </TITLE></HEAD>\n<BODY BGCOLOR=\"FFFFFF\">\n<CENTER>\n\
<H1> Exact Query Error </H1></CENTER>\n");

  switch(match)
  {
	case -1:	// no match
		top_p += "<P>No locations matched your query.</P>";
		break;

	case -2:	// more than one match
		top_p += "<P>More than one location matched your query.</P>\n";
		break;

	default:	// clear hit
  		load_location_page(s, filename, (unsigned long)match);
		return;
  }
  write (s, top_p.aschar(), top_p.length());
  end_page (s);
}

static void
badclientlocation (Dstr &filename) {
  log ("Bad client location: ", filename, LOG_NOTICE);
}

static void
rangeerror (Dstr &filename) {
  log ("Bad client location (range error): ", filename, LOG_NOTICE);
}

static void
location_page (int s, Dstr &filename) {
  unsigned long i;
  Dstr loc(filename);
  loc /= strlen("/locations/");
  if(!isdigit(loc[0]))
  {
	exact_location (s, filename, loc);
	return;
  }

  if (sscanf (loc.aschar(), "%lu.html", &i) != 1) {
    badclientlocation (filename);
    notfound_barf (s);
  }

  StationIndex *si = context->stationIndex();
  if (i >= si->length()) {
    rangeerror (filename);
    notfound_barf (s);
  }
  load_location_page(s, filename, i);
}

static void
calendar (int s, Dstr &filename) {
  unsigned long i;
  int isyear = 0;
  if (sscanf (filename.aschar(), "/calendar/month/%lu.html", &i) != 1) {
    isyear = 1;
    if (sscanf (filename.aschar(), "/calendar/year/%lu.html", &i) != 1) {
      badclientlocation (filename);
      notfound_barf (s);
    }
  }

  StationIndex *si = context->stationIndex();
  if (i >= si->length()) {
    rangeerror (filename);
    notfound_barf (s);
  }
  Station *st = (*si)[i]->load (context);

  Dstr top_p ("\
HTTP/1.0 200 OK\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> ");
  top_p += st->name;
  top_p += "</TITLE></HEAD>\n<BODY BGCOLOR=\"FFFFFF\">\n";

  Dstr tc_out;
  Timestamp start_tm = parse_time_control (filename, st->timeZone,
    context->settings, tc_out);
  struct tm *starttm = start_tm.get_tm (st->timeZone, context->settings);
  unsigned y = starttm->tm_year + 1900;
  unsigned m = starttm->tm_mon + 1;

  Timestamp end_tm;
  {
    char temp[80];
    if (isyear)
      sprintf (temp, "%4u-01-01 00:00", y);
    else
      sprintf (temp, "%4u-%02u-01 00:00", y, m);
    start_tm = Timestamp (temp, st->timeZone, context->settings);
    if (isyear)
      sprintf (temp, "%4u-01-01 00:00", y+1);
    else
      sprintf (temp, "%4u-%02u-01 00:00", y, m+1);
    end_tm = Timestamp (temp, st->timeZone, context->settings) - Interval(1);
  }

  if (!zoneinfo_doesnt_suck) {
    top_p += "<P> Warning:  this host machine is using an obsolete time zone\n\
database.  Summer Time (Daylight Savings Time) adjustments will not be done\n\
for some locations. </P>\n";
  }

  top_p += "<P>\n";
  Dstr text_out;
  context->calendarMode (st, start_tm, end_tm, text_out, 1);
  top_p += text_out;
  top_p += "</P>\n";

  add_disclaimer (top_p);
  write (s, top_p.aschar(), top_p.length());
  end_page (s, 1);
}

static void
purecppimage (int s) {
  Dstr top_p
    ("HTTP/1.0 200 OK\nMIME-version: 1.0\nContent-type: image/png\n\n");
  write (s, top_p.aschar(), top_p.length());
  write (s, pcpp, pcpplen);
  close (s);
  exit (0);
}

static void
graphimage (int s, Dstr &filename) {
  unsigned long i;
  if (sscanf (filename.aschar(), "/graphs/%lu.png", &i) != 1) {
    badclientlocation (filename);
    notfound_barf (s);
  }

  StationIndex *si = context->stationIndex();
  if (i >= si->length()) {
    rangeerror (filename);
    notfound_barf (s);
  }
  Station *st = (*si)[i]->load (context);

  Dstr tc_out;
  Timestamp start_tm = parse_time_control (filename, st->timeZone,
    context->settings, tc_out);

  Dstr top_p
    ("HTTP/1.0 200 OK\nMIME-version: 1.0\nPragma: no-cache\nContent-type: image/png\n\n");
  write (s, top_p.aschar(), top_p.length());

  RGBGraph g (context->settings->gw, context->settings->gh, context->colors);
  g.drawTides (st, start_tm);
  png_socket = s;  // See externC.cc
  g.writeAsPNG (socket_write_data_fn);
  close (s);
  exit (0);
}

static void
zones (int s, Dstr &filename) {
  Dstr zone(filename);
  zone /= strlen("/zones/");
  Dstr title;
  ZoneIndex::dnode *dn;
  if (zone.length()) {
    dn = zi[zone];
    if (!dn) {
      log ("Bad zone: ", filename, LOG_NOTICE);
      notfound_barf (s);
    }
    title = "Zone ";
    title += zone;
  } else {
    title = "Zone Index";
    dn = zi.top;
  }

  Dstr top_p ("\
HTTP/1.0 200 OK\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> ");
  top_p += title;
  top_p += "</TITLE></HEAD>\n<BODY BGCOLOR=\"FFFFFF\">\n<CENTER><H1> ";
  top_p += title;
  top_p += " </H1></CENTER>\n";

  if (dn->subzones) {
    ZoneIndex::dnode *sn;
    if (zone.length())
      sn = dn->subzones;
    else
      sn = dn;
    top_p += "\
<P> Most zones consist of a region followed by a city name.  Choose the\n\
zone that matches the country and time zone of the location that you want.\n\
For example:  places on the east coast of the U.S. are in zone\n\
:America/New_York; places in New Brunswick, Canada are in zone\n\
:America/Halifax.</P><UL>\n";
    while (sn) {
      top_p += "<LI><A HREF=\"/zones/";
      top_p += sn->name;
      top_p += "\">";
      top_p += sn->name;
      top_p += "</A></LI>\n";
      sn = sn->next;
    }
    top_p += "</UL>\n";
  }
  if (dn->il) {
    top_p += "<P> Click on any location to get tide predictions. </P>\n";
    struct ZoneIndex::indexlist *ix = dn->il;
    unsigned long i = 0;
    start_loclist (top_p);
    StationIndex *si = context->stationIndex();
    while (ix) {
      list_location (top_p, si, ix->i);
      if ((i % maxhtmltablen == 0) && (i != 0)) {
        end_loclist (top_p);
        start_loclist (top_p);
      }
      ix = ix->next;
      i++;
    }
    end_loclist (top_p);
  }
  if (zone.length()) {
    Dstr upzone(filename);
    upzone -= upzone.length()-1;
    upzone -= upzone.strrchr('/')+1;
    top_p += "<P> <A HREF=\"";
    top_p += upzone;
    top_p += "\">Back up</A></P>\n";
  }
  write (s, top_p.aschar(), top_p.length());
  end_page (s);
}

// Remove HTTP mangling from query string.
static void
demangle (Dstr &s, int sock) {
  Dstr buf;
  unsigned i = 0;
  while (i<s.length()) {
    if (s[i] == '+') {
      buf += ' ';
      i++;
    } else if (s[i] == '%') {
      char tiny[3];
      int temp;
      i++;
      tiny[0] = s[i++];
      tiny[1] = s[i++];
      tiny[2] = '\0';
      if (sscanf (tiny, "%x", &temp) != 1) {
        log ("Really nasty URL caught in demangle....", LOG_NOTICE);
        notfound_barf (sock);
      }
      buf += (char)temp;
    } else
      buf += s[i++];
  }
  s = buf;
}

static void
query (int s, Dstr &filename) {
  Dstr locq (filename);
  locq /= strlen("/query?location=");
  demangle (locq, s);

  // Strip leading whitespace
  while (locq.length())
    if (isspace (locq[0]))
      locq /= 1;
    else
      break;

  Dstr top_p ("\
HTTP/1.0 200 OK\n\
MIME-version: 1.0\n\
Content-type: text/html\n\
\n\
<HTML>\n\
<HEAD><TITLE> Query Results </TITLE></HEAD>\n<BODY BGCOLOR=\"FFFFFF\">\n<CENTER>\n\
<H1> Query Results </H1></CENTER>\n");

  unsigned long i, count = 0;
  StationIndex *si = context->stationIndex();
  for (i=0; i<si->length(); i++) {
    if ((*si)[i]->name %= locq) {
      if (count == 0) {
        top_p += "<P>Locations matching \"";
        top_p += locq;
        top_p += "\":</P><P>\n";
        start_loclist (top_p);
      }
      list_location (top_p, si, i);
      if ((count % maxhtmltablen == 0) && (count != 0)) {
        end_loclist (top_p);
        start_loclist (top_p);
      }
      count++;
    }
  }
  if (count)
    end_loclist (top_p);
  else {
    // Try harder.
    for (i=0; i<si->length(); i++) {
      if ((*si)[i]->name.strcasestr (locq) != -1) {
	if (count == 0) {
          top_p += "<P> Regular query found no matches; using more aggressive (substring) query. </P>\n";
	  top_p += "<P>Locations matching \"";
	  top_p += locq;
	  top_p += "\":</P><P>\n";
	  start_loclist (top_p);
	}
	list_location (top_p, si, i);
	if ((count % maxhtmltablen == 0) && (count != 0)) {
	  end_loclist (top_p);
	  start_loclist (top_p);
	}
	count++;
      }
    }
    if (count)
      end_loclist (top_p);
    else
      top_p += "<P> No locations matched your query.  However, that might\n\
just mean that something is spelled strangely in the XTide database.  To\n\
make sure that you aren't missing a location that is actually in there,\n\
you should check the indexes linked below.</P>\n";
  }
  write (s, top_p.aschar(), top_p.length());
  end_page (s);
}

static void
log_hits (struct sockaddr *addr) {
  Dstr msg;
  switch (addr->sa_family) {
  case AF_INET:
    {
      struct sockaddr_in *u = (struct sockaddr_in *)addr;
      unsigned long a = u->sin_addr.s_addr;
      char *c =
        // Archaic SunOS syntax:
        // inet_ntoa (&a);
        inet_ntoa (u->sin_addr);
      msg = c;
      msg += "  ";
      struct hostent *hp;
      if ((hp = gethostbyaddr ((char *)(&a), sizeof a, AF_INET)) == NULL)
        msg += "(?)";
      else
        msg += hp->h_name;
    }
    break;
  default:
    msg = "Non-Internet address (?)";
  }
  log (msg, LOG_INFO);
}

static void
handle_request (struct sockaddr *addr, int s) {
  char request[bufsize+1];
  Dstr buf, command, filename;

  log_hits (addr);

  // This will truncate long requests, but we generally won't lose
  // anything that matters.
  int len = read (s, request, bufsize); // Blah, blah, blah...

  if (len >= bufsize) {
    log ("Request too long", LOG_WARNING);
    toolong_barf (s);
  }
  request[len] = '\0';
  buf = request;

#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
  log ("-------------------------\n", request, LOG_DEBUG);
#endif

  buf /= command;
  if (command != "GET") {
    log ("Bad client command: ", command, LOG_NOTICE);
    unimp_barf (s);
  }

  buf /= filename;

  if (filename == "/")
    root_page (s);
  else if (filename == "/index.html")
    index_page (s);
  else if (filename == "/tricks.html")
    tricks (s);
  else if (filename == "/purec++.png")
    purecppimage (s);
  else if (filename %= "/locations/")
    location_page (s, filename);
  else if (filename %= "/graphs/")
    graphimage (s, filename);
  else if (filename %= "/zones/")
    zones (s, filename);
  else if (filename %= "/calendar/")
    calendar (s, filename);
  else if (filename %= "/query?location=")
    query (s, filename);
  else {
    log ("Client tried to get ", filename, LOG_NOTICE);
    notfound_barf (s);
  }

  exit (0);
}

static void
dont_be_root() {
  struct passwd *nb = getpwnam ("nobody");
  if (nb) {
    if (setuid (nb->pw_uid) == 0)
      return;
    else
      xperror ("setuid");
  } else
    xperror ("getpwnam");
  log ("Can't set uid to 'nobody.'  Hope that's OK....", LOG_WARNING);
}

int main (int argc, char **argv)
{
  set_daemon_mode();
  pid_t fpid = fork();
  if (fpid == -1) {
    xperror ("fork");
    exit (-1);
  }
  if (fpid)
    exit (0);

  int listen_socket, new_socket, ts;
  unsigned short portnum = 80;
  fd_set rdset;
  FD_ZERO (&rdset);

  if (argc >= 2)
    if (sscanf (argv[1], "%hu", &portnum) != 1)
      portnum = 80;

  setup_socket (portnum, listen_socket);
  dont_be_root();
  FD_SET (listen_socket, &rdset);

  Settings settings;
  {
    ConfigDefaults cd;
    UserDefaults ud;
    CommandLineSettings cls (argc, argv);
    settings.supersedeBy (cd);
    settings.supersedeBy (ud);
    settings.supersedeBy (cls);
  }

  context = new TideContext (argc, argv, new Colors (settings), &settings);
  // Build the indices.
  zi.add (context->stationIndex());
  // Avoid issuing the zoneinfo warning a million times.
  {
    Timestamp nowarn(time(NULL));
    zoneinfo_doesnt_suck = nowarn.zoneinfo_doesnt_suck(&settings);
  }
  log ("Accepting connections.", LOG_NOTICE);

  ts = -1;
  struct sockaddr addr;
  while (1) {
    fd_set trdset, twrset, texset;
    trdset = rdset;
    // I seem to remember that some platforms barf if you provide
    // null pointers for wrset and exset; best to be safe.
    FD_ZERO (&twrset);
    FD_ZERO (&texset);
    ts = select (listen_socket+1, &trdset, &twrset, &texset, NULL);
    if (ts > 0) {
      // Type of length is set by configure.
      acceptarg3_t length = sizeof (addr);
      if ((new_socket = accept (listen_socket, &addr, &length)) != -1) {
        if (!fork())
          handle_request (&addr, new_socket);
        close (new_socket);
      }
    }
    // Clean up zombies.
    while (waitpid (-1, NULL, WNOHANG|WUNTRACED) > 0);
  }

  exit (0);
}
