

#include "lispenvironment.h"
#include "standard.h"
#include "lispeval.h"
#include "lispatom.h"
#include "lispparser.h"
#include "stdfileio.h"
#include "stringio.h"
#include "lisperror.h"
#include "infixparser.h"
#include "lispuserfunc.h"
#include "mathuserfunc.h"
#include "platmath.h"
#include "numbers.h"
#include "anumber.h"
#include "arrayclass.h"
#include "patternclass.h"

#define InternalEval aEnvironment.iEvaluator->Eval


void LispArithmetic2(LispEnvironment& aEnvironment, LispPtr& aResult,
                     LispPtr& aArguments,
                     LispStringPtr (*func)(LispCharPtr f1, LispCharPtr f2,LispHashTable& aHashTable,LispInt aPrecision),
                    LispBoolean arbbase=LispFalse);

void LispArithmetic1(LispEnvironment& aEnvironment, LispPtr& aResult,
                     LispPtr& aArguments,
                     LispStringPtr (*func)(LispCharPtr f1, LispHashTable& aHashTable,LispInt aPrecision));


void LispCompare2(LispEnvironment& aEnvironment, LispPtr& aResult,
                     LispPtr& aArguments,
                     LispBoolean (*func)(LispCharPtr f1, LispCharPtr f2, LispHashTable& aHashTable,LispInt aPrecision));


void LispQuote(LispEnvironment& aEnvironment,
               LispPtr& aResult,
               LispPtr& aArguments)
{
    TESTARGS(2);
    aResult.Set(Argument(aArguments,1).Get()->Copy(LispFalse));
}
void LispEval(LispEnvironment& aEnvironment,LispPtr& aResult,
              LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr result;
    InternalEval(aEnvironment, result, Argument(aArguments,1));
    InternalEval(aEnvironment, aResult, result);
}

static void InternalSetVar(LispEnvironment& aEnvironment, LispPtr& aResult,
                LispPtr& aArguments,LispBoolean aMacroMode)
{
    TESTARGS(3);

    LispStringPtr varstring=NULL;

    if (aMacroMode)
    {
        LispPtr result;
        InternalEval(aEnvironment, result, Argument(aArguments,1));
        varstring = result.Get()->String();
    }
    else
    {
        varstring = Argument(aArguments,1).Get()->String();
    }
    Check(varstring != NULL,KLispErrInvalidArg);
    LispPtr result;
    InternalEval(aEnvironment, result, Argument(aArguments,2));
    aEnvironment.SetVariable(varstring, result);
    InternalTrue(aEnvironment,aResult);
}


void LispSetVar(LispEnvironment& aEnvironment, LispPtr& aResult,
                LispPtr& aArguments)
{
    InternalSetVar(aEnvironment, aResult,aArguments,LispFalse);
}
void LispMacroSetVar(LispEnvironment& aEnvironment, LispPtr& aResult,
                LispPtr& aArguments)
{
    InternalSetVar(aEnvironment, aResult,aArguments,LispTrue);
}


static void InternalClearVar(LispEnvironment& aEnvironment,
                      LispPtr& aResult, LispPtr& aArguments,
                      LispBoolean aMacroMode)
{
    LispIterator iter(Argument(aArguments,1));
    while (iter())
    {
        LispStringPtr str;

        if (aMacroMode)
        {
            LispPtr result;
            InternalEval(aEnvironment, result, *iter.Ptr());
            str = result.Get()->String();
        }
        else
        {
            str = iter()->String();
        }

        Check(str != NULL, KLispErrInvalidArg);
        aEnvironment.UnsetVariable(str);
        iter.GoNext();
    }
    InternalTrue(aEnvironment,aResult);
}

void LispClearVar(LispEnvironment& aEnvironment,
                  LispPtr& aResult,LispPtr& aArguments)
{
    InternalClearVar(aEnvironment,aResult, aArguments, LispFalse);
}
void LispMacroClearVar(LispEnvironment& aEnvironment,
                  LispPtr& aResult,LispPtr& aArguments)
{
    InternalClearVar(aEnvironment,aResult, aArguments, LispTrue);
}


void LispMultiply(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, MultiplyFloat);
}

void LispAdd(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispInt length = InternalListLength(aArguments);
    if (length == 2)
    {
        LispArithmetic1(aEnvironment, aResult, aArguments, PlusFloat);
    }
    else
    {
        LispArithmetic2(aEnvironment, aResult, aArguments, AddFloat);
    }
}

void LispSubtract(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispInt length = InternalListLength(aArguments);
    if (length == 2)
    {
        LispArithmetic1(aEnvironment, aResult, aArguments, NegateFloat);
    }
    else
    {
        LispArithmetic2(aEnvironment, aResult, aArguments, SubtractFloat);
    }
}


void LispDivide(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, DivideFloat);
}


void LispSin(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, SinFloat);
}

void LispCos(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, CosFloat);
}

void LispTan(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, TanFloat);
}

void LispArcSin(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, ArcSinFloat);
}

void LispArcCos(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, ArcCosFloat);
}

void LispArcTan(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, ArcTanFloat);
}

void LispSqrt(LispEnvironment& aEnvironment, LispPtr& aResult,
              LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, SqrtFloat);
}

void LispFloor(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, FloorFloat);
}

void LispCeil(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, CeilFloat);
}

void LispAbs(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, AbsFloat);
}

void LispMod(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, ModFloat);
}

void LispDiv(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, DivFloat);
}

void LispLog(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, LnFloat);
}

void LispExp(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, ExpFloat);
}

void LispPower(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, PowerFloat);
}

void LispLessThan(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispCompare2(aEnvironment, aResult, aArguments, LessThan);
}

void LispGreaterThan(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispCompare2(aEnvironment, aResult, aArguments, GreaterThan);
}


void LispCompare2(LispEnvironment& aEnvironment, LispPtr& aResult,
                     LispPtr& aArguments,
                     LispBoolean (*func)(LispCharPtr f1, LispCharPtr f2, LispHashTable& aHashTable,LispInt aPrecision))
{
    TESTARGS(3);

    LispStringPtr str1;
    LispStringPtr str2;

    LispPtr result1;
    LispPtr result2;
    InternalEval(aEnvironment, result1, Argument(aArguments,1));
    InternalEval(aEnvironment, result2, Argument(aArguments,2));

    str1 = result1.Get()->String();
    str2 = result2.Get()->String();
    Check(str1 != NULL && str2 != NULL,KLispErrInvalidArg);
    Check(IsNumber(str1->String(),LispTrue) &&
        IsNumber(str2->String(),LispTrue),KLispErrInvalidArg);

    InternalBoolean(aEnvironment,aResult,
                    func(str1->String(),str2->String(),
                         aEnvironment.HashTable(),
                         aEnvironment.Precision()));
}

void LispPi(LispEnvironment& aEnvironment, LispPtr& aResult,
            LispPtr& aArguments)
{
    TESTARGS(1);
    aResult.Set(LispAtom::New(PiFloat(aEnvironment.HashTable(),
                                         aEnvironment.Precision())));
}

void LispArithmetic1(LispEnvironment& aEnvironment, LispPtr& aResult,
                     LispPtr& aArguments,
                     LispStringPtr (*func)(LispCharPtr f1, LispHashTable& aHashTable,LispInt aPrecision))
{
    TESTARGS(2);

    LispStringPtr str1;

    LispPtr result1;
    InternalEval(aEnvironment, result1, Argument(aArguments,1));

    str1 = result1.Get()->String();
    Check(str1 != NULL,KLispErrInvalidArg);
    Check(IsNumber(str1->String(),LispTrue),KLispErrInvalidArg);
    aResult.Set(LispAtom::New(func(str1->String(),
                                   aEnvironment.HashTable(),
                                   aEnvironment.Precision())));
}

void LispArithmetic2(LispEnvironment& aEnvironment, LispPtr& aResult,
                     LispPtr& aArguments,
                     LispStringPtr (*func)(LispCharPtr f1, LispCharPtr f2,LispHashTable& aHashTable,LispInt aPrecision),
                    LispBoolean arbbase)
{
    TESTARGS(3);

    LispStringPtr str1;
    LispStringPtr str2;

    LispPtr result1;
    LispPtr result2;
    InternalEval(aEnvironment, result1, Argument(aArguments,1));
    InternalEval(aEnvironment, result2, Argument(aArguments,2));

    str1 = result1.Get()->String();
    str2 = result2.Get()->String();
    Check(str1 != NULL && str2 != NULL,KLispErrInvalidArg);
    if (!arbbase)
    {
        Check(IsNumber(str1->String(),LispTrue) &&
              IsNumber(str2->String(),LispTrue),KLispErrInvalidArg);
    }

    aResult.Set(LispAtom::New(func(str1->String(),str2->String(),
                                   aEnvironment.HashTable(),
                                   aEnvironment.Precision())));
}



void LispGcd(LispEnvironment& aEnvironment, LispPtr& aResult,
             LispPtr& aArguments)
{
    TESTARGS(3);

    LispStringPtr str1;
    LispStringPtr str2;

    LispPtr result1;
    LispPtr result2;
    InternalEval(aEnvironment, result1, Argument(aArguments,1));
    InternalEval(aEnvironment, result2, Argument(aArguments,2));

    str1 = result1.Get()->String();
    str2 = result2.Get()->String();
    Check(str1 != NULL && str2 != NULL,KLispErrInvalidArg);
    Check(IsNumber(str1->String(),LispFalse) &&
        IsNumber(str2->String(),LispFalse),KLispErrInvalidArg);
    aResult.Set(LispAtom::New(GcdInteger(str1->String(),str2->String(),
                                         aEnvironment.HashTable())));
}











void LispFullForm(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    InternalEval(aEnvironment, aResult, Argument(aArguments,1));
    LispPrinter printer;
    StdUserOutput output;
    printer.Print(aResult, output);
    output.Write("\n");
}


void LispHead(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    CheckIsList(evaluated);
    InternalNth(aResult, evaluated,1);
}

void LispNth(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(3);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    Check(evaluated.Get() != NULL,KLispErrNotList);
    Check(evaluated.Get()->SubList() != NULL,KLispErrNotList);

    LispPtr index;
    InternalEval(aEnvironment, index, Argument(aArguments,2));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);

    LispInt ind = InternalAsciiToInt(index.Get()->String()->String());
    
    InternalNth(aResult, evaluated,ind);
}


void LispTail(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    CheckIsList(evaluated);
    LispPtr first;
    InternalTail(first, evaluated);
    InternalTail(aResult, first);
    LispPtr head;
    head.Set(LispAtom::New(aEnvironment.HashTable().LookUp("List")));
    head.Get()->Next().Set(aResult.Get()->SubList()->Get());
    aResult.Get()->SubList()->Set(head.Get());
}

void LispUnList(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    CheckIsList(evaluated);

    InternalTail(aResult, evaluated);
}

void LispListify(LispEnvironment& aEnvironment, LispPtr& aResult,
                 LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    Check(evaluated.Get() != NULL,KLispErrNotList);
    Check(evaluated.Get()->SubList() != NULL,KLispErrNotList);

    LispPtr head;
    head.Set(LispAtom::New(aEnvironment.HashTable().LookUp("List")));
    head.Get()->Next().Set(evaluated.Get()->SubList()->Get());
    aResult.Set(LispSubList::New(head.Get()));
}




void LispDestructiveReverse(LispEnvironment& aEnvironment, LispPtr& aResult,
                            LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    CheckIsList(evaluated);

    // copy the list to revert first!
    LispPtr reversed;
    reversed.Set(LispAtom::New(aEnvironment.HashTable().LookUp("List")));
    InternalReverseList(reversed.Get()->Next(), evaluated.Get()->SubList()->Get()->Next());
    aResult.Set(LispSubList::New(reversed.Get()));
}


void LispLength(LispEnvironment& aEnvironment, LispPtr& aResult,
                LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    LispPtr* subList = evaluated.Get()->SubList();
    if (subList != NULL)
    {
        LispChar s[20];
        LispInt num = InternalListLength(subList->Get()->Next());
        InternalIntToAscii(s,num);
        aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(s)));
        return;
    }
    LispStringPtr string = evaluated.Get()->String();
    if (InternalIsString(string))
    {
        LispChar s[20];
        LispInt num = string->NrItems()-3;
        InternalIntToAscii(s,num);
        aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(s)));
        return;
    }
    //hier
    GenericClass *gen = evaluated.Get()->Generic();
    if (gen != NULL)
    if (StrEqual(gen->TypeName(),"\"Array\""))
    {
        LispInt size=((ArrayClass*)gen)->Size();
        LispChar s[20];
        InternalIntToAscii(s,size);
        aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(s)));
        return;
    }
    //hier
    CheckIsList(evaluated);
}


void LispList(LispEnvironment& aEnvironment, LispPtr& aResult,
              LispPtr& aArguments)
{
    LispPtr all;
    all.Set(LispAtom::New(aEnvironment.HashTable().LookUp("List")));
    LispIterator tail(all);
    tail.GoNext();
    LispIterator iter = Argument(aArguments,1);
    while (iter())
    {
        LispPtr evaluated;
        InternalEval(aEnvironment, evaluated, *iter.Ptr());
        tail.Ptr()->Set(evaluated.Get());
        tail.GoNext();
        iter.GoNext();
    }
    
    aResult.Set(LispSubList::New(all.Get()));
}


void LispConcatenate(LispEnvironment& aEnvironment, LispPtr& aResult,
              LispPtr& aArguments)
{
    LispPtr all;
    all.Set(LispAtom::New(aEnvironment.HashTable().LookUp("List")));
    LispIterator tail(all);
    tail.GoNext();


    LispIterator iter = Argument(aArguments,1);
    while (iter())
    {
        LispPtr evaluated;
        InternalEval(aEnvironment, evaluated, *iter.Ptr());
        CheckIsList(evaluated);
        InternalFlatCopy(*tail.Ptr(),evaluated.Get()->SubList()->Get()->Next());
        while (tail() != NULL)
            tail.GoNext();

        iter.GoNext();
    }
    
    aResult.Set(LispSubList::New(all.Get()));
}

void LispConcatenateStrings(LispEnvironment& aEnvironment, LispPtr& aResult,
              LispPtr& aArguments)
{
    LispString str;
    str.SetNrItems(0);
    str.Append('\"');
    
    LispIterator iter = Argument(aArguments,1);
    while (iter())
    {
        LispPtr evaluated;
        InternalEval(aEnvironment, evaluated, *iter.Ptr());
        CheckIsString(evaluated);

        LispInt i;
        LispInt length = evaluated.Get()->String()->NrItems()-2;
        LispCharPtr ptr=evaluated.Get()->String()->String();
        for (i=1;i<length;i++)
            str.Append(ptr[i]);
        
        iter.GoNext();
    }
    
    str.Append('\"');
    str.Append('\0');

    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(str.String())));
}


static void InternalDelete(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments, LispInt aDestructive)
{
    TESTARGS(3);
    LispPtr evaluated;
//    InternalEval(aEnvironment, evaluated, aArguments.Get()->Next());
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    CheckIsList(evaluated);

    LispPtr copied;
    if (aDestructive)
    {
        copied.Set(evaluated.Get()->SubList()->Get());
    }
    else
    {
        InternalFlatCopy(copied,*evaluated.Get()->SubList());
    }

    LispPtr index;
    InternalEval(aEnvironment, index, Argument(aArguments,2));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);
    LispInt ind = InternalAsciiToInt(index.Get()->String()->String());
    Check(ind>0,KLispErrInvalidArg);

    LispIterator iter(copied);
    while (ind>0)
    {
        iter.GoNext();
        ind--;
    }
    Check(iter() != NULL, KLispErrListNotLongEnough);
    LispPtr next;
    next.Set(iter()->Next().Get());
    iter.Ptr()->Set(next.Get());
    aResult.Set(LispSubList::New(copied.Get()));
}
void LispDelete(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    InternalDelete(aEnvironment, aResult,aArguments,LispFalse);
}
void LispDestructiveDelete(LispEnvironment& aEnvironment, LispPtr& aResult,
                           LispPtr& aArguments)
{
    InternalDelete(aEnvironment, aResult,aArguments,LispTrue);
}

void LispFlatCopy(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    CheckIsList(evaluated);
    LispPtr copied;
    InternalFlatCopy(copied,*evaluated.Get()->SubList());
    aResult.Set(LispSubList::New(copied.Get()));
}


static void InternalInsert(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments, LispInt aDestructive)
{
    TESTARGS(4);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    CheckIsList(evaluated);

    LispPtr copied;
    if (aDestructive)
    {
        copied.Set(evaluated.Get()->SubList()->Get());
    }
    else
    {
        InternalFlatCopy(copied,*evaluated.Get()->SubList());
    }
    
    LispPtr index;
    InternalEval(aEnvironment, index, Argument(aArguments,2));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);
    LispInt ind = InternalAsciiToInt(index.Get()->String()->String());
    Check(ind>0,KLispErrInvalidArg);

    LispIterator iter(copied);
    while (ind>0)
    {
        iter.GoNext();
        ind--;
    }

    LispPtr toInsert;
    InternalEval(aEnvironment, toInsert, Argument(aArguments,3));
    toInsert.Get()->Next().Set(iter());
    iter.Ptr()->Set(toInsert.Get());
    aResult.Set(LispSubList::New(copied.Get()));
}

void LispInsert(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    InternalInsert(aEnvironment, aResult,aArguments,LispFalse);
}

void LispDestructiveInsert(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    InternalInsert(aEnvironment, aResult,aArguments,LispTrue);
}







static void InternalReplace(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments, LispInt aDestructive)
{
    TESTARGS(4);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    CheckIsList(evaluated);

    LispPtr copied;
    if (aDestructive)
    {
        copied.Set(evaluated.Get()->SubList()->Get());
    }
    else
    {
        InternalFlatCopy(copied,*evaluated.Get()->SubList());
    }
    
    LispPtr index;
    InternalEval(aEnvironment, index, Argument(aArguments,2));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);
    LispInt ind = InternalAsciiToInt(index.Get()->String()->String());
    Check(ind>0,KLispErrInvalidArg);

    LispIterator iter(copied);
    while (ind>0)
    {
        iter.GoNext();
        ind--;
    }

    LispPtr toInsert;
    InternalEval(aEnvironment, toInsert, Argument(aArguments,3));
    Check(iter.Ptr() != NULL, KLispErrInvalidArg);
    Check(iter.Ptr()->Get() != NULL, KLispErrInvalidArg);
    toInsert.Get()->Next().Set(iter.Ptr()->Get()->Next().Get());
    iter.Ptr()->Set(toInsert.Get());
    aResult.Set(LispSubList::New(copied.Get()));
}

void LispReplace(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    InternalReplace(aEnvironment, aResult,aArguments,LispFalse);
}

void LispDestructiveReplace(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    InternalReplace(aEnvironment, aResult,aArguments,LispTrue);
}
















void LispNot(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    InternalNot(aResult, aEnvironment, evaluated);
}

void LispLazyAnd(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispPtr evaluated;

    LispIterator iter(Argument(aArguments,1));
    while (iter())
    {
        InternalEval(aEnvironment, evaluated, *iter.Ptr());
        if (IsFalse(aEnvironment, evaluated))
        {
            InternalFalse(aEnvironment,aResult);
            return;
        }
        iter.GoNext();
    }

    /*TODO weg?
     LispInt nrArguments = InternalListLength(aArguments);
    LispInt arg;
    for (arg=1;arg<nrArguments;arg++)
    {
        InternalEval(aEnvironment, evaluated, Argument(aArguments,arg));
        if (IsFalse(aEnvironment, evaluated))
        {
            InternalFalse(aEnvironment,aResult);
            return;
        }
    }
        */
    InternalTrue(aEnvironment,aResult);
}

void LispLazyOr(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispPtr evaluated;

    LispIterator iter(Argument(aArguments,1));
    while (iter())
    {
        InternalEval(aEnvironment, evaluated, *iter.Ptr());
        if (IsTrue(aEnvironment, evaluated))
        {
            InternalTrue(aEnvironment,aResult);
            return;
        }
        iter.GoNext();
    }

/*TODO
    LispInt nrArguments = InternalListLength(aArguments);
    LispInt arg;
    for (arg=1;arg<nrArguments;arg++)
    {
        InternalEval(aEnvironment, evaluated, Argument(aArguments,arg));
        if (IsTrue(aEnvironment, evaluated))
        {
            InternalTrue(aEnvironment,aResult);
            return;
        }
    }
    */
    InternalFalse(aEnvironment,aResult);
}

void LispEquals(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(3);
    LispPtr evaluated1;
    InternalEval(aEnvironment, evaluated1, Argument(aArguments,1));
    LispPtr evaluated2;
    InternalEval(aEnvironment, evaluated2, Argument(aArguments,2));

    InternalBoolean(aEnvironment,aResult,
                    InternalEquals(aEnvironment, evaluated1, evaluated2));
}


void LispWrite(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispPtr evaluated;
    LispInt nrArguments = InternalListLength(aArguments);
    LispInt arg;
    for (arg=1;arg<nrArguments;arg++)
    {
        InternalEval(aEnvironment, evaluated, Argument(aArguments,arg));
        aEnvironment.CurrentPrinter().Print(evaluated,
                                            *aEnvironment.CurrentOutput());
    }
    InternalTrue(aEnvironment,aResult);
}

static void RepeatChar(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments, LispCharPtr aString)
{
    LispInt nrArguments = InternalListLength(aArguments);
    Check(nrArguments == 1 || nrArguments == 2,KLispErrWrongNumberOfArgs);
    LispInt nrTimes=1;
    if (nrArguments == 2)
    {
        LispPtr index;
        InternalEval(aEnvironment, index, Argument(aArguments,1));
        Check(index.Get() != NULL, KLispErrInvalidArg);
        Check(index.Get()->String() != NULL, KLispErrInvalidArg);
        nrTimes = InternalAsciiToInt(index.Get()->String()->String());
    }
    Check(nrTimes>=0,KLispErrInvalidArg);
    LispInt i;
    for (i=0;i<nrTimes;i++)
        aEnvironment.CurrentOutput()->Write(aString);
    InternalTrue(aEnvironment,aResult);
}

void LispSpace(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    RepeatChar(aEnvironment, aResult, aArguments, " ");
}

void LispNewLine(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    RepeatChar(aEnvironment, aResult, aArguments, "\n");
}


void LispWriteString(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    Check(evaluated.Get()!= NULL,KLispErrInvalidArg);
    LispStringPtr str = evaluated.Get()->String();
    Check(str != NULL,KLispErrInvalidArg);
    Check((*str)[0] == '\"',KLispErrInvalidArg);
    Check((*str)[str->NrItems()-2] == '\"',KLispErrInvalidArg);

    LispInt i=1;
    LispInt nr=str->NrItems()-2;
    //((*str)[i] != '\"')
    for (i=1;i<nr;i++)
    {
        aEnvironment.CurrentOutput()->PutChar((*str)[i]);
    }
    InternalTrue(aEnvironment,aResult);
}

void LispProgBody(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    // Allow accessing previous locals.
    LispLocalFrame frame(aEnvironment,LispFalse);

    InternalTrue(aEnvironment,aResult);
    
    // Evaluate args one by one.
    LispInt nrArguments = InternalListLength(aArguments);
    LispInt arg;
    for (arg=1;arg<nrArguments;arg++)
    {
        InternalEval(aEnvironment, aResult, Argument(aArguments,arg));
    }
}

static void InternalNewLocal(LispEnvironment& aEnvironment, LispPtr& aResult,
                             LispPtr& aArguments, LispBoolean aMacroMode)
{
    LispInt nrArguments = InternalListLength(aArguments);
    LispInt arg;
    for (arg=1;arg<nrArguments;arg++)
    {
        LispStringPtr variable;
        if (aMacroMode)
        {
            LispPtr result;
            InternalEval(aEnvironment, result,  Argument(aArguments,arg));
            variable = result.Get()->String();
        }
        else
        {
            variable = Argument(aArguments,arg).Get()->String();
        }
        Check(variable != NULL,KLispErrInvalidArg);
        aEnvironment.NewLocal(variable,NULL);
    }
    InternalTrue(aEnvironment,aResult);
}

void LispNewLocal(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    InternalNewLocal(aEnvironment, aResult,aArguments, LispFalse);
}

void LispMacroNewLocal(LispEnvironment& aEnvironment, LispPtr& aResult,
                       LispPtr& aArguments)
{
    InternalNewLocal(aEnvironment, aResult,aArguments, LispTrue);
}


void LispWhile(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(3);

    LispPtr predicate;
    InternalEval(aEnvironment, predicate, Argument(aArguments,1));

    while (IsTrue(aEnvironment,predicate))
    {
        LispPtr evaluated;
        InternalEval(aEnvironment, evaluated, Argument(aArguments,2));
        InternalEval(aEnvironment, predicate, Argument(aArguments,1));

    }
    Check(IsFalse(aEnvironment,predicate),KLispErrInvalidArg);
    InternalTrue(aEnvironment,aResult);
}


void LispInFix(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(3);

    // Get operator
    Check(Argument(aArguments,1).Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = Argument(aArguments,1).Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    //HIER LispString oper;
    //HIER InternalUnstringify(oper, orig);
    
    LispPtr precedence;
    InternalEval(aEnvironment, precedence, Argument(aArguments,2));
    Check(precedence.Get()->String() != NULL, KLispErrInvalidArg);
    LispInt prec = InternalAsciiToInt(precedence.Get()->String()->String());
    Check(prec < KMaxPrecedence, KLispErrInvalidArg);
    //HIER aEnvironment.InFix().SetOperator(prec,aEnvironment.HashTable().LookUp(oper.String()));
    aEnvironment.InFix().SetOperator(prec,aEnvironment.HashTable().LookUpUnStringify(orig->String()));

    InternalTrue(aEnvironment,aResult);
}



static void SingleFix(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments, LispOperators& aOps)
{
    TESTARGS(2);

    // Get operator
    Check(Argument(aArguments,1).Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = Argument(aArguments,1).Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    //HIER LispString oper;
    //HIER InternalUnstringify(oper, orig);
    aOps.SetOperator(0,aEnvironment.HashTable().LookUpUnStringify(orig->String()));
    InternalTrue(aEnvironment,aResult);
}
void LispPreFix(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    SingleFix(aEnvironment, aResult,aArguments, aEnvironment.PreFix());
}
void LispPostFix(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    SingleFix(aEnvironment, aResult,aArguments, aEnvironment.PostFix());
}
void LispBodied(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    SingleFix(aEnvironment, aResult,aArguments, aEnvironment.Bodied());
}


void LispAtomize(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(2);

    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get operator
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    //HIER LispString oper;
    //HIER InternalUnstringify(oper, orig);
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUpUnStringify(orig->String())));
}


void LispStringify(LispEnvironment& aEnvironment, LispPtr& aResult,
                   LispPtr& aArguments)
{
    TESTARGS(2);

    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get operator
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);

    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUpStringify(orig->String())));

/*TODO
    LispString oper;
    InternalStringify(oper, orig);
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(oper.String())));
*/
}




void LispLoad(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(2);
    Check(aEnvironment.iSecure == 0, KLispErrSecurityBreach);

    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get file name
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);

    InternalLoad(aEnvironment,orig);
    InternalTrue(aEnvironment,aResult);
}


static void InternalRuleBase(LispEnvironment& aEnvironment, LispPtr& aResult,
                             LispPtr& aArguments, LispBoolean aMacroMode)
{
    TESTARGS(3);
    
    // Get operator
//HIER    LispString oper;
    LispPtr args;
    LispStringPtr orig=NULL;
    
    if (aMacroMode)
    {
        LispPtr result;
        InternalEval(aEnvironment, result, Argument(aArguments,1));
        orig = result.Get()->String();
        Check(orig != NULL, KLispErrInvalidArg);
        //HIER InternalUnstringify(oper, orig);
        InternalEval(aEnvironment, args, Argument(aArguments,2));
    }
    else
    {
        Check(Argument(aArguments,1).Get() != NULL, KLispErrInvalidArg);
        orig = Argument(aArguments,1).Get()->String();
        Check(orig != NULL, KLispErrInvalidArg);
        //HIER InternalUnstringify(oper, orig);
        args.Set(Argument(aArguments,2).Get());
    }
    
    // The arguments
    CheckIsList(args);

    // Finally define the rule base
    aEnvironment.DeclareRuleBase(aEnvironment.HashTable().LookUpUnStringify(orig->String()),
                                 args.Get()->SubList()->Get()->Next());
    
    // Return LispTrue
    InternalTrue(aEnvironment,aResult);
}

void LispRuleBase(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    InternalRuleBase(aEnvironment, aResult, aArguments, LispFalse);
}
void LispMacroRuleBase(LispEnvironment& aEnvironment, LispPtr& aResult,
                       LispPtr& aArguments)
{
    InternalRuleBase(aEnvironment, aResult, aArguments, LispTrue);
}



void LispHoldArg(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(3);
    
    // Get operator
    Check(Argument(aArguments,1).Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = Argument(aArguments,1).Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
//HIER    LispString oper;
//HIER    InternalUnstringify(oper, orig);

    // The arguments
    LispStringPtr tohold = Argument(aArguments,2).Get()->String();
    Check(tohold != NULL, KLispErrInvalidArg);

    aEnvironment.HoldArgument(aEnvironment.HashTable().LookUpUnStringify(orig->String()),
                              tohold);
    // Return LispTrue
    InternalTrue(aEnvironment,aResult);
}

static void InternalNewRule(LispEnvironment& aEnvironment, LispPtr& aResult,
                            LispPtr& aArguments, LispBoolean aMacroMode)
{
    TESTARGS(6);

    //HIER LispString oper;
    LispInt arity;
    LispInt precedence;

    LispPtr ar;
    LispPtr pr;
    LispPtr predicate;
    LispPtr body;
    LispStringPtr orig=NULL;
    
    // Get operator
    if (aMacroMode)
    {
        LispPtr result;
        InternalEval(aEnvironment, result, Argument(aArguments,1));
        Check(result.Get() != NULL, KLispErrInvalidArg);
        orig = result.Get()->String();
        Check(orig != NULL, KLispErrInvalidArg);
//HIER        InternalUnstringify(oper, orig);

        InternalEval(aEnvironment, ar, Argument(aArguments,2));
        InternalEval(aEnvironment, pr, Argument(aArguments,3));
        InternalEval(aEnvironment, predicate, Argument(aArguments,4));
        InternalEval(aEnvironment, body, Argument(aArguments,5));
    }
    else
    {
        Check(Argument(aArguments,1).Get() != NULL, KLispErrInvalidArg);
        orig = Argument(aArguments,1).Get()->String();
        Check(orig != NULL, KLispErrInvalidArg);
//HIER        InternalUnstringify(oper, orig);
        ar.Set(Argument(aArguments,2).Get());
        pr.Set(Argument(aArguments,3).Get());
        predicate.Set(Argument(aArguments,4).Get());
        body.Set(Argument(aArguments,5).Get());
    }
    
    // The arity
    Check(ar.Get() != NULL, KLispErrInvalidArg);
    Check(ar.Get()->String() != NULL, KLispErrInvalidArg);
    arity = InternalAsciiToInt(ar.Get()->String()->String());

    // The precedence
    Check(pr.Get() != NULL, KLispErrInvalidArg);
    Check(pr.Get()->String() != NULL, KLispErrInvalidArg);
    precedence = InternalAsciiToInt(pr.Get()->String()->String());
    
    // Finally define the rule base
    aEnvironment.DefineRule(aEnvironment.HashTable().LookUpUnStringify(orig->String()),
                            arity,
                            precedence,
                            predicate,
                            body );

    // Return LispTrue
    InternalTrue(aEnvironment,aResult);
}

void LispNewRule(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    InternalNewRule(aEnvironment, aResult,aArguments, LispFalse);
}

void LispMacroNewRule(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    InternalNewRule(aEnvironment, aResult,aArguments, LispTrue);
}


void LispUnFence(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(3);
    
    // Get operator
//HIER    LispString oper;

    // Get operator
    Check(Argument(aArguments,1).Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = Argument(aArguments,1).Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
//HIER    InternalUnstringify(oper, orig);

    // The arity
    Check(Argument(aArguments,2).Get() != NULL, KLispErrInvalidArg);
    Check(Argument(aArguments,2).Get()->String() != NULL, KLispErrInvalidArg);
    LispInt arity = InternalAsciiToInt(Argument(aArguments,2).Get()->String()->String());

    aEnvironment.UnFenceRule(aEnvironment.HashTable().LookUpUnStringify(orig->String()),
                            arity);
    
    // Return LispTrue
    InternalTrue(aEnvironment,aResult);
}



void LispIsFunction(LispEnvironment& aEnvironment,LispPtr& aResult,
              LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr result;
    InternalEval(aEnvironment, result, Argument(aArguments,1));
    InternalBoolean(aEnvironment,aResult,
                    result.Get()->SubList()!=NULL);
}
void LispIsAtom(LispEnvironment& aEnvironment,LispPtr& aResult,
              LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr result;
    InternalEval(aEnvironment, result, Argument(aArguments,1));
    InternalBoolean(aEnvironment,aResult,
                    result.Get()->String()!=NULL);
}
void LispIsNumber(LispEnvironment& aEnvironment,LispPtr& aResult,
              LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr result;
    InternalEval(aEnvironment, result, Argument(aArguments,1));
    if (result.Get()->String() == NULL)
    {
        InternalFalse(aEnvironment,aResult);
    }
    else
    {
        InternalBoolean(aEnvironment,aResult,
                        IsNumber(result.Get()->String()->String(),LispTrue));
    }
}

void LispIsInteger(LispEnvironment& aEnvironment,LispPtr& aResult,
              LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr result;
    InternalEval(aEnvironment, result, Argument(aArguments,1));
    if (result.Get()->String() == NULL)
    {
        InternalFalse(aEnvironment,aResult);
    }
    else
    {
        InternalBoolean(aEnvironment,aResult,
                        IsNumber(result.Get()->String()->String(),LispFalse));
    }
}


void LispIsList(LispEnvironment& aEnvironment,LispPtr& aResult,
              LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr result;
    InternalEval(aEnvironment, result, Argument(aArguments,1));
    InternalBoolean(aEnvironment,aResult,InternalIsList(result));
}


void LispIsString(LispEnvironment& aEnvironment,LispPtr& aResult,
              LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr result;
    InternalEval(aEnvironment, result, Argument(aArguments,1));
    InternalBoolean(aEnvironment,aResult,
                    InternalIsString(result.Get()->String()));
}

void LispIsBound(LispEnvironment& aEnvironment,LispPtr& aResult,
              LispPtr& aArguments)
{
    TESTARGS(2);
    LispStringPtr str = Argument(aArguments,1).Get()->String();
    if (str)
    {
        LispPtr* val = aEnvironment.GetVariable(str);
        if (val)
            if (val->Get())
            {
                InternalTrue(aEnvironment,aResult);
                return;
            }
    }
    InternalFalse(aEnvironment,aResult);
}



void LispIf(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispInt nrArguments = InternalListLength(aArguments);
    Check(nrArguments == 3 || nrArguments == 4,KLispErrWrongNumberOfArgs);

    LispPtr predicate;
    InternalEval(aEnvironment, predicate, Argument(aArguments,1));

    if (IsTrue(aEnvironment,predicate))
    {
        InternalEval(aEnvironment, aResult, Argument(aArguments,2));
    }
    else
    {
        Check(IsFalse(aEnvironment,predicate),KLispErrInvalidArg);
        if (nrArguments == 4)
        {
            InternalEval(aEnvironment, aResult, Argument(aArguments,3));
        }
        else
        {
            InternalFalse(aEnvironment,aResult);
        }
    }
}



void LispTryRetract(LispEnvironment& aEnvironment, LispPtr& aResult,
                 LispPtr& aArguments)
{
    TESTARGS(3);

    // Get operator
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    LispString oper;
    InternalUnstringify(oper, orig);
    
    LispPtr arity;
    InternalEval(aEnvironment, arity, Argument(aArguments,2));
    Check(arity.Get()->String() != NULL, KLispErrInvalidArg);
    LispInt ar = InternalAsciiToInt(arity.Get()->String()->String());
    aEnvironment.TryRetract(aEnvironment.HashTable().LookUp(oper.String()), ar);
    InternalTrue(aEnvironment,aResult);
}


void LispPrecision(LispEnvironment& aEnvironment, LispPtr& aResult,
                   LispPtr& aArguments)
{
    TESTARGS(2);

    LispPtr index;
    InternalEval(aEnvironment, index, Argument(aArguments,1));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);

    LispInt ind = InternalAsciiToInt(index.Get()->String()->String());
    aEnvironment.SetPrecision(ind);
    InternalTrue(aEnvironment,aResult);
}



void LispDefaultDirectory(LispEnvironment& aEnvironment, LispPtr& aResult,
                          LispPtr& aArguments)
{
    TESTARGS(2);

    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get file name
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    LispString oper;
    InternalUnstringify(oper, orig);
    aEnvironment.iInputDirectories.Append(new LispString(oper.String()));
    InternalTrue(aEnvironment,aResult);
}


void LispFromFile(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(3);

    Check(aEnvironment.iSecure == 0, KLispErrSecurityBreach);
    
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get file name
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
//HIER    LispString oper;
//HIER    InternalUnstringify(oper, orig);

    LispStringPtr hashedname = aEnvironment.HashTable().LookUpUnStringify(orig->String());
    LispRamFile* ramFile=aEnvironment.iRamDisk.LookUp(hashedname);

    InputStatus oldstatus = aEnvironment.iInputStatus;
    aEnvironment.iInputStatus.SetTo(hashedname->String());

    if (ramFile != NULL)
    {
        StringInput newInput(*(ramFile->Contents()),aEnvironment.iInputStatus);
        LispLocalInput localInput(aEnvironment, &newInput);

        // Evaluate the body
        InternalEval(aEnvironment, aResult, Argument(aArguments,2));
    }
    else
    {
        //TODO make the file api platform independent!!!!
        // Open file
        LispLocalFile localFP(aEnvironment, hashedname->String(),LispTrue,
                              aEnvironment.iInputDirectories);
        Check(localFP.iOpened != 0, KLispErrFileNotFound);
        FILEINPUT newInput(localFP,aEnvironment.iInputStatus);
        LispLocalInput localInput(aEnvironment, &newInput);

        // Evaluate the body
        InternalEval(aEnvironment, aResult, Argument(aArguments,2));
    }
    aEnvironment.iInputStatus.RestoreFrom(oldstatus);
    //Return the result
}


void LispFromString(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(3);

    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get file name
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    LispString oper;
    InternalUnstringify(oper, orig);

    InputStatus oldstatus = aEnvironment.iInputStatus;
    aEnvironment.iInputStatus.SetTo("String");
    StringInput newInput(oper,aEnvironment.iInputStatus);
    LispLocalInput localInput(aEnvironment, &newInput);

    // Evaluate the body
    InternalEval(aEnvironment, aResult, Argument(aArguments,2));
    aEnvironment.iInputStatus.RestoreFrom(oldstatus);

    //Return the result
}


void LispRead(LispEnvironment& aEnvironment, LispPtr& aResult,
              LispPtr& aArguments)
{
    LispTokenizer tok;
    InfixParser parser(tok,
                       *aEnvironment.CurrentInput(),
                       aEnvironment.HashTable(),
                       aEnvironment.PreFix(),
                       aEnvironment.InFix(),
                       aEnvironment.PostFix(),
                       aEnvironment.Bodied());
    // Read expression
    parser.Parse(aResult);
}


void LispReadToken(LispEnvironment& aEnvironment, LispPtr& aResult,
                   LispPtr& aArguments)
{
    LispTokenizer tok;
    LispStringPtr result;
    result = tok.NextToken(*aEnvironment.CurrentInput(),
                           aEnvironment.HashTable());
    aResult.Set(LispAtom::New(result));
}


void LispToFile(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(3);
    Check(aEnvironment.iSecure == 0, KLispErrSecurityBreach);

    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get file name
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    LispString oper;
    InternalUnstringify(oper, orig);

    //TODO make the file api platform independent!!!!
    // Open file for writing
    LispLocalFile localFP(aEnvironment, oper.String(),LispFalse,
                          aEnvironment.iInputDirectories);
    Check(localFP.iOpened != 0, KLispErrFileNotFound);
    StdFileOutput newOutput(localFP);
    LispLocalOutput localOutput(aEnvironment, &newOutput);

    // Evaluate the body
    InternalEval(aEnvironment, aResult, Argument(aArguments,2));

    //Return the result
}



void LispCheck(LispEnvironment& aEnvironment,LispPtr& aResult,
               LispPtr& aArguments)
{
    TESTARGS(3);
    CheckIsString(Argument(aArguments,2));
    InternalEval(aEnvironment, aResult, Argument(aArguments,1));
    if (!IsTrue(aEnvironment,aResult))
    {
        aEnvironment.SetUserError(Argument(aArguments,2).Get()->String()->String());
        Check(0,KLispErrUser);
    }
}



void LispSystemCall(LispEnvironment& aEnvironment,LispPtr& aResult,
               LispPtr& aArguments)
{
    TESTARGS(2);
    Check(aEnvironment.iSecure == 0, KLispErrSecurityBreach);

    LispPtr result;
    InternalEval(aEnvironment, result, Argument(aArguments,1));
    CheckIsString(result);

    LispString command;
    InternalUnstringify(command, result.Get()->String());
    system(command.String());

    InternalTrue(aEnvironment,aResult);
}





void LispFastSin(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatSin);
}

void LispFastCos(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatCos);
}
void LispFastTan(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatTan);
}

void LispFastArcSin(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatArcSin);
}
void LispFastArcCos(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatArcCos);
}
void LispFastArcTan(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatArcTan);
}
void LispFastExp(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatExp);
}
void LispFastLog(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatLn);
}

void LispFastPower(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, PlatPower);
}

void LispFastSqrt(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatSqrt);
/*TODO
    TESTARGS(2);
    LispPtr result1;
    InternalEval(aEnvironment, result1, Argument(aArguments,1));
    LispStringPtr str1;
    str1 = result1.Get()->String();
    Check(str1 != NULL,KLispErrInvalidArg);
    Check(IsNumber(str1->String(),LispTrue),KLispErrInvalidArg);
    PlatSqrt(aResult,result1, aEnvironment);
    */
}

void LispFastPi(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(1);
    aResult.Set(LispAtom::New(PlatPi(aEnvironment.HashTable(),
                                         aEnvironment.Precision())));
}
void LispFastFloor(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatFloor);
}

void LispFastCeil(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatCeil);
}
void LispFastMod(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, PlatMod);
}

void LispFastAbs(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic1(aEnvironment, aResult, aArguments, PlatAbs);
}













void LispShiftLeft(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, ShiftLeft);
}
void LispShiftRight(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, ShiftRight);
}


void LispFromBase(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, FromBase,LispTrue);
}
void LispToBase(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, ToBase);
}


void LispMaxEvalDepth(LispEnvironment& aEnvironment, LispPtr& aResult,
                      LispPtr& aArguments)
{
    TESTARGS(2);

    LispPtr index;
    InternalEval(aEnvironment, index, Argument(aArguments,1));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);

    LispInt ind = InternalAsciiToInt(index.Get()->String()->String());
    aEnvironment.iMaxEvalDepth = ind;
    InternalTrue(aEnvironment,aResult);
}


void LispDefLoad(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(2);
    Check(aEnvironment.iSecure == 0, KLispErrSecurityBreach);

    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get file name
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);

    LoadDefFile(aEnvironment, orig);
    InternalTrue(aEnvironment,aResult);
}

void LispUse(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(2);
    Check(aEnvironment.iSecure == 0, KLispErrSecurityBreach);

    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get file name
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);

    InternalUse(aEnvironment,orig);
    InternalTrue(aEnvironment,aResult);
}

void LispRightAssociative(LispEnvironment& aEnvironment, LispPtr& aResult,
                          LispPtr& aArguments)
{
    TESTARGS(2);
    // Get operator
    Check(Argument(aArguments,1).Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = Argument(aArguments,1).Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
//HIER    LispString oper;
//HIER    InternalUnstringify(oper, orig);
    aEnvironment.InFix().SetRightAssociative(aEnvironment.HashTable().LookUpUnStringify(orig->String()));
    InternalTrue(aEnvironment,aResult);
}


void LispLeftPrecedence(LispEnvironment& aEnvironment, LispPtr& aResult,
                          LispPtr& aArguments)
{
    TESTARGS(3);
    // Get operator
    Check(Argument(aArguments,1).Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = Argument(aArguments,1).Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
//HIER    LispString oper;
//HIER    InternalUnstringify(oper, orig);

    LispPtr index;
    InternalEval(aEnvironment, index, Argument(aArguments,2));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);
    LispInt ind = InternalAsciiToInt(index.Get()->String()->String());

    aEnvironment.InFix().SetLeftPrecedence(aEnvironment.HashTable().LookUpUnStringify(orig->String()),ind);
    InternalTrue(aEnvironment,aResult);
}


void LispRightPrecedence(LispEnvironment& aEnvironment, LispPtr& aResult,
                          LispPtr& aArguments)
{
    TESTARGS(3);
    // Get operator
    Check(Argument(aArguments,1).Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = Argument(aArguments,1).Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
//HIER    LispString oper;
//HIER    InternalUnstringify(oper, orig);

    LispPtr index;
    InternalEval(aEnvironment, index, Argument(aArguments,2));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);
    LispInt ind = InternalAsciiToInt(index.Get()->String()->String());

    aEnvironment.InFix().SetRightPrecedence(aEnvironment.HashTable().LookUpUnStringify(orig->String()),ind);
    InternalTrue(aEnvironment,aResult);
}
















//hier
static LispInFixOperator* OperatorInfo(LispEnvironment& aEnvironment,
                                       LispPtr& aArguments,
                                       LispOperators & aOperators)
{
    TESTARGS(2);
    // Get operator
    Check(Argument(aArguments,1).Get() != NULL, KLispErrInvalidArg);

    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);


    //
    LispInFixOperator* op = aOperators.LookUp(
                                          aEnvironment.HashTable().LookUpUnStringify(orig->String()));
    return op;
}


void LispIsInFix(LispEnvironment& aEnvironment, LispPtr& aResult,
               LispPtr& aArguments)
{

    LispInFixOperator* op = OperatorInfo(aEnvironment,
                                         aArguments,
                                         aEnvironment.InFix());
    if (op != NULL)
        InternalTrue( aEnvironment, aResult);
    else
        InternalFalse(aEnvironment, aResult);
}

void LispGetPrecedence(LispEnvironment& aEnvironment, LispPtr& aResult,
                       LispPtr& aArguments)
{

    LispInFixOperator* op = OperatorInfo(aEnvironment,
                                         aArguments,
                                         aEnvironment.InFix());
    Check(op!=NULL, KLispErrIsNotInFix);

    LispChar buf[30];
    InternalIntToAscii(buf, op->iPrecedence);
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(buf)));
}


void LispGetLeftPrecedence(LispEnvironment& aEnvironment, LispPtr& aResult,
                           LispPtr& aArguments)
{

    LispInFixOperator* op = OperatorInfo(aEnvironment,
                                         aArguments,
                                         aEnvironment.InFix());
    Check(op!=NULL, KLispErrIsNotInFix);

    LispChar buf[30];
    InternalIntToAscii(buf, op->iLeftPrecedence);
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(buf)));
}
void LispGetRightPrecedence(LispEnvironment& aEnvironment, LispPtr& aResult,
                            LispPtr& aArguments)
{

    LispInFixOperator* op = OperatorInfo(aEnvironment,
                                         aArguments,
                                         aEnvironment.InFix());
    Check(op!=NULL, KLispErrIsNotInFix);

    LispChar buf[30];
    InternalIntToAscii(buf, op->iRightPrecedence);
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(buf)));
}



void LispIsPreFix(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{

    LispInFixOperator* op = OperatorInfo(aEnvironment,
                                         aArguments,
                                         aEnvironment.PreFix());
    if (op != NULL)
        InternalTrue( aEnvironment, aResult);
    else
        InternalFalse(aEnvironment, aResult);
}

void LispIsPostFix(LispEnvironment& aEnvironment, LispPtr& aResult,
                   LispPtr& aArguments)
{

    LispInFixOperator* op = OperatorInfo(aEnvironment,
                                         aArguments,
                                         aEnvironment.PostFix());
    if (op != NULL)
        InternalTrue( aEnvironment, aResult);
    else
        InternalFalse(aEnvironment, aResult);
}

void LispGetPrecision(LispEnvironment& aEnvironment, LispPtr& aResult,
                      LispPtr& aArguments)
{
    TESTARGS(1);
    LispChar buf[30];
    InternalIntToAscii(buf, aEnvironment.Precision());
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(buf)));
}

void LispBitAnd(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, BitAnd);
}
void LispBitOr(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, BitOr);
}
void LispBitXor(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispArithmetic2(aEnvironment, aResult, aArguments, BitXor);
}


void LispToString(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    TESTARGS(2);

    LispString oper;
    StringOutput newOutput(oper);

    LispLocalOutput localOutput(aEnvironment, &newOutput);

    // Evaluate the body
    InternalEval(aEnvironment, aResult, Argument(aArguments,1));

    //Return the result
//HIER    LispString result;
//HIER    InternalStringify(result, &oper);
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUpStringify(oper.String())));
}

void LispSecure(LispEnvironment& aEnvironment,LispPtr& aResult,
              LispPtr& aArguments)
{
    TESTARGS(2);
    LispSecureFrame security(aEnvironment);
    InternalEval(aEnvironment, aResult, Argument(aArguments,1));
}


void LispFindFile(LispEnvironment& aEnvironment,LispPtr& aResult,
              LispPtr& aArguments)
{

    TESTARGS(2);

    Check(aEnvironment.iSecure == 0, KLispErrSecurityBreach);
    
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get file name
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    LispString oper;
    InternalUnstringify(oper, orig);

    LispChar filename[1024];//TODO FIXME
    InternalFindFile(oper.String(), aEnvironment.iInputDirectories,
                     filename);

    LispString res(filename,1);
    

    //HIER LispString result;
    //HIER InternalStringify(result, &res);
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUpStringify(res.String())));
}


void LispIsGeneric(LispEnvironment& aEnvironment,LispPtr& aResult,
                   LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    if (evaluated.Get()->Generic() != NULL)
        InternalTrue( aEnvironment, aResult);
    else
        InternalFalse(aEnvironment, aResult);
}

void LispGenericTypeName(LispEnvironment& aEnvironment,LispPtr& aResult,
                         LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    Check(evaluated.Get()->Generic() != NULL,KLispErrInvalidArg);

    LispCharPtr name = evaluated.Get()->Generic()->TypeName();
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(name)));
}

void GenArrayCreate(LispEnvironment& aEnvironment,LispPtr& aResult,
                    LispPtr& aArguments)
{
    TESTARGS(3);

    LispPtr sizearg;
    InternalEval(aEnvironment, sizearg, Argument(aArguments,1));

    Check(sizearg.Get() != NULL, KLispErrInvalidArg);
    Check(sizearg.Get()->String() != NULL, KLispErrInvalidArg);

    LispInt size = InternalAsciiToInt(sizearg.Get()->String()->String());

    LispPtr initarg;
    InternalEval(aEnvironment, initarg, Argument(aArguments,2));
     
    ArrayClass *array = new ArrayClass(size,initarg.Get());
    aResult.Set(LispGenericClass::New(array));
}

void GenArraySize(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    GenericClass *gen = evaluated.Get()->Generic();
    Check(gen != NULL,KLispErrInvalidArg);
    Check(StrEqual(gen->TypeName(),"\"Array\""),KLispErrInvalidArg);
    LispInt size=((ArrayClass*)gen)->Size();
    LispChar s[20];
    InternalIntToAscii(s,size);
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(s)));
}

void GenArrayGet(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(3);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    GenericClass *gen = evaluated.Get()->Generic();
    Check(gen != NULL,KLispErrInvalidArg);
    Check(StrEqual(gen->TypeName(),"\"Array\""),KLispErrInvalidArg);

    LispPtr sizearg;
    InternalEval(aEnvironment, sizearg, Argument(aArguments,2));

    Check(sizearg.Get() != NULL, KLispErrInvalidArg);
    Check(sizearg.Get()->String() != NULL, KLispErrInvalidArg);

    LispInt size = InternalAsciiToInt(sizearg.Get()->String()->String());


    Check(size>0 && size<=((ArrayClass*)gen)->Size(),KLispErrInvalidArg);
    LispObject* object = ((ArrayClass*)gen)->GetElement(size);

    aResult.Set(object->Copy(LispFalse));
}


void GenArraySet(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(4);

    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    GenericClass *gen = evaluated.Get()->Generic();
    Check(gen != NULL,KLispErrInvalidArg);
    Check(StrEqual(gen->TypeName(),"\"Array\""),KLispErrInvalidArg);

    LispPtr sizearg;
    InternalEval(aEnvironment, sizearg, Argument(aArguments,2));

    Check(sizearg.Get() != NULL, KLispErrInvalidArg);
    Check(sizearg.Get()->String() != NULL, KLispErrInvalidArg);

    LispInt size = InternalAsciiToInt(sizearg.Get()->String()->String());
    Check(size>0 && size<=((ArrayClass*)gen)->Size(),KLispErrInvalidArg);

    LispPtr obj;
    InternalEval(aEnvironment, obj, Argument(aArguments,3));
    ((ArrayClass*)gen)->SetElement(size,obj.Get());
    InternalTrue( aEnvironment, aResult);
}

void LispTrace(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(2);
    LispLocalEvaluator local(aEnvironment,new TracedEvaluator);
    InternalEval(aEnvironment, aResult, Argument(aArguments,1));
}


void LispReadLisp(LispEnvironment& aEnvironment, LispPtr& aResult,
                  LispPtr& aArguments)
{
    LispTokenizer tok;
    LispParser parser(tok,
                      *aEnvironment.CurrentInput(),
                      aEnvironment.HashTable());
    // Read expression
    parser.Parse(aResult);
}


void LispTraceRule(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(3);
    LispPtr *ptr = aArguments.Get()->Next().Get()->SubList();
    LispUserFunction* userfunc=NULL;
    if (ptr != NULL)
        userfunc = GetUserFunction(aEnvironment,ptr);
    LispLocalTrace trace(userfunc);
    InternalEval(aEnvironment, aResult, Argument(aArguments,2));
}

void LispType(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(2);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));
    LispPtr* subList = evaluated.Get()->SubList();
    LispObject* head = NULL;
    if (!subList)
    {
        goto EMPTY;
    }
    head = subList->Get();
    if (!head->String())
        goto EMPTY;
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUpStringify(head->String()->String())));
    return;
    
EMPTY:
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp("\"\"")));
    return;
}



void LispStringMid(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(4);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,3));
    CheckIsString(evaluated);
    LispStringPtr orig = evaluated.Get()->String();
    
    LispPtr index;
    InternalEval(aEnvironment, index, Argument(aArguments,1));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);
    LispInt from = InternalAsciiToInt(index.Get()->String()->String());

    Check(from>0,KLispErrInvalidArg);
    
    InternalEval(aEnvironment, index, Argument(aArguments,2));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);
    LispInt count = InternalAsciiToInt(index.Get()->String()->String());

    
    LispString str;
    str.SetNrItems(0);
    str.Append('\"');
    LispInt i;
    Check(from+count<orig->NrItems()-1, KLispErrInvalidArg);
    for (i=from;i<from+count;i++)
        str.Append((*orig)[i]);
    str.Append('\"');
    str.Append('\0');
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(str.String())));
}


void LispSetStringMid(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(4);
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,3));
    CheckIsString(evaluated);
    LispStringPtr orig = evaluated.Get()->String();
    LispPtr index;
    InternalEval(aEnvironment, index, Argument(aArguments,1));
    Check(index.Get() != NULL, KLispErrInvalidArg);
    Check(index.Get()->String() != NULL, KLispErrInvalidArg);
    LispInt from = InternalAsciiToInt(index.Get()->String()->String());

    Check(from>0,KLispErrInvalidArg);
    
    LispPtr ev2;
    InternalEval(aEnvironment, ev2, Argument(aArguments,2));
    CheckIsString(ev2);
    LispStringPtr replace = ev2.Get()->String();

    LispString str(orig->String());
    LispInt i;
    LispInt count = replace->NrItems();
    Check(from+count-3<orig->NrItems()-1, KLispErrInvalidArg);

    for (i=0;i<count-3;i++)
        str[i+from] = (*replace)[i+1];
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(str.String())));
}



void LispFindFunction(LispEnvironment& aEnvironment,LispPtr& aResult,
                      LispPtr& aArguments)
{
    TESTARGS(2);
    Check(aEnvironment.iSecure == 0, KLispErrSecurityBreach);
    
    LispPtr evaluated;
    InternalEval(aEnvironment, evaluated, Argument(aArguments,1));

    // Get file name
    Check(evaluated.Get() != NULL, KLispErrInvalidArg);
    LispStringPtr orig = evaluated.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    LispString oper;
    InternalUnstringify(oper, orig);

    LispMultiUserFunction* multiUserFunc =
        aEnvironment.MultiUserFunction(aEnvironment.HashTable().LookUp(oper.String()));
    if (multiUserFunc != NULL)
    {
        LispDefFile* def = multiUserFunc->iFileToOpen;
        if (def != NULL)
        {
            //HIER LispString result;
            //HIER InternalStringify(result, &res);
            aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp(def->iFileName->String())));
            return;
        }
    }
    aResult.Set(LispAtom::New(aEnvironment.HashTable().LookUp("\"\"")));
}




















void GenPatternCreate(LispEnvironment& aEnvironment,LispPtr& aResult,
                      LispPtr& aArguments)
{
    TESTARGS(3);
    LispPtr pattern;
    InternalEval(aEnvironment, pattern, Argument(aArguments,1));
    LispPtr postpredicate;
    InternalEval(aEnvironment, postpredicate, Argument(aArguments,2));

    LispIterator iter(pattern);
    Check(iter() != NULL,KLispErrInvalidArg);
    Check(iter()->SubList() != NULL,KLispErrInvalidArg);
    iter.GoSub();
    Check(iter() != NULL,KLispErrInvalidArg);
    iter.GoNext();

    LispPtr *ptr = iter.Ptr();


    YacasPatternPredicateBase* matcher =
        new YacasPatternPredicateBase(aEnvironment, *ptr,postpredicate);
    PatternClass *p = new PatternClass(matcher);
    aResult.Set(LispGenericClass::New(p));
}
void GenPatternMatches(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(3);
    LispPtr pattern;
    InternalEval(aEnvironment, pattern, Argument(aArguments,1));
    GenericClass *gen = pattern.Get()->Generic();
    Check(gen != NULL,KLispErrInvalidArg);
    Check(StrEqual(gen->TypeName(),"\"Pattern\""),KLispErrInvalidArg);

    LispPtr list;
    InternalEval(aEnvironment, list, Argument(aArguments,2));

    PatternClass *patclass = (PatternClass*)gen;

    LispIterator iter(list);
    Check(iter() != NULL,KLispErrInvalidArg);
    Check(iter()->SubList() != NULL,KLispErrInvalidArg);
    iter.GoSub();
    Check(iter() != NULL,KLispErrInvalidArg);
    iter.GoNext();

    LispPtr *ptr = iter.Ptr();
    Check(ptr != NULL,KLispErrInvalidArg);
    InternalBoolean(aEnvironment,aResult,patclass->Matches(aEnvironment,*ptr));
}

void LispRuleBaseDefined(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(3);
    LispPtr name;
    InternalEval(aEnvironment, name, Argument(aArguments,1));
    LispStringPtr orig = name.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    LispString oper;
    InternalUnstringify(oper, orig);

    LispPtr sizearg;
    InternalEval(aEnvironment, sizearg, Argument(aArguments,2));
    Check(sizearg.Get() != NULL, KLispErrInvalidArg);
    Check(sizearg.Get()->String() != NULL, KLispErrInvalidArg);

    LispInt arity = InternalAsciiToInt(sizearg.Get()->String()->String());

    LispUserFunction* userFunc = aEnvironment.UserFunction(aEnvironment.HashTable().LookUp(oper.String()),arity);
    InternalBoolean(aEnvironment,aResult,userFunc != NULL);
}

void LispRuleBaseArgList(LispEnvironment& aEnvironment,LispPtr& aResult, LispPtr& aArguments)
{
    TESTARGS(3);
    LispPtr name;
    InternalEval(aEnvironment, name, Argument(aArguments,1));
    LispStringPtr orig = name.Get()->String();
    Check(orig != NULL, KLispErrInvalidArg);
    LispString oper;
    InternalUnstringify(oper, orig);

    LispPtr sizearg;
    InternalEval(aEnvironment, sizearg, Argument(aArguments,2));
    Check(sizearg.Get() != NULL, KLispErrInvalidArg);
    Check(sizearg.Get()->String() != NULL, KLispErrInvalidArg);

    LispInt arity = InternalAsciiToInt(sizearg.Get()->String()->String());

    LispUserFunction* userFunc = aEnvironment.UserFunction(aEnvironment.HashTable().LookUp(oper.String()),arity);
    Check(userFunc != NULL, KLispErrInvalidArg);

    LispPtr& list = userFunc->ArgList();
    LispPtr head;
    head.Set(LispAtom::New(aEnvironment.HashTable().LookUp("List")));
    head.Get()->Next().Set(list.Get());
    aResult.Set(LispSubList::New(head.Get()));
}


