#ifndef __autil_config_data__
#define __autil_config_data__

#include "exception.hh"
#include "string_map.hh"

// The ConfigData class is used to hold configuration information.
// it has a set of keys which it will except.  Inserting are even
// trying to look at a key that it does not know will produce
// an error.  Extra accepted keys can be added with the set_extra 
// method.

namespace autil {

  struct UnknownKey : public Exception {
    string key;
    UnknownKey(const string & k);
  };

  struct KeyInfo {
    const char * name;
    enum Type {Bool, String, Int, List};
    Type  type;
    const char * def;
    const char * desc; // null if internal value
   };

  class ConfigData : public StringMap {
  public:
    struct Module {
      const char * name;
      const KeyInfo * begin;
      const KeyInfo * end;
    };
  
  private:
    class PossibleElementsEmul;
    friend class PossibleElementsEmul;

    string          name_;
    const KeyInfo * main_begin;
    const KeyInfo * main_end;
    const KeyInfo * extra_begin;
    const KeyInfo * extra_end;
    const Module  * modules_begin;
    const Module  * modules_end;

    void fancy_add(const string & key, const string & value, AddAction);
    void simple_add(const string & key, const string & value, AddAction);
    
  public:
    ConfigData(const string & n,
	       const KeyInfo  * mainbegin, const KeyInfo * mainend);
    ConfigData(const string & n,
	       const KeyInfo  * mainbegin, const KeyInfo * mainend,
	       const Module * modbegin, const Module * modend);

    const char * name() const {return name_.c_str();}

    void set_extra(const KeyInfo * begin, const KeyInfo * end);

    static const char * base_name(const char * name);
    const KeyInfo * keyinfo(const string & key) const;
    VirEmulation<const KeyInfo *> * possible_elements(bool include_extra = true) const;
    
    string get_default(const string & key) const;

    // these methods, unlike lookup will
    // a) return the default if the value is not set
    // b) give an error if the key is not requested as known
    // c) give an error if the value is not in the right format
    string retrieve (const string & key) const;
    string retrieve_list (const string & key) const;
    void retrieve_list (const string & key, MutableContainer &) const;
    bool retrieve_bool(const string & key) const;
    int retrieve_int(const string & key) const;
    
    void insert(const string & key, const string & value);
    void replace(const string & key, const string & value);
    
    void write_to_stream(ostream & out, bool include_extra = false) const;

  };

  extern const ConfigData empty_config_data;

}

#endif
