/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPModule.cpp

	Contains:	Implementation of RTPModule object. 
	
	$Log: RTPModule.cpp,v $
	Revision 1.2  1999/02/19 23:07:23  ds
	Created
	

*/

#include "RTPModule.h"
#include "RTPSession.h"
#include "RTSPSession.h"
#include "OS.h"


RTPModule**		RTPModuleInterface::sRTPModuleArray = NULL;
UInt32			RTPModuleInterface::sNumRTPModules = 0;	

StrPtrLen 		RTPModule::sDefaultName("Default", 7);

StrPtrLen RTPModuleInterface::sPublicHeader("DESCRIBE, SETUP, PLAY, PAUSE, TEARDOWN, OPTIONS", 47);


RTSPProtocol::RTSPStatusCode RTPModuleInterface::DoDescribe(RTSPRequestInterface* inRequest)
{
	Assert(sRTPModuleArray != NULL);
	Assert(inRequest != NULL);
	void* theCookie = NULL;
	RTSPSessionInterface* theRTSPSession = NULL;

	if (inRequest->GetSession()->GetCookieA() != NULL)
		((RTPModule*)inRequest->GetSession()->GetCookieA())->DestroyCookie(inRequest->GetSession()->GetCookieB());

	for (UInt32 x = 0; x < sNumRTPModules; x++)
	{
		if ((NULL != sRTPModuleArray[x]) && (sRTPModuleArray[x]->IsRequestProcessingModule()))
		{
			RTSPProtocol::RTSPStatusCode theErr = sRTPModuleArray[x]->Describe(inRequest, &theCookie);
			if (theErr != RTSPProtocol::kSuccessOK)
				return theErr;
			
			//if this module associates a cookie with the response, make sure to
			//store that cookie in the RTSP session so we can grab it later.
			if (inRequest->HasResponseBeenSent())
			{
				theRTSPSession = inRequest->GetSession();
				
				if ( theCookie != NULL )
				{						
					//set cookie A to be the target module
					theRTSPSession->SetCookieA(sRTPModuleArray[x]);
					theRTSPSession->SetCookieB(theCookie);
				}
					
				return RTSPProtocol::kSuccessOK;
			}
			//+ rt 6.17.99
			else if ( theCookie != NULL )
			{	
				// use cookie owning session to destroy the old cookie + 6.25.99 rt
				sRTPModuleArray[x]->DestroyCookie(theCookie);
				//printf( "\n !HasResponseBeenSent(), new filesession cookie destroyed -- would have leaked!" );
			}
			//-
		}
	}
	
	return RTSPProtocol::kSuccessOK;
}


RTSPProtocol::RTSPStatusCode RTPModuleInterface::NewSession(RTSPRequestInterface* inRequest, RTPSession* inSessionP)
{
	Assert(sRTPModuleArray != NULL);
	Assert(inRequest != NULL);
	Assert(inSessionP != NULL);
	void* theCookie = NULL;

	for (UInt32 x = 0; x < sNumRTPModules; x++)
	{
		if ((NULL != sRTPModuleArray[x]) && (sRTPModuleArray[x]->IsRequestProcessingModule()))
		{
			//if this module associated a cookie with its describe, make sure to pass that
			//cookie back into the module
			if (inRequest->GetSession()->GetCookieA() == sRTPModuleArray[x])
			{
				theCookie = inRequest->GetSession()->GetCookieB();
				//the module is now responsible for this cookie, so we can clear it
				inRequest->GetSession()->SetCookieA(NULL);
				inRequest->GetSession()->SetCookieB(NULL);
			}
			RTSPProtocol::RTSPStatusCode theErr = sRTPModuleArray[x]->NewSession(inRequest, inSessionP, theCookie);
			
			if ((theErr != RTSPProtocol::kSuccessOK) || (inSessionP->IsBound()))
			{
				return theErr;
			}
		}
	}
	return RTSPProtocol::kSuccessOK;
}

void RTPModuleInterface::ExecuteRTCPModules(RTPStream* inStream, StrPtrLen* inPacket)
{
	Assert(sRTPModuleArray != NULL);
	Assert(inPacket != NULL);
	Assert(inStream != NULL);

	for (UInt32 x = 0; x < sNumRTPModules; x++)
	{
		if ((NULL != sRTPModuleArray[x]) && (sRTPModuleArray[x]->IsRTCPProcessingModule()))
			sRTPModuleArray[x]->ProcessRTCPPacket(inStream, inPacket);
	}
}

void RTPModuleInterface::PreProcessRTSPRequest(RTSPRequestInterface* inRTSPRequest, RTPSession* inSession)
{
	//Pre-process the inRTSPRequest
	for (UInt32 preModIdx = 0; preModIdx < sNumRTPModules; preModIdx++)
	{
		if ( NULL != sRTPModuleArray[preModIdx] && sRTPModuleArray[preModIdx]->IsPreProcessingModule() )
			sRTPModuleArray[preModIdx]->PreProcessRTSPRequest(inRTSPRequest, inSession);
	}
}

void RTPModuleInterface::PostProcessRTSPRequest(RTSPRequestInterface* inRTSPRequest, RTPSession* inSession)
{
	//Post-process the inRTSPRequest
	for (UInt32 postModIdx = 0; postModIdx < sNumRTPModules; postModIdx++)
	{
		if ( NULL != sRTPModuleArray[postModIdx] && sRTPModuleArray[postModIdx]->IsPostProcessingModule() )
			sRTPModuleArray[postModIdx]->PostProcessRTSPRequest(inRTSPRequest, inSession);
	}
}

void RTPModuleInterface::ProcessTimeout(RTPSession* inSession)
{
	for (UInt32 timeModIdx = 0; timeModIdx < sNumRTPModules; timeModIdx++)
	{
		if ( NULL != sRTPModuleArray[timeModIdx] && sRTPModuleArray[timeModIdx]->IsTimeoutProcessingModule() )
			sRTPModuleArray[timeModIdx]->ProcessTimeout(inSession);
	}
}

RTSPProtocol::RTSPStatusCode RTPModuleInterface::DoSetup(RTSPRequestInterface *inRTSPRequest, StrPtrLen* inSessionID)
{
	RTSPProtocol::RTSPStatusCode rtspError = RTSPProtocol::kSuccessOK;
	
	//if there is no session ID for this setup, this must be a new session
	if (( inSessionID == NULL ) || ( inSessionID->Len == 0 ))
	{
		//this is a brand spanking new session. At this point, we need to create
		//a new RTPSession object that will represent this session until it completes.
		//Then, we need to pass the session onto one of the modules

		//First of all, ask the server if it's ok to add a new session
		rtspError = RTPServerInterface::IsOkToAddNewSession(inRTSPRequest);
		if (rtspError != RTSPProtocol::kSuccessOK)
			return rtspError;

		RTPSession* theRTPSession = new ('Rses') RTPSession();
		rtspError = RTPModuleInterface::NewSession(inRTSPRequest, theRTPSession);
		
		//none of the RTP modules wanted this session.
		if (!theRTPSession->IsBound())
		{
			//we don't need to send a kill event, there's no way this task object
			//could be getting any events.
			delete theRTPSession;
			
			//if there hasn't been a response sent to the client yet, make sure
			//to do so
			if ((rtspError == RTSPProtocol::kSuccessOK) && (!inRTSPRequest->HasResponseBeenSent()))
				return inRTSPRequest->SendErrorResponse(RTSPProtocol::kClientNotFound,
														RTSPMessages::kNoRTPModuleAvailable);
		}
		else
		{
			//A module has taken this session under its wing.
			//So, generate a session ID for this session
			StrPtrLen theNewSessionID;
			char theSessionIDBuffer[kMaxSessionIDLength];
			theNewSessionID.Ptr = theSessionIDBuffer;

			QTSS_ErrorCode activationError = QTSS_DupName;
			while (activationError == QTSS_DupName)
			{
				UInt32 theLength = RTPModuleInterface::GenerateNewSessionID(theSessionIDBuffer);
				theNewSessionID.Len = theLength;
				
				//ok, some module has bound this session, we can activate it.
				//At this point, we may find out that this new session ID is a duplicate.
				//If that's the case, we'll simply retry until we get a unique ID
				activationError = theRTPSession->Activate(theNewSessionID);
				theRTPSession = NULL;//it is IMPERATIVE that we consider this a dead pointer now. 
			}
			Assert(activationError == QTSS_NoErr);
			
			//Append this new session ID to the response
			inRTSPRequest->AppendSessionHeader(&theNewSessionID,
									RTSPServerInterface::GetRTSPPrefs()->GetRTSPTimeoutAsString());

			//one of the modules grabbed this session. Invoke that module with this
			//setup inRTSPRequest
			rtspError = RTPModuleInterface::SendToRTPModule(inRTSPRequest, &theNewSessionID);
		}
	}
	else
		//there is already a module bound to this session, so just pass the request
		//onto the module
		rtspError = RTPModuleInterface::SendToRTPModule(inRTSPRequest, inSessionID);

	return RTSPProtocol::kSuccessOK;
}


RTSPProtocol::RTSPStatusCode RTPModuleInterface::ProcessRequest( RTSPRequestInterface *inRTSPRequest, StrPtrLen* inSessionID)
{
	Assert(inRTSPRequest != NULL);
	Assert(sRTPModuleArray != NULL);
	
	RTSPProtocol::RTSPStatusCode returnErr = RTSPProtocol::kSuccessOK;
	
	//by default, timeout should be enabled
	inRTSPRequest->GetSession()->SetTimeoutEnabled(true);
	
	//Really process the inRTSPRequest
	switch ( inRTSPRequest->GetMethod() )
	{
		case RTSPProtocol::kDescribeMethod:
			//There isn't any RTP session associated with describes, so the SendToRTPModule
			//function won't get called, so we should call the preprocessors & postprocessors
			//separately, passing in NULL for the RTP session
			
			RTPModuleInterface::PreProcessRTSPRequest(inRTSPRequest, NULL);
			returnErr = RTPModuleInterface::DoDescribe(inRTSPRequest);
			RTPModuleInterface::PostProcessRTSPRequest(inRTSPRequest, NULL);
		break;
		
		case RTSPProtocol::kSetupMethod:
			returnErr = RTPModuleInterface::DoSetup(inRTSPRequest, inSessionID);
		break;
		
		case RTSPProtocol::kOptionsMethod:
		{
			//I'm not sure if this is really the best place to put this,
			//but it works...
			inRTSPRequest->AppendHeader(RTSPProtocol::kPublicHeader, &sPublicHeader);
			inRTSPRequest->SendHeader();
			returnErr = RTSPProtocol::kSuccessOK;
		}
		break;
		
		case RTSPProtocol::kPlayMethod:
		{
			//because the QT client doesn't allow the TCP connection to go away,
			//after we've issued a play, disable the RTSP timeout
			if (RTSPServerInterface::GetRTSPPrefs()->DisableTimeoutOnPlay())
				inRTSPRequest->GetSession()->SetTimeoutEnabled(false);
				
			//Yes, we intentionally want to fall through here...
		}
		default:
		{
			returnErr = RTPModuleInterface::SendToRTPModule(inRTSPRequest, inSessionID);
		}
	}
	
	return returnErr;
	
}


RTSPProtocol::RTSPStatusCode RTPModuleInterface::SendToRTPModule(RTSPRequestInterface *inRTSPRequest, StrPtrLen* inSessionID)
{
	Assert(inRTSPRequest != NULL);
	
	//The order of things here is very very important

	if ((inSessionID == NULL) || (inSessionID->Len == 0))
		return inRTSPRequest->SendErrorResponse(RTSPProtocol::kClientBadRequest,
												RTSPMessages::kSessionIDRequired,
												inRTSPRequest->GetQTSSParameter(qtssMethodParam));
	
	OSRefResolver osRefResolver(RTPServerInterface::GetRTPSessionMap(), NULL);	//auto releaser
	OSRef* theRef = osRefResolver.Resolve(inSessionID);

	if (theRef == NULL)
		return inRTSPRequest->SendErrorResponse(RTSPProtocol::kClientSessionNotFound,
										RTSPMessages::kNoSessionID);
		
	RTPSession* theRTPSession = (RTPSession*)theRef->GetObject();
	
	//Make sure that the session can't be timed out when it's active!
	theRTPSession->DisarmTimeout();
	
	RTSPProtocol::RTSPStatusCode rtspError = RTSPProtocol::kSuccessOK;

	//Before invoking the module, grab the session mutex so we are sure it isn't
	//sending packets at the same time.
	{
		OSMutexLocker locker( theRTPSession->GetSessionMutex() );	

		//add the latest RTSP info to the RTP session
		StrPtrLen* theURL = inRTSPRequest->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(theURL != NULL);
		
		theRTPSession->GetSessionDictionary()->SetStandardValue(SessionDictionary::kLastRTSPURL, theURL->Ptr, theURL->Len);
		
		StrPtrLen* theUserAgent = inRTSPRequest->GetHeaderValue(RTSPProtocol::kUserAgentHeader);
		Assert(theUserAgent != NULL);
		
		theRTPSession->GetSessionDictionary()->SetStandardValue(SessionDictionary::kLastRTSPUserAgent, theUserAgent->Ptr, theUserAgent->Len);
		
		//run the preprocessor modules now.
		RTPModuleInterface::PreProcessRTSPRequest(inRTSPRequest, theRTPSession);

		//Invoke the module
		rtspError = theRTPSession->GetModule()->ProcessRTSPRequest(inRTSPRequest, theRTPSession);
			
		//run the postprocessor modules now.
		RTPModuleInterface::PostProcessRTSPRequest(inRTSPRequest, theRTPSession);

		//sokay to release the session mutex now
	}	

	//Rearm the timeout. This also resets it to its maximum time.
	theRTPSession->RearmTimeout();
	
	//It's ok to release the Session Ref at this point. We cannot reference the session
	//pointer after doing this, because the session may get destroyed!
	return rtspError;
}

UInt32 RTPModuleInterface::GenerateNewSessionID(char* ioBuffer)
{
	//RANDOM NUMBER GENERATOR
	
	//We want to make our session IDs as random as possible, so use a bunch of
	//current server statistics to generate a random SInt64.

	//Generate the random number in two UInt32 parts. The first UInt32 uses
	//statistics out of a random RTP session.
	SInt64 theMicroseconds = OS::Microseconds();
	::srand((unsigned int)theMicroseconds);
	UInt32 theFirstRandom = ::rand();
	
	{
		OSMutexLocker locker(RTPServerInterface::GetRTPSessionMap()->GetMutex());
		OSRefHashTable* theHashTable = RTPServerInterface::GetRTPSessionMap()->GetHashTable();
		if (theHashTable->GetNumEntries() > 0)
		{
			theFirstRandom %= theHashTable->GetNumEntries();
			theFirstRandom >>= 2;
			
			OSRefHashTableIter theIter(theHashTable);
			//Iterate through the session map, finding a random session
			for (UInt32 theCount = 0; theCount < theFirstRandom; theIter.Next(), theCount++)
				Assert(!theIter.IsDone());
			
			RTPSession* theSession = (RTPSession*)theIter.GetCurrent()->GetObject();
			theFirstRandom += theSession->GetPacketsSent();
			theFirstRandom += (UInt32)theSession->GetSessionCreateTime();
			theFirstRandom += (UInt32)theSession->GetPlayTime();
			theFirstRandom += (UInt32)theSession->GetBytesSent();
			theFirstRandom += (UInt32)theSession->GetMediaSrcRef();
		}
	}
	//Generate the first half of the random number
	::srand((unsigned int)theFirstRandom);
	theFirstRandom = ::rand();
	
	//Now generate the second half
	UInt32 theSecondRandom = ::rand();
	theSecondRandom += RTPServerInterface::GetCurrentBandwidthInBits();
	theSecondRandom += RTPServerInterface::GetAvgBandwidthInBits();
	theSecondRandom += RTPServerInterface::GetPacketsPerSecond();
	theSecondRandom += (UInt32)RTPServerInterface::GetTotalBytes();
	theSecondRandom += RTPServerInterface::GetTotalSessions();
	
	::srand((unsigned int)theSecondRandom);
	theSecondRandom = ::rand();
	
	SInt64 theSessionID = (SInt64)theFirstRandom;
	theSessionID <<= 32;
	theSessionID += (SInt64)theSecondRandom;
	::sprintf(ioBuffer, "%qd", theSessionID);
	Assert(::strlen(ioBuffer) < kMaxSessionIDLength);
	return ::strlen(ioBuffer);
}

void  RTPModuleInterface::SessionClosing( RTSPSessionInterface *inRTSPSessionInterface)
{
	//execute all RTP modules that want to see this event
	for (UInt32 scModIdx = 0; scModIdx < sNumRTPModules; scModIdx++)
	{
		if ( NULL != sRTPModuleArray[scModIdx] && sRTPModuleArray[scModIdx]->IsRTSPSessionClosingModule() )
			sRTPModuleArray[scModIdx]->ProcessRTSPSessionClosing(inRTSPSessionInterface);
	}

	//if there was a cookie associated with this session, we need to make
	//sure the RTP module has a chance to clear it. We've stored the module
	//pointer as cookie A, and its cookie as cookie B
	if (inRTSPSessionInterface->GetCookieA() != NULL)
		((RTPModule*)inRTSPSessionInterface->GetCookieA())->DestroyCookie(inRTSPSessionInterface->GetCookieB());
}

void RTPModuleInterface::InitModules()
{
	Assert(sRTPModuleArray != NULL);
	UInt32 x = 0;
	
	for ( x = 0; x < sNumRTPModules; x++)
	{
		if (NULL != sRTPModuleArray[x])
		{
			bool successful = sRTPModuleArray[x]->Initialize();
			if (!successful)
			{
				delete sRTPModuleArray[x];
				sRTPModuleArray[x] = NULL;
			}
		}
	}
}




RTPModule* RTPModuleInterface::GetRTPModuleByIndex(UInt32 inIndex)
{
	Assert(sRTPModuleArray != NULL);
	if (inIndex < sNumRTPModules)
		return sRTPModuleArray[inIndex];
	else
		return NULL;
}
