/*
	puzzle.c - animate puzzle

	Author:	Susumu Shiohara (shiohara@tpp.epson.co.jp)

		Copyright 1993-1997 by Susumu Shiohara

			All Rights Reserved

*/

#include "xslideshow.h"
#include "animproto.h"

typedef enum {
	UP=0,
	DOWN,
	LEFT,
	RIGHT,
	DMAX
}DIRS;

typedef struct {
	int	dir[DMAX];
	int	col;
	int	row;
}puzzleData;

typedef struct {
	DIRS	dir;
	int	count;
}historyData;

static	puzzleData current;
static	Window *wd=NULL;
static	Window firstw;
static	int xof,yof,pw,ph,pofx,pofy,steps,puzzleCount;
static	historyData *history=NULL;
static	int hasRes=False;
static	int mapsize;

/* Exchange the block at dir for current block */
#if defined(__STDC__) || defined(__cplusplus)
static void moveCurrent(DIRS dir,int count)
#else
static void moveCurrent(dir,count)
DIRS dir;
int	count;
#endif
{
int x1,y1,x2,y2,col,row;
Window tmpw;

	while(count--){
	switch(dir){
		case UP   : col=current.col;   row=current.row-1 ; break;
		case DOWN : col=current.col;   row=current.row+1 ; break;
		case LEFT : col=current.col-1; row=current.row   ; break;
		case RIGHT: col=current.col+1; row=current.row   ; break;
		default: break;
	}

	x1 = xof + pofx + current.col * steps;
	y1 = yof + pofy + current.row * steps;
	x2 = xof + pofx + col * steps;
	y2 = yof + pofy + row * steps;

	tmpw = wd[current.row * pw + current.col];
	XLowerWindow(theDisp,tmpw);

	/* Move the block at dir on the bresenham line */
	move_on_bresenham_line(
						wd[row * pw + col],
						x2, y2,
						x1, y1,
						2, app_data.moveOnTheLineTicks);

	/* Move the current block on the bresenham line */
	XMoveWindow(theDisp,tmpw,x2,y2);

	wd[current.row * pw + current.col] = wd[row * pw + col];
	wd[row * pw + col] = tmpw;

	/* Update current block informations */
	current.col = col;
	current.row = row;

	if(row == 0)		current.dir[UP]	  = 0;
	else				current.dir[UP]	  = 1;
	if(row == (ph - 1))	current.dir[DOWN] = 0;
	else				current.dir[DOWN] = 1;
	if(col == 0)		current.dir[LEFT] = 0;
	else				current.dir[LEFT] = 1;
	if(col == (pw - 1))	current.dir[RIGHT]= 0;
	else				current.dir[RIGHT]= 1;
	}
}

static void clearUnderPicture()
{
	XClearArea(
			theDisp,
			theWindow,
			xof + pofx,
			yof + pofy,
			pw * steps,
			ph * steps,
			False
			);
}

static void PuzzleIt()
{
int i,j,count,countmax,m;
DIRS predir,dir;
Window tmpw;
XImage *subImage;
Pixmap pic,subPic;
XSetWindowAttributes xswa;
XGCValues gcv;
GC gc;

	if(app_data.puzzleSteps > MIN(gim.subImageList->width,gim.subImageList->height))
		steps = MIN(gim.subImageList->width,gim.subImageList->height);
	else
		steps = app_data.puzzleSteps;

	if(steps == 0) steps = 1;

	xof = (windowWidth - gim.subImageList->width) / 2;
	yof = (windowHeight - gim.subImageList->height) / 2;

	if((windowWidth > gim.subImageList->width) && (windowHeight > gim.subImageList->height)){

		pw = gim.subImageList->width / steps;
		ph = gim.subImageList->height / steps;
		pofx = (gim.subImageList->width % steps) / 2;
		pofy = (gim.subImageList->height % steps) / 2;

	}
	else if ((windowWidth > gim.subImageList->width) && (windowHeight <= gim.subImageList->height)){

		pw = gim.subImageList->width / steps;
		ph = windowHeight / steps;
		pofx = (gim.subImageList->width % steps) / 2;
		pofy = (windowHeight % steps + gim.subImageList->height - windowHeight) / 2;

	}
	else if ((windowWidth <= gim.subImageList->width) && (windowHeight > gim.subImageList->height)){

		pw = windowWidth / steps;
		ph = gim.subImageList->height / steps;
		pofx = (windowWidth % steps + gim.subImageList->width - windowWidth) / 2;
		pofy = (gim.subImageList->height % steps) / 2;

	}
	else /* if ((windowWidth <= gim.subImageList->width) && (windowHeight <= gim.subImageList->height)) */ {

		pw = windowWidth / steps;
		ph = windowHeight / steps;
		pofx = (windowWidth % steps + gim.subImageList->width - windowWidth) / 2;
		pofy = (windowHeight % steps + gim.subImageList->height - windowHeight) / 2;
	}

	/* Can't puzzle 1x?, ?x1, 2x2 blocks */
	if(pw==1 || ph==1 || (pw==2 && ph==2)) {
		hasRes = False;
		return; 
	}

	gcv.function = GXcopy;
	gcv.foreground = bcol;
	gcv.background = bcol;
	if((pic = XCreatePixmap( theDisp,theWindow,
							gim.subImageList->width,gim.subImageList->height,
							DefaultDepth(theDisp,theScreenNumber))) == (Pixmap)0){
		fprintf(stderr,"xslideshow: Pixmap creation error.\n");
		goodbyekiss();
	}

	gc = XCreateGC(theDisp,pic,GCFunction | GCForeground | GCBackground,&gcv );
	XPutImage(theDisp,pic,gc,theImage,0,0,0,0,gim.subImageList->width,gim.subImageList->height);
	XFreeGC(theDisp,gc);

	m = pw * ph;
	puzzleCount = m * 2;
	history = (historyData *)XtMalloc(sizeof(historyData) * puzzleCount);
	if(history == (historyData *)NULL){
		fprintf(stderr,"xslideshow: memory allocation error.\n");
		goodbyekiss();
	}

	wd = (Window *)XtMalloc(sizeof(Window) * m);
	if(wd == (Window *)NULL){
		fprintf(stderr,"xslideshow: memory allocation error.\n");
		goodbyekiss();
	}

	/* Separate the picture into small blocks */
	firstw = XCreateSimpleWindow( theDisp,theWindow,
								xof + pofx,yof + pofy,
								steps - app_data.borderWidth * 2,
								steps - app_data.borderWidth * 2,
								app_data.borderWidth,
								bcol, bcol	);

	if(firstw == (Window)0) {
		XFreePixmap(theDisp,pic);

		XtFree((char *)history);
		history = NULL;

		XtFree((char *)wd);
		wd = NULL;

		hasRes = False;
		return;
	}

	xswa.override_redirect = True;
	XChangeWindowAttributes(theDisp,firstw,CWOverrideRedirect,&xswa);

	for(i = 0; i < m; i++){

		wd[i] = XCreateSimpleWindow( theDisp,theWindow,
									xof + pofx + (i % pw) * steps,
									yof + pofy + (i / pw) * steps,
									steps - app_data.borderWidth * 2,
									steps - app_data.borderWidth * 2,
									app_data.borderWidth,
									bcol, bcol	);

		if(wd[i] == (Window)0) {

			for(j = 0; j < i; j++)
				XDestroyWindow(theDisp,wd[j]);

			XDestroyWindow(theDisp,firstw);
			XFreePixmap(theDisp,pic);

			XtFree((char *)history);
			history = NULL;

			XtFree((char *)wd);
			wd = NULL;

			hasRes = False;
			return;

		}

		xswa.override_redirect = True;
		XChangeWindowAttributes(theDisp,wd[i],CWOverrideRedirect,&xswa);

		subPic = XCreatePixmap( theDisp,wd[i],
								steps, steps,
								DefaultDepth(theDisp,theScreenNumber)	);

		subImage = XGetImage( theDisp,pic,
								pofx + (i % pw) * steps,
								pofy + (i / pw) * steps,	
								steps, steps,
								XAllPlanes(), ZPixmap	);

		gc = XCreateGC(theDisp,wd[i],GCFunction | GCForeground | GCBackground,&gcv );
		XPutImage(theDisp,subPic,gc,subImage,0,0,0,0,steps,steps);
		XSetWindowBackgroundPixmap(theDisp,wd[i],subPic);

		XDestroyImage(subImage);
		XFreePixmap(theDisp,subPic);
		XFreeGC(theDisp,gc);

		myevent();	/* Get X event */
	}

	XFreePixmap(theDisp,pic);
	tmpw   = wd[0];
	wd[0]  = firstw;
	firstw = tmpw;
	XLowerWindow(theDisp,firstw);


	/* Puzzle (Move block randomly) */
	current.dir[UP]   =0;
	current.dir[DOWN] =1;
	current.dir[LEFT] =0;
	current.dir[RIGHT]=1;
	current.col=0;
	current.row=0;
	predir=dir=UP;

	for(i = 0; i < puzzleCount; i++){

		while(True){
			dir = myrandom() % DMAX;
			if(predir != dir && current.dir[dir]){
				switch(dir){
				case UP	  : countmax = current.row;	       predir = DOWN ; break;
				case DOWN : countmax = ph - current.row-1; predir = UP   ; break;
				case LEFT : countmax = current.col;	       predir = RIGHT; break;
				case RIGHT: countmax = pw - current.col-1; predir = LEFT ; break;
				default: break;
				}
				break;
			}
		}

		count = (myrandom() % countmax) + 1;
		moveCurrent(dir,count);
		history[i].dir  = dir;
		history[i].count = count;
	}

	/* Display all blocks */
	for(i = 0; i < m; i++) {
		XLowerWindow(theDisp,wd[i]);
		XMapWindow(theDisp,wd[i]);
	}

	hasRes = True;
	return;
}

static void Resolve()
{
DIRS dir;
int	i;

	/* The current block was placed on history[puzzleCount-1] */
	for(i = puzzleCount - 1; i >= 0; i--){
		switch(history[i].dir){
		case UP	  : dir = DOWN ; break;
		case DOWN : dir = UP   ; break;
		case LEFT : dir = RIGHT; break;
		case RIGHT: dir = LEFT ; break;
		default: break;
		}
		moveCurrent(dir,history[i].count);
		myusleep(app_data.puzzleTicks);
		if(gotSomeAction == True)
			return;
	}
	/* Display the first block, Then finished */
	XMapWindow(theDisp,firstw);
}

static void freeResources()
{
int i,m;

	m = pw * ph;
	for(i = 0; i < m; i++)
		XDestroyWindow(theDisp,wd[i]);

	XDestroyWindow(theDisp,firstw);

	XtFree((char *)history);
	history = NULL;

	XtFree((char *)wd);
	wd = NULL;

	hasRes = False;

	myusleep(0); /* Flush */
}

#if defined(__STDC__) || defined(__cplusplus)
ActionStatus xpuzzleshow(char *fname)
#else
ActionStatus xpuzzleshow(fname)
char *fname;
#endif
{
	mapsize = gim.subImageList->mapsize;

	PreDisplay();

	if(app_data.showFileName)
		createFileNameWindow(fname);

	PuzzleIt();

	ShowImage(	(windowWidth - gim.subImageList->width) / 2,
				(windowHeight - gim.subImageList->height) / 2,
				gim.subImageList->width, gim.subImageList->height);

	if(app_data.puzzleFade){
		PreFadeColors(mapsize);
		FadeColors(False,mapsize);
	}

	StoreColors(xcolors,mapsize);

	if(hasRes) {
		clearUnderPicture();
		myusleep(3000000); /* wait 3 sec */
		Resolve();
	}

	if(app_data.showFileName)
		raiseFileNameWindow();

	return(mywait());
}

#if defined(__STDC__) || defined(__cplusplus)
void postxpuzzleshow(char *fname)
#else
void postxpuzzleshow(fname)
char *fname;
#endif
{
	if(app_data.showFileName)
		removeFileNameWindow();

	if(app_data.puzzleFade)
		FadeColors(True,mapsize);

	if(hasRes) {
		freeResources();
		XClearWindow(theDisp,theWindow);
	}
}

