------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              E X P _ C H 6                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.173 $                            --
--                                                                          --
--   Copyright (C) 1992,1993,1994,1995,1996 Free Software Foundation, Inc.  --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Checks;   use Checks;
with Einfo;    use Einfo;
with Errout;   use Errout;
with Expander; use Expander;
with Exp_Ch2;  use Exp_Ch2;
with Exp_Ch7;  use Exp_Ch7;
with Exp_Ch9;  use Exp_Ch9;
with Exp_Disp; use Exp_Disp;
with Exp_Intr; use Exp_Intr;
with Exp_TSS;  use Exp_TSS;
with Exp_Util; use Exp_Util;
with Freeze;   use Freeze;
with Inline;   use Inline;
with Lib.Writ; use Lib.Writ;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Sem;      use Sem;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;

package body Exp_Ch6 is

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Expand_Actuals (N : Node_Id; Subp : Entity_Id);
   --  For each actual of an in-out parameter which is a numeric conversion
   --  of the form T(A), where A denotes a variable, we insert the declaration:
   --
   --  Temp : T := T(A);
   --
   --  prior to the call. Then we replace the actual with a reference to Temp,
   --  and append the assignment:
   --
   --  A := T' (Temp);
   --
   --  after the call. Here T' is the actual type of variable A.
   --  For out parameters, the initial declaration has no expression.
   --  If A is not an entity name,  we generate instead:
   --
   --  Var  : T' renames A;
   --  Temp : T := Var;       --  omitting expression for out parameter.
   --  ...
   --  Var := T' (Temp);
   --
   --  For other in-out parameters, we emit the required constraint checks
   --  before and/or after the call.

   procedure Expand_Protected_Subprogram_Call
     (N    : Node_Id;
      Subp : Entity_Id;
      Scop : Entity_Id);
   --  A call to a protected subprogram within the protected object may appear
   --  as a regular call. The list of actuals must be expanded to contain a
   --  reference to the object itself, and the call becomes a call to the
   --  corresponding protected subprogram.

   --------------------
   -- Expand_Actuals --
   --------------------

   procedure Expand_Actuals (N : Node_Id; Subp : Entity_Id) is
      Loc       : constant Source_Ptr := Sloc (N);
      Actual    : Node_Id;
      Formal    : Entity_Id;
      N_Node    : Node_Id;
      Post_Call : List_Id;
      E_Formal  : Entity_Id;

      procedure Add_In_Out_Checks;
      --  For In and In-Out parameters, emit constraint checks before the
      --  call. For In-Out and Out parameters, if the actual is a constrained
      --  access type, emit a constraint on the resulting value. Other
      --  scalar types receive checks directly from the back-end, and
      --  assignment to composite types are checked in the body.

      function Make_Var (V_Typ : Entity_Id; Expr : Node_Id) return Entity_Id;
      --  If parameter is not an entity name, create a temporary to be used
      --  for checks on exit.

      procedure Add_In_Out_Checks is
         Expr  : Node_Id;
         Init  : Node_Id;
         Temp  : Entity_Id;
         Var   : Entity_Id;
         V_Typ : Entity_Id;

      begin
         Temp  := Make_Defining_Identifier (Loc, New_Internal_Name ('T'));

         if Nkind (Actual) = N_Type_Conversion then
            V_Typ := Etype (Expression (Actual));
            Var   := Make_Var (V_Typ, Expression (Actual));
         else
            V_Typ := Etype (Actual);
            Var   := Make_Var (V_Typ, Actual);
         end if;

         if Ekind (Formal) = E_In_Out_Parameter then
            if Nkind (Actual) = N_Type_Conversion then
               Init :=
                 Convert_To (Etype (Formal), New_Occurrence_Of (Var, Loc));
            else
               Init := New_Occurrence_Of (Var, Loc);
            end if;

         else
            Init := Empty;
         end if;

         N_Node :=
           Make_Object_Declaration (Loc,
             Defining_Identifier => Temp,
             Object_Definition   =>
               New_Occurrence_Of (Etype (Formal), Loc),
             Expression => Init);
         Insert_Action (N, N_Node);

         if Nkind (Actual) = N_Type_Conversion then

            --  Use the reverse conversion on exit.

            Expr := Convert_To (V_Typ, New_Occurrence_Of (Temp, Loc));

         else
            Expr := New_Occurrence_Of (Temp, Loc);
         end if;

         Rewrite_Substitute_Tree
           (Actual, New_Reference_To (Defining_Identifier (N_Node), Loc));
         Analyze (Actual);

         Append_To (Post_Call,
           Make_Assignment_Statement (Loc,
             Name       => New_Occurrence_Of (Var, Loc),
             Expression => Expr));
      end Add_In_Out_Checks;

      function Make_Var (V_Typ : Entity_Id; Expr : Node_Id) return Entity_Id is
         Var : Entity_Id;

      begin
         if Is_Entity_Name (Expr) then
            Var := Entity (Expr);

         else
            Var := Make_Defining_Identifier (Loc, New_Internal_Name ('T'));

            N_Node :=
              Make_Object_Renaming_Declaration (Loc,
                Defining_Identifier => Var,
                Subtype_Mark        => New_Occurrence_Of (V_Typ, Loc),
                Name                => Relocate_Node (Expr));

            Insert_Before_And_Analyze (N, N_Node);
         end if;

         return Var;
      end Make_Var;

   --  Start of processing for Expand_Actuals

   begin
      Formal := First_Formal (Subp);
      Actual := First_Actual (N);

      Post_Call := New_List;

      while Present (Formal) loop
         E_Formal := Etype (Formal);

         if Is_Scalar_Type (E_Formal) then

            --  ??? not sure why is this necessary
            null;

         --  RM 6.4.1 (11)

         elsif Ekind (Formal) /= E_Out_Parameter then
            Apply_Constraint_Check (Actual, E_Formal);

         --  Out parameter case. Not constraint checks on access type
         --  RM 6.4.1 (13)

         elsif Is_Access_Type (E_Formal) then
            null;

         --  RM 6.4.1 (14)

         elsif Has_Discriminants (Base_Type (E_Formal))
           or else Present (Base_Init_Proc (E_Formal))
         then
            Apply_Constraint_Check (Actual, E_Formal);

         --  RM 6.4.1 (15)

         else
            Apply_Constraint_Check (Actual, Base_Type (E_Formal));
         end if;

         --  ??? I don't understand what this code is about (CC). Isn't
         --  apply_range_check called with a type incompatible with the
         --  expression

         if Is_Array_Type (E_Formal)
           and then Nkind (Actual) = N_Type_Conversion
           and then Ekind (Formal) /= E_In_Parameter
         then
            Apply_Range_Check (Expression (Actual), E_Formal);
         end if;

         --  Why do we only do something for numeric/enum types here???

         if Nkind (Actual) = N_Type_Conversion
           and then (Is_Numeric_Type (E_Formal)
                       or else Is_Enumeration_Type (E_Formal))
           and then Ekind (Formal) /= E_In_Parameter
         then
            Add_In_Out_Checks;

         --  references to components of packed arrays are expanded at this
         --  point, rather than at the point of analysis of the actuals, to
         --  handle the expansion of the assignment to (in)-out parameters.

         elsif Nkind (Actual) = N_Indexed_Component
           and then Is_Packed (Etype (Prefix (Actual)))
         then
            Add_In_Out_Checks;

         --  It is neccessary to exclude tagged types because of "downward
         --  conversion" errors and a strange assertion error in namet
         --  from gnatf in bug 1215-001.

         elsif Is_Access_Type (E_Formal)
           and then (E_Formal /= Etype (Actual)
                      or else Nkind (Actual) = N_Type_Conversion)
           and then not Is_Tagged_Type (Directly_Designated_Type (E_Formal))
           and then Ekind (Formal) /= E_In_Parameter
         then
            Add_In_Out_Checks;

         end if;

         Formal := Next_Formal (Formal);
         Actual := Next_Actual (Actual);
      end loop;

      --  Find right place to put post call stuff if it is present

      if not Is_Empty_List (Post_Call) then

         --  If call is not a list member, it must be the triggering
         --  statement of a triggering alternative, and we can add
         --  the post call stuff to the corresponding statement list.

         if not Is_List_Member (N) then
            declare
               P : constant Node_Id := Parent (N);

            begin
               pragma Assert (Nkind (P) = N_Triggering_Alternative);

               if Present (Statements (P)) then
                  Insert_List_Before_And_Analyze
                    (First (Statements (P)), Post_Call);
               else
                  Set_Statements (P, Post_Call);
               end if;
            end;

         --  Otherwise, normal case where N is in a statement sequence,
         --  just put the post-call stuff after the call statement.

         else
            Insert_List_After (N, Post_Call);
         end if;
      end if;

      --  The call node itself is re-analyzed in Expand_Call.

   end Expand_Actuals;

   -----------------
   -- Expand_Call --
   -----------------

   --  This procedure handles expansion of function calls and procedure call
   --  statements (i.e. it serves as the body for Expand_N_Function_Call and
   --  Expand_N_Procedure_Call_Statement. Processing for calls includes:

   --    Provide values of actuals for all formals in Extra_Formals list
   --    Replace "call" to enumeration literal function by literal itself
   --    Rewrite call to predefined operator as operator
   --    Replace actuals to in-out parameters that are numeric conversions,
   --     with explicit assignment to temporaries before and after the call.

   --   Note that the list of actuals has been filled with default expressions
   --   during semantic analysis of the call. Only the extra actuals required
   --   for the 'Constrained attribute and for accessibility checks are added
   --   at this point.

   procedure Expand_Call (N : Node_Id) is
      Loc           : constant Source_Ptr := Sloc (N);
      Subp          : Entity_Id;
      Parent_Subp   : Entity_Id;
      Parent_Formal : Entity_Id;
      Actual        : Node_Id;
      Formal        : Entity_Id;
      Prev          : Node_Id := Empty;
      Scop          : Entity_Id;
      Extra_Actuals : List_Id := No_List;

      procedure Add_Actual_Parameter (Insert_Param : Node_Id);
      --  Adds one entry to the end of the actual parameter list. Used for
      --  default parameters and for extra actuals (for Extra_Formals).
      --  The argument is an N_Parameter_Association node.

      procedure Add_Extra_Actual (Expr : Node_Id; EF : Entity_Id);
      --  Adds an extra actual to the list of extra actuals. Expr
      --  is the expression for the value of the actual, EF is the
      --  entity for the extra formal.

      --------------------------
      -- Add_Actual_Parameter --
      --------------------------

      procedure Add_Actual_Parameter (Insert_Param : Node_Id) is
         Actual_Expr : constant Node_Id :=
                         Explicit_Actual_Parameter (Insert_Param);

      begin
         --  Case of insertion is first named actual

         if No (Prev) or else
            Nkind (Parent (Prev)) /= N_Parameter_Association
         then
            Set_Next_Named_Actual (Insert_Param, First_Named_Actual (N));
            Set_First_Named_Actual (N, Actual_Expr);

            if No (Prev) then
               if not Present (Parameter_Associations (N)) then
                  Set_Parameter_Associations (N, New_List);
                  Append (Insert_Param, Parameter_Associations (N));
               end if;
            else
               Insert_After (Prev, Insert_Param);
            end if;

         --  Case of insertion is not first named actual

         else
            Set_Next_Named_Actual
              (Insert_Param, Next_Named_Actual (Parent (Prev)));
            Set_Next_Named_Actual (Parent (Prev), Actual_Expr);
            Append (Insert_Param, Parameter_Associations (N));
         end if;

         Prev := Actual_Expr;
      end Add_Actual_Parameter;

      ----------------------
      -- Add_Extra_Actual --
      ----------------------

      procedure Add_Extra_Actual (Expr : Node_Id; EF : Entity_Id) is
         Loc : constant Source_Ptr := Sloc (Expr);

      begin
         if Extra_Actuals = No_List then
            Extra_Actuals := New_List;
            Set_Parent (Extra_Actuals, N);
         end if;

         Append_To (Extra_Actuals,
           Make_Parameter_Association (Loc,
             Explicit_Actual_Parameter => Expr,
             Selector_Name =>
               Make_Identifier (Loc, Chars (EF))));

         Analyze_And_Resolve (Expr, Etype (EF));

      end Add_Extra_Actual;

   --  Start of processing for Expand_Call

   begin
      if Nkind (Name (N)) = N_Explicit_Dereference then
         Subp := Etype (Name (N));
         Parent_Subp := Empty;

      --  Case of call to simple entry, where the Name is a selected component
      --  whose prefix is the task, and whose selector name is the entry name

      elsif Nkind (Name (N)) = N_Selected_Component then
         Subp := Entity (Selector_Name (Name (N)));
         Parent_Subp := Empty;

      --  Case of call to member of entry family, where Name is an indexed
      --  component, with the prefix being a selected component giving the
      --  task and entry family name, and the index being the entry index.

      elsif Nkind (Name (N)) = N_Indexed_Component then
         Subp := Entity (Selector_Name (Prefix (Name (N))));
         Parent_Subp := Empty;

      --  Normal case

      else
         Subp := Entity (Name (N));
         Parent_Subp := Alias (Subp);

         if Ekind (Subp) = E_Entry then
            Parent_Subp := Empty;
         end if;
      end if;

      --  First step, compute  extra actuals, corresponding to any
      --  Extra_Formals present. Note that we do not access
      --  Extra_Formals directly, instead we simply  note the presence of
      --  the extra formals as we process the regular formals and collect
      --  the corresponding actuals in Extra_Actuals.

      Formal := First_Formal (Subp);
      Actual := First_Actual (N);

      while Present (Formal) loop

         Prev   := Actual;

         --  Create possible extra actual for constrained case. Usually,
         --  the extra actual is of the form actual'constrained, but since
         --  this attribute is only available for unconstrained records,
         --  TRUE is expanded if the type of the formal happens to be
         --  constrained (for instance when this procedure is inherited
         --  from an unconstrained record to a constrained one) or if the
         --  actual has no discriminant (its type is constrained). An
         --  exception to this is the case of a private type without
         --  discriminants. In this case we pass FALSE because the
         --  object has underlying discriminants with defaults.

         if Present (Extra_Formal_Constrained (Formal)) then
            if Ekind (Etype (Prev)) in Private_Kind
              and then not Has_Discriminants (Base_Type (Etype (Prev)))
            then
               Add_Extra_Actual (
                 New_Occurrence_Of (Standard_False, Loc),
                 Extra_Formal_Constrained (Formal));

            elsif Is_Constrained (Etype (Formal))
              or else not Has_Discriminants (Etype (Prev))
            then
               Add_Extra_Actual (
                 New_Occurrence_Of (Standard_True, Loc),
                 Extra_Formal_Constrained (Formal));

            else
               Add_Extra_Actual (
                 Make_Attribute_Reference (Sloc (Prev),
                   Prefix => Duplicate_Subexpr (Prev, Name_Req => True),
                   Attribute_Name => Name_Constrained),
                 Extra_Formal_Constrained (Formal));
            end if;
         end if;

         --  Create possible extra actual for accessibility level.

         if Present (Extra_Formal_Accessibility (Formal)) then
            if Is_Entity_Name (Prev) then

               --  When passing an access parameter as the actual to another
               --  access parameter we need to pass along the actual's own
               --  associated access level parameter.

               if Ekind (Entity (Prev)) in Formal_Kind
                 and then Ekind (Etype (Prev)) = E_Anonymous_Access_Type
               then
                  declare
                     Parm_Ent : constant Entity_Id := Param_Entity (Prev);

                  begin
                     pragma Assert (Present (Parm_Ent));

                     if Present (Extra_Formal_Accessibility (Parm_Ent)) then
                        Add_Extra_Actual (
                          New_Occurrence_Of
                            (Extra_Formal_Accessibility (Parm_Ent), Loc),
                          Extra_Formal_Accessibility (Formal));

                     --  If the actual access parameter does not have an
                     --  associated extra formal providing its scope level,
                     --  then treat the actual as having library-level
                     --  accessibility.

                     else
                        Add_Extra_Actual (
                          Make_Integer_Literal
                            (Loc, Scope_Depth (Standard_Standard)),
                          Extra_Formal_Accessibility (Formal));
                     end if;
                  end;

               --  The actual is a normal access value, so just pass the
               --  level of the actual's access type.

               else
                  Add_Extra_Actual (
                    Make_Integer_Literal
                      (Loc, Type_Access_Level (Etype (Prev))),
                    Extra_Formal_Accessibility (Formal));
               end if;

            else
               case Nkind (Prev) is

                  when N_Attribute_Reference =>

                     case Get_Attribute_Id (Attribute_Name (Prev)) is

                        --  For X'Access, pass on the level of the prefix X

                        when Attribute_Access =>
                           Add_Extra_Actual (
                             Make_Integer_Literal
                               (Loc, Object_Access_Level (Prefix (Prev))),
                             Extra_Formal_Accessibility (Formal));

                        --  Treat the unchecked attributes as library-level

                        when Attribute_Unchecked_Access |
                           Attribute_Unrestricted_Access =>
                           Add_Extra_Actual (
                             Make_Integer_Literal
                               (Loc, Scope_Depth (Standard_Standard)),
                             Extra_Formal_Accessibility (Formal));

                        --  No other cases of attributes returning access
                        --  values that can be passed to access parameters

                        when others =>
                           pragma Assert (False); null;

                     end case;

                  --  For allocators we pass the level of the execution of
                  --  the called subprogram, which is one greater than the
                  --  current scope level.

                  when N_Allocator =>
                     Add_Extra_Actual (
                       Make_Integer_Literal
                         (Loc, UI_Add (Scope_Depth (Current_Scope), 1)),
                       Extra_Formal_Accessibility (Formal));

                  --  For other cases we simply pass the level of the
                  --  actual's access type.

                  when others =>
                     Add_Extra_Actual (
                       Make_Integer_Literal
                         (Loc, Type_Access_Level (Etype (Prev))),
                       Extra_Formal_Accessibility (Formal));

               end case;
            end if;
         end if;

         Actual := Next_Actual (Actual);
         Formal := Next_Formal (Formal);
      end loop;

      --  Deals with Dispatch_Call if we still have a call, before expanding
      --  extra actuals since this will be done on the re-analysis of the
      --  dispatching call

      if (Nkind (N) = N_Function_Call
           or else Nkind (N) =  N_Procedure_Call_Statement)
        and then Present (Controlling_Argument (N))
      then
         Expand_Dispatch_Call (N);
         return;

         --  Similarly, do not add extra actuals for an entry call whose entity
         --  is a protected procedure, because it will be rewritten as a
         --  protected procedure call and reanalyzed.

      elsif Nkind (N) = N_Entry_Call_Statement
        and then Ekind (Subp) = E_Procedure
        and then Is_Protected_Type (Scope (Subp))
      then
         null;
      else
         --  During that loop we gathered the extra actuals (the ones that
         --  correspond to Extra_Formals), so now they can be appended.

         while Is_Non_Empty_List (Extra_Actuals) loop
            Add_Actual_Parameter (Remove_Head (Extra_Actuals));
         end loop;
      end if;

      if Nkind (N) /= N_Entry_Call_Statement
        and then No (Controlling_Argument (N))
        and then Present (Parent_Subp)
      then
         while Present (Alias (Parent_Subp)) loop
            Parent_Subp := Alias (Parent_Subp);
         end loop;

         Set_Entity (Name (N), Parent_Subp);

         --  Add an explicit conversion for parameter of the derived type.
         --  This is only done for scalar in-parameters. Others are expanded
         --  in expand_actuals.

         Formal := First_Formal (Subp);
         Parent_Formal := First_Formal (Parent_Subp);
         Actual := First_Actual (N);

         if not Is_Intrinsic_Subprogram (Parent_Subp) then

            while Present (Formal) loop

               if (Etype (Formal) /= Etype (Parent_Formal))
                 and then Is_Scalar_Type (Etype (Formal))
                 and then Ekind (Formal) = E_In_Parameter
               then
                  Rewrite_Substitute_Tree (Actual,
                    Convert_To (Etype (Parent_Formal),
                      Relocate_Node (Actual)));

                  Analyze (Actual);
                  Resolve (Actual, Etype (Parent_Formal));
                  Set_Do_Range_Check (Actual, True);
               end if;

               Formal := Next_Formal (Formal);
               Parent_Formal := Next_Formal (Parent_Formal);
               Actual := Next_Actual (Actual);
            end loop;
         end if;

         Subp := Parent_Subp;
      end if;

      --  Some more special cases for cases other than explicit dereference

      if Nkind (Name (N)) /= N_Explicit_Dereference then

         --  Calls to an enumeration literal are replaced by the literal
         --  The only way that this case occurs is when we have a call to
         --  a function that is a renaming of an enumeration literal. The
         --  normal case of a direct reference to an enumeration literal
         --  has already been dealt with by Resolve_Call

         if Ekind (Subp) = E_Enumeration_Literal then
            Rewrite_Substitute_Tree (N, Name (N));
         end if;

      else
         if Ekind (Etype (Prefix (Name (N)))) =
                               E_Access_Protected_Subprogram_Type
         then
            --  If this is a call through an access to protected operation,
            --  the prefix has the form (object'address, operation'access).
            --  Rewrite as a for other protected calls: the object is the
            --  first parameter of the list of actuals.

            declare
               Call : Node_Id;
               Parm : List_Id;
               Nam  : Node_Id;
               Obj  : Node_Id;
               Ptr  : Node_Id := Prefix (Name (N));
               T    : Entity_Id := Equivalent_Type (Etype (Ptr));
               D_T  : Entity_Id := Designated_Type (Etype (Ptr));

            begin
               Obj := Make_Selected_Component (Loc,
                 Prefix => Unchecked_Convert_To (T, Ptr),
                 Selector_Name => New_Occurrence_Of (First_Entity (T), Loc));

               Nam := Make_Selected_Component (Loc,
                 Prefix => Unchecked_Convert_To (T, Ptr),
                 Selector_Name => New_Occurrence_Of (
                   Next_Entity (First_Entity (T)), Loc));

               Nam := Make_Explicit_Dereference (Loc, Nam);

               if Present (Parameter_Associations (N))  then
                  Parm := New_List_Copy (Parameter_Associations (N));
               else
                  Parm := New_List;
               end if;

               Prepend (Obj, Parm);

               if Etype (D_T) = Standard_Void_Type then
                  Call := Make_Procedure_Call_Statement (Loc,
                    Name => Nam,
                    Parameter_Associations => Parm);
               else
                  Call := Make_Function_Call (Loc,
                    Name => Nam,
                    Parameter_Associations => Parm);
               end if;

               Set_Etype (Call, Etype (D_T));

               Rewrite_Substitute_Tree (N, Call);
               Analyze (Nam);
               Analyze (Obj);
               return;
            end;
         end if;
      end if;

      --  functions returning controlled objects need special attention

      if Controlled_Type (Etype (Subp))
        and then not Is_Return_By_Reference_Type (Etype (Subp))
      then
         Expand_Ctrl_Function_Call (N);
         return;
      end if;

      if Ekind (Subp) = E_Procedure
         or else Ekind (Subp) = E_Entry
         or else Ekind (Subp) = E_Entry_Family
      then
         Expand_Actuals (N, Subp);
      end if;

      --  If this is a call to an intrinsic subprogram, then perform the
      --  appropriate expansion to the corresponding tree node.

      if Is_Intrinsic_Subprogram (Subp) then
         Expand_Intrinsic_Call (N, Subp);
         return;
      end if;

      if Ekind (Subp) = E_Function
        or else Ekind (Subp) = E_Procedure
      then

         if Is_Inlined (Subp) then
            Add_Inlined_Body (N, Subp);
         end if;
      end if;

      --  Check for a protected subprogram.  This is either an intra-object
      --  call, or a protected function call. Protected procedure calls are
      --  rewritten as entry calls and handled accordingly.

      Scop := Scope (Subp);

      if Nkind (N) /= N_Entry_Call_Statement
        and then Is_Protected_Type (Scop)
      then
         --  If the call is an internal one,  it is rewritten as a call to
         --  to the corresponding unprotected subprogram.

         Expand_Protected_Subprogram_Call (N, Subp, Scop);
      end if;

   end Expand_Call;

   --------------------------------------
   -- Expand_Protected_Subprogram_Call --
   --------------------------------------

   procedure Expand_Protected_Subprogram_Call
     (N    : Node_Id;
      Subp : Entity_Id;
      Scop : Entity_Id)
   is
      Loc   : constant Source_Ptr := Sloc (N);
      Param : Entity_Id;
      Corr  : Entity_Id;
      Proc  : Entity_Id;
      Rec   : Node_Id;

   begin
      --  If the protected object is not an enclosing scope, this is
      --  an inter-object function call. Inter-object procedure
      --  calls are expanded by Exp_Ch9.Build_Simple_Entry_Call.
      --  The call is intra-object only if the the subprogram being
      --  called is in the protected body being compiled, and if the
      --  protected object in the call is statically the enclosing type.
      --  The object may be an component of some other data structure,
      --  in which case this must be handled as an inter-object call.

      if not In_Open_Scopes (Scop)
        or else not Is_Entity_Name (Name (N))
      then
         if Nkind (Name (N)) = N_Selected_Component then
            Rec := Prefix (Name (N));

         elsif Nkind (Name (N)) = N_Indexed_Component then
            Rec := Prefix (Prefix (Name (N)));

         else
            null;
            pragma Assert (False);
         end if;

         Rewrite_Substitute_Tree (N,
           Build_Protected_Subprogram_Call (N,
             Name => New_Occurrence_Of (Subp, Sloc (N)),
             Rec =>  Convert_Concurrent (Rec, Etype (Rec)),
             External => True));

      else
         Rec := Make_Identifier (Loc, Name_uObject);
         Set_Etype (Rec, Corresponding_Record_Type (Scop));

         --  Find enclosing protected operation, and retrieve its first
         --  parameter, which denotes the enclosing protected object.
         --  If the enclosing operation is an entry, we are immediately
         --  within the protected body, and we can retrieve the object
         --  from the service entries procedure. A barrier function has
         --  has the same signature as an entry. A barrier function is
         --  compiled within the protected object, but unlike protected
         --  operations its never needs locks, so that its protected body
         --  subprogram points to itself.

         Proc := Current_Scope;
         while Present (Proc)
           and then Scope (Proc) /= Scop
         loop
            Proc := Scope (Proc);
         end loop;

         Corr := Protected_Body_Subprogram (Proc);

         if No (Corr) then

            --  Previous error left expansion incomplete.
            --  Nothing to do on this call.

            return;
         end if;

         Param := Defining_Identifier
           (First
             (Parameter_Specifications (Parent (Corr))));

         if Is_Subprogram (Proc)
           and then Proc /= Corr
         then
            --  Protected function or procedure.

            Set_Entity (Rec, Param);

            --  What is this about and why is it necessary???

            Set_Analyzed (Rec);

            --  A protected procedure cannot be called internally from
            --  a protected function.

            if Ekind (Corr) = E_Function
              and then Ekind (Subp) /= E_Function
            then
               Error_Msg_N
                ("within a protected function object is constant", N);
               --  ??? this is wrong, error messages should not be output
               --  ??? in expander, since then they do not appear in -gnatc.
            end if;

         else
            --  Entry or barrier function for entry body.
            --  The first parameter of the entry body procedure is a
            --  pointer to the object. We create a local variable
            --  of the proper type, duplicating what is done to define
            --  _object later on.

            declare
               Decls : List_Id;
               Obj_Ptr : Entity_Id :=  Make_Defining_Identifier
                                         (Loc, New_Internal_Name ('T'));
            begin
               Decls := New_List (
                 Make_Full_Type_Declaration (Loc,
                   Defining_Identifier => Obj_Ptr,
                     Type_Definition =>
                        Make_Access_To_Object_Definition (Loc,
                          Subtype_Indication =>
                            New_Reference_To
                         (Corresponding_Record_Type (Scop), Loc))));

               Insert_Actions (N, Decls);

               Rec :=
                 Make_Explicit_Dereference (Loc,
                   Unchecked_Convert_To (Obj_Ptr,
                     New_Occurrence_Of (Param, Loc)));
            end;
         end if;

         Rewrite_Substitute_Tree (N,
           Build_Protected_Subprogram_Call (N,
             Name     => Name (N),
             Rec      => Rec,
             External => False));

         --  What is this about, and why is it necessary ???
         --  Even if the setting of analyzed on Rec is legitimate,
         --  it would protect the result of this analyze anyway!

         if not Is_Subprogram (Proc) then
            Analyze (Rec);
         end if;
      end if;

      Analyze (N);
   end Expand_Protected_Subprogram_Call;

   ----------------------------
   -- Expand_N_Function_Call --
   ----------------------------

   procedure Expand_N_Function_Call (N : Node_Id) is
   begin
      Expand_Call (N);
   end Expand_N_Function_Call;

   ---------------------------------------
   -- Expand_N_Procedure_Call_Statement --
   ---------------------------------------

   procedure Expand_N_Procedure_Call_Statement (N : Node_Id) is
   begin
      Expand_Call (N);
   end Expand_N_Procedure_Call_Statement;

   ------------------------------
   -- Expand_N_Subprogram_Body --
   ------------------------------

   --  Add return statement if last statement in body is not a return
   --  statement (this makes things easier on Gigi which does not want
   --  to have to handle a missing return).

   --  Add call to Activate_Tasks if body is a task activator

   procedure Expand_N_Subprogram_Body (N : Node_Id) is
      Loc      : constant Source_Ptr := Sloc (N);
      H        : constant Node_Id    := Handled_Statement_Sequence (N);
      Spec_Id  : Entity_Id;
      Except_H : Node_Id;
      Scop     : Entity_Id;
      Dec      : Node_Id;
      Next_Op : Node_Id;

      procedure Add_Termination (S : List_Id);
      --  Append to S a return statement in the procedure case or a Raise
      --  Program_Error in the function case if the last statement is not
      --  already a return or a goto statement.

      procedure Add_Termination (S : List_Id) is
         Last_S : constant Node_Id := Last (S);
         Loc_S  : constant Source_Ptr := Sloc (Last_S);

      begin
         if Nkind (Last_S) /= N_Return_Statement
           and then Nkind (Last_S) /= N_Goto_Statement
           and then Nkind (Last_S) /= N_Raise_Statement
         then
            if Ekind (Spec_Id) = E_Procedure then
               Append_To (S, Make_Return_Statement (Loc_S));

            elsif Ekind (Spec_Id) = E_Function then
               Append_To (S,
                 Make_Raise_Statement (Loc_S,
                   Name => New_Occurrence_Of (Standard_Program_Error, Loc_S)));
            end if;
         end if;
      end Add_Termination;

   --  Start of processing for Expand_N_Subprogram_Body

   begin
      --  Get entities for subprogram body and spec

      if Present (Corresponding_Spec (N)) then
         Spec_Id := Corresponding_Spec (N);
      else
         Spec_Id := Defining_Unit_Simple_Name (Specification (N));
         Make_Default_Expr_Functions (N, Spec_Id);
      end if;

      --  Returns_By_Ref flag is normally set when the subprogram is frozen
      --  but subprograms with no specs are not frozen

      declare
         Typ  : constant Entity_Id := Etype (Spec_Id);
         Utyp : constant Entity_Id := Underlying_Type (Typ);

      begin
         if not Acts_As_Spec (N) then
            null;

         elsif Is_Return_By_Reference_Type (Typ) then
            Set_Returns_By_Ref (Spec_Id);

         elsif Present (Utyp) and then Controlled_Type (Utyp) then
            Set_Returns_By_Ref (Spec_Id);
         end if;
      end;

      --  Now, add a termination for all possible syntactic ends of the
      --  subprogram.  We don't bother to reanalyze the new body with the added
      --  return statement, since it would involve a lot of unnecessary work
      --  that would achieve precisely nothing.

      Add_Termination (Statements (H));

      if Present (Exception_Handlers (H)) then
         Except_H := First_Non_Pragma (Exception_Handlers (H));

         while Present (Except_H) loop
            Add_Termination (Statements (Except_H));
            Except_H := Next_Non_Pragma (Except_H);
         end loop;
      end if;

      Scop := Scope (Spec_Id);

      --  Add discriminal renamings to protected subprograms.
      --  Install new discriminals for expansion of the next
      --  subprogram of this protected type, if any.

      if Is_List_Member (N)
        and then Present (Parent (List_Containing (N)))
        and then Nkind (Parent (List_Containing (N))) = N_Protected_Body
      then
         Add_Discriminal_Declarations
           (Declarations (N), Scop, Name_uObject, Loc);
         Add_Private_Declarations (Declarations (N), Scop, Name_uObject, Loc);

         --  Associate privals and discriminals with the next protected
         --  operation body to be expanded. These are used to expand
         --  references to private data objects and discriminants,
         --  respectively.

         Next_Op := Next_Protected_Operation (N);

         if Present (Next_Op) then
            Dec := Parent (Base_Type (Scop));
            Set_Privals (Dec, Next_Op, Loc);
            Set_Discriminals (Dec, Next_Op, Loc);
         end if;

      end if;

   end Expand_N_Subprogram_Body;

   -------------------------------------
   -- Expand_N_Subprogram_Declaration --
   -------------------------------------

   --  The first task to be performed is the construction of
   --  default expression functions for in parameters with default values.
   --  These are parameterless inlined functions that are used to evaluate
   --  default expressions that are more complicated than simple literals
   --  or identifiers referencing constants and variables.

   --  If the declaration appears within a protected body, it is a private
   --  operation of the protected type. We must create the corresponding
   --  protected subprogram an associated formals. For a normal protected
   --  operation, this is done when expanding the protected type declaration.

   procedure Expand_N_Subprogram_Declaration (N : Node_Id) is
      Loc      : constant Source_Ptr := Sloc (N);
      Subp     : Entity_Id := Defining_Unit_Simple_Name (Specification (N));
      Scop     : Entity_Id := Scope (Subp);
      Prot_Sub : Entity_Id;
      Prot_Bod : Node_Id;

   begin
      Make_Default_Expr_Functions (N,
        Defining_Unit_Simple_Name (Specification (N)));

      if Is_List_Member (N)
        and then Present (Parent (List_Containing (N)))
        and then Nkind (Parent (List_Containing (N))) = N_Protected_Body
      then
         if No (Protected_Body_Subprogram (Subp)) then
            Prot_Sub :=
              Make_Subprogram_Declaration (Loc,
                Specification =>
                  Build_Protected_Sub_Specification
                    (N, Scop, Unprotected => True));

            --  The protected subprogram is declared outside of the protected
            --  body. Given that the body has frozen all entities so far, we
            --  freeze the subprogram explicitly.

            Prot_Bod := Parent (List_Containing (N));
            Insert_Before (Prot_Bod, Prot_Sub);

            New_Scope (Scope (Scop));
            Analyze (Prot_Sub);
            Set_Protected_Body_Subprogram (Subp,
              Defining_Unit_Name (Specification (Prot_Sub)));
            Pop_Scope;
         end if;
      end if;
   end Expand_N_Subprogram_Declaration;

   -----------------------
   -- Freeze_Subprogram --
   -----------------------

   procedure Freeze_Subprogram (N : Node_Id) is
      E : constant Entity_Id := Entity (N);

   begin
      --  When a primitive is frozen, enter its name in the corresponding
      --  dispatch table. If the DTC_Entity field is not set this is an
      --  overridden primitive that can be ignored.

      if Is_Dispatching_Operation (E)
        and then not Is_Abstract (E)
        and then Present (DTC_Entity (E))
        and then not Is_CPP_Class (Scope (DTC_Entity (E)))
      then
         Insert_After (N, Fill_DT_Entry (Sloc (N), E));
      end if;

      --  Mark functions that return by reference. Note that it cannot be
      --  part of the normal semantic analysis of the spec since the
      --  underlying returned type may not be known yet (for private types)

      declare
         Typ  : constant Entity_Id := Etype (E);
         Utyp : constant Entity_Id := Underlying_Type (Typ);

      begin
         if Is_Return_By_Reference_Type (Typ) then
            Set_Returns_By_Ref (E);

         elsif Present (Utyp) and then Controlled_Type (Utyp) then
            Set_Returns_By_Ref (E);
         end if;
      end;

   end Freeze_Subprogram;

end Exp_Ch6;
