/****************************************************************************
 * This module is based on Twm, but has been siginificantly modified 
 * by Rob Nation 
 ****************************************************************************/
/*****************************************************************************/
/**       Copyright 1988 by Evans & Sutherland Computer Corporation,        **/
/**                          Salt Lake City, Utah                           **/
/**  Portions Copyright 1989 by the Massachusetts Institute of Technology   **/
/**                        Cambridge, Massachusetts                         **/
/**                                                                         **/
/**                           All Rights Reserved                           **/
/**                                                                         **/
/**    Permission to use, copy, modify, and distribute this software and    **/
/**    its documentation  for  any  purpose  and  without  fee is hereby    **/
/**    granted, provided that the above copyright notice appear  in  all    **/
/**    copies and that both  that  copyright  notice  and  this  permis-    **/
/**    sion  notice appear in supporting  documentation,  and  that  the    **/
/**    names of Evans & Sutherland and M.I.T. not be used in advertising    **/
/**    in publicity pertaining to distribution of the  software  without    **/
/**    specific, written prior permission.                                  **/
/**                                                                         **/
/**    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD    **/
/**    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-    **/
/**    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR    **/
/**    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-    **/
/**    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA    **/
/**    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER    **/
/**    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE    **/
/**    OR PERFORMANCE OF THIS SOFTWARE.                                     **/
/*****************************************************************************/


/***********************************************************************
 *
 * mwm window border drawing code
 *
 ***********************************************************************/

#include "mwm.h"

void            DrawLinePattern(Window win,
				GC ReliefGC,
				GC ShadowGC,
				int num_coords,
				int *x_coord,
				int *y_coord,
				int *line_style,
				int th);

int
lbut_styles[2] = {
    22,
    55
};
int rbut_style_min[2] = {
    22,
    22
};
int rbut_style_max[2] = {
    55,
    55
};

/* macro to change window background color/pixmap */
#define ChangeWindowColor(window) {\
        if(NewColor)\
           {\
             XChangeWindowAttributes(dpy,window,valuemask, &attributes);\
             XClearWindow(dpy,window);\
           }\
         }

#define IsHighlight(w)	(Scr.Hilite == (w))
extern Window   PressedW;

static void
SetValAttr(MwmWindow *t, unsigned long *valuemask,
	   XSetWindowAttributes *attributes, ComponentInfo *comp) {

    *valuemask = CWBorderPixel;
    if (IsHighlight(t)) {
	attributes->border_pixel = comp->active_background;
	if (Scr.d_depth < 2 &&
	    comp->active_background_pixmap != XmUNSPECIFIED_PIXMAP) {
	    attributes->background_pixmap = comp->active_background_pixmap;
	    *valuemask |= CWBackPixmap;
	}
	else {
	    attributes->background_pixel = comp->active_background;
	    *valuemask |= CWBackPixel;
	}
    }
    else {
	attributes->border_pixel = comp->background;
	if (Scr.d_depth < 2 &&
	    comp->background_pixmap != XmUNSPECIFIED_PIXMAP) {
	    attributes->background_pixmap = comp->background_pixmap;
	    *valuemask |= CWBackPixmap;
	}
	else {
	    attributes->background_pixel = comp->background;
	    *valuemask |= CWBackPixel;
	}
    }
}

/****************************************************************************
 *
 * Redraws the windows borders
 *
 ****************************************************************************/
void
SetBorder(MwmWindow *t, Bool onoroff, Bool force, Bool Mapped,
	  Window expose_win)
{
    Window          w = None;
    int             y, x;
    Bool            NewColor = False;
    XSetWindowAttributes attributes;
    unsigned long   valuemask;
    static unsigned int corners[4];

    corners[0] = TOP_HILITE | LEFT_HILITE;
    corners[1] = TOP_HILITE | RIGHT_HILITE;
    corners[2] = BOTTOM_HILITE | LEFT_HILITE;
    corners[3] = BOTTOM_HILITE | RIGHT_HILITE;

    if (!t)
	return;

    if (onoroff) {
	/* don't re-draw just for kicks */
	if ((!force) && (Scr.Hilite == t))
	    return;

	if (Scr.Hilite != t)
	    NewColor = True;

	/* make sure that the previously highlighted window got unhighlighted */
	if ((Scr.Hilite != t) && (Scr.Hilite != NULL))
	    SetBorder(Scr.Hilite, False, False, True, None);

	/* set the keyboard focus */
	if ((Mapped) && (t->flags & MAPPED) && (Scr.Hilite != t))
	    w = t->w;
	else if ((t->flags & ICONIFIED) &&
		 (Scr.Hilite != t) && (!(t->flags & SUPPRESSICON)))
	    w = t->icon_w;
	Scr.Hilite = t;
    }
    else {
	/* don't re-draw just for kicks */
	if ((!force) && (Scr.Hilite != t))
	    return;

	if (Scr.Hilite == t) {
	    Scr.Hilite = NULL;
	    NewColor = True;
	}
    }

    if ((Scr.Pager_w) && !(t->flags & STICKY)) {
	if (NewColor) {
	    if (Scr.d_depth < 2 &&
		Mwm.components[MWM_PAGER].background_pixmap
		 != XmUNSPECIFIED_PIXMAP)
		XSetWindowBackgroundPixmap(dpy,
					   t->pager_view,
					   Mwm.components[MWM_PAGER].background_pixmap);
	    else
		XSetWindowBackground(dpy, t->pager_view,
				     Mwm.components[MWM_PAGER].background);
	    XClearWindow(dpy, t->pager_view);
	}
	if ((t->icon_image != NULL) && (Scr.PagerFont.height > 0)) {
	    NewFontAndColor(Scr.PagerFont.font->fid,
			    Mwm.components[MWM_PAGER].foreground,
			    Mwm.components[MWM_PAGER].background);
	    XDrawImageString(dpy, t->pager_view, Scr.FontGC, 2, Scr.PagerFont.y + 2,
			     t->icon_image, strlen(t->icon_image));
	}
    }

    if (t->flags & ICONIFIED) {
	DrawIconWindow(t);
	return;
    }

    SetValAttr(t, &valuemask, &attributes, &Mwm.components[MWM_BORDER]);

    if (t->flags & (TITLE | BORDER)) {
	XSetWindowBorder(dpy, t->Parent, attributes.background_pixel);
	XSetWindowBorder(dpy, t->frame, attributes.background_pixel);
    }


    if (t->flags & TITLE) {
	SetValAttr(t, &valuemask, &attributes, &Mwm.components[MWM_TITLE_AREA]);
	ChangeWindowColor(t->title_w);
	if (t->menub != None) {
	    if (IsHighlight(t)) {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_MENU_BUTTON].active_top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_MENU_BUTTON].active_bottom_shadow_color);
	    }
	    else {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_MENU_BUTTON].top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_MENU_BUTTON].bottom_shadow_color);
	    }
	    SetValAttr(t, &valuemask, &attributes, &Mwm.components[MWM_MENU_BUTTON]);
	    ChangeWindowColor(t->menub);
	    if (flush_expose(t->menub) || (expose_win == t->menub) ||
		(expose_win == None)) {
		RelieveWindow(t, t->menub, 0, 0, t->title_height,
			      t->title_height,
			      (PressedW == t->menub
				? Scr.bot_GC
				: Scr.top_GC),
			      (PressedW == t->menub
				? Scr.top_GC
				: Scr.bot_GC),
			      BOTTOM_HILITE);
		DrawPattern(t->menub, Scr.top_GC, Scr.bot_GC,
			    lbut_styles[0],
			    lbut_styles[1],
			    t->title_height);
	    }
	}
	if (t->maximizeb != None) {
	    SetValAttr(t, &valuemask, &attributes,
		       &Mwm.components[MWM_MAXIMIZE_BUTTON]);
	    ChangeWindowColor(t->maximizeb);
	    if (IsHighlight(t)) {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_MAXIMIZE_BUTTON].active_top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_MAXIMIZE_BUTTON].active_bottom_shadow_color);
	    }
	    else {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_MAXIMIZE_BUTTON].top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_MAXIMIZE_BUTTON].bottom_shadow_color);
	    }
	    if (flush_expose(t->maximizeb) || (expose_win == t->maximizeb) ||
		(expose_win == None)) {
		RelieveWindow(t, t->maximizeb, 0, 0, t->title_height,
			      t->title_height,
			      (PressedW == t->maximizeb
				? Scr.bot_GC
				: Scr.top_GC),
			      (PressedW == t->maximizeb
				? Scr.top_GC
				: Scr.bot_GC),
			      BOTTOM_HILITE);
		DrawPattern(t->maximizeb, Scr.top_GC, Scr.bot_GC,
			    rbut_style_max[0],
			    rbut_style_max[1],
			    t->title_height);
 	    }
	}
	if (t->minimizeb != None) {
	    SetValAttr(t, &valuemask, &attributes, &Mwm.components[MWM_MINIMIZE_BUTTON]);
	    ChangeWindowColor(t->minimizeb);
	    if (IsHighlight(t)) {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_MINIMIZE_BUTTON].active_top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_MINIMIZE_BUTTON].active_bottom_shadow_color);
	    }
	    else {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_MINIMIZE_BUTTON].top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_MINIMIZE_BUTTON].bottom_shadow_color);
	    }
	    if (flush_expose(t->minimizeb) || (expose_win == t->minimizeb) ||
		(expose_win == None)) {
		RelieveWindow(t, t->minimizeb, 0, 0, t->title_height,
			      t->title_height,
			      (PressedW == t->minimizeb
				? Scr.bot_GC
				: Scr.top_GC),
			      (PressedW == t->minimizeb
				? Scr.top_GC
				: Scr.bot_GC),
			      BOTTOM_HILITE);
		DrawPattern(t->minimizeb, Scr.top_GC, Scr.bot_GC,
			    rbut_style_min[0],
			    rbut_style_min[1],
			    t->title_height);
 	    }
	}
	SetTitleBar(t, onoroff, False);
    }

    if (t->flags & BORDER) {
	int i;

	/* draw relief lines */
	y = t->frame_height - 2 * t->corner_width;
	x = t->frame_width - 2 * t->corner_width + t->bw;

	for (i = 0; i < 4; i++) {
	    SetValAttr(t, &valuemask, &attributes, &Mwm.components[MWM_BORDER]);
	    ChangeWindowColor(t->sides[i]);
	    if (IsHighlight(t)) {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_BORDER].active_top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_BORDER].active_bottom_shadow_color);
	    }
	    else {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_BORDER].top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_BORDER].bottom_shadow_color);
	    }
	    if ((flush_expose(t->sides[i])) || (expose_win == t->sides[i]) ||
		(expose_win == None)) {
		GC              sgc, rgc;

		sgc = Scr.bot_GC;
		rgc = Scr.top_GC;
		/* index    side
		 * 0        TOP
		 * 1        RIGHT
		 * 2        BOTTOM
		 * 3        LEFT
		 */

		RelieveWindow(t, t->sides[i], 0, 0,
			      ((i % 2) ? t->boundary_width : x),
			      ((i % 2) ? y : t->boundary_width),
			      rgc, sgc, (0x0001 << i));
	    }

	    SetValAttr(t, &valuemask, &attributes, &Mwm.components[MWM_RESIZE_HANDLES]);
	    ChangeWindowColor(t->corners[i]);
	    if (IsHighlight(t)) {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_RESIZE_HANDLES].active_top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_RESIZE_HANDLES].active_bottom_shadow_color);
	    }
	    else {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_RESIZE_HANDLES].top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_RESIZE_HANDLES].bottom_shadow_color);
	    }
	    if ((flush_expose(t->corners[i])) || (expose_win == t->corners[i]) ||
		(expose_win == None)) {
		GC              rgc, sgc;

		rgc = Scr.top_GC;
		sgc = Scr.bot_GC;
		RelieveWindow(t, t->corners[i], 0, 0, t->corner_width,
		      ((i / 2) ? t->corner_width + t->bw : t->corner_width),
			      rgc, sgc, corners[i]);
		if (t->boundary_width > 1)
		    RelieveParts(t, i, ((i / 2) ? rgc : sgc), ((i % 2) ? rgc : sgc));
		else
		    RelieveParts(t, i, ((i / 2) ? sgc : sgc), ((i % 2) ? sgc : sgc));
	    }
	}
    }
    else {			/* no decorative border */
	/* for mono - put a black border on 
	 * for color, make it the color of the decoration background */
	if (t->boundary_width < 2) {
	    flush_expose(t->frame);
	    XSetWindowBorder(dpy, t->frame, Mwm.components[MWM_BORDER].background);
	    XSetWindowBorder(dpy, t->Parent, Mwm.components[MWM_BORDER].background);
	    if (Mwm.components[MWM_BORDER].background_pixmap)
		XSetWindowBackgroundPixmap(dpy, t->frame, Mwm.components[MWM_BORDER].background_pixmap);
	    XClearWindow(dpy, t->frame);
	    if (Mwm.components[MWM_BORDER].background_pixmap)
		XSetWindowBackgroundPixmap(dpy, t->Parent, Mwm.components[MWM_BORDER].background_pixmap);
	    XClearWindow(dpy, t->Parent);
	}
	else {
	    GC              rgc, sgc;

	    if (IsHighlight(t)) {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_BORDER].active_top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_BORDER].active_bottom_shadow_color);
	    }
	    else {
		XSetForeground(dpy, Scr.top_GC,
			 Mwm.components[MWM_BORDER].top_shadow_color);
		XSetForeground(dpy, Scr.bot_GC,
			 Mwm.components[MWM_BORDER].bottom_shadow_color);
	    }
	    XSetWindowBorder(dpy, t->Parent, Mwm.components[MWM_BORDER].background);
	    XSetWindowBorder(dpy, t->frame, Mwm.components[MWM_BORDER].background);

	    rgc = Scr.top_GC;
	    sgc = Scr.bot_GC;
	    ChangeWindowColor(t->frame);
	    if ((flush_expose(t->frame)) || (expose_win == t->frame) ||
		(expose_win == None)) {
		if (t->boundary_width > 2) {
		    RelieveWindow(t, t->frame, t->boundary_width - 1 - t->bw,
				  t->boundary_width - 1 - t->bw,
				  t->frame_width -
				  (t->boundary_width << 1) + 2 + 3 * t->bw,
				  t->frame_height -
				  (t->boundary_width << 1) + 2 + 3 * t->bw,
				  sgc, rgc,
				  TOP_HILITE | LEFT_HILITE | RIGHT_HILITE |
				  BOTTOM_HILITE);
		    RelieveWindow(t, t->frame, 0, 0, t->frame_width + t->bw,
				  t->frame_height + t->bw, rgc, sgc,
				  TOP_HILITE | LEFT_HILITE | RIGHT_HILITE |
				  BOTTOM_HILITE);
		}
		else {
		    RelieveWindow(t, t->frame, 0, 0, t->frame_width + t->bw,
				  t->frame_height + t->bw, rgc, rgc,
				  TOP_HILITE | LEFT_HILITE | RIGHT_HILITE |
				  BOTTOM_HILITE);
		}
	    }
	    else {
		XSetWindowBackground(dpy, t->Parent, Mwm.components[MWM_BORDER].background);
	    }
	}
    }
}


/****************************************************************************
 *
 *  Redraws just the title bar
 *
 ****************************************************************************/
void
SetTitleBar(MwmWindow * t, Bool onoroff, Bool NewTitle)
{
    int             hor_off, w;
    GC              ReliefGC, ShadowGC, tGC;
    Pixel           Forecolor, BackColor;

    if (!t)
	return;
    if (!(t->flags & TITLE))
	return;

    if (onoroff) {
	XSetForeground(dpy, Scr.top_GC,
		 Mwm.components[MWM_TITLE_AREA].active_top_shadow_color);
	XSetForeground(dpy, Scr.bot_GC,
		 Mwm.components[MWM_TITLE_AREA].active_bottom_shadow_color);
	Forecolor = Mwm.components[MWM_TITLE_AREA].active_foreground;
	BackColor = Mwm.components[MWM_TITLE_AREA].active_background;
	ReliefGC = (PressedW == t->title_w ? Scr.bot_GC : Scr.top_GC);
	ShadowGC = (PressedW == t->title_w ? Scr.top_GC : Scr.bot_GC);
    }
    else {
	XSetForeground(dpy, Scr.top_GC,
		 Mwm.components[MWM_BORDER].top_shadow_color);
	XSetForeground(dpy, Scr.bot_GC,
		 Mwm.components[MWM_BORDER].bottom_shadow_color);
	Forecolor = Mwm.components[MWM_TITLE_AREA].foreground;
	BackColor = Mwm.components[MWM_TITLE_AREA].background;
	ReliefGC = (PressedW == t->title_w ? Scr.bot_GC : Scr.top_GC);
	ShadowGC = (PressedW == t->title_w ? Scr.top_GC : Scr.bot_GC);
    }
    flush_expose(t->title_w);

    if (t->name != (char *) NULL) {
	w = XTextWidth(Scr.WindowFont.font, t->name, strlen(t->name));
	if (w > t->title_width - 12)
	    w = t->title_width - 4;
	if (w < 0)
	    w = 0;
    }
    else
	w = 0;


    hor_off = (t->title_width - w) / 2;

    NewFontAndColor(Scr.WindowFont.font->fid, Forecolor, BackColor);

    if (NewTitle)
	XClearWindow(dpy, t->title_w);

    /* for mono, we clear an area in the title bar where the window
     * title goes, so that its more legible. For color, no need */
    if (Scr.d_depth < 2) {
	RelieveWindow(t, t->title_w, 0, 0, hor_off - 2, t->title_height,
		      ReliefGC, ShadowGC, BOTTOM_HILITE);
	RelieveWindow(t, t->title_w, hor_off + w + 2, 0,
		      t->title_width - w - hor_off - 2, t->title_height,
		      ReliefGC, ShadowGC, BOTTOM_HILITE);
	XFillRectangle(dpy, t->title_w,
		       (PressedW == t->title_w ? ShadowGC : ReliefGC),
		       hor_off - 2, 0, w + 4, t->title_height);

	XDrawLine(dpy, t->title_w, ShadowGC, hor_off + w + 1, 0, hor_off + w + 1,
		  t->title_height);
	if (t->name != (char *) NULL)
	    XDrawString(dpy, t->title_w, Scr.FontGC, hor_off, Scr.WindowFont.y + 1,
			t->name, strlen(t->name));
    }
    else {
	if (t->name != (char *) NULL)
	    XDrawString(dpy, t->title_w, Scr.FontGC, hor_off, Scr.WindowFont.y + 1,
			t->name, strlen(t->name));
	RelieveWindow(t, t->title_w, 0, 0, t->title_width, t->title_height,
		      ReliefGC, ShadowGC, BOTTOM_HILITE);
    }

    XFlush(dpy);
}




/****************************************************************************
 *
 *  Draws the relief pattern around a window
 *
 ****************************************************************************/
#ifdef __GNUC__
inline void
#else
void
#endif
RelieveWindow(MwmWindow * t, Window win,
	      int x, int y, int w, int h,
	      GC ReliefGC, GC ShadowGC, int hilite)
{
    XSegment        seg[4];
    int             i;
    int             edge;

    edge = 0;
    if ((win == t->sides[0]) || (win == t->sides[1]) ||
	(win == t->sides[2]) || (win == t->sides[3]))
	edge = -1;
    if (win == t->corners[0])
	edge = 1;
    if (win == t->corners[1])
	edge = 2;
    if (win == t->corners[2])
	edge = 3;
    if (win == t->corners[3])
	edge = 4;

    i = 0;
    seg[i].x1 = x;
    seg[i].y1 = y;
    seg[i].x2 = w + x - 1;
    seg[i++].y2 = y;

    seg[i].x1 = x;
    seg[i].y1 = y;
    seg[i].x2 = x;
    seg[i++].y2 = h + y - 1;

    XDrawSegments(dpy, win, ReliefGC, seg, i);

    i = 0;
    seg[i].x1 = x;
    seg[i].y1 = y + h - 1;
    seg[i].x2 = w + x - 1;
    seg[i++].y2 = y + h - 1;

    seg[i].x1 = x + w - 1;
    seg[i].y1 = y;
    seg[i].x2 = x + w - 1;
    seg[i++].y2 = y + h - 1;

    XDrawSegments(dpy, win, ShadowGC, seg, i);
}

void
RelieveParts(MwmWindow * t, int i, GC hor, GC vert)
{
    XSegment        seg[2];
    int             n = 0;

    if (t->boundary_width < 3) {
	switch (i) {
	case 0:
	    seg[0].x1 = t->boundary_width - 1;
	    seg[0].x2 = t->corner_width;
	    seg[0].y1 = t->boundary_width - 1;
	    seg[0].y2 = t->boundary_width - 1;
	    n = 1;
	    break;
	case 1:
	    seg[0].x1 = 0;
	    seg[0].x2 = t->corner_width - t->boundary_width /* -1 */ ;
	    seg[0].y1 = t->boundary_width - 1;
	    seg[0].y2 = t->boundary_width - 1;
	    n = 1;
	    break;
	case 2:
	    seg[0].x1 = t->boundary_width - 1;
	    seg[0].x2 = t->corner_width - 2;
	    seg[0].y1 = t->corner_width - t->boundary_width + t->bw;
	    seg[0].y2 = t->corner_width - t->boundary_width + t->bw;
	    n = 1;
	    break;
	case 3:
	    seg[0].x1 = 0;
	    seg[0].x2 = t->corner_width - t->boundary_width;
	    seg[0].y1 = t->corner_width - t->boundary_width + t->bw;
	    seg[0].y2 = t->corner_width - t->boundary_width + t->bw;
	    n = 1;
	    break;
	}
	XDrawSegments(dpy, t->corners[i], hor, seg, n);
	switch (i) {
	case 0:
	    seg[0].y1 = t->boundary_width - 1;
	    seg[0].y2 = t->corner_width;
	    seg[0].x1 = t->boundary_width - 1;
	    seg[0].x2 = t->boundary_width - 1;
	    n = 1;
	    break;
	case 1:
	    seg[0].y1 = t->boundary_width - 1;
	    seg[0].y2 = t->corner_width - 2;
	    seg[0].x1 = t->corner_width - t->boundary_width;
	    seg[0].x2 = t->corner_width - t->boundary_width;
	    n = 1;
	    break;
	case 2:
	    seg[0].y1 = 0;
	    seg[0].y2 = t->corner_width - t->boundary_width;
	    seg[0].x1 = t->boundary_width - 1;
	    seg[0].x2 = t->boundary_width - 1;
	    n = 1;
	    break;
	case 3:
	    seg[0].y1 = 0;
	    seg[0].y2 = t->corner_width - t->boundary_width + t->bw;
	    seg[0].x1 = t->corner_width - t->boundary_width;
	    seg[0].x2 = t->corner_width - t->boundary_width;
	    n = 1;
	    break;
	}
	XDrawSegments(dpy, t->corners[i], vert, seg, 1);
    }
    else {
	switch (i) {
	case 0:
	    seg[0].x1 = t->boundary_width - 2;
	    seg[0].x2 = t->corner_width;
	    seg[0].y1 = t->boundary_width - 2;
	    seg[0].y2 = t->boundary_width - 2;

	    seg[1].x1 = t->boundary_width - 2;
	    seg[1].x2 = t->corner_width;
	    seg[1].y1 = t->boundary_width - 1;
	    seg[1].y2 = t->boundary_width - 1;
	    n = 2;
	    break;
	case 1:
	    seg[0].x1 = 1;
	    seg[0].x2 = t->corner_width - t->boundary_width;
	    seg[0].y1 = t->boundary_width - 2;
	    seg[0].y2 = t->boundary_width - 2;

	    seg[1].x1 = 0;
	    seg[1].x2 = t->corner_width - t->boundary_width - 1;
	    seg[1].y1 = t->boundary_width - 1;
	    seg[1].y2 = t->boundary_width - 1;
	    n = 2;
	    break;
	case 2:
	    seg[0].x1 = t->boundary_width - 1;
	    seg[0].x2 = t->corner_width - 2;
	    seg[0].y1 = t->corner_width - t->boundary_width + 1;
	    seg[0].y2 = t->corner_width - t->boundary_width + 1;
	    n = 1;
	    if (t->boundary_width > 3) {
		seg[1].x1 = t->boundary_width - 2;
		seg[1].x2 = t->corner_width - 3;
		seg[1].y1 = t->corner_width - t->boundary_width + 2;
		seg[1].y2 = t->corner_width - t->boundary_width + 2;
		n = 2;
	    }
	    break;
	case 3:
	    seg[0].x1 = 0;
	    seg[0].x2 = t->corner_width - t->boundary_width;
	    seg[0].y1 = t->corner_width - t->boundary_width + 1;
	    seg[0].y2 = t->corner_width - t->boundary_width + 1;
	    n = 1;
	    if (t->boundary_width > 3) {
		seg[0].x2 = t->corner_width - t->boundary_width + 1;

		seg[1].x1 = 0;
		seg[1].x2 = t->corner_width - t->boundary_width + 1;
		seg[1].y1 = t->corner_width - t->boundary_width + 2;
		seg[1].y2 = t->corner_width - t->boundary_width + 2;
		n = 2;
	    }
	    break;
	}
	XDrawSegments(dpy, t->corners[i], hor, seg, n);
	switch (i) {
	case 0:
	    seg[0].y1 = t->boundary_width - 2;
	    seg[0].y2 = t->corner_width;
	    seg[0].x1 = t->boundary_width - 2;
	    seg[0].x2 = t->boundary_width - 2;

	    seg[1].y1 = t->boundary_width - 2;
	    seg[1].y2 = t->corner_width;
	    seg[1].x1 = t->boundary_width - 1;
	    seg[1].x2 = t->boundary_width - 1;
	    n = 2;
	    break;
	case 1:
	    seg[0].y1 = t->boundary_width - 1;
	    seg[0].y2 = t->corner_width - 2;
	    seg[0].x1 = t->corner_width - t->boundary_width;
	    seg[0].x2 = t->corner_width - t->boundary_width;
	    n = 1;
	    if (t->boundary_width > 3) {
		seg[1].y1 = t->boundary_width - 2;
		seg[1].y2 = t->corner_width - 3;
		seg[1].x1 = t->corner_width - t->boundary_width + 1;
		seg[1].x2 = t->corner_width - t->boundary_width + 1;
		n = 2;
	    }
	    break;
	case 2:
	    seg[0].y1 = 1;
	    seg[0].y2 = t->corner_width - t->boundary_width + 1;
	    seg[0].x1 = t->boundary_width - 2;
	    seg[0].x2 = t->boundary_width - 2;
	    n = 1;

	    if (t->boundary_width > 3) {
		seg[1].y1 = 0;
		seg[1].y2 = t->corner_width - t->boundary_width;
		seg[1].x1 = t->boundary_width - 1;
		seg[1].x2 = t->boundary_width - 1;
	    }
	    break;
	case 3:
	    seg[0].y1 = 0;
	    seg[0].y2 = t->corner_width - t->boundary_width + 1;
	    seg[0].x1 = t->corner_width - t->boundary_width;
	    seg[0].x2 = t->corner_width - t->boundary_width;
	    n = 1;

	    if (t->boundary_width > 3) {
		seg[0].y2 = t->corner_width - t->boundary_width + 2;
		seg[1].y1 = 0;
		seg[1].y2 = t->corner_width - t->boundary_width + 2;
		seg[1].x1 = t->corner_width - t->boundary_width + 1;
		seg[1].x2 = t->corner_width - t->boundary_width + 1;
		n = 2;
	    }
	    break;
	}
	XDrawSegments(dpy, t->corners[i], vert, seg, n);
    }
}
/****************************************************************************
 *
 *  Draws a little pattern within a window
 *
 ****************************************************************************/
#ifdef __GNUC__
inline void
#else
void
#endif
DrawPattern(Window win, GC ShadowGC, GC ReliefGC, int h1, int w1, int t1)
{
    XSegment        seg[2];
    int             i, h, b, u, w, r, l;

    h = t1 * h1 / 200;
    b = (t1 >> 1) + h;
    u = t1 - b - 1;
    w = t1 * w1 / 200;
    r = (t1 >> 1) + w;
    l = t1 - r - 1;

    i = 0;
    seg[i].x1 = l;
    seg[i].y1 = u;
    seg[i].x2 = r;
    seg[i++].y2 = u;

    seg[i].x1 = l;
    seg[i].y1 = u;
    seg[i].x2 = l;
    seg[i++].y2 = b;
    XDrawSegments(dpy, win, ShadowGC, seg, i);

    i = 0;
    seg[i].x1 = l;
    seg[i].y1 = b;
    seg[i].x2 = r;
    seg[i++].y2 = b;

    seg[i].x1 = r;
    seg[i].y1 = u;
    seg[i].x2 = r;
    seg[i++].y2 = b;
    XDrawSegments(dpy, win, ReliefGC, seg, i);
}

/****************************************************************************
 *
 *  Draws a little pattern within a window (more complex)
 *
 ****************************************************************************/
void
DrawLinePattern(Window win,
		GC ReliefGC,
		GC ShadowGC,
		int num_coords,
		int *x_coord,
		int *y_coord,
		int *line_style,
		int th)
{
    int             i;

    for (i = 1; i < num_coords; i++) {
	XDrawLine(dpy, win, line_style[i] ? ReliefGC : ShadowGC,
		  th * x_coord[i - 1] / 100,
		  th * y_coord[i - 1] / 100,
		  th * x_coord[i] / 100,
		  th * y_coord[i] / 100);
    }
}



/***********************************************************************
 *
 *  Procedure:
 *      Setupframe - set window sizes, this was called from either
 *              AddWindow, EndResize, or HandleConfigureNotify.
 *
 *  Inputs:
 *      tmp_win - the MwmWindow pointer
 *      x       - the x coordinate of the upper-left outer corner of the frame
 *      y       - the y coordinate of the upper-left outer corner of the frame
 *      w       - the width of the frame window w/o border
 *      h       - the height of the frame window w/o border
 *
 *  Special Considerations:
 *      This routine will check to make sure the window is not completely
 *      off the display, if it is, it'll bring some of it back on.
 *
 *      The tmp_win->frame_XXX variables should NOT be updated with the
 *      values of x,y,w,h prior to calling this routine, since the new
 *      values are compared against the old to see whether a synthetic
 *      ConfigureNotify event should be sent.  (It should be sent if the
 *      window was moved but not resized.)
 *
 ************************************************************************/

void
SetupFrame(MwmWindow * tmp_win, int x, int y, int w, int h, Bool sendEvent)
{
    XEvent          client_event;
    XWindowChanges  frame_wc, xwc;
    unsigned long   frame_mask, xwcm;
    int             cx, cy, i;
    Bool            Resized = False;
    MwmWindow      *t;
    int             xwidth, ywidth;

    /* if windows is not being maximized, save size in case of maximization */
    if (!(tmp_win->flags & MAXIMIZED)) {
	tmp_win->orig_x = x;
	tmp_win->orig_y = y;
	tmp_win->orig_wd = w;
	tmp_win->orig_ht = h;
    }
    if (Scr.flags & DontMoveOff) {
	if (x + Scr.Vx + w < 16)
	    x = 16 - Scr.Vx - w;
	if (y + Scr.Vy + h < 16)
	    y = 16 - Scr.Vy - h;
    }
    if (x >= Scr.d_width + Scr.VxMax - Scr.Vx - 16)
	x = Scr.d_width + Scr.VxMax - Scr.Vx - 16;
    if (y >= Scr.d_height + Scr.VyMax - Scr.Vy - 16)
	y = Scr.d_height + Scr.VyMax - Scr.Vy - 16;

    /*
     * According to the July 27, 1988 ICCCM draft, we should send a
     * "synthetic" ConfigureNotify event to the client if the window
     * was moved but not resized.
     */
    if ((x != tmp_win->frame_x || y != tmp_win->frame_y) &&
	(w == tmp_win->frame_width && h == tmp_win->frame_height))
	sendEvent = TRUE;

    if ((w != tmp_win->frame_width) || (h != tmp_win->frame_height))
	Resized = True;

    if (Resized) {
	tmp_win->title_width = w -
	    (3) * tmp_win->title_height
	    - 2 * tmp_win->boundary_width + tmp_win->bw;


	if (tmp_win->title_width < 1)
	    tmp_win->title_width = 1;

	if (tmp_win->flags & TITLE) {
	    xwcm = CWWidth | CWX | CWY;
	    tmp_win->title_x = tmp_win->boundary_width +
		(1) * tmp_win->title_height;
	    if (tmp_win->title_x >= w - tmp_win->boundary_width)
		tmp_win->title_x = -10;
	    tmp_win->title_y = tmp_win->boundary_width;

	    xwc.width = tmp_win->title_width;
	    xwc.x = tmp_win->title_x;
	    xwc.y = tmp_win->title_y;
	    XConfigureWindow(dpy, tmp_win->title_w, xwcm, &xwc);


	    xwcm = CWX | CWY;
	    xwc.y = tmp_win->boundary_width;

	    xwc.x = tmp_win->boundary_width;
	    if (tmp_win->menub != None) {
		if (xwc.x + tmp_win->title_height < w - tmp_win->boundary_width)
		    XConfigureWindow(dpy, tmp_win->menub, xwcm, &xwc);
		else {
		    xwc.x = -tmp_win->title_height;
		    XConfigureWindow(dpy, tmp_win->menub, xwcm, &xwc);
		}
		xwc.x += tmp_win->title_height;
	    }

	    xwc.x = w - tmp_win->boundary_width + tmp_win->bw;
	    if (tmp_win->maximizeb != None) {
		xwc.x -= tmp_win->title_height;
		if (xwc.x > tmp_win->boundary_width)
		    XConfigureWindow(dpy, tmp_win->maximizeb, xwcm, &xwc);
		else {
		    xwc.x = -tmp_win->title_height;
		    XConfigureWindow(dpy, tmp_win->maximizeb, xwcm, &xwc);
		}
	    }
	    if (tmp_win->minimizeb != None) {
		xwc.x -= tmp_win->title_height;
		if (xwc.x > tmp_win->boundary_width)
		    XConfigureWindow(dpy, tmp_win->minimizeb, xwcm, &xwc);
		else {
		    xwc.x = -tmp_win->title_height;
		    XConfigureWindow(dpy, tmp_win->minimizeb, xwcm, &xwc);
		}
	    }
	}

	if (tmp_win->flags & BORDER) {
	    xwidth = w - 2 * tmp_win->corner_width + tmp_win->bw;
	    ywidth = h - 2 * tmp_win->corner_width;
	    xwcm = CWWidth | CWHeight | CWX | CWY;
	    if (xwidth < 2)
		xwidth = 2;
	    if (ywidth < 2)
		ywidth = 2;

	    for (i = 0; i < 4; i++) {
		if (i == 0) {
		    xwc.x = tmp_win->corner_width;
		    xwc.y = 0;
		    xwc.height = tmp_win->boundary_width;
		    xwc.width = xwidth;
		}
		else if (i == 1) {
		    xwc.x = w - tmp_win->boundary_width + tmp_win->bw;
		    xwc.y = tmp_win->corner_width;
		    xwc.width = tmp_win->boundary_width;
		    xwc.height = ywidth;

		}
		else if (i == 2) {
		    xwc.x = tmp_win->corner_width;
		    xwc.y = h - tmp_win->boundary_width + tmp_win->bw;
		    xwc.height = tmp_win->boundary_width + tmp_win->bw;
		    xwc.width = xwidth;
		}
		else {
		    xwc.x = 0;
		    xwc.y = tmp_win->corner_width;
		    xwc.width = tmp_win->boundary_width;
		    xwc.height = ywidth;
		}
		XConfigureWindow(dpy, tmp_win->sides[i], xwcm, &xwc);
	    }

	    xwcm = CWX | CWY;
	    for (i = 0; i < 4; i++) {
		if (i % 2)
		    xwc.x = w - tmp_win->corner_width + tmp_win->bw;
		else
		    xwc.x = 0;

		if (i / 2)
		    xwc.y = h - tmp_win->corner_width;
		else
		    xwc.y = 0;

		XConfigureWindow(dpy, tmp_win->corners[i], xwcm, &xwc);
	    }

	}
    }
    tmp_win->attr.width = w - 2 * tmp_win->boundary_width;
    tmp_win->attr.height = h - tmp_win->title_height
	- 2 * tmp_win->boundary_width;
    /* may need to omit the -1 for shaped windows, next two lines */
    cx = tmp_win->boundary_width - tmp_win->bw;
    cy = tmp_win->title_height + tmp_win->boundary_width - tmp_win->bw;

    XResizeWindow(dpy, tmp_win->w, tmp_win->attr.width,
		  tmp_win->attr.height);
    XMoveResizeWindow(dpy, tmp_win->Parent, cx, cy,
		      tmp_win->attr.width, tmp_win->attr.height);

    /* 
     * fix up frame and assign size/location values in tmp_win
     */
    frame_wc.x = tmp_win->frame_x = x;
    frame_wc.y = tmp_win->frame_y = y;
    frame_wc.width = tmp_win->frame_width = w;
    frame_wc.height = tmp_win->frame_height = h;
    frame_mask = (CWX | CWY | CWWidth | CWHeight);
    XConfigureWindow(dpy, tmp_win->frame, frame_mask, &frame_wc);


    if ((Resized) && (tmp_win->wShaped)) {
	SetShape(tmp_win, w);
    }
    XSync(dpy, 0);
    if (sendEvent) {
	client_event.type = ConfigureNotify;
	client_event.xconfigure.display = dpy;
	client_event.xconfigure.event = tmp_win->w;
	client_event.xconfigure.window = tmp_win->w;

	client_event.xconfigure.x = x + tmp_win->boundary_width;
	client_event.xconfigure.y = y + tmp_win->title_height +
	    tmp_win->boundary_width;
	client_event.xconfigure.width = w - 2 * tmp_win->boundary_width;
	client_event.xconfigure.height = h - 2 * tmp_win->boundary_width -
	    tmp_win->title_height;

	client_event.xconfigure.border_width = tmp_win->bw;
	/* Real ConfigureNotify events say we're above title window, so ... */
	/* what if we don't have a title ????? */
	client_event.xconfigure.above = tmp_win->frame;
	client_event.xconfigure.override_redirect = False;
	XSendEvent(dpy, tmp_win->w, False, StructureNotifyMask, &client_event);
    }
    if (tmp_win == Scr.MwmPager) {
	MoveResizeViewPortIndicator();
	for (t = Scr.MwmRoot.next; t != NULL; t = t->next) {
	    MoveResizePagerView(t);
	}
    }
    else
	MoveResizePagerView(tmp_win);
}


/****************************************************************************
 *
 * Sets up the shaped window borders 
 * 
 ****************************************************************************/
void
SetShape(MwmWindow * tmp_win, int w)
{
    XRectangle      rect;

    XShapeCombineShape(dpy, tmp_win->frame, ShapeBounding,
		       tmp_win->boundary_width,
		       tmp_win->title_height + tmp_win->boundary_width,
		       tmp_win->w,
		       ShapeBounding, ShapeSet);
    if (tmp_win->title_w) {
	/* windows w/ titles */
	rect.x = tmp_win->boundary_width;
	rect.y = tmp_win->title_y;
	rect.width = w - 2 * tmp_win->boundary_width + tmp_win->bw;
	rect.height = tmp_win->title_height;


	XShapeCombineRectangles(dpy, tmp_win->frame, ShapeBounding,
				0, 0, &rect, 1, ShapeUnion, Unsorted);
    }
}

/********************************************************************
 *
 * Sets the input focus to the indicated window.
 *
 **********************************************************************/

void
SetFocus(Window w, MwmWindow * Fw)
{
    int             i;
    extern Time     lastTimestamp;

    /* XmEXPLICIT keyboard focus policy queue manipulation */
    if (Fw && Fw != Scr.Focus && Fw != &Scr.MwmRoot) {
	MwmWindow      *tmp_win1, *tmp_win2;

	tmp_win1 = Fw->prev;
	tmp_win2 = Fw->next;

	if (tmp_win1)
	    tmp_win1->next = tmp_win2;
	if (tmp_win2)
	    tmp_win2->prev = tmp_win1;

	Fw->next = Scr.MwmRoot.next;
	if (Scr.MwmRoot.next)
	    Scr.MwmRoot.next->prev = Fw;
	Scr.MwmRoot.next = Fw;
	Fw->prev = &Scr.MwmRoot;
    }

    if (Mwm.number_of_screens > 1) {
	XQueryPointer(dpy, Scr.root_win, &JunkRoot, &JunkChild,
		      &JunkX, &JunkY, &JunkX, &JunkY, &JunkMask);
	if (JunkRoot != Scr.root_win) {
	    if ((Mwm.keyboard_focus_policy == XmEXPLICIT) &&
		(Scr.Ungrabbed != NULL)) {
		/* Need to grab buttons for focus window */
		XSync(dpy, 0);
		for (i = 0; i < 3; i++)
		    if (Scr.buttons2grab & (1 << i)) {
			XGrabButton(dpy, (i + 1), 0, Scr.Ungrabbed->frame, True,
			       ButtonPressMask, GrabModeSync, GrabModeAsync,
				    None, Scr.MwmCursors[SYS_CURS]);
			XGrabButton(dpy, (i + 1), LockMask, Scr.Ungrabbed->frame, True,
			       ButtonPressMask, GrabModeSync, GrabModeAsync,
				    None, Scr.MwmCursors[SYS_CURS]);
		    }
		Scr.Focus = NULL;
		Scr.Ungrabbed = NULL;
		XSetInputFocus(dpy, Scr.no_focus_win, RevertToParent, lastTimestamp);
	    }
	    return;
	}
    }

    if ((Fw != NULL) && (Fw->Desk != Scr.CurrentDesk)) {
	Fw = NULL;
	w = Scr.no_focus_win;
    }

    if ((Mwm.keyboard_focus_policy == XmEXPLICIT) && (Scr.Ungrabbed != Fw)) {
	/* need to grab all buttons for window that we are about to
	 * unfocus */
	if (Scr.Ungrabbed != NULL) {
	    XSync(dpy, 0);
	    for (i = 0; i < 3; i++)
		if (Scr.buttons2grab & (1 << i)) {
		    XGrabButton(dpy, (i + 1), 0, Scr.Ungrabbed->frame, True,
			 ButtonPressMask, GrabModeSync, GrabModeAsync, None,
				Scr.MwmCursors[SYS_CURS]);
		    XGrabButton(dpy, (i + 1), LockMask, Scr.Ungrabbed->frame, True,
			 ButtonPressMask, GrabModeSync, GrabModeAsync, None,
				Scr.MwmCursors[SYS_CURS]);
		}
	    Scr.Ungrabbed = NULL;
	}
	/* if we do click to focus, remove the grab on mouse events that
	 * was made to detect the focus change */
	if ((Mwm.keyboard_focus_policy == XmEXPLICIT) && (Fw != NULL)) {
	    for (i = 0; i < 3; i++)
		if (Scr.buttons2grab & (1 << i)) {
		    XUngrabButton(dpy, (i + 1), 0, Fw->frame);
		    XUngrabButton(dpy, (i + 1), LockMask, Fw->frame);
		}
	    Scr.Ungrabbed = Fw;
	}
    }
    if ((Fw) && (Fw->flags & ICONIFIED) && (Fw->icon_w))
	w = Fw->icon_w;

    if (!((Fw) && (Fw->wmhints) && (Fw->wmhints->flags & InputHint) &&
	       (Fw->wmhints->input == False))) {
	/* Window will accept input focus */
	XSetInputFocus(dpy, w, RevertToParent, lastTimestamp);
	Scr.Focus = Fw;
    }
    else if ((Scr.Focus) && (Scr.Focus->Desk == Scr.CurrentDesk)) {
	/* Window doesn't want focus. Leave focus alone */
	/* XSetInputFocus (dpy,Scr.Hilite->w , RevertToParent, lastTimestamp); */
    }
    else {
	XSetInputFocus(dpy, Scr.no_focus_win, RevertToParent, lastTimestamp);
	Scr.Focus = NULL;
    }


    if ((Fw) && (Fw->flags & WM_TAKES_FOCUS))
	send_clientmessage(w, _XA_WM_TAKE_FOCUS, lastTimestamp);

    XSync(dpy, 0);

}
