/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: print.c,v 1.3 94/07/19 19:58:02 nau Exp $";

/* printing routines
 */
#include <stdio.h>
#include <time.h>

#include "global.h"

#include "data.h"
#include "misc.h"
#include "print.h"
#include "transform.h"

/* ----------------------------------------------------------------------
 * some local defines
 */
#define	DEFAULT_OFFSET		300

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	PrintHeader(void);
static	void	PrintArc(ArcTypePtr);
static	void	PrintText(TextTypePtr);
static	void	PrintRect(RectTypePtr);
static	void	PrintLine(LineTypePtr);
static	void	PrintPinOrVia(PinTypePtr, Cardinal);
static	void	PrintLayer(LayerTypePtr);
static	void	PrintElementOutline(ElementTypePtr);

/* ----------------------------------------------------------------------
 * some local identifiers
 */
static	FILE	*FP;				/* output file */
static	char	*Header[] = {
	"%!PS-Adobe-2.0 EPSF-1.2",
	"%%Creator:pcb2ps",
	"%%EndComments" };

static	char	*PSFunctions[] = {
	"",
	"% scaling constant",
	"/mil {0.072 mul} def",
	"/cm  {28.346456 mul} def",
	"",
	"% some colors",
	"/Black {0.0 setgray} def",
	"/White {1.0 setgray} def",
	"",
	"/PV {",
	"% pin or via, get x, y and thickness from stack",
	"	/thickness exch def /y exch def /x exch def",
	"% draw pin",
	"	Black newpath",
	"	x y thickness 2 div 0 360 arc",
	"	closepath fill stroke",
	"% draw drilling whole	",
	"	White newpath",
	"	x y thickness 8 div 0 360",
	"	arc closepath fill stroke",
	"} def",
	"",
	"/L0 {",
	"% line up and right, get x, y, delta and thickness from stack",
	"	/thickness exch def /delta exch def /y exch def /x exch def",
	"% the line itself",
	"	Black newpath",
	"	x y thickness 2 div 180 360 arc",
	"	x y delta add thickness 2 div 0 180 arc",
	"	closepath fill stroke",
	"} def",
	"",
	"/L1 {",
	"% line up and right, get x, y, delta and thickness from stack",
	"	/thickness exch def /delta exch def /y exch def /x exch def",
	"% the line itself",
	"	Black newpath	",
	"	x y thickness 2 div 135 315 arc",
	"	x delta add y delta add thickness 2 div 315 135 arc",
	"	closepath fill stroke",
	"} def",
	"",
	"/L2 {",
	"% line straight right, get x, y, delta and thickness from stack with x2 > x1",
	"	/thickness exch def /delta exch def /y exch def /x exch def",
	"% the line itself",
	"	Black newpath	",
	"	x y thickness 2 div 90 270 arc",
	"	x delta add y thickness 2 div 270 90 arc",
	"	closepath fill stroke",
	"} def",
	"",
	"/L3 {",
	"% line down and right, get x, y, delta and thickness from stack",
	"	/thickness exch def /delta exch def /y exch def /x exch def",
	"% the line itself",
	"	Black newpath	",
	"	x y thickness 2 div 45 225 arc",
	"	x delta add y delta sub thickness 2 div 225 45 arc",
	"	closepath fill stroke",
	"} def",
	"",
	"/R {",
	"% filled rectangle, get x, y, width and height from stack",
	"	/height exch def /width exch def /y exch def /x exch def",
	"% the rectangle",
	"	Black newpath",
	"	x y moveto",
	"	x width add y lineto",
	"	x width add y height add lineto",
	"	x y height add lineto",
	"	closepath fill stroke",
	"} def",
	"",
	"/A {",
	"% arc for elements, get x, y, width, height, thickness",
	"% startangle and delta-angle from stack",
		"/delta exch def /start exch def /thickness exch def",
		"/height exch def /width exch def /y exch def /x exch def",
	"% draw it",
	"	gsave Black thickness setlinewidth x y translate",
	"% scaling is less then zero because the coord system has to be swapped",
	"	0 width sub height div 1 scale",
	"	0 0 height start start delta add arc stroke",
	"	0.0 setlinewidth grestore",
	"} def",
	"",
	"% setup some things",
	"0.0 setlinewidth" };

/* ----------------------------------------------------------------------
 * prints PostScript header with function definition
 */
static void PrintHeader(void)
{
	int		i;
	time_t	currenttime;

	for (i = 0; i < ENTRIES(Header); i++)
		fprintf(FP, "%s\n", Header[i]);

	currenttime = time(NULL);
	fprintf(FP, "%% PCB:  %s\n", PCB->Name);
	fprintf(FP, "%% Date: %s\n", asctime(localtime(&currenttime)));

	for (i = 0; i < ENTRIES(PSFunctions); i++)
		fprintf(FP, "%s\n", PSFunctions[i]);
	fflush(FP);
}

/* ----------------------------------------------------------------------
 * prints an element arc
 */
static void PrintArc(ArcTypePtr Arc)
{
	fprintf(FP, "%d %d %d %d %d %d %d A\n",
		(int) Arc->X, (int) Arc->Y, (int) Arc->Width, (int) Arc->Height,
		(int) Arc->Thickness, Arc->StartAngle, Arc->Delta);
}

/* ----------------------------------------------------------------------
 * prints a line
 */
static void PrintLine(LineTypePtr Line)
{
	switch(Line->Direction)
	{
		case 0:
			fprintf(FP, "%d %d %d %d L0\n",
				(int) Line->X1, (int) Line->Y1,
				(int) (Line->Y2 -Line->Y1), (int) Line->Thickness);
			break;

		case 1:
		case 2:
		case 3:
			fprintf(FP, "%d %d %d %d L%c\n",
				(int) Line->X1, (int) Line->Y1,
				(int) (Line->X2 -Line->X1), (int) Line->Thickness,
				(char) (Line->Direction +'0'));
			break;
	}
}

/* ----------------------------------------------------------------------
 * prints a rectangle
 */
static void PrintRect(RectTypePtr Rect)
{
	fprintf(FP, "%d %d %d %d R\n",
		(int) Rect->X, (int) Rect->Y, (int) Rect->Width, (int) Rect->Height);
}

/* ----------------------------------------------------------------------
 * prints a text
 * the routine is identical to DrawText() in module dra.c except
 * that DrawLine() and DrawRect() are replaced by their corresponding
 * printing routines
 */
static void PrintText(TextTypePtr Text)
{
	Position			x = Text->X,
						y = Text->Y,
						x0 = 0;
	unsigned char		*string = (unsigned char *) Text->TextString;
	Cardinal			n;
	LineTypePtr			line;
	LineType			newline;
	FontTypePtr			font = &PCB->Font;
	RectType			defaultsymbol;

		/* if text has to be mirrored, get x0 with the unrotated
		 * width of the surrounding box (which is the length of the string)
		 */
	if (TEST_FLAG(MIRRORFLAG, Text))
	{
		if (Text->Direction == 0 || Text->Direction == 2)
			x0 = Text->X +Text->Rect.Width/2;
		else
			x0 = Text->X +Text->Rect.Height/2;
	}
	while (string && *string)
	{
			/* print lines if symbol is valid and data is present */
		if (*string <= MAX_FONTPOSITION && font->Symbol[*string].Valid)
		{
			for (n = font->Symbol[*string].LineN, line = font->Symbol[*string].Line; n; n--, line++)
			{
					/* convert coordinates */
				newline = *line;
				newline.X1 += x;
				newline.Y1 += y;
				newline.X2 += x;
				newline.Y2 += y;

					/* do some mirroring and rotations */
				if (TEST_FLAG(MIRRORFLAG, Text))
				{
					newline.X1 = 2*x0 -newline.X1;
					newline.X2 = 2*x0 -newline.X2;
				}
				RotateLine(&newline, Text->X, Text->Y, Text->Direction);
				PrintLine(&newline);
			}
		}
		else
		{
				/* the default symbol is a rectangle */
			defaultsymbol = PCB->Font.DefaultSymbol;
			defaultsymbol.X += x;
			defaultsymbol.Y += y;

				/* do some mirroring and rotations */
			if (TEST_FLAG(MIRRORFLAG, Text))
				defaultsymbol.X = 2*x0 -defaultsymbol.X -defaultsymbol.Width;
			RotateRect(&defaultsymbol, Text->X, Text->Y, Text->Direction);
			PrintRect(&defaultsymbol);
		}
				/* move on to next cursor position */
		x += font->Symbol[*string].Width +font->Symbol[*string].Delta;
		string++;
	}
}

/* ----------------------------------------------------------------------
 * prints a via or pin
 */
static void PrintPinOrVia(PinTypePtr Ptr, Cardinal Count)
{
	for(; Count; Count--, Ptr++)
		fprintf(FP, "%d %d %d PV\n",
			(int) Ptr->X, (int) Ptr->Y, (int) Ptr->Thickness);
}

/* ----------------------------------------------------------------------
 * prints layer data
 */
static void PrintLayer(LayerTypePtr Layer)
{
	Cardinal	n;
	LineTypePtr	line;
	TextTypePtr	text;
	RectTypePtr	rect;

	for (n = Layer->LineN, line = Layer->Line; n; n--, line++)
		PrintLine(line);
	for (n = Layer->RectN, rect = Layer->Rect; n; n--, rect++)
		PrintRect(rect);
	for (n = Layer->TextN, text = Layer->Text; n; n--, text++)
		PrintText(text);
	fflush(FP);
}

/* ----------------------------------------------------------------------
 * prints package outline
 */
static void PrintElementOutline(ElementTypePtr Element)
{
	Cardinal	n;
	LineTypePtr	line;
	ArcTypePtr	arc;
	TextType	text;

	for (n = Element->LineN, line = Element->Line; n; n--, line++)
		PrintLine(line);

	for (n = Element->ArcN, arc = Element->Arc; n; n--, arc++)
		PrintArc(arc);

		/* create dummy text */
	text.X = Element->TextX;
	text.Y = Element->TextY;
	text.Direction = Element->Direction;
	text.TextString = TEST_FLAG(CANONICALFLAG, PCB) ? Element->CanonicalName : Element->NameOnPCB;
	text.Flags = 0;
	PrintText(&text);
}

/* ----------------------------------------------------------------------
 * generates PostScript output of currently visible layers
 * mirroring is on by default because X11 and PostScript use a different
 * direction for the y axis
 * options:
 *   -m             mirror
 *   -r             rotate 90 degree
 *   -ox value      x-offset in mil
 *   -oy value      y-offset in mil
 *   -scale value   scaling
 *
 * An usage message is printed by the calling function if value <> 0
 * is returned
 */
int PrintPS(int argc, char *argv[], FILE *OutputFile)
{
	Position	minx, miny,
				maxx, maxy;
	float		scale = 1.0;
	int			offsetx = DEFAULT_OFFSET,
				offsety = DEFAULT_OFFSET;
	Boolean		mirror = True,		/* see description at top of function */
				rotate = False;
	Cardinal	i;

		/* parse command-line, filename at the end is already removed */
	for (i = 1; i < argc; i++)
	{
		if (!strcmp(argv[i], "-m"))
		{
			mirror = False;
			continue;
		}
		if (!strcmp(argv[i], "-r"))
		{
			rotate = True;
			continue;
		}
		if (!strcmp(argv[i], "-scale"))
		{
			if (++i >= argc || (scale = atof(argv[i])) <= 0.0)
				return(1);
			continue;
		}
		if (!strcmp(argv[i], "-ox"))
		{
			if (++i >= argc)
				return(1);
			offsetx = atoi(argv[i]);
			continue;
		}
		if (!strcmp(argv[i], "-oy"))
		{
			if (++i >= argc)
				return(1);
			offsety = atoi(argv[i]);
			continue;
		}
		return(1);
	}

		/* copy filehandle and print PostScript header and
		 * function definitions
		 */
	FP = OutputFile;
	PrintHeader();

		/* get information about size of layout */
	GetMinMaxCoordinates(PCB, &minx, &miny, &maxx, &maxy);

		/* insert scaling and transformation commands */
	fprintf(FP, "%% PCBMIN(%d,%d), PCBMAX(%d,%d)\n",
		(int) minx, (int) miny, (int) maxx, (int) maxy);
	fprintf(FP, "%% PCBOFFSET(%d,%d), PCBSCALE(%.5f)\n", offsetx, offsety, scale);

		/* print first label */
	fputs("% PCBSTARTDATA don't remove it -------------------------\n", FP);
	fputs("gsave\n", FP);

		/* PS unit is 1 inch equal to 72 dots, PCB unit is 0.001 inch */
	scale *= 0.072;
	if (mirror && rotate)
		fprintf(FP, "%.3f %.3f scale %d %d translate -90 rotate\n",
			scale, -scale, -miny+offsetx, minx-offsety);
	if (!mirror && rotate)
		fprintf(FP, "%.3f %.3f scale %d %d translate -90 rotate\n",
			scale, scale, -miny+offsetx, maxx+offsety);
	if (mirror && !rotate)
		fprintf(FP, "%.3f %.3f scale %d %d translate\n",
			scale, -scale, -minx+offsetx, -maxy-offsety);
	if (!mirror && !rotate)
		fprintf(FP, "%.3f %.3f scale %d %d translate\n",
			scale, scale, -minx+offsetx, -miny+offsety);

		/* print data, vias and pins at last */
	for (i = 0; i < MAX_LAYER; i++)
		if (PCB->Layer[i].On)
			PrintLayer(&PCB->Layer[i]);
	if (PCB->ElementOn)
		for (i = 0; i < PCB->ElementN; i++)
			PrintElementOutline(&PCB->Element[i]);
	if (PCB->PinOn)
		for (i = 0; i < PCB->ElementN; i++)
			PrintPinOrVia(PCB->Element[i].Pin, PCB->Element[i].PinN);
	if (PCB->ViaOn)
		PrintPinOrVia(PCB->Via, PCB->ViaN);

		/* print trailing commands */
	fputs("grestore\n", FP);
	fputs("% PCBENDDATA don't remove it -------------------------\n", FP);
	fputs("showpage\n\004", FP);
	return(0);
}
