/* IluInit2.java */
/* Chris Jacobi, September 29, 1997 2:51 pm PDT */

/*
 * Copyright (c) 1996, 1997 Xerox Corporation.  All Rights Reserved.  
 * Unlimited use, reproduction, and distribution of this software is
 * permitted.  Any copy of this software must include both the above
 * copyright notice of Xerox Corporation and this paragraph.  Any
 * distribution of this software must comply with all applicable United
 * States export control laws.  This software is made available AS IS,
 * and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 * PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 * THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 * CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 * XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */
 
/* $Id: IluInit2.java,v 1.25 1997/09/29 21:52:05 jacobi Exp $ */
 
/*
 * Native code in IluJava_IluInit.c
 */


package xerox.ilu;

/** 
 * Helper class which takes care of initialization of ILU classes and
 * libraries in the right order. <br>
 * Not for applications. <p>
 *
 * @see IluInit
 * @see IluDebug
 */
 
/*friendly*/ final class IluInit2 {
    
    /* Flag to prevent recursive initializations */
    private static boolean initialized = false;

    /* Invokes initialization procedures for various native packages */
    private native static void initializeLibrary();

    /* Prevent loading inmcompatible library */
    private static final int expectedLibraryVersion = 18;
    
    /* Native code in IluJava_IluInit.c */
    private native static int checkLibrary_9_9_1997();
    private static void checkLibrary() {
        if (checkLibrary_9_9_1997() != expectedLibraryVersion) {
            IluDebug.panic("bad ilu library");
        } 
    } //checkLibrary
    
    
    /* 
     * Monitor to initialize just once.
     * While this makes sure initialization happens only once
     * applications can still fail if they don't initialize
     * ILU before forking threads; that is rare and requires 
     * applications to fork before initializing classes
     * which reference ILU stuff.
     */
    private static final synchronized boolean mustInit() {
       boolean wasinit = initialized;
       initialized = true;
       return ! wasinit;
    } //mustInit
    
    private static void println(java.lang.String s) {
        java.lang.System.err.println(s);
    } //println
    
    private static void log(java.lang.String s) {
        println("IluInit2 did " + s);
    } //log
    
    //friendly*/ java.lang.Object key = new java.lang.Object();
    
    /** Initialization procedures.  Idempotent */
    public static final void init () {
        if (! initialized) {
            if (mustInit()) { 
                if (! IluInit.doInit()){return;}
                try { 
                    /* Allow different native libraries set up by
                     * appropriate use of properties.  The standard
                     * ilu build simply sets up one library. It is 
                     * conceivable however that a site installs multiple
                     * ilu versions.
                     */
                    java.lang.String libname = IluEnvironment.getStringProp(
                        "xerox.ilu.nativeLibrary"
                        );
                    if (libname == null) {
                        libname = "IluJava";
                    }
                    try {
                        java.lang.System.loadLibrary(libname);
                    } catch (UnsatisfiedLinkError e) {
                        /* This error caused enough traffic on our
                         * bugs list to warrant carefull message
                         */
                        println("**** Unsatisfied link  " +  e);
                        println("Couldn't load native ilu library " + libname);
                        if ( libname.equals("IluJava")) {
                            println("differ for native and green threads");
                            println("or JavaSoft JDK versus Microsoft SDK");
                        } else {
                            println("Using a non-standard library name");
                        }
                        println("Is the LD_LIBRARY_PATH set correctly?");
                        IluDebug.panic("LD_LIBRARY_PATH");
                    }
                    boolean trace = (IluDebug.initsFlag > 0);
                    xerox.ilu.IluDebug.init();
                    if (trace) log("IluDebug");
                    xerox.ilu.IluDebugHooks.init();
                    if (trace) log("IluDebugHooks");
                    checkLibrary();
                    xerox.ilu.IluAlarmTech.init();
                    if (trace) log("IluAlarmTech");
                    initializeLibrary();
                    if (trace) log("library");
                    xerox.ilu.IluRT0.init();
                    if (trace) log("IluRT0");
                    xerox.ilu.IluWPBase.init();
                    if (trace) log("IluWPBase");
                    xerox.ilu.IluTypeCode.init();
                    if (trace) log("IluTypeCode");  
                    xerox.ilu.IluRootObjectHelper.init();
                    if (trace) log("IluRootObjectHelper");  
                    xerox.ilu.IluSurrogateConnection.init();
                    if (trace) log("IluSurrogateConnection");
                    xerox.ilu.IluDebugHooks.defineDebugHook(null);
                    if (trace) log("def DH");
                    xerox.ilu.IluDebugHooks.defineCheckFailure(null);
                    if (trace) log("def CF");
                    xerox.ilu.IluIdentityType.init();
                    if (trace) log("def ID");
                    xerox.ilu.IluServer.init();
                    if (trace) log("def S");
                    if (xerox.basics.Environment.getBooleanProp(
                                "ilu.interactivewait", false)) {
                        //This is for setting breakpoints...
                        IluDebug.interactiveWait();
                    }
                } catch (java.lang.Exception e) {
                    println("** Failed initializing Ilu " +  e);
                    e.printStackTrace();
                    IluDebug.panic("initializing ilu");
                }
            }
        }
    } //init
    
    
    static {init();}

} // IluInit2

