/* IluRT0.java */
/* Chris Jacobi, October 21, 1997 4:04 pm PDT */

/*
 * Copyright (c) 1996, 1997 Xerox Corporation.  All Rights Reserved.  
 * Unlimited use, reproduction, and distribution of this software is
 * permitted.  Any copy of this software must include both the above
 * copyright notice of Xerox Corporation and this paragraph.  Any
 * distribution of this software must comply with all applicable United
 * States export control laws.  This software is made available AS IS,
 * and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 * PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 * THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 * CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 * XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */
 
/* $Id: IluRT0.java,v 1.43 1997/10/21 23:46:44 jacobi Exp $ */ 
 
/* 
 * See also IluJava_IluRT0.c
 */

package xerox.ilu;
import xerox.ilu.IluClassRep;
import xerox.ilu.IluSystemException;
import xerox.ilu.IluObject;
import xerox.ilu.IluOInt;
import xerox.ilu.IluRuntimeError;

/**
 * Top level implementation of ilu features.  <p>
 *
 * Applications rather use Ilu.java.
 * @see Ilu
 */
public class IluRT0 {

    /** 
     * Initializes IluRT0 
     */
    public static void init() {
        IluOInt.init();
        IluInit.init();
    } //init

    
    /** 
     * Shortcut 
     */
    public static void throwErr(java.lang.String s) {
        s = "**xerox.ilu.IluRT0." + s;
        IluDebug.clientPreError(s);
        throw new xerox.ilu.IluRuntimeError(s);
    } //throwErr
    
    
    /** 
     * Shortcut 
     */
    public static void panic(java.lang.String s) {
        s = "**xerox.ilu.IluRT0." + s;
        IluDebug.panic(s);
    } //panic
    
    
    /** 
     * Importing an object.<p>
     * Object might be created as consequence of this call. 
     */
    public static xerox.ilu.IluObject  
    objectFromSBH (
                java.lang.String sbh, 
                xerox.ilu.IluClassRep iluClass
                ) throws xerox.ilu.IluSystemException {
        xerox.ilu.IluObject obj = null;
        if (sbh == null) throwErr("null ih");
        xerox.ilu.IluOInt oi = oiFromSBH(sbh, iluClass);
        if (oi == null) {
            IluDebug.clientPreError("objectFromSBH finds no object");
            throw new xerox.ilu.IluNoObjectException(
                "no object for [sbh = '" + sbh + "']"
                );
        }
        obj = oi.getTheObject();
                if (IluDebug.traceObjects()>0) {
            IluDebug.log.println("! ILU: objectFromSBH found " 
                + obj + " oi:" + oi);
        }
        return obj;
    } //objectFromSBH
    
    
    /**
     * Native sub-part for objectFromSBH. <p> 
     * New "unfinished" IluOInt might be created.
     */
    /*friends*/ static native xerox.ilu.IluOInt  
    oiFromSBH (
        java.lang.String sbh, 
        xerox.ilu.IluClassRep iluClass
          ) throws xerox.ilu.IluSystemException;
    

    /** 
     * Importing an object.<p>
     * Object might be created as consequence of this call. 
     */
    public static xerox.ilu.IluObject 
    lookupObject(
                java.lang.String sid, 
                java.lang.String ih, 
                xerox.ilu.IluClassRep iluClass
             ) throws xerox.ilu.IluSystemException {
        xerox.ilu.IluObject obj = null;
        if (sid == null) throwErr("null sid");
        if (ih == null) throwErr("null ih");
        xerox.ilu.IluOInt oi = oiFromLookup(sid, ih, iluClass);
        if (oi == null) {
            IluDebug.clientPreError("lookup finds no object");
            throw new xerox.ilu.IluNoObjectException(
                "no object for [sid = '" + sid + "' ih = '" + ih + "']"
                );
        }
        obj = oi.getTheObject();
        if (IluDebug.traceObjects()>0) {
            IluDebug.log.println("! ILU: lookupObject found " 
                + obj + " oi:" + oi);
        }
        return obj;
    } //lookupObject
    
    
    /**
     * Native sub-part for lookupObject <p>
     * New "unfinished" IluOInt might be created.
     */
    private static native xerox.ilu.IluOInt  
    oiFromLookup (
        java.lang.String sid, 
        java.lang.String ih, 
        xerox.ilu.IluClassRep iluClass
        ) throws xerox.ilu.IluSystemException;
    
    
    /**
     * Generates a string that's unique over space and time. <p>
     * A server with nothing better to use might call this to 
     * get an ID.
     */
    public native static java.lang.String  
    inventID();
    
    
    /** 
     * deprecated: use interface instead.
     * Literals for the lifetime argument of registerTrueObject.
     * Remember obj to prevent java garbage collection.
     * Forget obj to subject it to java garbage collection. 
     * Unspec requires ilu set a value depending on other arguments. 
     */
    public static final int unspec = IluLifetimeArgs.iluLifetimeUnspec;
    public static final int remember = IluLifetimeArgs.iluLifetimeRemember;
    public static final int forget = IluLifetimeArgs.iluLifetimeForget;
    
    
    /**
     * Create true objects.<p>
     *
     * Life time is important and must be determined.  Normally the ilu
     * object stays alive it is garbage collected.  There are other
     * schemes to prevent garbage collection. <p>
     
     * 
     * Return value is intermediate object.  <p>
     *
     * @see IluLifetimeArgs
     * @see IluIHProposer
     */
    public static java.lang.Object 
    registerTrueObject(
                java.lang.String ih,
                xerox.ilu.IluObject obj, 
                xerox.ilu.IluServer server, 
                IluClassRep iluClass,
                int lifetime //0 gives default behaviour,
                ) throws IluSystemException {
        xerox.ilu.IluOInt oi;
        if (obj == null) {
            throwErr("null object");
        }
        if (iluClass == null) {
            iluClass = IluClassAccess.iluClassFromJavaClass(obj.getClass());
            if (iluClass == null) {throwErr("IluClass of object unknown");}
        }
        iluClass.checkClass();
        if (ih == null) {
            if (obj instanceof IluIHProposer) {
                ih = ((IluIHProposer)obj).proposeIluIH();
            }
            if (ih == null) {
                ih = inventID();
            }
        }
        if (server == null) {
            server = IluServer.defaultServer();
        }
        server.finishInitialization(); 
        oi = xerox.ilu.IluOInt.newOI(obj, iluClass, server); 
              //throws exception if object already known to Ilu
        if (oi.jjClassRep == null) {
            throwErr("registerTrueObject jjClassRep missing ");
        }
        registerTrue(oi, ih, server);
        if (oi.yIluKernelObject == 0) {
            oi.forgetOI();
            throwErr("registerTrueObject: object already exists");
        }
        if (oi.yServer == 0) {
            oi.forgetOI();
            throwErr("server has been destroyed");
        }
        oi.setLifetime(lifetime);
        return oi;
    } //registerTrueObject
    
    
    /**
     * Native part of registerTrueObject
     * check out jjoi.yIluKernelObject to look for success
     */
    private static native void
    registerTrue(
        xerox.ilu.IluOInt jjoi, 
        java.lang.String ih, 
        xerox.ilu.IluServer server);
    
    
    /** 
     * Publishes the SBH of the object in the local object domain
     */
    public static void
    publishTrueObject(xerox.ilu.IluObject obj0) 
            throws xerox.ilu.IluSystemException {
        xerox.ilu.IluObject obj = (IluObject) obj0;
        if (obj instanceof IluSurrogateObject) throwErr("is surrogate");
        IluOInt oi = IluOInt.peek(obj);
        if (oi == null) {
            //object unknown; try to register it right here and now...
            java.lang.String ih = null;
            if (obj instanceof IluIHProposer) {
                ih = ((IluIHProposer)obj).proposeIluIH();
            }
            if (ih == null) {
                throwErr("publishTrueObject does not invent instance handles");
            }
            IluClassRep iluClass =
                IluClassAccess.iluClassFromJavaClass(obj.getClass());
            if (iluClass != null) {
                IluRT0.registerTrueObject(
                    ih,       //ih
                    obj, 
                    null,     //server
                    iluClass,
                    0         //lifetime
                    );
                oi = IluOInt.peek(obj);
            }
            if (oi == null) {
                throwErr("object not registerd with Ilu " + obj);
            }
        }
        if (IluDebug.traceObjects()>0) {
            IluDebug.log.println("! ILU:publishTrueObject " 
                + obj + " oi:" + oi); 
        }
        oi.publishOI();
    } //publishTrueObject
    
    
    /** 
     * withdrawObject is undo of publishTrueObject
     */
    public static void
    withdrawObject(xerox.ilu.IluObject obj) {
        if (obj instanceof IluSurrogateObject) throwErr("is surrogate");
        IluOInt oi = IluOInt.peek(obj);
        if (IluDebug.traceObjects()>0) {
            IluDebug.log.println("! ILU withdrawObject " 
                + obj + " oi:" + oi); 
        }
        if (oi != null) oi.withdrawOI();
    } //withdrawObject
    
    
    /** 
     * May return NIL if the object's server isn't exported through any
     * port; may return an invalid SBH if the cached one references a
     * closed port.
     */
    public static java.lang.String 
    sbhOfObject(xerox.ilu.IluObject obj) {
        IluOInt oi = IluOInt.peek(obj);
        if (oi == null) return null;
        return oi.sbhOfOI();
    } //sbhOfObject
    
        
    /** 
     * Returns OMG IIOP-specified IOR string for object.
     * May return null if object is not exported through IIOP.
     */
    public static java.lang.String 
    iorOfObject(xerox.ilu.IluObject obj) {
        IluOInt oi = IluOInt.peek(obj);
        if (oi == null) return null;
        return oi.iorOfOI();
    } //iorOfObject
    

    /** 
     * Destroy ILU-ness of object.  Only for impatient applications,
     * normal applications leave this to the garbage collector.
     * Do NOT use from applets; error behavior is not yet defined.
     *
     * True objects only.
     */
    public static void
    destroyObject(xerox.ilu.IluObject obj) {
        if (obj instanceof IluSurrogateObject) throwErr("is surrogate");
        IluOInt oi = IluOInt.peek(obj);
        if (IluDebug.traceObjects()>0) {
            IluDebug.log.println("! ILU destroyObject ob: " 
                + obj + " oi:" + oi); 
        }
        if (oi != null) {
            oi.destroyRudeOI();
        }
    } //destroyObject
    
    
    /** 
     * Test whether object is alive
     */
    public static void
    ping(xerox.ilu.IluObject obj) {
        IluOInt oi = IluOInt.peek(obj);
        if (oi == null) {
            throw new xerox.ilu.IluNoObjectException();
        }
        oi.nativePingOI();
    } //ping
    
    /** 
     * Test whether object is known to ILU
     */
    public static boolean
    known(xerox.ilu.IluObject obj) {
        IluOInt oi = IluOInt.peek(obj);
        if (oi == null) return false;
        return true;
    } //known
    
    
    /** 
     * Returns HTTP URL for an object.
     * May return null if object is not exported through an HTTP ilu_Port
     */
    public static java.lang.String 
    URLOfObject(xerox.ilu.IluObject obj) {
        IluOInt oi = IluOInt.peek(obj);
        if (oi == null) return null;
        return oi.nativeURLOfObject();
    } //URLOfObject

    
    /**
     * Prevent garbage collection of true object.  
     * Object will not be garbage collected as long as its server is alive. 
     */
    public static void 
    retainTrueObject(xerox.ilu.IluObject obj) {
        IluOInt.retain(obj);
    } //retainTrueObject
    
    
    /**
     * Allows garbage collection of true object.  
     * (undo of all previous calls of retainTrueObject).  
     */
    public static void 
    exposeTrueObject(xerox.ilu.IluObject obj) {
        IluOInt.expose(obj);
    } //exposeTrueObject
    
    
    /**
     * Returns whether obj is a surrogate object.
     * i.e. not a true object.  
     */
    public static boolean 
    isSurrogate(xerox.ilu.IluObject obj) {
        IluOInt oi = IluOInt.peek(obj);
        if (oi == null) return false;
        IluServer s = oi.jjServer;
        if (s == null) return true; 
        return s.isSurrogateServer();
    } //isSurrogate
    
    
    /**
     * Returns whether two objects are siblings. 
     * (Whether they belong to the same ilu server)<p>
     *
     * This is much more lightweight then figuring out their IluServer
     */
    public static boolean 
    objectsAreSiblings(xerox.ilu.IluObject obj1, xerox.ilu.IluObject obj2) {
        IluOInt oi1 = IluOInt.peek(obj1);
        IluOInt oi2 = IluOInt.peek(obj2);
        if (oi1 == null) return false;
        if (oi2 == null) return false;
        if (oi1.yServer == 0) return false;
        return (oi1.yServer == oi2.yServer);
    } //objectsAreSiblings
    
    
    /**
     * Returns the ilu version.
     */
    public native static java.lang.String iluVersion();
    
    
    static {
        init();
    }

} // IluRT0



