/*
Copyright (c) 1991-1995 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: connect.c,v 1.98 1997/11/25 23:11:54 spreitze Exp $ */
/* Last edited by Mike Spreitzer November 25, 1997 3:10 pm PST */

#define _POSIX_SOURCE

#ifdef MACOS
#pragma segment ilu
#endif

#include "iluntrnl.h"

#include "object.h"
#include "server.h"
#include "connect.h"
#include "transprt.h"
#include "protocol.h"

#ifdef HAVE_GETRLIMIT
#include <sys/time.h>
#include <sys/resource.h>
#endif /* HAVE_GETRLIMIT */

/*L2 unconstrained*/
/*L1 >= {conn's server if k=psl; cmu if k=lru}*/

static void 
_ilu_LinkConnection(ilu_ConnLinks * head, ilu_Connection conn,
		    ilu_ConnLinkKind k)
{
  _ilu_Assert((head->prev == NIL && head->next == NIL) ||
	      (head->prev != NIL && head->next != NIL &&
	       head->prev->co_links[k].next == NIL &&
	       head->next->co_links[k].prev == NIL),
	      "LinkConnection");
  conn->co_links[k].prev = NIL;
  conn->co_links[k].next = head->next;
  if (conn->co_links[k].next != NIL)
    conn->co_links[k].next->co_links[k].prev = conn;
  else
    head->prev = conn;
  head->next = conn;
  return;
}

static void 
_ilu_UnlinkConnection(ilu_ConnLinks * head, ilu_Connection conn,
		      ilu_ConnLinkKind k)
{
  _ilu_Assert((conn->co_links[k].prev == NIL)
	      ? head->next == conn
	      : conn->co_links[k].prev->co_links[k].next == conn,
	      "UnlinkConnection 1");
  _ilu_Assert((conn->co_links[k].next == NIL)
	      ? head->prev == conn
	      : conn->co_links[k].next->co_links[k].prev == conn,
	      "UnlinkConnection 2");
  if (conn->co_links[k].prev != NIL)
    conn->co_links[k].prev->co_links[k].next = conn->co_links[k].next;
  else
    head->next = conn->co_links[k].next;
  if (conn->co_links[k].next != NIL)
    conn->co_links[k].next->co_links[k].prev = conn->co_links[k].prev;
  else
    head->prev = conn->co_links[k].prev;
  return;
}

/*L1, L2 unconstrained*/

ilu_Server ilu_ServerOfConnection(ilu_Connection conn)
{
  return (conn -> co_server);
}

/**L1     =    {cmu, conn's server}; Main Remnant;
   L2    >=    {conn's waitmu};
   L2 disjoint {conn's iomu}*/
ilu_boolean
_ilu_BlockingWaitForInputOnConnection(ilu_Connection conn,
				      ilu_FineTime * limit,
				      ILU_ERRS((broken_locks,
						interrupted,
						internal)) * err)
{
  ilu_Transport   t = connection_transport(conn);
  ilu_Server      s = connection_server(conn);
  int             disabled = 1;
  ilu_boolean     domore = TRUE;
  if (!ilu_Check(conn->co_waiting && !conn->co_closed, err))
    return FALSE;
  while (disabled && domore) {
    while (_ilu_waitsDisabled && !conn->co_closing) {
      if (!ilu_CMWait2(_ilu_connAbleCC, server_lock(s), ilu_cmu, err))
	return FALSE;
      if (!ilu_Check(conn->co_waiting && !conn->co_closed, err))
	return FALSE;
    }
    if (conn->co_closing)
      return ILU_CLER(*err);
    if (!_ilu_ExitServerMutex(s, TRUE, err))
      return FALSE;
    if (!ilu_ExitMutex(ilu_cmu, TRUE, err))
      return FALSE;
    domore = transport_wait_for_input(t, &disabled, limit, err);
    if (!ilu_ReEnterMutex(ilu_cmu, err))
      return FALSE;
    if (!_ilu_EnterServerMutex(s, TRUE, err))
      return FALSE;
  }
  return domore;
}

/*Main Invariant holds; L2 disjoint conn's mutexes*/
ilu_boolean
ilu_BlockingWaitForInputOnConnection(ilu_Connection conn,
				     ilu_FineTime * limit)
{
  ilu_Server      s;
  ILU_ERRS((broken_locks, interrupted, internal)) lerr;
  ilu_boolean     ans = FALSE;
  ilu_Call_s      dummyCall = {1};
  _ilu_Assert(conn != NIL, "BlockingWaitForInputOnConnection(NIL,..)");
  s = connection_server(conn);
  _ilu_Assert(s != NIL,
	      "BlockingWaitForInputOnConnection(&{s=NIL,..},..)");
  (void) ilu_EnterMutex(ilu_cmu, &lerr);
  ILU_MUST_BE_SUCCESS(lerr);
  (void) _ilu_EnterServerMutex(s, FALSE, &lerr);
  ILU_MUST_BE_SUCCESS(lerr);
  if (connection_closed(conn))
    goto dun2;
  if (_ilu_CanCondition() && connection_concurrent(conn)) {
    _ilu_EnterConnWait(conn, &dummyCall, FALSE, &lerr);
    ILU_MUST_BE_SUCCESS(lerr);
  } else
    _ilu_TakeConnWait(conn, &dummyCall);
  ans = _ilu_BlockingWaitForInputOnConnection(conn, limit, &lerr);
  ILU_ERR_SWITCH(lerr) {
    ILU_SUCCESS_CASE 0;
    ILU_ERR_CASE(interrupted, x) ans = TRUE;
    /* i.e., assert !(broken_locks||internal) */
  } ILU_ERR_ENDSWITCH;
  (void) _ilu_ReleaseConnWait(conn, &dummyCall, TRUE, &lerr);
dun2:
  (void) _ilu_ExitServerMutex(s, TRUE, &lerr);
dun1:
  (void) ilu_ExitMutex(ilu_cmu, TRUE, &lerr);
dun0:
  ILU_MUST_BE_SUCCESS(lerr);
  return (ans);
}

/*L1.sup < trmu; L2 >= {xmu, ymu}*/
ilu_boolean
_ilu_SetTransportInputHandler(ilu_Transport trans,
			      ilu_TIH tih,
			      ILU_ERRS((no_memory, no_resources,
					internal)) * err)
{
  if (tih && trans->tr_inBuff != NIL &&
      trans->tr_inNext < trans->tr_inLimit)
    return FALSE;
  return ((*trans->tr_class->tc_set_input_handler)
	  (trans, tih, err));
}

/*L1 >= {conn's server}; (L1-{conn's server}).sup = cmu*/
/*L2 disjoint {conn's iomu, callmu, waitmu}*/
extern          ilu_boolean
_ilu_InnerSetConnectionInputHandler(ilu_Connection conn,
				ilu_TransportInputHandler tih_proc,
				    ilu_refany tih_rock,
				    ILU_ERRS((no_memory, internal,
					   no_resources, bad_param,
					      bad_locks)) * err)
{
  ilu_Server      s;
  ilu_Transport   trans;
  ilu_Mutex       sm;
  ilu_boolean     immediate = FALSE;
  static ilu_Call_s dumy = {0};
  if (!conn)
    return ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_nil, FALSE);
  if (_ilu_CanCondition())
    return ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_threading, FALSE);
  s = conn->co_server;
  if (!s)
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_broken, FALSE);
  if (conn->co_mucall || conn->co_ioing || conn->co_waiting)
    return ILU_ERR_CONS0(bad_locks, err, FALSE);
  ILU_CLER(*err);
  if (conn->co_closed) {
    if (!tih_proc)
      return TRUE;
    immediate = TRUE;
    goto level2;
  }
  _ilu_TakeConnWait(conn, &dumy);
  if (!_ilu_EnterConnCall(conn, &dumy, TRUE, err))
    goto level3;
  if (!_ilu_TakeConnIO(conn, TRUE, err))
    goto level4;
  trans = conn->co_transport;
  conn->co_tih.proc = tih_proc;
  conn->co_tih.rock = tih_rock;
  immediate = !_ilu_SetTransportInputHandler(trans,
				  (tih_proc ? &conn->co_tih : NIL),
					     err);
  (void) _ilu_ReleaseConnIO(conn, TRUE, err);
level4:
  (void) _ilu_ReleaseConnCall(conn, &dumy, TRUE, err);
level3:
  (void) _ilu_ReleaseConnWait(conn, &dumy, TRUE, err);
  if (ILU_ERRNOK(*err))
    return FALSE;
  if (!immediate || !tih_proc)
    return TRUE;
level2:
  (void) _ilu_ExitServerMutex(s, FALSE, err);
  if (ILU_ERRNOK(*err))
    return FALSE;
  (void) ilu_ExitMutex(ilu_cmu, FALSE, err);
  if (ILU_ERRNOK(*err))
    goto ret1;
  (*tih_proc) (tih_rock);
  if (!ilu_ReEnterMutex(ilu_cmu, err))
    return FALSE;
ret1:
  if (!_ilu_EnterServerMutex(s, TRUE, err))
    return FALSE;
  return ILU_ERROK(*err);
}

/*Main Invariant holds; L2 disjoint {conn's iomu, callmu, waitmu}*/
extern          ilu_boolean
ilu_SetConnectionInputHandler(ilu_Connection conn,
			      ilu_TransportInputHandler tih_proc,
			      ilu_refany tih_rock,
			      ILU_ERRS((no_memory, internal,
					no_resources, bad_param,
					bad_locks)) * err)
{
  ilu_Server      s;
  ilu_Transport   trans;
  ilu_Mutex       sm;
  ilu_boolean     immediate = FALSE;
  static ilu_Call_s dumy = {0};
  if (!conn)
    return ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_nil, FALSE);
  if (_ilu_CanCondition())
    return ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_threading, FALSE);
  s = conn->co_server;
  if (!s)
    return ILU_ERR_CONS1(internal, err, minor, ilu_im_broken, FALSE);
  if (!ilu_EnterMutex(ilu_cmu, err))
    goto dun0;
  if (!_ilu_EnterServerMutex(s, FALSE, err))
    goto dun1;
  if (conn->co_mucall || conn->co_ioing || conn->co_waiting) {
    ILU_ERR_CONS0(bad_locks, err, (void) 0);
    goto dun2;
  }
  if (conn->co_closed) {
    immediate = TRUE;
    goto dun2;
  }
  _ilu_TakeConnWait(conn, &dumy);
  if (!_ilu_EnterConnCall(conn, &dumy, TRUE, err))
    goto dun3;
  if (!_ilu_TakeConnIO(conn, TRUE, err))
    goto dun4;
  trans = conn->co_transport;
  conn->co_tih.proc = tih_proc;
  conn->co_tih.rock = tih_rock;
  immediate = !_ilu_SetTransportInputHandler(trans,
				  (tih_proc ? &conn->co_tih : NIL),
					     err);
  (void) _ilu_ReleaseConnIO(conn, TRUE, err);
dun4:
  (void) _ilu_ReleaseConnCall(conn, &dumy, TRUE, err);
dun3:
  (void) _ilu_ReleaseConnWait(conn, &dumy, TRUE, err);
dun2:
  (void) _ilu_ExitServerMutex(s, TRUE, err);
dun1:
  (void) ilu_ExitMutex(ilu_cmu, TRUE, err);
dun0:
  if (immediate && ILU_ERROK(*err) && tih_proc)
    (*tih_proc) (tih_rock);
  return ILU_ERROK(*err);
}

/*L1 >= {cmu, conn's server}, L1.sup < trmu; L2 disjoint conn*/
extern          ilu_boolean
ilu_ClearConnectionInputHandler(ilu_Connection conn,
				ILU_ERRS((no_memory, internal,
					  no_resources)) * err)
{
  ilu_Transport   trans = conn->co_transport;
  ilu_boolean     regd = FALSE;
  static ilu_Call_s dumy = {0};
  if (conn->co_waiting || conn->co_ioing)
    return ILU_ERR_CONS0(bad_locks, err, FALSE);
  if (conn->co_closed)
    return ILU_CLER(*err);
  _ilu_TakeConnWait(conn, &dumy);
  if (!_ilu_TakeConnIO(conn, TRUE, err))
    goto dun3;
  regd = ((*trans->tr_class->tc_set_input_handler)
	  (trans, NIL, err));
  (void) _ilu_ReleaseConnIO(conn, TRUE, err);
dun3:
  (void) _ilu_ReleaseConnWait(conn, &dumy, TRUE, err);
  if (ILU_ERRNOK(*err))
    return FALSE;
  if (!ilu_Check(regd, err))
    return FALSE;
  return TRUE;
}

/*L1 >= {cmu, conn's server}; L1.sup < trmu*/
/*L2 >= {conn's iomu}*/
void 
_ilu_CloseIoingConnection(ilu_Connection conn,
			  ilu_boolean set_cfails,
			  ilu_ConnShutdownReason reason)
{
  ilu_Error       err;
  ilu_Protocol    proto = connection_protocol(conn);
  ilu_Transport   trans = conn->co_transport;
  ilu_Server      server = conn->co_server;
  ilu_integer     dfd;
  ilu_Call_s      dummycall = {0};

  _ilu_Assert(conn->co_ioing, "bad call on _ilu_CloseIoingConnection");
  if (set_cfails)
    server->sr_cfails = ilu_TRUE;
  if (connection_closed(conn))
    return;
#ifdef ENABLE_DEBUGGING
  if (connection_incoming(conn)) {
    ILU_NOTE(CONNECTION_DEBUG,
	  ("_ilu_CloseIoingConnection:  incoming conn %p trans %p"
	   " via %s from %s to %s; cfails=%d; waiting=%d; reason=%d;"
	   " closing was %d.\n",
	   conn, trans, conn->co_pinfo,
	   conn_peerinfo(conn),
	   server->sr_id, set_cfails != 0,
	   conn->co_waiting != 0, reason, conn->co_closing != 0));
  } else {
    ilu_string      t = _ilu_StringifyTinfo(conn_tinfo(conn), &err);
    ILU_MUST_BE_SUCCESS(err);
    ILU_NOTE(CONNECTION_DEBUG,
	  ("_ilu_CloseIoingConnection:  outgoing conn %p trans %p"
	   " via %s %s to %s; cfails=%d; waiting=%d; reason=%d; closing was %d.\n",
	   conn, trans, conn->co_pinfo, t,
	   server->sr_id, set_cfails != 0,
	   conn->co_waiting != 0, reason, conn->co_closing != 0));
    ilu_free(t);
  }
#endif				/* ENABLE_DEBUGGING */
  if (connection_protocol(conn)->pr_conn_closing != NULLFN) {
    protocol_close_connection(connection_protocol(conn),
			      connection_protocol_data(conn),
			      reason, &err);
    ILU_HANDLED(err);
  };
  if (conn->co_waiting) {
    if (conn->co_closing)
      return;
    conn->co_closing = TRUE;
    if (_ilu_CanCondition()) {
      (void) transport_disableWait(trans, &err);
      ILU_MUST_BE_SUCCESS(err);
      ilu_CondNotify(_ilu_connAbleCC, &err);
    } else
      (void) transport_interruptST(trans, &err);
    ILU_MUST_BE_SUCCESS(err);
    return;
  }
  conn->co_waiting = &dummycall;
  conn->co_closed = TRUE;
  if (_ilu_CanCondition()) {
    if (conn->co_closing) {
      transport_enableWait(trans, &err);
      ILU_MUST_BE_SUCCESS(err);
    }
    ilu_CondNotify(_ilu_connAbleCC, &err);
    ILU_MUST_BE_SUCCESS(err);
  }
  if ((conn->co_mucall == NIL) && (conn->co_nOuts == 0))
    _ilu_UnlinkConnection(&ilu_idleConns, conn, ilu_lru);
  if (connection_incoming(conn)) {
    ilu_Port        p = conn->co_port;
    _ilu_UnlinkConnection(&p->po_connHead, conn, ilu_psl);
    _ilu_LinkConnection(&p->po_closedConns, conn, ilu_psl);
  } else {
    _ilu_UnlinkConnection(&server->sr_connHead, conn, ilu_psl);
    _ilu_LinkConnection(&server->sr_closedConns, conn, ilu_psl);
  }
  transport_close(trans, &dfd, &err);
  ILU_MUST_BE_SUCCESS(err);
  ilu_DeltaFD(-dfd);
  protocol_free_data_block(proto, connection_protocol_data(conn));
  conn->co_protocol_data = NIL;
  while (conn->co_replies != NIL) {
    ilu_ReplyList   r = conn->co_replies;
    ilu_ReplyList   next = r->rp_next;
    (void) protocol_abandon_delayed_interp(proto, conn, r->rp_queued,
					   &err);
    ILU_MUST_BE_SUCCESS(err);
    ilu_free(r);
    conn->co_replies = next;
  }
  if (conn->co_auth_info != NIL)
    ilu_DestroyPassport(conn->co_auth_info, &err);
  ILU_MUST_BE_SUCCESS(err);
  if (_ilu_CanCondition()) {
    err = _ilu_NotifyCondition(conn->co_cc);
    ILU_MUST_BE_SUCCESS(err);
  }
  _ilu_Assert(conn->co_waiting == &dummycall, "ilu_CloseIoIngConn vs. wait in connect.c");
  conn->co_waiting = NIL;
  return;
}

/*L1, L2 unconstrained*/
static struct _ilu_Call_s fauxcall = {0};

/* Usable on either incoming or outgoing connections. */
/*L1 = {conn's server, cmu}; L2 disjoint {conn's callmu, iomu}*/
ILU_ERRS((bad_locks, broken_locks))
_ilu_CloseConnection(ilu_Connection connection, ilu_ConnShutdownReason reason)
{
  ilu_Error       err;
  if (!_ilu_EnterConnIO(connection, FALSE, &err))
    return err;
  _ilu_CloseIoingConnection(connection, FALSE, reason);
  (void) _ilu_ReleaseConnIO(connection, TRUE, &err);
  return err;
}

/* L1 >= {conn's server}; L2 disjoint {conn's callmu, iomu} */
void _ilu_MaybeFreeConnection(ilu_Connection conn)
{
  ilu_Server s = conn->co_server;
  if (conn->co_nOuts || conn->co_nCalls ||
      conn->co_lsrCares || !conn->co_closed || conn->co_serialer ||
      conn->co_waiting || conn->co_mucall || conn->co_ioing)
    return;
  if (connection_incoming(conn)) {
    ilu_free(conn_peerinfo(conn));
    conn_peerinfo(conn) = NIL;
  } else {
    ilu_free(conn_tinfo(conn));
    conn_tinfo(conn) = NIL;
  }
  ilu_free(conn->co_pinfo);
  conn->co_pinfo = NIL;
  /*
   * conn->co_transport should be closed and free by now.  What
   * about co_auth_info and co_conn_identity?  co_replies should be
   * empty by now.
   */
  if (_ilu_CanCondition())
    ilu_DestroyCondition(conn->co_cc);
  conn->co_cc = NIL;
  /*
   * Should already be unlinked from server & idle list, and be in
   * closed list.
   */
  if (conn->co_port) {
    ILU_ERRS((internal)) lerr;
    ilu_Port        p = conn->co_port;
    _ilu_UnlinkConnection(&p->po_closedConns, conn, ilu_psl);
    (void) _ilu_MaybeFreePort(p, &lerr);
    ILU_MUST_BE_SUCCESS(lerr);
  } else
    _ilu_UnlinkConnection(&conn->co_server->sr_closedConns, conn,
			  ilu_psl);
  ILU_NOTE(CONNECTION_DEBUG,
	   ("connect.c: freeing ilu_Connection %p %s %p=%s;\n"
   "\tserver's ports=%s %s %s, conns:=%s %s, objs=%d, LSSes=%d.\n",
	    conn, (connection_incoming(conn) ? "into" : "to"),
	    s, s->sr_id,
	    (server_ports(s) ? "X" : "0"),
	    (s->sr_local_port ? "X" : "0"),
	    (s->sr_closedPorts.pl_next ? "X" : "0"),
	    (s->sr_connHead.next ? "X" : "0"),
	    (s->sr_closedConns.next ? "X" : "0"),
	    (s->sr_objs ? ilu_hash_PairsInTable(s->sr_objs) : 0),
	    _ilu_ServerLSSCount(s)));
  ilu_free(conn);
  return;
}

/*L1, L2 unconstrained; call only when single-threaded */
ilu_boolean ilu_ConnectionServingP(ilu_Connection conn)
{
  return (conn->co_nCalls > 0);
}

/*L1.sup < cmu; L2 disjoint conn's mutexes*/
ilu_boolean
ilu_DoneServingConnection(ilu_Connection conn,
			  ILU_ERRS((bad_param, broken_locks,
				    bad_locks, internal)) * err)
{
  ilu_Server      s = conn->co_server;
  if (!(connection_incoming(conn) && conn->co_lsrCares)) {
   ILU_NOTE(CONNECTION_DEBUG,
	("ilu_DoneServingConnection, returning error because: incoming = %s, lsrCares = %s)\n",
	 (connection_incoming(conn) ? "True" : "False"), 
	 (conn->co_lsrCares ? "True" : "False")));
    return ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_duh, FALSE);
  }
  ILU_NOTE(CONNECTION_DEBUG,
	("ilu_DoneServingConnection(%p, tr=%p, %s from %s)\n",
	 conn, conn->co_transport, conn->co_pinfo,
	 conn_peerinfo(conn)));
  if (!ilu_EnterMutex(ilu_cmu, err))
    return FALSE;
  if (!_ilu_EnterServerMutex(s, FALSE, err))
    goto dun1;
  *err = _ilu_CloseConnection(conn, ilu_ConnShutdownReason_ProcessTermination);
  conn->co_lsrCares = FALSE;
  if (ILU_ERROK(*err))
    _ilu_MaybeFreeConnection(conn);
  (void) _ilu_ExitServerMutex(s, TRUE, err);
dun1:
  (void) ilu_ExitMutex(ilu_cmu, TRUE, err);
  return ILU_ERROK(*err);
}

/*Main Invariant, L2 >= {conn's iomu}*/
extern          ilu_boolean
_ilu_CloseConnWithIo(ilu_Connection conn, ilu_boolean set_cfails,
		     ilu_ConnShutdownReason reason, ILU_ERRS((IoErrs)) * err)
{
  ilu_Server      s = conn->co_server;
  if (!ilu_ReEnterMutex(ilu_cmu, err))
    return FALSE;
  if (!_ilu_EnterServerMutex(s, TRUE, err))
    goto dun1;
  _ilu_CloseIoingConnection(conn, set_cfails, reason);
  (void) _ilu_ExitServerMutex(s, TRUE, err);
dun1:
  (void) ilu_ExitMutex(ilu_cmu, TRUE, err);
  return ILU_ERROK(*err);
}

/*L1 >= {cmu}*/
/*L2 unconstrained*/

ilu_integer     ilu_fdbudget = 16;	/* # FDs allowed */
ilu_integer     ilu_fdstaken = 0;	/* # FDs used (incl idle) */

void 
ilu_FullDeltaFD(ilu_integer n, char *file, int line)
{
  ILU_NOTE(CONNECTION_DEBUG,
	("ilu_fdstaken(was %ld) += %ld at %s:%d\n",
	 (long int) ilu_fdstaken,
	 (long int) n, file, line));
  ilu_fdstaken += n;
  return;
}

ilu_ConnLinks ilu_idleConns = {NIL, NIL};

/*L1.sup < cmu; L2 unconstrained*/
ilu_cardinal ilu_GetFDBudget(void)
{
  ilu_cardinal ans;
  ilu_AcquireMutex(ilu_cmu);
  ans = ilu_fdbudget;
  ilu_ReleaseMutex(ilu_cmu);
  return ans;
}

/*Main Invariant holds; L2 otherwise unconstrained*/
extern ilu_cardinal ilu_SetFDBudget(ilu_cardinal n)
{
  ILU_ERRS((WrongLocks)) err;
  ilu_cardinal    ans;
#ifdef HAVE_GETRLIMITo
  struct rlimit resourcelimit;
  /* restrict to no more than the processes current resource limit */
  if (! getrlimit(RLIMIT_NOFILE, &resourcelimit))
    ILU_NOTE(CONNECTION_DEBUG, ("ilu_SetFDBudget couldn't determine resource limit RLIMIT_NOFILE, errno = %d\n", errno));
      else {
	if (n > resourcelimit.rlim_cur - 1) {
	  ILU_NOTE(CONNECTION_DEBUG, ("ilu_SetFDBudget(%d) limiting to RLIMIT_NOFILE - 1 which is %d\n", n, resourcelimit.rlim_cur - 1));
	  n = resourcelimit.rlim_cur - 1;
	}
      }
#else
#ifdef WIN32
  if (n > FD_SETSIZE) {
    ILU_NOTE(CONNECTION_DEBUG, ("ilu_SetFDBudget(%d) limiting to FD_SETSIZE which is %d\n", n, FD_SETSIZE));
    n = FD_SETSIZE;
  }
#endif /* WIN32 */
#endif /* HAVE_GETRLIMIT */

  ilu_AcquireMutex(ilu_cmu);
  err = _ilu_ReduceFdsTo(n);
  ILU_MUST_BE_SUCCESS(err);
  ilu_fdbudget = ilu_fdstaken;
  if (ilu_fdbudget < 0 || n > (ilu_cardinal) ilu_fdbudget)
    ilu_fdbudget = n;
  ans = ilu_fdbudget;
  ilu_ReleaseMutex(ilu_cmu);
  return ans;
}

/*Main Remnant holds*/

/*L1.sup = cmu*/
ILU_ERRS((internal)) _ilu_ReduceFdsTo(ilu_integer goal)
{
  ilu_Connection  next, cur = ilu_idleConns.next;
  ilu_Error       err;
  _ilu_HoldMutex(ilu_cmu);
  ILU_NOTE(CONNECTION_DEBUG, ("_ilu_ReduceFdsTo: goal %d, ilu_fdbudget is %d\n", goal, ilu_fdbudget)); 
  while ((cur != NIL) && ((ilu_fdstaken) > goal)) {
    ilu_Server      s = cur->co_server;
    next = cur->co_links[ilu_lru].next;
    _ilu_AcquireServerMutex(s);
    /*
     * (cur) not closed, because this thread holds cmu, which
     * another thread would need to close (cur).
     */
    if (!(cur->co_mucall || cur->co_nOuts || cur->co_waiting))
      /* What about waiting input? */
      if (0 < ((*cur->co_transport->tr_class->tc_closeDFd)
	       (cur->co_transport))) {
	(void) _ilu_EnterConnCall(cur, &fauxcall, TRUE, &err);
	/* can't block or err */
	ILU_MUST_BE_SUCCESS(err);
	(void) _ilu_EnterConnIO(cur, TRUE, &err);
	/* can't block or err */
	ILU_MUST_BE_SUCCESS(err);
	ILU_NOTE(CONNECTION_DEBUG, ("_ilu_ReduceFdsTo calling _ilu_CloseIoingConnection( %p, FALSE)\n", cur));
	_ilu_CloseIoingConnection(cur, FALSE, ilu_ConnShutdownReason_ResourceManagement);
	(void) _ilu_ReleaseConnIO(cur, TRUE, &err);
	ILU_MUST_BE_SUCCESS(err);
	(void) _ilu_ReleaseConnCall(cur, &fauxcall, TRUE, &err);
	ILU_MUST_BE_SUCCESS(err);
      }
    _ilu_ReleaseServerMutex(s);
    cur = next;
  }
  return ILU_NO_ERR;
}

/*L1.sup = s; L1 >= {cmu}*/
/*L2 unconstrained*/
ilu_Connection 
_ilu_CreateConnection(ilu_Transport bs,
		      ilu_TransportInfo tinfo,	/* NIL for incoming */
		      ilu_string peerinfo,	/* NIL for outgoing */
		      ilu_Protocol pr, ilu_string pinfo,
		      ilu_Port port,		/* NIL for outgoing */
		      ilu_Server s,
		      ilu_Passport pp,
		      ILU_ERRS((no_memory, internal/check)) * err)
{
  ilu_Error		lerr = ILU_INIT_NO_ERR;
  ilu_Connection	ans;
  static struct _ilu_Connection_s initConn = {0};

  ans = (ilu_Connection) ilu_MallocE(sizeof(*ans), err);
  if (ans == NIL) {
    if (pp) {
      ilu_Error       lerr;
      ilu_DestroyPassport(pp, &lerr);
      ILU_HANDLED(lerr);
    }
    return ans;
  }
  *ans = initConn;
  _ilu_HoldMutex(server_lock(s));
  ans->co_protocol = pr;
  ans->co_port = port;
  ans->co_protocol_data = protocol_create_data_block(pr, pinfo, err);
  if (ILU_ERRNOK(*err))
    goto faild;
  ans->co_pinfo = ilu_StrdupE(pinfo, err);
  if (ILU_ERRNOK(*err))
    goto faild;
  ans->co_transport = bs;
  if (port != NIL)
    conn_peerinfo(ans) = ilu_StrdupE(peerinfo, err);
  else
    conn_tinfo(ans) = _ilu_CopyTinfo(tinfo, err);
  if (ILU_ERRNOK(*err))
    goto faild;
  ans->co_conn_identity.ii_owned_by_passport = ilu_FALSE;
  if (port != NIL) {
    ans->co_conn_identity.ii_type = ilu_ConnectionIdentity;
    ans->co_conn_identity.ii_info = (ilu_refany) conn_peerinfo(ans);
    ans->co_auth_info = pp;
    if (!ilu_AddIdentity (pp, &ans->co_conn_identity, err))
      goto faild;
  } else {
    ans->co_conn_identity.ii_type = ilu_NoIdentity;
    ans->co_auth_info = NIL;
  };
  ans->co_server = s;
  if (_ilu_CanCondition()) {
    char            buf[24];
    sprintf(buf, "%p", ans);
    ans->co_cc = _ilu_CreateCondition(s->sr_id, buf, &lerr);
    ILU_MUST_BE_SUCCESS(lerr);
  } else
    ans->co_cc = NIL;
  ans->co_lsrCares = TRUE;
  ans->co_next_sn = 1;
  if (connection_incoming(ans)) {
    _ilu_LinkConnection(&port->po_connHead, ans, ilu_psl);
  } else {
    _ilu_LinkConnection(&s->sr_connHead, ans, ilu_psl);
  }
  _ilu_LinkConnection(&ilu_idleConns, ans, ilu_lru);
#ifdef ENABLE_DEBUGGING
  if ((ilu_DebugLevel & CONNECTION_DEBUG) != 0) {
    if (connection_incoming(ans))
      ilu_DebugPrintf("_ilu_CreateConnection:  %p; CV %p, transport %p\n",
		      ans, ans->co_cc, bs);
    else {
      ilu_string      t = _ilu_StringifyTinfo(tinfo, err);
      if (ILU_ERRNOK(*err))
	goto faild;
      ilu_DebugPrintf("_ilu_CreateConnection:  %p; CV %p, transport %p\n",
		      ans, ans->co_cc, bs);
      ilu_free(t);
    }
  }
#endif /* ENABLE_DEBUGGING */
  return (ans);
faild:
  if (connection_incoming(ans) && conn_peerinfo(ans) != NIL)
    ilu_free(conn_peerinfo(ans));
  else if (!connection_incoming(ans) && conn_tinfo(ans) != NIL)
    ilu_free(conn_tinfo(ans));
  if (ans->co_pinfo != NIL)
    ilu_free(ans->co_pinfo);
  if (ans->co_protocol_data != NIL)
    (*pr->pr_free_data_block) (ans->co_protocol_data);
  ilu_free(ans);
  if (pp) {
    ilu_Error       lerr;
    ilu_DestroyPassport(pp, &lerr);
    ILU_HANDLED(lerr);
  }
  return NIL;
}

/*L1, L2 unconstrained*/

ilu_boolean ilu_ThreadPerRequest(ilu_Connection conn)
{
  return (connection_protocol(conn)->pr_concurrent_requests);
}

/*L1 >= {conn's server}*/
/*L2 as implied by name*/

/*L1 >= {conn's server}*/
ilu_boolean
_ilu_TakeConnIO(ilu_Connection conn, ilu_boolean hard,
		   ILU_ERRS((bad_locks)) * err)
{
  _ilu_HoldMutex(server_lock(conn->co_server));
  if (conn->co_ioing) {
    if (hard)
      ILU_ERR_CONS0(broken_locks, err, 0);
    else
      ILU_ERR_CONS0(bad_locks, err, 0);
    return FALSE;
  }
  ILU_NOTE(LOCK_DEBUG,
	("TakeConnIO(%p) succeeds.\n", conn));
  conn->co_ioing = TRUE;
  return ILU_CLER(*err);
}

/*L1 = {conn's server, cmu}*/
ilu_boolean
_ilu_EnterConnIO(ilu_Connection conn, ilu_boolean hard,
		   ILU_ERRS((bad_locks)) * err)
{
  _ilu_HoldMutex(server_lock(conn->co_server));
  _ilu_HoldMutex(ilu_cmu);
  while (conn->co_ioing) {
    /*
     * We shouldn't actually get here unless the I/O mutex is held
     * by another thread, in which case the runtime should have
     * provided a wait-on-condition operation.  We could also get
     * here if the caller mistakenly holds the I/O mutex.
     */
    ILU_NOTE(LOCK_DEBUG,
	     ("EnterConnIO(%p) waiting.\n", conn));
    (void) ilu_CMWait2(conn->co_cc, server_lock(conn->co_server),
		       ilu_cmu, err);
    ILU_NOTE(LOCK_DEBUG,
	     ("EnterConnIO(%p) resuming.\n", conn));
    if (ILU_ERRNOK(*err))
      return FALSE;
  }
  ILU_NOTE(LOCK_DEBUG,
	   ("EnterConnIO(%p) succeeds.\n", conn));
  conn->co_ioing = TRUE;
  return ILU_CLER(*err);
}

/*L1 >= {cmu, conn's server}; L1.sup < trmu*/
ilu_boolean
_ilu_ReleaseConnIO(ilu_Connection conn, ilu_boolean hard,
		   ILU_ERRS((bad_locks, broken_locks)) * err)
{
  _ilu_HoldMutex(ilu_cmu);
  _ilu_HoldMutex(server_lock(conn->co_server));
  if (!conn->co_ioing) {
    if (hard)
      return ILU_ERR_CONS0(broken_locks, err, FALSE);
    else
      return ILU_ERR_CONS0(bad_locks, err, FALSE);
  }
  ILU_NOTE(LOCK_DEBUG,
	("ReleaseConnIO(%p)\n", conn));
  if (conn->co_server->sr_closing == TRUE)
    _ilu_CloseIoingConnection(conn, ilu_FALSE, ilu_ConnShutdownReason_ProcessTermination);
  conn->co_ioing = FALSE;
  if (_ilu_CanCondition()) {
    ilu_Error       lerr;
    lerr = _ilu_NotifyCondition(conn->co_cc);
    ILU_ERR_SWITCH(lerr) {
      ILU_SUCCESS_CASE;
      ILU_ERR_CASE(CantCondition, v) {
	ILU_HANDLED(lerr);
	return ILU_ERR_CONS0(broken_locks, err, FALSE);
      }
    } ILU_ERR_ENDSWITCH;
  }
  return TRUE;
}

/*L1 = alt?{conn's server}:{conn's server, cmu}*/
static          ilu_boolean
FullEnterConnCall(ilu_Connection conn, ilu_Call call,
		  ilu_boolean hard, ilu_boolean alt,
		  ILU_ERRS((bad_locks, broken_locks)) * err)
{
  _ilu_HoldMutex(server_lock(conn->co_server));
  if (conn->co_mucall == call) {
    if (hard)
      return ILU_ERR_CONS0(broken_locks, err, FALSE);
    else
      return ILU_ERR_CONS0(bad_locks, err, FALSE);
  }
  while (conn->co_mucall) {
    /*
     * We shouldn't actually get here unless the call mutex is held
     * by another thread, in which case the runtime should have
     * provided a wait-on-condition operation.  We could also get
     * here if the caller mistakenly holds the call mutex on behalf
     * of a different call.
     */
    ILU_NOTE(LOCK_DEBUG,
	     ("EnterConnCall(%p, %p) waits for %p\n",
	      conn, call, conn->co_mucall));
    (void) ilu_CMWait2(conn->co_cc, server_lock(conn->co_server),
		    (alt ? server_lock(conn->co_server) : ilu_cmu),
		       err);
    ILU_NOTE(LOCK_DEBUG,
	     ("EnterConnCall(%p, %p) resumes from %p\n",
	      conn, call, conn->co_mucall));
    if (ILU_ERRNOK(*err))
      return FALSE;
  }
  ILU_NOTE(LOCK_DEBUG,
	   ("EnterConnCall(%p, %p) succeeds\n",
	    conn, call));
  conn->co_mucall = call;
  if (conn->co_nOuts == 0 && !connection_closed(conn)) {
    _ilu_Assert(!alt, "FullEnterConnCall alt vs. co_nOuts");
    _ilu_HoldMutex(ilu_cmu);
    _ilu_UnlinkConnection(&ilu_idleConns, conn, ilu_lru);
  }
  return ILU_CLER(*err);
}

/*L1 = {conn's server, cmu}*/
ilu_boolean
_ilu_EnterConnCall(ilu_Connection conn, ilu_Call call,
		   ilu_boolean hard,
		   ILU_ERRS((bad_locks, broken_locks)) * err)
{
  return FullEnterConnCall(conn, call, hard, FALSE, err);
}

/* L1 = {conn's server}; conn->co_nOuts > 0 */
ilu_boolean
_ilu_AltEnterConnCall(ilu_Connection conn, ilu_Call call,
		      ilu_boolean hard,
		      ILU_ERRS((bad_locks, broken_locks)) * err)
{
  _ilu_Assert(conn->co_nOuts > 0, "AltEnterConnCall");
  return FullEnterConnCall(conn, call, hard, TRUE, err);
}

ilu_boolean
_ilu_ReleaseConnCall(ilu_Connection conn, ilu_Call call,
		     ilu_boolean hard,
		     ILU_ERRS((bad_locks, broken_locks)) * err)
{
  _ilu_HoldMutex(server_lock(conn->co_server));
  ILU_NOTE(LOCK_DEBUG,
	("ReleaseConnCall(%p, %p, holder=%p)\n",
	 conn, call, conn->co_mucall));
  if (conn->co_mucall != call) {
    if (ILU_ERRNOK(*err))
      return FALSE;
    else if (hard)
      return ILU_ERR_CONS0(broken_locks, err, FALSE);
    else
      return ILU_ERR_CONS0(bad_locks, err, FALSE);
  }
  conn->co_mucall = NIL;
  if (conn->co_nOuts == 0 && !connection_closed(conn)) {
    _ilu_HoldMutex(ilu_cmu);
    _ilu_LinkConnection(&ilu_idleConns, conn, ilu_lru);
  }
  if (_ilu_CanCondition()) {
    ilu_Error       lerr;
    lerr = _ilu_NotifyCondition(conn->co_cc);
    if (ILU_ERRNOK(lerr)) {
      ILU_HANDLED(lerr);
      if (ILU_ERRNOK(*err))
	return FALSE;
      return ILU_ERR_CONS0(broken_locks, err, FALSE);
    }
  }
  return TRUE;
}

/*L1 >= {cmu, conn's server}*/
/**before: L2 disjoint {(conn)'s waitmu};
   after:  L2    >=    {(conn)'s waitmu}*/

void 
_ilu_FullTakeConnWait(ilu_Connection conn, ilu_Call call,
		      const char *file, int line)
{
  ILU_NOTE(CONNECTION_DEBUG,
	("ilu_TakeConnWait(conn=%p, call=%p)@%s:%d\n",
	 conn, call, file, line));
  _ilu_Assert(!conn->co_waiting, "ilu_TakeConnWait in connect.c");
  conn->co_waiting = call;
  return;
}

ilu_boolean
_ilu_FullEnterConnWait(ilu_Connection conn, ilu_Call call,
		       ilu_boolean hard,
		       ILU_ERRS((bad_locks, broken_locks,
				 internal/broken)) * err,
		       const char *file, int line)
{
  if (!ilu_Check(call != NIL, err))
    return FALSE;
  _ilu_HoldMutex(server_lock(conn->co_server));
  _ilu_HoldMutex(ilu_cmu);
  while (conn->co_waiting) {
    ILU_NOTE(LOCK_DEBUG,
	     ("EnterConnWait(%p, %p) waiting.\n", conn, call));
    (void) ilu_CMWait2(conn->co_cc, server_lock(conn->co_server),
		       ilu_cmu, err);
    ILU_NOTE(LOCK_DEBUG,
	     ("EnterConnWait(%p, %p) resuming.\n", conn, call));
    if (ILU_ERRNOK(*err))
      return FALSE;
  }
  ILU_NOTE(LOCK_DEBUG,
	   ("EnterConnWait(%p, %p) succeeds.\n", conn, call));
  conn->co_waiting = call;
  return ILU_CLER(*err);
}

/*L1 >= {cmu, conn's server}; L1.sup < trmu*/
/**before: L2    >=    {(conn)'s waitmu};
   after:  L2 disjoint {(conn)'s waitmu}*/
ilu_boolean
_ilu_FullReleaseConnWait(ilu_Connection conn, ilu_Call call,
			 ilu_boolean hard,
			 ILU_ERRS((bad_locks, broken_locks)) * err,
			 const char *file, int line)
{
  ilu_Error       lerr;
  ilu_boolean     ans;
  ILU_NOTE(CONNECTION_DEBUG,
	("ilu_ReleaseConnWait(conn=%p, call=%p)@%s:%d\n",
	 conn, call, file, line));
  if (!ilu_Check(call != NIL, err))
    return FALSE;
  if (conn->co_waiting != call) {
    if (ILU_ERRNOK(*err))
      return FALSE;
    else if (hard)
      return ILU_ERR_CONS0(broken_locks, err, FALSE);
    else
      return ILU_ERR_CONS0(bad_locks, err, FALSE);
  }
  conn->co_waiting = NIL;
  if (conn->co_closing && !conn->co_closed) {
    ilu_boolean     takeio = !conn->co_ioing;
    if (takeio)
      (void) _ilu_TakeConnIO(conn, TRUE, &lerr);
    _ilu_CloseIoingConnection(conn, FALSE, ilu_ConnShutdownReason_ProcessTermination);
    if (takeio)
      (void) _ilu_ReleaseConnIO(conn, TRUE, &lerr);
  } else if (_ilu_CanCondition())
    (void) ilu_CondNotify(conn->co_cc, &lerr);
  else
    ILU_CLER(lerr);
  ans = ILU_ERROK(lerr);
  if (ILU_ERROK(*err))
    *err = lerr;
  else
    ILU_HANDLED(lerr);
  return ans;
}

/*L1.sup < cmu; L2 unconstrained*/

static ilu_Connection handoff = NIL;
static ilu_boolean handoffReady = FALSE;
ilu_Condition   _ilu_connHandoffChange = NIL;

ilu_boolean
_ilu_HandOffNewConnection(ilu_Connection conn,
			  ilu_Error * err)
{
  ILU_ERRS((bad_locks, broken_locks, internal)) lerr;
  if (!ilu_Check(handoffReady, &lerr))
    goto faild;
  if (!ilu_EnterMutex(ilu_cmu, &lerr))
    goto faild;
  while (handoff != NIL) {
    if (!ilu_CMWait1(_ilu_connHandoffChange, ilu_cmu, &lerr))
      goto faild;
  }
  handoff = conn;
  if (!ilu_CondNotify(_ilu_connHandoffChange, &lerr))
    goto faild;
  if (!ilu_ExitMutex(ilu_cmu, TRUE, &lerr))
    goto faild;
  return TRUE;
faild:
  if (ILU_ERROK(*err))
    *err = lerr;
  else
    ILU_HANDLED(lerr);
  return FALSE;
}

ilu_Connection 
ilu_OtherNewConnection(ILU_ERRS((internal)) * err)
{
  ilu_Connection  ans;
  if (!ilu_Check(handoffReady, err))
    return NIL;
  if (!ilu_EnterMutex(ilu_cmu, err))
    return NIL;
  while (handoff == NIL) {
    if (!ilu_CMWait1(_ilu_connHandoffChange, ilu_cmu, err))
      return NIL;
  }
  ans = handoff;
  handoff = NIL;
  if (!ilu_CondNotify(_ilu_connHandoffChange, err))
    return NIL;
  if (!ilu_ExitMutex(ilu_cmu, TRUE, err))
    return NIL;
  return ans;
}

/*Main Invariant holds; L2 not further constrained*/
ilu_boolean
ilu_NewConnectionGetterForked(ILU_ERRS((internal)) * err)
{
  if (!ilu_Check(_ilu_connHandoffChange != NIL, err))
    return FALSE;
  handoffReady = TRUE;
  return TRUE;
}

/* Main Invariant holds; L2 no further constrained */

ilu_Pipeline
ilu_GetPipeline(ILU_ERRS((no_memory, no_resources, bad_locks,
			  broken_locks, bad_param)) * err)
{
  ilu_Pipeline    pl = ilu_MallocE(sizeof(*pl), err);
  if (!pl)
    return pl;
  pl->pl_nCalls = 0;
  pl->pl_lsrCares = TRUE;
  return pl;
}

ilu_boolean
ilu_ReleasePipeline(ilu_Pipeline pl,
		    ILU_ERRS((bad_locks, broken_locks,
			      bad_param)) *err)
{
  if (!pl)
    return ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_duh, FALSE);
  if (!ilu_EnterMutex(ilu_cmu, err))
    return FALSE;
  pl->pl_lsrCares = FALSE;
  _ilu_MaybeFreePipeline(pl);
  if (!ilu_ExitMutex(ilu_cmu, TRUE, err))
    return FALSE;
  return TRUE;
}

/* L1 >= {cmu}; L2 unconstrained */
void _ilu_MaybeFreePipeline(ilu_Pipeline pl)
{
  if (pl && !(pl->pl_lsrCares || pl->pl_nCalls))
    ilu_free(pl);
  return;
}

/* L1 = {server}; Main Remnant holds */
ilu_Serializer
ilu_InnerGetSerializer(ilu_Server server,
		  ILU_ERRS((no_memory, no_resources, bad_locks,
			    broken_locks, bad_param)) * err)
{
  ilu_Serializer  si = ilu_MallocE(sizeof(*si), err);
  if (!si)
    return NIL;
  if (!ilu_DeltaServerHolds(server, 1, err))
    goto dun;
  si->si_server = server;
  si->si_conn = NIL;
  si->si_nCalls = 0;
  si->si_lsrCares = TRUE;
  si->si_connecting = FALSE;
  si->si_connChg = NIL;
dun:
  if (ILU_ERRNOK(*err)) {
    ilu_free(si);
    return NIL;
  }
  return si;
}

ilu_Serializer
ilu_GetSerializer(ilu_Server server,
		  ILU_ERRS((no_memory, no_resources, bad_locks,
			    broken_locks, bad_param)) * err)
{
  ilu_Serializer  ans;
  if (!ilu_EnterServerMutex(server, FALSE, err))
    return NIL;
  ans = ilu_InnerGetSerializer(server, err);
  if (!ilu_ExitServerMutex(server, TRUE, err))
    return NIL;
  return ans;
}

ilu_boolean
ilu_ReleaseSerializer(ilu_Serializer si,
		      ILU_ERRS((bad_locks, broken_locks,
				bad_param)) * err)
{
  ilu_Server      server;
  if (!si || !(server = si->si_server))
    return ILU_ERR_CONS1(bad_param, err, minor, ilu_bpm_duh, FALSE);
  if (!ilu_EnterServerMutex(server, FALSE, err))
    return FALSE;
  if (!ilu_DeltaServerHolds(server, -1, err))
    goto dun1;
  si->si_lsrCares = FALSE;
  _ilu_MaybeFreeSerializer(si);
dun1:
  if (!ilu_ExitServerMutex(server, TRUE, err))
    return FALSE;
  return TRUE;
}

/* L1 >= {si->si_server}; L2 unconstrained */
void _ilu_MaybeFreeSerializer(ilu_Serializer si)
{
  if (si && !(si->si_lsrCares || si->si_nCalls)) {
    if (si->si_conn) {
      si->si_conn->co_serialer = NIL;
      si->si_conn = NIL;
    }
    if (si->si_connChg) {
      ILU_ERRS((CantCondition)) lerr;
      lerr = ilu_DestroyCondition(si->si_connChg);
      ILU_MUST_BE_SUCCESS(lerr);
    }
    ilu_free(si);
  }
  return;
}
