/*
Copyright (c) 1996-1997 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: cppparsenarrow.hpp,v 1.1 1997/02/13 07:04:51 srjohnso Exp $
*/

/* ISL Parser backend for C++ */

#ifndef __cppparsenarrow_hpp_
#define __cppparsenarrow_hpp_ 1

#include "cppportability.hpp"


/*

"Narrowing" is equivalent to "downcasting" from a pointer to one (base) type to a pointer to a more specific (derived) type; it is a restricted form of the "dynamic cast" defined by RTTI. Unfortunately, dynamic_cast does not allow a downcast in the face of virtual inheritance (ANSI C++ April 1995 Working Paper, X3J16/95-0087). Further, not all compilers support RTTI (MSVC4.1 does; SunPro4.1 does not; G++2.7.1's support is in alpha).

The macros defined here support type-safe downcasting within an inheritance hierarchy, irrespective of virtual/non-virtual inheritance. Expanding one of these macros within a class declaration causes the definition of a "narrow" function member (ie, a class method called "narrow"). A "narrow" method on class "Foo" attempts to convert it's argument (a pointer) to a pointer to "Foo", and returns the result. The result will be NULL if the actual object referred to by the argument is not of class "Foo", nor of a class derived from "Foo"; or, if the argument itself is NULL.

Which macro to use is determined by the number of (immediate) base classes for the class at hand (currently, there are macros for a maximum of 2 bases):

  CppParseNarrow_FUNCS0: No base class.
  CppParseNarrow_FUNCS1: 1 base class.
  CppParseNarrow_FUNCS2: 2 base classes.

The first argument to each macro is the name of the class at hand; succeeding arguments are the names of the base classes (Note that it is not necessary to supply all of the actual bases, only those that you want to be part of the "narrowing lattice"). Further, the top of a "narrowing lattice" must derive from the class CppParseNarrow_Root (the argument type of all "narrow" functions is of type "CppParseNarrow_Root *").

The example below illustrates use of the macros:


  class A: public CppParseNarrow_Root {
    CppParseNarrow_FUNCS0(A); 
    <other A members>
  };

  class B : public A {
    CppParseNarrow_FUNCS1(B, A);
    <other B members>
  };

  class C1: virtual public B {
    CppParseNarrow_FUNCS1(C1, B);
    <other C1 members>
  };

  class C2: virtual public B {
    CppParseNarrow_FUNCS1(C2, B);
    <other C2 members>
  };

  class D: public C1, public C2 {
    CppParseNarrow_FUNCS2(D, C1, C2);
    <other D members>
  };

  C2  c2;
  B * bptr;

  bptr = &c2;

  if (B::narrow(bptr) != NULL)
    cout << "bptr * is a B" << endl;
  else
    cout << "bptr * is not a B" << endl;

  if (C1::narrow(bptr) != NULL)
    cout << "bptr * is a C1" << endl;
  else
    cout << "bptr * is not a C1" << endl;

  if (C2::narrow(bptr) != NULL)
    cout << "bptr * is a C2" << endl;
  else
    cout << "bptr * is not a C2" << endl;

  if (D::narrow(bptr) != NULL)
    cout << "bptr * is a D" << endl;
  else
    cout << "bptr * is not a D" << endl;


The above code will generate the following output:

  bptr * is a B
  bptr * is not a C1
  bptr * is a C2
  bptr * is not a D


Note: Macros are used to define methods directly on the class at hand rather than simply inheriting them (from a template class); the latter would require clumsy syntax at the point of invocation to resolve inheritance ambiguity. However, each macro-defined function delegates it's work to a template class static function member; this makes debugging easier (breakpoints can be set).

*/


/*
  "Public". Classes at top of "narrowing lattice" need to inherit from this.
*/

class CppParseNarrow_Root {
  public:
    virtual void * asSuper (void *) const {
      return (NULL);
    }
};


/*
   "Private" macros/templates.

   The value returned by __cppparsenarrow_hpp_CommonFuncsT::classID() can be any value unique to class T.  For the sake of encapsulation, an address of something introduced by __cppparsenarrow_hpp_CommonFuncsT (eg, narrow) would be slightly preferreable;  however, this will fail with some compilers (MSVC4.1 generates inconsistent values, leading to infinite recursion when comparisons fail; SunPro4.1 generates a compile-time warning). Using the address of T::narrow fails under G++2.7.1 (at runtime; apparently, different addresses are generated at different times, probably depending on the state of template instantiation). Thus, we introduce the static int member T::cid, whose sole purpose is to provide an address with a value unique to T.
*/



template <class X>  // MSVC Compiler Note #1
class __cppparsenarrow_hpp_CommonFuncsT {
  public:
    static void * classID () {
      return STATIC_CAST(void*, &X::cid);
    }
    static X * narrow (const CppParseNarrow_Root * r) {
      X * rtn;
      if (r == NULL)
        rtn = STATIC_CAST(X*, NULL);
      else
        rtn = STATIC_CAST(X*, (r->asSuper(classID())));
      return rtn;
    }
}; 

template <class T>
class __cppparsenarrow_hpp_Node0T {
   public:
    static void * asSuper (const T * t, void * id) {
      void * rtn;
      if (id == __cppparsenarrow_hpp_CommonFuncsT<T>::classID())
        rtn = STATIC_CAST(void*, CONST_CAST(T*, t));
      else
        rtn = STATIC_CAST(void*, NULL);
      return rtn;
    }
}; 

template <class T, class BASE>
class __cppparsenarrow_hpp_Node1T {
   public:
    static void * asSuper (const T * t, void * id) {
      void * rtn;
      if (id == __cppparsenarrow_hpp_CommonFuncsT<T>::classID())
        rtn = STATIC_CAST(void*, CONST_CAST(T*, t));
      else
        rtn = t->BASE::asSuper(id);
      return rtn;
    }
}; 

template <class T, class BASE1, class BASE2>
class __cppparsenarrow_hpp_Node2T {
   public:
    static void * asSuper (const T * t, void * id) {
      void * rtn;
      if (id == __cppparsenarrow_hpp_CommonFuncsT<T>::classID())
        rtn = STATIC_CAST(void*, CONST_CAST(T*, t));
      else if ((rtn = t->BASE1::asSuper(id)) == NULL)
        rtn = t->BASE2::asSuper(id);
      return rtn;
    }
};


#ifdef __GNUC__  // G++2.7.1 Compiler Note #1

  #define __cppparsenarrow_hpp_COMMON_FUNCS(T)                    \
    public:                                                       \
      static T * narrow (const CppParseNarrow_Root * r) {         \
        return (__cppparsenarrow_hpp_CommonFuncsT<T>::narrow(r)); \
      }                                                           \
      static int cid  // see note above on cid

#else

  #define __cppparsenarrow_hpp_COMMON_FUNCS(T)                    \
    public:                                                       \
      static T * narrow (CppParseNarrow_Root * r) {               \
        return (__cppparsenarrow_hpp_CommonFuncsT<T>::narrow(r)); \
      }                                                           \
      static const T * narrow (const CppParseNarrow_Root * r) {   \
        return (__cppparsenarrow_hpp_CommonFuncsT<T>::narrow(r)); \
      }                                                           \
      static int cid  // see note above on cid

#endif  // __GNUC__


/*
   "Public" macros. Expand these in your class definitions.
*/

#define CppParseNarrow_FUNCS0(T)                                  \
  __cppparsenarrow_hpp_COMMON_FUNCS(T);                           \
  public:                                                         \
    virtual void * asSuper (void * id) const {                    \
      return (__cppparsenarrow_hpp_Node0T<T>::asSuper(this, id)); \
    }
 
#define CppParseNarrow_FUNCS1(T, BASE)                                  \
  __cppparsenarrow_hpp_COMMON_FUNCS(T);                                 \
  public:                                                               \
    virtual void * asSuper (void * id) const {                          \
      return (__cppparsenarrow_hpp_Node1T<T, BASE>::asSuper(this, id)); \
    }
 
#define CppParseNarrow_FUNCS2(T, BASE1, BASE2)                                  \
  __cppparsenarrow_hpp_COMMON_FUNCS(T);                                         \
  public:                                                                       \
    virtual void * asSuper (void * id) const {                                  \
      return (__cppparsenarrow_hpp_Node2T<T, BASE1, BASE2>::asSuper(this, id)); \
    }
  


#endif  // __cppparsenarrow_hpp_


/*
    G++2.7.1 Compiler Note #1 - When "narrow" is overloaded based on const/non-const argument, the compiler cannot determine which version to call (the ANSI C++ April 1995 Working Paper, X3J16/95-0087 implies that it should be able to). The single method defined here will work for both const and non-const cases, although it violates the spirit of "constness". Oh well...

    MSVC4.1 Compiler Note #1 - We use the name "X" instead of the usual "T" for the first template parameter, because - for some strange reason - the compiler doesn't like "T" (warning 2027: "use of undefined type 'T', and other errors).

*/
