// proc.h				emacs, this is written in -*-c++-*-
//
// This program is free software. See the file COPYING for details.
// Author: Mattias Engdegrd, 1997, 1998

#ifndef PROC_H
#define PROC_H

#include <unistd.h>
#include <sys/time.h>
#include <qstring.h>
#include <qwindefs.h>
#include <qintdict.h>
#include "svec.h"

enum fields {
    F_PID, F_PPID, F_PGID, F_SID, F_TTY, F_TPGID,
    F_USER,
    F_PRI, F_NICE,
    F_PLCY, F_RPRI,
    F_MAJFLT, F_MINFLT, F_TRS, F_DRS, F_SIZE, F_SWAP, F_RSS, F_SHARE, F_DT,
    F_STAT,
    F_FLAGS,
    F_WCHAN,
    F_UID,
    F_WCPU, F_CPU, F_MEM,
    F_START, F_TIME,
    F_COMM, F_CMDLINE,
    F_END = -1 };

class Details;

class Sockinfo
{
public:
    enum proto_t { TCP, UDP };
    proto_t proto;
    unsigned char st;
    unsigned char tr;
    unsigned local_addr;
    unsigned rem_addr;
    unsigned short local_port;
    unsigned short rem_port;
    unsigned tx_queue;
    unsigned rx_queue;
    unsigned tm_when;
    unsigned rexmits;
    int uid;
    int timeout;
    int inode;
};

class Mapsinfo
{	
public:
    // members rearranged to minimize padding (should be done in all structs)
    unsigned long from, to;
    unsigned long offset;
    unsigned long inode;
    QString filename;		// null if Linux 2.0 and name unknown
    char perm[4];		// "rwx[ps]"; last is private/shared flag
    unsigned char minor, major;
};

class Fileinfo
{
public:
    Fileinfo(int descr, QString name) : fd(descr), filename(name) {};
    int fd;
    QString filename;		// "major:minor inode" in Linux 2.0
};

class Procinfo
{
public:
    Procinfo(int pid);
    ~Procinfo();

    int readproc(int pid);
    static void read_common();
    static void read_loadavg();
    static int read_file(char *name, char *buf, int max);
    bool read_fds();
    static bool read_socket_list(Sockinfo::proto_t proto, char *pseudofile);
    static void read_sockets();
    static void invalidate_sockets();
    bool read_maps();

    // accessors for lazy evaluation
    int get_policy();
    int get_rtprio();

    int pid;

    // from /proc/XX/status
    int uid, euid, suid, fsuid;
    int gid, egid, sgid, fsgid;

    // from /proc/XX/cmdline
    QString cmdline;

    // from /proc/XX/stat
    QString comm;
    char state;
    int ppid;
    int pgrp;
    int session;
    int tty;
    int tpgid;
    unsigned long flags;
    unsigned long minflt;
    unsigned long cminflt;
    unsigned long majflt;
    unsigned long cmajflt;
    long utime;
    long stime;
    long cutime;
    long cstime;
    int priority;
    int nice;
    unsigned long timeout;
    unsigned long starttime;
    unsigned long rss;
    unsigned long wchan;

    // from /proc/XX/statm
    // (not required to be long but it's more rational)
    unsigned long size;		// total memory (K)
    unsigned long resident;	// pages in resident set (non-swapped) (K)
    unsigned long share;	// shared memory pages (mmaped) (K)
    unsigned long trs;      	// text resident set size (K)
    unsigned long lrs;		// shared-lib resident set size (K)
    unsigned long drs;		// data resident set size (K)
    unsigned long dt;		// dirty pages (number of pages, not K)

    // from gettimeofday(2)
    struct timeval tv;

    // from sched_getscheduler(2)
    int policy;		// -1 = uninitialized
    // from sched_getparam(2)
    int rtprio;		// 0-99, higher can pre-empt lower (-1 = uninitialized)

    // computed %cpu and %mem since last update
    float wcpu;
    float pcpu;
    float pmem;

    // from /proc/loadavg
    static float loadavg[3];

    // from /proc/meminfo (these are in kB)
    static int mem_total, mem_free, mem_shared, mem_buffers;
    static int mem_cached;
    static int swap_total, swap_free;

    // from /proc/stat
    enum { CPU_USER, CPU_NICE, CPU_SYSTEM, CPU_IDLE };
    const int CPUTIMES = CPU_IDLE + 1;
    static unsigned cpu_time[CPUTIMES];
    static unsigned old_cpu_time[CPUTIMES];
    static unsigned boot_time;

    // from /proc/uptime
    static int uptime;	// seconds of uptime

    // from /proc/net/{tcp,udp}
    static QIntDict<Sockinfo> socks;
    static bool socks_current;	// true if the socks list is current

    // from /proc/XX/fd
    Svec<Fileinfo*> *fd_files;	// file names or NULL if not read
    Svec<int> *sock_inodes;	// socket inodes or NULL if not read

    Details *details;		// details window or NULL (backlink)
    
    // from /proc/XX/maps
    Svec<Mapsinfo*> *maps;	// maps or NULL if not read

    bool selected;	// true if selected in current view
    bool viewed;	// true if part of current process view

    const MAX_CMD_LEN = 4096;
    static int pagesize;
};

class Category
{
public:
    Category(char *heading, char *explain) : name(heading), help(explain) {};

    virtual int alignment() = 0;
    virtual QString string(Procinfo *p) = 0;
    virtual int width() = 0;
    virtual int compare(Procinfo *a, Procinfo *b);
    virtual int gap() { return 0; };

    char *name;
    char *help;
    int index;
};

class Cat_int : public Category
{
public:    
    Cat_int(char *heading, char *explain, int w, int Procinfo::*member);
    virtual int alignment() { return AlignRight; };
    virtual QString string(Procinfo *p);
    virtual int width() { return field_width; };
    virtual int compare(Procinfo *a, Procinfo *b);

protected:
    int Procinfo::*int_member;
    int field_width;
};

class Cat_uintl : public Category
{
public:
    Cat_uintl(char *heading, char *explain, int w,
	      unsigned long Procinfo::*member);
    virtual int alignment() { return AlignRight; };
    virtual QString string(Procinfo *p);
    virtual int width() { return field_width; };
    virtual int compare(Procinfo *a, Procinfo *b);

protected:
    unsigned long Procinfo::*uintl_member;
    int field_width;
};

class Cat_hex : public Cat_uintl
{
public:
    Cat_hex(char *heading, char *explain, int w,
	    unsigned long Procinfo::*member);
    virtual QString string(Procinfo *p);
};

class Cat_swap : public Category
{
public:
    Cat_swap(char *heading, char *explain);
    virtual int alignment() { return AlignRight; };
    virtual QString string(Procinfo *p);
    virtual int width() { return 56; };
    virtual int compare(Procinfo *a, Procinfo *b);
};

class Cat_string : public Category
{
public:
    Cat_string(char *heading, char *explain, QString Procinfo::*member = 0);
    virtual int alignment() { return AlignLeft; };
    virtual QString string(Procinfo *p);
    virtual int width() { return -1; };
    virtual int gap() { return 8; };

protected:
    QString Procinfo::*str_member;
};

class Cat_user : public Cat_string
{
public:
    Cat_user(char *heading, char *explain);
    virtual QString string(Procinfo *p);
};

class Cat_wchan : public Cat_string
{
public:
    Cat_wchan(char *heading, char *explain);
    virtual QString string(Procinfo *p);
};

class Cat_cmdline : public Cat_string
{
public:
    Cat_cmdline(char *heading, char *explain);
    virtual QString string(Procinfo *p);
};

class Cat_state : public Category
{
public:
    Cat_state(char *heading, char *explain);
    virtual int alignment() { return AlignLeft; };
    virtual QString string(Procinfo *p);
    virtual int width() { return 44; };
    virtual int gap() { return 8; };
};

class Cat_policy : public Category
{
public:
    Cat_policy(char *heading, char *explain);
    virtual int alignment() { return AlignLeft; };
    virtual QString string(Procinfo *p);
    virtual int width() { return 46; };
    virtual int gap() { return 8; };
    virtual int compare(Procinfo *a, Procinfo *b);
};

class Cat_rtprio : public Category
{
public:
    Cat_rtprio(char *heading, char *explain);
    virtual int alignment() { return AlignRight; };
    virtual QString string(Procinfo *p);
    virtual int width() { return 36; };
    virtual int compare(Procinfo *a, Procinfo *b);
};

class Cat_time : public Category
{
public:
    Cat_time(char *heading, char *explain);
    virtual int alignment() { return AlignRight; };
    virtual QString string(Procinfo *p);
    virtual int width() { return 48; };
    virtual int compare(Procinfo *a, Procinfo *b);
};

class Cat_start : public Category
{
public:
    Cat_start(char *heading, char *explain);
    virtual int alignment() { return AlignRight; };
    virtual QString string(Procinfo *p);
    virtual int width() { return 56; };
    virtual int compare(Procinfo *a, Procinfo *b);
};

class Cat_percent : public Category
{
public:
    Cat_percent(char *heading, char *explain, int w, float Procinfo::*member);
    virtual int alignment() { return AlignRight; };
    virtual QString string(Procinfo *p);
    virtual int width() { return field_width; };
    virtual int compare(Procinfo *a, Procinfo *b);

protected:
    float Procinfo::*float_member;
    int field_width;
};

class Cat_tty : public Category
{
public:
    Cat_tty(char *heading, char *explain);
    virtual int alignment() { return AlignLeft; };
    virtual QString string(Procinfo *p);
    virtual int width() { return 36; };
    virtual int gap() { return 8; };
};

class Proc
{
public:
    Proc();
    void refresh();

    Svec<Procinfo *> procs;
    Svec<Category *> cats;
};

class Procview
{
public:
    Procview(Proc *p);
    static int compare(Procinfo **a, Procinfo **b);
    static int compare_backwards(Procinfo **a, Procinfo **b);
    void refresh();
    void rebuild();
    void set_fields();
    void set_fields_list(int fields[]);
    void add_cat(Category *c);
    void remove_cat(int index);
    int findCol(int field);
    
    Svec<Procinfo *> procs;
    Svec<Procinfo *> old_procs;
    Svec<Category *> cats;
    Category *sortcat;
    bool reversed;		// true if sorted backwards

    enum procstates {ALL, OWNED, NROOT, RUNNING} viewproc;
    enum fieldstates {USER = RUNNING + 1, JOBS, MEM, CUSTOM} viewfields;

    // lists of fields to be used for different views, terminated by -1:
    static int user_fields[];
    static int jobs_fields[];
    static int mem_fields[];

    static float avg_factor;		// exponential factor for averaging

    const int cpu_avg_time = 30 * 1000;	// averaging time for WCPU (ms)

private:
    static Category *static_sortcat;	// kludge: to be used by compare

    Proc *proc;
};

#endif	// PROC_H
