/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State Universtiy
 *	GDB
 *
 *	$Id: terror.c,v 6.1 96/11/23 19:58:19 nevin Rel $
 *
 *	Function:	- Trollius error messages
 */

#include <string.h>
#include <unistd.h>

#ifdef WIN32
#include <windows.h>
#endif

#include <portable.h>
#include <terror.h>

/*
 * local variables
 */
static char		*lam_emsg[] = {
		"undiagnosed error",		/* EUNKNOWN */
		"remote node has died",		/* EDEADNODE */
		"out of exit descriptors",	/* ENOEXDESCRIPTORS */
		"out of descriptors",		/* EFULL */
		"not a process",		/* ENOTPROCESS */
		"invalid request",		/* EBADREQUEST */
		"block not allocated",		/* ENOTMALLOCED */
		"GER overflow",			/* EGERFLOW */
		"service unavailable",		/* ESERVICE */
		"bad magic number",		/* EMAGIC */
		"blown stack",			/* EBLOWN */
		"impossible, ergo a bug",	/* EIMPOSSIBLE */
		"operation timed out",		/* LAM_ERR_TO */
		"unknown host",			/* EBADHOST */
		"no running daemon",		/* ENOKERNEL */
		"not attached to daemon",	/* ENOTATTACHED */
		"operation would block",	/* LAM_ERR_BLOCK */
		"improper message size",	/* EMSGLEN */
		"end of file",			/* EEOF */
		"cast spanning tree collision",	/* ECOLLIDE */
		"invalid parameter",		/* LAM_ERR_INVAL */
		"unused error code",		/* unused */
		"can't find shared segment",	/* ESHARE */
		"invalid link index",		/* EBADLINK */
		"bad acknowledgement",		/* EACK */
		"improper command invocation",	/* EUSAGE */
		"bad/unknown option",		/* EOPTION */
		"unused error code",		/* unused */
		"invalid node",			/* EBADNODE */
		"invalid ID specification",	/* EBADIDSPEC */
		"no matching nodes found",	/* ENONODES */
		"out of route descriptors",	/* ENORDESCRIPTORS */
		"out of buffer descriptors",	/* ENOBFDESCRIPTORS */
		"out of boot descriptors",	/* ENOBTDESCRIPTORS */
		"out of flat descriptors",	/* ENOFLDESCRIPTORS */
		"out of kenya descriptors",	/* ENOPDESCRIPTORS */
		"out of process descriptors",	/* ENOKDESCRIPTORS */
		"bad moses request",		/* EBADMOSESREQ */
		"unused error code",		/* unused */
		"invalid address tag",		/* EBADTAG */
		"delete from list failed",	/* EDELETE */
		"no bootstrap present",		/* ENOBOOTSTRAP */
		"invalid record format",	/* EBADREC */
		"bad ascii number",		/* EBADASCIINUM */
		"booted system is too big",	/* EHIGHWATER */
		"process already in group",	/* EINGROUP */
		"limit 32 group procs/node",	/* EGMAXPROC */
		"group has a single process",	/* EGSINGLETON */
		"not a group process",		/* EGNOTPROC */
		"invalid group",		/* EBADGROUP */
		"MPI already initialized",	/* EMPIINIT */
		"MPI not initialized",		/* EMPINOINIT */
		"duplicate virtual circuit",	/* LAM_ERR_DUPVC */
		"invalid virtual circuit",	/* LAM_ERR_BADVC */
		"name is published",		/* EPUBLISHED */
		"name is not published",	/* ENOTPUBLISHED */
			};

static char		*baderror = "error code ";

/*
 * extern variables
 */
extern int		sys_nerr;
extern char		*sys_errlist[];

/*
 * external functions
 */
extern void		sfh_itoa();		/* int to ascii */

/*
 *	lam_errorstr
 *
 *	Function:	- obtains error message string
 *			- uses Trollius, errno and system message tables
 *			- message is stored user provided space
 *	Accepts:	- user's error message buffer
 *			- buffer length
 */
void
lam_errorstr(emsgbuf, emsgbuflen)

char			*emsgbuf;
unint			emsgbuflen;

{
	int		ecode;			/* errno code */
	int		found = 0;
	char		anum[16];		/* ascii error code */

	if ((emsgbuf == 0) || (emsgbuflen < 1)) return;

	*emsgbuf = '\0';
	emsgbuflen--;
	ecode = LAMGetLastError();
/*
 * Look for C runtime error message (errno).
 */
	if ((ecode >= LAM_ERRNO_BASE) &&
			(ecode < (LAM_ERRNO_BASE + sys_nerr))) {
		strncat(emsgbuf, sys_errlist[ecode], emsgbuflen);
	}
/*
 * Look for LAM error message.
 */
	else if ((ecode >= ELOW) && (ecode <= EHIGH)) {
		strncat(emsgbuf, lam_emsg[ecode - ELOW], emsgbuflen);
		found = 1;
	}
/*
 * Error code is obtained from the system.
 */
	else {
#ifdef WIN32
		found = FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, ecode,
				MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
				(LPTSTR) emsgbuf, emsgbuflen, NULL);
#else
		found = 0;
#endif
/*
 * Return a bad error code message.
 */
		if (!found) {
			sfh_itoa(ecode, anum);
			strncat(emsgbuf, baderror, emsgbuflen);

			if (strlen(baderror) < emsgbuflen) {
				strncat(emsgbuf, anum, 
					emsgbuflen - strlen(baderror));
			}
		}
	}
}

/*
 *	lam_perror
 *
 *	Function:	- prints error messages on stderr
 *	Accepts:	- user's error message
 */
void
lam_perror(umsg)

CONST char		*umsg;

{
	char		emsgbuf[LAM_EMSGBUFLEN];
	unint		l;

	if ((umsg == 0) || (*umsg == '\0')) {
		lam_errorstr(emsgbuf, LAM_EMSGBUFLEN - 1);
		strcat(emsgbuf, "\n");
	}

	else {
		*emsgbuf = '\0';
		l = strlen(umsg);
		strncat(emsgbuf, umsg, LAM_EMSGBUFLEN - 2);
/*
 * Continue building the error message if I have enough room for
 * the separator (2), at least one character of the error message,
 * a newline, and a string termination.
 */
		if (l <= (LAM_EMSGBUFLEN - 5)) {
			strcat(emsgbuf, ": ");
			lam_errorstr(emsgbuf + l + 2,
					(LAM_EMSGBUFLEN - (l + 2)) - 1);
		}

		strcat(emsgbuf, "\n");
	}

	write(2, emsgbuf, strlen(emsgbuf));
}

/*
 * backwards compatibility
 */
void terror(umsg) CONST char *umsg; { lam_perror(umsg); }
