/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: alltoallv.c,v 6.1 96/11/23 22:50:47 nevin Rel $
 *
 *	Function:	- send/recv var. len. data from each node to all nodes
 *	Accepts:	- send buffer
 *			- send counts
 *			- send displacements
 *			- send datatype
 *			- recv buffer
 *			- recv counts
 *			- recv displacements
 *			- recv datatype
 *			- communicator
 *	Returns:	- MPI_SUCCESS or an MPI error code
 */

#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <terror.h>

/*
 * local functions
 */
static int		c2c_alltoallv();
static int		lamd_alltoallv();


int
MPI_Alltoallv(sbuf, scounts, sdisps, sdtype,
		rbuf, rcounts, rdisps, rdtype, comm)

void			*sbuf;
int			*scounts;
int			*sdisps;
MPI_Datatype		sdtype;
void			*rbuf;
int			*rcounts;
int			*rdisps;
MPI_Datatype		rdtype;
MPI_Comm		comm;

{
/*
 * Check for invalid arguments.
 */
	if ((comm == MPI_COMM_NULL) || LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm, BLKMPIALLTOALLV,
				lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if ((sdtype == MPI_DATATYPE_NULL) || (rdtype == MPI_DATATYPE_NULL)) {
		return(lam_errfunc(comm, BLKMPIALLTOALLV,
				lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if ((scounts == 0) || (rcounts == 0)) {
		return(lam_errfunc(comm, BLKMPIALLTOALLV,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if ((sdisps == 0) || (rdisps == 0)) {
		return(lam_errfunc(comm, BLKMPIALLTOALLV,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}

	LAM_TRACE(lam_tr_cffstart(BLKMPIALLTOALLV));

	return(RPI_SPLIT(lamd_alltoallv, c2c_alltoallv,
				(sbuf, scounts, sdisps, sdtype,
				rbuf, rcounts, rdisps, rdtype, comm)));
}


/*
 *	c2c_alltoallv
 *
 *	Function:	- MPI_Alltoallv for the C2C RPI
 *	Accepts:	- same as MPI_Alltoallv
 *	Returns:	- MPI_SUCCESS or an MPI error code
 */
static int
c2c_alltoallv(sbuf, scounts, sdisps, sdtype,
		rbuf, rcounts, rdisps, rdtype, comm)

void			*sbuf;
int			*scounts;
int			*sdisps;
MPI_Datatype		sdtype;
void			*rbuf;
int			*rcounts;
int			*rdisps;
MPI_Datatype		rdtype;
MPI_Comm		comm;

{
	int		i;			/* favourite index */
	int		size;			/* group size */
	int		rank;			/* my rank */
	int		nreqs;			/* # requests */
	int		err;			/* error code */
	char		*psnd;			/* ptr send buffer */
	char		*prcv;			/* ptr recv buffer */
	MPI_Aint	sndextent;		/* send datatype extent */
	MPI_Aint	rcvextent;		/* recv datatype extent */
	MPI_Request	*req;			/* request array */
	MPI_Request	*preq;			/* ptr request */
	MPI_Status	*stat;			/* status array */
/*
 * Initialize.
 */
	MPI_Comm_size(comm, &size);
	MPI_Comm_rank(comm, &rank);

	MPI_Type_extent(sdtype, &sndextent);
	MPI_Type_extent(rdtype, &rcvextent);
/*
 * Allocate arrays of requests and status structures.
 */
	nreqs = 2 * (size - 1);

	if (nreqs > 0) {
		req = (MPI_Request *) malloc((unsigned)
					nreqs * sizeof(MPI_Request));
		stat = (MPI_Status *) malloc((unsigned)
					nreqs * sizeof(MPI_Status));

		if ((req == 0) || (stat == 0)) {
			if (req) free((char *) req);
			if (stat) free((char *) stat);
			return(lam_errfunc(comm, BLKMPIALLTOALLV,
					lam_mkerr(MPI_ERR_OTHER, errno)));
		}
	}
	else {
		req = 0;
		stat = 0;
	}
/*
 * Switch to collective communicator.
 */
	lam_mkcoll(comm);
/*
 * simple optimization
 */
	psnd = ((char *) sbuf) + (sdisps[rank] * sndextent);
	prcv = ((char *) rbuf) + (rdisps[rank] * rcvextent);

	err = lam_dtsndrcv(psnd, scounts[rank], sdtype,
			prcv, rcounts[rank], rdtype, BLKMPIALLTOALLV, comm);

	if (err != MPI_SUCCESS) {
		if (req) free((char *) req);
		if (stat) free((char *) stat);
		lam_mkpt(comm);
		return(lam_errfunc(comm, BLKMPIALLTOALLV, err));
	}
/*
 * If only one process, generate run time trace and we're done.
 */
	if (size == 1) {
		lam_mkpt(comm);
		LAM_TRACE(lam_tr_cffend(BLKMPIALLTOALLV, -1, comm,
				sdtype, 0));

		lam_resetfunc(BLKMPIALLTOALLV);
		return(MPI_SUCCESS);
	}
/*
 * Initiate all send/recv to/from others.
 */
	preq = req;

	for (i = 0; i < size; ++i) {

		if (i == rank) continue;

		prcv = ((char *) rbuf) + (rdisps[i] * rcvextent);

		err = MPI_Recv_init(prcv, rcounts[i], rdtype,
					i, BLKMPIALLTOALLV, comm, preq++);

		if (err != MPI_SUCCESS) {
			free((char *) req);
			free((char *) stat);
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPIALLTOALLV, err));
		}
	}

	for (i = 0; i < size; ++i) {

		if (i == rank) continue;

		psnd = ((char *) sbuf) + (sdisps[i] * sndextent);

		err = MPI_Send_init(psnd, scounts[i], sdtype,
					i, BLKMPIALLTOALLV, comm, preq++);

		if (err != MPI_SUCCESS) {
			free((char *) req);
			free((char *) stat);
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPIALLTOALLV, err));
		}
	}
/*
 * Start all requests.
 */
	err = MPI_Startall(nreqs, req);

	if (err != MPI_SUCCESS) {
		free((char *) req);
		free((char *) stat);
		lam_mkpt(comm);
		return(lam_errfunc(comm, BLKMPIALLTOALLV, err));
	}
/*
 * Wait for them all.
 */
	err = MPI_Waitall(nreqs, req, stat);

	lam_mkpt(comm);
	free((char *) stat);

	if (err != MPI_SUCCESS) {
		free((char *) req);
		return(lam_errfunc(comm, BLKMPIALLTOALLV, err));
	}
/*
 * Free the requests.
 */
	for (i = 0, preq = req; i < nreqs; ++i, ++preq) {

		err = MPI_Request_free(preq);
		if (err != MPI_SUCCESS) {
			free((char *) req);
			return(lam_errfunc(comm, BLKMPIALLTOALLV, err));
		}
	}

	free((char *) req);
	LAM_TRACE(lam_tr_cffend(BLKMPIALLTOALLV, -1, comm, sdtype, 0));

	lam_resetfunc(BLKMPIALLTOALLV);
	return(MPI_SUCCESS);
}

/*
 *	lamd_alltoallv
 *
 *	Function:	- MPI_Alltoallv for the LAMD RPI
 *	Accepts:	- same as MPI_Alltoallv
 *	Returns:	- MPI_SUCCESS or an MPI error code
 */
static int
lamd_alltoallv(sbuf, scounts, sdisps, sdtype,
		rbuf, rcounts, rdisps, rdtype, comm)

void			*sbuf;
int			*scounts;
int			*sdisps;
MPI_Datatype		sdtype;
void			*rbuf;
int			*rcounts;
int			*rdisps;
MPI_Datatype		rdtype;
MPI_Comm		comm;

{
	int		i;			/* favourite index */
	int		size;			/* group size */
	int		rank;			/* my rank */
	int		err;			/* error code */
	char		*psnd;			/* ptr send buffer */
	char		*prcv;			/* ptr recv buffer */
	MPI_Aint	sndextent;		/* send datatype extent */
	MPI_Aint	rcvextent;		/* recv datatype extent */
	MPI_Status	stat;			/* status array */
/*
 * Initialize.
 */
	MPI_Comm_size(comm, &size);
	MPI_Comm_rank(comm, &rank);

	MPI_Type_extent(sdtype, &sndextent);
	MPI_Type_extent(rdtype, &rcvextent);
/*
 * Switch to collective communicator.
 */
	lam_mkcoll(comm);
/*
 * simple optimization
 */
	psnd = ((char *) sbuf) + (sdisps[rank] * sndextent);
	prcv = ((char *) rbuf) + (rdisps[rank] * rcvextent);

	err = lam_dtsndrcv(psnd, scounts[rank], sdtype,
			prcv, rcounts[rank], rdtype, BLKMPIALLTOALLV, comm);

	if (err != MPI_SUCCESS) {
		lam_mkpt(comm);
		return(lam_errfunc(comm, BLKMPIALLTOALLV, err));
	}
/*
 * Do sendrecv's with others if any.
 */
	for (i = 0; i < size; ++i) {

		if (i == rank) continue;

		psnd = ((char *) sbuf) + (sdisps[i] * sndextent);
		prcv = ((char *) rbuf) + (rdisps[i] * rcvextent);
		
		err = MPI_Sendrecv(psnd, scounts[i], sdtype, i,
			BLKMPIALLTOALLV, prcv, rcounts[i], rdtype, i,
			BLKMPIALLTOALLV, comm, &stat);

		if (err != MPI_SUCCESS) {
			lam_mkpt(comm);
			return(lam_errfunc(comm, BLKMPIALLTOALLV, err));
		}
	}	

	lam_mkpt(comm);
	LAM_TRACE(lam_tr_cffend(BLKMPIALLTOALLV, -1, comm, sdtype, 0));
	
	lam_resetfunc(BLKMPIALLTOALLV);
	return(MPI_SUCCESS);
}
