/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio	State University
 *	RBD/NJN
 *
 *	$Id: MPI.c,v 6.1 96/11/22 13:44:37 nevin Rel $
 *
 *	Function:	- F77 support
 */

#include <lam_config.h>

#include <stdlib.h>
#include <string.h>

#include <args.h>
#include <mpi.h>
#include <MPISYS.h>
#include <mpisys.h>
#include <terror.h>
#include <typical.h>

/*
 * public functions
 */
char			*lam_F2C_string();
int			lam_F2C_argv();
int			lam_F2C_comm();
int			lam_F2C_type();
void			lam_F2C_stat();
void			lam_C2F_string();
void			lam_C2F_stat();
int			lam_F_typealloc();
int			lam_F_typefind();
void			lam_F_maketype();
void			lam_F_typefree();
void			mpi_null_copy_fn_();
void			mpi_null_delete_fn_();

/*
 * public variables
 */
int			lam_F_comm_parent;
int			lam_F_maxtype =	0;
int			lam_F_maxpredef	= 0;
int			lam_F_ntypes = 0;
int			lam_F_typesize = 0;
void			**lam_F_types =	0;
char			*lam_F_bottom;
void			*lam_F_argvnull;
void			*lam_F_argvsnull;
void			*lam_F_errdontcare;

/*
 * These variables are used to link against MPI F77 constants which
 * correspond to addresses, e.g. MPI_BOTTOM, and are implemented via common
 * blocks.  They must have the same size and alignment constraints as the
 * corresponding F77 common blocks.
 */
#if SIZEOF_LONG_DOUBLE == 16
long double		mpi_bottom_[1];
long double		mpi_argv_null_[1];
long double		mpi_argvs_null_[1];
long double		mpi_errcodes_dontcare_[1];
#else
double			mpi_bottom_[2];
double			mpi_argv_null_[2];
double			mpi_argvs_null_[2];
double			mpi_errcodes_dontcare_[2];
#endif

/*
 * Null	functions.
 */
void mpi_null_copy_fn_() { }
void mpi_null_delete_fn_() { }

/*
 *	lam_F2C_comm
 *
 *	Function:	- converts Fortran communicator	handle to C handle
 *	Accepts:	- Fortran communicator handle
 *			- C communicator handle	(out)
 *	Returns:	- MPI_SUCCESS
 */
int
lam_F2C_comm(comm_fort,	pcomm_c)

int			comm_fort;
MPI_Comm		*pcomm_c;

{
	*pcomm_c = (MPI_Comm) GETHDL(comm_fort);
	return(MPI_SUCCESS);
}

/*
 *	lam_F2C_type
 *
 *	Accepts:	- Fortran datatype handle
 *			- C datatype handle (output)
 *	Returns:	- MPI_SUCCESS
 */
int
lam_F2C_type(type_fort,	ptype_c)

int			type_fort;
MPI_Datatype		*ptype_c;

{
	*ptype_c = (MPI_Datatype) GETHDL(type_fort);
	return(MPI_SUCCESS);
}

/*
 *	lam_F_typealloc
 *
 *	Function:	- allocate entry in type lookup	table
 *	Returns:	- table	index or LAMERROR
 */
int
lam_F_typealloc()

{
	int		i;
	unsigned	newsize;
	void		**p;
/*
 * If there is free space at the top of	the table, allocate it.
 */
	if (lam_F_maxtype < (lam_F_typesize - 1)) {
		++lam_F_ntypes;
		return(++lam_F_maxtype);
	}
/*
 * If there is room in the table, search for an	empty entry.
 */
	if (lam_F_ntypes < lam_F_typesize) {
		for (i = 0, p =	lam_F_types; i < lam_F_typesize; ++i, ++p) {
			if (*p == 0) {
				++lam_F_ntypes;
				return(i);
			}
		}

		return(LAMERROR);
	}
/*
 * Double the table size and allocate entry from its top.
 */
	newsize	= 2 * lam_F_typesize;
	p = (void **) realloc((char *) lam_F_types,
				(unsigned) newsize * sizeof(void *));
	if (p == 0) return(LAMERROR);

	memset((char *)	(p + lam_F_typesize), 0,
		lam_F_typesize * sizeof(void *));

	lam_F_types = p;
	lam_F_typesize = newsize;

	++lam_F_ntypes;
	return(++lam_F_maxtype);
}

/*
 *	lam_F_typefind
 *
 *	Function:	- locate given entry
 *	Accepts:	- pointer (entry)
 *	Returns:	- table	index or LAMERROR
 */
int
lam_F_typefind(pointer)

void			*pointer;

{
	int		i;
	int		rem;
	void		**p;

	rem = lam_F_ntypes;
	for (i = 0, p =	lam_F_types;
			(i < lam_F_typesize) &&	(rem > 0); ++i,	++p) {
		if (*p)	{
			--rem;
			if (*p == pointer) return(i);
		}
	}

	return(LAMERROR);
}

/*
 *	lam_F_typefree
 *
 *	Function:	- free entry in	type lookup table
 *	Accepts:	- table	index
 */
void
lam_F_typefree(index)

int			index;

{
	void		**p;

	if (index < 0) return;

	p = lam_F_types	+ index;
	*p = 0;
	--lam_F_ntypes;

	if (index == lam_F_maxtype) {
		for (--lam_F_maxtype, --p;
				lam_F_maxtype >= 0; --lam_F_maxtype, --p) {
			if (*p)	break;
		}
	}
}

/*
 *	lam_F_maketype
 *
 *	Function:	- create a type	mapping
 *			- set error code appropriately
 *	Accepts:	- new type (out)
 *			- error	code (inout)
 *			- new type handle
 */
void
lam_F_maketype(newtype,	errcode, handle)

int			*newtype;
int			*errcode;
void			*handle;

{
	if ((*errcode == MPI_SUCCESS) && (handle != 0))	{
		if ((*newtype =	lam_F_typealloc()) < 0)	{
			*errcode = lam_mkerr(MPI_ERR_OTHER, errno);
			*newtype = -1;
		} else {
			lam_F_types[*newtype] =	handle;
		}
	} else {
		*newtype = -1;
	}
}

/*
 *	lam_C2F_stat
 *
 *	Function:	- copy MPI_Status from C to F77
 *			- datatype entry not touched
 *	Accepts:	- C status
 *			- F77 status (out)
 */
void
lam_C2F_stat(cstat, f77stat)

MPI_Status		*cstat;
int			*f77stat;

{
	f77stat[0] = cstat->MPI_SOURCE;
	f77stat[1] = cstat->MPI_TAG;
	f77stat[2] = cstat->MPI_ERROR;
	f77stat[3] = cstat->st_count;
	f77stat[4] = cstat->st_nelem;
	f77stat[5] = cstat->st_length;
}

/*
 *	lam_F2C_stat
 *
 *	Function:	- copy MPI_Status from F77 to C
 *	Accepts:	- F77 status
 *			- C status (out)
 */
void
lam_F2C_stat(f77stat, cstat)

int			*f77stat;
MPI_Status		*cstat;

{
	cstat->MPI_SOURCE = f77stat[0];
	cstat->MPI_TAG = f77stat[1];
	cstat->MPI_ERROR = f77stat[2];
	cstat->st_count	= f77stat[3];
	cstat->st_nelem	= f77stat[4];
	cstat->st_length = f77stat[5];
	cstat->st_dtype	= (f77stat[6] <	0) ? MPI_DATATYPE_NULL :
				(MPI_Datatype) GETHDL(f77stat[6]);
}

/*
 *	lam_F2C_string
 * 
 *	Function:	- creates a C string from an F77 string
 *	Accepts:	- F77 string
 *			- string length
 *	Returns:	- C string or NULL
 */
char *
lam_F2C_string(str, len)

char			*str;
int			len;

{
	char		*cstr;
	char		*end;
	int		i;
/*
 * Leading and trailing	blanks are discarded.
 */
	end = str + len	- 1;
	
	for (i = 0; (i < len) && (*str == ' ');	i++, str++) ;

	if (i >= len) {
		len = 0;
	} else {
		for ( ;	(end > str) && (*end ==	' '); end--) ;

		len = end - str	+ 1;
	}
/*
 * Allocate space for the C string.
 */
	if ((cstr = malloc(len + 1)) ==	0) {
		return((char *)	0);
	}
/*
 * Copy	F77 string into	C string and NULL terminate it.
 */
	if (len	> 0) {
		strncpy(cstr, str, len);
	}
	cstr[len] = 0;

	return(cstr);
}

/*
 *	lam_C2F_string
 * 
 *	Function:	- copy a C string into a F77 string
 *	Accepts:	- C string
 *			- F77 string (out)
 *			- F77 string length
 */
void
lam_C2F_string(cstr, f77str, len)

char			*cstr;
char			*f77str;
int			len;

{
	int		i;
	
	strncpy(f77str,	cstr, len);

	for (i = strlen(cstr); i < len;	i++) {
		f77str[i] = ' ';
	}
}

/*
 *	lam_F2C_argv
 * 
 *	Function:	- creates a C argument vector from an F77 array 
 *			  of strings (terminated by a blank string)
 *	Accepts:	- F77 array of strings
 *			- length of strings
 *			- C argument vector (out)
 *	Returns:	- 0 or LAMERROR
 */
int
lam_F2C_argv(array, len, argv)

char			*array;
int			len;
char			***argv;

{
	int		argc = 0;
	char		*cstr;

	*argv = 0;
	
	while (1) {
		cstr = lam_F2C_string(array, len);
		if (cstr == 0) {
			argvfree(*argv);
			return(LAMERROR);
		}

		if (*cstr == 0) break;

		if (argvadd(&argc, argv, cstr)) {
			argvfree(*argv);
			return(LAMERROR);
		}

		free(cstr);
		array += len;
	}

	return(0);
}
