/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	NJN
 *
 *	$Id: shm.sol.c,v 6.1 96/11/22 13:35:14 nevin Rel $
 *
 *	Function:	- Solaris shared memory low-level routines.
 *			- Uses Solaris semaphores for locking.
 */

#include <lam_config.h>

#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>
#include <terror.h>
#include <typical.h>
#include <t_types.h>

/*
 * global functions
 */
int			_shm_serverinit();
int			_shm_clientinit();
int			_shm_cleanup();

/*
 * external functions
 */
extern void		_sysv_slop_add();
extern void		_sysv_slop_del();
extern void		_sysv_slop_clean();


/*
 *	_shm_clientinit
 *
 *	Function:	- client side shared memory initialization
 *	Accepts:	- process
 *			- LAM msg to fill with info for client
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_clientinit(proc, msg)

struct c2c_proc		*proc;
struct nmsg		*msg;

{
	proc->cp_shm = (int) msg->nh_data[1];
/*
 * Attach the shared memory segment.
 */
	proc->cp_shmbuf = (char *) shmat(proc->cp_shm, (char *) 0, 0);
	if (proc->cp_shmbuf == (char *) -1) return(LAMERROR);
/*
 * Mark it for deletion. It will be removed once all attached
 * processes detach.
 */
	if (shmctl(proc->cp_shm, IPC_RMID, (struct shmid_ds *) 0) < 0) {
		return(LAMERROR);
	}

	proc->cp_shmin = proc->cp_shmbuf + 2 * CACHELINESIZE;
	proc->cp_shmout = proc->cp_shmbuf + SHMBUFSIZE + 2 * CACHELINESIZE;
/*
 * Set the locks.
 */
	proc->cp_iwtlock = (sema_t *) proc->cp_shmbuf;
	proc->cp_irdlock = (sema_t *) (proc->cp_shmbuf + CACHELINESIZE);
	proc->cp_owtlock = (sema_t *) (proc->cp_shmbuf + SHMBUFSIZE); 
	proc->cp_ordlock = (sema_t *) (proc->cp_shmbuf
					+ SHMBUFSIZE + CACHELINESIZE);

	return(0);
}

/*
 *	_shm_serverinit
 *
 *	Function:	- server side shared memory initialization
 *	Accepts:	- process
 *			- LAM msg to fill with info for client
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_serverinit(proc, msg)

struct c2c_proc		*proc;
struct nmsg		*msg;

{
	int		shmid;
	int		err;
/*
 * Create the shared memory segment.
 */
	shmid = shmget(IPC_PRIVATE, SHMBLKSIZE, 0600 | IPC_CREAT);
	if (shmid < 0) return(LAMERROR);
    
	proc->cp_shmbuf = (char *) shmat(shmid, (char *) 0, 0);
	if (proc->cp_shmbuf == (char *) -1) {
		shmctl(shmid, IPC_RMID, (struct shmid_ds *) 0);
		return(LAMERROR);
	}

	proc->cp_shmout = proc->cp_shmbuf + 2 * CACHELINESIZE;
	proc->cp_shmin = proc->cp_shmbuf + SHMBUFSIZE + 2 * CACHELINESIZE;
	proc->cp_shm = shmid;
/*
 * Initialize the locks.
 */
	proc->cp_owtlock = (sema_t *) proc->cp_shmbuf;
	proc->cp_ordlock = (sema_t *) (proc->cp_shmbuf + CACHELINESIZE);
	proc->cp_iwtlock = (sema_t *) (proc->cp_shmbuf + SHMBUFSIZE);
	proc->cp_irdlock = (sema_t *) (proc->cp_shmbuf
					+ SHMBUFSIZE + CACHELINESIZE);

	err = sema_init(proc->cp_owtlock, 1, USYNC_PROCESS, (void *) 0);
	if (err) { errno = err; return(LAMERROR); }
	err = sema_init(proc->cp_iwtlock, 1, USYNC_PROCESS, (void *) 0);
	if (err) { errno = err; return(LAMERROR); }
	err = sema_init(proc->cp_ordlock, 0, USYNC_PROCESS, (void *) 0);
	if (err) { errno = err; return(LAMERROR); }
	err = sema_init(proc->cp_irdlock, 0, USYNC_PROCESS, (void *) 0);
	if (err) { errno = err; return(LAMERROR); }
/*
 * Set information to pass to client.
 */
	msg->nh_data[1] = (int4) shmid;
/*
 * Register id for cleanup.
 */
	_sysv_slop_add(1, 'm', shmid);

	return(0);
}

/*
 *	_shm_cleanup
 *
 *	Function:	- clean up a process's shared memory structures
 *	Accepts:	- c2c process
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_cleanup(proc)

struct c2c_proc		*proc;

{
	if (shmdt(proc->cp_shmbuf) < 0) {
		return(LAMERROR);
	}
	_sysv_slop_del('m', proc->cp_shm);
	return(0);
}

/*
 *      _shm_mop_up
 *
 *      Function:       - complete mop up of persistent state
 */
void
_shm_mop_up()

{
	_sysv_slop_clean();
}
