# Hey emacs! This is a -*- Perl -*- script!
# util.pl -- Perl utility functions for lintian

# Copyright (C) 1998 by Chistian Schwarz
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, you can find it on the World Wide
# Web at http://www.gnu.org/copyleft/gpl.html, or write to the Free
# Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
# MA 02111-1307, USA.

# general function to read dpkg control files
# this function can parse output of `dpkg-deb -f', .dsc, 
# and .changes files (and probably all similar formats)
# arguments:
#    $file
# output:
#    list of hashes
#    (a hash contains one sections,
#    keys in hash are lower case letters of control fields)
sub read_dpkg_control {
  my ($file) = @_;

  my @data;
  my $cur_section = 0;
  my $open_section = 0;
  my $last_tag;

  open(CONTROL,$file) or fail("cannot open control file $file for reading: $!");

  while (<CONTROL>) {
    chop;

    # tabs at the beginning are illegal, but handle them anyways
    s/^\t/ \t/o;

    # empty line?
    if (/^\s*$/) {
      if ($open_section) {
	# end of current section
	$cur_section++;
	$open_section = 0;
      }
    }
    # pgp sig?
    elsif (/^-----BEGIN PGP SIGNATURE/) {
      # skip until end of signature
      while (<CONTROL>) {
	last if /^-----END PGP SIGNATURE/o;
      }
    }
    # other pgp control?
    elsif (/^-----BEGIN PGP/) {
      # ignore
    }
    # new empty field?
    elsif (/^(\S+):\s*$/o) {
      $open_section = 1;

      my ($tag) = (lc $1);
      $data[$cur_section]->{$tag} = '';

      $last_tag = $tag;
    }
    # new field?
    elsif (/^(\S+): (.*)$/o) {
      $open_section = 1;

      my ($tag,$value) = (lc $1,$2);
      $data[$cur_section]->{$tag} = $value;

      $last_tag = $tag;
    }
    # continued field?
    elsif (/^ (.*)$/o) {
      $open_section or fail("syntax error in control file: $_");

      $data[$cur_section]->{$last_tag} .= "\n".$1;
    }
  }

  close(CONTROL) or fail("broken input pipe for control file $file: $!");

  return @data;
}

sub get_deb_info {
  my ($file) = @_;

  # `dpkg-deb -f $file' is very slow. Instead, we use ar and tar.

  my @data = read_dpkg_control("ar p \"$file\" control.tar.gz | tar xfzO - control |");

  return $data[0];
}

sub get_dsc_info {
  my ($file) = @_;

  my @data = read_dpkg_control($file);

  return $data[0];
}

# ------------------------

sub fail {
  if ($_[0]) {
    print STDERR "error: $_[0]\n";
  } elsif ($!) {
    print STDERR "error: $!\n";
  } else {
    print STDERR "error.\n";
  }
  print STDERR "$_[1]\n" if $_[1];
  exit 1;
}

1;
