/* vi:set ts=8 sts=0 sw=8:
 * $Id: prefs.c,v 1.22 1998/10/18 03:56:54 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include <values.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#ifndef GTK_HAVE_FEATURES_1_1_0
#include "gtkfontsel.h"
#endif
#include "main.h"
#include "menu.h"
#include "file.h"
#include "dialog.h"
#include "win.h"
#include "msgbar.h"
#include "msgbox.h"
#include "prefs.h"


/*** global variables ***/
char *		appdir;		/* $HOME/PREFS_DIR/ */
GtkPositionType	tabpos;		/* doc tab position */
guint		autosave;	/* minutes for autosave (0 = disables) */
char *		printcmd;	/* print cmd. 'lpr' */
char *		tmpdir;		/* temp directory. '/var/tmp' */
long		options;	/* bit flag options.  see prefs.h */
ubyte		maxrecent;	/* max number of recently opened docs */
guint		msgbox_max_msg;	/* max num of msgs in msg box */
guint		msgbox_per_del;	/* percentage of msgs to delete in msg box */
int		win_height;	/* window height */
int		win_width;	/* window width */
int		win_xpos;	/* window xposition */
int		win_ypos;	/* window yposition */
GdkColor	text_fg_color;	/* GDK-styled text foreground color */
GdkColor	text_bg_color;	/* GDK-styled text background color */
GdkFont *	text_font;	/* text font */



/*** local definitions ***/
#define COL_VAL_GDK	65535.0		/* GdkColor is 16-bit color */
#define PREFS_NAME_LEN	32		/* max len for pref name in rc file */
#define PREFS_DIR	".gnp/"		/* should be in $HOME */
#define APPRC		"apprc"		/* stores app-specific prefs */
#define APPGTKRC	"appgtkrc"	/* stores app-specific GTK prefs */
#define GTKRC		"gtkrc"		/* stores gtk-related prefs */
#define MAX_FONT_LEN	MAXPATH
#define MAX_RGB_STR	32
#define DEFAULT_FONT	"-adobe-courier-medium-r-normal-*-*-120-*-*-m-*-iso8859-1"

typedef enum {
	PrefGeneric,	/* generic */
	PrefBool,	/* its boolean, but the variable must be a "long" */
	PrefString,	/* (char *) */
	PrefInt,	/* int or uint */
	PrefFloat,	/* float */
	PrefByte,	/* byte */
	PrefShort	/* short */
} prefs_type_t;

typedef struct {
	char *		name;	/* pref name */
	prefs_type_t	type;	/* pref type */
	void		(*f)();	/* callback used when updating prefs */
	void *		cbdata;	/* callback data */
	void * 		addr;	/* address of data value */
	long		min;	/* min size/len/value */
	long		max;	/* max size/length/value */
	gulong		val;	/* multi-purpose value.  currently used for */
				/* max string len and boolean bit flags */
} prefs_t;

/* used by color selection */
typedef enum { Foreground, Background } which_text_t;
typedef struct {
	GtkWidget *	cs;
	which_text_t	which;
} prefs_color_sel_t;


/*** local variables ***/
static bool_t		pre102 = FALSE;	/* using gnp+ before v1.0.2 */
static GtkPositionType	prefs_tabpos;	/* prefs tab position */
static char *		text_fg_str;	/* fg color (rgb string) */
static char *		text_bg_str;	/* bg color (rgb string) */
static char *		text_font_str;	/* text font string */

static char *	appgtkrc = NULL;	/* app specific gtk rc */
static void	appgtk_rc_update(gpointer data);	/* need fwd ref */

/*
 * to add a new configurable preference option:
 *
 * 1. declare a global variable up in the "global variables" section.  make
 * sure that it's one of the following types: int, bool, float, byte, short,
 * or "char *".
 *
 * 2. add a corresponding entry below into the variable 'app_prefs'.  your
 * best bet is to just cut and paste an existing entry and see how it works.
 * keep the entries in alphabetical order.
 *
 * 3. create and add the necessary widgets into the Preferences window popup
 * dialog.  the most commonly used ones are a button and a text entry.  be
 * sure to add these widgets into the wgtopt_list.  see one of the existing
 * routines as an example (e.g., prefs_frame_toolbar_create()).
 *
 *
 * i know this setup may not be easiest way for another developer to use.  for
 * example, ideally one would like to do something such as the following:
 *
 *	prefs_set_data("doc_tab_position", "top");
 *	prefs_get_data("print_command", buf);
 *
 * however, the big downside to this is that when getting data, you need
 * routines to translate the values *ANYWAY*.  in other words, every time you
 * successfully get data, you still have to determing *what* that data is and
 * *how* it affects preferences.  for instance, if data is supposed to
 * represent an integer, you'd have to then convert it to an integer and
 * determine what to do.
 *
 * so although the prefs_set_data() and prefs_get_data() looks nice on the
 * outside, you still have to write a whole bunch of code to figure out what
 * do with it once you've gotten it.  the tabular method that i'm using
 * effectively puts the "what", "how", and "where" into the table itself.
 * so all you have to do is specify it ONCE, and the routines that manipulate
 * the table will take care of it so you don't have to.
 *
 * the prefs_set_data() and prefs_get_data() technique may work well if it
 * were provided as a library (e.g., Gnome does this), but since this part of
 * the code is not intended to be a library, (i think) the tabular method is
 * better.
 */
static prefs_t		app_prefs[] = {
#ifdef USE_AUTO_INDENT
	{ "use_auto_indent",
		PrefBool,
		NULL_CALLBACK, NULL,
		&options,
		0, 1, AUTO_INDENT
	},
#endif
	{ "autosave_timeout",
		PrefInt,
		NULL_CALLBACK, NULL,
		&autosave,
		0, 1440, 0
	},
	{ "doc_tab_position",
		PrefInt,
		win_redraw_doc_tab, NULL,
		&tabpos,
		0, 3, 0
	},
	{ "max_num_recent",
		PrefByte,
		NULL_CALLBACK, NULL,
		&maxrecent,
		0, 255, 0
	},
	{ "msgbox_max_msg",
		PrefInt,
		NULL_CALLBACK, NULL,
		&msgbox_max_msg,
		0, 32768, 0
	},
	{ "msgbox_per_del",
		PrefInt,
		NULL_CALLBACK, NULL,
		&msgbox_per_del,
		0, 100, 0
	},
	{ "prefs_tab_position",
		PrefInt,
		NULL_CALLBACK, NULL,
		&prefs_tabpos,
		0, 3, 0
	},
	{ "print_command",
		PrefString,
		NULL_CALLBACK, NULL,
		&printcmd,
		0, 0, MAXPATH
	},
	{ "save_win_pos",
		PrefBool,
		NULL_CALLBACK, NULL,
		&options,
		0, 1, SAVE_WIN_POS
	},
	{ "save_win_height",
		PrefBool,
		NULL_CALLBACK, NULL,
		&options,
		0, 1, SAVE_WIN_HEIGHT
	},
	{ "save_win_width",
		PrefBool,
		NULL_CALLBACK, NULL,
		&options,
		0, 1, SAVE_WIN_WIDTH
	},
	{ "show_doc_tabs",
		PrefBool,
		NULL_CALLBACK, NULL,
		&options,
		0, 1, SHOW_TABS
	},
#ifdef WANT_PROJECT
	{ "show_project_bar",
		PrefBool,
		win_redraw_prjbar, NULL,
		&options,
		0, 1, SHOW_PRJBAR
	},
#endif	/* WANT_PROJECT */
	{ "show_message_bar",
		PrefBool,
		win_redraw_msgbar, NULL,
		&options,
		0, 1, SHOW_MSGBAR
	},
	{ "show_toolbar",
		PrefBool,
		win_redraw_toolbar, NULL,
		&options,
		0, 1, SHOW_TOOLBAR
	},
	{ "show_tooltips",
		PrefBool,
		NULL_CALLBACK, NULL,
		&options,
		0, 1, SHOW_TOOLTIPS
	},
	{ "text_bg_color",
		PrefString,
		NULL_CALLBACK, NULL,
		&text_bg_str,
		0, 1, MAX_RGB_STR
	},
	{ "text_fg_color",
		PrefString,
		NULL_CALLBACK, NULL,
		&text_fg_str,
		0, 0, MAX_RGB_STR
	},
	{ "text_font",
		PrefString,
		NULL_CALLBACK, NULL,
		&text_font_str,
		0, 0, MAX_FONT_LEN
	},
	{ "tmp_directory",
		PrefString,
		NULL_CALLBACK, NULL,
		&tmpdir,
		0, 0, MAXPATH
	},
	{ "toolbar_piconly",
		PrefBool,
		win_redraw_toolbar, NULL,
		&options,
		0, 1, PIC_ONLY_TOOLBAR
	},
	{ "toolbar_pictext",
		PrefBool,
		win_redraw_toolbar, NULL,
		&options,
		0, 1, PIC_TEXT_TOOLBAR
	},
	{ "toolbar_textonly",
		PrefBool,
		win_redraw_toolbar, NULL,
		&options,
		0, 1, TEXT_ONLY_TOOLBAR
	},
	{ "use_msgbox",
		PrefBool,
		NULL_CALLBACK, NULL,
		&options,
		0, 1, USE_MSGBOX
	},
	{ "use_wordwrap",
		PrefBool,
		NULL_CALLBACK, NULL,
		&options,
		0, 1, USE_WORDWRAP
	},
	{ "win_height",
		PrefInt,
		NULL_CALLBACK, NULL,
		&win_height,
		-1, 8191, 0
	},
	{ "win_width",
		PrefInt,
		NULL_CALLBACK, NULL,
		&win_width,
		-1, 8191, 0
	},
	{ "win_xpos",
		PrefInt,
		NULL_CALLBACK, NULL,
		&win_xpos,
		-1, 8191, 0
	},
	{ "win_ypos",
		PrefInt,
		NULL_CALLBACK, NULL,
		&win_ypos,
		-1, 8191, 0
	},
	{ NULL, PrefBool, NULL_CALLBACK, NULL, NULL, 0, 0, 0 }
};

/*
 * in order for the preferences to take effect and be saved *after* the user
 * clicked "Ok" (as opposed to the preferences taking effect immediately each
 * time the user selects an option in the prefs window), we build a list of
 * items.  each item contains the following:
 *
 *	1. a pointer into the correct entry into the app_prefs table.  since
 *	once the user selects "Ok", we're effectively saving the new options
 *	anyway, so let's just reuse the code used for saving the prefs.
 *
 *	2. the widget representing the options (e.g., a check box).  this
 *	is used so we know what state it's in.  or rather, what the user
 *	has selected.  the widget is implied by what the preference is.
 *	if it's PrefBool, it must be a button widget.  all others are most
 *	likely a text entry type.
 *
 * once the user hits the "Save Preferences" button, we then scan this list,
 * and depending on the state of the widget/button/etc, we can look in the
 * app_prefs table to see what to set and to what value.
 */
typedef enum {
	GenericType,
	EntryType,
	ButtonType,
	ComboType
} wgttype_t;

typedef struct {
	prefs_t *pap;		/* pointer into app_prefs table */
	GtkWidget *wgt;		/* the widget */
	wgttype_t type;		/* type of widget */
	void *data;		/* data to use */
} wgtopt_t;
static GList *wgtopt_list = NULL;	/* list of wgtopt_t's */

/* other global variables within this file */
static char *		apprc     = NULL;	/* full path to apprc */
static char *		gtkrc     = NULL;	/* full path to gtkrc */
static GtkWidget *	prefs_win = NULL;	/* preferences popup window */
static GtkWidget *	prefs_nb = NULL;	/* notebook for prefs window */
static GList *		tabpos_list = NULL;	/* text for combo widget */


/*** local function prototypes ***/
static char *		prefs_tf(long opts, long mask);
static void		prefs_read(FILE *fp);
static void		prefs_bool_set(prefs_t *pap, char *data);
static void		prefs_byte_set(prefs_t *pap, char *data);
static void		prefs_float_set(prefs_t *pap, char *data);
static void		prefs_int_set(prefs_t *pap, char *data);
static void		prefs_string_set(prefs_t *pap, char *data);
static void		prefs_dialog_create(win_t *w);
static void		prefs_win_destroy(GtkWidget *wgt, gpointer cbdata);
static void		prefs_save_cb(GtkWidget *wgt, gpointer cbdata);
static void		prefs_wgtopt_list_add(
				GtkWidget *, wgttype_t, void *, char *);
static void		prefs_wgtopt_list_free(void);
static void		prefs_page_document_create(GtkWidget *nb);
static void		prefs_page_misc_create(GtkWidget *nb);
static void		prefs_frame_toolbar_create(GtkWidget *parent_vbox);
static void		prefs_frame_doctab_create(GtkWidget *parent_vbox);
static void		prefs_page_appearance_create(GtkWidget *nb);
static void		prefs_frame_msgbar_create(GtkWidget *parent);
static void		prefs_page_window_create(GtkWidget *nb);
static void		prefs_page_fonts_colors(GtkWidget *nb);
static void		prefs_frame_msgbox_create(GtkWidget *parent);
static void		prefs_update(void);
static void		prefs_tab_pos_change(void);
static void		prefs_text_fg_cb(GtkWidget *wgt, gpointer cbdata);
static void		prefs_text_bg_cb(GtkWidget *wgt, gpointer cbdata);
static GtkWidget *	prefs_check_button_with_label_create(
				GtkWidget *, char *, long , char *);
static GtkWidget *	prefs_entry_with_label_create(
				GtkWidget *, char *, char *, char *);
static GtkWidget *	prefs_spin_button_with_label_create(
				GtkWidget *, char *, float , float , float ,
				float , float , float , char *);
static GtkWidget *	prefs_radio_button_with_label_create(
				GtkWidget *, GtkWidget *, char *, long, char *);
static GtkWidget *	prefs_pulldown_with_label_create(
				GtkWidget *, char *, GList *, char *);
static void		prefs_color_sel_changed(GtkWidget *, gpointer cbdata);
static void		prefs_color_sel_ok(GtkWidget *wgt, gpointer cbdata);
static void		prefs_color_sel_create(which_text_t which);
static void		prefs_color_sel_close(GtkWidget *, gpointer cbdata);
static void		prefs_color_sel_destroy(GtkWidget *, gpointer cbdata);
static void		prefs_font_sel_cb(GtkWidget *wgt, gpointer cbdata);
static void		prefs_font_sel_ok(GtkWidget *wgt, gpointer cbdata);
static void		prefs_font_sel_cancel(GtkWidget *wgt, gpointer cbdata);


/*** global function definitions ***/
/*
 * PUBLIC: prefs_cb
 *
 * callback invoked from menu to bring up preferences window.
 */
void
prefs_cb(GtkWidget *wtg, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (prefs_win != NULL) {
		gdk_window_raise(prefs_win->window);
		return;
	}

	prefs_dialog_create(w);
} /* prefs_cb */


/*
 * PUBLIC: prefs_init
 *
 * initializes application wide preferences.  called from main().
 * TODO: specify/use system-wide app-defaults file
 */
void
prefs_init(void)
{
	char *homedir, *buf;
	FILE *fp;
	int num;

	/* initialize to default values */
	appdir         = NULL;
	printcmd       = g_strdup("lpr %s");
	tmpdir         = g_strdup("/var/tmp");
	tabpos         = GTK_POS_TOP;
	autosave       = 5;
	options        = DEFAULT_OPTIONS;
	maxrecent      = 4;
	msgbox_max_msg = 200;
	msgbox_per_del = 50;
	win_height     = 470;
	win_width      = -1;
	win_xpos       = -1;
	win_ypos       = -1;
	text_fg_str    = g_strdup("0 0 0");	/* always black */
	text_bg_str    = g_strdup("65535 65535 65535");	/* 16-bit white */
	text_font_str  = g_strdup(DEFAULT_FONT);
	text_font      = NULL;
	prefs_tabpos   = GTK_POS_TOP;

	if ((homedir = getenv("HOME")) == NULL)
		return;

	appdir = (char *)g_malloc(strlen(homedir) + strlen(PREFS_DIR) + 3);
	sprintf(appdir, "%s/%s/", homedir, PREFS_DIR);
	num = mkdir(appdir, 0777);
	if (num == -1 && errno != ESUCCESS && errno != EEXIST) {
		buf = (char *)g_malloc(strlen(appdir) + 32);
		sprintf(buf, "prefs_init: could not create '%s'", appdir);
		perror(buf);
		g_free(buf);
		g_free(appdir);
		return;
	} 
#if NOT_QUITE
	else if (errno == ESUCCESS) {
		/* hmm, this doesn't work the way i intended to at this
		 * this point because there's no top-level dialog window
		 * associated with it.  consequently, this dialog box just
		 * pops up where-ever it wants.
		 */
		(void)do_dialog_ok(
			"gnotepad+ Initialized", " Created $HOME/.app/ ");
	}
#endif

	/* parse gtkrc specific attributes */
	if (gtkrc)
		g_free(gtkrc);
	num = strlen(appdir) + strlen(GTKRC) + 1;
	gtkrc = (char *)g_malloc(num);
	g_snprintf(gtkrc, num, "%s%s", appdir, GTKRC);
	gtk_rc_parse(gtkrc);

	/* now read apprc to get application specific stuff */
	num = strlen(appdir) + strlen(APPRC) + 1;
	apprc = (char *)g_malloc(num);
	g_snprintf(apprc, num, "%s%s", appdir, APPRC);
	
	if ((fp = fopen(apprc, "r")) == NULL) {
		if (errno == ENOENT)	/* apprc doesn't exist, so create it */
			prefs_save();	/* using default values */
		else {
			buf = (char *)g_malloc(strlen(apprc) + 32);
			sprintf(buf, "prefs_init: could not open '%s'", apprc);
			perror(buf);
			g_free(buf);
		}
	}

	prefs_read(fp);
	prefs_text_color_update();
	text_font = (text_font_str) ? gdk_font_load(text_font_str) : NULL;

	/*
	 * now that we have the application specific stuff, create a gtk
	 * formattted rc file which contains app settings that may override
	 * anything in gtkrc, and finally, use the settings specified in this
	 * new file.
	 */
	num = strlen(appdir) + strlen(APPGTKRC) + 1;
	if (appgtkrc)
		g_free(appgtkrc);
	appgtkrc = (char *)g_malloc(num);
	g_snprintf(appgtkrc, num, "%s%s", appdir, APPGTKRC);
	appgtk_rc_update(appgtkrc);
} /* prefs_init */


/*
 * PUBLIC: prefs_save
 *
 * save application-wide preferences.  basically, scan the app_prefs table and
 * write out the values corresponding to each entry.
 */
void
prefs_save(void)
{
	FILE *fp;
	char *buf;
	prefs_t *pap;

	if ((fp = fopen(apprc, "w")) == NULL) {
		buf = (char *)g_malloc(strlen(apprc) + 32);
		sprintf(buf, "prefs_save: could not open '%s'", apprc);
		perror(buf);
		g_free(buf);
		return;
	}

	(void)file_lock(apprc, fp, TRUE, FALSE, TRUE);
	fprintf(fp,
		"# %s %s initialization file.  "
		"This file is automatically generated.\n"
		"# Data in this file is in no particular order.\n"
		"# Although you could edit this by hand, "
		"it is NOT recommended!\n",
		APP_NAME, APP_VERSION);

	buf = (char *)g_malloc(PREFS_NAME_LEN + MAXPATH);
	buf[0] = '\0';
	pap = app_prefs;
	while (pap->name) {
		if (pap->addr == NULL) {
			printf("prefs_save: '%s' has NULL addr\n", pap->name);
			pap++;
			continue;
		}

		switch (pap->type) {
		case PrefBool:
			g_snprintf(buf, PREFS_NAME_LEN + MAXPATH,
				"%s = %s",
				pap->name,
				prefs_tf(*(long *)(pap->addr), pap->val));
			break;
		case PrefString:
			g_snprintf(buf, PREFS_NAME_LEN + MAXPATH,
				"%s = %s",
				pap->name, *(char **)(pap->addr));
			break;
		case PrefInt:
			g_snprintf(buf, PREFS_NAME_LEN + MAXPATH,
				"%s = %d",
				pap->name, *(int *)(pap->addr));
			break;
		case PrefByte:
			g_snprintf(buf, PREFS_NAME_LEN + MAXPATH,
				"%s = %d",
				pap->name, *(byte *)(pap->addr));
			break;
		case PrefFloat:
			g_snprintf(buf, PREFS_NAME_LEN + MAXPATH,
				"%s = %f",
				pap->name, *(float *)(pap->addr));
			break;
		default:
			printf("prefs_save: ignoring '%s' (unknown type=%d)\n",
				pap->name, pap->type);
			break;
		} /* switch pap->type */

		if (buf[0] != '\0') {
			fprintf(fp, "%s\n", buf);
			GNPDBG_PREFS(("prefs_save: wrote = '%s'\n", buf));
			buf[0] = '\0';
		}

		pap++;
	} /* while pap->name */

	(void)file_unlock(apprc, fp);
	fclose(fp);
	g_free(buf);
} /* prefs_save */


/*
 * PRIVATE: prefs_read
 *
 * actually read the prefs file.  each line's first token is looked up in the
 * app_prefs table.  if it's not in there, then that line is ignored.
 */
static void
prefs_read(FILE *fp)
{
	char *buf, *data, *tok;
	prefs_t *pap;
	bool_t done;

	if (fp == NULL) {
		if ((fp = fopen(apprc, "r")) == NULL) {
			buf = (char *)g_malloc(strlen(apprc) + 32);
			sprintf(buf, "prefs_save: could not open '%s'", apprc);
			perror(buf);
			g_free(buf);
			return;
		}
	}

	(void)file_lock(apprc, fp, FALSE, FALSE, TRUE);
	buf = (char *)g_malloc(PREFS_NAME_LEN + MAXPATH);
	if (fgets(buf, PREFS_NAME_LEN + MAXPATH, fp)) {
		char *p;
		/* XXX - horrible hack for versioning */
		if ((p = strstr(buf, "gnotepad+ 1.0"))) {
			p += strlen("gnotepad+ 1.0") + 1;
			if (!isdigit((int)(*(p+1))))
				*(p+1) = '\0';
			else
				*(p+2) = '\0';
			if (atoi(p) < 2)
				pre102 = TRUE;
		}
	}

	while (fgets(buf, PREFS_NAME_LEN + MAXPATH, fp)) {
		buf[strlen(buf) - 1] = '\0';	/* remove \n */
		if ((tok = strtok(buf, "=")) == NULL)
			continue;

		/* trim any trailing spaces */
		while (isspace((int)tok[strlen(tok) - 1]))
			tok[strlen(tok) - 1] = '\0';

		data = tok + strlen(tok) + 2;
		while (isspace((int)(*data)))	/* skip to actual value */
			data++;

		GNPDBG_PREFS(("prefs_read: tok  = '%s'\n", tok));
		GNPDBG_PREFS(("prefs_read: data = '%s'\n", data));

		/*
		 * TODO - eventually, if app_prefs gets big, this should be a
		 * binary search instead of a linear traversal.
		 */
		done = FALSE;
		pap = app_prefs;
		while (pap->name) {
			if (pap->addr == NULL) {
				printf("prefs_read: '%s' has NULL addr\n",
					pap->name);
				pap++;
				continue;
			}

			if (strncmp(tok, pap->name, strlen(pap->name)) == 0) {
				switch (pap->type) {
				case PrefBool:
					prefs_bool_set(pap, data);
					break;
				case PrefString:
					prefs_string_set(pap, data);
					break;
				case PrefInt:
					prefs_int_set(pap, data);
					break;
				case PrefByte:
					prefs_byte_set(pap, data);
					break;
				case PrefFloat:
					prefs_float_set(pap, data);
					break;
				default:
					printf("prefs_read: ignoring '%s' "
						"(unknown type=%d)\n",
						pap->name, pap->type);
					break;
				}
				done = TRUE;
			} /* if attr match */

			pap++;

		} /* while pap->name */
	} /* while fgets */

	(void)file_unlock(apprc, fp);
	fclose(fp);
	g_free(buf);
} /* prefs_read */


/*
 * PRIVATE: appgtk_rc_update
 *
 * creates an application specific rc file in gtk format.
 */
static void
appgtk_rc_update(gpointer data)
{
	char *appgtkrc = (char *)data;
	FILE *fp;
	char *buf;

	if ((fp = fopen(appgtkrc, "w")) == NULL) {
		buf = (char *)g_malloc(strlen(appgtkrc) + 64);
		sprintf(buf, "appgtk_rc_update: could not open '%s'", appgtkrc);
		perror(buf);
		g_free(buf);
		return;
	}

	(void)file_lock(appgtkrc, fp, TRUE, FALSE, TRUE);

	fprintf(fp, "style \"text\"\n{\n\tfont = \"%s\"\n", text_font_str);
	fprintf(fp, "\tbase[NORMAL] = { %0.1f, %0.1f, %0.1f }\n", 
		text_bg_color.red   / COL_VAL_GDK,
		text_bg_color.green / COL_VAL_GDK,
		text_bg_color.blue  / COL_VAL_GDK);
	fprintf(fp, "\ttext[NORMAL] = { %0.1f, %0.1f, %0.1f }\n", 
		text_fg_color.red   / COL_VAL_GDK,
		text_fg_color.green / COL_VAL_GDK,
		text_fg_color.blue  / COL_VAL_GDK);
	fprintf(fp, "}\nwidget_class \"*GtkText\" style \"text\"\n");

	(void)file_unlock(appgtkrc, fp);
	fclose(fp);
	gtk_rc_parse(appgtkrc);
} /* appgtk_rc_update */


/*
 * PRIVATE: prefs_string_set
 *
 * set a preference whose setting/value is a "string"
 */
static void
prefs_string_set(prefs_t *pap, char *data)
{
	int len;
	char **string = (char **)(pap->addr);

	if (*string)
		g_free(*string);
	*string = NULL;

	if (data == NULL)
		return;

	if (data[0] == '\0')
		return;

	len = MIN(strlen(data), pap->val);
	*string = (char *)g_malloc(len + 1);
	strncpy(*string, data, len);
	(*string)[len] = '\0';
} /* prefs_string_set */


/*
 * PRIVATE: prefs_byte_set
 *
 * set a preference whose setting/value is an integer
 */
static void
prefs_byte_set(prefs_t *pap, char *data)
{
	byte *num = (byte *)(pap->addr);

	*num = (byte)atoi(data);
	if (*num > pap->max)
		*num = (byte)(pap->max);
	if (*num < pap->min)
		*num = (byte)(pap->min);
} /* prefs_byte_set */


/*
 * PRIVATE: prefs_float_set
 *
 * set a preference whose setting/value is a float
 */
static void
prefs_float_set(prefs_t *pap, char *data)
{
	float *num = (float *)(pap->addr);

	sscanf(data, "%f", num);
	if (*num > pap->max)
		*num = (float)(pap->max);
	if (*num < pap->min)
		*num = (float)(pap->min);
} /* prefs_float_set */


/*
 * PRIVATE: prefs_int_set
 *
 * set a preference whose setting/value is an integer
 */
static void
prefs_int_set(prefs_t *pap, char *data)
{
	int *num = (int *)(pap->addr);

	*num = atoi(data);
	if (*num > pap->max)
		*num = (int)(pap->max);
	if (*num < pap->min)
		*num = (int)(pap->min);
} /* prefs_int_set */


/*
 * PRIVATE: prefs_bool_set
 *
 * set a preference whose setting/value is a boolean.
 * NOTE!!  the convention here is that booleans use bitmasks to toggle their
 * settings.  the variable itself must be stored in a "long".
 */
static void
prefs_bool_set(prefs_t *pap, char *data)
{
	long *opts;

	opts = (long *)(pap->addr);

	if (tolower((int)data[0]) == 't' || data[0] == '1') {
		*opts |= pap->val;
	} else if (tolower((int)data[0]) == 'f' || data[0] == '0') {
		*opts &= ~pap->val;
	}
} /* prefs_bool_set */


/*
 * PRIVATE: prefs_tf
 *
 * returns the string 'true' or 'false'
 */
static char *
prefs_tf(long opts, long mask)
{
	return (opts & mask) ? "true" : "false";
} /* prefs_tf */


/*
 * PRIVATE: prefs_dialog_create
 *
 * creates the preferences window.
 *
 * 1. Appearance
 * 2. Document
 * 3. Window
 * 4. Misc
 */
static void
prefs_dialog_create(win_t *w)
{
	GtkWidget *vbox, *hbox, *tmp;

	/* create top level */
	prefs_win = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(prefs_win), "gnotepad+ Preferences");
	gtk_signal_connect(GTK_OBJECT(prefs_win), "destroy",
		GTK_SIGNAL_FUNC(prefs_win_destroy), NULL);
	gtk_container_border_width(GTK_CONTAINER(prefs_win), 5);

	/* create vbox to hold everything */
	vbox = GTK_DIALOG(prefs_win)->vbox;
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);

	/* create notebook */
	prefs_nb = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(prefs_nb), prefs_tabpos);
	gtk_box_pack_start(GTK_BOX(vbox), prefs_nb, TRUE, TRUE, 0);

	/* create various pages of the notebook */
	prefs_page_appearance_create(prefs_nb);
	prefs_page_document_create(prefs_nb);
	prefs_page_window_create(prefs_nb);
	prefs_page_fonts_colors(prefs_nb);
	prefs_page_misc_create(prefs_nb);

	/* lastly, the buttons */
#if 0
	/* if we don't use gtk_dialog_new() and just create a regular top level
	   window, we can use the following code snippit to have more control
	   over how the buttons are laid out */
	hbox = gtk_hbutton_box_new();
	gtk_container_border_width(GTK_CONTAINER(hbox), 5);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
#endif
	hbox = GTK_DIALOG(prefs_win)->action_area;
	tmp = gtk_button_new_with_label("Save");
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(prefs_save_cb), w);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, TRUE, TRUE, 0);
	tmp = gtk_button_new_with_label("Tab Position");
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(prefs_tab_pos_change), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, TRUE, TRUE, 0);
	tmp = gtk_button_new_with_label("Cancel");
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(prefs_win_destroy), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(tmp);

	gtk_widget_show_all(prefs_win);
} /* prefs_dialog_create */


/*
 * PRIVATE: prefs_page_window_create
 *
 * creates the page titled "Window"
 *
 * 3. Window
 *	save window height on exit	(check button)
 *	default window height		(spin button)
 *	save window width on exit	(check button)
 *	default window width		(spin button)
 *
 *	save window position on exit	(check button)
 *	default window x-position	(spin button)
 *	default window y-position	(spin button)
 */
static void
prefs_page_window_create(GtkWidget *nb)
{
	GtkWidget *frame, *vbox, *tmp;

	frame = gtk_frame_new("Window Settings");
	gtk_container_border_width(GTK_CONTAINER(frame), 10);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* save window height on exit */
	(void)prefs_check_button_with_label_create(
			vbox,
			" Save Window height on exit ",
			IS_SAVE_WIN_HEIGHT(),
			"save_win_height");

	/* default window height */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Default height (-1 for auto) ",
			(float)win_height, -1.0, 2048.0, 1.0, 1.0, 0,
			"win_height");

	/* save window width on exit */
	(void)prefs_check_button_with_label_create(
			vbox,
			" Save Window width on exit ",
			IS_SAVE_WIN_WIDTH(),
			"save_win_width");

	/* default window width */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Default width (-1 for auto) ",
			(float)win_width, -1.0, 2048.0, 1.0, 1.0, 0,
			"win_width");

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 0);

	/* save window position on exit */
	(void)prefs_check_button_with_label_create(
			vbox,
			" Save Window Position on Exit ",
			IS_SAVE_WIN_POS(),
			"save_win_pos");

	/* default window xpos */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Default x-pos (-1 for auto) ",
			(float)win_xpos, -1.0, 2048.0, 1.0, 1.0, 0,
			"win_xpos");

	/* default window ypos */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Default y-pos (-1 for auto) ",
			(float)win_ypos, -1.0, 2048.0, 1.0, 1.0, 0,
			"win_ypos");

	/* stuff into notebook */
	tmp = gtk_label_new("Window");
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);
} /* prefs_page_window_create */


/*
 * PRIVATE: prefs_page_document_create
 *
 * creates the notebook page titled "Document"
 *
 * 2. Document
 *	autosave		(spin button)
 *	max num of recent docs	(spin button)
 *	wordwrap on/off		(check button)
 *	fonts			(popup must return string)
 *	colors			(popup must return string)
 */
static void
prefs_page_document_create(GtkWidget *nb)
{
	GtkWidget *frame, *vbox, *tmp;

	frame = gtk_frame_new("Document Settings");
	gtk_container_border_width(GTK_CONTAINER(frame), 10);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* autosave spin button */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Autosave minutes (0 disables) ",
			(float)autosave, 0.0, 1440.0, 1.0, 1.0, 0.0,
			"autosave_timeout");

	/* max num of recent doc */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Maximum # of recent documents ",
			(float)maxrecent, 0.0, 255.0, 1.0, 1.0, 0.0,
			"max_num_recent");

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 0);

	/* word wrap */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Use wordwrap",
			IS_USE_WORDWRAP(),
			"use_wordwrap");

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 0);

	/* stuff into notebook */
	tmp = gtk_label_new("Document");
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);
} /* prefs_page_document_create */


/*
 * PRIVATE: prefs_page_fonts_colors
 *
 * creates the notebook page titled "Fonts and Colors"
 *
 * Fonts and Colors
 *	Font Selection		(regular push button)
 *	Text Foreground Color	(regular push button)
 *	Text Background Color	(regular push button)
 */
static void
prefs_page_fonts_colors(GtkWidget *nb)
{
	GtkWidget *frame, *vbox, *tmp;

	frame = gtk_frame_new("Fonts and Colors");
	gtk_container_border_width(GTK_CONTAINER(frame), 10);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	tmp = gtk_button_new_with_label("Font Selection");
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 10);
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(prefs_font_sel_cb), NULL);
	prefs_wgtopt_list_add(tmp, GenericType, NULL, "text_font");

	tmp = gtk_button_new_with_label("Text Foreground Color");
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 10);
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(prefs_text_fg_cb), NULL);
	prefs_wgtopt_list_add(tmp, GenericType, NULL, "text_fg_color");

	tmp = gtk_button_new_with_label("Text Background Color");
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 10);
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		GTK_SIGNAL_FUNC(prefs_text_bg_cb), NULL);
	prefs_wgtopt_list_add(tmp, GenericType, NULL, "text_bg_color");

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 10);
	tmp = gtk_label_new(
		" NOTE: Fonts and Color changes take effect \n"
		" the next time you run gnotepad+ ");
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 10);

	/* stuff into notebook */
	tmp = gtk_label_new("Fonts and Colors");
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);
} /* prefs_page_fonts_colors */


/*
 * PRIVATE: prefs_text_fg_cb
 *
 * callback when button pushed is for changing the text foreground color
 */
static void
prefs_text_fg_cb(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_create(Foreground);
} /* prefs_text_fg_cb */


/*
 * PRIVATE: prefs_text_bg_cb
 *
 * callback when button pushed is for changing the text background color
 */
static void
prefs_text_bg_cb(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_create(Background);
} /* prefs_text_bg_cb */


/*
 * PRIVATE: prefs_color_sel_create
 *
 * creates the color selection dialog and sets up the callbacks.
 * some of this code was taken from testgtk.c
 *
 * rgb_str: points to either text_fg_str or text_bg_str.
 */
static void
prefs_color_sel_create(which_text_t which)
{
	gdouble color[4];
	GtkWidget *colsel;
	prefs_color_sel_t *pcsp;
	char *which_txt[] = {
		"Text Foreground Color Selection",
		"Text Background Color Selection"
	};


	colsel = gtk_color_selection_dialog_new(which_txt[which]);

	pcsp = (prefs_color_sel_t *)g_malloc(sizeof(prefs_color_sel_t));
	pcsp->cs = colsel;
	pcsp->which = which;

	if (which == Foreground) {
		sscanf(text_fg_str, "%lf %lf %lf",
			&(color[0]), &(color[2]), &(color[1]));
	} else {
		sscanf(text_bg_str, "%lf %lf %lf",
			&(color[0]), &(color[2]), &(color[1]));
	}
	color[0] /= COL_VAL_GDK;
	color[1] /= COL_VAL_GDK;
	color[2] /= COL_VAL_GDK;
	gtk_color_selection_set_color(GTK_COLOR_SELECTION(
		GTK_COLOR_SELECTION_DIALOG(colsel)->colorsel), color);

	gtk_color_selection_set_opacity(GTK_COLOR_SELECTION(
		GTK_COLOR_SELECTION_DIALOG(colsel)->colorsel), FALSE);

	gtk_color_selection_set_update_policy(GTK_COLOR_SELECTION(
		GTK_COLOR_SELECTION_DIALOG(colsel)->colorsel),
		GTK_UPDATE_CONTINUOUS);

	gtk_signal_connect(GTK_OBJECT(colsel), "destroy",
		GTK_SIGNAL_FUNC(prefs_color_sel_destroy), pcsp);

	gtk_signal_connect(GTK_OBJECT(
		GTK_COLOR_SELECTION_DIALOG(colsel)->colorsel), "color_changed",
		GTK_SIGNAL_FUNC(prefs_color_sel_changed), colsel);

	gtk_signal_connect(GTK_OBJECT(
		GTK_COLOR_SELECTION_DIALOG(colsel)->ok_button), "clicked",
		GTK_SIGNAL_FUNC(prefs_color_sel_ok), pcsp);

	gtk_signal_connect(GTK_OBJECT(
		GTK_COLOR_SELECTION_DIALOG(colsel)->cancel_button), "clicked",
		GTK_SIGNAL_FUNC(prefs_color_sel_close), pcsp);

	gtk_widget_show(colsel);
} /* prefs_color_sel_create */


/*
 * PRIVATE: prefs_text_color_update
 *
 * reads the RGB string values and converts them into what GDK expects.
 */
void
prefs_text_color_update(void)
{
	GdkColormap *gdkcmap;

	gdkcmap = gdk_colormap_get_system();

	sscanf(text_fg_str, "%hu %hu %hu", &(text_fg_color.red),
					   &(text_fg_color.blue),
					   &(text_fg_color.green));
	GNPDBG_PREFS(("prefs_text_color_update: FG pixel %lu, "
		      "red %u, blue %u, green %u\n",
		      text_fg_color.pixel, text_fg_color.red,
		      text_fg_color.blue, text_fg_color.green));

	if (pre102) {
		g_free(text_bg_str);
		text_bg_str = g_strdup("65535 65535 65535");
	}
	sscanf(text_bg_str, "%hu %hu %hu", &(text_bg_color.red),
					   &(text_bg_color.blue),
					   &(text_bg_color.green));
	GNPDBG_PREFS(("prefs_text_color_update: BG pixel %lu, "
		      "red %u, blue %u, green %u\n",
		      text_bg_color.pixel, text_bg_color.red,
		      text_bg_color.blue, text_bg_color.green));

	if (!gdk_color_alloc(gdkcmap, &text_fg_color))
		g_error("prefs_text_color_update: couldn't alloc fg color");
	if (!gdk_color_alloc(gdkcmap, &text_bg_color))
		g_error("prefs_text_color_update: couldn't alloc bg color");
} /* prefs_text_color_update */


/*
 * PRIVATE: prefs_color_sel_destroy
 *
 * callback for the "destroy" event for the color selection dialog
 */
static void
prefs_color_sel_destroy(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_t *pcsp = (prefs_color_sel_t *)cbdata;

	if (pcsp) {
		g_free(pcsp);
		pcsp = NULL;
	}
} /* prefs_color_sel_destroy */


/*
 * PRIVATE: prefs_color_sel_close
 *
 * callback for the "Close" button from the color selection dialog
 */
static void
prefs_color_sel_close(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_t *pcsp = (prefs_color_sel_t *)cbdata;

	g_assert(pcsp != NULL);
	gtk_widget_destroy(pcsp->cs);
} /* prefs_color_sel_close */


/*
 * PRIVATE: prefs_color_sel_ok
 *
 * callback for the "Ok" button in the color selection dialog.
 * gets the color from the color selection dialog, then converts it into a
 * character string containing RGB values.
 */
static void
prefs_color_sel_ok(GtkWidget *wgt, gpointer cbdata)
{
	prefs_color_sel_t *pcsp = (prefs_color_sel_t *)cbdata;
	GtkColorSelection *colorsel;
	gdouble color[4];
	char buf[MAX_RGB_STR];

	g_assert(pcsp != NULL);
	colorsel = GTK_COLOR_SELECTION(
			GTK_COLOR_SELECTION_DIALOG(pcsp->cs)->colorsel);
	gtk_color_selection_get_color(colorsel, color);

	/*
	 * NOTE!  gtk_color_selection_get_color() returns the color in the
	 * order red-blue-green, instead of red-green-blue.
	 */
	GNPDBG_PREFS(("prefs_color_sel_ok: colors = (red %f) (green %f) "
		      "(blue %f)\n", color[0], color[2], color[1]));
	g_snprintf(buf, MAX_RGB_STR, "%u %u %u",
		   (unsigned)(color[0] * COL_VAL_GDK),
		   (unsigned)(color[2] * COL_VAL_GDK),
		   (unsigned)(color[1] * COL_VAL_GDK));
	GNPDBG_PREFS(("prefs_color_sel_ok: buf = '%s'\n", buf));

	switch (pcsp->which) {
	case Foreground:
		if (text_fg_str)
			g_free(text_fg_str);
		text_fg_str = g_strdup(buf);
		GNPDBG_PREFS(("prefs_color_sel_ok: text_fg_color = '%s'\n",
			      text_fg_str));
		break;
	case Background:
		if (text_bg_str)
			g_free(text_bg_str);
		text_bg_str = g_strdup(buf);
		GNPDBG_PREFS(("prefs_color_sel_ok: text_bg_color = '%s'\n",
			      text_bg_str));
		break;
	} /* switch pcsp->which */

	gtk_widget_destroy(pcsp->cs);
} /* prefs_color_sel_ok */


/*
 * PRIVATE: prefs_color_sel_changed
 *
 * callback if color has changed in the color selection dialog
 */
static void
prefs_color_sel_changed(GtkWidget *wgt, gpointer cbdata)
{
	GtkColorSelectionDialog *cs = (GtkColorSelectionDialog *)cbdata;
	GtkColorSelection *colorsel;
	gdouble color[4];

	colorsel = GTK_COLOR_SELECTION(cs->colorsel);
	gtk_color_selection_get_color(colorsel, color);
} /* prefs_color_sel_changed */


/*
 * PRIVATE: prefs_font_sel_cb
 *
 * creates font selection dialog box and sets up callbacks.
 */
static void
prefs_font_sel_cb(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget *tmp;

	tmp = gtk_font_selection_dialog_new("Font Selection");
	gtk_signal_connect(
		GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(tmp)->ok_button),
		"clicked",
		GTK_SIGNAL_FUNC(prefs_font_sel_ok),
		tmp);

	gtk_signal_connect(
		GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(tmp)->cancel_button),
		"clicked",
		GTK_SIGNAL_FUNC(prefs_font_sel_cancel),
		tmp);

	gtk_signal_connect(
		GTK_OBJECT(tmp),
		"destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroy),
		tmp);

	(void)gtk_font_selection_dialog_set_font_name(
		GTK_FONT_SELECTION_DIALOG(tmp), text_font_str);

	gtk_widget_show(tmp);
} /* prefs_font_sel_cb */


/*
 * PRIVATE: prefs_font_sel_ok
 *
 * callback for "Ok" button on font selection dialog.
 * gets the font string from the dialog, and calls gdk_font_load() to load it
 * from the system.
 */
static void
prefs_font_sel_ok(GtkWidget *wgt, gpointer cbdata)
{
	GtkFontSelectionDialog *fsd = (GtkFontSelectionDialog *)cbdata;
	char *newfont;

	newfont = g_strdup(gtk_font_selection_dialog_get_font_name(fsd));
	if (newfont == NULL) {
		GNPDBG_PREFS(("prefs_font_sel_ok: no font chosen\n"));
		return;
	}

	if (text_font_str)
		g_free(text_font_str);
	text_font_str = g_strdup(newfont);
	GNPDBG_PREFS(("prefs_font_sel_ok: text_font_str '%s'\n",
		       text_font_str));
	if ((text_font = gdk_font_load(text_font_str)) == NULL)
		text_font = gdk_font_load(DEFAULT_FONT);

	gtk_widget_destroy(GTK_WIDGET(fsd));
} /* prefs_font_sel_ok */


/*
 * PRIVATE: prefs_font_sel_cancel
 *
 * callback for "Cancel" button on font selection dialog
 */
static void
prefs_font_sel_cancel(GtkWidget *wgt, gpointer cbdata)
{
	GtkFontSelectionDialog *fsd = (GtkFontSelectionDialog *)cbdata;

	gtk_widget_destroy(GTK_WIDGET(fsd));
} /* prefs_font_sel_cancel */


/*
 * PRIVATE: prefs_frame_msgbar_create
 *
 * creates the frame to contain options for the message bar
 */
static void
prefs_frame_msgbar_create(GtkWidget *parent)
{
	GtkWidget *frame, *tmp, *vbox;

	frame = gtk_frame_new("Message Bar");
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* 'show msg bar' check box */
	tmp = prefs_check_button_with_label_create(
			vbox,
			"Show msg bar",
			IS_SHOW_MSGBAR(),
			"show_message_bar");

	/* stuff into parent */
	gtk_box_pack_start(GTK_BOX(parent), frame, TRUE, TRUE, 0);
} /* prefs_frame_msgbar_create */


/*
 * PRIVATE: prefs_frame_doctab_create
 *
 * creates the frame to contain options for the document tabs
 */
static void
prefs_frame_doctab_create(GtkWidget *parent)
{
	GtkWidget *frame, *vbox;

	g_list_free(tabpos_list);
	tabpos_list = NULL;
	tabpos_list = g_list_append(tabpos_list, "Left");
	tabpos_list = g_list_append(tabpos_list, "Right");
	tabpos_list = g_list_append(tabpos_list, "Top");
	tabpos_list = g_list_append(tabpos_list, "Bottom");

	frame = gtk_frame_new("Document Tabs");
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* 'show doc tabs' check box */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Show doc tabs",
			IS_SHOW_TABS(),
			"show_doc_tabs");

	/* tab position */
	(void)prefs_pulldown_with_label_create(
			vbox,
			" Tab position ",
			tabpos_list,
			"doc_tab_position"
			);

	/* stuff into parent */
	gtk_box_pack_start(GTK_BOX(parent), frame, TRUE, TRUE, 0);
} /* prefs_frame_doctab_create */


/*
 * PRIVATE: prefs_page_appearance_create
 *
 * creates the notebook page titled "Appearance"
 *
 * 1. Appearance
 *	toolbar on/off		(check button)
 *	toolbar style		(radio buttons)
 *	tooltips on/off		(check button)
 *	doc tabs on/off		(check button)
 *	doc tabs position	(pulldown combo entry text)
 *	msgbar on/off		(check button)
 */
static void
prefs_page_appearance_create(GtkWidget *nb)
{
	GtkWidget *frame, *vbox, *tmp, *hbox;

	frame = gtk_frame_new("Appearance Settings");
	gtk_container_border_width(GTK_CONTAINER(frame), 10);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	prefs_frame_toolbar_create(vbox);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);

	prefs_frame_doctab_create(hbox);
	prefs_frame_msgbar_create(hbox);

	/* stuff into notebook */
	tmp = gtk_label_new("Appearance");
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);
} /* prefs_page_appearance_create */


/*
 * PRIVATE: prefs_frame_toolbar_create
 *
 * creates the frame to contain options for the toolbar
 */
static void
prefs_frame_toolbar_create(GtkWidget *parent)
{
	GtkWidget *frame, *hbox, *vbox, *radio;

	frame = gtk_frame_new("Toolbar Settings");
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* put first two checkboxes into one vbox */
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

	/* 'show toolbar' check box */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Show Toolbar",
			IS_SHOW_TOOLBAR(),
			"show_toolbar");

	/* 'show tooltips' check box */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Show Tooltips",
			IS_SHOW_TOOLTIPS(),
			"show_tooltips");

	/* now another vbox into the hbox */
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

	/* radio buttons for toolbar style */
	radio = prefs_radio_button_with_label_create(
			vbox,
			NULL,
			"Pictures and Text",
			IS_TEXT_PIC_TOOLBAR(),
			"toolbar_pictext");
	(void)prefs_radio_button_with_label_create(
			vbox,
			radio,
			"Pictures Only",
			IS_PIC_TOOLBAR(),
			"toolbar_piconly");
	(void)prefs_radio_button_with_label_create(
			vbox,
			radio,
			"Text Only",
			IS_TEXT_TOOLBAR(),
			"toolbar_textonly");

	/* stuff into parent */
	gtk_box_pack_start(GTK_BOX(parent), frame, FALSE, FALSE, 0);
} /* prefs_frame_toolbar_create */


/*
 * PRIVATE: prefs_page_misc_create
 *
 * creates the notebook page titled "Miscellaneous"
 *
 * 4. Misc
 *	tmpdir				(text entry string)
 *	printcmd			(text entry string)
 *	msgbox on/off			(check button)
 *	max # msgs			(spin button w/ text entry)
 *	msgbox truncate percentage	(spin button w/ text entry)
 */
static void
prefs_page_misc_create(GtkWidget *nb)
{
	GtkWidget *frame, *vbox, *tmp;

	frame = gtk_frame_new("Miscellaneous");
	gtk_container_border_width(GTK_CONTAINER(frame), 10);
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* print cmd */
	(void)prefs_entry_with_label_create(
			vbox, " Print command ", printcmd, "print_cmd");

	/* tmp dir */
	(void)prefs_entry_with_label_create(
			vbox, " Temp directory ", tmpdir, "tmp_directory");

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 0);

	prefs_frame_msgbox_create(vbox);

	/* stuff into notebook */
	tmp = gtk_label_new("Miscellaneous");
	gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);
} /* prefs_page_misc_create */


/*
 * PRIVATE: prefs_frame_msgbox_create
 *
 * creates the frame containing the options for the msgbox
 *
 *	msgbox on/off			(check button)
 *	max # msgs			(spin button w/ text entry)
 *	msgbox truncate percentage	(spin button w/ text entry)
 */
static void
prefs_frame_msgbox_create(GtkWidget *parent)
{
	GtkWidget *frame, *vbox;

	frame = gtk_frame_new("Message Box");
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	/* use msg box */
	(void)prefs_check_button_with_label_create(
			vbox,
			"Use message box",
			IS_USE_MSGBOX(),
			"use_msgbox");

	/* msgbox max msgs */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" Max # of messages ",
			(float)msgbox_max_msg, 0.0, 1024.0, 1.0, 1.0, 0,
			"msgbox_max_msg");

	/* msgbox del msgs */
	(void)prefs_spin_button_with_label_create(
			vbox,
			" % msgs to delete ",
			(float)msgbox_per_del, 0.0, 100.0, 1.0, 1.0, 0,
			"msgbox_per_del");

	/* stuff into parent */
	gtk_box_pack_start(GTK_BOX(parent), frame, FALSE, FALSE, 0);
} /* prefs_frame_msgbox_create */


/*
 * PRIVATE: prefs_pulldown_with_label_create
 *
 * convenience routine for creating combo widget (pulldown widget with text)
 */
static GtkWidget *
prefs_pulldown_with_label_create(
	GtkWidget *parent,
	char *labeltext,
	GList *itemlist,
	char *prefname)
{
	GtkWidget *hbox, *tmp;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), hbox, FALSE, FALSE, 0);
	tmp = gtk_label_new(labeltext);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);

	tmp = gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(tmp), itemlist);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(tmp)->entry),
		(char *)g_list_nth_data(itemlist, tabpos));
	gtk_widget_set_usize(GTK_COMBO(tmp)->entry, 65, 0);
	gtk_editable_select_region(GTK_EDITABLE(GTK_COMBO(tmp)->entry), 0, -1);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
	prefs_wgtopt_list_add(GTK_WIDGET(GTK_EDITABLE(GTK_COMBO(tmp)->entry)),
		ComboType, itemlist, prefname);

	return tmp;
} /* prefs_pulldown_with_label_create */


/*
 * PRIVATE: prefs_check_button_with_label_create
 *
 * convenience routine for creating a check button and label, and then packing
 * them into a parent (box).
 */
static GtkWidget *
prefs_check_button_with_label_create(
	GtkWidget *parent,
	char *labeltext,
	long expr,
	char *prefname)
{
	GtkWidget *tmp;

	tmp = gtk_check_button_new_with_label(labeltext);
	gtk_box_pack_start(GTK_BOX(parent), tmp, FALSE, FALSE, 0);
	if (expr)
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(tmp), TRUE);
	prefs_wgtopt_list_add(tmp, ButtonType, NULL, prefname);

	return tmp;
} /* prefs_check_button_with_label_create */


/*
 * PRIVATE: prefs_entry_with_label_create
 *
 * convenience routine for creating a text entry widget and label
 */
static GtkWidget *
prefs_entry_with_label_create(
	GtkWidget *parent,
	char *labeltext,
	char *entrytext,
	char *prefname)
{
	GtkWidget *hbox, *tmp;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), hbox, FALSE, FALSE, 0);
	tmp = gtk_label_new(labeltext);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
	tmp = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
	gtk_entry_set_text(GTK_ENTRY(tmp), entrytext);
	prefs_wgtopt_list_add(tmp, EntryType, NULL, prefname);

	return tmp;
} /* prefs_entry_with_label_create */


/*
 * PRIVATE: prefs_spin_button_with_label_create
 *
 * convenience routine for creating a spin button and a label
 */
static GtkWidget *
prefs_spin_button_with_label_create(
	GtkWidget *parent,
	char *labeltext,
	float adj_value,
	float adj_lower,
	float adj_upper,
	float adj_step_increment,
	float adj_page_increment,
	float adj_page_size,
	char *prefname)
{
	GtkWidget *tmp, *hbox, *entry;
	GtkObject *adj;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), hbox, FALSE, FALSE, 0);
	tmp = gtk_label_new(labeltext);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
	adj = gtk_adjustment_new(adj_value, adj_lower, adj_upper,
		adj_step_increment, adj_page_increment, adj_page_size);
	tmp = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0, 0);
	gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(tmp), TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, TRUE, 0);
	gtk_widget_set_usize(tmp, 65, 0);
	entry = GTK_WIDGET(&(GTK_SPIN_BUTTON(tmp)->entry));
	prefs_wgtopt_list_add(GTK_WIDGET(entry), EntryType, NULL, prefname);

	return tmp;
} /* prefs_spin_button_with_label_create */


/*
 * PRIVATE: prefs_radio_button_with_label_create
 *
 * convenience routine for creating a radio button and a label.
 */
static GtkWidget *
prefs_radio_button_with_label_create(
	GtkWidget *parent,
	GtkWidget *group,
	char *labeltext,
	long expr,
	char *prefname)
{
	GtkWidget *tmp;

	if (group != NULL)
		tmp = gtk_radio_button_new_with_label(gtk_radio_button_group(
				GTK_RADIO_BUTTON(group)), labeltext);
	else
		tmp = gtk_radio_button_new_with_label(NULL, labeltext);
	gtk_box_pack_start(GTK_BOX(parent), tmp, FALSE, FALSE, 0);
	if (expr)
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(tmp), TRUE);
	prefs_wgtopt_list_add(tmp, ButtonType, NULL, prefname);

	return tmp;

} /* prefs_radio_button_with_label_create */


/*
 * PRIVATE: prefs_wgtopt_list_add
 *
 * adds a widget and its corresponding information into the wgtopt_list.
 */
static void
prefs_wgtopt_list_add(GtkWidget *wgt, wgttype_t type, void *data,
	char *prefname)
{
	wgtopt_t *wdata;
	prefs_t *pap;

	wdata = (wgtopt_t *)g_malloc0(sizeof(wgtopt_t));
	wdata->wgt = wgt;

	pap = app_prefs;
	while (pap->name) {
		if (strncmp(prefname, pap->name, strlen(pap->name)) == 0) {
			wdata->pap = pap;
			wdata->type = type;
			wdata->data = data;
			wgtopt_list = g_list_append(wgtopt_list, wdata);
			break;
		}
		pap++;
	}
} /* prefs_wgtopt_list_add */


/*
 * PRIVATE: prefs_update
 *
 * reads from wgtopt_list and updates in-memory preferences.  invoked just
 * before calling prefs_save() to save the new changes.
 */
static void
prefs_update(void)
{
	GList *wolp, *dp;
	wgtopt_t *wdata;
	char buf[8], *data;
	int num;

	wolp = wgtopt_list;
	while (wolp) {
		wdata = (wgtopt_t *)(wolp->data);
		GNPDBG_PREFS(("prefs_update: updating '%s''\n",
			       wdata->pap->name));
		switch (wdata->pap->type) {
		case PrefBool:
			if (GTK_TOGGLE_BUTTON(GTK_BUTTON(wdata->wgt))->active)
				prefs_bool_set(wdata->pap, "true");
			else
				prefs_bool_set(wdata->pap, "false");
			break;
		case PrefString:
			if (wdata->type != GenericType)
				prefs_string_set(wdata->pap,
						 gtk_entry_get_text(
						 	GTK_ENTRY(wdata->wgt)));
			break;
		case PrefInt:
			/*
			 * ugh this is awful.  if the preference variable
			 * stores an integer, we have to allow at least a
			 * couple of ways to represent it in the prefs window.
			 * the obvious way is the text entry method, where we
			 * can simply use atoi() on the text entry.  however,
			 * in the case of document tab positioning, the value
			 * is an integer, but we want to present it as text to
			 * the user (e.g., "Left", "Right", "Top", "Bottom").
			 * to do this, we need some special case code to
			 * handle it.  i will probably rethink this in the
			 * future to see if there's a better way to do this.
			 */
			data = gtk_entry_get_text(GTK_ENTRY(wdata->wgt));

			if (wdata->type == EntryType) {
				prefs_int_set(wdata->pap, data);
			} else if (wdata->type == ComboType) {
				dp = (GList *)(wdata->data);
				num = 0;
				while (dp) {
					if (strcmp((char *)(dp->data),
						data) == 0) {
						g_snprintf(buf, 8, "%d", num);
						prefs_int_set(wdata->pap, buf);
						break;
					}
					num++;
					dp = dp->next;
				}
			}

			break;
		case PrefByte:
			prefs_byte_set(wdata->pap,
				gtk_entry_get_text(GTK_ENTRY(wdata->wgt)));
			break;
		case PrefFloat:
			prefs_float_set(wdata->pap,
				gtk_entry_get_text(GTK_ENTRY(wdata->wgt)));
			break;
		default:
			printf("prefs_update: ignoring '%s' (unknown type=%d)"
			       "\n", wdata->pap->name, wdata->pap->type);
			break;
		} /* switch */

		if (wdata->pap->f) {
			GNPDBG_PREFS(("prefs_update: invoking callback for "
				      "'%s''\n", wdata->pap->name));
			(wdata->pap->f)();
		}
		wolp = wolp->next;
	} /* while wolp */
} /* prefs_update */


/*
 * PRIVATE: prefs_wgtopt_list_free
 *
 * cleans up and frees the wgtopt_list
 */
static void
prefs_wgtopt_list_free(void)
{
	GList *wolp;
	wgtopt_t *wdata;

	while (wgtopt_list) {
		wolp = wgtopt_list;
		wgtopt_list = g_list_remove_link(wgtopt_list, wolp);
		wdata = (wgtopt_t *)(wolp->data);
		g_free(wdata);
	}
} /* prefs_wgtopt_list_free */


/*
 * PRIVATE: prefs_save_cb
 *
 * callback invoked when user clicks on "Ok" button
 */
static void
prefs_save_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_widget_hide(prefs_win);
	prefs_update();
	prefs_save();
	prefs_wgtopt_list_free();
	prefs_win_destroy(NULL, NULL);
	appgtk_rc_update(appgtkrc);
	msgbox_printf("Preferences saved...");
	msgbar_printf(w, "Preferences saved...");
} /* prefs_save_cb */


/*
 * PRIVATE: prefs_win_destroy
 *
 * destroys the prefs window and cleans up the wgtopt_list
 */
static void
prefs_win_destroy(GtkWidget *wgt, gpointer cbdata)
{
	prefs_wgtopt_list_free();
	gtk_widget_destroy(prefs_win);
	prefs_win = NULL;
} /* prefs_win_destroy */


/*
 * PRIVATE: prefs_tab_pos_change
 *
 * for all windows, redraw the document tabs.  called from prefs_save, when
 * the preferences may have changed.
 */
static void
prefs_tab_pos_change(void)
{
	switch (prefs_tabpos) {
	case GTK_POS_LEFT:
		prefs_tabpos = GTK_POS_TOP;
		break;
	case GTK_POS_RIGHT:
		prefs_tabpos = GTK_POS_BOTTOM;
		break;
	case GTK_POS_TOP:
		prefs_tabpos = GTK_POS_RIGHT;
		break;
	case GTK_POS_BOTTOM:
		prefs_tabpos = GTK_POS_LEFT;
		break;
	}
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(prefs_nb), prefs_tabpos);
} /* prefs_tab_pos_change */


/* the end */
