/* vi:set ts=8 sts=0 sw=8:
 * $Id: toolbar.c,v 1.7 1998/10/15 21:03:39 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glib.h>
#include "main.h"
#include "doc.h"
#include "print.h"
#include "clipboard.h"
#include "toolbar.h"
#include "msgbar.h"
#include "prefs.h"

#include "xpm/tb_new.xpm"
#include "xpm/tb_open.xpm"
#include "xpm/tb_save.xpm"
#include "xpm/tb_cancel.xpm"
#include "xpm/tb_print.xpm"
#include "xpm/tb_cut.xpm"
#include "xpm/tb_copy.xpm"
#include "xpm/tb_paste.xpm"
#include "xpm/tb_search.xpm"
#include "xpm/tb_prefs.xpm"
#include "xpm/tb_exit.xpm"


/*** local definitions ***/
#define PIC_TEXT	"Toolbar with icons and text"
#define PIC_ONLY	"Toolbar with icons only"
#define TEXT_ONLY	"Toolbar with text only"
#define TOOLBAR_ON	"Toolbar ON"
#define TOOLBAR_OFF	"Toolbar OFF"
#define TOOLTIPS_ON	"Tooltips ON"
#define TOOLTIPS_OFF	"Tooltips OFF"


/*** local variable declarations ***/
static toolbar_data_t main_tb[] = {
	{ " New ", "New File", "new", tb_new_xpm,
		(GtkSignalFunc)doc_new_cb, NULL },
	{ " Open ", "Open File", "open", tb_open_xpm,
		(GtkSignalFunc)doc_open_cb, NULL },
	{ " Close ", "Close File", "close", tb_cancel_xpm,
		(GtkSignalFunc)doc_close_cb, NULL },
	{ " Save ", "Save File", "save", tb_save_xpm,
		(GtkSignalFunc)doc_save_cb, NULL },
	{ " Print ", "Print File", "print", tb_print_xpm,
		(GtkSignalFunc)print_cb, NULL },
	{ " SPACE ", NULL, NULL, NULL, NULL, NULL },
	{ " Cut ", "Cut Text", "cut", tb_cut_xpm,
		(GtkSignalFunc)cut_cb, NULL },
	{ " Copy ", "Copy Text", "copy", tb_copy_xpm,
		(GtkSignalFunc)copy_cb, NULL },
	{ " Paste ", "Paste Text", "paste", tb_paste_xpm,
		(GtkSignalFunc)paste_cb, NULL },
	{ " Find ", "Find Text", "Find", tb_search_xpm,
		(GtkSignalFunc)NULL, NULL },
	{ " SPACE ", NULL, NULL, NULL, NULL, NULL },
	{ " Prefs ", "Preferences", "prefs", tb_prefs_xpm,
		(GtkSignalFunc)prefs_cb, NULL },
	{ " Exit ", "Exit App", "exit", tb_exit_xpm,
		(GtkSignalFunc)win_close_all_cb, NULL },
	{ NULL, NULL, NULL, NULL, NULL }
};

static toolbar_data_t dlw_tb[] = {
	{ " Save ", "Save file", "Toolbar/Save", tb_save_xpm,
		(GtkSignalFunc)doc_save_cb, NULL },
	{ " Close ", "Close the current file", "Toolbar/Close", tb_cancel_xpm,
		(GtkSignalFunc)doc_close_cb, NULL },
	{ " Print ", "Print file", "Toolbar/Print", tb_print_xpm,
		(GtkSignalFunc)print_cb, NULL },
	{ " SPACE ", NULL, NULL, NULL, NULL, NULL },
	{ " Ok ", "Close list window", "Ok", tb_exit_xpm,
		(GtkSignalFunc)doc_list_destroy, NULL },
	{ NULL, NULL, NULL, NULL, NULL }
};


/*** local function prototypes ***/
static GtkWidget *new_pixmap(char **icon, GtkWidget *gdkw, GtkWidget *w);
static GtkWidget *toolbar_create_common(toolbar_data_t *tbdata, win_t *w);
static void tb_off(win_t *w);
static void tb_on(win_t *w);


/*** global function definitions ***/
/*
 * redraws the toolbar for a specified window
 */
void
tb_redraw(win_t *w)
{
	if (IS_SHOW_TOOLBAR())
		tb_on(w);
	else
		tb_off(w);

	if (IS_TEXT_PIC_TOOLBAR())
		tb_pic_text(NULL, w);
	else if (IS_PIC_TOOLBAR())
		tb_pic_only(NULL, w);
	else
		tb_text_only(NULL, w);
} /* tb_redraw */


/*
 * PUBLIC: tb_pic_only
 *
 * updates toolbar to show buttons with icons only
 */
void
tb_pic_only(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_toolbar_set_style(GTK_TOOLBAR(w->toolbar), GTK_TOOLBAR_ICONS);
	if (GTK_WIDGET_VISIBLE(w->toolbar) && IS_SHOW_TOOLBAR()) {
		gtk_widget_hide(w->toolbar);
		gtk_widget_show(w->toolbar);
	}

	if (w->dlw_toolbar) {
		gtk_toolbar_set_style(
			GTK_TOOLBAR(w->dlw_toolbar), GTK_TOOLBAR_ICONS);
		if (GTK_WIDGET_VISIBLE(w->dlw_toolbar) && IS_SHOW_TOOLBAR()) {
			gtk_widget_hide(w->dlw_toolbar);
			gtk_widget_show(w->dlw_toolbar);
		}
	}

	SET_PIC_ONLY_TOOLBAR();
	CLEAR_PIC_TEXT_TOOLBAR();
	CLEAR_TEXT_ONLY_TOOLBAR();
	msgbar_printf(w, PIC_ONLY);
} /* tb_pic_only */


/*
 * PUBLIC: tb_pic_text
 *
 * updates toolbar to show buttons with icons and text
 */
void
tb_pic_text(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_toolbar_set_style(GTK_TOOLBAR(w->toolbar), GTK_TOOLBAR_BOTH);
	if (GTK_WIDGET_VISIBLE(w->toolbar) && IS_SHOW_TOOLBAR()) {
		gtk_widget_hide(w->toolbar);
		gtk_widget_show(w->toolbar);
	}

	if (w->dlw_toolbar) {
		gtk_toolbar_set_style(
			GTK_TOOLBAR(w->dlw_toolbar), GTK_TOOLBAR_BOTH);
		if (GTK_WIDGET_VISIBLE(w->dlw_toolbar) && IS_SHOW_TOOLBAR()) {
			gtk_widget_hide(w->dlw_toolbar);
			gtk_widget_show(w->dlw_toolbar);
		}
	}

	SET_PIC_TEXT_TOOLBAR();
	CLEAR_TEXT_ONLY_TOOLBAR();
	CLEAR_PIC_ONLY_TOOLBAR();
	msgbar_printf(w, PIC_TEXT);
} /* tb_pic_text */


/*
 * PUBLIC: tb_text_only
 *
 * updates toolbar to show buttons with text only
 */
void
tb_text_only(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_toolbar_set_style(GTK_TOOLBAR(w->toolbar), GTK_TOOLBAR_TEXT);
	if (GTK_WIDGET_VISIBLE(w->toolbar) && IS_SHOW_TOOLBAR()) {
		gtk_widget_hide(w->toolbar);
		gtk_widget_show(w->toolbar);
	}

	if (w->dlw_toolbar) {
		gtk_toolbar_set_style(
			GTK_TOOLBAR(w->dlw_toolbar), GTK_TOOLBAR_TEXT);
		if (GTK_WIDGET_VISIBLE(w->dlw_toolbar) && IS_SHOW_TOOLBAR()) {
			gtk_widget_hide(w->dlw_toolbar);
			gtk_widget_show(w->dlw_toolbar);
		}
	}

	SET_TEXT_ONLY_TOOLBAR();
	CLEAR_PIC_TEXT_TOOLBAR();
	CLEAR_PIC_ONLY_TOOLBAR();

	msgbar_printf(w, TEXT_ONLY);
} /* tb_text_only */


/*
 * PUBLIC: tb_toggle
 *
 * toggles toolbar
 */
void
tb_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);

	if (IS_SHOW_TOOLBAR()) {
		tb_off(w);
		CLEAR_SHOW_TOOLBAR();
		msgbar_printf(w, TOOLBAR_OFF);
	} else {
		tb_on(w);
		SET_SHOW_TOOLBAR();
		msgbar_printf(w, TOOLBAR_ON);
	}
} /* tb_toggle */


/*
 * PUBLIC: tb_ttip_off
 *
 * turns OFF tooltips
 */
void
tb_ttip_off(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_toolbar_set_tooltips(GTK_TOOLBAR(w->toolbar), FALSE);
	if (w->dlw_toolbar_h)
		gtk_toolbar_set_tooltips(GTK_TOOLBAR(w->dlw_toolbar), FALSE);
	CLEAR_SHOW_TOOLTIPS();
	msgbar_printf(w, TOOLTIPS_OFF);
} /* tb_ttip_off */


/*
 * PUBLIC: tb_ttip_on
 *
 * turns ON tooltips
 */
void
tb_ttip_on(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_toolbar_set_tooltips(GTK_TOOLBAR(w->toolbar), TRUE);
	if (w->dlw_toolbar_h)
		gtk_toolbar_set_tooltips(GTK_TOOLBAR(w->dlw_toolbar), TRUE);
	SET_SHOW_TOOLTIPS();
	msgbar_printf(w, TOOLTIPS_ON);
} /* tb_ttip_on */


/*
 * PUBLIC: toolbar_create_dlw
 *
 * creates main toolbar that goes below menu bar
 */
void
toolbar_create_dlw(win_t *w)
{
	GtkWidget *toolbar;

	toolbar = toolbar_create_common(dlw_tb, w);
#ifdef GTK_HAVE_FEATURES_1_1_0
	/*
	 * do we want to set the relief to GTK_RELIEF_NONE?  setting the
	 * relief on the toolbar widget doesn't set the relief for the handle
	 * box.
	 */
	gtk_toolbar_set_button_relief(GTK_TOOLBAR(toolbar), GTK_RELIEF_NORMAL);
#endif
	GTK_WIDGET_UNSET_FLAGS(toolbar, GTK_CAN_FOCUS);
	w->dlw_toolbar = toolbar;
	w->dlw_toolbar_h = gtk_handle_box_new();
	gtk_container_add(GTK_CONTAINER(w->dlw_toolbar_h), toolbar);
	if (IS_SHOW_TOOLBAR())
		gtk_widget_show(w->dlw_toolbar_h);
} /* toolbar_create_dlw */


/*
 * PUBLIC: toolbar_main_init
 *
 * creates main toolbar that goes below menu bar
 */
void
toolbar_main_init(win_t *w)
{
	GtkWidget *toolbar;

	gtk_widget_realize(w->toplev);
	toolbar = toolbar_create_common(main_tb, w);
#ifdef GTK_HAVE_FEATURES_1_1_0
	gtk_toolbar_set_button_relief(GTK_TOOLBAR(toolbar), GTK_RELIEF_NORMAL);
#endif
	GTK_WIDGET_UNSET_FLAGS(toolbar, GTK_CAN_FOCUS);
	w->toolbar = toolbar;
	w->toolbar_h = gtk_handle_box_new();
	gtk_container_add(GTK_CONTAINER(w->toolbar_h), toolbar);
	gtk_box_pack_start(GTK_BOX(w->mainbox), w->toolbar_h, FALSE, TRUE, 0);

	if (IS_SHOW_TOOLBAR())
		gtk_widget_show(w->toolbar_h);
} /* toolbar_main_init */


/*** local function definitions */
/*
 * PRIVATE: new_pixmap
 *
 * taken from testgtk.c
 */
static GtkWidget*
new_pixmap(char **icon, GtkWidget *gtkw, GtkWidget *wgt)
{
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	pixmap = gdk_pixmap_create_from_xpm_d(gtkw->window, &mask,
				&gtkw->style->bg[GTK_STATE_NORMAL], icon);
	return gtk_pixmap_new(pixmap, mask);
} /* new pixmap */


static void
tb_off(win_t *w)
{
	if (GTK_WIDGET_VISIBLE(w->toolbar_h))
		gtk_widget_hide(w->toolbar_h);
	
	if (w->dlw_toolbar_h && GTK_WIDGET_VISIBLE(w->dlw_toolbar_h))
		gtk_widget_hide(w->dlw_toolbar_h);
} /* tb_on */


static void
tb_on(win_t *w)
{
	if (!GTK_WIDGET_VISIBLE(w->toolbar_h))
		gtk_widget_show_all(w->toolbar_h);

	if (w->dlw_toolbar_h && !GTK_WIDGET_VISIBLE(w->dlw_toolbar_h))
		gtk_widget_show_all(w->dlw_toolbar_h);
} /* tb_on */


/*
 * PRIVATE: toolbar_create_common
 *
 * common routine to create a toolbar.
 *
 * in: toolbar_data_t and pointer to callback data of gE_data type
 * out: GtkWidget *toolbar
 */
static GtkWidget *
toolbar_create_common(toolbar_data_t *tbdata, win_t *w)
{
	GtkWidget *tb;
	toolbar_data_t *tbdp = tbdata;

	g_assert(tbdp != NULL);

	if (IS_TEXT_TOOLBAR()) {
		tb = gtk_toolbar_new(
				GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_TEXT);
	} else if (IS_PIC_TOOLBAR()) {
		tb = gtk_toolbar_new(
				GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	} else {
		SET_PIC_TEXT_TOOLBAR();
		tb = gtk_toolbar_new(
				GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);
	}

	while (tbdp->text != NULL) {
		gtk_toolbar_append_item(
			GTK_TOOLBAR(tb),
			tbdp->text,
			tbdp->tooltip_text,
			tbdp->tooltip_private_text,
			new_pixmap(tbdp->icon, w->toplev, tb),
			(GtkSignalFunc)tbdp->callback,
			(tbdp->cbdata) ? tbdp->cbdata : (gpointer)w);

		tbdp++;

		if (tbdp->text != NULL && strcmp(tbdp->text, " SPACE ") == 0) {
			gtk_toolbar_append_space(GTK_TOOLBAR(tb));
			tbdp++;
		}
	}

	/* set according to preferences */
	gtk_toolbar_set_tooltips(GTK_TOOLBAR(tb), IS_SHOW_TOOLTIPS());

	/*
	 * this is OK to show even if prefs says don't show, because it's
	 * the toolbar handle that gets shown
	 */
	gtk_widget_show(tb);
	return tb;
} /* toolbar_create_common */


/* the end */
