/*
	Audio File Library

	Copyright 1998, Michael Pruett <michael@68k.org>

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License as
	published by the Free Software Foundation; either version 2 of
	the License, or (at your option) any later version.

	This library is distributed in the hope that it will be
	useful, but WITHOUT ANY WARRANTY; without even the implied
	warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
	PURPOSE.  See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this program; if not, write to the Free
	Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
	MA 02111-1307, USA.
*/

/*
	wave.c

	This file contains code for parsing RIFF WAVE format sound files.
*/

#include <assert.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "audiofile.h"
#include "util.h"
#include "afinternal.h"
#include "byteorder.h"
#include "wave.h"

void ParseFormat (AFfilehandle filehandle, FILE *fp, u_int32_t id, size_t size)
{
	u_int16_t	formatTag, channelCount;
	u_int32_t	sampleRate, averageBytesPerSecond;
	u_int16_t	blockAlign;

	assert(filehandle != NULL);
	assert(fp != NULL);

	fread(&formatTag, 1, 2, fp);
	formatTag = LENDIAN_TO_HOST_INT16(formatTag);

	/* only PCM format data is supported at the present */
	assert(formatTag == WAVE_FORMAT_PCM);

	fread(&channelCount, 1, 2, fp);
	channelCount = LENDIAN_TO_HOST_INT16(channelCount);
	filehandle->channelCount = channelCount;

	fread(&sampleRate, 1, 4, fp);
	sampleRate = LENDIAN_TO_HOST_INT32(sampleRate);
	filehandle->sampleRate = sampleRate;

	fread(&averageBytesPerSecond, 1, 4, fp);
	averageBytesPerSecond = LENDIAN_TO_HOST_INT32(averageBytesPerSecond);

	fread(&blockAlign, 1, 2, fp);
	blockAlign = LENDIAN_TO_HOST_INT16(blockAlign);

	if (formatTag == WAVE_FORMAT_PCM)
	{
		u_int16_t	bitsPerSample;

		fread(&bitsPerSample, 1, 2, fp);
		bitsPerSample = LENDIAN_TO_HOST_INT16(bitsPerSample);

		filehandle->sampleWidth = bitsPerSample;
	}
}

void ParseData (AFfilehandle filehandle, FILE *fp, u_int32_t id, size_t size)
{
	u_int32_t	frameSize;

	assert(filehandle != NULL);
	assert(fp != NULL);

	frameSize = filehandle->channelCount * (filehandle->sampleWidth / 8);

	filehandle->dataStart = ftell(fp);
	filehandle->trackBytes = size;
	filehandle->frameCount = size / frameSize;
}

int parsewave (AFfilehandle file)
{
	u_int32_t	type, size, formtype;
	u_int32_t	index = 0;

	assert(file != NULL);
	assert(file->fp != NULL);

	fread(&type, 4, 1, file->fp);
	fread(&size, 4, 1, file->fp);
	size = LENDIAN_TO_HOST_INT32(size);
	fread(&formtype, 4, 1, file->fp);

	assert(!memcmp(&type, "RIFF", 4));
	assert(!memcmp(&formtype, "WAVE", 4));
	
#ifdef DEBUG
	printf("size: %d\n", size);
#endif

	index += 4;

	/* include the offset of the form type */
	while (index < size)
	{
		u_int32_t	chunkid, chunksize;

#ifdef DEBUG
		printf("index: %d\n", index);
#endif
		fread(&chunkid, 4, 1, file->fp);
		chunkid = BENDIAN_TO_HOST_INT32(chunkid);

		fread(&chunksize, 4, 1, file->fp);
		chunksize = LENDIAN_TO_HOST_INT32(chunksize);

#ifdef DEBUG
		printid(chunkid);
		printf(" size: %d\n", chunksize);
#endif

		switch (chunkid)
		{
			/* the common chunk is required */
			case 'data':
				ParseData(file, file->fp, chunkid, chunksize);
				break;
			case 'fmt ':
				ParseFormat(file, file->fp, chunkid, chunksize);
				break;
		}

		index += chunksize + 8;

		/* all chunks must be aligned on an even number of bytes */
		if ((index % 2) != 0)
			index++;

		fseek(file->fp, index + 8, SEEK_SET);
	}
}
