// Fl.C

// fltk (Fast Light Tool Kit) version 0.99
// Copyright (C) 1998 Bill Spitzak

// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.

// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.

// Written by Bill Spitzak spitzak@d2.com

// This file contains the Fl:: methods that are system-independent

#include <FL/Fl.H>
#include <FL/Fl_Window.H>
#include <stdlib.h>

int Fl::damage_;
Fl_Widget *Fl::belowmouse_;
Fl_Widget *Fl::pushed_;
Fl_Widget *Fl::focus_;
Fl_Widget *Fl::selection_owner_;
int Fl::e_x, Fl::e_y, Fl::e_x_root, Fl::e_y_root;
int Fl::e_state;
int Fl::e_clicks;
int Fl::e_is_click;
int Fl::e_keysym;
char *Fl::e_text;
int Fl::e_length;

int Fl::event_inside(int x,int y,int w,int h) /*const*/ {
  int mx = event_x();
  int my = event_y();
  return (mx >= x && mx < x+w && my >= y && my < y+h);
}

int Fl::event_inside(const Fl_Widget *o) /*const*/ {
  return event_inside(o->x(),o->y(),o->w(),o->h());
}

int Fl::run() {
  while (wait());
  return 0;
}

////////////////////////////////////////////////////////////////

void Fl::focus(Fl_Widget *o) {
  Fl_Widget *p = focus_;
  if (o != p) {
    for (; p && !p->contains(o); p = p->parent()) p->handle(FL_UNFOCUS);
    focus_ = o;
  }
}

void Fl::belowmouse(Fl_Widget *o) {
  Fl_Widget *p = belowmouse_;
  if (o != p) {
    event_is_click(0);
    for (; p && !p->contains(o); p = p->parent()) p->handle(FL_LEAVE);
    belowmouse_ = o;
  }
}

Fl_Window *fl_xfocus;	// which window X thinks has focus
Fl_Window *fl_xmousewin; // which window X thinks has FL_ENTER
Fl_Window *Fl::grab_;	// most recent Fl::grab()
Fl_Window *Fl::modal_;

// Update modal(), focus() and other state according to x state.
// This is called whenever a window is added or hidden, and whenever
// X says the focus or mouse window have changed, and when grab_ is
// changed.
void fl_fix_focus() {
  if (Fl::grab_)
    Fl::modal_ = Fl::grab_;
  else {
    Fl_Window* w = Fl::first_window();
    while (w && w->parent()) w = Fl::next_window(w);
    Fl::modal_ = w && w->modal() ? w : 0;
  }
  Fl_Window *w = fl_xfocus;
  while (w && w->parent()) w = w->window();
  if (w) {
    if (Fl::modal()) w = Fl::modal();
    if (!w->contains(Fl::focus()))
      if (!w->take_focus()) Fl::focus(w);
  } else
    Fl::focus(0);

  if (Fl::pushed()) {
    if (Fl::modal() && !Fl::modal()->contains(Fl::pushed()))
      Fl::pushed_ = Fl::modal();
    return; // don't change belowmouse() when pushed() is true
  }

  w = fl_xmousewin;
  if (w) {
    if (Fl::modal()) w = Fl::modal();
    if (!w->contains(Fl::belowmouse())) {
      Fl::belowmouse(w); w->handle(FL_ENTER);}
  } else
    Fl::belowmouse(0);
}

////////////////////////////////////////////////////////////////
// ~Fl_Widget() calls this, to allow all the internal pointers to
// an object to be cleared.  This is also called by hide or deactivate
// on objects, it seems like the right thing to do.

#ifndef WIN32
Fl_Widget *fl_selection_requestor; // from Fl_cutpaste.C
#endif

void fl_throw_focus(Fl_Widget *o) {
  if (o->contains(Fl::pushed())) Fl::pushed(0);
  if (o->contains(Fl::selection_owner())) Fl::selection_owner(0);
#ifndef WIN32
  if (o->contains(fl_selection_requestor)) fl_selection_requestor = 0;
#endif
  int fix = 0;
  if (o->contains(Fl::belowmouse())) {Fl::belowmouse(0); fix = 1;}
  if (o->contains(Fl::focus())) {Fl::focus(0); fix = 1;}
  if (fix) fl_fix_focus();
}

////////////////////////////////////////////////////////////////
// Event handlers:

struct handler_link {
  int (*handle)(int);
  const handler_link *next;
};

int fl_backstop_handle(int event); // in Fl_x.C

static const handler_link backstop_link = {fl_backstop_handle, 0};

static const handler_link *handlers = &backstop_link;

void Fl::add_handler(int (*h)(int)) {
  handler_link *l = new handler_link;
  l->handle = h;
  l->next = handlers;
  handlers = l;
}

void fl_send_event(int event, Fl_Widget *widget) {
  if (widget && widget->handle(event)) return;
  for (const handler_link *h = handlers; ; h = h->next)
    if (h->handle(event)) return;
}

////////////////////////////////////////////////////////////////
// These are called when the system wants to send a particular event
// to a particular window.  We may want to change the window to another
// widget, especially if there is a modal one up.  Often this requires
// changing the x/y position of the event:

int Fl_Window::handle(int event) {
  return Fl_Group::handle(event);
}

void Fl::pushed(Fl_Widget *o) {
  pushed_ = o;
}

void fl_send_push(Fl_Window* window) {
  if (!Fl::pushed()) {
    if (Fl::modal() && window != Fl::modal()) {
      window = Fl::modal();
      Fl::e_x = Fl::e_x_root - window->x_root();
      Fl::e_y = Fl::e_y_root - window->y_root();
    }
    Fl::pushed_ = window;
  }
  fl_send_event(FL_PUSH, Fl::pushed());
}

void fl_send_move(Fl_Window* window) {
  fl_xmousewin = window;
  if (Fl::pushed()) {fl_send_event(FL_DRAG, Fl::pushed()); return;}
  fl_fix_focus();
  if (Fl::modal() && window != Fl::modal()) {
    window = Fl::modal();
    Fl::e_x = Fl::e_x_root - window->x_root();
    Fl::e_y = Fl::e_y_root - window->y_root();
  }
  fl_send_event(FL_MOVE,window);
}

void fl_send_release(Fl_Window* window) {
  // unfortunately pushed() must be zero when the callback() is done:
  Fl_Widget* o = Fl::pushed_; Fl::pushed_ = 0;
//  if (!o) o = Fl::first_window();
  fl_send_event(FL_RELEASE, o);
  fl_fix_focus();
  if (window->contains(Fl::belowmouse())) window->handle(FL_MOVE);
}

void fl_send_keyboard(Fl_Window* window) {
  fl_xfocus = window; fl_fix_focus();
  
  Fl_Widget *o;
  // Try it as keystroke, sending it to focus and all parents:
  for (o = Fl::focus(); o; o = o->parent())
    if (o->handle(FL_KEYBOARD)) return;

  // Try it as shortcut, sending to mouse widget and all parents:
  for (o = Fl::belowmouse(); o; o = o->parent())
    if (o->handle(FL_SHORTCUT)) return;

#if 0
  // Try sending shortcut to all other windows:
  if (!Fl::modal()) {
    for (Fl_Window* w = Fl::first_window(); w; w = Fl::next_window(w))
      if (w != o && !w->parent() && w->handle(FL_SHORTCUT)) return;
  }
#endif

  // try using add_handle() functions:
  fl_send_event(FL_SHORTCUT,0);
}

////////////////////////////////////////////////////////////////

void Fl::default_atclose(Fl_Window* window, void* v) {
  if (!window) exit(0); // Quit
  for (Fl_Window* w = first_window(); w; w = next_window(w))
    if (w != window && !w->parent()) { // we found another window
      window->hide();
      return;
    }
  // no other window, so let's quit
  // recursively call this to quit, so user's atclose() is called:
  atclose(0,v);
}

void (*Fl::atclose)(Fl_Window*, void*) = Fl::default_atclose;

// End of Fl.C //
