// Fl_x.C

// fltk (Fast Light Tool Kit) version 0.99
// Copyright (C) 1998 Bill Spitzak

// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.

// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.

// Written by Bill Spitzak spitzak@d2.com

#ifdef WIN32
#include "Fl_win32.C"
#else

// This file contains X-specific code for fltk which is always linked
// in.  Several other files contain X-specific code as well (search for
// x.H), and either have #ifdef's or have a parallel source file
// (such as foo.C and foo_win32.C) for compilation on non-X systems.

#include <config.h>
#include <FL/Fl.H>
#include <FL/x.H>
#include <FL/Fl_Window.H>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>

////////////////////////////////////////////////////////////////
// interface to poll/select call:

#if HAVE_POLL
#include <poll.h>
#else
#include <unistd.h>
struct pollfd {int fd; short events; short revents;};
#define POLLIN 1
#define POLLOUT 4
#define POLLERR 8
#ifdef __sgi // on IRIX select is broken in C++, fix it:
//inline static void bzero(void *b, int l) {memset(b,0,l);}
extern "C" int	select( int, fd_set *, fd_set *, fd_set *, struct timeval * );
#endif
#endif

#define MAXFD 8
#if !HAVE_POLL
static fd_set fdsets[3];
static int maxfd;
#endif
static int nfds;
static struct pollfd fds[MAXFD];
static struct {
  void (*cb)(int, void*);
  void* arg;
} fd[MAXFD];

void Fl::add_fd(int n, int events, void (*cb)(int, void*), void *v) {
  int i;
  if (nfds < MAXFD) {i = nfds; nfds++;} else {i = MAXFD-1;}
  fds[i].fd = n;
  fds[i].events = events;
#if !HAVE_POLL
  if (events & POLLIN) FD_SET(n, &fdsets[0]);
  if (events & POLLOUT) FD_SET(n, &fdsets[1]);
  if (events & POLLERR) FD_SET(n, &fdsets[2]);
  if (n > maxfd) maxfd = n;
#endif
  fd[i].cb = cb;
  fd[i].arg = v;
}

void Fl::add_fd(int fd, void (*cb)(int, void*), void* v) {
  Fl::add_fd(fd,POLLIN,cb,v);
}

void Fl::remove_fd(int n) {
  int i,j;
  for (i=j=0; i<nfds; i++) {
    if (fds[i].fd == n);
    else {if (j<i) {fd[j]=fd[i]; fds[j]=fds[i];} j++;}
  }
  nfds = j;
#if !HAVE_POLL
  FD_CLR(n, &fdsets[0]);
  FD_CLR(n, &fdsets[1]);
  FD_CLR(n, &fdsets[2]);
  if (n == maxfd) maxfd--;
#endif
}

// Timeouts are insert-sorted into order.  This works good if there
// are only a small number:

#define MAXTIMEOUT 8

static int numtimeouts;
static struct {
  double time;
  void (*cb)(void*);
  void* arg;
} timeout[MAXTIMEOUT+1];

void Fl::add_timeout(double t, void (*cb)(void *), void *v) {
  int i;
  if (numtimeouts<MAXTIMEOUT) numtimeouts++;
  for (i=0; i<numtimeouts-1; i++) {
    if (timeout[i].time > t) {
      for (int j=numtimeouts-1; j>i; j--) timeout[j] = timeout[j-1];
      break;
    }
  }
  timeout[i].time = t;
  timeout[i].cb = cb;
  timeout[i].arg = v;
}

void Fl::remove_timeout(void (*cb)(void *), void *v) {
  int i,j;
  for (i=j=0; i<numtimeouts; i++) {
    if (timeout[i].cb == cb && timeout[i].arg==v) ;
    else {if (j<i) timeout[j]=timeout[i]; j++;}
  }
  numtimeouts = j;
}

void (*Fl::idle)();

// I have attempted to write these to not call reset() unnecessarily,
// since on some systems this results in a time-wasting context switch
// to the system when Clock() is called.  This means that only
// Fl::reset() and Fl::wait(n) are guaranteed to reset the clock that
// timeouts are measured from.  Fl::wait(), Fl::check(), and
// Fl::ready() may or may not change the clock.

static int initclock; // if false we didn't call reset() last time

#include <sys/times.h>
#undef CLK_TCK
#define CLK_TCK 100	// CLK_TCK did not work on IRIX

double Fl::reset() {

  // This should be a value that increases uniformly over time:
  struct tms foo;
  unsigned long newclock = times(&foo); // this must call real-time clock!
  const int TICKS_PER_SECOND = CLK_TCK; // divisor of the value to get seconds

  static unsigned long prevclock;
  if (!initclock) {prevclock = newclock; initclock = 1; return 0;}
  double t = double(newclock-prevclock)/TICKS_PER_SECOND;
  prevclock = newclock;

  // elapse the pending timeouts:
  for (int i=0; i<numtimeouts; i++) timeout[i].time -= t;
  return t;
}

void call_timeouts() {
  if (!numtimeouts || timeout[0].time > 0) return;
  struct {
    void (*cb)(void *);
    void *arg;
  } temp[MAXTIMEOUT];
  int i,j,k;
  // copy all expired timeouts to temp array:
  for (i=j=0; j<numtimeouts && timeout[j].time <= 0; i++,j++) {
    temp[i].cb = timeout[j].cb;
    temp[i].arg= timeout[j].arg;
  }
  // remove them from source array:
  for (k=0; j<numtimeouts;) timeout[k++] = timeout[j++];
  numtimeouts = k;
  // and then call them:
  for (k=0; k<i; k++) temp[k].cb(temp[k].arg);
}

static void do_queued_events();

static double innards(int timeout_flag, double time) {

  // Fix OpenGL and other broken libraries that call XEventsQueued
  // unnecessarily and thus cause the file descriptor to not be ready:
  if (XQLength(fl_display)) {do_queued_events(); return time;}

#if !HAVE_POLL
  fd_set fdt[3];
  fdt[0] = fdsets[0];
  fdt[1] = fdsets[1];
  fdt[2] = fdsets[2];
#endif
  int n;

  if (!timeout_flag) {
    initclock = 0; // don't waste time calling system for current time
#if HAVE_POLL
    n = ::poll(fds, nfds, -1);
#else
    n = ::select(maxfd+1,&fdt[0],&fdt[1],&fdt[2],0);
#endif
  } else {
    if (time > 0) time -= Fl::reset();
#if HAVE_POLL
    int n = ::poll(fds, nfds, time > 0.0 ? int(time*1000) : 0);
#else
    timeval t;
    if (time <= 0.0) {
      t.tv_sec = 0;
      t.tv_usec = 0;
    } else {
      t.tv_sec = int(time);
      t.tv_usec = int(1000000 * (time-t.tv_sec));
    }
    n = ::select(maxfd+1,&fdt[0],&fdt[1],&fdt[2],&t);
#endif
  }
  if (n > 0) {
    for (int i=0; i<nfds; i++) {
#if HAVE_POLL
      if (fds[i].revents) fd[i].cb(fds[i].fd, fd[i].arg);
#else
      int f = fds[i].fd;
      short revents = 0;
      if (FD_ISSET(f,&fdt[0])) revents |= POLLIN;
      if (FD_ISSET(f,&fdt[1])) revents |= POLLOUT;
      if (FD_ISSET(f,&fdt[2])) revents |= POLLERR;
      if (fds[i].events & revents) fd[i].cb(f, fd[i].arg);
#endif
    }
    if (!time) time = -.00000001; // so check() can return non-zero
  }
  return time;
}

int Fl::wait() {
  flush();
  if (!numtimeouts && !idle) {
    innards(0, 0.0);
  } else {
    innards(1, idle ? 0.0 : timeout[0].time);
    reset();	// expire timeouts during poll call
    call_timeouts();
    if (idle) idle();
  }
  return 1;
}

double Fl::wait(double time) {
  flush();
  double wait_time = idle ? 0.0 : time;
  if (numtimeouts && timeout[0].time < time) wait_time = timeout[0].time;
  double remaining_wait = innards(1, wait_time);
  time = (time-wait_time+remaining_wait) - reset();
  call_timeouts();
  if (idle) idle();
  return time;
}

int Fl::check() {
  flush();
  double t = innards(1, 0.0);
  if (numtimeouts) {reset(); call_timeouts();}
//if (idle) idle(); // ??? should it do this ???
  return t < 0.0;
}

int Fl::ready() {
//if (idle) return 1; // ??? should it do this ???
  if (XQLength(fl_display)) return 1;
  if (numtimeouts) {reset(); if (timeout[0].time <= 0) return 1;}
#if HAVE_POLL
  return ::poll(fds, nfds, 0);
#else
  timeval t;
  t.tv_sec = 0;
  t.tv_usec = 0;
  fd_set fdt[3];
  fdt[0] = fdsets[0];
  fdt[1] = fdsets[1];
  fdt[2] = fdsets[2];
  return ::select(maxfd+1,&fdt[0],&fdt[1],&fdt[2],&t);
#endif
}

////////////////////////////////////////////////////////////////

Display *fl_display;
int fl_screen;
XVisualInfo *fl_visual;
Colormap fl_colormap;
Fl_Window *Fl_Window::current_;
Window fl_window;
GC fl_gc;

Atom fl_wm_delete_window;
Atom fl__wm_quit_app;
Atom fl_wm_protocols;
Atom fl__motif_wm_hints;

static void fd_callback(int,void *) {
  if (XEventsQueued(fl_display,1)) do_queued_events();
}

static int io_error_handler(Display*) {Fl::fatal("X I/O error"); return 0;}

static int xerror_handler(Display* d, XErrorEvent* e) {
  char buf1[128], buf2[128];
  sprintf(buf1, "XRequest.%d", e->request_code);
  XGetErrorDatabaseText(d,"",buf1,buf1,buf2,128);
  XGetErrorText(d, e->error_code, buf1, 128);
  Fl::warning("%s: %s 0x%lx", buf2, buf1, e->resourceid);
  return 0;
}

void fl_open_display() {
  if (fl_display) return;

  XSetIOErrorHandler(io_error_handler);
  XSetErrorHandler(xerror_handler);

  Display *d = XOpenDisplay(0);
  if (!d) Fl::fatal("Can't open display: %s",XDisplayName(0));

  fl_display = d;

  fl_wm_delete_window = XInternAtom(d,"WM_DELETE_WINDOW",0);
  fl__wm_quit_app = XInternAtom(d,"_WM_QUIT_APP",0);
//fl_wm_save_yourself = XInternAtom(d,"WM_SAVE_YOURSELF",0);
  fl_wm_protocols = XInternAtom(d,"WM_PROTOCOLS",0);
  fl__motif_wm_hints = XInternAtom(d,"_MOTIF_WM_HINTS",0);
  Fl::add_fd(ConnectionNumber(d), POLLIN, fd_callback);

  fl_screen = DefaultScreen(fl_display);
// construct an XVisualInfo that matches the default Visual:
  XVisualInfo templt; int num;
  templt.visualid = XVisualIDFromVisual(DefaultVisual(fl_display,fl_screen));
  fl_visual = XGetVisualInfo(fl_display, VisualIDMask, &templt, &num);
  fl_colormap = DefaultColormap(fl_display,fl_screen);
}

void fl_close_display() {
  Fl::remove_fd(ConnectionNumber(fl_display));
  XCloseDisplay(fl_display);
}

int Fl::h() {
  fl_open_display();
  return DisplayHeight(fl_display,fl_screen);
}

int Fl::w() {
  fl_open_display();
  return DisplayWidth(fl_display,fl_screen);
}

void Fl::get_mouse(int &x, int &y) {
  fl_open_display();
  Window root = RootWindow(fl_display, fl_screen);
  Window c; int mx,my,cx,cy; unsigned int mask;
  XQueryPointer(fl_display,root,&root,&c,&mx,&my,&cx,&cy,&mask);
  x = mx;
  y = my;
}

////////////////////////////////////////////////////////////////

Fl_X* Fl_X::first;

static Fl_X* freelist;

Fl_Window* fl_find(ulong xid) {
  Fl_X *window;
  for (Fl_X **pp = &Fl_X::first; (window = *pp); pp = &window->next)
    if (window->xid == xid) {
      if (window != Fl_X::first && !Fl::modal()) {
	// make this window be first to speed up searches
	// this is not done if modal is true to avoid messing up modal stack
	*pp = window->next;
	window->next = Fl_X::first;
	Fl_X::first = window;
      }
      return window->w;
    }
  return 0;
}

////////////////////////////////////////////////////////////////

extern Fl_Window *fl_xfocus;	// in Fl.C
extern Fl_Window *fl_xmousewin; // in Fl.C
void fl_fix_focus(); // in Fl.C

static Fl_Window *fl_selection_window; // window X thinks selection belongs to

// call this to free a selection (or change the owner):
void Fl::selection_owner(Fl_Widget *owner) {
  if (selection_owner_ && owner != selection_owner_)
    selection_owner_->handle(FL_SELECTIONCLEAR);
  selection_owner_ = owner;
  if (!owner)
    fl_selection_window = 0;
  else if (!fl_selection_window) {
    Fl_X *w = Fl_X::first;
    if (w) {	// we lose if there are no windows
      XSetSelectionOwner(fl_display, XA_PRIMARY, w->xid, fl_event_time);
      fl_selection_window = w->w;
    }
  }
}

////////////////////////////////////////////////////////////////

XEvent fl_xevent; // the last xevent read
ulong fl_event_time; // the last timestamp from an xevent (not all have em...)
static Fl_Window* send_motion; // window to send motion event to, if any

// record event mouse position and state from an XEvent
// checkdouble is true for button and keyboard down events

static void set_event_xy(int checkdouble=0) {
  static int px, py;
  static ulong ptime;
  Fl::e_x_root = fl_xevent.xbutton.x_root;
  Fl::e_y_root = fl_xevent.xbutton.y_root;
  Fl::e_x = fl_xevent.xbutton.x;
  Fl::e_y = fl_xevent.xbutton.y;
  Fl::e_state = fl_xevent.xbutton.state << 16;
  fl_event_time = fl_xevent.xbutton.time;
  if (abs(Fl::e_x_root-px)+abs(Fl::e_y_root-py) > 3 
      || fl_event_time >= ptime+1000)
    Fl::e_is_click = 0;
  if (checkdouble) {
    if (Fl::e_is_click == Fl::e_keysym)
      Fl::e_clicks++;
    else {
      Fl::e_clicks = 0;
      Fl::e_is_click = Fl::e_keysym;
    }
    px = Fl::e_x_root;
    py = Fl::e_y_root;
    ptime = fl_event_time;
  }
  send_motion = 0;
}

// event processors in Fl.C:
void fl_send_event(int event, Fl_Widget *widget);
void fl_send_push(Fl_Window*);
void fl_send_move(Fl_Window*);
void fl_send_release(Fl_Window*);
void fl_send_keyboard(Fl_Window*);

char fl_key_vector[32]; // used by Fl::get_key()

////////////////////////////////////////////////////////////////

static Fl_Window* fl_resize_bug_fix; // prevents echoing of resize X events

static void do_queued_events() {
  send_motion = 0;

  while (XEventsQueued(fl_display,QueuedAfterReading)) {
    XNextEvent(fl_display, &fl_xevent);

    switch (fl_xevent.type) { // events where we don't care about window

    case KeymapNotify:
      memcpy(fl_key_vector, fl_xevent.xkeymap.key_vector, 32);
      break;

    case MappingNotify:
      XRefreshKeyboardMapping(&fl_xevent.xmapping);
      break;
    }

    Fl_Window* window = fl_find(fl_xevent.xany.window);
    if (!window) goto DEFAULT;

    switch (fl_xevent.type) {

    case ClientMessage:
      if ((Atom)(fl_xevent.xclient.data.l[0])==fl_wm_delete_window) {
	// Close box clicked
	if (Fl::modal()) {
	  if (window != Fl::modal()) break;
	} else {
	  Fl::atclose(window, window->user_data());
	}
	window->do_callback();
      } else if ((Atom)fl_xevent.xclient.data.l[0]==fl__wm_quit_app) {
	// 4DWM "Quit" picked off window menu
	Fl::atclose(0, window->user_data());
      } else goto DEFAULT;
      break;

    case MapNotify:
      ((Fl_Widget*)window)->show();
      break;

    case UnmapNotify:
      ((Fl_Widget*)window)->hide();
      break;

    case Expose:
    case GraphicsExpose:
#if 1	// try to keep windows on top even if WM_TRANSIENT_FOR does not work:
      if (Fl::first_window()->non_modal() && window != Fl::first_window())
	Fl::first_window()->show();
#endif
      window->expose(2, fl_xevent.xexpose.x, fl_xevent.xexpose.y,
		     fl_xevent.xexpose.width, fl_xevent.xexpose.height);
      break;

    case ButtonPress:
      Fl::e_keysym = FL_Button + fl_xevent.xbutton.button;
      set_event_xy(1);
      Fl::e_state |= (FL_BUTTON1 << (fl_xevent.xbutton.button-1));
      fl_send_push(window);
      break;

    case MotionNotify:
      set_event_xy(); send_motion = window;
      break;

    case ButtonRelease:
      Fl::e_keysym = FL_Button + fl_xevent.xbutton.button;
      set_event_xy();
      Fl::e_state &= ~(FL_BUTTON1 << (fl_xevent.xbutton.button-1));
      fl_send_release(window);
      break;

    case FocusIn:
      Fl::e_keysym = 0; // make sure it is not confused with navigation key
      fl_xfocus = window;
      fl_fix_focus();
      break;

    case FocusOut:
      Fl::e_keysym = 0; // make sure it is not confused with navigation key
      fl_xfocus = 0;
      fl_fix_focus();
      break;

    case KeyPress: {
      static int got_backspace;
      static char buffer[21];
      KeySym keysym;
      int i = fl_xevent.xkey.keycode; fl_key_vector[i/8] |= (1 << (i%8));
      int len = XLookupString(&(fl_xevent.xkey), buffer, 20, &keysym, 0);
      if (!len) {
	if (keysym < 0x400) {
	  // turn all latin-2,3,4 characters into 8-bit codes:
	  buffer[0] = char(keysym);
	  len = 1;
	} else if (keysym >= 0xff95 && keysym < 0xffa0) {
	  // Make NumLock irrelevant (always on):
	  // This lookup table turns the XK_KP_* functions back into the
	  // ascii characters.  This won't work on non-PC layout keyboards,
	  // but are there any of those left??
	  buffer[0] = "7486293150."[keysym-0xff95];
	  len = 1;
	  keysym = FL_KP+buffer[0];
#if 0
	} else if (keysym >= FL_KP && keysym <= FL_KP_Last) {
	  // Make sure keypad numbers produce digits even if the Xlib
	  // did not do it.  Do any do this, however?
	  buffer[0] = char(keysym&0x7f);
	  len = 1;
#endif
	}
      }
      buffer[len] = 0;
      if (!got_backspace) {
	// Backspace kludge: until user hits the backspace key, assumme
	// it is missing and use the Delete key for that purpose:
	if (keysym == FL_Delete) keysym = FL_BackSpace;
	else if (keysym == FL_BackSpace) got_backspace = 1;
      }
      Fl::e_keysym = int(keysym);
      Fl::e_text = buffer;
      Fl::e_length = len;
      set_event_xy(1);
      if (Fl::event_state(FL_CTRL) && keysym == '-') buffer[0] = 0x1f; // ^_

      fl_send_keyboard(window);}
      break;

    case KeyRelease: {
      int i = fl_xevent.xkey.keycode; fl_key_vector[i/8] &= ~(1 << (i%8));
      set_event_xy();
      goto DEFAULT;}

    case EnterNotify:
      // Force the colormap, even though the X docs say don't do this:
      XInstallColormap(fl_display, Fl_X::i(window)->colormap);
      // X sends extra enter events to the parent when mouse enters
      // child, we have to ignore them:
      if (fl_xevent.xcrossing.detail == NotifyInferior) goto DEFAULT;
      // okay, this event is good, use it:
      set_event_xy();
      Fl::e_state = fl_xevent.xcrossing.state << 16;
      fl_xmousewin = window; fl_fix_focus();
      break;

    case LeaveNotify:
      if (fl_xevent.xcrossing.detail == NotifyInferior) goto DEFAULT;
      set_event_xy();
      Fl::e_state = fl_xevent.xcrossing.state << 16;
      if (window == fl_xmousewin) {fl_xmousewin = 0; fl_fix_focus();}
      break;

    case ConfigureNotify: {
      if (window->parent()) goto DEFAULT; // assumme we are ok for children
      int x = fl_xevent.xconfigure.x;
      int y = fl_xevent.xconfigure.y;
      // avoid bug (?) in 4DWM, it reports position of 0,0 on resize:
      if (!x && !y) {
	Window r, c; int X, Y; unsigned int m;
	XQueryPointer(fl_display, fl_xid(window), &r, &c, &x, &y, &X, &Y, &m);
	x = x-X; y = y-Y;
      }
      fl_resize_bug_fix = window;
      window->resize(x, y,
		     fl_xevent.xconfigure.width, fl_xevent.xconfigure.height);
      fl_resize_bug_fix = 0;}
      break;

    default:
    DEFAULT:
      fl_send_event(0,0);
    }
  }
  if (send_motion) {
    Fl_Window* w = send_motion;
    send_motion = 0;
    fl_send_move(w);
  }
}

int fl_backstop_handle(int event) {
  if (event == FL_PUSH) {
    // raise windows that are clicked on:
    Fl_X::first->w->show();
  } else if (event == FL_SHORTCUT && Fl::event_key()==FL_Escape) {
    // Escape key closes windows:
    Fl_Window* w = Fl::modal();
    if (!w) {
      w = Fl_X::first->w;
      for (;;) {
	Fl_Window* p = w->window();
	if (!p) break;
	w = p;
      }
      Fl::atclose(w, w->user_data());
    }
    w->do_callback();
  }
  return 1;
}

////////////////////////////////////////////////////////////////

void Fl_Window::resize(int X,int Y,int W,int H) {
  if (shown() && this != fl_resize_bug_fix) {
    // tell X window manager to change window size:
    if (!(flags()&FL_FORCE_POSITION) && X == x() && Y == y())
      XResizeWindow(fl_display, i->xid, W>0 ? W : 1, H>0 ? H : 1);
    else if (W != w() || H != h())
      XMoveResizeWindow(fl_display, i->xid, X, Y, W>0 ? W : 1, H>0 ? H : 1);
    else
      XMoveWindow(fl_display, i->xid, X, Y);
    if (!parent()) return; // wait for response from server
//  XSync(fl_display,0); // this seems to reduce flashing a lot
  } else {
    // This is to avoid echoing resize events back to window manager:
    fl_resize_bug_fix = 0;
  }
  if (X != x() || Y != y()) set_flag(FL_FORCE_POSITION);
  if (W != w() || H != h()) Fl_Group::resize(X,Y,W,H); else {x(X); y(Y);}
  // Notice that this does *not* set any redraw bits.  I assumme
  // I will receive damage for the whole window from X.  I think
  // that "ForgetGravity" forces the expose event for the entire
  // window, but this may not be true on some implementations.
}

////////////////////////////////////////////////////////////////

// A subclass of Fl_Window may call this to associate an X window it
// creates with the Fl_Window:

Fl_X* Fl_X::set_xid(Fl_Window* w, ulong xid, Colormap colormap) {
  Fl_X* x = freelist;
  if (x) freelist = x->next; else x = new Fl_X;
  x->xid = xid;
  x->other_xid = 0;
  x->setwindow(w);
  x->colormap = colormap;
  x->next = Fl_X::first;
  x->region = 0;
  Fl_X::first = x;
  return x;
}

// More commonly a subclass calls this, because it hides the really
// ugly parts of X and sets all the stuff for a window that is set
// normally.  The global variables like fl_show_iconic are so that
// subclasses of *that* class may change the behavior...

char fl_show_iconic;	// hack for iconize()
int fl_background_pixel = -1; // hack to speed up bg box drawing

static const int inactiveEventMask =
ExposureMask|StructureNotifyMask|EnterWindowMask;

static const int XEventMask =
ExposureMask|StructureNotifyMask
|KeyPressMask|KeyReleaseMask|KeymapStateMask|FocusChangeMask
|ButtonPressMask|ButtonReleaseMask
|EnterWindowMask|LeaveWindowMask
|PointerMotionMask;

void Fl_X::make_xid(Fl_Window* w, XVisualInfo *visual, Colormap colormap)
{
  Fl_Group::current(0); // get rid of very common user bug: forgot end()
  w->clear_damage(); // wait for X expose events

  ulong root = w->parent() ?
    fl_xid(w->window()) : RootWindow(fl_display, fl_screen);

  XSetWindowAttributes attr;
  int mask = CWBorderPixel|CWColormap|CWEventMask|CWBitGravity;
  attr.event_mask = (w->active()) ? XEventMask : inactiveEventMask;
  attr.colormap = colormap;
  attr.border_pixel = 0;
  attr.bit_gravity = 0; // StaticGravity;
  attr.override_redirect = 0;
  if (Fl::grab()) {
    attr.save_under = 1; mask |= CWSaveUnder;
    if (!w->border()) {attr.override_redirect = 1; mask |= CWOverrideRedirect;}
  }
  if (fl_background_pixel >= 0) {
    attr.background_pixel = fl_background_pixel;
    mask |= CWBackPixel;
  }
  Fl_X* x =
    set_xid(w, XCreateWindow(fl_display,
			     root,
			     w->x(), w->y(),
			     w->w()>0 ? w->w() : 1,
			     w->h()>0 ? w->h() : 1,
			     0, // borderwidth
			     visual->depth,
			     InputOutput,
			     visual->visual,
			     mask, &attr), colormap);
  XInstallColormap(fl_display, colormap);

  if (!w->parent() && !attr.override_redirect) {
    // Communicate all kinds 'o junk to the X Window Manager:

    w->label(w->label(), w->iconlabel());

    // set the WM_PROTOCOLS to WM_DELETE_WINDOW and _WM_QUIT_APP:
    Atom atoms[2];
    atoms[0] = fl_wm_delete_window;
    atoms[1] = fl__wm_quit_app;
    XChangeProperty(fl_display, x->xid, fl_wm_protocols,
		    XA_ATOM, 32, 0, (uchar *)atoms, 2);

    // send size limits and border:
    x->sendxjunk();

    // set the class property, which controls the icon used:
    if (w->xclass()) {
      char buffer[1024];
      char *p; const char *q;
      // truncate on any punctuation, because they break XResource lookup:
      for (p = buffer, q = w->xclass(); isalnum(*q)||(*q&128);) *p++ = *q++;
      *p++ = 0;
      // create the capitalized version:
      q = buffer;
      *p = toupper(*q++); if (*p++ == 'X') *p++ = toupper(*q++);
      while ((*p++ = *q++));
      XChangeProperty(fl_display, x->xid, XA_WM_CLASS, XA_STRING, 8, 0,
		      (unsigned char *)buffer, p-buffer-1);
    }

    if (w->non_modal()) { // find some other window to be "transient for":
      Fl_X* y = x->next;
      while (y && (y->w->parent() || y->w->non_modal())) y = y->next;
      if (y) XSetTransientForHint(fl_display, x->xid, x->next->xid);
    }

    if (fl_show_iconic) {
      XWMHints hints;
      hints.flags = StateHint;
      hints.initial_state = 3;
      XSetWMHints(fl_display, x->xid, &hints);
      fl_show_iconic = 0;
    }

  }

  XMapRaised(fl_display, x->xid);
  fl_fix_focus();
}

void Fl::flush() {
  Fl_Group::current(0); // get rid of very common user bug: forgot end()
  Fl_X* x = Fl_X::first;
  if (!x) return;
  if (damage()) {
    damage_ = 0;
    for (; x; x = x->next) {
      if (x->w->damage() && x->w->visible()) {
	x->flush();
	x->w->clear_damage();
	if (x->region) {XDestroyRegion(x->region); x->region = 0;}
      }
    }
  }
  XFlush(fl_display);
}

void Fl::redraw() {
  for (Fl_X* x = Fl_X::first; x; x = x->next) x->w->clear_damage(~0);
}

Fl_Window* Fl::first_window() {Fl_X* x = Fl_X::first; return x ? x->w : 0;}

Fl_Window* Fl::next_window(const Fl_Window* w) {
  Fl_X* x = Fl_X::i(w)->next; return x ? x->w : 0;}

////////////////////////////////////////////////////////////////
// hide() destroys the X window:

void Fl_Window::hide() {
  if (!shown()) return;

  // remove from the list of windows:
  Fl_X* x = i;
  Fl_X** pp = &Fl_X::first;
  for (; *pp != x; pp = &(*pp)->next) if (!*pp) return;
  *pp = x->next;

  // recursively remove any subwindows:
  for (Fl_X *w = Fl_X::first; w;) {
    if (w->w->window() == this) {w->w->hide(); w = Fl_X::first;}
    else w = w->next;
  }

  XDestroyWindow(fl_display, x->xid);
  i = 0;
  x->next = freelist; freelist = x;

  // Make sure no events are sent to this window:
  if (contains(Fl::pushed())) Fl::pushed(Fl_X::first->w);
  if (this == fl_xmousewin) fl_xmousewin = 0;
  if (this == fl_xfocus) fl_xfocus = 0;
  Fl_Widget::hide();

  // Try to move the selection to another window:
  if (this == fl_selection_window) {
    fl_selection_window = 0;
    Fl::selection_owner(Fl::selection_owner());
  }
}

Fl_Window::~Fl_Window() {
  hide();
}

////////////////////////////////////////////////////////////////
// Send X window stuff that can be changed over time:

void Fl_X::sendxjunk() {
  if (w->parent()) return; // it's not a window manager window!

  if (!w->size_range_set) { // default size_range based on resizable():
    if (w->resizable()) {
      Fl_Widget *o = w->resizable();
      int minw = o->w(); if (minw > 100) minw = 100;
      int minh = o->h(); if (minh > 100) minh = 100;
      w->size_range(w->w() - o->w() + minw, w->h() - o->h() + minh, 0, 0);
    } else {
      w->size_range(w->w(), w->h(), w->w(), w->h());
    }
    return; // because this recursively called here
  }

  XSizeHints hints;
  hints.min_width = w->minw;
  hints.min_height = w->minh;
  hints.max_width = w->maxw;
  hints.max_height = w->maxh;
  hints.width_inc = w->dw;
  hints.height_inc = w->dh;

  // see the file /usr/include/X11/Xm/MwmUtil.h:
  // fill all fields to avoid bugs in kwm and perhaps other window managers:
  // MWM_HINTS_FUNCTIONS|MWM_HINTS_DECORATIONS, MWM_FUNC_ALL, MWM_DECOR_ALL
  long prop[5] = {1|2, 1, 1, 0, 0};

  if (hints.min_width != hints.max_width ||
      hints.min_height != hints.max_height) { // resizable
    hints.flags = PMinSize;
    if (hints.max_width >= hints.min_width ||
	hints.max_height >= hints.min_height) {
      hints.flags = PMinSize|PMaxSize;
      // unfortunately we can't set just one maximum size.  Guess a
      // value for the other one.  Some window managers will make the
      // window fit on screen when maximized, others will put it off screen:
      if (hints.max_width < hints.min_width) hints.max_width = Fl::w();
      if (hints.max_height < hints.min_height) hints.max_height = Fl::h();
    }
    if (hints.width_inc && hints.height_inc) hints.flags |= PResizeInc;
    if (w->aspect) {
      // stupid X!  It could insist that the corner go on the
      // straight line between min and max...
      hints.min_aspect.x = hints.max_aspect.x = hints.min_width;
      hints.min_aspect.y = hints.max_aspect.y = hints.min_height;
      hints.flags |= PAspect;
    }
  } else { // not resizable:
    hints.flags = PMinSize|PMaxSize;
    prop[1]= 1|2|16; // MWM_FUNC_ALL | MWM_FUNC_RESIZE | MWM_FUNC_MAXIMIZE
  }

  if (w->flags() & Fl_Window::FL_FORCE_POSITION) {
    hints.flags |= USPosition;
    hints.x = w->x();
    hints.y = w->y();
  }

  if (!w->border()) prop[2] = 0; // no decorations

  XSetWMNormalHints(fl_display, xid, &hints);
  XChangeProperty(fl_display, xid,
		  fl__motif_wm_hints, fl__motif_wm_hints,
		  32, 0, (unsigned char *)prop, 5);
}

void Fl_Window::size_range_() {
  size_range_set = 1;
  if (shown()) i->sendxjunk();
}

////////////////////////////////////////////////////////////////

// returns pointer to the filename, or null if name ends with '/'
const char *filename_name(const char *name) {
  const char *p,*q;
  for (p=q=name; *p;) if (*p++ == '/') q = p;
  return q;
}

void Fl_Window::label(const char *name,const char *iname) {
  Fl_Widget::label(name);
  iconlabel_ = iname;
  if (shown() && !parent()) {
    if (!name) name = "";
    XChangeProperty(fl_display, i->xid, XA_WM_NAME,
		    XA_STRING, 8, 0, (uchar*)name, strlen(name));
    if (!iname) iname = filename_name(name);
    XChangeProperty(fl_display, i->xid, XA_WM_ICON_NAME, 
		    XA_STRING, 8, 0, (uchar*)iname, strlen(iname));
  }
}

void Fl_Window::label(const char *name) {
  label(name, iconlabel());
}

void Fl_Window::iconlabel(const char *iname) {
  label(label(), iname);
}

////////////////////////////////////////////////////////////////
// Implement the virtual functions for the base Fl_Window class:

// If the box is a filled rectangle, we can make the redisplay *look*
// faster by using X's background pixel erasing.  We can make it
// actually *be* faster by drawing the frame only, this is done by
// setting fl_boxcheat, which is seen by code in fl_drawbox.C:
Fl_Window *fl_boxcheat;
static inline int can_boxcheat(uchar b) {return (b==1 || (b&2) && b<=15);}

void Fl_Window::show() {
  if (!shown()) {
    fl_open_display();
    if (can_boxcheat(box())) fl_background_pixel = int(fl_xpixel(color()));
    Fl_X::make_xid(this);
    fl_background_pixel = -1;
  } else {
    XMapRaised(fl_display, i->xid);
  }
}

// make X drawing go into this window (called by subclass flush() impl.)
void Fl_Window::make_current() {
  static GC gc;	// the GC used by all X windows
  if (!gc) gc = XCreateGC(fl_display, i->xid, 0, 0);
  fl_window = i->xid;
  fl_gc = gc;
  current_ = this;
}

#include <FL/fl_draw.H>

// Current meaning of damage() bits on a window:
// 1 = a child needs redrawing
// 2 = region exposed, can be fixed by copying from back buffer
// 4 = region needs redrawing
// 128 = complete redraw, so don't waste time with region

// X expose events and cropped damage call this:
void Fl_Window::expose(uchar flags,int X,int Y,int W,int H) {
  if (i) {
    if (!i->region) i->region = XCreateRegion();
    XRectangle R;
    R.x = X; R.y = Y; R.width = W; R.height = H;
    XUnionRectWithRegion(&R, i->region, i->region);
  }
  damage(flags);
}

void Fl_Window::flush() {
  make_current();
  if (damage() & ~6) {
    draw();
  } else {
    fl_clip_region(i->region); i->region = 0;
    draw();
    fl_pop_clip();
  }
}

#endif
// End of Fl_x.C
